/**********************************************************************
zyGrib: meteorological GRIB file viewer
Copyright (C) 2008-2010 - Jacques Zaninetti - http://www.zygrib.org

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
***********************************************************************/

// Analyse MeteoBlue data file

#include <iostream>
#include <list>
#include <cmath>
#include <cassert>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "MblueRecord.h"
#include "Util.h"



//===============================================================================
//===============================================================================
MbluePoint::MbluePoint () 
{
	ok = false;
	for (int i=0; i<MB_NBVALUES; i++) {
		values[i] = GRIB_NOTDEF;
	}
}
//--------------------------------------------------------------------------------
MbluePoint::MbluePoint ( double px, double py,
					 int  nbPoints,
					 MbluePoint *p1,
					 MbluePoint *p2,
					 MbluePoint *p3,
					 MbluePoint *p4 )
{
	ok = true;
	double d1, d2, d3, d4;
	double v1, v2, v3, v4;
	double k1, k2, k3, k4;
	
	if (nbPoints < 2) {
		ok = false;
		for (int i=0; i<MB_NBVALUES; i++) {
			values[i] = GRIB_NOTDEF;
		}
		return;
	}
	if (nbPoints==4) {
			d1 = (px-p1->x)*(px-p1->x)+(py-p1->y)*(py-p1->y);
			d2 = (px-p2->x)*(px-p2->x)+(py-p2->y)*(py-p2->y);
			d3 = (px-p3->x)*(px-p3->x)+(py-p3->y)*(py-p3->y);
			d4 = (px-p4->x)*(px-p4->x)+(py-p4->y)*(py-p4->y);
			k1 = 1.0/(d1*d1+1e-12);
			k2 = 1.0/(d2*d2+1e-12);
			k3 = 1.0/(d3*d3+1e-12);
			k4 = 1.0/(d4*d4+1e-12);
			for (int i=0; i<MB_NBVALUES; i++) {
				v1 = p1->getValue (i);
				v2 = p2->getValue (i);
				v3 = p3->getValue (i);
				v4 = p4->getValue (i);
				if (v1==GRIB_NOTDEF || v2==GRIB_NOTDEF || v3==GRIB_NOTDEF || v4==GRIB_NOTDEF)
					values[i] = GRIB_NOTDEF;
				else
					values[i] =  (k1*v1 + k2*v2 + k3*v3 + k4*v4)/(k1+k2+k3+k4);
			}
	}
	else if (nbPoints==3) {
			d1 = (px-p1->x)*(px-p1->x)+(py-p1->y)*(py-p1->y);
			d2 = (px-p2->x)*(px-p2->x)+(py-p2->y)*(py-p2->y);
			d3 = (px-p3->x)*(px-p3->x)+(py-p3->y)*(py-p3->y);
			k1 = 1.0/(d1*d1+1e-12);
			k2 = 1.0/(d2*d2+1e-12);
			k3 = 1.0/(d3*d3+1e-12);
			for (int i=0; i<MB_NBVALUES; i++) {
				v1 = p1->getValue (i);
				v2 = p2->getValue (i);
				v3 = p3->getValue (i);
				if (v1==GRIB_NOTDEF || v2==GRIB_NOTDEF || v3==GRIB_NOTDEF)
					values[i] = GRIB_NOTDEF;
				else
					values[i] =  (k1*v1 + k2*v2 + k3*v3)/(k1+k2+k3);
			}
	}
	else if (nbPoints==2) {
			d1 = (px-p1->x)*(px-p1->x)+(py-p1->y)*(py-p1->y);
			d2 = (px-p2->x)*(px-p2->x)+(py-p2->y)*(py-p2->y);
			k1 = 1.0/(d1*d1+1e-12);
			k2 = 1.0/(d2*d2+1e-12);
			for (int i=0; i<MB_NBVALUES; i++) {
				v1 = p1->getValue (i);
				v2 = p2->getValue (i);
				if (v1==GRIB_NOTDEF || v2==GRIB_NOTDEF)
					values[i] = GRIB_NOTDEF;
				else
					values[i] =  (k1*v1 + k2*v2)/(k1+k2);
			}
	}
}

//--------------------------------------------------------------------------------
MbluePoint::MbluePoint (ZUFILE *file, int nbdata) 
{
	ok = true;
	nbvalues = nbdata;
	for (int i=0; i<MB_NBVALUES; i++) {
		values[i] = GRIB_NOTDEF;
	}
	int nb =0;
	if (nb<nbdata)  { read_float32 (file, &x); nb++; }
	if (nb<nbdata)  { read_float32 (file, &y); nb++; }
	if (nb<nbdata)  { read_float32 (file, &values [MB_temp2m]); nb++; }
	if (nb<nbdata)  { read_float32 (file, &values [MB_rh2m]); nb++; }
	if (nb<nbdata)  { read_float32 (file, &values [MB_precip]); nb++; }
	if (nb<nbdata)  { read_float32 (file, &values [MB_windu10m]); nb++; } 
	if (nb<nbdata)  { read_float32 (file, &values [MB_windv10m]); nb++; }
	if (nb<nbdata)  { read_float32 (file, &values [MB_clouds]); nb++; } 
	if (nb<nbdata)  { read_float32 (file, &values [MB_gust]); nb++; }
	if (nb<nbdata)  { read_float32 (file, &values [MB_pmsl]); nb++; } 
	if (nb<nbdata)  { read_float32 (file, &values [MB_tempsfc]); nb++; } 
	if (nb<nbdata)  { read_float32 (file, &values [MB_rainprob]); nb++; }
	float tmp;
	for ( ; nb<nbdata; nb++)
		read_float32 (file, &tmp);	// flush data if necessary

	// Adjust values
	if (x<-360 || x>360 || y<-90 || y>90)
		ok = false;
	
	if (values[MB_temp2m] != GRIB_NOTDEF)
		values[MB_temp2m] += 273.15;	// kelvin
	if (values[MB_tempsfc] != GRIB_NOTDEF)
		values[MB_tempsfc] += 273.15;	// kelvin

	if (values[MB_rh2m]<0 || values[MB_rh2m]>100)
			values[MB_rh2m] = GRIB_NOTDEF;
	if (values[MB_clouds]<0 || values[MB_clouds]>100)
			values[MB_clouds] = GRIB_NOTDEF;
	if (values[MB_rainprob]<0 || values[MB_rainprob]>100)
			values [MB_rainprob] = GRIB_NOTDEF;
	
	if (values[MB_pmsl]<84000 || values[MB_pmsl]>112000)
			values[MB_pmsl] = GRIB_NOTDEF;
	
	// Compute dew point with the Magnus-Tetens formula
	if (values[MB_temp2m] != GRIB_NOTDEF && values[MB_rh2m] != GRIB_NOTDEF)
		values[MB_dewpoint2m] 
				= DataRecordAbstract::dewpointMagnusTetens (values[MB_temp2m], values[MB_rh2m]);
}

//-------------------------------------------------------------
int MbluePoint::getDataIndex (DataCode dtc)
{
	if (dtc.equals (GRB_WIND_VX, LV_ABOV_GND, 10))
		return MB_windu10m;
	else if (dtc.equals (GRB_WIND_VY, LV_ABOV_GND, 10))
		return MB_windv10m;
	else if (dtc.equals (GRB_PRESSURE, LV_MSL, 0))
		return MB_pmsl;
	else if (dtc.equals (GRB_CLOUD_TOT, LV_ATMOS_ALL, 0))
		return MB_clouds;
	else if (dtc.equals (GRB_PRECIP_TOT, LV_GND_SURF, 0))
		return MB_precip;
	else if (dtc.equals (GRB_TEMP, LV_ABOV_GND, 2))
		return MB_temp2m;
	else if (dtc.equals (GRB_TEMP, LV_GND_SURF, 0))
		return MB_tempsfc;
	else if (dtc.equals (GRB_HUMID_REL, LV_ABOV_GND, 2))
		return MB_rh2m;
	else if (dtc.equals (GRB_DEWPOINT, LV_ABOV_GND, 2))
		return MB_dewpoint2m;
	else {
		//DBG("unknown data %d %d %d", dataType,levelType,levelValue);
		return -1;
	}
}	

//-------------------------------------------------------------
double MbluePoint::getValue (DataCode dtc)
{
	return getValue (getDataIndex (dtc));
}	

//-------------------------------------------------------------
double MbluePoint::getValue (int index)
{
	if (index>=0 && index<MB_NBVALUES) {
		return values [index];
	}
	else
		return GRIB_NOTDEF;
}	



//===========================================================
//===========================================================
//===========================================================
// MblueRecord
//===========================================================
//===========================================================
//===========================================================
MblueRecord::MblueRecord (ZUFILE *file)
{
	clusters = NULL;
	regularGrid = NULL;
	smoothPressureGrid = NULL;
	Ni = 0;
	Nj = 0;
	if (file != NULL) 
	{
		ok = readFile (file);
		if (ok) {
			makeClusters ();
			DBG("nb points : %d", allPoints.size());
			makeVirtualRegularGrid ();
			makeSmoothPressureGrid ();
		}
	}
}
//-----------------------------------------------------
MblueRecord::~MblueRecord ()
{
	if (clusters != NULL)
		delete [] clusters;
	if (regularGrid != NULL) {
		for (int i=0; i<Ni*Nj; i++)
				delete regularGrid [i];
		delete [] regularGrid;
	}
	if (smoothPressureGrid != NULL)
		delete [] smoothPressureGrid;
	Util::cleanListPointers (allPoints);
}
//-----------------------------------------------------
void MblueRecord::makeSmoothPressureGrid ()
{
	if (!ok)
		return;
	smoothPressureGrid = new float [Ni*Nj];
	assert (smoothPressureGrid);
	int i, j;
	float v0,v1,v2,v3,v4,v5,v6,v7,v8;
	DataCode dtc (GRB_PRESSURE,LV_MSL,0);
	int ind = MbluePoint::getDataIndex (dtc);
	
    for (j=1; j<Nj-1; j++) {
        for (i=1; i<Ni-1; i++) {
            v0 = getValueOnRegularGrid (ind, i, j );
            v1 = getValueOnRegularGrid (ind, i-1, j-1 );
            v2 = getValueOnRegularGrid (ind, i  , j-1 );
            v3 = getValueOnRegularGrid (ind, i+1, j-1 );
            v4 = getValueOnRegularGrid (ind, i-1, j );
            v5 = getValueOnRegularGrid (ind, i+1, j );
            v6 = getValueOnRegularGrid (ind, i-1, j+1 );
            v7 = getValueOnRegularGrid (ind, i  , j+1 );
            v8 = getValueOnRegularGrid (ind, i+1, j+1 );
			if (   v0 != GRIB_NOTDEF  && v1 != GRIB_NOTDEF  && v2 != GRIB_NOTDEF
				&& v3 != GRIB_NOTDEF  && v4 != GRIB_NOTDEF  && v5 != GRIB_NOTDEF  
				&& v6 != GRIB_NOTDEF  && v7 != GRIB_NOTDEF  && v8 != GRIB_NOTDEF 
			) {
				smoothPressureGrid [i+j*Ni] = (v0 +v1+v2+v3+v4+v5+v6+v7+v8) / 9.0;
			}
			else {
				smoothPressureGrid [i+j*Ni] = GRIB_NOTDEF;
			}
		}
	}
    j = 0;
	for (i=1; i<Ni-1; i++) {
		v0 = getValueOnRegularGrid (ind, i, j );
		v4 = getValueOnRegularGrid (ind, i-1, j );
		v5 = getValueOnRegularGrid (ind, i+1, j );
		v6 = getValueOnRegularGrid (ind, i-1, j+1 );
		v7 = getValueOnRegularGrid (ind, i  , j+1 );
		v8 = getValueOnRegularGrid (ind, i+1, j+1 );
		if (v0 != GRIB_NOTDEF  && v4 != GRIB_NOTDEF && v5 != GRIB_NOTDEF  
				&& v6 != GRIB_NOTDEF && v7 != GRIB_NOTDEF  && v8 != GRIB_NOTDEF)
			smoothPressureGrid [i+j*Ni] = (v0 +v4+v5+v6+v7+v8) / 6.0;
		else
			smoothPressureGrid [i+j*Ni] = GRIB_NOTDEF;
	}
    j = Nj-1;
	for (i=1; i<Ni-1; i++) {
		v0 = getValueOnRegularGrid (ind, i, j );
		v1 = getValueOnRegularGrid (ind, i-1, j-1 );
		v2 = getValueOnRegularGrid (ind, i  , j-1 );
		v3 = getValueOnRegularGrid (ind, i+1, j-1 );
		v4 = getValueOnRegularGrid (ind, i-1, j );
		v5 = getValueOnRegularGrid (ind, i+1, j );
		if (v0 != GRIB_NOTDEF  && v1 != GRIB_NOTDEF  && v2 != GRIB_NOTDEF
				&& v3 != GRIB_NOTDEF  && v4 != GRIB_NOTDEF  && v5 != GRIB_NOTDEF)
			smoothPressureGrid [i+j*Ni] = (v0 +v1+v2+v3+v4+v5) / 6.0;
		else
			smoothPressureGrid [i+j*Ni] = GRIB_NOTDEF;
	}
	i = 0;
    for (j=1; j<Nj-1; j++) {
		v0 = getValueOnRegularGrid (ind, i, j );
		v2 = getValueOnRegularGrid (ind, i  , j-1 );
		v3 = getValueOnRegularGrid (ind, i+1, j-1 );
		v5 = getValueOnRegularGrid (ind, i+1, j );
		v7 = getValueOnRegularGrid (ind, i  , j+1 );
		v8 = getValueOnRegularGrid (ind, i+1, j+1 );
		if (v0 != GRIB_NOTDEF  && v2 != GRIB_NOTDEF  && v3 != GRIB_NOTDEF
				&& v5 != GRIB_NOTDEF  && v7 != GRIB_NOTDEF  && v8 != GRIB_NOTDEF)
			smoothPressureGrid [i+j*Ni] = (v0 +v2+v3+v5+v7+v8) / 6.0;
		else
			smoothPressureGrid [i+j*Ni] = GRIB_NOTDEF;
	}
	i = Ni-1;
    for (j=1; j<Nj-1; j++) {
		v0 = getValueOnRegularGrid (ind, i, j );
		v1 = getValueOnRegularGrid (ind, i-1, j-1 );
		v2 = getValueOnRegularGrid (ind, i  , j-1 );
		v4 = getValueOnRegularGrid (ind, i-1, j );
		v6 = getValueOnRegularGrid (ind, i-1, j+1 );
		v7 = getValueOnRegularGrid (ind, i  , j+1 );
		if (v0 != GRIB_NOTDEF  && v1 != GRIB_NOTDEF  && v2 != GRIB_NOTDEF
				&& v4 != GRIB_NOTDEF  && v6 != GRIB_NOTDEF  && v7 != GRIB_NOTDEF)
			smoothPressureGrid [i+j*Ni] = (v0 +v1+v2+v4+v6+v7) / 6.0;
		else
			smoothPressureGrid [i+j*Ni] = GRIB_NOTDEF;
	}
	i = 0;
	j = 0;
		v0 = getValueOnRegularGrid (ind, i, j );
		v5 = getValueOnRegularGrid (ind, i+1, j );
		v7 = getValueOnRegularGrid (ind, i  , j+1 );
		v8 = getValueOnRegularGrid (ind, i+1, j+1 );
		if (v0 != GRIB_NOTDEF && v5 != GRIB_NOTDEF && v7 != GRIB_NOTDEF && v8 != GRIB_NOTDEF)
			smoothPressureGrid [i+j*Ni] = (v0 +v5+v7+v8) / 4.0;
		else
			smoothPressureGrid [i+j*Ni] = GRIB_NOTDEF;
	i = Ni-1;
	j = 0;
		v0 = getValueOnRegularGrid (ind, i, j );
		v4 = getValueOnRegularGrid (ind, i-1, j );
		v6 = getValueOnRegularGrid (ind, i-1, j+1 );
		v7 = getValueOnRegularGrid (ind, i  , j+1 );
		if (v0 != GRIB_NOTDEF && v4 != GRIB_NOTDEF && v6 != GRIB_NOTDEF && v7 != GRIB_NOTDEF)
			smoothPressureGrid [i+j*Ni] = (v0 +v4+v6+v7) / 4.0;
		else
			smoothPressureGrid [i+j*Ni] = GRIB_NOTDEF;
	i = 0;
	j = Nj-1;
		v0 = getValueOnRegularGrid (ind, i, j );
		v2 = getValueOnRegularGrid (ind, i  , j-1 );
		v3 = getValueOnRegularGrid (ind, i+1, j-1 );
		v5 = getValueOnRegularGrid (ind, i+1, j );
		if (v0 != GRIB_NOTDEF && v2 != GRIB_NOTDEF && v3 != GRIB_NOTDEF && v5 != GRIB_NOTDEF)
			smoothPressureGrid [i+j*Ni] = (v0 +v2+v3+v5) / 4.0;
		else
			smoothPressureGrid [i+j*Ni] = GRIB_NOTDEF;
		
	i = Ni-1;
	j = Nj-1;
		v0 = getValueOnRegularGrid (ind, i, j );
		v1 = getValueOnRegularGrid (ind, i-1, j-1 );
		v2 = getValueOnRegularGrid (ind, i  , j-1 );
		v4 = getValueOnRegularGrid (ind, i-1, j );
		if (v0 != GRIB_NOTDEF && v1 != GRIB_NOTDEF  && v2 != GRIB_NOTDEF && v4 != GRIB_NOTDEF)
			smoothPressureGrid [i+j*Ni] = (v0 +v1+v2+v4) / 4.0;
		else
			smoothPressureGrid [i+j*Ni] = GRIB_NOTDEF;
}
//-----------------------------------------------------
void MblueRecord::makeVirtualRegularGrid ()
{
	if (!ok)
		return;
	double dens = allPoints.size() / ((xmax-xmin)*(ymax-ymin));
	double dt = 1 / sqrt (dens);
	if (dt<0.1)
			dt = 0.1;
	// Virtual grid of size dt x dt
	Ni = (int) ceil ((xmax-xmin)/dt);
	Nj = (int) ceil ((ymax-ymin)/dt);
	if (Ni < 2)
		Ni = 2;
	if (Nj < 2)
		Nj = 2;
	
	Di = (xmax-xmin)/(Ni-1) - 1e-12;
	Dj = (ymax-ymin)/(Nj-1) - 1e-12;
	
	DBG("Ni=%d, Nj=%d,  Ni*Nj=%d,   Di=%f Dj=%f ", Ni, Nj, Ni*Nj, Di,Dj);

	regularGrid = new MbluePoint* [Ni*Nj];
	assert (regularGrid);
	
	MbluePoint *p1, *p2, *p3, *p4;
	int  nb;
	double px, py;
	
	for (int i=0; i<Ni; i++) {
		for (int j=0; j<Nj; j++) {
			px = getX(i);
			py = getY(j);
			nb = findNeighbour_clusters (px, py, &p1,&p2,&p3,&p4);
			regularGrid [i+j*Ni] = 
							new MbluePoint (px, py, 
											nb, p1, p2, p3, p4);
		}
	}
}
//-----------------------------------------------------
void MblueRecord::makeClusters ()
{
	int clustersize = 3;	// mean number of points in each cluster
	int nbclusters = allPoints.size()/clustersize;
	double k = (xmax-xmin)/(ymax-ymin);
	clustersNBY = (int) (sqrt (nbclusters/k));
	clustersNBX = (int) (k*clustersNBY);
	if (clustersNBX<1)
			clustersNBX = 1;
	if (clustersNBY<1)
			clustersNBY = 1;
DBG("clustersNBX %d   clustersNBY %d  tot=%d", clustersNBX,clustersNBY, clustersNBX*clustersNBY)	;
	clusters = new std::list <MbluePoint *> [clustersNBX*clustersNBY];
	assert (clusters);
	std::list <MbluePoint *>::iterator it;
	for (it=allPoints.begin(); it !=allPoints.end(); it++)
	{
		MbluePoint *p = *it;
		getCluster (p->x,p->y).push_back(p);
	}
}

//-----------------------------------------------------
std::list <MbluePoint *> & MblueRecord::getCluster (double x, double y) const
{
	double dx = (xmax-xmin)/clustersNBX;
	double dy = (ymax-ymin)/clustersNBY;
	int i, j;
	i = (int) floor((x-xmin)/dx);
	j = (int) floor((y-ymin)/dy);
	// avoid rounding errors
	if (i<0) i=0;
	if (j<0) j=0;
	if (i>=clustersNBX) i=clustersNBX-1;
	if (j>=clustersNBY) j=clustersNBY-1;
	return clusters[j*clustersNBX+i];
}

//-----------------------------------------------------
std::list <std::list <MbluePoint *> > * MblueRecord::getClustersList (
					double x, double y) const
{
 	std::list <std::list <MbluePoint *> > *listall;
	listall = new std::list <std::list <MbluePoint *> >;
	assert(listall);
	
	double dx = (xmax-xmin)/clustersNBX;
	double dy = (ymax-ymin)/clustersNBY;
	int i, j;
	i = (int) floor((x-xmin)/dx);
	j = (int) floor((y-ymin)/dy);
	// avoid rounding errors
	if (i<0) i=0;
	if (j<0) j=0;
	if (i>=clustersNBX) i=clustersNBX-1;
	if (j>=clustersNBY) j=clustersNBY-1;

	listall->push_back (clusters[j*clustersNBX+i]);
	if (i>0) 
		listall->push_back (clusters[j*clustersNBX+(i-1)]);
	if (i<clustersNBX-1) 
		listall->push_back (clusters[j*clustersNBX+(i+1)]);
	if (j>0) 
		listall->push_back (clusters[(j-1)*clustersNBX+i]);
	if (j<clustersNBY-1) 
		listall->push_back (clusters[(j+1)*clustersNBX+i]);

	if (i>0 && j>0) 
		listall->push_back (clusters[(j-1)*clustersNBX+(i-1)]);
	if (i<clustersNBX-1 && j<clustersNBY-1) 
		listall->push_back (clusters[(j+1)*clustersNBX+(i+1)]);
	if (i>0 && j<clustersNBY-1) 
		listall->push_back (clusters[(j+1)*clustersNBX+(i-1)]);
	if (i<clustersNBX-1 && j>0) 
		listall->push_back (clusters[(j-1)*clustersNBX+(i+1)]);

	return listall;
}

//-----------------------------------------------------
int  MblueRecord::findNeighbour_clusters (
					double lon, double lat,
					MbluePoint **p1, MbluePoint **p2, 
					MbluePoint **p3, MbluePoint **p4 ) const
{
	double d1,d2,d3,d4;
	d1 = d2 =  d3 =  d4 = 1e100;
	
	*p1 = *p2 = *p3 = *p4 = NULL;

	std::list <std::list <MbluePoint *> > *clusterslist = getClustersList (lon,lat);
	std::list <std::list <MbluePoint *> >::iterator it; 
	int nb=0;
	if (clusterslist != NULL) 
	{
		for (it=clusterslist->begin(); it!=clusterslist->end(); it++)
		{
			std::list <MbluePoint *> cluster = *it;
			nb += findBestNeighboursInCluster (
						lon, lat, cluster,
						p1, p2, p3, p4,  
						&d1, &d2, &d3 , &d4 );
		}
		delete clusterslist;
	}
	
	if (nb > 4)
		nb = 4;
	return nb;
}
//-------------------------------------------------------
int MblueRecord::findBestNeighboursInCluster (
						double lon, double lat,
						std::list <MbluePoint *> cluster,
						MbluePoint **p1, MbluePoint **p2, 
						MbluePoint **p3, MbluePoint **p4, 
						double *d1, double *d2, double *d3, double *d4
			) const
{	
	double d;
	int nb=0;
	std::list <MbluePoint *>::iterator it;
	for (it=cluster.begin(); it !=cluster.end(); it++)
	{
		MbluePoint *p = *it;
		d = (p->x-lon)*(p->x-lon)+(p->y-lat)*(p->y-lat);
		if (d < *d1) {
			*p4 = *p3;
			*p3 = *p2;
			*p2 = *p1;
			*p1 = p;
			*d4 = *d3;
			*d3 = *d2;
			*d2 = *d1;
			*d1 = d;
			nb ++;
		}
		else if (d < *d2) {
			*p4 = *p3;
			*p3 = *p2;
			*p2 = p;
			*d4 = *d3;
			*d3 = *d2;
			*d2 = d;
			nb ++;
		}
		else if (d < *d3) {
			*p4 = *p3;
			*p3 = p;
			*d4 = *d3;
			*d3 = d;
			nb ++;
		}
		else if (d < *d4) {
			*p4 = p;
			*d4 = d;
			nb ++;
		}
	}
	
	if (nb > 4)
		nb = 4;
	return nb;
}

//-----------------------------------------------------
bool MblueRecord::read_int32 (ZUFILE *file, int *val)
{
	unsigned char buf[16];
	if (zu_read(file,buf,4) != 4)
		return false;
	*val = ((int)(buf[0])<<24) + ((int)(buf[1])<<16) + ((int)(buf[2])<<8) + (int)buf[3];
	//DBG("%02x %02x %02x %02x -> %d", buf[0],buf[1],buf[2],buf[3], *val);
	return true;
}

//-----------------------------------------------------
bool MbluePoint::read_float32 (ZUFILE *file, float *val)
{
	char buf[16];
	if (zu_read(file,buf,4) != 4)
		return false;
	*val = *( (float*)((void*)buf) );
	return true;
}

//-----------------------------------------------------
bool MblueRecord::readFile (ZUFILE *file)
{
	char buf[1024];
	int   year,month,day,href,hcur; 
	
	if (zu_read(file,buf,10) != 8)
		return false;
	buf[8] = '\0';
	if (strcmp(buf, "MBZYGRIB") != 0)
		return false;
	
	if (zu_read(file,buf,4) != 4)
		return false;
	buf[4] = '\0';
	year = atoi (buf);
	if (year <1970 || year > 3333) 	// here is the year 3333 bug
		return false;

	if (zu_read(file,buf,2) != 2)
		return false;
	buf[2] = '\0';
	month = atoi (buf);
	if (month <1 || month > 12) 
		return false;

	if (zu_read(file,buf,2) != 2)
		return false;
	buf[2] = '\0';
	day = atoi (buf);
	if (day<1 || day> 31) 
		return false;

	if (zu_read(file,buf,2) != 2)
		return false;
	buf[2] = '\0';
	href = atoi (buf);
	if (href<0 || href>24) 
		return false;

	if (zu_read(file,buf,3) != 3)
		return false;
	buf[3] = '\0';
	hcur = atoi (buf);
	if (hcur<0 || hcur>999) 
		return false;
	
	refDate = UTC_mktime (year,month,day, href,0,0);
	curDate = UTC_mktime (year,month,day, hcur,0,0);

	if (zu_read(file,buf,8) != 8)
		return false;

	int nblines, nbdata;
	if (!read_int32(file, &nblines))
		return false;
	if (!read_int32(file, &nbdata))
		return false;
	
	if (nblines<1 || nblines >1000000000) 
		return false;
	if (nbdata<1 || nbdata>100) 
		return false;
	
	xmin =  1e6;
	xmax = -1e6;
	ymin =  1e6;
	ymax = -1e6;
	for (int i=0; i<nblines; i++)
	{
		MbluePoint *point = new MbluePoint (file, nbdata);
		assert (point);
		if (point->isOk()) {
			allPoints.push_back (point);
			if (xmin > point->x) xmin=point->x;
			if (xmax < point->x) xmax=point->x;
			if (ymin > point->y) ymin=point->y;
			if (ymax < point->y) ymax=point->y;
		}
		else
			delete point;
		
	}
	
	if (   (xmax - xmin) < 0.01
		|| (ymax - ymin) < 0.01
		|| allPoints.size() < 4
	) {
		ok = false;
	}
	return true;
}	

//--------------------------------------------------------------------
bool MblueRecord::getZoneExtension (double *x0,double *y0, double *x1,double *y1)
{
	*x0 = xmin;
	*y0 = ymin;
	*x1 = xmax;
	*y1 = ymax;
	return ok;
}
//--------------------------------------------------------------------
double  MblueRecord::getInterpolatedValue (
				int index, 
				double px, double py,
				bool interpolateValues) const
{
    if (!ok) {
        return GRIB_NOTDEF;
    }
    if (!isPointInMap(px,py)) {
        px += 360.0;               // tour du monde à droite ?
        if (!isPointInMap(px,py)) {
            px -= 2*360.0;              // tour du monde à gauche ?
            if (!isPointInMap(px,py)) {
                return GRIB_NOTDEF;
            }
        }
    }
	MbluePoint *p1, *p2, *p3, *p4;
	double d1, d2, d3, d4;
	double v1, v2, v3, v4;
	double k1, k2, k3, k4;
	
	int  nb = findNeighbour_clusters (px, py, &p1,&p2,&p3,&p4);
	if (nb < 2)
		return GRIB_NOTDEF;
	
	if (! interpolateValues) {
			return p1->getValue (index);
	}
	if (nb==4) {
			d1 = (px-p1->x)*(px-p1->x)+(py-p1->y)*(py-p1->y);
			d2 = (px-p2->x)*(px-p2->x)+(py-p2->y)*(py-p2->y);
			d3 = (px-p3->x)*(px-p3->x)+(py-p3->y)*(py-p3->y);
			d4 = (px-p4->x)*(px-p4->x)+(py-p4->y)*(py-p4->y);
			k1 = 1.0/(d1+1e-12);
			k2 = 1.0/(d2+1e-12);
			k3 = 1.0/(d3+1e-12);
			k4 = 1.0/(d4+1e-12);
			v1 = p1->getValue (index);
			v2 = p2->getValue (index);
			v3 = p3->getValue (index);
			v4 = p4->getValue (index);
			if (v1!=GRIB_NOTDEF && v2!=GRIB_NOTDEF && v3!=GRIB_NOTDEF && v4!=GRIB_NOTDEF)
				return (k1*v1 + k2*v2 + k3*v3 + k4*v4)/(k1+k2+k3+k4);
	}
	else if (nb==3) {
			d1 = (px-p1->x)*(px-p1->x)+(py-p1->y)*(py-p1->y);
			d2 = (px-p2->x)*(px-p2->x)+(py-p2->y)*(py-p2->y);
			d3 = (px-p3->x)*(px-p3->x)+(py-p3->y)*(py-p3->y);
			k1 = 1.0/(d1+1e-12);
			k2 = 1.0/(d2+1e-12);
			k3 = 1.0/(d3+1e-12);
			v1 = p1->getValue (index);
			v2 = p2->getValue (index);
			v3 = p3->getValue (index);
			if (v1!=GRIB_NOTDEF && v2!=GRIB_NOTDEF && v3!=GRIB_NOTDEF)
				return (k1*v1 + k2*v2 + k3*v3)/(k1+k2+k3);
	}
	else if (nb==2) {
			d1 = (px-p1->x)*(px-p1->x)+(py-p1->y)*(py-p1->y);
			d2 = (px-p2->x)*(px-p2->x)+(py-p2->y)*(py-p2->y);
			k1 = 1.0/(d1+1e-12);
			k2 = 1.0/(d2+1e-12);
			v1 = p1->getValue (index);
			v2 = p2->getValue (index);
			if (v1!=GRIB_NOTDEF && v2!=GRIB_NOTDEF)
				return (k1*v1 + k2*v2)/(k1+k2);
	}
	return GRIB_NOTDEF;
}

//--------------------------------------------------------------------
double  MblueRecord::getInterpolatedValue (
				DataCode dtc, 
				double px, double py,
				bool interpolate) const
{
	bool fastInterpolation = true;
	if (fastInterpolation)
			return getInterpolatedValueUsingRegularGrid (
						dtc, 
						px, py, interpolate );
	else
			return getInterpolatedValue (
						MbluePoint::getDataIndex (dtc),
						px, py, interpolate );

}

//--------------------------------------------------------------------	
double MblueRecord::getValueOnRegularGrid (DataCode dtc, int i, int j) const
{	
	if (i>=0 && i<Ni && j>=0 && j<Nj) 
	{
		if (dtc.equals (GRB_PRESSURE, LV_MSL, 0)) {
			return getSmoothPressureMSL (i, j);
		}
		else {
			MbluePoint *pt = regularGrid [i+j*Ni];
			return pt->getValue (dtc); 
		}
	}
	else
		return GRIB_NOTDEF;
} 
//--------------------------------------------------------------------	
double MblueRecord::getValueOnRegularGrid (int ind, int i, int j) const
{	
	if (i>=0 && i<Ni && j>=0 && j<Nj) 
	{
		MbluePoint *pt = regularGrid [i+j*Ni];
		return pt->getValue (ind); 
	}
	else {
		return GRIB_NOTDEF;
	}
}  

//--------------------------------------------------------------------	
int     MblueRecord::getNi ()  const
{
	return Ni;
}
int     MblueRecord::getNj ()  const
{
	return Nj;
}
double  MblueRecord::getDeltaX ()  const
{
	return Di;
}
double  MblueRecord::getDeltaY ()  const
{
	return Dj;
}
//--------------------------------------------------------------------	
double  MblueRecord::getX (int i)  const
{
	return xmin + i*Di;
}
double  MblueRecord::getY (int j)  const
{
	return ymin + j*Dj;
}

