/**********************************************************************
zyGrib: meteorological GRIB file viewer
Copyright (C) 2008 - Jacques Zaninetti - http://www.zygrib.org

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
***********************************************************************/

#ifndef UTIL_H
#define UTIL_H

#include <vector>
#include <cmath>
#include <cassert>

#include <QApplication>
#include <QObject>
#include <QString>
#include <QDateTime>
#include <QSettings>
#include <QLocale>
#include <QDir>

class Util : public QObject
{
    Q_OBJECT
    
    public:
        
    static void     setSetting(const QString &key, const QVariant &value);
    static QVariant getSetting(const QString &key, const QVariant &defaultValue);
	static bool     isDirWritable (const QDir &dir);
	
    //-------------------------------------------------
    static QString formatDegres    (float x,bool inf100=false);   // 123.4 -> 123°24.00'
    static QString formatPosition  (float x, float y);    // 123°24.00'W 45°67.89'N
    static QString formatLongitude (float x);
    static QString formatLatitude  (float y);

    static QString formatDateLong  (time_t t);
    static QString formatDateShort (time_t t);
    static QString formatTime      (time_t t);
    static QString formatDateTimeLong  (time_t t);
    static QString formatDateTimeShort (time_t t);
    
    static QString   formatDayName (QDate dt);
    static QDateTime applyTimeZone (time_t t, QString *suffix=NULL);
    
    static QString formatSpeed (float meterspersecond);
    static QString formatDistance (float mille);
    static QString formatAltitude (float meter);
    static QString formatTemperature (float tempKelvin);
    static QString formatTemperature_short (float tempKelvin);
    static QString formatPercentValue (float v);
    static QString formatSnowDepth (float meter);

    static int    kmhToBeaufort   (float v);
    static float  kmhToBeaufort_F (float v);
    static float  BeaufortToKmh_F (float bf);

    static double distancePointSegment (double a,double b,     // point
								  double x0, double y0,  // segment start
								  double x1, double y1   // segment end
								);

    //-------------------------------------------------
    template <typename T>
        static void orderMinMax(T &a, T &b)
                    { if (a>b) {T c; c=a; a=b; b=c;} }
    
    //-------------------------------------------------
    template <typename T>
        static void orderMaxMin(T &a, T &b)
                    { if (a<b) {T c; c=a; a=b; b=c;} }
    
    //-------------------------------------------------
    template <typename T>
        static bool isInRange(T v, T min, T max)
                    { return (v>=min && v<=max); }
    
    //-------------------------------------------------
    template <typename T>
        static T inRange(T v, T min, T max)
                    {
                        if (v<=min) return min;
                        else if (v>=max) return max;
                        else return v;
                    }
    
    //--------------------------------------------------------
    template <typename T>
        static void cleanListPointers( std::list<T *> & ls)
        {
            typename std::list<T *>::iterator it;
            for (it=ls.begin(); it!=ls.end(); it++) {
                delete *it;
                *it = NULL;
            }
            ls.clear();
        }
    //--------------------------------------------------------
	template <typename T>
		static void cleanVectorPointers( std::vector<T *> & ls) {
			typename std::vector<T *>::iterator it;
			for (it=ls.begin(); it!=ls.end(); it++) {
				delete *it;
				*it = NULL;
			}
			ls.clear();
		}

};

//======================================================================
inline int Util::kmhToBeaufort(float v) {
    return (int)(kmhToBeaufort_F(v)+0.5);
}
//-----------------------------------------------------------------------------
inline float Util::kmhToBeaufort_F(float v) {
    float bf = pow( v*v/9.0 , 0.33333);
    if (bf > 12.0)
        bf = 12.0;
    else if (bf < 0.0)
        bf = 0.0;
    return bf;
}
//-----------------------------------------------------------------------------
inline float Util::BeaufortToKmh_F(float bf) {
    float v = sqrt(bf*bf*bf*9.0);
    return v;
}


#endif
