# MailSupport mixin

import string, re, sys, traceback
from string import split,join,find,lower,rfind,atoi,strip
from types import *

import TextFormatter
from Utils import html_unquote,DLOG
from Defaults import AUTO_UPGRADE

class MailSupport:                 
    """
    This mixin class provides subscription and wikimail support.

    Responsibilities: manage a list of subscribers for both this page and
    it's folder, and expose these in the ZMI; also provide wikimail
    utilities and auto-upgrade support.

    A "subscriber" is a string which may be either an email address or a
    CMF member username. A list of these is kept in the page's and/or
    folder's subscriber_list property.

    For the moment, it's still called "email" in arguments to avoid
    breaking legacy dtml (eg subscribeform).
    """
    subscriber_list = []
    _properties=(
        {'id':'subscriber_list', 'type': 'lines', 'mode': 'w'},
        )

    ## private ###########################################################

    def _getSubscribers(self, parent=0):
        """
        Return (a copy of!) this page's subscriber list.
        
        With parent flag, manage the parent folder's subscriber list instead.
        """
        if AUTO_UPGRADE: self._upgradeSubscribers()
        if parent:
            if hasattr(self.folder(),'subscriber_list'):
                return self.folder().subscriber_list[:]
            else:
                return []
        else:
            return self.subscriber_list[:]

    def _setSubscribers(self, subscriberlist, parent=0):
        """
        Set this page's subscriber list. 
        With parent flag, manage the parent folder's subscriber list instead.
        """
        if AUTO_UPGRADE: self._upgradeSubscribers()
        if parent:
            self.folder().subscriber_list = subscriberlist
        else:
            self.subscriber_list = subscriberlist

    def _resetSubscribers(self, parent=0):
        """
        Clear this page's subscriber list.
        With parent flag, manage the parent folder's subscriber list instead.
        """
        self._setSubscribers([],parent)

    def _upgradeSubscribers(self):
        """
        Upgrade old subscriber lists, both this page's and the folder's.

        Called as needed, ie on each access and also from ZWikiPage.upgrade()
        (set AUTO_UPGRADE=0 in Default.py to disable).
        
        XXX Lord have mercy! couldn't this be simpler
        """
        # upgrade the folder first; we'll check attributes then properties
        changed = 0
        f = self.folder().aq_base

        # migrate an old zwiki subscribers attribute
        oldsubs = None
        if (hasattr(f, 'subscribers') and
            type(f.subscribers) is StringType):
            if f.subscribers:
                oldsubs = split(re.sub(r'[ \t]+',r'',f.subscribers),',')
            try:
                del f.subscribers
            except KeyError:
                DLOG('failed to delete self.folder().subscribers')
            changed = 1

        # migrate a wikifornow _subscribers attribute
        oldsubs = None
        if hasattr(f, '_subscribers'):
            oldsubs = f._subscribers.keys()
            try:
                del f._subscribers
            except KeyError:
                DLOG('failed to delete self.folder()._subscribers')
            changed = 1
        if not hasattr(f, 'subscriber_list'):
            f.subscriber_list = []
        # copy old subscribers to subscriber_list, unless it's already
        # got some
        # XXX merge instead
        if oldsubs and not f.subscriber_list:
            f.subscriber_list = oldsubs

        # update _properties
        props = map(lambda x:x['id'], f._properties)
        if 'subscribers' in props:
            f._properties = filter(lambda x:x['id'] != 'subscribers',
                                   f._properties)
            changed = 1
        if not 'subscriber_list' in props:
            f._properties = f._properties + \
                ({'id':'subscriber_list','type':'lines','mode':'w'},)

        if changed:
            DLOG('upgraded %s folder subscriber list' % (f.id))

        # now do the page..
        changed = 0
        self = self.aq_base

        # migrate an old zwiki subscribers attribute
        oldsubs = None
        if (hasattr(self, 'subscribers') and
            type(self.subscribers) is StringType):
            if self.subscribers:
                oldsubs = split(re.sub(r'[ \t]+',r'',self.subscribers),',')
            try:
                del self.subscribers
            except KeyError:
                DLOG('failed to delete %s.subscribers' % (self.id()))
            changed = 1
        # copy old subscribers to subscriber_list, unless it's already
        # got some
        # XXX merge instead
        if oldsubs and not self.subscriber_list:
            self.subscriber_list = oldsubs

        # migrate a wikifornow _subscribers attribute
        oldsubs = None
        if hasattr(self, '_subscribers'):
            oldsubs = self._subscribers.keys()
            try:
                del self._subscribers
            except KeyError:
                DLOG('failed to delete %s._subscribers' % (self.id()))
            changed = 1
        if oldsubs and not self.subscriber_list:
            self.subscriber_list = oldsubs

        # update _properties
        props = map(lambda x:x['id'], self._properties)
        if 'subscribers' in props:
            self._properties = filter(lambda x:x['id'] != 'subscribers',
                                      self._properties)
            changed = 1
        if not 'subscriber_list' in props:
            self._properties = self._properties + \
                ({'id':'subscriber_list','type':'lines','mode':'w'},)

        if changed:
            DLOG('upgraded %s subscriber list' % (self.id()))

    ## page subscription api #############################################

    def subscriberList(self, parent=0):
        """
        Return a list of this page's subscribers
        With parent flag, manage the parent folder's subscriber list instead.
        """
        return self._getSubscribers(parent)

    def subscriberCount(self, parent=0):
        """
        Return the number of subscribers currently subscribed to this page
        With parent flag, count the parent folder's subscriber list
        instead.
        """
        return len(self.subscriberList(parent))

    def isSubscriber(self,email,parent=0):
        """
        Is email currently subscribed to this page ?
        With parent flag, check the parent folder's subscriber list instead.

        Now that email may be an email address or a username,
        we must work a bit harder:

        - for a username, check whether it's subscribed as-is. Don't
        bother looking up it's email address.

        - for an email address, check whether it's subscribed as-is
        (case-insensitively), but also look for member username(s) which
        have that email address and check whether any of those are
        subscribed.
        """
        subscriber = email
        if subscriber:
            email = self.emailAddressFrom(subscriber)
            if email: email = string.lower(email)
            usernames = self.usernamesFrom(subscriber)
            for sub in self.subscriberList(parent):
                if not sub: continue
                if ((email and (string.lower(sub) == email)) or
                    (usernames and (sub in usernames))):
                    return 1
        return 0
               
    def subscribe(self, email, REQUEST=None, parent=0):
        """
        Add email as a subscriber to this page.  With parent flag, add to
        the parent folder's subscriber list instead.
        """
        subscriber = email
        if subscriber:
            if not self.isSubscriber(subscriber,parent):
                subs = self._getSubscribers(parent)
                subs.append(subscriber)
                self._setSubscribers(subs,parent)

        if REQUEST:
            REQUEST.RESPONSE.redirect(
                REQUEST['URL1']+'/subscribeform?email='+subscriber)

    def unsubscribe(self, email, REQUEST=None, parent=0):
        """
        Remove email from this page's subscriber list.  With parent
        flag, remove from the parent folder's subscriber list instead.
        Does not attempt to look up the username from an email address
        or vice-versa, so you must unsubscribe the correct one.
        """
        subscriber = email
        if self.isSubscriber(subscriber,parent):
            sl = self.subscriberList(parent)
            for s in sl:
                if string.lower(s) == string.lower(subscriber):
                    sl.remove(s)
            self._setSubscribers(sl,parent)

        if REQUEST:
            REQUEST.RESPONSE.redirect(
                REQUEST['URL1']+'/subscribeform?email='+subscriber)

    ## folder subscription api ###########################################

    def wikiSubscriberList(self):
        """whole-wiki version of subscriberList"""
        return self.subscriberList(parent=1)

    def wikiSubscriberCount(self):
        """whole-wiki version of subscriberCount"""
        return self.subscriberCount(parent=1)

    def isWikiSubscriber(self,email):
        """whole-wiki version of isSubscriber"""
        return self.isSubscriber(email,parent=1)

    def wikiSubscribe(self, email, REQUEST=None):
        """whole-wiki version of subscribe"""
        return self.subscribe(email,REQUEST,parent=1)

    def wikiUnsubscribe(self, email, REQUEST=None):
        """whole-wiki version of unsubscribe"""
        return self.unsubscribe(email,REQUEST,parent=1)

    ## misc api methods ##################################################

    def allSubscribers(self):
        """
        List all subscribers to this page or the wiki.
        """
        subs = []
        subs.extend(self.subscriberList()) # copy, don't reference
        for s in self.wikiSubscriberList():
            if not (s in subs): subs.append(s)
        return subs

    def allSubscriptionsFor(self, email):
        """
        Return the ids of all pages to which a subscriber is subscribed
        ('whole_wiki' indicates a wiki subscription).
        """
        subscriber = email
        # simple-minded implementation
        subscriptions = []
        if self.isWikiSubscriber(subscriber):
            subscriptions.append('whole_wiki')
        # use catalog ?
        for id, page in self.folder().objectItems(spec='ZWiki Page'):
            if page.isSubscriber(subscriber):
                subscriptions.append(id)
        return subscriptions

    def otherPageSubscriptionsFor(self, email):
        """
        Ack, this was too hard in DTML. Return the ids of all pages to
        which a subscriber is subscribed, excluding the current page and
        'whole_wiki'.
        """
        subscriber = email
        subs = self.allSubscriptionsFor(subscriber)
        thispage = self.id()
        if thispage in subs: subs.remove(thispage)
        if 'whole_wiki' in subs: subs.remove('whole_wiki')
        return subs

    ## utilities ################################################

    def isMailoutEnabled(self):
        """
        True if mailout has been configured
        """
        if (hasattr(self,'MailHost') and
            (getattr(self.folder(),'mail_from',None) or
             getattr(self.folder(),'mail_replyto',None))):
            return 1
        else:
            return 0

    def isEmailAddress(self,s):
        """
        True if s looks like an email address.
        """
        if '@' in s: return 1
        else: return 0
    
    def isUsername(self,s):
        """
        True if s looks like a username.
        """
        return not self.isEmailAddress(s)
    
    def emailAddressFrom(self,subscriber):
        """
        Convert subscriber to an email address, if needed.

        If subscriber is an email address, return as-is.  Otherwise assume
        it's a username and try to look up the corresponding CMF member's
        email address.  Otherwise return None.
        """
        if self.isEmailAddress(subscriber):
            return subscriber
        else:
            folder = self.folder()
            try:
                user = folder.portal_membership.getMemberById(subscriber)
                member = folder.portal_memberdata.wrapUser(user)
                return member.email
            except AttributeError:
                return None

    def emailAddressesFrom(self,subscribers):
        """
        Convert a list of subscribers to a list of email addresses.
        """
        emails = []
        for s in subscribers:
            e = self.emailAddressFrom(s)
            if e: emails.append(e)
        return emails

    def usernamesFrom(self,subscriber):
        """
        Convert subscriber to username(s) if needed and return as a list.

        Ie if subscriber is a username, return that username; if
        subscriber is an email address, return the usernames of any CMF
        members with that email address.
        """
        if self.isUsername(subscriber):
            return [subscriber]
        else:
            email = string.lower(subscriber)
            usernames = []
            folder = self.folder()
            try:
                for user in folder.portal_membership.listMembers():
                    member = folder.portal_memberdata.wrapUser(user)
                    if string.lower(member.email) == email:
                        usernames.append(member.name)
            except AttributeError:
                pass
            return usernames

    def sendMailToSubscribers(self, text, REQUEST, subjectSuffix='',
                              subject=''):
        """
        Send mail to page subscribers.
        
        If a mailhost and mail_from property have been configured and
        there are subscribers to this page, email text to them.  So as not
        to prevent page edits, catch any mail-sending errors (and log and
        try to forward them to an admin).
        """
        recipients = self.emailAddressesFrom(self.allSubscribers())
        if not recipients: return
        try:
            self.sendMailTo(recipients,text,REQUEST,subjectSuffix,subject)
        except:
            type, val, tb = sys.exc_info()
            err = string.join(traceback.format_exception(type,val,tb),'')
            DLOG('failed to send mail to %s: %s' % (recipients,err))
            admin = getattr(self.folder(),'mail_admin',None)
            if admin:
                try: self.sendMailTo(
                    [admin],text or '(no text)',REQUEST,
                    subjectSuffix='ERROR, subscriber mailout failed')
                except:
                    type, val, tb = sys.exc_info()
                    err = string.join(
                        traceback.format_exception(type,val,tb),'')
                    DLOG('failed to send error report to admin: %s' % err)

    def sendMailTo(self, recipients, text, REQUEST, subjectSuffix='',
                   subject=''):
        """
        Send mail to the specified recipients.

        If a mailhost and mail_from property have been configured,
        attempt to email text to recipients. 
        """
        if not self.isMailoutEnabled(): return

        # ugLy temp hack
        # strip out the message heading typically prepended on *Discussion
        # pages
        mailouttext = self.formatMailout(text)

        # gather bits and pieces
        mhost=self.MailHost
        username = self.zwiki_username_or_ip(REQUEST)
        mail_from = getattr(self.folder(),'mail_from','')
        mail_replyto = getattr(self.folder(),'mail_replyto','')
        replytohdr = mail_replyto or mail_from
        listid = mail_from
        # primary recipient, alternatives:
        # 1. "To: ;" causes messy cc header in replies, while
        # 2. "To: replytohdr" sends a copy back to the wiki which
        # may be the cause of intermittent slow comments
        # XXX for now use 2 & allow it to be overridden 
        tohdr = getattr(self.folder(),'mail_to','') or replytohdr
        if mail_from:
            if (not re.match(r'[0-9\.\s]*$',username) and
                not self.isEmailAddress(username)):
                fromhdr = '%s (%s)' % (mail_from,username)
            else:
                fromhdr = mail_from
        else:
            if self.isEmailAddress(username):
                fromhdr = username
            elif re.match(r'[0-9\.\s]*$',username):
                fromhdr = mail_replyto
            else:
                fromhdr = '%s (%s)' % (mail_replyto,username)
        pageurl = self.page_url()
        if getattr(self.folder(),'mail_page_name',1):
            pagename = '[%s] ' % self.title_or_id()
        else:
            pagename = ''
        signature = getattr(self.folder(),
                            'mail_signature',
                            '--\nforwarded from %s' % pageurl)
        # if using mailin.py's checkrecipient option you can uncomment
        # this line to set the page name as Reply-to's real name:
        #replytohdr = replytohdr + ' (%s)'%self.title_or_id()

        # send message - XXX templatize this
        msg = """\
From: %s
Reply-To: %s
To: %s
Bcc: %s
Subject: %s
X-BeenThere: %s
X-Zwiki-Version: %s
Precedence: bulk
List-Id: %s <%s>
List-Post: <mailto:%s>
List-Subscribe: <%s/subscribeform>
List-Unsubscribe: <%s/subscribeform>
List-Archive: <%s>
List-Help: <%s>

%s
%s
""" \
        % (fromhdr,
           replytohdr,
           tohdr,
           join(filter(lambda x:strip(x)!='', recipients), ', '),
           join([strip(getattr(self.folder(),'mail_subject_prefix',
                               '')), #getattr(self.folder(),'title'))),
                 #strip(self.id()),
                 pagename,
                 subject,
                 strip(subjectSuffix)],''),
           listid,
           self.zwiki_version(),
           self.folder().title, listid,
           listid,
           pageurl,
           pageurl,
           pageurl,
           self.wiki_url(),
           mailouttext,
           signature,
           )

        # testing support - messages with subject [test] or originating
        # from TestPage are not mailed out to subscribers
        if (subject == '[test]' or self.title_or_id() == 'TestPage'):
            # log and drop it
            DLOG('discarding test mailout:\n%s' % msg)
            # I tried redirecting to a test SMTP server but it blocked
            #DLOG('diverting test mailout to test server:\n%s' % msg)
            #try:
            #    self.TestMailHost.send(msg)
            #    DLOG('sent mailout to test server')
            #    DLOG('TestMailHost info:',
            #         self.TestMailHost.smtp_host,
            #         self.TestMailHost.smtp_port)
            #except:
            #    type, val, tb = sys.exc_info()
            #    err = string.join(
            #        traceback.format_exception(type,val,tb),'')
            #    DLOG('failed to send mailout to test server:',
            #         err,
            #         self.TestMailHost.smtp_host,
            #         self.TestMailHost.smtp_port)
            # instead, I sent as usual and hacked mailman to drop it
            # do add a X-No-Archive header
            #msg = re.sub(r'(?m)(List-Help:.*$)',
            #             r'\1\nX-No-Archive: yes',
            #             msg)
            #DLOG('sending mailout:\n%s' % msg)
            #mhost.send(msg)
        else:
            DLOG('sending mailout:\n%s' % msg)
            mhost.send(msg)

    def formatMailout(self, text):
        """
        Format some text (usually a page diff) for email delivery.

        See testCommentFormatting/test_formatMailout for what's really
        going on here.
        """
        if not text: return ''
        
        # strip zwiki comment heading and citation formatting, if present
        text = re.sub(r'(?i)(<img[^>]*?>)?<b>.*?<br>\n',r'',text)
        text = re.sub(r'(?sm)^<br>><i>(.*?)</i>',r'>\1',text)

        # wrap (but don't fill or pad) long lines
        # ideally would like to fill each paragraph like emacs' fill-paragraph
        formatter = TextFormatter.TextFormatter((
            {'width':78, 'fill':0, 'pad':0},
            ))
        # work around textformatter foibles
        savenl = ''
        if text[0] == '\n' and text[1] != '\n':
            savenl = '\n'; text = text[1:]
        text=formatter.compose((text,))
        text = savenl + text
        # textformatter chops off a trailing newline.. since textDiff adds
        # one, and we are usually formatting its output, just leave it be

        # strip leading newlines
        text = re.sub(r'(?s)^\n+',r'',text)
        # strip trailing newlines
        text = re.sub(r'(?s)\n+$',r'\n',text)
        # lose any html quoting
        text = html_unquote(text)
        return text

