#
# Generic base test case for working with CMF-style portals
#
# The fixture consists of:
#
#   - a portal object (self.portal)
#   - a user folder inside the portal
#   - a user with role 'Member' inside the user folder
#   - the user's memberarea (self.folder)
#
# And the end of the setup process the user is logged in.
#
# The twist is that the portal object itself is *not* created
# by the PortalTestCase class. The user of this base class is
# responsible for providing a portal object to the machinery.
#
# You must create a portal object before starting the tests
# and/or override getPortal() to return the object serving as
# the "portal".
#

# $Id: PortalTestCase.py,v 1.19 2003/11/28 23:12:09 shh42 Exp $

import ZopeTestCase

import unittest

from AccessControl import getSecurityManager
from AccessControl.SecurityManagement import newSecurityManager
from AccessControl.SecurityManagement import noSecurityManager
from Acquisition import aq_base

portal_name = 'portal'
user_name   = ZopeTestCase.user_name


class PortalTestCase(unittest.TestCase):
    '''Base test case for testing CMF-style portals

       __implements__ = (IPortalTestCase, ISimpleSecurity, IExtensibleSecurity)

       See doc/IZopeTestCase.py for more.
    '''

    _configure_portal = 1

    def getPortal(self):
        '''Returns the portal object, i.e. the "fixture root".
           Override if you don't like the default.
        '''
        return self.app[portal_name]

    def afterSetUp(self):
        '''Called after setUp() has completed. This is
           far and away the most useful hook.
        '''
        pass

    def beforeTearDown(self):
        '''Called before tearDown() is executed.
           Note that tearDown() is not called if
           setUp() fails.
        '''
        pass

    def afterClear(self):
        '''Called after the fixture has been cleared.
           Note that this is done during setUp() *and*
           tearDown().
        '''
        pass

    def beforeSetUp(self):
        '''Called before the ZODB connection is opened,
           at the start of setUp(). By default begins
           a new transaction.
        '''
        get_transaction().begin()

    def beforeClose(self):
        '''Called before the ZODB connection is closed,
           at the end of tearDown(). By default aborts
           the transaction.
        '''
        get_transaction().abort()

    def setUp(self):
        '''Sets up the fixture. Do not override,
           use the hooks instead.
        '''
        self._clear()
        self.beforeSetUp()
        self.app = self._app()
        self.portal = self.getPortal()
        if self._configure_portal:
            self._setupUserFolder()
            self._setupUser()
            self.login()
            self._setupHomeFolder()
        self._refreshSkinData()
        self.afterSetUp()

    def tearDown(self):
        '''Tears down the fixture. Do not override,
           use the hooks instead.
        '''
        self.beforeTearDown()
        self._clear(1)

    def _app(self):
        '''Returns the app object for a test.'''
        return ZopeTestCase.app()

    def _setupUserFolder(self):
        '''Creates the user folder if none is provided.'''
        if not hasattr(aq_base(self.portal), 'acl_users'):
            self.portal.manage_addUserFolder()

    def _setupUser(self):
        '''Creates the portal user.'''
        uf = self.portal.acl_users
        uf._doAddUser(user_name, 'secret', ['Member'], [])

    def _setupHomeFolder(self):
        '''Creates the portal user's home folder.'''
        self.createMemberarea(user_name)
        pm = self.portal.portal_membership
        self.folder = pm.getHomeFolder(user_name)

    def createMemberarea(self, member_id):
        '''Creates a user's memberarea. May be overridden.'''
        pm = self.portal.portal_membership
        pm.createMemberarea(member_id)

    def _refreshSkinData(self):
        '''Refreshes the magic _v_skindata attribute.'''
        if hasattr(self.portal, '_v_skindata'):
            self.portal._v_skindata = None
        if hasattr(self.portal, 'setupCurrentSkin'):
            self.portal.setupCurrentSkin()

    def _clear(self, call_close_hook=0):
        '''Clears the fixture.'''
        # No more cleanups here. We rely
        # on transaction abort.
        if call_close_hook:
            self.beforeClose()
        self._close()
        self.logout()
        self.afterClear()

    def _close(self):
        '''Closes the ZODB connection.'''
        get_transaction().abort()
        ZopeTestCase.closeConnections()

    # Security interfaces

    def setRoles(self, roles, name=user_name):
        '''Changes the user's roles.'''
        uf = self.portal.acl_users
        uf._doChangeUser(name, None, roles, [])
        if name == getSecurityManager().getUser().getId():
            self.login(name)

    def setPermissions(self, permissions, role='Member'):
        '''Changes the user's permissions.'''
        self.portal.manage_role(role, permissions)

    def login(self, name=user_name):
        '''Logs in.'''
        uf = self.portal.acl_users
        user = uf.getUserById(name)
        if not hasattr(user, 'aq_base'):
            user = user.__of__(uf)
        newSecurityManager(None, user)

    def logout(self):
        '''Logs out.'''
        noSecurityManager()


# b/w compatibility names
_portal_name = portal_name

