#
# Tests the ExtFile product
#

import os, sys
if __name__ == '__main__':
    execfile(os.path.join(sys.path[0], 'framework.py'))

from Testing import ZopeTestCase

ZopeTestCase.installProduct('ExtFile')


standard_perms = ZopeTestCase.standard_permissions
copymove_perms = ['View management screens', 'Add ExtImages', 'Add ExtFiles', 'Delete objects']
access_perms = ['View management screens']
change_perms = ['Change ExtFile/ExtImage']

from cgi import FieldStorage
from ZPublisher.HTTPRequest import FileUpload

from TestBase import TestBase
from TestBase import gifImage, jpegImage



class TestExtFileAdd(TestBase):
    '''Test ExtFile/ExtImage creation and content-type detection'''

    def testAddImage1(self):
        '''Add Image from file name'''
        self.addExtImage(id='image', file=gifImage)
        assert self.image.get_size() == self._fsize(gifImage)

    def testAddImage2(self):
        '''Add Image from file handle'''
        self.addExtImage(id='image', file=open(gifImage, 'rb'))
        assert self.image.get_size() == self._fsize(gifImage)

    def testAddImage3(self):
        '''Add Image from file upload'''
        environ = {'REQUEST_METHOD': 'POST', 'CONTENT_TYPE': 'image/gif'}
        file = FileUpload(FieldStorage(fp=open(gifImage, 'rb'), environ=environ))
        self.addExtImage(id='image', file=file)
        assert self.image.get_size() == self._fsize(gifImage)

    def testContentType1(self):
        '''Detect GIF content type from file extension'''
        self.addExtFile(id='image.gif', file=gifImage)
        assert self.image.content_type == 'image/gif', 'Content type: ' + self.image.content_type
            
    def testContentType2(self):
        '''Detect GIF content type from file upload headers'''
        environ = {'REQUEST_METHOD': 'POST', 'CONTENT_TYPE': 'image/gif'}
        file = FileUpload(FieldStorage(fp=open(gifImage, 'rb'), environ=environ))
        self.addExtFile(id='image', file=file)
        assert self.image.content_type == 'image/gif', 'Content type: ' + self.image.content_type

    def testContentType3(self):
        '''Detect GIF content type from file contents'''
        self.addExtImage(id='image', file=gifImage)
        assert self.image.content_type == 'image/gif', 'Content type: ' + self.image.content_type
            
    def testContentType4(self):
        '''Detect GIF content type from string buffer'''
        self.addExtImage(id='image', file=jpegImage)
        self.image.manage_upload(file=open(gifImage, 'rb').read())
        assert self.image.content_type == 'image/gif', 'Content type: ' + self.image.content_type
        # check manage_upload works
        assert self.image.get_size() == self._fsize(gifImage)

    def testContentType5(self):
        '''Detect JPEG content type from file contents'''
        self.addExtImage(id='image', file=jpegImage)
        assert self.image.content_type == 'image/jpeg', 'Content type: ' + self.image.content_type

    def testContentType6(self):
        '''Detect JPEG content type from PUT' image'''
        self.addExtImage(id='image', file=open(gifImage, 'rb'))
        request = self.app.REQUEST
        request['BODYFILE'] = open(jpegImage, 'rb')
        self.image.PUT(request, request.RESPONSE)
        assert self.image.content_type == 'image/jpeg'
        # check PUT works
        assert self.image.get_size() == self._fsize(jpegImage)



class TestExtFileRepository(TestBase):
    '''Test file repository handling'''

    def afterSetUp(self):
        TestBase.afterSetUp(self)
        self.folder.manage_addFolder('subfolder')
        self.subfolder = self.folder['subfolder']

    def testName1(self):
        '''Create a unique file name'''
        self.addExtImage(id='image', file=gifImage)
        dispatcher = self.subfolder.manage_addProduct['ExtFile']
        dispatcher.manage_addExtImage(id='image', file=gifImage)
        self.failUnless(self._exists('image.gif'))
        self.failIf(self._exists('image.gif.undo'))
        self.failUnless(self._exists('image1.gif'))
        self.failIf(self._exists('image1.gif.undo'))
         
    def testName2(self):
        '''Create a unique preview name'''
        self.addExtImage(id='image', file=jpegImage)
        self.image.manage_create_prev(100, 100, ratio=1)
        self.failUnless(self._exists('image.jpg'))
        self.failIf(self._exists('image.jpg.undo'))
        self.failUnless(self._exists('image1.jpg'))
        self.failIf(self._exists('image1.jpg.undo'))
         
    def testUndoName1(self):
        '''Create a .undo file'''
        self.addExtImage(id='image', file=gifImage)
        self.folder.manage_delObjects(['image'])
        self.failIf(self._exists('image.gif'))
        self.failUnless(self._exists('image.gif.undo'))
         
    def testUndoName2(self):
        '''Do not create a .undo file'''
        self.addExtImage(id='image', file=gifImage)
        self.image.manage_file_upload(gifImage)
        self.failUnless(self._exists('image.gif'))
        self.failIf(self._exists('image.gif.undo'))
        self.failIf(self._exists('image1.gif'))



class TestExtFileCopyPaste(TestBase):
    '''Tests cut/copy/paste/rename/clone'''

    def afterSetUp(self):
        TestBase.afterSetUp(self)
        self.folder.manage_addFolder('subfolder')
        self.subfolder = self.folder['subfolder']
        self.setPermissions(standard_perms+copymove_perms)
        self.addExtFile(id='image.gif', file=gifImage)
        get_transaction().commit(1) # need a _p_jar

    def testClone(self):
        '''Clone a file'''
        #print self._listdir()
        self.subfolder.manage_clone(self.image, 'image.gif')
        #print self._listdir()
        self.failUnless(self._exists('image.gif'))
        self.failIf(self._exists('image.gif.undo'))
        # This only works in Zope 2.6+
        self.failUnless(self._exists('image1.gif'))
        #print self._listdir()
        #print self.image.get_filename()
        #print self.subfolder.image.get_filename()

    def testCopyPaste(self):
        '''Copy and paste a file'''
        cb = self.folder.manage_copyObjects(['image.gif'])
        self.subfolder.manage_pasteObjects(cb)
        self.failUnless(self._exists('image.gif'))
        self.failIf(self._exists('image.gif.undo'))
        self.failUnless(self._exists('image1.gif'))
        assert self.subfolder['image.gif'].get_size() == self._fsize(gifImage)
         
    def testCutPaste(self):
        '''Cut and paste a file'''
        cb = self.folder.manage_cutObjects(['image.gif'])
        self.subfolder.manage_pasteObjects(cb)
        self.failUnless(self._exists('image.gif'))
        self.failIf(self._exists('image.gif.undo'))
        self.failIf(self._exists('image1.gif'))

    def testRename(self):
        '''Rename a file'''
        self.folder.manage_renameObject('image.gif', 'image44.gif')
        self.failUnless(self._exists('image.gif'))
        self.failIf(self._exists('image.gif.undo'))
        self.failIf(self._exists('image1.gif'))

    def testCOPY(self):
        '''Webdav copy a file'''
        req = self.app.REQUEST
        req.environ['HTTP_DEPTH'] = 'infinity'
        req.environ['HTTP_DESTINATION'] = 'http://dummy.org/%s/subfolder/image.gif' %ZopeTestCase._folder_name
        self.folder['image.gif'].COPY(req, req.RESPONSE)
        self.failUnless(self._exists('image.gif'))
        self.failIf(self._exists('image.gif.undo'))
        self.failUnless(self._exists('image1.gif'))
        assert self.subfolder['image.gif'].get_size() == self._fsize(gifImage)



class TestExtImageCopyPaste(TestBase):
    '''Tests cut/copy/paste/rename/clone'''

    def afterSetUp(self):
        TestBase.afterSetUp(self)
        self.folder.manage_addFolder('subfolder')
        self.subfolder = self.folder['subfolder']
        self.setPermissions(standard_perms+copymove_perms)
        self.addExtImage(id='image.gif', file=gifImage)
        # Add a preview image as well
        self.folder['image.gif'].manage_file_upload(file=gifImage, is_preview=1)
        get_transaction().commit(1) # need a _p_jar

    def testClone(self):
        '''Clone an image'''
        self.subfolder.manage_clone(self.image, 'image.gif')
        self.failUnless(self._exists('image.gif'))
        self.failIf(self._exists('image.gif.undo'))
        self.failUnless(self._exists('image1.gif'))
        self.failIf(self._exists('image1.gif.undo'))
        # This only works in Zope 2.6+
        self.failUnless(self._exists('image2.gif'))
        self.failUnless(self._exists('image3.gif'))

    def testCopyPaste(self):
        '''Copy and paste an image'''
        cb = self.folder.manage_copyObjects(['image.gif'])
        self.subfolder.manage_pasteObjects(cb)
        self.failUnless(self._exists('image.gif'))
        self.failIf(self._exists('image.gif.undo'))
        self.failUnless(self._exists('image1.gif'))
        self.failIf(self._exists('image1.gif.undo'))
        self.failUnless(self._exists('image2.gif'))
        self.failUnless(self._exists('image3.gif'))
        assert self.subfolder['image.gif'].get_size() == self._fsize(gifImage)
         
    def testCutPaste(self):
        '''Cut and paste an image'''
        cb = self.folder.manage_cutObjects(['image.gif'])
        self.subfolder.manage_pasteObjects(cb)
        self.failUnless(self._exists('image.gif'))
        self.failIf(self._exists('image.gif.undo'))
        self.failUnless(self._exists('image1.gif'))
        self.failIf(self._exists('image1.gif.undo'))
        self.failIf(self._exists('image2.gif'))
        self.failIf(self._exists('image3.gif'))

    def testRename(self):
        '''Rename an image'''
        self.folder.manage_renameObject('image.gif', 'image44.gif')
        self.failUnless(self._exists('image.gif'))
        self.failIf(self._exists('image.gif.undo'))
        self.failUnless(self._exists('image1.gif'))
        self.failIf(self._exists('image1.gif.undo'))
        self.failIf(self._exists('image2.gif'))
        self.failIf(self._exists('image3.gif'))

    def testCOPY(self):
        '''Webdav copy an image'''
        req = self.app.REQUEST
        req.environ['HTTP_DEPTH'] = 'infinity'
        req.environ['HTTP_DESTINATION'] = 'http://dummy.org/%s/subfolder/image.gif' %ZopeTestCase._folder_name
        self.folder['image.gif'].COPY(req, req.RESPONSE)
        self.failUnless(self._exists('image.gif'))
        self.failIf(self._exists('image.gif.undo'))
        self.failUnless(self._exists('image1.gif'))
        self.failIf(self._exists('image1.gif.undo'))
        self.failUnless(self._exists('image2.gif'))
        self.failUnless(self._exists('image3.gif'))
        assert self.subfolder['image.gif'].get_size() == self._fsize(gifImage)



if __name__ == '__main__':
    framework()
else:
    import unittest
    def test_suite():
        suite = unittest.TestSuite()
        suite.addTest(unittest.makeSuite(TestExtFileAdd))
        suite.addTest(unittest.makeSuite(TestExtFileRepository))
        suite.addTest(unittest.makeSuite(TestExtFileCopyPaste))
        suite.addTest(unittest.makeSuite(TestExtImageCopyPaste))
        return suite

