#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import <Renaissance/Renaissance.h>
#import "AppController.h"
#import "Archive.h"
#import "ZipArchive.h"
#import "TarArchive.h"
#import "RarArchive.h"
#import "LhaArchive.h"
#import "FileInfo.h"
#import "PreferencesController.h"
#import "Preferences.h"
#import "ArchiveService.h"
#import "NSFileManager+Custom.h"
#import "TableViewDataSource.h"
#import "ZipperCell.h"

static NSDictionary *_fileExtMappings = nil;

@interface AppController (PrivateAPI)
- (BOOL)openArchiveWithPath:(NSString *)path;
- (void)addRatioColumn;
- (void)removeRatioColumn;
- (void)extractIncludingPathInfo:(BOOL)includePathInfo;
- (void)setArchive:(Archive *)archive;
- (void)openFile:(NSString *)file withDefaultApp:(NSString *)defaultApp;
@end

@implementation AppController : NSObject

+ (void)initialize
{
	if (_fileExtMappings == nil)
	{
		_fileExtMappings = [[NSDictionary dictionaryWithObjectsAndKeys:
			[ZipArchive class], @"zip",
			[ZipArchive class], @"jar",
			[TarArchive class], @"tar",
			[TarArchive class], @"gz",
			[TarArchive class], @"tgz",
			[TarArchive class], @"bz2",
			[RarArchive class], @"rar",
			[LhaArchive class], @"lha",
			nil] retain];
	}
}

- (id)init
{
	[super init];
	_filename = nil;
	_tableViewDataSource = nil;
	return self;
}

- (void)dealloc
{
	[_archive release];
	[_filename release];
	[super dealloc];
}

//------------------------------------------------------------------------------
// NSApp delegate methods
//------------------------------------------------------------------------------
- (void)applicationDidFinishLaunching:(NSNotification *)note
{	
	NSSize size;
	NSTableColumn *tableColumn;
	NSCell *cell = [[ZipperCell alloc] init];

	// register our service provider
	[NSApp setServicesProvider:[[ArchiveService alloc] init]];
	// TODO remove me!
	NSUpdateDynamicServices();

    [NSBundle loadGSMarkupNamed:@"Zipper" owner:self];
	
	// allow for a little bit more size between the columns
	size = [_tableView intercellSpacing];
	size.width += 5.0;
	[_tableView setIntercellSpacing:size];
	
	// right-align the size column
	tableColumn = [_tableView tableColumnWithIdentifier:COL_ID_SIZE];
	[[tableColumn dataCell] setAlignment:NSRightTextAlignment];
	// set the ZipperCell for the name column
	tableColumn = [_tableView tableColumnWithIdentifier:COL_ID_NAME];
	[tableColumn setDataCell: cell];
	[cell release];

	// when double-clicking an archive the openFile: methods will run
	// before this method. Since at this point in time the app isn't fully initialized
	// yet, a flag is set and it's our responsibility here to re-call the open method
	if (_filename != nil)
	{
		[self openArchiveWithPath:_filename];
		ASSIGN(_filename, nil);
	}
}

- (BOOL)openArchiveWithPath:(NSString *)path
{
	if (_tableView == nil)
	{
		// The app is not completely loaded yet. Defer opening the file until it is up and running
		ASSIGN(_filename, path);
	}
	else
	{
		NSString *extension = [[path pathExtension] lowercaseString];
	
		if ([extension isEqual:@""] == NO)
		{
			id archiveClass;
	
			archiveClass = [_fileExtMappings objectForKey:extension];
			NSAssert(archiveClass != nil, @"No archive class registered. This should not happen");
			if ([archiveClass hasRatio])
			{
				[self addRatioColumn];
			}
			else
			{
				[self removeRatioColumn];
			}
			
			if ([archiveClass executableDoesExist] == NO)
			{
				// ouch, no executable set (yet)
				NSString *message;
				PreferencesController *prefsController;
		
				message = [NSString stringWithFormat:
					@"Missing unarchiver for file extension %@. Please provide a value", extension];
				NSRunAlertPanel(@"Error in Preferences", message, nil, nil, nil);
				
				// bring up the prefs panel
				prefsController = [[PreferencesController alloc] init];
				[prefsController setArchiveForRequiredType:archiveClass];
				[prefsController showPreferencesPanel];
			}
			[self setArchive:[archiveClass newWithPath:path]];
			[_archive sortByFilename];
			[_tableView reloadData];
		}		
		return YES;
	}
	return NO;
}

/*" this delegate method will be called before we actually display our GUI "*/
- (BOOL)application:(NSApplication *)sender openFile:(NSString *)filename
{
	return [self openArchiveWithPath:filename];
}

/*" this delegate method will be called before we actually display our GUI "*/
- (BOOL)application:(NSApplication *)sender openTempFile:(NSString *)filename
{
	return [self openArchiveWithPath:filename];
}

/**
 * do cleanup, especially remove temporary files that were created while we ran
 */
-(void)applicationWillTerminate:(NSNotification *)aNotification
{
	NSEnumerator *cursor;
	NSString *element;
	
	// clean up all temporary Zipper directories
	cursor = [[[NSFileManager defaultManager] directoryContentsAtPath:NSTemporaryDirectory()]
		objectEnumerator];
	while ((element = [cursor nextObject]) != nil)
	{
		if ([element hasPrefix:@"Zipper"])
		{
			NSString *path;
			
			path = [NSString pathWithComponents:[NSArray arrayWithObjects:NSTemporaryDirectory(),
				element, nil]];
			[[NSFileManager defaultManager] removeFileAtPath:path handler:nil];
		}
	}
}
 
//-----------------------------------------------------------------------------
// validating the UI
//-----------------------------------------------------------------------------
- (BOOL)validateMenuItem:(NSMenuItem *)anItem
{
	SEL	action = [anItem action];
	
	if (sel_eq(action, @selector(openFile:)) ||	sel_eq(action, @selector(showPreferences:))) 
	{
		return YES;
	}
	
	// enable the extract menu item only if we have something to extract
	if (_archive == nil)
	{
		return NO;
	}
	// do not enable the 'extract flat' menu item if the extractor does not support it
	if (([anItem tag] == 2) && ([[_archive class] canExtractWithoutFullPath] == NO))
	{
		return NO;
	}
	return YES;
}

//------------------------------------------------------------------------------
// action methods
//------------------------------------------------------------------------------
- (void)extract:(id)sender
{
	[self extractIncludingPathInfo:YES];
}

- (void)extractFlat:(id)sender
{
	[self extractIncludingPathInfo:NO];
}

- (void)extractIncludingPathInfo:(BOOL)includePathInfo
{
	NSOpenPanel *openPanel;
	int rc;

	if (_archive == nil)
	{
		return;
	}

	openPanel = [NSOpenPanel openPanel];
	[openPanel setTitle:@"Extract to"];
	[openPanel setCanChooseFiles:NO];
	[openPanel setCanChooseDirectories:YES];
	
	rc = [openPanel runModalForDirectory:[Preferences lastExtractDirectory] file:nil types:nil];
	if (rc == NSOKButton)
	{
		NSString *message;
		NSString *path = [openPanel directory];

		if ([_tableView selectedRow] == -1)
		{
			// no rows selected ... extract the whole archive
			[_archive expandFiles:nil withPathInfo:includePathInfo toPath:path];
		}
		else
		{
			NSNumber *rowIndex;

			// retrieve selected rows && extract them
			NSMutableArray *extractFiles = [NSMutableArray array];
			NSEnumerator *rows = [_tableView selectedRowEnumerator];
			while ((rowIndex = [rows nextObject]) != nil)
			{
				[extractFiles addObject:[_archive elementAtIndex:[rowIndex intValue]]]; 
			}
			[_archive expandFiles:extractFiles withPathInfo:includePathInfo toPath:path];
		}
		
		// save the selected directory for later
		[Preferences setLastExtractDirectory:path];

		message = [NSString stringWithFormat:@"Successfully expanded to %@", path];
		NSRunAlertPanel(@"Expand", message, nil, nil, nil);
	}
}

- (void)openFile:(id)sender
{
	int rc;
	
	NSOpenPanel *openPanel = [NSOpenPanel openPanel];
	[openPanel setTitle:@"Open archive"];
	[openPanel setCanChooseFiles:YES];
	[openPanel setCanChooseDirectories:NO];
	
	rc = [openPanel runModalForDirectory:[Preferences lastOpenDirectory]
		file:nil types:[_fileExtMappings allKeys]];
	if (rc == NSOKButton)
	{
		NSString *filename = [openPanel filename];
	
		[self application:NSApp openFile:filename];
		
		// save the directory for later reuse
		[Preferences setLastOpenDirectory:[filename stringByDeletingLastPathComponent]];
		
		// reset any rows that might still be selected from a previous run and reload data
		[_tableView deselectAll:self];
		[_tableView reloadData];
	}
}

- (void)showPreferences:(id)sender
{
	[[[PreferencesController alloc] init] showPreferencesPanel];
}

- (void)view:(id)sender
{
	NSEnumerator *enumerator;
	NSMutableArray *filesToExtract;
	NSNumber *row;
	NSString *tempDir;
	FileInfo *info;
	
	// collect all files to extract
	filesToExtract = [NSMutableArray array];
	enumerator = [_tableView selectedRowEnumerator];
	while ((row = [enumerator nextObject]) != nil)
	{
		[filesToExtract addObject:[_archive elementAtIndex:[row intValue]]];
	}
	
	tempDir = [[NSFileManager defaultManager] createTemporaryDirectory];
	[_archive expandFiles:filesToExtract withPathInfo:YES toPath:tempDir];
	// NSWorkspace hopefully knows how to handle the file
	enumerator = [filesToExtract objectEnumerator];
	while ((info = [enumerator nextObject]) != nil)
	{
		NSString *tempFile;
		
		tempFile = [NSString pathWithComponents:[NSArray arrayWithObjects:tempDir, 
			[info fullPath], nil]];
		[self openFile:tempFile withDefaultApp:[Preferences defaultOpenApp]];
	}
}

//------------------------------------------------------------------------------
// private API
//------------------------------------------------------------------------------
- (void)addRatioColumn
{
	NSTableColumn *ratioColumn;
	int colIndex;
		
	// do it only if the tableView was already loaded
	if (_tableView != nil)
	{	
		ratioColumn = [_tableView tableColumnWithIdentifier:COL_ID_RATIO];
		if (ratioColumn == nil)
		{
			ratioColumn = [[[NSTableColumn alloc] initWithIdentifier:COL_ID_RATIO] autorelease];
			[[ratioColumn headerCell] setStringValue:@"Ratio"];
			[ratioColumn setWidth:50.0];
			[[ratioColumn dataCell] setAlignment:NSRightTextAlignment];
			[_tableView addTableColumn:ratioColumn];
		}
		
		// here we have a ratio column for sure
		colIndex = [_tableView columnWithIdentifier:COL_ID_RATIO];
		[_tableView moveColumn:colIndex toColumn:3];

		[_tableView reloadData];
	}
}

- (void)removeRatioColumn;
{	
	if (_tableView != nil)
	{		
		NSTableColumn *ratioCol;

		ratioCol = [_tableView tableColumnWithIdentifier:COL_ID_RATIO];
		if (ratioCol != nil)
		{
			[_tableView removeTableColumn:ratioCol];
		}
	}
}

/**
 * Returns the DataSource for our table view
 */
- (TableViewDataSource *)tableViewDataSource
{
	if (_tableViewDataSource == nil)
	{
		_tableViewDataSource = [[TableViewDataSource alloc] init];
		[_tableViewDataSource setArchive:_archive];
	}
	return _tableViewDataSource;
}

- (void)setArchive:(Archive *)archive
{
	ASSIGN(_archive, archive);
	// make sure the data source knows the archive as well
	[_tableViewDataSource setArchive:archive];
}

- (void)openFile:(NSString *)file withDefaultApp:(NSString *)defaultApp;
{
	NSString *extension;
	int rc;
	
	extension = [file pathExtension];
	// this sux: if the file does not have an extension, NSWorkspace does not know how to 
	// handle it. Handling of files should be based on file's contents instead of its 
	// extension, like the unix command 'file' does.
	if ([extension isEqual:@""])
	{
		if (defaultApp == nil)
		{
			NSRunAlertPanel(@"Could not open", @"No default open application set in preferences", 
				nil, nil, nil);
			return;
		}
		else
		{
			rc = [[NSWorkspace sharedWorkspace] openFile:file withApplication:defaultApp];
		}
	}
	else
	{
		// we have a valid path extension, try opening with NSWorkspace's default app
		rc = [[NSWorkspace sharedWorkspace] openFile:file];
		if (rc == NO)
		{
			// NSWorkspace could not open the file, try again with the default app we've been given
			if (defaultApp != nil)
			{
				rc = [[NSWorkspace sharedWorkspace] openFile:file withApplication:defaultApp];
			}
			else
			{
				NSRunAlertPanel(@"Could not open", @"No default open application set in preferences", 
					nil, nil, nil);
				return;
			}
		}
	}
			
	if (rc == NO)
	{
		NSRunAlertPanel(@"Could not open", @"Don't know how to open file %@", nil, nil, nil, file);
	}
}

- (void)tableViewDoubleAction:(id)sender
{
	[self view:sender];
}
	
@end
