/*
    libzint - the open source barcode library
    Copyright (C) 2008-2020 Robin Stuart <rstuart114@gmail.com>

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:

    1. Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
    3. Neither the name of the project nor the names of its contributors
       may be used to endorse or promote products derived from this software
       without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
    ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
    IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
    ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
    DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
    OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
    HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
    LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
    OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
    SUCH DAMAGE.
 */
/* vim: set ts=4 sw=4 et : */
/*
 * Adapted from GNU LIBICONV library and patched to add 2 duplicate mappings
 * for compatibility with GB 18030 subset:
 * 1) U+00B7 to 0xA1A4 (duplicate of U+30FB)
 * 2) U+2014 to 0xA1AA (duplicate of U+2015)
 */
/*
 * Copyright (C) 1999-2001, 2012, 2016 Free Software Foundation, Inc.
 * This file is part of the GNU LIBICONV Library.
 *
 * The GNU LIBICONV Library is free software; you can redistribute it
 * and/or modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The GNU LIBICONV Library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with the GNU LIBICONV Library; see the file COPYING.LIB.
 * If not, see <https://www.gnu.org/licenses/>.
 */
#include <string.h>
#include "common.h"
#include "gb2312.h"

INTERNAL int utf_to_eci(const int eci, const unsigned char source[], unsigned char dest[], size_t *length); /* Convert Unicode to other encodings */

/*
 * GB2312.1980-0 (libiconv-1.16/lib/gb2312.h)
 */

/* ZINT: Table converted from GB 2312 to EUC-CN values using tools/cnv_gb2312.php:
 *
while ($line = fgets(STDIN)) {
    echo preg_replace_callback('/0x([0-9a-f]{4})/', function ($matches) {
        return '0x' . dechex(hexdec($matches[1]) + 0x8080);
    }, $line);
}
 */
static const unsigned short gb2312_2charset[7445] = {
  0xa1e8, 0xa1ec, 0xa1a7, 0xa1e3, 0xa1c0, 0xa1c1, 0xa8a4, 0xa8a2,
  0xa8a8, 0xa8a6, 0xa8ba, 0xa8ac, 0xa8aa, 0xa8b0, 0xa8ae, 0xa1c2,
  0xa8b4, 0xa8b2, 0xa8b9, 0xa8a1, 0xa8a5, 0xa8a7, 0xa8a9, 0xa8ad,
  0xa8b1, 0xa8a3, 0xa8ab, 0xa8af, 0xa8b3, 0xa8b5, 0xa8b6, 0xa8b7,
  0xa8b8, 0xa1a6, 0xa1a5, 0xa6a1, 0xa6a2, 0xa6a3, 0xa6a4, 0xa6a5,
  0xa6a6, 0xa6a7, 0xa6a8, 0xa6a9, 0xa6aa, 0xa6ab, 0xa6ac, 0xa6ad,
  0xa6ae, 0xa6af, 0xa6b0, 0xa6b1, 0xa6b2, 0xa6b3, 0xa6b4, 0xa6b5,
  0xa6b6, 0xa6b7, 0xa6b8, 0xa6c1, 0xa6c2, 0xa6c3, 0xa6c4, 0xa6c5,
  0xa6c6, 0xa6c7, 0xa6c8, 0xa6c9, 0xa6ca, 0xa6cb, 0xa6cc, 0xa6cd,
  0xa6ce, 0xa6cf, 0xa6d0, 0xa6d1, 0xa6d2, 0xa6d3, 0xa6d4, 0xa6d5,
  0xa6d6, 0xa6d7, 0xa6d8, 0xa7a7, 0xa7a1, 0xa7a2, 0xa7a3, 0xa7a4,
  0xa7a5, 0xa7a6, 0xa7a8, 0xa7a9, 0xa7aa, 0xa7ab, 0xa7ac, 0xa7ad,
  0xa7ae, 0xa7af, 0xa7b0, 0xa7b1, 0xa7b2, 0xa7b3, 0xa7b4, 0xa7b5,
  0xa7b6, 0xa7b7, 0xa7b8, 0xa7b9, 0xa7ba, 0xa7bb, 0xa7bc, 0xa7bd,
  0xa7be, 0xa7bf, 0xa7c0, 0xa7c1, 0xa7d1, 0xa7d2, 0xa7d3, 0xa7d4,
  0xa7d5, 0xa7d6, 0xa7d8, 0xa7d9, 0xa7da, 0xa7db, 0xa7dc, 0xa7dd,
  0xa7de, 0xa7df, 0xa7e0, 0xa7e1, 0xa7e2, 0xa7e3, 0xa7e4, 0xa7e5,
  0xa7e6, 0xa7e7, 0xa7e8, 0xa7e9, 0xa7ea, 0xa7eb, 0xa7ec, 0xa7ed,
  0xa7ee, 0xa7ef, 0xa7f0, 0xa7f1, 0xa7d7, 0xa1aa, 0xa1ac, 0xa1ae,
  0xa1af, 0xa1b0, 0xa1b1, 0xa1ad, 0xa1eb, 0xa1e4, 0xa1e5, 0xa1f9,
  0xa1e6, 0xa1ed, 0xa2f1, 0xa2f2, 0xa2f3, 0xa2f4, 0xa2f5, 0xa2f6,
  0xa2f7, 0xa2f8, 0xa2f9, 0xa2fa, 0xa2fb, 0xa2fc, 0xa1fb, 0xa1fc,
  0xa1fa, 0xa1fd, 0xa1ca, 0xa1c7, 0xa1c6, 0xa1cc, 0xa1d8, 0xa1de,
  0xa1cf, 0xa1ce, 0xa1c4, 0xa1c5, 0xa1c9, 0xa1c8, 0xa1d2, 0xa1d3,
  0xa1e0, 0xa1df, 0xa1c3, 0xa1cb, 0xa1d7, 0xa1d6, 0xa1d5, 0xa1d9,
  0xa1d4, 0xa1dc, 0xa1dd, 0xa1da, 0xa1db, 0xa1d1, 0xa1cd, 0xa1d0,
  0xa2d9, 0xa2da, 0xa2db, 0xa2dc, 0xa2dd, 0xa2de, 0xa2df, 0xa2e0,
  0xa2e1, 0xa2e2, 0xa2c5, 0xa2c6, 0xa2c7, 0xa2c8, 0xa2c9, 0xa2ca,
  0xa2cb, 0xa2cc, 0xa2cd, 0xa2ce, 0xa2cf, 0xa2d0, 0xa2d1, 0xa2d2,
  0xa2d3, 0xa2d4, 0xa2d5, 0xa2d6, 0xa2d7, 0xa2d8, 0xa2b1, 0xa2b2,
  0xa2b3, 0xa2b4, 0xa2b5, 0xa2b6, 0xa2b7, 0xa2b8, 0xa2b9, 0xa2ba,
  0xa2bb, 0xa2bc, 0xa2bd, 0xa2be, 0xa2bf, 0xa2c0, 0xa2c1, 0xa2c2,
  0xa2c3, 0xa2c4, 0xa9a4, 0xa9a5, 0xa9a6, 0xa9a7, 0xa9a8, 0xa9a9,
  0xa9aa, 0xa9ab, 0xa9ac, 0xa9ad, 0xa9ae, 0xa9af, 0xa9b0, 0xa9b1,
  0xa9b2, 0xa9b3, 0xa9b4, 0xa9b5, 0xa9b6, 0xa9b7, 0xa9b8, 0xa9b9,
  0xa9ba, 0xa9bb, 0xa9bc, 0xa9bd, 0xa9be, 0xa9bf, 0xa9c0, 0xa9c1,
  0xa9c2, 0xa9c3, 0xa9c4, 0xa9c5, 0xa9c6, 0xa9c7, 0xa9c8, 0xa9c9,
  0xa9ca, 0xa9cb, 0xa9cc, 0xa9cd, 0xa9ce, 0xa9cf, 0xa9d0, 0xa9d1,
  0xa9d2, 0xa9d3, 0xa9d4, 0xa9d5, 0xa9d6, 0xa9d7, 0xa9d8, 0xa9d9,
  0xa9da, 0xa9db, 0xa9dc, 0xa9dd, 0xa9de, 0xa9df, 0xa9e0, 0xa9e1,
  0xa9e2, 0xa9e3, 0xa9e4, 0xa9e5, 0xa9e6, 0xa9e7, 0xa9e8, 0xa9e9,
  0xa9ea, 0xa9eb, 0xa9ec, 0xa9ed, 0xa9ee, 0xa9ef, 0xa1f6, 0xa1f5,
  0xa1f8, 0xa1f7, 0xa1f4, 0xa1f3, 0xa1f0, 0xa1f2, 0xa1f1, 0xa1ef,
  0xa1ee, 0xa1e2, 0xa1e1, 0xa1a1, 0xa1a2, 0xa1a3, 0xa1a8, 0xa1a9,
  0xa1b4, 0xa1b5, 0xa1b6, 0xa1b7, 0xa1b8, 0xa1b9, 0xa1ba, 0xa1bb,
  0xa1be, 0xa1bf, 0xa1fe, 0xa1b2, 0xa1b3, 0xa1bc, 0xa1bd, 0xa4a1,
  0xa4a2, 0xa4a3, 0xa4a4, 0xa4a5, 0xa4a6, 0xa4a7, 0xa4a8, 0xa4a9,
  0xa4aa, 0xa4ab, 0xa4ac, 0xa4ad, 0xa4ae, 0xa4af, 0xa4b0, 0xa4b1,
  0xa4b2, 0xa4b3, 0xa4b4, 0xa4b5, 0xa4b6, 0xa4b7, 0xa4b8, 0xa4b9,
  0xa4ba, 0xa4bb, 0xa4bc, 0xa4bd, 0xa4be, 0xa4bf, 0xa4c0, 0xa4c1,
  0xa4c2, 0xa4c3, 0xa4c4, 0xa4c5, 0xa4c6, 0xa4c7, 0xa4c8, 0xa4c9,
  0xa4ca, 0xa4cb, 0xa4cc, 0xa4cd, 0xa4ce, 0xa4cf, 0xa4d0, 0xa4d1,
  0xa4d2, 0xa4d3, 0xa4d4, 0xa4d5, 0xa4d6, 0xa4d7, 0xa4d8, 0xa4d9,
  0xa4da, 0xa4db, 0xa4dc, 0xa4dd, 0xa4de, 0xa4df, 0xa4e0, 0xa4e1,
  0xa4e2, 0xa4e3, 0xa4e4, 0xa4e5, 0xa4e6, 0xa4e7, 0xa4e8, 0xa4e9,
  0xa4ea, 0xa4eb, 0xa4ec, 0xa4ed, 0xa4ee, 0xa4ef, 0xa4f0, 0xa4f1,
  0xa4f2, 0xa4f3, 0xa5a1, 0xa5a2, 0xa5a3, 0xa5a4, 0xa5a5, 0xa5a6,
  0xa5a7, 0xa5a8, 0xa5a9, 0xa5aa, 0xa5ab, 0xa5ac, 0xa5ad, 0xa5ae,
  0xa5af, 0xa5b0, 0xa5b1, 0xa5b2, 0xa5b3, 0xa5b4, 0xa5b5, 0xa5b6,
  0xa5b7, 0xa5b8, 0xa5b9, 0xa5ba, 0xa5bb, 0xa5bc, 0xa5bd, 0xa5be,
  0xa5bf, 0xa5c0, 0xa5c1, 0xa5c2, 0xa5c3, 0xa5c4, 0xa5c5, 0xa5c6,
  0xa5c7, 0xa5c8, 0xa5c9, 0xa5ca, 0xa5cb, 0xa5cc, 0xa5cd, 0xa5ce,
  0xa5cf, 0xa5d0, 0xa5d1, 0xa5d2, 0xa5d3, 0xa5d4, 0xa5d5, 0xa5d6,
  0xa5d7, 0xa5d8, 0xa5d9, 0xa5da, 0xa5db, 0xa5dc, 0xa5dd, 0xa5de,
  0xa5df, 0xa5e0, 0xa5e1, 0xa5e2, 0xa5e3, 0xa5e4, 0xa5e5, 0xa5e6,
  0xa5e7, 0xa5e8, 0xa5e9, 0xa5ea, 0xa5eb, 0xa5ec, 0xa5ed, 0xa5ee,
  0xa5ef, 0xa5f0, 0xa5f1, 0xa5f2, 0xa5f3, 0xa5f4, 0xa5f5, 0xa5f6,
  0xa1a4, 0xa8c5, 0xa8c6, 0xa8c7, 0xa8c8, 0xa8c9, 0xa8ca, 0xa8cb,
  0xa8cc, 0xa8cd, 0xa8ce, 0xa8cf, 0xa8d0, 0xa8d1, 0xa8d2, 0xa8d3,
  0xa8d4, 0xa8d5, 0xa8d6, 0xa8d7, 0xa8d8, 0xa8d9, 0xa8da, 0xa8db,
  0xa8dc, 0xa8dd, 0xa8de, 0xa8df, 0xa8e0, 0xa8e1, 0xa8e2, 0xa8e3,
  0xa8e4, 0xa8e5, 0xa8e6, 0xa8e7, 0xa8e8, 0xa8e9, 0xa2e5, 0xa2e6,
  0xa2e7, 0xa2e8, 0xa2e9, 0xa2ea, 0xa2eb, 0xa2ec, 0xa2ed, 0xa2ee,
  0xd2bb, 0xb6a1, 0xc6df, 0xcdf2, 0xd5c9, 0xc8fd, 0xc9cf, 0xcfc2,
  0xd8a2, 0xb2bb, 0xd3eb, 0xd8a4, 0xb3f3, 0xd7a8, 0xc7d2, 0xd8a7,
  0xcac0, 0xc7f0, 0xb1fb, 0xd2b5, 0xb4d4, 0xb6ab, 0xcbbf, 0xd8a9,
  0xb6aa, 0xc1bd, 0xd1cf, 0xc9a5, 0xd8ad, 0xb8f6, 0xd1be, 0xe3dc,
  0xd6d0, 0xb7e1, 0xb4ae, 0xc1d9, 0xd8bc, 0xcde8, 0xb5a4, 0xceaa,
  0xd6f7, 0xc0f6, 0xbed9, 0xd8af, 0xc4cb, 0xbec3, 0xd8b1, 0xc3b4,
  0xd2e5, 0xd6ae, 0xceda, 0xd5a7, 0xbaf5, 0xb7a6, 0xc0d6, 0xc6b9,
  0xc5d2, 0xc7c7, 0xb9d4, 0xb3cb, 0xd2d2, 0xd8bf, 0xbec5, 0xc6f2,
  0xd2b2, 0xcfb0, 0xcfe7, 0xcae9, 0xd8c0, 0xc2f2, 0xc2d2, 0xc8e9,
  0xc7ac, 0xc1cb, 0xd3e8, 0xd5f9, 0xcac2, 0xb6fe, 0xd8a1, 0xd3da,
  0xbff7, 0xd4c6, 0xbba5, 0xd8c1, 0xcee5, 0xbeae, 0xd8a8, 0xd1c7,
  0xd0a9, 0xd8bd, 0xd9ef, 0xcdf6, 0xbfba, 0xbdbb, 0xbaa5, 0xd2e0,
  0xb2fa, 0xbae0, 0xc4b6, 0xcfed, 0xbea9, 0xcda4, 0xc1c1, 0xc7d7,
  0xd9f1, 0xd9f4, 0xc8cb, 0xd8e9, 0xd2da, 0xcab2, 0xc8ca, 0xd8ec,
  0xd8ea, 0xd8c6, 0xbdf6, 0xc6cd, 0xb3f0, 0xd8eb, 0xbdf1, 0xbde9,
  0xc8d4, 0xb4d3, 0xc2d8, 0xb2d6, 0xd7d0, 0xcacb, 0xcbfb, 0xd5cc,
  0xb8b6, 0xcfc9, 0xd9da, 0xd8f0, 0xc7aa, 0xd8ee, 0xb4fa, 0xc1ee,
  0xd2d4, 0xd8ed, 0xd2c7, 0xd8ef, 0xc3c7, 0xd1f6, 0xd6d9, 0xd8f2,
  0xd8f5, 0xbcfe, 0xbcdb, 0xc8ce, 0xb7dd, 0xb7c2, 0xc6f3, 0xd8f8,
  0xd2c1, 0xcee9, 0xbcbf, 0xb7fc, 0xb7a5, 0xd0dd, 0xd6da, 0xd3c5,
  0xbbef, 0xbbe1, 0xd8f1, 0xc9a1, 0xceb0, 0xb4ab, 0xd8f3, 0xc9cb,
  0xd8f6, 0xc2d7, 0xd8f7, 0xceb1, 0xd8f9, 0xb2ae, 0xb9c0, 0xd9a3,
  0xb0e9, 0xc1e6, 0xc9ec, 0xcbc5, 0xcbc6, 0xd9a4, 0xb5e8, 0xb5ab,
  0xcebb, 0xb5cd, 0xd7a1, 0xd7f4, 0xd3d3, 0xcce5, 0xbace, 0xd9a2,
  0xd9dc, 0xd3e0, 0xd8fd, 0xb7f0, 0xd7f7, 0xd8fe, 0xd8fa, 0xd9a1,
  0xc4e3, 0xd3b6, 0xd8f4, 0xd9dd, 0xd8fb, 0xc5e5, 0xc0d0, 0xd1f0,
  0xb0db, 0xbcd1, 0xd9a6, 0xd9a5, 0xd9ac, 0xd9ae, 0xd9ab, 0xcab9,
  0xd9a9, 0xd6b6, 0xb3de, 0xd9a8, 0xc0fd, 0xcacc, 0xd9aa, 0xd9a7,
  0xd9b0, 0xb6b1, 0xb9a9, 0xd2c0, 0xcfc0, 0xc2c2, 0xbdc4, 0xd5ec,
  0xb2e0, 0xc7c8, 0xbfeb, 0xd9ad, 0xd9af, 0xceea, 0xbaee, 0xc7d6,
  0xb1e3, 0xb4d9, 0xb6ed, 0xd9b4, 0xbfa1, 0xd9de, 0xc7ce, 0xc0fe,
  0xd9b8, 0xcbd7, 0xb7fd, 0xd9b5, 0xd9b7, 0xb1a3, 0xd3e1, 0xd9b9,
  0xd0c5, 0xd9b6, 0xd9b1, 0xd9b2, 0xc1a9, 0xd9b3, 0xbcf3, 0xd0de,
  0xb8a9, 0xbee3, 0xd9bd, 0xd9ba, 0xb0b3, 0xd9c2, 0xd9c4, 0xb1b6,
  0xd9bf, 0xb5b9, 0xbef3, 0xccc8, 0xbaf2, 0xd2d0, 0xd9c3, 0xbde8,
  0xb3ab, 0xd9c5, 0xbeeb, 0xd9c6, 0xd9bb, 0xc4df, 0xd9be, 0xd9c1,
  0xd9c0, 0xd5ae, 0xd6b5, 0xc7e3, 0xd9c8, 0xbcd9, 0xd9ca, 0xd9bc,
  0xd9cb, 0xc6ab, 0xd9c9, 0xd7f6, 0xcda3, 0xbda1, 0xd9cc, 0xc5bc,
  0xcdb5, 0xd9cd, 0xd9c7, 0xb3a5, 0xbffe, 0xb8b5, 0xc0fc, 0xb0f8,
  0xb4f6, 0xd9ce, 0xd9cf, 0xb4a2, 0xd9d0, 0xb4df, 0xb0c1, 0xd9d1,
  0xc9b5, 0xcff1, 0xd9d2, 0xc1c5, 0xd9d6, 0xc9ae, 0xd9d5, 0xd9d4,
  0xd9d7, 0xcbdb, 0xbda9, 0xc6a7, 0xd9d3, 0xd9d8, 0xd9d9, 0xc8e5,
  0xc0dc, 0xb6f9, 0xd8a3, 0xd4ca, 0xd4aa, 0xd0d6, 0xb3e4, 0xd5d7,
  0xcfc8, 0xb9e2, 0xbfcb, 0xc3e2, 0xb6d2, 0xcdc3, 0xd9ee, 0xd9f0,
  0xb5b3, 0xb6b5, 0xbea4, 0xc8eb, 0xc8ab, 0xb0cb, 0xb9ab, 0xc1f9,
  0xd9e2, 0xc0bc, 0xb9b2, 0xb9d8, 0xd0cb, 0xb1f8, 0xc6e4, 0xbedf,
  0xb5e4, 0xd7c8, 0xd1f8, 0xbce6, 0xcade, 0xbcbd, 0xd9e6, 0xd8e7,
  0xc4da, 0xb8d4, 0xc8bd, 0xb2e1, 0xd4d9, 0xc3b0, 0xc3e1, 0xdaa2,
  0xc8df, 0xd0b4, 0xbefc, 0xc5a9, 0xb9da, 0xdaa3, 0xd4a9, 0xdaa4,
  0xd9fb, 0xb6ac, 0xb7eb, 0xb1f9, 0xd9fc, 0xb3e5, 0xbef6, 0xbff6,
  0xd2b1, 0xc0e4, 0xb6b3, 0xd9fe, 0xd9fd, 0xbebb, 0xc6e0, 0xd7bc,
  0xdaa1, 0xc1b9, 0xb5f2, 0xc1e8, 0xbcf5, 0xb4d5, 0xc1dd, 0xc4fd,
  0xbcb8, 0xb7b2, 0xb7ef, 0xd9ec, 0xc6be, 0xbfad, 0xbbcb, 0xb5ca,
  0xdbc9, 0xd0d7, 0xcdb9, 0xb0bc, 0xb3f6, 0xbbf7, 0xdbca, 0xbaaf,
  0xd4e4, 0xb5b6, 0xb5f3, 0xd8d6, 0xc8d0, 0xb7d6, 0xc7d0, 0xd8d7,
  0xbfaf, 0xdbbb, 0xd8d8, 0xd0cc, 0xbbae, 0xebbe, 0xc1d0, 0xc1f5,
  0xd4f2, 0xb8d5, 0xb4b4, 0xb3f5, 0xc9be, 0xc5d0, 0xc5d9, 0xc0fb,
  0xb1f0, 0xd8d9, 0xb9ce, 0xb5bd, 0xd8da, 0xd6c6, 0xcba2, 0xc8af,
  0xc9b2, 0xb4cc, 0xbfcc, 0xb9f4, 0xd8db, 0xd8dc, 0xb6e7, 0xbcc1,
  0xccea, 0xcff7, 0xd8dd, 0xc7b0, 0xb9d0, 0xbda3, 0xccde, 0xc6ca,
  0xd8e0, 0xd8de, 0xd8df, 0xb0fe, 0xbee7, 0xcaa3, 0xbcf4, 0xb8b1,
  0xb8ee, 0xd8e2, 0xbdcb, 0xd8e4, 0xd8e3, 0xc5fc, 0xd8e5, 0xd8e6,
  0xc1a6, 0xc8b0, 0xb0ec, 0xb9a6, 0xbcd3, 0xcef1, 0xdbbd, 0xc1d3,
  0xb6af, 0xd6fa, 0xc5ac, 0xbdd9, 0xdbbe, 0xdbbf, 0xc0f8, 0xbea2,
  0xc0cd, 0xdbc0, 0xcac6, 0xb2aa, 0xd3c2, 0xc3e3, 0xd1ab, 0xdbc2,
  0xc0d5, 0xdbc3, 0xbfb1, 0xc4bc, 0xc7da, 0xdbc4, 0xd9e8, 0xc9d7,
  0xb9b4, 0xcef0, 0xd4c8, 0xb0fc, 0xb4d2, 0xd0d9, 0xd9e9, 0xdecb,
  0xd9eb, 0xd8b0, 0xbbaf, 0xb1b1, 0xb3d7, 0xd8ce, 0xd4d1, 0xbdb3,
  0xbfef, 0xcfbb, 0xd8d0, 0xb7cb, 0xd8d1, 0xc6a5, 0xc7f8, 0xd2bd,
  0xd8d2, 0xc4e4, 0xcaae, 0xc7a7, 0xd8a6, 0xc9fd, 0xcee7, 0xbbdc,
  0xb0eb, 0xbbaa, 0xd0ad, 0xb1b0, 0xd7e4, 0xd7bf, 0xb5a5, 0xc2f4,
  0xc4cf, 0xb2a9, 0xb2b7, 0xb1e5, 0xdfb2, 0xd5bc, 0xbfa8, 0xc2ac,
  0xd8d5, 0xc2b1, 0xd8d4, 0xced4, 0xdae0, 0xcec0, 0xd8b4, 0xc3ae,
  0xd3a1, 0xcea3, 0xbcb4, 0xc8b4, 0xc2d1, 0xbeed, 0xd0b6, 0xdae1,
  0xc7e4, 0xb3a7, 0xb6f2, 0xccfc, 0xc0fa, 0xc0f7, 0xd1b9, 0xd1e1,
  0xd8c7, 0xb2de, 0xc0e5, 0xbaf1, 0xd8c8, 0xd4ad, 0xcfe1, 0xd8c9,
  0xd8ca, 0xcfc3, 0xb3f8, 0xbec7, 0xd8cb, 0xdbcc, 0xc8a5, 0xcfd8,
  0xc8fe, 0xb2ce, 0xd3d6, 0xb2e6, 0xbcb0, 0xd3d1, 0xcbab, 0xb7b4,
  0xb7a2, 0xcae5, 0xc8a1, 0xcadc, 0xb1e4, 0xd0f0, 0xc5d1, 0xdbc5,
  0xb5fe, 0xbfda, 0xb9c5, 0xbee4, 0xc1ed, 0xdfb6, 0xdfb5, 0xd6bb,
  0xbdd0, 0xd5d9, 0xb0c8, 0xb6a3, 0xbfc9, 0xcca8, 0xdfb3, 0xcab7,
  0xd3d2, 0xd8cf, 0xd2b6, 0xbac5, 0xcbbe, 0xccbe, 0xdfb7, 0xb5f0,
  0xdfb4, 0xd3f5, 0xb3d4, 0xb8f7, 0xdfba, 0xbacf, 0xbcaa, 0xb5f5,
  0xcdac, 0xc3fb, 0xbaf3, 0xc0f4, 0xcdc2, 0xcff2, 0xdfb8, 0xcfc5,
  0xc2c0, 0xdfb9, 0xc2f0, 0xbefd, 0xc1df, 0xcdcc, 0xd2f7, 0xb7cd,
  0xdfc1, 0xdfc4, 0xb7f1, 0xb0c9, 0xb6d6, 0xb7d4, 0xbaac, 0xccfd,
  0xbfd4, 0xcbb1, 0xc6f4, 0xd6a8, 0xdfc5, 0xcee2, 0xb3b3, 0xcefc,
  0xb4b5, 0xcec7, 0xbaf0, 0xcee1, 0xd1bd, 0xdfc0, 0xb4f4, 0xb3ca,
  0xb8e6, 0xdfbb, 0xc4c5, 0xdfbc, 0xdfbd, 0xdfbe, 0xc5bb, 0xdfbf,
  0xdfc2, 0xd4b1, 0xdfc3, 0xc7ba, 0xced8, 0xc4d8, 0xdfca, 0xdfcf,
  0xd6dc, 0xdfc9, 0xdfda, 0xceb6, 0xbac7, 0xdfce, 0xdfc8, 0xc5de,
  0xc9eb, 0xbaf4, 0xc3fc, 0xbed7, 0xdfc6, 0xdfcd, 0xc5d8, 0xd5a6,
  0xbacd, 0xbecc, 0xd3bd, 0xb8c0, 0xd6e4, 0xdfc7, 0xb9be, 0xbfa7,
  0xc1fc, 0xdfcb, 0xdfcc, 0xdfd0, 0xdfdb, 0xdfe5, 0xdfd7, 0xdfd6,
  0xd7c9, 0xdfe3, 0xdfe4, 0xe5eb, 0xd2a7, 0xdfd2, 0xbfa9, 0xd4db,
  0xbfc8, 0xdfd4, 0xcfcc, 0xdfdd, 0xd1ca, 0xdfde, 0xb0a7, 0xc6b7,
  0xdfd3, 0xbae5, 0xb6df, 0xcddb, 0xb9fe, 0xd4d5, 0xdfdf, 0xcfec,
  0xb0a5, 0xdfe7, 0xdfd1, 0xd1c6, 0xdfd5, 0xdfd8, 0xdfd9, 0xdfdc,
  0xbba9, 0xdfe0, 0xdfe1, 0xdfe2, 0xdfe6, 0xdfe8, 0xd3b4, 0xb8e7,
  0xc5b6, 0xdfea, 0xc9da, 0xc1a8, 0xc4c4, 0xbfde, 0xcff8, 0xd5dc,
  0xdfee, 0xb2b8, 0xbadf, 0xdfec, 0xdbc1, 0xd1e4, 0xcbf4, 0xb4bd,
  0xb0a6, 0xdff1, 0xccc6, 0xdff2, 0xdfed, 0xdfe9, 0xdfeb, 0xdfef,
  0xdff0, 0xbbbd, 0xdff3, 0xdff4, 0xbba3, 0xcadb, 0xcea8, 0xe0a7,
  0xb3aa, 0xe0a6, 0xe0a1, 0xdffe, 0xcdd9, 0xdffc, 0xdffa, 0xbfd0,
  0xd7c4, 0xc9cc, 0xdff8, 0xb0a1, 0xdffd, 0xdffb, 0xe0a2, 0xe0a8,
  0xb7c8, 0xc6a1, 0xc9b6, 0xc0b2, 0xdff5, 0xc5be, 0xd8c4, 0xdff9,
  0xc4f6, 0xe0a3, 0xe0a4, 0xe0a5, 0xd0a5, 0xe0b4, 0xcce4, 0xe0b1,
  0xbfa6, 0xe0af, 0xceb9, 0xe0ab, 0xc9c6, 0xc0ae, 0xe0ae, 0xbaed,
  0xbab0, 0xe0a9, 0xdff6, 0xe0b3, 0xe0b8, 0xb4ad, 0xe0b9, 0xcfb2,
  0xbac8, 0xe0b0, 0xd0fa, 0xe0ac, 0xd4fb, 0xdff7, 0xc5e7, 0xe0ad,
  0xd3f7, 0xe0b6, 0xe0b7, 0xe0c4, 0xd0e1, 0xe0bc, 0xe0c9, 0xe0ca,
  0xe0be, 0xe0aa, 0xc9a4, 0xe0c1, 0xe0b2, 0xcac8, 0xe0c3, 0xe0b5,
  0xcecb, 0xcbc3, 0xe0cd, 0xe0c6, 0xe0c2, 0xe0cb, 0xe0ba, 0xe0bf,
  0xe0c0, 0xe0c5, 0xe0c7, 0xe0c8, 0xe0cc, 0xe0bb, 0xcbd4, 0xe0d5,
  0xe0d6, 0xe0d2, 0xe0d0, 0xbcce, 0xe0d1, 0xb8c2, 0xd8c5, 0xd0ea,
  0xc2ef, 0xe0cf, 0xe0bd, 0xe0d4, 0xe0d3, 0xe0d7, 0xe0dc, 0xe0d8,
  0xd6f6, 0xb3b0, 0xd7ec, 0xcbbb, 0xe0da, 0xcefb, 0xbad9, 0xe0e1,
  0xe0dd, 0xd2ad, 0xe0e2, 0xe0db, 0xe0d9, 0xe0df, 0xe0e0, 0xe0de,
  0xe0e4, 0xc6f7, 0xd8ac, 0xd4eb, 0xe0e6, 0xcac9, 0xe0e5, 0xb8c1,
  0xe0e7, 0xe0e8, 0xe0e9, 0xe0e3, 0xbabf, 0xcce7, 0xe0ea, 0xcff9,
  0xe0eb, 0xc8c2, 0xbdc0, 0xc4d2, 0xe0ec, 0xe0ed, 0xc7f4, 0xcbc4,
  0xe0ee, 0xbbd8, 0xd8b6, 0xd2f2, 0xe0ef, 0xcdc5, 0xb6da, 0xe0f1,
  0xd4b0, 0xc0a7, 0xb4d1, 0xcea7, 0xe0f0, 0xe0f2, 0xb9cc, 0xb9fa,
  0xcdbc, 0xe0f3, 0xc6d4, 0xe0f4, 0xd4b2, 0xc8a6, 0xe0f6, 0xe0f5,
  0xe0f7, 0xcdc1, 0xcaa5, 0xd4da, 0xdbd7, 0xdbd9, 0xdbd8, 0xb9e7,
  0xdbdc, 0xdbdd, 0xb5d8, 0xdbda, 0xdbdb, 0xb3a1, 0xdbdf, 0xbbf8,
  0xd6b7, 0xdbe0, 0xbef9, 0xb7bb, 0xdbd0, 0xccae, 0xbfb2, 0xbbb5,
  0xd7f8, 0xbfd3, 0xbfe9, 0xbce1, 0xccb3, 0xdbde, 0xb0d3, 0xceeb,
  0xb7d8, 0xd7b9, 0xc6c2, 0xc0a4, 0xccb9, 0xdbe7, 0xdbe1, 0xc6ba,
  0xdbe3, 0xdbe8, 0xc5f7, 0xdbea, 0xdbe9, 0xbfc0, 0xdbe6, 0xdbe5,
  0xb4b9, 0xc0ac, 0xc2a2, 0xdbe2, 0xdbe4, 0xd0cd, 0xdbed, 0xc0dd,
  0xdbf2, 0xb6e2, 0xdbf3, 0xdbd2, 0xb9b8, 0xd4ab, 0xdbec, 0xbfd1,
  0xdbf0, 0xdbd1, 0xb5e6, 0xdbeb, 0xbfe5, 0xdbee, 0xdbf1, 0xdbf9,
  0xb9a1, 0xb0a3, 0xc2f1, 0xb3c7, 0xdbef, 0xdbf8, 0xc6d2, 0xdbf4,
  0xdbf5, 0xdbf7, 0xdbf6, 0xdbfe, 0xd3f2, 0xb2ba, 0xdbfd, 0xdca4,
  0xdbfb, 0xdbfa, 0xdbfc, 0xc5e0, 0xbbf9, 0xdca3, 0xdca5, 0xccc3,
  0xb6d1, 0xddc0, 0xdca1, 0xdca2, 0xc7b5, 0xb6e9, 0xdca7, 0xdca6,
  0xdca9, 0xb1a4, 0xb5cc, 0xbfb0, 0xd1df, 0xb6c2, 0xdca8, 0xcbfa,
  0xebf3, 0xcbdc, 0xcbfe, 0xccc1, 0xc8fb, 0xdcaa, 0xccee, 0xdcab,
  0xdbd3, 0xdcaf, 0xdcac, 0xbeb3, 0xcafb, 0xdcad, 0xc9ca, 0xc4b9,
  0xc7bd, 0xdcae, 0xd4f6, 0xd0e6, 0xc4ab, 0xb6d5, 0xdbd4, 0xb1da,
  0xdbd5, 0xdbd6, 0xbabe, 0xc8c0, 0xcabf, 0xc8c9, 0xd7b3, 0xc9f9,
  0xbfc7, 0xbaf8, 0xd2bc, 0xe2ba, 0xb4a6, 0xb1b8, 0xb8b4, 0xcfc4,
  0xd9e7, 0xcfa6, 0xcde2, 0xd9ed, 0xb6e0, 0xd2b9, 0xb9bb, 0xe2b9,
  0xe2b7, 0xb4f3, 0xccec, 0xccab, 0xb7f2, 0xd8b2, 0xd1eb, 0xbabb,
  0xcaa7, 0xcdb7, 0xd2c4, 0xbfe4, 0xbcd0, 0xb6e1, 0xdec5, 0xdec6,
  0xdbbc, 0xd1d9, 0xc6e6, 0xc4ce, 0xb7ee, 0xb7dc, 0xbffc, 0xd7e0,
  0xc6f5, 0xb1bc, 0xdec8, 0xbdb1, 0xccd7, 0xdeca, 0xdec9, 0xb5ec,
  0xc9dd, 0xb0c2, 0xc5ae, 0xc5ab, 0xc4cc, 0xbce9, 0xcbfd, 0xbac3,
  0xe5f9, 0xc8e7, 0xe5fa, 0xcdfd, 0xd7b1, 0xb8be, 0xc2e8, 0xc8d1,
  0xe5fb, 0xb6ca, 0xbccb, 0xd1fd, 0xe6a1, 0xc3ee, 0xe6a4, 0xe5fe,
  0xe6a5, 0xcdd7, 0xb7c1, 0xe5fc, 0xe5fd, 0xe6a3, 0xc4dd, 0xe6a8,
  0xe6a7, 0xc3c3, 0xc6de, 0xe6aa, 0xc4b7, 0xe6a2, 0xcabc, 0xbde3,
  0xb9c3, 0xe6a6, 0xd0d5, 0xceaf, 0xe6a9, 0xe6b0, 0xd2a6, 0xbdaa,
  0xe6ad, 0xe6af, 0xc0d1, 0xd2cc, 0xbca7, 0xe6b1, 0xd2f6, 0xd7cb,
  0xcdfe, 0xcdde, 0xc2a6, 0xe6ab, 0xe6ac, 0xbdbf, 0xe6ae, 0xe6b3,
  0xe6b2, 0xe6b6, 0xe6b8, 0xc4ef, 0xc4c8, 0xbeea, 0xc9ef, 0xe6b7,
  0xb6f0, 0xc3e4, 0xd3e9, 0xe6b4, 0xe6b5, 0xc8a2, 0xe6bd, 0xe6b9,
  0xc6c5, 0xcdf1, 0xe6bb, 0xe6bc, 0xbbe9, 0xe6be, 0xe6ba, 0xc0b7,
  0xd3a4, 0xe6bf, 0xc9f4, 0xe6c3, 0xe6c4, 0xd0f6, 0xc3bd, 0xc3c4,
  0xe6c2, 0xe6c1, 0xe6c7, 0xcfb1, 0xebf4, 0xe6ca, 0xe6c5, 0xbcde,
  0xc9a9, 0xbcb5, 0xcfd3, 0xe6c8, 0xe6c9, 0xe6ce, 0xe6d0, 0xe6d1,
  0xe6cb, 0xb5d5, 0xe6cc, 0xe6cf, 0xc4db, 0xe6c6, 0xe6cd, 0xe6d2,
  0xe6d4, 0xe6d3, 0xe6d5, 0xd9f8, 0xe6d6, 0xe6d7, 0xd7d3, 0xe6dd,
  0xe6de, 0xbfd7, 0xd4d0, 0xd7d6, 0xb4e6, 0xcbef, 0xe6da, 0xd8c3,
  0xd7ce, 0xd0a2, 0xc3cf, 0xe6df, 0xbcbe, 0xb9c2, 0xe6db, 0xd1a7,
  0xbaa2, 0xc2cf, 0xd8ab, 0xcaeb, 0xe5ee, 0xe6dc, 0xb7f5, 0xc8e6,
  0xc4f5, 0xe5b2, 0xc4fe, 0xcbfc, 0xe5b3, 0xd5ac, 0xd3ee, 0xcad8,
  0xb0b2, 0xcbce, 0xcdea, 0xbaea, 0xe5b5, 0xe5b4, 0xd7da, 0xb9d9,
  0xd6e6, 0xb6a8, 0xcdf0, 0xd2cb, 0xb1a6, 0xcab5, 0xb3e8, 0xc9f3,
  0xbfcd, 0xd0fb, 0xcad2, 0xe5b6, 0xbbc2, 0xcfdc, 0xb9ac, 0xd4d7,
  0xbaa6, 0xd1e7, 0xcffc, 0xbcd2, 0xe5b7, 0xc8dd, 0xbfed, 0xb1f6,
  0xcbde, 0xbcc5, 0xbcc4, 0xd2fa, 0xc3dc, 0xbfdc, 0xb8bb, 0xc3c2,
  0xbaae, 0xd4a2, 0xc7de, 0xc4af, 0xb2ec, 0xb9d1, 0xe5bb, 0xc1c8,
  0xd5af, 0xe5bc, 0xe5be, 0xb4e7, 0xb6d4, 0xcbc2, 0xd1b0, 0xb5bc,
  0xcad9, 0xb7e2, 0xc9e4, 0xbdab, 0xcebe, 0xd7f0, 0xd0a1, 0xc9d9,
  0xb6fb, 0xe6d8, 0xbce2, 0xb3be, 0xc9d0, 0xe6d9, 0xb3a2, 0xdecc,
  0xd3c8, 0xdecd, 0xd2a2, 0xdece, 0xbecd, 0xdecf, 0xcaac, 0xd2fc,
  0xb3df, 0xe5ea, 0xc4e1, 0xbea1, 0xceb2, 0xc4f2, 0xbed6, 0xc6a8,
  0xb2e3, 0xbed3, 0xc7fc, 0xcceb, 0xbdec, 0xcedd, 0xcaba, 0xc6c1,
  0xe5ec, 0xd0bc, 0xd5b9, 0xe5ed, 0xcaf4, 0xcdc0, 0xc2c5, 0xe5ef,
  0xc2c4, 0xe5f0, 0xe5f8, 0xcdcd, 0xc9bd, 0xd2d9, 0xe1a8, 0xd3ec,
  0xcbea, 0xc6f1, 0xe1ac, 0xe1a7, 0xe1a9, 0xe1aa, 0xe1af, 0xb2ed,
  0xe1ab, 0xb8da, 0xe1ad, 0xe1ae, 0xe1b0, 0xb5ba, 0xe1b1, 0xe1b3,
  0xe1b8, 0xd1d2, 0xe1b6, 0xe1b5, 0xc1eb, 0xe1b7, 0xd4c0, 0xe1b2,
  0xe1ba, 0xb0b6, 0xe1b4, 0xbff9, 0xe1b9, 0xe1bb, 0xe1be, 0xe1bc,
  0xd6c5, 0xcfbf, 0xe1bd, 0xe1bf, 0xc2cd, 0xb6eb, 0xd3f8, 0xc7cd,
  0xb7e5, 0xbefe, 0xe1c0, 0xe1c1, 0xe1c7, 0xb3e7, 0xc6e9, 0xb4de,
  0xd1c2, 0xe1c8, 0xe1c6, 0xe1c5, 0xe1c3, 0xe1c2, 0xb1c0, 0xd5b8,
  0xe1c4, 0xe1cb, 0xe1cc, 0xe1ca, 0xeffa, 0xe1d3, 0xe1d2, 0xc7b6,
  0xe1c9, 0xe1ce, 0xe1d0, 0xe1d4, 0xe1d1, 0xe1cd, 0xe1cf, 0xe1d5,
  0xe1d6, 0xe1d7, 0xe1d8, 0xe1da, 0xe1db, 0xcea1, 0xe7dd, 0xb4a8,
  0xd6dd, 0xd1b2, 0xb3b2, 0xb9a4, 0xd7f3, 0xc7c9, 0xbede, 0xb9ae,
  0xced7, 0xb2ee, 0xdbcf, 0xbcba, 0xd2d1, 0xcbc8, 0xb0cd, 0xcfef,
  0xd9e3, 0xbded, 0xb1d2, 0xcad0, 0xb2bc, 0xcba7, 0xb7ab, 0xcaa6,
  0xcfa3, 0xe0f8, 0xd5ca, 0xe0fb, 0xe0fa, 0xc5c1, 0xccfb, 0xc1b1,
  0xe0f9, 0xd6e3, 0xb2af, 0xd6c4, 0xb5db, 0xb4f8, 0xd6a1, 0xcfaf,
  0xb0ef, 0xe0fc, 0xe1a1, 0xb3a3, 0xe0fd, 0xe0fe, 0xc3b1, 0xc3dd,
  0xe1a2, 0xb7f9, 0xbbcf, 0xe1a3, 0xc4bb, 0xe1a4, 0xe1a5, 0xe1a6,
  0xb4b1, 0xb8c9, 0xc6bd, 0xc4ea, 0xb2a2, 0xd0d2, 0xe7db, 0xbbc3,
  0xd3d7, 0xd3c4, 0xb9e3, 0xe2cf, 0xd7af, 0xc7ec, 0xb1d3, 0xb4b2,
  0xe2d1, 0xd0f2, 0xc2ae, 0xe2d0, 0xbfe2, 0xd3a6, 0xb5d7, 0xe2d2,
  0xb5ea, 0xc3ed, 0xb8fd, 0xb8ae, 0xc5d3, 0xb7cf, 0xe2d4, 0xe2d3,
  0xb6c8, 0xd7f9, 0xcda5, 0xe2d8, 0xe2d6, 0xcafc, 0xbfb5, 0xd3b9,
  0xe2d5, 0xe2d7, 0xc1ae, 0xc0c8, 0xe2db, 0xe2da, 0xc0aa, 0xc1ce,
  0xe2dc, 0xe2dd, 0xe2de, 0xdbc8, 0xd1d3, 0xcda2, 0xbda8, 0xdec3,
  0xd8a5, 0xbfaa, 0xdbcd, 0xd2ec, 0xc6fa, 0xc5aa, 0xdec4, 0xb1d7,
  0xdfae, 0xcabd, 0xdfb1, 0xb9ad, 0xd2fd, 0xb8a5, 0xbaeb, 0xb3da,
  0xb5dc, 0xd5c5, 0xc3d6, 0xcfd2, 0xbba1, 0xe5f3, 0xe5f2, 0xe5f4,
  0xcde4, 0xc8f5, 0xb5af, 0xc7bf, 0xe5f6, 0xecb0, 0xe5e6, 0xb9e9,
  0xb5b1, 0xc2bc, 0xe5e8, 0xe5e7, 0xe5e9, 0xd2cd, 0xe1ea, 0xd0ce,
  0xcdae, 0xd1e5, 0xb2ca, 0xb1eb, 0xb1f2, 0xc5ed, 0xd5c3, 0xd3b0,
  0xe1dc, 0xe1dd, 0xd2db, 0xb3b9, 0xb1cb, 0xcdf9, 0xd5f7, 0xe1de,
  0xbeb6, 0xb4fd, 0xe1df, 0xbadc, 0xe1e0, 0xbbb2, 0xc2c9, 0xe1e1,
  0xd0ec, 0xcdbd, 0xe1e2, 0xb5c3, 0xc5c7, 0xe1e3, 0xe1e4, 0xd3f9,
  0xe1e5, 0xd1ad, 0xe1e6, 0xcea2, 0xe1e7, 0xb5c2, 0xe1e8, 0xbbd5,
  0xd0c4, 0xe2e0, 0xb1d8, 0xd2e4, 0xe2e1, 0xbcc9, 0xc8cc, 0xe2e3,
  0xecfe, 0xecfd, 0xdfaf, 0xe2e2, 0xd6be, 0xcdfc, 0xc3a6, 0xe3c3,
  0xd6d2, 0xe2e7, 0xe2e8, 0xd3c7, 0xe2ec, 0xbfec, 0xe2ed, 0xe2e5,
  0xb3c0, 0xc4ee, 0xe2ee, 0xd0c3, 0xbaf6, 0xe2e9, 0xb7de, 0xbbb3,
  0xccac, 0xcbcb, 0xe2e4, 0xe2e6, 0xe2ea, 0xe2eb, 0xe2f7, 0xe2f4,
  0xd4f5, 0xe2f3, 0xc5ad, 0xd5fa, 0xc5c2, 0xb2c0, 0xe2ef, 0xe2f2,
  0xc1af, 0xcbbc, 0xb5a1, 0xe2f9, 0xbcb1, 0xe2f1, 0xd0d4, 0xd4b9,
  0xe2f5, 0xb9d6, 0xe2f6, 0xc7d3, 0xe2f0, 0xd7dc, 0xeda1, 0xe2f8,
  0xeda5, 0xe2fe, 0xcad1, 0xc1b5, 0xbbd0, 0xbfd6, 0xbae3, 0xcba1,
  0xeda6, 0xeda3, 0xeda2, 0xbbd6, 0xeda7, 0xd0f4, 0xeda4, 0xbade,
  0xb6f7, 0xe3a1, 0xb6b2, 0xccf1, 0xb9a7, 0xcfa2, 0xc7a1, 0xbfd2,
  0xb6f1, 0xe2fa, 0xe2fb, 0xe2fd, 0xe2fc, 0xc4d5, 0xe3a2, 0xd3c1,
  0xe3a7, 0xc7c4, 0xcfa4, 0xe3a9, 0xbab7, 0xe3a8, 0xbbda, 0xe3a3,
  0xe3a4, 0xe3aa, 0xe3a6, 0xcef2, 0xd3c6, 0xbbbc, 0xd4c3, 0xc4fa,
  0xeda8, 0xd0fc, 0xe3a5, 0xc3f5, 0xe3ad, 0xb1af, 0xe3b2, 0xbcc2,
  0xe3ac, 0xb5bf, 0xc7e9, 0xe3b0, 0xbeaa, 0xcdef, 0xbbf3, 0xcce8,
  0xe3af, 0xe3b1, 0xcfa7, 0xe3ae, 0xcea9, 0xbbdd, 0xb5eb, 0xbee5,
  0xb2d2, 0xb3cd, 0xb1b9, 0xe3ab, 0xb2d1, 0xb5ac, 0xb9df, 0xb6e8,
  0xcfeb, 0xe3b7, 0xbbcc, 0xc8c7, 0xd0ca, 0xe3b8, 0xb3ee, 0xeda9,
  0xd3fa, 0xd3e4, 0xedaa, 0xe3b9, 0xd2e2, 0xe3b5, 0xd3de, 0xb8d0,
  0xe3b3, 0xe3b6, 0xb7df, 0xe3b4, 0xc0a2, 0xe3ba, 0xd4b8, 0xb4c8,
  0xe3bb, 0xbbc5, 0xc9f7, 0xc9e5, 0xc4bd, 0xedab, 0xc2fd, 0xbbdb,
  0xbfae, 0xcebf, 0xe3bc, 0xbfb6, 0xb1ef, 0xd4f7, 0xe3be, 0xedad,
  0xe3bf, 0xbaa9, 0xedac, 0xe3bd, 0xe3c0, 0xbab6, 0xb6ae, 0xd0b8,
  0xb0c3, 0xedae, 0xedaf, 0xc0c1, 0xe3c1, 0xc5b3, 0xe3c2, 0xdcb2,
  0xedb0, 0xb8ea, 0xceec, 0xeaa7, 0xd0e7, 0xcaf9, 0xc8d6, 0xcfb7,
  0xb3c9, 0xced2, 0xbde4, 0xe3de, 0xbbf2, 0xeaa8, 0xd5bd, 0xc6dd,
  0xeaa9, 0xeaaa, 0xeaac, 0xeaab, 0xeaae, 0xeaad, 0xbdd8, 0xeaaf,
  0xc2be, 0xb4c1, 0xb4f7, 0xbba7, 0xece6, 0xece5, 0xb7bf, 0xcbf9,
  0xb1e2, 0xece7, 0xc9c8, 0xece8, 0xece9, 0xcad6, 0xded0, 0xb2c5,
  0xd4fa, 0xc6cb, 0xb0c7, 0xb4f2, 0xc8d3, 0xcdd0, 0xbfb8, 0xbfdb,
  0xc7a4, 0xd6b4, 0xc0a9, 0xded1, 0xc9a8, 0xd1ef, 0xc5a4, 0xb0e7,
  0xb3b6, 0xc8c5, 0xb0e2, 0xb7f6, 0xc5fa, 0xb6f3, 0xd5d2, 0xb3d0,
  0xbcbc, 0xb3ad, 0xbef1, 0xb0d1, 0xd2d6, 0xcae3, 0xd7a5, 0xcdb6,
  0xb6b6, 0xbfb9, 0xd5db, 0xb8a7, 0xc5d7, 0xded2, 0xbfd9, 0xc2d5,
  0xc7c0, 0xbba4, 0xb1a8, 0xc5ea, 0xc5fb, 0xcca7, 0xb1a7, 0xb5d6,
  0xc4a8, 0xded3, 0xd1ba, 0xb3e9, 0xc3f2, 0xb7f7, 0xd6f4, 0xb5a3,
  0xb2f0, 0xc4b4, 0xc4e9, 0xc0ad, 0xded4, 0xb0e8, 0xc5c4, 0xc1e0,
  0xb9d5, 0xbedc, 0xcdd8, 0xb0ce, 0xcdcf, 0xded6, 0xbed0, 0xd7be,
  0xded5, 0xd5d0, 0xb0dd, 0xc4e2, 0xc2a3, 0xbcf0, 0xd3b5, 0xc0b9,
  0xc5a1, 0xb2a6, 0xd4f1, 0xc0a8, 0xcac3, 0xded7, 0xd5fc, 0xb9b0,
  0xc8ad, 0xcba9, 0xded9, 0xbfbd, 0xc6b4, 0xd7a7, 0xcab0, 0xc4c3,
  0xb3d6, 0xb9d2, 0xd6b8, 0xeafc, 0xb0b4, 0xbfe6, 0xccf4, 0xcdda,
  0xd6bf, 0xc2ce, 0xcece, 0xcca2, 0xd0ae, 0xc4d3, 0xb5b2, 0xded8,
  0xd5f5, 0xbcb7, 0xbbd3, 0xb0a4, 0xc5b2, 0xb4ec, 0xd5f1, 0xeafd,
  0xdeda, 0xcda6, 0xcdec, 0xcee6, 0xdedc, 0xcdb1, 0xc0a6, 0xd7bd,
  0xdedb, 0xb0c6, 0xbab4, 0xc9d3, 0xc4f3, 0xbee8, 0xb2b6, 0xc0cc,
  0xcbf0, 0xbcf1, 0xbbbb, 0xb5b7, 0xc5f5, 0xdee6, 0xdee3, 0xbedd,
  0xdedf, 0xb4b7, 0xbddd, 0xdee0, 0xc4ed, 0xcfc6, 0xb5e0, 0xb6de,
  0xcada, 0xb5f4, 0xdee5, 0xd5c6, 0xdee1, 0xcccd, 0xc6fe, 0xc5c5,
  0xd2b4, 0xbef2, 0xc2d3, 0xccbd, 0xb3b8, 0xbdd3, 0xbfd8, 0xcdc6,
  0xd1da, 0xb4eb, 0xdee4, 0xdedd, 0xdee7, 0xeafe, 0xc2b0, 0xdee2,
  0xd6c0, 0xb5a7, 0xb2f4, 0xdee8, 0xdef2, 0xdeed, 0xdef1, 0xc8e0,
  0xd7e1, 0xdeef, 0xc3e8, 0xcce1, 0xb2e5, 0xd2be, 0xdeee, 0xdeeb,
  0xced5, 0xb4a7, 0xbfab, 0xbebe, 0xbdd2, 0xdee9, 0xd4ae, 0xdede,
  0xdeea, 0xc0bf, 0xdeec, 0xb2f3, 0xb8e9, 0xc2a7, 0xbdc1, 0xdef5,
  0xdef8, 0xb2ab, 0xb4a4, 0xb4ea, 0xc9a6, 0xdef6, 0xcbd1, 0xb8e3,
  0xdef7, 0xdefa, 0xdef9, 0xccc2, 0xb0e1, 0xb4ee, 0xe5ba, 0xd0af,
  0xb2eb, 0xeba1, 0xdef4, 0xc9e3, 0xdef3, 0xb0da, 0xd2a1, 0xb1f7,
  0xccaf, 0xdef0, 0xcba4, 0xd5aa, 0xdefb, 0xb4dd, 0xc4a6, 0xdefd,
  0xc3fe, 0xc4a1, 0xdfa1, 0xc1cc, 0xdefc, 0xbeef, 0xc6b2, 0xb3c5,
  0xc8f6, 0xcbba, 0xdefe, 0xdfa4, 0xd7b2, 0xb3b7, 0xc1c3, 0xc7cb,
  0xb2a5, 0xb4e9, 0xd7ab, 0xc4ec, 0xdfa2, 0xdfa3, 0xdfa5, 0xbab3,
  0xdfa6, 0xc0de, 0xc9c3, 0xb2d9, 0xc7e6, 0xdfa7, 0xc7dc, 0xdfa8,
  0xeba2, 0xcbd3, 0xdfaa, 0xdfa9, 0xb2c1, 0xc5ca, 0xdfab, 0xd4dc,
  0xc8c1, 0xdfac, 0xbef0, 0xdfad, 0xd6a7, 0xeab7, 0xebb6, 0xcad5,
  0xd8fc, 0xb8c4, 0xb9a5, 0xb7c5, 0xd5fe, 0xb9ca, 0xd0a7, 0xf4cd,
  0xb5d0, 0xc3f4, 0xbec8, 0xebb7, 0xb0bd, 0xbdcc, 0xc1b2, 0xb1d6,
  0xb3a8, 0xb8d2, 0xc9a2, 0xb6d8, 0xebb8, 0xbeb4, 0xcafd, 0xc7c3,
  0xd5fb, 0xb7f3, 0xcec4, 0xd5ab, 0xb1f3, 0xecb3, 0xb0df, 0xecb5,
  0xb6b7, 0xc1cf, 0xf5fa, 0xd0b1, 0xd5e5, 0xced3, 0xbdef, 0xb3e2,
  0xb8ab, 0xd5b6, 0xedbd, 0xb6cf, 0xcbb9, 0xd0c2, 0xb7bd, 0xecb6,
  0xcaa9, 0xc5d4, 0xecb9, 0xecb8, 0xc2c3, 0xecb7, 0xd0fd, 0xecba,
  0xecbb, 0xd7e5, 0xecbc, 0xecbd, 0xc6ec, 0xcede, 0xbcc8, 0xc8d5,
  0xb5a9, 0xbec9, 0xd6bc, 0xd4e7, 0xd1ae, 0xd0f1, 0xeab8, 0xeab9,
  0xeaba, 0xbab5, 0xcab1, 0xbff5, 0xcdfa, 0xeac0, 0xb0ba, 0xeabe,
  0xc0a5, 0xeabb, 0xb2fd, 0xc3f7, 0xbbe8, 0xd2d7, 0xcef4, 0xeabf,
  0xeabc, 0xeac3, 0xd0c7, 0xd3b3, 0xb4ba, 0xc3c1, 0xd7f2, 0xd5d1,
  0xcac7, 0xeac5, 0xeac4, 0xeac7, 0xeac6, 0xd6e7, 0xcfd4, 0xeacb,
  0xbbce, 0xbdfa, 0xc9ce, 0xeacc, 0xc9b9, 0xcffe, 0xeaca, 0xd4ce,
  0xeacd, 0xeacf, 0xcded, 0xeac9, 0xeace, 0xceee, 0xbbde, 0xb3bf,
  0xc6d5, 0xbeb0, 0xcefa, 0xc7e7, 0xbea7, 0xead0, 0xd6c7, 0xc1c0,
  0xd4dd, 0xead1, 0xcfbe, 0xead2, 0xcaee, 0xc5af, 0xb0b5, 0xead4,
  0xead3, 0xf4df, 0xc4ba, 0xb1a9, 0xe5df, 0xead5, 0xcaef, 0xead6,
  0xead7, 0xc6d8, 0xead8, 0xead9, 0xd4bb, 0xc7fa, 0xd2b7, 0xb8fc,
  0xeac2, 0xb2dc, 0xc2fc, 0xd4f8, 0xcce6, 0xd7ee, 0xd4c2, 0xd3d0,
  0xebc3, 0xc5f3, 0xb7fe, 0xebd4, 0xcbb7, 0xebde, 0xc0ca, 0xcdfb,
  0xb3af, 0xc6da, 0xebfc, 0xc4be, 0xceb4, 0xc4a9, 0xb1be, 0xd4fd,
  0xcaf5, 0xd6ec, 0xc6d3, 0xb6e4, 0xbbfa, 0xd0e0, 0xc9b1, 0xd4d3,
  0xc8a8, 0xb8cb, 0xe8be, 0xc9bc, 0xe8bb, 0xc0ee, 0xd0d3, 0xb2c4,
  0xb4e5, 0xe8bc, 0xd5c8, 0xb6c5, 0xe8bd, 0xcaf8, 0xb8dc, 0xccf5,
  0xc0b4, 0xd1ee, 0xe8bf, 0xe8c2, 0xbabc, 0xb1ad, 0xbddc, 0xeabd,
  0xe8c3, 0xe8c6, 0xe8cb, 0xe8cc, 0xcbc9, 0xb0e5, 0xbcab, 0xb9b9,
  0xe8c1, 0xcdf7, 0xe8ca, 0xcef6, 0xd5ed, 0xc1d6, 0xe8c4, 0xc3b6,
  0xb9fb, 0xd6a6, 0xe8c8, 0xcae0, 0xd4e6, 0xe8c0, 0xe8c5, 0xe8c7,
  0xc7b9, 0xb7e3, 0xe8c9, 0xbfdd, 0xe8d2, 0xe8d7, 0xe8d5, 0xbcdc,
  0xbccf, 0xe8db, 0xe8de, 0xe8da, 0xb1fa, 0xb0d8, 0xc4b3, 0xb8cc,
  0xc6e2, 0xc8be, 0xc8e1, 0xe8cf, 0xe8d4, 0xe8d6, 0xb9f1, 0xe8d8,
  0xd7f5, 0xc4fb, 0xe8dc, 0xb2e9, 0xe8d1, 0xbced, 0xbfc2, 0xe8cd,
  0xd6f9, 0xc1f8, 0xb2f1, 0xe8df, 0xcac1, 0xe8d9, 0xd5a4, 0xb1ea,
  0xd5bb, 0xe8ce, 0xe8d0, 0xb6b0, 0xe8d3, 0xe8dd, 0xc0b8, 0xcaf7,
  0xcba8, 0xc6dc, 0xc0f5, 0xe8e9, 0xd0a3, 0xe8f2, 0xd6ea, 0xe8e0,
  0xe8e1, 0xd1f9, 0xbacb, 0xb8f9, 0xb8f1, 0xd4d4, 0xe8ef, 0xe8ee,
  0xe8ec, 0xb9f0, 0xccd2, 0xe8e6, 0xcea6, 0xbff2, 0xb0b8, 0xe8f1,
  0xe8f0, 0xd7c0, 0xe8e4, 0xcda9, 0xc9a3, 0xbbb8, 0xbddb, 0xe8ea,
  0xe8e2, 0xe8e3, 0xe8e5, 0xb5b5, 0xe8e7, 0xc7c5, 0xe8eb, 0xe8ed,
  0xbdb0, 0xd7ae, 0xe8f8, 0xe8f5, 0xcdb0, 0xe8f6, 0xc1ba, 0xe8e8,
  0xc3b7, 0xb0f0, 0xe8f4, 0xe8f7, 0xb9a3, 0xc9d2, 0xc3ce, 0xcee0,
  0xc0e6, 0xcbf3, 0xccdd, 0xd0b5, 0xcae1, 0xe8f3, 0xbcec, 0xe8f9,
  0xc3de, 0xc6e5, 0xb9f7, 0xb0f4, 0xd7d8, 0xbcac, 0xc5ef, 0xccc4,
  0xe9a6, 0xc9ad, 0xe9a2, 0xc0e2, 0xbfc3, 0xe8fe, 0xb9d7, 0xe8fb,
  0xe9a4, 0xd2ce, 0xe9a3, 0xd6b2, 0xd7b5, 0xe9a7, 0xbdb7, 0xe8fc,
  0xe8fd, 0xe9a1, 0xcdd6, 0xd2ac, 0xe9b2, 0xe9a9, 0xb4aa, 0xb4bb,
  0xe9ab, 0xd0a8, 0xe9a5, 0xb3fe, 0xe9ac, 0xc0e3, 0xe9aa, 0xe9b9,
  0xe9b8, 0xe9ae, 0xe8fa, 0xe9a8, 0xbfac, 0xe9b1, 0xe9ba, 0xc2a5,
  0xe9af, 0xb8c5, 0xe9ad, 0xd3dc, 0xe9b4, 0xe9b5, 0xe9b7, 0xe9c7,
  0xc0c6, 0xe9c5, 0xe9b0, 0xe9bb, 0xb0f1, 0xe9bc, 0xd5a5, 0xe9be,
  0xe9bf, 0xe9c1, 0xc1f1, 0xc8b6, 0xe9bd, 0xe9c2, 0xe9c3, 0xe9b3,
  0xe9b6, 0xbbb1, 0xe9c0, 0xbcf7, 0xe9c4, 0xe9c6, 0xe9ca, 0xe9ce,
  0xb2db, 0xe9c8, 0xb7ae, 0xe9cb, 0xe9cc, 0xd5c1, 0xc4a3, 0xe9d8,
  0xbae1, 0xe9c9, 0xd3a3, 0xe9d4, 0xe9d7, 0xe9d0, 0xe9cf, 0xc7c1,
  0xe9d2, 0xe9d9, 0xb3c8, 0xe9d3, 0xcff0, 0xe9cd, 0xb3f7, 0xe9d6,
  0xe9da, 0xccb4, 0xcfad, 0xe9d5, 0xe9dc, 0xe9db, 0xe9de, 0xe9d1,
  0xe9dd, 0xe9df, 0xc3ca, 0xc7b7, 0xb4ce, 0xbbb6, 0xd0c0, 0xeca3,
  0xc5b7, 0xd3fb, 0xeca4, 0xeca5, 0xc6db, 0xbfee, 0xeca6, 0xeca7,
  0xd0aa, 0xc7b8, 0xb8e8, 0xeca8, 0xd6b9, 0xd5fd, 0xb4cb, 0xb2bd,
  0xcee4, 0xc6e7, 0xcde1, 0xb4f5, 0xcbc0, 0xbcdf, 0xe9e2, 0xe9e3,
  0xd1ea, 0xe9e5, 0xb4f9, 0xe9e4, 0xd1b3, 0xcae2, 0xb2d0, 0xe9e8,
  0xe9e6, 0xe9e7, 0xd6b3, 0xe9e9, 0xe9ea, 0xe9eb, 0xe9ec, 0xecaf,
  0xc5b9, 0xb6ce, 0xd2f3, 0xb5ee, 0xbbd9, 0xecb1, 0xd2e3, 0xcee3,
  0xc4b8, 0xc3bf, 0xb6be, 0xd8b9, 0xb1c8, 0xb1cf, 0xb1d1, 0xc5fe,
  0xb1d0, 0xc3ab, 0xd5b1, 0xeba4, 0xbac1, 0xccba, 0xeba5, 0xeba7,
  0xeba8, 0xeba6, 0xeba9, 0xebab, 0xebaa, 0xebac, 0xcacf, 0xd8b5,
  0xc3f1, 0xc3a5, 0xc6f8, 0xebad, 0xc4ca, 0xebae, 0xebaf, 0xebb0,
  0xb7d5, 0xb7fa, 0xebb1, 0xc7e2, 0xebb3, 0xbaa4, 0xd1f5, 0xb0b1,
  0xebb2, 0xebb4, 0xb5aa, 0xc2c8, 0xc7e8, 0xebb5, 0xcbae, 0xe3df,
  0xd3c0, 0xd9db, 0xcda1, 0xd6ad, 0xc7f3, 0xd9e0, 0xbbe3, 0xbaba,
  0xe3e2, 0xcfab, 0xe3e0, 0xc9c7, 0xbab9, 0xd1b4, 0xe3e1, 0xc8ea,
  0xb9af, 0xbdad, 0xb3d8, 0xcedb, 0xccc0, 0xe3e8, 0xe3e9, 0xcdf4,
  0xccad, 0xbcb3, 0xe3ea, 0xe3eb, 0xd0da, 0xc6fb, 0xb7da, 0xc7df,
  0xd2ca, 0xced6, 0xe3e4, 0xe3ec, 0xc9f2, 0xb3c1, 0xe3e7, 0xc6e3,
  0xe3e5, 0xedb3, 0xe3e6, 0xc9b3, 0xc5e6, 0xb9b5, 0xc3bb, 0xe3e3,
  0xc5bd, 0xc1a4, 0xc2d9, 0xb2d7, 0xe3ed, 0xbba6, 0xc4ad, 0xe3f0,
  0xbeda, 0xe3fb, 0xe3f5, 0xbad3, 0xb7d0, 0xd3cd, 0xd6ce, 0xd5d3,
  0xb9c1, 0xd5b4, 0xd1d8, 0xd0b9, 0xc7f6, 0xc8aa, 0xb2b4, 0xc3da,
  0xe3ee, 0xe3fc, 0xe3ef, 0xb7a8, 0xe3f7, 0xe3f4, 0xb7ba, 0xc5a2,
  0xe3f6, 0xc5dd, 0xb2a8, 0xc6fc, 0xc4e0, 0xd7a2, 0xc0e1, 0xe3f9,
  0xe3fa, 0xe3fd, 0xcca9, 0xe3f3, 0xd3be, 0xb1c3, 0xedb4, 0xe3f1,
  0xe3f2, 0xe3f8, 0xd0ba, 0xc6c3, 0xd4f3, 0xe3fe, 0xbde0, 0xe4a7,
  0xe4a6, 0xd1f3, 0xe4a3, 0xe4a9, 0xc8f7, 0xcfb4, 0xe4a8, 0xe4ae,
  0xc2e5, 0xb6b4, 0xbdf2, 0xe4a2, 0xbae9, 0xe4aa, 0xe4ac, 0xb6fd,
  0xd6de, 0xe4b2, 0xe4ad, 0xe4a1, 0xbbee, 0xcddd, 0xc7a2, 0xc5c9,
  0xc1f7, 0xe4a4, 0xc7b3, 0xbdac, 0xbdbd, 0xe4a5, 0xd7c7, 0xb2e2,
  0xe4ab, 0xbcc3, 0xe4af, 0xbbeb, 0xe4b0, 0xc5a8, 0xe4b1, 0xd5e3,
  0xbfa3, 0xe4ba, 0xe4b7, 0xe4bb, 0xe4bd, 0xc6d6, 0xbac6, 0xc0cb,
  0xb8a1, 0xe4b4, 0xd4a1, 0xbaa3, 0xbdfe, 0xe4bc, 0xcdbf, 0xc4f9,
  0xcffb, 0xc9e6, 0xd3bf, 0xcfd1, 0xe4b3, 0xe4b8, 0xe4b9, 0xcce9,
  0xccce, 0xc0d4, 0xe4b5, 0xc1b0, 0xe4b6, 0xced0, 0xbbc1, 0xb5d3,
  0xc8f3, 0xbda7, 0xd5c7, 0xc9ac, 0xb8a2, 0xe4ca, 0xe4cc, 0xd1c4,
  0xd2ba, 0xbaad, 0xbad4, 0xe4c3, 0xb5ed, 0xd7cd, 0xe4c0, 0xcffd,
  0xe4bf, 0xc1dc, 0xccca, 0xcae7, 0xc4d7, 0xccd4, 0xe4c8, 0xe4c7,
  0xe4c1, 0xe4c4, 0xb5ad, 0xd3d9, 0xe4c6, 0xd2f9, 0xb4e3, 0xbbb4,
  0xc9ee, 0xb4be, 0xbbec, 0xd1cd, 0xcced, 0xedb5, 0xc7e5, 0xd4a8,
  0xe4cb, 0xd7d5, 0xe4c2, 0xbda5, 0xe4c5, 0xd3e6, 0xe4c9, 0xc9f8,
  0xe4be, 0xd3e5, 0xc7fe, 0xb6c9, 0xd4fc, 0xb2b3, 0xe4d7, 0xcec2,
  0xe4cd, 0xcebc, 0xb8db, 0xe4d6, 0xbfca, 0xd3ce, 0xc3ec, 0xc5c8,
  0xe4d8, 0xcdc4, 0xe4cf, 0xe4d4, 0xe4d5, 0xbafe, 0xcfe6, 0xd5bf,
  0xe4d2, 0xe4d0, 0xe4ce, 0xcde5, 0xcaaa, 0xc0a3, 0xbda6, 0xe4d3,
  0xb8c8, 0xe4e7, 0xd4b4, 0xe4db, 0xc1ef, 0xe4e9, 0xd2e7, 0xe4df,
  0xe4e0, 0xcfaa, 0xcbdd, 0xe4da, 0xe4d1, 0xe4e5, 0xc8dc, 0xe4e3,
  0xc4e7, 0xe4e2, 0xe4e1, 0xb3fc, 0xe4e8, 0xb5e1, 0xd7cc, 0xe4e6,
  0xbbac, 0xd7d2, 0xcccf, 0xebf8, 0xe4e4, 0xb9f6, 0xd6cd, 0xe4d9,
  0xe4dc, 0xc2fa, 0xe4de, 0xc2cb, 0xc0c4, 0xc2d0, 0xb1f5, 0xccb2,
  0xb5ce, 0xe4ef, 0xc6af, 0xc6e1, 0xe4f5, 0xc2a9, 0xc0ec, 0xd1dd,
  0xe4ee, 0xc4ae, 0xe4ed, 0xe4f6, 0xe4f4, 0xc2fe, 0xe4dd, 0xe4f0,
  0xcafe, 0xd5c4, 0xe4f1, 0xd1fa, 0xe4eb, 0xe4ec, 0xe4f2, 0xceab,
  0xc5cb, 0xc7b1, 0xc2ba, 0xe4ea, 0xc1ca, 0xccb6, 0xb3b1, 0xe4fb,
  0xe4f3, 0xe4fa, 0xe4fd, 0xe4fc, 0xb3ce, 0xb3ba, 0xe4f7, 0xe4f9,
  0xe4f8, 0xc5ec, 0xc0bd, 0xd4e8, 0xe5a2, 0xb0c4, 0xe5a4, 0xe5a3,
  0xbca4, 0xe5a5, 0xe5a1, 0xe4fe, 0xb1f4, 0xe5a8, 0xe5a9, 0xe5a6,
  0xe5a7, 0xe5aa, 0xc6d9, 0xe5ab, 0xe5ad, 0xe5ac, 0xe5af, 0xe5ae,
  0xb9e0, 0xe5b0, 0xe5b1, 0xbbf0, 0xece1, 0xc3f0, 0xb5c6, 0xbbd2,
  0xc1e9, 0xd4ee, 0xbec4, 0xd7c6, 0xd4d6, 0xb2d3, 0xecbe, 0xeac1,
  0xc2af, 0xb4b6, 0xd1d7, 0xb3b4, 0xc8b2, 0xbfbb, 0xecc0, 0xd6cb,
  0xecbf, 0xecc1, 0xecc5, 0xbee6, 0xccbf, 0xc5da, 0xbebc, 0xecc6,
  0xb1fe, 0xecc4, 0xd5a8, 0xb5e3, 0xecc2, 0xc1b6, 0xb3e3, 0xecc3,
  0xcbb8, 0xc0c3, 0xccfe, 0xc1d2, 0xecc8, 0xbae6, 0xc0d3, 0xd6f2,
  0xd1cc, 0xbfbe, 0xb7b3, 0xc9d5, 0xecc7, 0xbbe2, 0xcccc, 0xbdfd,
  0xc8c8, 0xcfa9, 0xcde9, 0xc5eb, 0xb7e9, 0xd1c9, 0xbab8, 0xecc9,
  0xecca, 0xbbc0, 0xeccb, 0xece2, 0xb1ba, 0xb7d9, 0xbdb9, 0xeccc,
  0xd1e6, 0xeccd, 0xc8bb, 0xecd1, 0xecd3, 0xbbcd, 0xbce5, 0xeccf,
  0xc9b7, 0xc3ba, 0xece3, 0xd5d5, 0xecd0, 0xd6f3, 0xecd2, 0xecce,
  0xecd4, 0xecd5, 0xc9bf, 0xcfa8, 0xd0dc, 0xd1ac, 0xc8db, 0xecd6,
  0xcef5, 0xcaec, 0xecda, 0xecd9, 0xb0be, 0xecd7, 0xecd8, 0xece4,
  0xc8bc, 0xc1c7, 0xecdc, 0xd1e0, 0xecdb, 0xd4ef, 0xecdd, 0xdbc6,
  0xecde, 0xb1ac, 0xecdf, 0xece0, 0xd7a6, 0xc5c0, 0xebbc, 0xb0ae,
  0xbef4, 0xb8b8, 0xd2af, 0xb0d6, 0xb5f9, 0xd8b3, 0xcbac, 0xe3dd,
  0xc6ac, 0xb0e6, 0xc5c6, 0xebb9, 0xebba, 0xebbb, 0xd1c0, 0xc5a3,
  0xeaf2, 0xc4b2, 0xc4b5, 0xc0ce, 0xeaf3, 0xc4c1, 0xceef, 0xeaf0,
  0xeaf4, 0xc9fc, 0xc7a3, 0xccd8, 0xcefe, 0xeaf5, 0xeaf6, 0xcfac,
  0xc0e7, 0xeaf7, 0xb6bf, 0xeaf8, 0xeaf9, 0xeafa, 0xeafb, 0xeaf1,
  0xc8ae, 0xe1eb, 0xb7b8, 0xe1ec, 0xe1ed, 0xd7b4, 0xe1ee, 0xe1ef,
  0xd3cc, 0xe1f1, 0xbff1, 0xe1f0, 0xb5d2, 0xb1b7, 0xe1f3, 0xe1f2,
  0xbafc, 0xe1f4, 0xb9b7, 0xbed1, 0xc4fc, 0xbadd, 0xbdc6, 0xe1f5,
  0xe1f7, 0xb6c0, 0xcfc1, 0xcaa8, 0xe1f6, 0xd5f8, 0xd3fc, 0xe1f8,
  0xe1fc, 0xe1f9, 0xe1fa, 0xc0ea, 0xe1fe, 0xe2a1, 0xc0c7, 0xe1fb,
  0xe1fd, 0xe2a5, 0xc1d4, 0xe2a3, 0xe2a8, 0xb2fe, 0xe2a2, 0xc3cd,
  0xb2c2, 0xe2a7, 0xe2a6, 0xe2a4, 0xe2a9, 0xe2ab, 0xd0c9, 0xd6ed,
  0xc3a8, 0xe2ac, 0xcfd7, 0xe2ae, 0xbaef, 0xe9e0, 0xe2ad, 0xe2aa,
  0xbbab, 0xd4b3, 0xe2b0, 0xe2af, 0xe9e1, 0xe2b1, 0xe2b2, 0xe2b3,
  0xcca1, 0xe2b4, 0xe2b5, 0xd0fe, 0xc2ca, 0xd3f1, 0xcdf5, 0xe7e0,
  0xe7e1, 0xbec1, 0xc2ea, 0xe7e4, 0xe7e3, 0xcde6, 0xc3b5, 0xe7e2,
  0xbbb7, 0xcfd6, 0xc1e1, 0xe7e9, 0xe7e8, 0xe7f4, 0xb2a3, 0xe7ea,
  0xe7e6, 0xe7ec, 0xe7eb, 0xc9ba, 0xd5e4, 0xe7e5, 0xb7a9, 0xe7e7,
  0xe7ee, 0xe7f3, 0xd6e9, 0xe7ed, 0xe7f2, 0xe7f1, 0xb0e0, 0xe7f5,
  0xc7f2, 0xc0c5, 0xc0ed, 0xc1f0, 0xe7f0, 0xe7f6, 0xcbf6, 0xe8a2,
  0xe8a1, 0xd7c1, 0xe7fa, 0xe7f9, 0xe7fb, 0xe7f7, 0xe7fe, 0xe7fd,
  0xe7fc, 0xc1d5, 0xc7d9, 0xc5fd, 0xc5c3, 0xc7ed, 0xe8a3, 0xe8a6,
  0xe8a5, 0xe8a7, 0xbaf7, 0xe7f8, 0xe8a4, 0xc8f0, 0xc9aa, 0xe8a9,
  0xb9e5, 0xd1fe, 0xe8a8, 0xe8aa, 0xe8ad, 0xe8ae, 0xc1a7, 0xe8af,
  0xe8b0, 0xe8ac, 0xe8b4, 0xe8ab, 0xe8b1, 0xe8b5, 0xe8b2, 0xe8b3,
  0xe8b7, 0xe8b6, 0xb9cf, 0xf0ac, 0xf0ad, 0xc6b0, 0xb0ea, 0xc8bf,
  0xcddf, 0xcecd, 0xeab1, 0xeab2, 0xc6bf, 0xb4c9, 0xeab3, 0xd5e7,
  0xddf9, 0xeab4, 0xeab5, 0xeab6, 0xb8ca, 0xdfb0, 0xc9f5, 0xccf0,
  0xc9fa, 0xc9fb, 0xd3c3, 0xcba6, 0xb8a6, 0xf0ae, 0xb1c2, 0xe5b8,
  0xccef, 0xd3c9, 0xbcd7, 0xc9ea, 0xb5e7, 0xc4d0, 0xb5e9, 0xeeae,
  0xbbad, 0xe7de, 0xeeaf, 0xb3a9, 0xeeb2, 0xeeb1, 0xbde7, 0xeeb0,
  0xceb7, 0xc5cf, 0xc1f4, 0xdbce, 0xeeb3, 0xd0f3, 0xc2d4, 0xc6e8,
  0xb7ac, 0xeeb4, 0xb3eb, 0xbbfb, 0xeeb5, 0xe7dc, 0xeeb6, 0xbdae,
  0xf1e2, 0xcae8, 0xd2c9, 0xf0da, 0xf0db, 0xf0dc, 0xc1c6, 0xb8ed,
  0xbece, 0xf0de, 0xc5b1, 0xf0dd, 0xd1f1, 0xf0e0, 0xb0cc, 0xbdea,
  0xd2df, 0xf0df, 0xb4af, 0xb7e8, 0xf0e6, 0xf0e5, 0xc6a3, 0xf0e1,
  0xf0e2, 0xb4c3, 0xf0e3, 0xd5ee, 0xccdb, 0xbed2, 0xbcb2, 0xf0e8,
  0xf0e7, 0xf0e4, 0xb2a1, 0xd6a2, 0xd3b8, 0xbeb7, 0xc8ac, 0xf0ea,
  0xd1f7, 0xd6cc, 0xbadb, 0xf0e9, 0xb6bb, 0xcdb4, 0xc6a6, 0xc1a1,
  0xf0eb, 0xf0ee, 0xf0ed, 0xf0f0, 0xf0ec, 0xbbbe, 0xf0ef, 0xccb5,
  0xf0f2, 0xb3d5, 0xb1d4, 0xf0f3, 0xf0f4, 0xf0f6, 0xb4e1, 0xf0f1,
  0xf0f7, 0xf0fa, 0xf0f8, 0xf0f5, 0xf0fd, 0xf0f9, 0xf0fc, 0xf0fe,
  0xf1a1, 0xcec1, 0xf1a4, 0xf1a3, 0xc1f6, 0xf0fb, 0xcadd, 0xb4f1,
  0xb1f1, 0xccb1, 0xf1a6, 0xf1a7, 0xf1ac, 0xd5ce, 0xf1a9, 0xc8b3,
  0xf1a2, 0xf1ab, 0xf1a8, 0xf1a5, 0xf1aa, 0xb0a9, 0xf1ad, 0xf1af,
  0xf1b1, 0xf1b0, 0xf1ae, 0xd1a2, 0xf1b2, 0xf1b3, 0xb9ef, 0xb5c7,
  0xb0d7, 0xb0d9, 0xd4ed, 0xb5c4, 0xbdd4, 0xbbca, 0xf0a7, 0xb8de,
  0xf0a8, 0xb0a8, 0xf0a9, 0xcdee, 0xf0aa, 0xf0ab, 0xc6a4, 0xd6e5,
  0xf1e4, 0xf1e5, 0xc3f3, 0xd3db, 0xd6d1, 0xc5e8, 0xd3af, 0xd2e6,
  0xeec1, 0xb0bb, 0xd5b5, 0xd1ce, 0xbce0, 0xbad0, 0xbff8, 0xb8c7,
  0xb5c1, 0xc5cc, 0xcaa2, 0xc3cb, 0xeec2, 0xc4bf, 0xb6a2, 0xedec,
  0xc3a4, 0xd6b1, 0xcfe0, 0xedef, 0xc5ce, 0xb6dc, 0xcaa1, 0xeded,
  0xedf0, 0xedf1, 0xc3bc, 0xbfb4, 0xedee, 0xedf4, 0xedf2, 0xd5e6,
  0xc3df, 0xedf3, 0xedf6, 0xd5a3, 0xd1a3, 0xedf5, 0xc3d0, 0xedf7,
  0xbff4, 0xbeec, 0xedf8, 0xccf7, 0xd1db, 0xd7c5, 0xd5f6, 0xedfc,
  0xedfb, 0xedf9, 0xedfa, 0xedfd, 0xbea6, 0xcbaf, 0xeea1, 0xb6bd,
  0xeea2, 0xc4c0, 0xedfe, 0xbdde, 0xb2c7, 0xb6c3, 0xeea5, 0xd8ba,
  0xeea3, 0xeea6, 0xc3e9, 0xb3f2, 0xeea7, 0xeea4, 0xcfb9, 0xeea8,
  0xc2f7, 0xeea9, 0xeeaa, 0xdeab, 0xc6b3, 0xc7c6, 0xd6f5, 0xb5c9,
  0xcbb2, 0xeeab, 0xcdab, 0xeeac, 0xd5b0, 0xeead, 0xf6c4, 0xdbc7,
  0xb4a3, 0xc3ac, 0xf1e6, 0xcab8, 0xd2d3, 0xd6aa, 0xeff2, 0xbed8,
  0xbdc3, 0xeff3, 0xb6cc, 0xb0ab, 0xcaaf, 0xedb6, 0xedb7, 0xcef9,
  0xb7af, 0xbff3, 0xedb8, 0xc2eb, 0xc9b0, 0xedb9, 0xc6f6, 0xbfb3,
  0xedbc, 0xc5f8, 0xd1d0, 0xd7a9, 0xedba, 0xedbb, 0xd1e2, 0xedbf,
  0xedc0, 0xedc4, 0xedc8, 0xedc6, 0xedce, 0xd5e8, 0xedc9, 0xedc7,
  0xedbe, 0xc5e9, 0xc6c6, 0xc9e9, 0xd4d2, 0xedc1, 0xedc2, 0xedc3,
  0xedc5, 0xc0f9, 0xb4a1, 0xb9e8, 0xedd0, 0xedd1, 0xedca, 0xedcf,
  0xcef8, 0xcbb6, 0xedcc, 0xedcd, 0xcff5, 0xedd2, 0xc1f2, 0xd3b2,
  0xedcb, 0xc8b7, 0xbcef, 0xc5f0, 0xedd6, 0xb5ef, 0xc2b5, 0xb0ad,
  0xcbe9, 0xb1ae, 0xedd4, 0xcdeb, 0xb5e2, 0xedd5, 0xedd3, 0xedd7,
  0xb5fa, 0xedd8, 0xedd9, 0xeddc, 0xb1cc, 0xc5f6, 0xbcee, 0xedda,
  0xccbc, 0xb2ea, 0xeddb, 0xc4eb, 0xb4c5, 0xb0f5, 0xeddf, 0xc0da,
  0xb4e8, 0xc5cd, 0xeddd, 0xbfc4, 0xedde, 0xc4a5, 0xede0, 0xede1,
  0xede3, 0xc1d7, 0xbbc7, 0xbdb8, 0xede2, 0xede4, 0xede6, 0xede5,
  0xede7, 0xcabe, 0xecea, 0xc0f1, 0xc9e7, 0xeceb, 0xc6ee, 0xecec,
  0xc6ed, 0xeced, 0xecf0, 0xd7e6, 0xecf3, 0xecf1, 0xecee, 0xecef,
  0xd7a3, 0xc9f1, 0xcbee, 0xecf4, 0xecf2, 0xcfe9, 0xecf6, 0xc6b1,
  0xbcc0, 0xecf5, 0xb5bb, 0xbbf6, 0xecf7, 0xd9f7, 0xbdfb, 0xc2bb,
  0xecf8, 0xecf9, 0xb8a3, 0xecfa, 0xecfb, 0xecfc, 0xd3ed, 0xd8ae,
  0xc0eb, 0xc7dd, 0xbacc, 0xd0e3, 0xcbbd, 0xcdba, 0xb8d1, 0xb1fc,
  0xc7ef, 0xd6d6, 0xbfc6, 0xc3eb, 0xeff5, 0xc3d8, 0xd7e2, 0xeff7,
  0xb3d3, 0xc7d8, 0xd1ed, 0xd6c8, 0xeff8, 0xeff6, 0xbbfd, 0xb3c6,
  0xbdd5, 0xd2c6, 0xbbe0, 0xcfa1, 0xeffc, 0xeffb, 0xeff9, 0xb3cc,
  0xc9d4, 0xcbb0, 0xeffe, 0xb0de, 0xd6c9, 0xeffd, 0xb3ed, 0xf6d5,
  0xcec8, 0xf0a2, 0xf0a1, 0xb5be, 0xbcda, 0xbbfc, 0xb8e5, 0xc4c2,
  0xf0a3, 0xcbeb, 0xf0a6, 0xd1a8, 0xbebf, 0xc7ee, 0xf1b6, 0xf1b7,
  0xbfd5, 0xb4a9, 0xf1b8, 0xcdbb, 0xc7d4, 0xd5ad, 0xf1b9, 0xf1ba,
  0xc7cf, 0xd2a4, 0xd6cf, 0xf1bb, 0xbdd1, 0xb4b0, 0xbebd, 0xb4dc,
  0xced1, 0xbfdf, 0xf1bd, 0xbffa, 0xf1bc, 0xf1bf, 0xf1be, 0xf1c0,
  0xf1c1, 0xc1fe, 0xc1a2, 0xcafa, 0xd5be, 0xbeba, 0xbeb9, 0xd5c2,
  0xbfa2, 0xcdaf, 0xf1b5, 0xbddf, 0xb6cb, 0xd6f1, 0xf3c3, 0xf3c4,
  0xb8cd, 0xf3c6, 0xf3c7, 0xb0ca, 0xf3c5, 0xf3c9, 0xcbf1, 0xf3cb,
  0xd0a6, 0xb1ca, 0xf3c8, 0xf3cf, 0xb5d1, 0xf3d7, 0xf3d2, 0xf3d4,
  0xf3d3, 0xb7fb, 0xb1bf, 0xf3ce, 0xf3ca, 0xb5da, 0xf3d0, 0xf3d1,
  0xf3d5, 0xf3cd, 0xbce3, 0xc1fd, 0xf3d6, 0xf3da, 0xf3cc, 0xb5c8,
  0xbdee, 0xf3dc, 0xb7a4, 0xbff0, 0xd6fe, 0xcdb2, 0xb4f0, 0xb2df,
  0xf3d8, 0xf3d9, 0xc9b8, 0xf3dd, 0xf3de, 0xf3e1, 0xf3df, 0xf3e3,
  0xf3e2, 0xf3db, 0xbfea, 0xb3ef, 0xf3e0, 0xc7a9, 0xbcf2, 0xf3eb,
  0xb9bf, 0xf3e4, 0xb2ad, 0xbbfe, 0xcbe3, 0xf3ed, 0xf3e9, 0xb9dc,
  0xf3ee, 0xf3e5, 0xf3e6, 0xf3ea, 0xc2e1, 0xf3ec, 0xf3ef, 0xf3e8,
  0xbcfd, 0xcfe4, 0xf3f0, 0xf3e7, 0xf3f2, 0xd7ad, 0xc6aa, 0xf3f3,
  0xf3f1, 0xc2a8, 0xb8dd, 0xf3f5, 0xf3f4, 0xb4db, 0xf3f6, 0xf3f7,
  0xf3f8, 0xc0ba, 0xc0e9, 0xc5f1, 0xf3fb, 0xf3fa, 0xb4d8, 0xf3fe,
  0xf3f9, 0xf3fc, 0xf3fd, 0xf4a1, 0xf4a3, 0xbbc9, 0xf4a2, 0xf4a4,
  0xb2be, 0xf4a6, 0xf4a5, 0xbcae, 0xc3d7, 0xd9e1, 0xc0e0, 0xf4cc,
  0xd7d1, 0xb7db, 0xf4ce, 0xc1a3, 0xc6c9, 0xb4d6, 0xd5b3, 0xf4d0,
  0xf4cf, 0xf4d1, 0xcbda, 0xf4d2, 0xd4c1, 0xd6e0, 0xb7e0, 0xc1b8,
  0xc1bb, 0xf4d3, 0xbeac, 0xb4e2, 0xf4d4, 0xf4d5, 0xbeab, 0xf4d6,
  0xf4db, 0xf4d7, 0xf4da, 0xbafd, 0xf4d8, 0xf4d9, 0xb8e2, 0xccc7,
  0xf4dc, 0xb2da, 0xc3d3, 0xd4e3, 0xbfb7, 0xf4dd, 0xc5b4, 0xf4e9,
  0xcfb5, 0xcec9, 0xcbd8, 0xcbf7, 0xbdf4, 0xd7cf, 0xc0db, 0xd0f5,
  0xf4ea, 0xf4eb, 0xf4ec, 0xf7e3, 0xb7b1, 0xf4ed, 0xd7eb, 0xf4ee,
  0xe6f9, 0xbec0, 0xe6fa, 0xbaec, 0xe6fb, 0xcfcb, 0xe6fc, 0xd4bc,
  0xbcb6, 0xe6fd, 0xe6fe, 0xbccd, 0xc8d2, 0xceb3, 0xe7a1, 0xb4bf,
  0xe7a2, 0xc9b4, 0xb8d9, 0xc4c9, 0xd7dd, 0xc2da, 0xb7d7, 0xd6bd,
  0xcec6, 0xb7c4, 0xc5a6, 0xe7a3, 0xcfdf, 0xe7a4, 0xe7a5, 0xe7a6,
  0xc1b7, 0xd7e9, 0xc9f0, 0xcfb8, 0xd6af, 0xd6d5, 0xe7a7, 0xb0ed,
  0xe7a8, 0xe7a9, 0xc9dc, 0xd2ef, 0xbead, 0xe7aa, 0xb0f3, 0xc8de,
  0xbde1, 0xe7ab, 0xc8c6, 0xe7ac, 0xbbe6, 0xb8f8, 0xd1a4, 0xe7ad,
  0xc2e7, 0xbef8, 0xbdca, 0xcdb3, 0xe7ae, 0xe7af, 0xbeee, 0xd0e5,
  0xcbe7, 0xccd0, 0xbccc, 0xe7b0, 0xbca8, 0xd0f7, 0xe7b1, 0xd0f8,
  0xe7b2, 0xe7b3, 0xb4c2, 0xe7b4, 0xe7b5, 0xc9fe, 0xceac, 0xc3e0,
  0xe7b7, 0xb1c1, 0xb3f1, 0xe7b8, 0xe7b9, 0xd7db, 0xd5c0, 0xe7ba,
  0xc2cc, 0xd7ba, 0xe7bb, 0xe7bc, 0xe7bd, 0xbcea, 0xc3e5, 0xc0c2,
  0xe7be, 0xe7bf, 0xbca9, 0xe7c0, 0xe7c1, 0xe7b6, 0xb6d0, 0xe7c2,
  0xe7c3, 0xe7c4, 0xbbba, 0xb5de, 0xc2c6, 0xb1e0, 0xe7c5, 0xd4b5,
  0xe7c6, 0xb8bf, 0xe7c8, 0xe7c7, 0xb7ec, 0xe7c9, 0xb2f8, 0xe7ca,
  0xe7cb, 0xe7cc, 0xe7cd, 0xe7ce, 0xe7cf, 0xe7d0, 0xd3a7, 0xcbf5,
  0xe7d1, 0xe7d2, 0xe7d3, 0xe7d4, 0xc9c9, 0xe7d5, 0xe7d6, 0xe7d7,
  0xe7d8, 0xe7d9, 0xbdc9, 0xe7da, 0xf3be, 0xb8d7, 0xc8b1, 0xf3bf,
  0xf3c0, 0xf3c1, 0xb9de, 0xcdf8, 0xd8e8, 0xbab1, 0xc2de, 0xeeb7,
  0xb7a3, 0xeeb9, 0xeeb8, 0xb0d5, 0xeebb, 0xd5d6, 0xd7ef, 0xd6c3,
  0xeebd, 0xcaf0, 0xeebc, 0xeebe, 0xeec0, 0xeebf, 0xd1f2, 0xc7bc,
  0xc3c0, 0xb8e1, 0xc1e7, 0xf4c6, 0xd0df, 0xf4c7, 0xcfdb, 0xc8ba,
  0xf4c8, 0xf4c9, 0xf4ca, 0xf4cb, 0xd9fa, 0xb8fe, 0xe5f1, 0xd3f0,
  0xf4e0, 0xcecc, 0xb3e1, 0xf1b4, 0xd2ee, 0xf4e1, 0xcfe8, 0xf4e2,
  0xc7cc, 0xb5d4, 0xb4e4, 0xf4e4, 0xf4e3, 0xf4e5, 0xf4e6, 0xf4e7,
  0xbab2, 0xb0bf, 0xf4e8, 0xb7ad, 0xd2ed, 0xd2ab, 0xc0cf, 0xbfbc,
  0xeba3, 0xd5df, 0xeac8, 0xf1f3, 0xb6f8, 0xcba3, 0xc4cd, 0xf1e7,
  0xf1e8, 0xb8fb, 0xf1e9, 0xbac4, 0xd4c5, 0xb0d2, 0xf1ea, 0xf1eb,
  0xf1ec, 0xf1ed, 0xf1ee, 0xf1ef, 0xf1f1, 0xf1f0, 0xc5d5, 0xf1f2,
  0xb6fa, 0xf1f4, 0xd2ae, 0xdec7, 0xcbca, 0xb3dc, 0xb5a2, 0xb9a2,
  0xc4f4, 0xf1f5, 0xf1f6, 0xc1c4, 0xc1fb, 0xd6b0, 0xf1f7, 0xf1f8,
  0xc1aa, 0xc6b8, 0xbedb, 0xf1f9, 0xb4cf, 0xf1fa, 0xedb2, 0xedb1,
  0xcbe0, 0xd2de, 0xcbc1, 0xd5d8, 0xc8e2, 0xc0df, 0xbca1, 0xebc1,
  0xd0a4, 0xd6e2, 0xb6c7, 0xb8d8, 0xebc0, 0xb8ce, 0xebbf, 0xb3a6,
  0xb9c9, 0xd6ab, 0xb7f4, 0xb7ca, 0xbce7, 0xb7be, 0xebc6, 0xebc7,
  0xb0b9, 0xbfcf, 0xebc5, 0xd3fd, 0xebc8, 0xebc9, 0xb7ce, 0xebc2,
  0xebc4, 0xc9f6, 0xd6d7, 0xd5cd, 0xd0b2, 0xebcf, 0xceb8, 0xebd0,
  0xb5a8, 0xb1b3, 0xebd2, 0xcca5, 0xc5d6, 0xebd3, 0xebd1, 0xc5df,
  0xebce, 0xcaa4, 0xebd5, 0xb0fb, 0xbafa, 0xd8b7, 0xf1e3, 0xebca,
  0xebcb, 0xebcc, 0xebcd, 0xebd6, 0xe6c0, 0xebd9, 0xbfe8, 0xd2c8,
  0xebd7, 0xebdc, 0xb8ec, 0xebd8, 0xbdba, 0xd0d8, 0xb0b7, 0xebdd,
  0xc4dc, 0xd6ac, 0xb4e0, 0xc2f6, 0xbcb9, 0xebda, 0xebdb, 0xd4e0,
  0xc6ea, 0xc4d4, 0xebdf, 0xc5a7, 0xd9f5, 0xb2b1, 0xebe4, 0xbdc5,
  0xebe2, 0xebe3, 0xb8ac, 0xcdd1, 0xebe5, 0xebe1, 0xc1b3, 0xc6a2,
  0xccf3, 0xebe6, 0xc0b0, 0xd2b8, 0xebe7, 0xb8af, 0xb8ad, 0xebe8,
  0xc7bb, 0xcdf3, 0xebea, 0xebeb, 0xebed, 0xd0c8, 0xebf2, 0xebee,
  0xebf1, 0xc8f9, 0xd1fc, 0xebec, 0xebe9, 0xb8b9, 0xcfd9, 0xc4e5,
  0xebef, 0xebf0, 0xccda, 0xcdc8, 0xb0f2, 0xebf6, 0xebf5, 0xb2b2,
  0xb8e0, 0xebf7, 0xb1ec, 0xccc5, 0xc4a4, 0xcfa5, 0xebf9, 0xeca2,
  0xc5f2, 0xebfa, 0xc9c5, 0xe2df, 0xebfe, 0xcdce, 0xeca1, 0xb1db,
  0xd3b7, 0xd2dc, 0xebfd, 0xebfb, 0xb3bc, 0xeab0, 0xd7d4, 0xf4ab,
  0xb3f4, 0xd6c1, 0xd6c2, 0xd5e9, 0xbeca, 0xf4a7, 0xd2a8, 0xf4a8,
  0xf4a9, 0xf4aa, 0xbecb, 0xd3df, 0xc9e0, 0xc9e1, 0xf3c2, 0xcae6,
  0xccf2, 0xe2b6, 0xcbb4, 0xcee8, 0xd6db, 0xf4ad, 0xf4ae, 0xf4af,
  0xf4b2, 0xbabd, 0xf4b3, 0xb0e3, 0xf4b0, 0xf4b1, 0xbda2, 0xb2d5,
  0xf4b6, 0xf4b7, 0xb6e6, 0xb2b0, 0xcfcf, 0xf4b4, 0xb4ac, 0xf4b5,
  0xf4b8, 0xf4b9, 0xcda7, 0xf4ba, 0xf4bb, 0xf4bc, 0xcbd2, 0xf4bd,
  0xf4be, 0xf4bf, 0xf4de, 0xc1bc, 0xbce8, 0xc9ab, 0xd1de, 0xe5f5,
  0xdcb3, 0xd2d5, 0xdcb4, 0xb0ac, 0xdcb5, 0xbdda, 0xdcb9, 0xd8c2,
  0xdcb7, 0xd3f3, 0xc9d6, 0xdcba, 0xdcb6, 0xdcbb, 0xc3a2, 0xdcbc,
  0xdcc5, 0xdcbd, 0xcedf, 0xd6a5, 0xdccf, 0xdccd, 0xdcd2, 0xbde6,
  0xc2ab, 0xdcb8, 0xdccb, 0xdcce, 0xdcbe, 0xb7d2, 0xb0c5, 0xdcc7,
  0xd0be, 0xdcc1, 0xbba8, 0xb7bc, 0xdccc, 0xdcc6, 0xdcbf, 0xc7db,
  0xd1bf, 0xdcc0, 0xdcca, 0xdcd0, 0xcead, 0xdcc2, 0xdcc3, 0xdcc8,
  0xdcc9, 0xb2d4, 0xdcd1, 0xcbd5, 0xd4b7, 0xdcdb, 0xdcdf, 0xcca6,
  0xdce6, 0xc3e7, 0xdcdc, 0xbfc1, 0xdcd9, 0xb0fa, 0xb9b6, 0xdce5,
  0xdcd3, 0xdcc4, 0xdcd6, 0xc8f4, 0xbfe0, 0xc9bb, 0xb1bd, 0xd3a2,
  0xdcda, 0xdcd5, 0xc6bb, 0xdcde, 0xd7c2, 0xc3af, 0xb7b6, 0xc7d1,
  0xc3a9, 0xdce2, 0xdcd8, 0xdceb, 0xdcd4, 0xdcdd, 0xbea5, 0xdcd7,
  0xdce0, 0xdce3, 0xdce4, 0xdcf8, 0xdce1, 0xdda2, 0xdce7, 0xbceb,
  0xb4c4, 0xc3a3, 0xb2e7, 0xdcfa, 0xdcf2, 0xdcef, 0xdcfc, 0xdcee,
  0xd2f0, 0xb2e8, 0xc8d7, 0xc8e3, 0xdcfb, 0xdced, 0xdcf7, 0xdcf5,
  0xbea3, 0xdcf4, 0xb2dd, 0xdcf3, 0xbcf6, 0xdce8, 0xbbc4, 0xc0f3,
  0xbcd4, 0xdce9, 0xdcea, 0xdcf1, 0xdcf6, 0xdcf9, 0xb5b4, 0xc8d9,
  0xbbe7, 0xdcfe, 0xdcfd, 0xd3ab, 0xdda1, 0xdda3, 0xdda5, 0xd2f1,
  0xdda4, 0xdda6, 0xdda7, 0xd2a9, 0xbac9, 0xdda9, 0xddb6, 0xddb1,
  0xddb4, 0xddb0, 0xc6ce, 0xc0f2, 0xc9af, 0xdcec, 0xddae, 0xddb7,
  0xdcf0, 0xddaf, 0xddb8, 0xddac, 0xddb9, 0xddb3, 0xddad, 0xc4aa,
  0xdda8, 0xc0b3, 0xc1ab, 0xddaa, 0xddab, 0xddb2, 0xbbf1, 0xddb5,
  0xd3a8, 0xddba, 0xddbb, 0xc3a7, 0xddd2, 0xddbc, 0xddd1, 0xb9bd,
  0xbed5, 0xbefa, 0xbaca, 0xddca, 0xddc5, 0xddbf, 0xb2cb, 0xddc3,
  0xddcb, 0xb2a4, 0xddd5, 0xddbe, 0xc6d0, 0xddd0, 0xddd4, 0xc1e2,
  0xb7c6, 0xddce, 0xddcf, 0xddc4, 0xddbd, 0xddcd, 0xccd1, 0xddc9,
  0xddc2, 0xc3c8, 0xc6bc, 0xceae, 0xddcc, 0xddc8, 0xddc1, 0xddc6,
  0xc2dc, 0xd3a9, 0xd3aa, 0xddd3, 0xcff4, 0xc8f8, 0xdde6, 0xddc7,
  0xdde0, 0xc2e4, 0xdde1, 0xddd7, 0xd6f8, 0xddd9, 0xddd8, 0xb8f0,
  0xddd6, 0xc6cf, 0xb6ad, 0xdde2, 0xbaf9, 0xd4e1, 0xdde7, 0xb4d0,
  0xddda, 0xbffb, 0xdde3, 0xdddf, 0xdddd, 0xb5d9, 0xdddb, 0xdddc,
  0xddde, 0xbdaf, 0xdde4, 0xdde5, 0xddf5, 0xc3c9, 0xcbe2, 0xddf2,
  0xd8e1, 0xc6d1, 0xddf4, 0xd5f4, 0xddf3, 0xddf0, 0xddec, 0xddef,
  0xdde8, 0xd0ee, 0xc8d8, 0xddee, 0xdde9, 0xddea, 0xcbf2, 0xdded,
  0xb1cd, 0xc0b6, 0xbcbb, 0xddf1, 0xddf7, 0xddf6, 0xddeb, 0xc5ee,
  0xddfb, 0xdea4, 0xdea3, 0xddf8, 0xc3ef, 0xc2fb, 0xd5e1, 0xceb5,
  0xddfd, 0xb2cc, 0xc4e8, 0xcadf, 0xc7be, 0xddfa, 0xddfc, 0xddfe,
  0xdea2, 0xb0aa, 0xb1ce, 0xdeac, 0xdea6, 0xbdb6, 0xc8ef, 0xdea1,
  0xdea5, 0xdea9, 0xdea8, 0xdea7, 0xdead, 0xd4cc, 0xdeb3, 0xdeaa,
  0xdeae, 0xc0d9, 0xb1a1, 0xdeb6, 0xdeb1, 0xdeb2, 0xd1a6, 0xdeb5,
  0xdeaf, 0xdeb0, 0xd0bd, 0xdeb4, 0xcaed, 0xdeb9, 0xdeb8, 0xdeb7,
  0xdebb, 0xbde5, 0xb2d8, 0xc3ea, 0xdeba, 0xc5ba, 0xdebc, 0xccd9,
  0xb7aa, 0xd4e5, 0xdebd, 0xdebf, 0xc4a2, 0xdec1, 0xdebe, 0xdec0,
  0xd5ba, 0xdec2, 0xf2ae, 0xbba2, 0xc2b2, 0xc5b0, 0xc2c7, 0xf2af,
  0xd0e9, 0xd3dd, 0xebbd, 0xb3e6, 0xf2b0, 0xf2b1, 0xcaad, 0xbae7,
  0xf2b3, 0xf2b5, 0xf2b4, 0xcbe4, 0xcfba, 0xf2b2, 0xcab4, 0xd2cf,
  0xc2ec, 0xcec3, 0xf2b8, 0xb0f6, 0xf2b7, 0xf2be, 0xb2cf, 0xd1c1,
  0xf2ba, 0xf2bc, 0xd4e9, 0xf2bb, 0xf2b6, 0xf2bf, 0xf2bd, 0xf2b9,
  0xf2c7, 0xf2c4, 0xf2c6, 0xf2ca, 0xf2c2, 0xf2c0, 0xf2c5, 0xd6fb,
  0xf2c1, 0xc7f9, 0xc9df, 0xf2c8, 0xb9c6, 0xb5b0, 0xf2c3, 0xf2c9,
  0xf2d0, 0xf2d6, 0xbbd7, 0xf2d5, 0xcddc, 0xd6eb, 0xf2d2, 0xf2d4,
  0xb8f2, 0xf2cb, 0xf2ce, 0xc2f9, 0xd5dd, 0xf2cc, 0xf2cd, 0xf2cf,
  0xf2d3, 0xf2d9, 0xd3bc, 0xb6ea, 0xcaf1, 0xb7e4, 0xf2d7, 0xf2d8,
  0xf2da, 0xf2dd, 0xf2db, 0xf2dc, 0xd1d1, 0xf2d1, 0xcdc9, 0xcecf,
  0xd6a9, 0xf2e3, 0xc3db, 0xf2e0, 0xc0af, 0xf2ec, 0xf2de, 0xf2e1,
  0xf2e8, 0xf2e2, 0xf2e7, 0xf2e6, 0xf2e9, 0xf2df, 0xf2e4, 0xf2ea,
  0xd3ac, 0xf2e5, 0xb2f5, 0xf2f2, 0xd0ab, 0xf2f5, 0xbbc8, 0xf2f9,
  0xf2f0, 0xf2f6, 0xf2f8, 0xf2fa, 0xf2f3, 0xf2f1, 0xbafb, 0xb5fb,
  0xf2ef, 0xf2f7, 0xf2ed, 0xf2ee, 0xf2eb, 0xf3a6, 0xf3a3, 0xf3a2,
  0xf2f4, 0xc8da, 0xf2fb, 0xf3a5, 0xc3f8, 0xf2fd, 0xf3a7, 0xf3a9,
  0xf3a4, 0xf2fc, 0xf3ab, 0xf3aa, 0xc2dd, 0xf3ae, 0xf3b0, 0xf3a1,
  0xf3b1, 0xf3ac, 0xf3af, 0xf2fe, 0xf3ad, 0xf3b2, 0xf3b4, 0xf3a8,
  0xf3b3, 0xf3b5, 0xd0b7, 0xf3b8, 0xd9f9, 0xf3b9, 0xf3b7, 0xc8e4,
  0xf3b6, 0xf3ba, 0xf3bb, 0xb4c0, 0xeec3, 0xf3bc, 0xf3bd, 0xd1aa,
  0xf4ac, 0xd0c6, 0xd0d0, 0xd1dc, 0xcfce, 0xbdd6, 0xd1c3, 0xbae2,
  0xe1e9, 0xd2c2, 0xf1c2, 0xb2b9, 0xb1ed, 0xf1c3, 0xc9c0, 0xb3c4,
  0xd9f2, 0xcba5, 0xf1c4, 0xd6d4, 0xf1c5, 0xf4c0, 0xf1c6, 0xd4ac,
  0xf1c7, 0xb0c0, 0xf4c1, 0xf4c2, 0xb4fc, 0xc5db, 0xccbb, 0xd0e4,
  0xcde0, 0xf1c8, 0xd9f3, 0xb1bb, 0xcfae, 0xb8a4, 0xf1ca, 0xf1cb,
  0xb2c3, 0xc1d1, 0xd7b0, 0xf1c9, 0xf1cc, 0xf1ce, 0xd9f6, 0xd2e1,
  0xd4a3, 0xf4c3, 0xc8b9, 0xf4c4, 0xf1cd, 0xf1cf, 0xbfe3, 0xf1d0,
  0xf1d4, 0xf1d6, 0xf1d1, 0xc9d1, 0xc5e1, 0xc2e3, 0xb9fc, 0xf1d3,
  0xf1d5, 0xb9d3, 0xf1db, 0xbad6, 0xb0fd, 0xf1d9, 0xf1d8, 0xf1d2,
  0xf1da, 0xf1d7, 0xc8ec, 0xcdca, 0xf1dd, 0xe5bd, 0xf1dc, 0xf1de,
  0xf1df, 0xcfe5, 0xf4c5, 0xbdf3, 0xf1e0, 0xf1e1, 0xcef7, 0xd2aa,
  0xf1fb, 0xb8b2, 0xbcfb, 0xb9db, 0xb9e6, 0xc3d9, 0xcad3, 0xeae8,
  0xc0c0, 0xbef5, 0xeae9, 0xeaea, 0xeaeb, 0xeaec, 0xeaed, 0xeaee,
  0xeaef, 0xbdc7, 0xf5fb, 0xf5fd, 0xf5fe, 0xf5fc, 0xbde2, 0xf6a1,
  0xb4a5, 0xf6a2, 0xf6a3, 0xecb2, 0xd1d4, 0xd9ea, 0xf6a4, 0xeeba,
  0xd5b2, 0xd3fe, 0xccdc, 0xcac4, 0xe5c0, 0xf6a5, 0xbeaf, 0xc6a9,
  0xdaa5, 0xbcc6, 0xb6a9, 0xb8bc, 0xc8cf, 0xbca5, 0xdaa6, 0xdaa7,
  0xccd6, 0xc8c3, 0xdaa8, 0xc6fd, 0xd1b5, 0xd2e9, 0xd1b6, 0xbcc7,
  0xbdb2, 0xbbe4, 0xdaa9, 0xdaaa, 0xd1c8, 0xdaab, 0xd0ed, 0xb6ef,
  0xc2db, 0xcbcf, 0xb7ed, 0xc9e8, 0xb7c3, 0xbef7, 0xd6a4, 0xdaac,
  0xdaad, 0xc6c0, 0xd7e7, 0xcab6, 0xd5a9, 0xcbdf, 0xd5ef, 0xdaae,
  0xd6df, 0xb4ca, 0xdab0, 0xdaaf, 0xd2eb, 0xdab1, 0xdab2, 0xdab3,
  0xcad4, 0xdab4, 0xcaab, 0xdab5, 0xdab6, 0xb3cf, 0xd6ef, 0xdab7,
  0xbbb0, 0xb5ae, 0xdab8, 0xdab9, 0xb9ee, 0xd1af, 0xd2e8, 0xdaba,
  0xb8c3, 0xcfea, 0xb2ef, 0xdabb, 0xdabc, 0xbdeb, 0xcedc, 0xd3ef,
  0xdabd, 0xcef3, 0xdabe, 0xd3d5, 0xbbe5, 0xdabf, 0xcbb5, 0xcbd0,
  0xdac0, 0xc7eb, 0xd6ee, 0xdac1, 0xc5b5, 0xb6c1, 0xdac2, 0xb7cc,
  0xbfce, 0xdac3, 0xdac4, 0xcbad, 0xdac5, 0xb5f7, 0xdac6, 0xc1c2,
  0xd7bb, 0xdac7, 0xccb8, 0xd2ea, 0xc4b1, 0xdac8, 0xb5fd, 0xbbd1,
  0xdac9, 0xd0b3, 0xdaca, 0xdacb, 0xcebd, 0xdacc, 0xdacd, 0xdace,
  0xb2f7, 0xdad1, 0xdacf, 0xd1e8, 0xdad0, 0xc3d5, 0xdad2, 0xdad3,
  0xdad4, 0xdad5, 0xd0bb, 0xd2a5, 0xb0f9, 0xdad6, 0xc7ab, 0xdad7,
  0xbdf7, 0xc3a1, 0xdad8, 0xdad9, 0xc3fd, 0xccb7, 0xdada, 0xdadb,
  0xc0be, 0xc6d7, 0xdadc, 0xdadd, 0xc7b4, 0xdade, 0xdadf, 0xb9c8,
  0xbbed, 0xb6b9, 0xf4f8, 0xf4f9, 0xcde3, 0xf5b9, 0xebe0, 0xcff3,
  0xbbbf, 0xbac0, 0xd4a5, 0xe1d9, 0xf5f4, 0xb1aa, 0xb2f2, 0xf5f5,
  0xf5f7, 0xbad1, 0xf5f6, 0xc3b2, 0xf5f9, 0xf5f8, 0xb1b4, 0xd5ea,
  0xb8ba, 0xb9b1, 0xb2c6, 0xd4f0, 0xcfcd, 0xb0dc, 0xd5cb, 0xbbf5,
  0xd6ca, 0xb7b7, 0xccb0, 0xc6b6, 0xb1e1, 0xb9ba, 0xd6fc, 0xb9e1,
  0xb7a1, 0xbcfa, 0xeada, 0xeadb, 0xccf9, 0xb9f3, 0xeadc, 0xb4fb,
  0xc3b3, 0xb7d1, 0xbad8, 0xeadd, 0xd4f4, 0xeade, 0xbcd6, 0xbbdf,
  0xeadf, 0xc1de, 0xc2b8, 0xd4df, 0xd7ca, 0xeae0, 0xeae1, 0xeae4,
  0xeae2, 0xeae3, 0xc9de, 0xb8b3, 0xb6c4, 0xeae5, 0xcaea, 0xc9cd,
  0xb4cd, 0xe2d9, 0xc5e2, 0xeae6, 0xc0b5, 0xd7b8, 0xeae7, 0xd7ac,
  0xc8fc, 0xd8d3, 0xd8cd, 0xd4de, 0xd4f9, 0xc9c4, 0xd3ae, 0xb8d3,
  0xb3e0, 0xc9e2, 0xf4f6, 0xbad5, 0xf4f7, 0xd7df, 0xf4f1, 0xb8b0,
  0xd5d4, 0xb8cf, 0xc6f0, 0xb3c3, 0xf4f2, 0xb3ac, 0xd4bd, 0xc7f7,
  0xf4f4, 0xf4f3, 0xcccb, 0xc8a4, 0xf4f5, 0xd7e3, 0xc5bf, 0xf5c0,
  0xf5bb, 0xf5c3, 0xf5c2, 0xd6ba, 0xf5c1, 0xd4be, 0xf5c4, 0xf5cc,
  0xb0cf, 0xb5f8, 0xf5c9, 0xf5ca, 0xc5dc, 0xf5c5, 0xf5c6, 0xf5c7,
  0xf5cb, 0xbee0, 0xf5c8, 0xb8fa, 0xf5d0, 0xf5d3, 0xbfe7, 0xb9f2,
  0xf5bc, 0xf5cd, 0xc2b7, 0xccf8, 0xbcf9, 0xf5ce, 0xf5cf, 0xf5d1,
  0xb6e5, 0xf5d2, 0xf5d5, 0xf5bd, 0xf5d4, 0xd3bb, 0xb3ec, 0xcca4,
  0xf5d6, 0xf5d7, 0xbee1, 0xf5d8, 0xccdf, 0xf5db, 0xb2c8, 0xd7d9,
  0xf5d9, 0xf5da, 0xf5dc, 0xf5e2, 0xf5e0, 0xf5df, 0xf5dd, 0xf5e1,
  0xf5de, 0xf5e4, 0xf5e5, 0xcce3, 0xe5bf, 0xb5b8, 0xf5e3, 0xf5e8,
  0xcca3, 0xf5e6, 0xf5e7, 0xf5be, 0xb1c4, 0xf5bf, 0xb5c5, 0xb2e4,
  0xf5ec, 0xf5e9, 0xb6d7, 0xf5ed, 0xf5ea, 0xf5eb, 0xb4da, 0xd4ea,
  0xf5ee, 0xb3f9, 0xf5ef, 0xf5f1, 0xf5f0, 0xf5f2, 0xf5f3, 0xc9ed,
  0xb9aa, 0xc7fb, 0xb6e3, 0xccc9, 0xeaa6, 0xb3b5, 0xd4fe, 0xb9ec,
  0xd0f9, 0xe9ed, 0xd7aa, 0xe9ee, 0xc2d6, 0xc8ed, 0xbae4, 0xe9ef,
  0xe9f0, 0xe9f1, 0xd6e1, 0xe9f2, 0xe9f3, 0xe9f5, 0xe9f4, 0xe9f6,
  0xe9f7, 0xc7e1, 0xe9f8, 0xd4d8, 0xe9f9, 0xbdce, 0xe9fa, 0xe9fb,
  0xbdcf, 0xe9fc, 0xb8a8, 0xc1be, 0xe9fd, 0xb1b2, 0xbbd4, 0xb9f5,
  0xe9fe, 0xeaa1, 0xeaa2, 0xeaa3, 0xb7f8, 0xbcad, 0xcae4, 0xe0ce,
  0xd4af, 0xcfbd, 0xd5b7, 0xeaa4, 0xd5de, 0xeaa5, 0xd0c1, 0xb9bc,
  0xb4c7, 0xb1d9, 0xc0b1, 0xb1e6, 0xb1e7, 0xb1e8, 0xb3bd, 0xc8e8,
  0xe5c1, 0xb1df, 0xc1c9, 0xb4ef, 0xc7a8, 0xd3d8, 0xc6f9, 0xd1b8,
  0xb9fd, 0xc2f5, 0xd3ad, 0xd4cb, 0xbdfc, 0xe5c2, 0xb7b5, 0xe5c3,
  0xbbb9, 0xd5e2, 0xbdf8, 0xd4b6, 0xcea5, 0xc1ac, 0xb3d9, 0xccf6,
  0xe5c6, 0xe5c4, 0xe5c8, 0xe5ca, 0xe5c7, 0xb5cf, 0xc6c8, 0xb5fc,
  0xe5c5, 0xcaf6, 0xe5c9, 0xc3d4, 0xb1c5, 0xbca3, 0xd7b7, 0xcdcb,
  0xcbcd, 0xcaca, 0xccd3, 0xe5cc, 0xe5cb, 0xc4e6, 0xd1a1, 0xd1b7,
  0xe5cd, 0xe5d0, 0xcdb8, 0xd6f0, 0xe5cf, 0xb5dd, 0xcdbe, 0xe5d1,
  0xb6ba, 0xcda8, 0xb9e4, 0xcac5, 0xb3d1, 0xcbd9, 0xd4ec, 0xe5d2,
  0xb7ea, 0xe5ce, 0xe5d5, 0xb4fe, 0xe5d6, 0xe5d3, 0xe5d4, 0xd2dd,
  0xc2df, 0xb1c6, 0xd3e2, 0xb6dd, 0xcbec, 0xe5d7, 0xd3f6, 0xb1e9,
  0xb6f4, 0xe5da, 0xe5d8, 0xe5d9, 0xb5c0, 0xd2c5, 0xe5dc, 0xe5de,
  0xe5dd, 0xc7b2, 0xd2a3, 0xe5db, 0xd4e2, 0xd5da, 0xe5e0, 0xd7f1,
  0xe5e1, 0xb1dc, 0xd1fb, 0xe5e2, 0xe5e4, 0xe5e3, 0xe5e5, 0xd2d8,
  0xb5cb, 0xe7df, 0xdaf5, 0xdaf8, 0xdaf6, 0xdaf7, 0xdafa, 0xd0cf,
  0xc4c7, 0xb0ee, 0xd0b0, 0xdaf9, 0xd3ca, 0xbaaa, 0xdba2, 0xc7f1,
  0xdafc, 0xdafb, 0xc9db, 0xdafd, 0xdba1, 0xd7de, 0xdafe, 0xc1da,
  0xdba5, 0xd3f4, 0xdba7, 0xdba4, 0xdba8, 0xbdbc, 0xc0c9, 0xdba3,
  0xdba6, 0xd6a3, 0xdba9, 0xdbad, 0xdbae, 0xdbac, 0xbac2, 0xbfa4,
  0xdbab, 0xdbaa, 0xd4c7, 0xb2bf, 0xdbaf, 0xb9f9, 0xdbb0, 0xb3bb,
  0xb5a6, 0xb6bc, 0xdbb1, 0xb6f5, 0xdbb2, 0xb1c9, 0xdbb4, 0xdbb3,
  0xdbb5, 0xdbb7, 0xdbb6, 0xdbb8, 0xdbb9, 0xdbba, 0xd3cf, 0xf4fa,
  0xc7f5, 0xd7c3, 0xc5e4, 0xf4fc, 0xf4fd, 0xf4fb, 0xbec6, 0xd0ef,
  0xb7d3, 0xd4cd, 0xccaa, 0xf5a2, 0xf5a1, 0xbaa8, 0xf4fe, 0xcbd6,
  0xf5a4, 0xc0d2, 0xb3ea, 0xcdaa, 0xf5a5, 0xf5a3, 0xbdb4, 0xf5a8,
  0xf5a9, 0xbdcd, 0xc3b8, 0xbfe1, 0xcbe1, 0xf5aa, 0xf5a6, 0xf5a7,
  0xc4f0, 0xf5ac, 0xb4bc, 0xd7ed, 0xb4d7, 0xf5ab, 0xf5ae, 0xf5ad,
  0xf5af, 0xd0d1, 0xc3d1, 0xc8a9, 0xf5b0, 0xf5b1, 0xf5b2, 0xf5b3,
  0xf5b4, 0xf5b5, 0xf5b7, 0xf5b6, 0xf5b8, 0xb2c9, 0xd3d4, 0xcacd,
  0xc0ef, 0xd6d8, 0xd2b0, 0xc1bf, 0xbdf0, 0xb8aa, 0xbcf8, 0xf6c6,
  0xf6c7, 0xf6c8, 0xf6c9, 0xf6ca, 0xf6cc, 0xf6cb, 0xf7e9, 0xf6cd,
  0xf6ce, 0xeec4, 0xeec5, 0xeec6, 0xd5eb, 0xb6a4, 0xeec8, 0xeec7,
  0xeec9, 0xeeca, 0xc7a5, 0xeecb, 0xeecc, 0xb7b0, 0xb5f6, 0xeecd,
  0xeecf, 0xeece, 0xb8c6, 0xeed0, 0xeed1, 0xeed2, 0xb6db, 0xb3ae,
  0xd6d3, 0xc4c6, 0xb1b5, 0xb8d6, 0xeed3, 0xeed4, 0xd4bf, 0xc7d5,
  0xbefb, 0xced9, 0xb9b3, 0xeed6, 0xeed5, 0xeed8, 0xeed7, 0xc5a5,
  0xeed9, 0xeeda, 0xc7ae, 0xeedb, 0xc7af, 0xeedc, 0xb2a7, 0xeedd,
  0xeede, 0xeedf, 0xeee0, 0xeee1, 0xd7ea, 0xeee2, 0xeee3, 0xbcd8,
  0xeee4, 0xd3cb, 0xccfa, 0xb2ac, 0xc1e5, 0xeee5, 0xc7a6, 0xc3ad,
  0xeee6, 0xeee7, 0xeee8, 0xeee9, 0xeeea, 0xeeeb, 0xeeec, 0xeeed,
  0xeeee, 0xeeef, 0xeef0, 0xeef1, 0xeef2, 0xeef4, 0xeef3, 0xeef5,
  0xcdad, 0xc2c1, 0xeef6, 0xeef7, 0xeef8, 0xd5a1, 0xeef9, 0xcfb3,
  0xeefa, 0xeefb, 0xeefc, 0xeefd, 0xefa1, 0xeefe, 0xefa2, 0xb8f5,
  0xc3fa, 0xefa3, 0xefa4, 0xbdc2, 0xd2bf, 0xb2f9, 0xefa5, 0xefa6,
  0xefa7, 0xd2f8, 0xefa8, 0xd6fd, 0xefa9, 0xc6cc, 0xefaa, 0xefab,
  0xc1b4, 0xefac, 0xcffa, 0xcbf8, 0xefae, 0xefad, 0xb3fa, 0xb9f8,
  0xefaf, 0xefb0, 0xd0e2, 0xefb1, 0xefb2, 0xb7e6, 0xd0bf, 0xefb3,
  0xefb4, 0xefb5, 0xc8f1, 0xcce0, 0xefb6, 0xefb7, 0xefb8, 0xefb9,
  0xefba, 0xd5e0, 0xefbb, 0xb4ed, 0xc3aa, 0xefbc, 0xefbd, 0xefbe,
  0xefbf, 0xcefd, 0xefc0, 0xc2e0, 0xb4b8, 0xd7b6, 0xbdf5, 0xcfc7,
  0xefc3, 0xefc1, 0xefc2, 0xefc4, 0xb6a7, 0xbcfc, 0xbee2, 0xc3cc,
  0xefc5, 0xefc6, 0xefc7, 0xefcf, 0xefc8, 0xefc9, 0xefca, 0xc7c2,
  0xeff1, 0xb6cd, 0xefcb, 0xefcc, 0xefcd, 0xb6c6, 0xc3be, 0xefce,
  0xefd0, 0xefd1, 0xefd2, 0xd5f2, 0xefd3, 0xc4f7, 0xefd4, 0xc4f8,
  0xefd5, 0xefd6, 0xb8e4, 0xb0f7, 0xefd7, 0xefd8, 0xefd9, 0xefda,
  0xefdb, 0xefdc, 0xefdd, 0xefde, 0xbeb5, 0xefe1, 0xefdf, 0xefe0,
  0xefe2, 0xefe3, 0xc1cd, 0xefe4, 0xefe5, 0xefe6, 0xefe7, 0xefe8,
  0xefe9, 0xefea, 0xefeb, 0xefec, 0xc0d8, 0xefed, 0xc1ad, 0xefee,
  0xefef, 0xeff0, 0xcfe2, 0xb3a4, 0xc3c5, 0xe3c5, 0xc9c1, 0xe3c6,
  0xb1d5, 0xceca, 0xb4b3, 0xc8f2, 0xe3c7, 0xcfd0, 0xe3c8, 0xbce4,
  0xe3c9, 0xe3ca, 0xc3c6, 0xd5a2, 0xc4d6, 0xb9eb, 0xcec5, 0xe3cb,
  0xc3f6, 0xe3cc, 0xb7a7, 0xb8f3, 0xbad2, 0xe3cd, 0xe3ce, 0xd4c4,
  0xe3cf, 0xe3d0, 0xd1cb, 0xe3d1, 0xe3d2, 0xe3d3, 0xe3d4, 0xd1d6,
  0xe3d5, 0xb2fb, 0xc0bb, 0xe3d6, 0xc0ab, 0xe3d7, 0xe3d8, 0xe3d9,
  0xe3da, 0xe3db, 0xb8b7, 0xdae2, 0xb6d3, 0xdae4, 0xdae3, 0xdae6,
  0xc8ee, 0xdae5, 0xb7c0, 0xd1f4, 0xd2f5, 0xd5f3, 0xbdd7, 0xd7e8,
  0xdae8, 0xdae7, 0xb0a2, 0xcdd3, 0xdae9, 0xb8bd, 0xbcca, 0xc2bd,
  0xc2a4, 0xb3c2, 0xdaea, 0xc2aa, 0xc4b0, 0xbdb5, 0xcfde, 0xdaeb,
  0xc9c2, 0xb1dd, 0xdaec, 0xb6b8, 0xd4ba, 0xb3fd, 0xdaed, 0xd4c9,
  0xcfd5, 0xc5e3, 0xdaee, 0xdaef, 0xdaf0, 0xc1ea, 0xccd5, 0xcfdd,
  0xd3e7, 0xc2a1, 0xdaf1, 0xcbe5, 0xdaf2, 0xcbe6, 0xd2fe, 0xb8f4,
  0xdaf3, 0xb0af, 0xcfb6, 0xd5cf, 0xcbed, 0xdaf4, 0xe3c4, 0xc1a5,
  0xf6bf, 0xf6c0, 0xf6c1, 0xc4d1, 0xc8b8, 0xd1e3, 0xd0db, 0xd1c5,
  0xbcaf, 0xb9cd, 0xeff4, 0xb4c6, 0xd3ba, 0xf6c2, 0xb3fb, 0xf6c3,
  0xb5f1, 0xf6c5, 0xd3ea, 0xf6a7, 0xd1a9, 0xf6a9, 0xf6a8, 0xc1e3,
  0xc0d7, 0xb1a2, 0xceed, 0xd0e8, 0xf6ab, 0xcff6, 0xf6aa, 0xd5f0,
  0xf6ac, 0xc3b9, 0xbbf4, 0xf6ae, 0xf6ad, 0xc4de, 0xc1d8, 0xcbaa,
  0xcfbc, 0xf6af, 0xf6b0, 0xf6b1, 0xc2b6, 0xb0d4, 0xc5f9, 0xf6b2,
  0xc7e0, 0xf6a6, 0xbeb8, 0xbeb2, 0xb5e5, 0xb7c7, 0xbfbf, 0xc3d2,
  0xc3e6, 0xd8cc, 0xb8ef, 0xbdf9, 0xd1a5, 0xb0d0, 0xf7b0, 0xf7b1,
  0xd0ac, 0xb0b0, 0xf7b2, 0xf7b3, 0xf7b4, 0xc7ca, 0xbecf, 0xf7b7,
  0xf7b6, 0xb1de, 0xf7b5, 0xf7b8, 0xf7b9, 0xcea4, 0xc8cd, 0xbaab,
  0xe8b8, 0xe8b9, 0xe8ba, 0xbec2, 0xd2f4, 0xd4cf, 0xc9d8, 0xd2b3,
  0xb6a5, 0xc7ea, 0xf1fc, 0xcfee, 0xcbb3, 0xd0eb, 0xe7ef, 0xcde7,
  0xb9cb, 0xb6d9, 0xf1fd, 0xb0e4, 0xcbcc, 0xf1fe, 0xd4a4, 0xc2ad,
  0xc1ec, 0xc6c4, 0xbeb1, 0xf2a1, 0xbcd5, 0xf2a2, 0xf2a3, 0xf2a4,
  0xd2c3, 0xc6b5, 0xcdc7, 0xf2a5, 0xd3b1, 0xbfc5, 0xcce2, 0xf2a6,
  0xf2a7, 0xd1d5, 0xb6ee, 0xf2a8, 0xf2a9, 0xb5df, 0xf2aa, 0xf2ab,
  0xb2fc, 0xf2ac, 0xf2ad, 0xc8a7, 0xb7e7, 0xeca9, 0xecaa, 0xecab,
  0xecac, 0xc6ae, 0xecad, 0xecae, 0xb7c9, 0xcab3, 0xe2b8, 0xf7cf,
  0xf7d0, 0xb2cd, 0xf7d1, 0xf7d3, 0xf7d2, 0xe2bb, 0xbca2, 0xe2bc,
  0xe2bd, 0xe2be, 0xe2bf, 0xe2c0, 0xe2c1, 0xb7b9, 0xd2fb, 0xbda4,
  0xcace, 0xb1a5, 0xcbc7, 0xe2c2, 0xb6fc, 0xc8c4, 0xe2c3, 0xbdc8,
  0xb1fd, 0xe2c4, 0xb6f6, 0xe2c5, 0xc4d9, 0xe2c6, 0xcfda, 0xb9dd,
  0xe2c7, 0xc0a1, 0xe2c8, 0xb2f6, 0xe2c9, 0xc1f3, 0xe2ca, 0xe2cb,
  0xc2f8, 0xe2cc, 0xe2cd, 0xe2ce, 0xcad7, 0xd8b8, 0xd9e5, 0xcfe3,
  0xf0a5, 0xdcb0, 0xc2ed, 0xd4a6, 0xcdd4, 0xd1b1, 0xb3db, 0xc7fd,
  0xb2b5, 0xc2bf, 0xe6e0, 0xcabb, 0xe6e1, 0xe6e2, 0xbed4, 0xe6e3,
  0xd7a4, 0xcdd5, 0xe6e5, 0xbcdd, 0xe6e4, 0xe6e6, 0xe6e7, 0xc2ee,
  0xbdbe, 0xe6e8, 0xc2e6, 0xbaa7, 0xe6e9, 0xe6ea, 0xb3d2, 0xd1e9,
  0xbfa5, 0xe6eb, 0xc6ef, 0xe6ec, 0xe6ed, 0xe6ee, 0xc6ad, 0xe6ef,
  0xc9a7, 0xe6f0, 0xe6f1, 0xe6f2, 0xe5b9, 0xe6f3, 0xe6f4, 0xc2e2,
  0xe6f5, 0xe6f6, 0xd6e8, 0xe6f7, 0xe6f8, 0xb9c7, 0xf7bb, 0xf7ba,
  0xf7be, 0xf7bc, 0xbaa1, 0xf7bf, 0xf7c0, 0xf7c2, 0xf7c1, 0xf7c4,
  0xf7c3, 0xf7c5, 0xf7c6, 0xf7c7, 0xcbe8, 0xb8df, 0xf7d4, 0xf7d5,
  0xf7d6, 0xf7d8, 0xf7da, 0xf7d7, 0xf7db, 0xf7d9, 0xd7d7, 0xf7dc,
  0xf7dd, 0xf7de, 0xf7df, 0xf7e0, 0xdbcb, 0xd8aa, 0xe5f7, 0xb9ed,
  0xbffd, 0xbbea, 0xf7c9, 0xc6c7, 0xf7c8, 0xf7ca, 0xf7cc, 0xf7cb,
  0xf7cd, 0xceba, 0xf7ce, 0xc4a7, 0xd3e3, 0xf6cf, 0xc2b3, 0xf6d0,
  0xf6d1, 0xf6d2, 0xf6d3, 0xf6d4, 0xf6d6, 0xb1ab, 0xf6d7, 0xf6d8,
  0xf6d9, 0xf6da, 0xf6db, 0xf6dc, 0xf6dd, 0xf6de, 0xcfca, 0xf6df,
  0xf6e0, 0xf6e1, 0xf6e2, 0xf6e3, 0xf6e4, 0xc0f0, 0xf6e5, 0xf6e6,
  0xf6e7, 0xf6e8, 0xf6e9, 0xf6ea, 0xf6eb, 0xf6ec, 0xf6ed, 0xf6ee,
  0xf6ef, 0xf6f0, 0xf6f1, 0xf6f2, 0xf6f3, 0xf6f4, 0xbea8, 0xf6f5,
  0xf6f6, 0xf6f7, 0xf6f8, 0xc8fa, 0xf6f9, 0xf6fa, 0xf6fb, 0xf6fc,
  0xf6fd, 0xf6fe, 0xf7a1, 0xf7a2, 0xf7a3, 0xf7a4, 0xf7a5, 0xf7a6,
  0xf7a7, 0xf7a8, 0xb1ee, 0xf7a9, 0xf7aa, 0xf7ab, 0xf7ac, 0xf7ad,
  0xc1db, 0xf7ae, 0xf7af, 0xc4f1, 0xf0af, 0xbca6, 0xf0b0, 0xc3f9,
  0xc5b8, 0xd1bb, 0xf0b1, 0xf0b2, 0xf0b3, 0xf0b4, 0xf0b5, 0xd1bc,
  0xd1ec, 0xf0b7, 0xf0b6, 0xd4a7, 0xcdd2, 0xf0b8, 0xf0ba, 0xf0b9,
  0xf0bb, 0xf0bc, 0xb8eb, 0xf0bd, 0xbae8, 0xf0be, 0xf0bf, 0xbee9,
  0xf0c0, 0xb6ec, 0xf0c1, 0xf0c2, 0xf0c3, 0xf0c4, 0xc8b5, 0xf0c5,
  0xf0c6, 0xf0c7, 0xc5f4, 0xf0c8, 0xf0c9, 0xf0ca, 0xf7bd, 0xf0cb,
  0xf0cc, 0xf0cd, 0xf0ce, 0xf0cf, 0xbad7, 0xf0d0, 0xf0d1, 0xf0d2,
  0xf0d3, 0xf0d4, 0xf0d5, 0xf0d6, 0xf0d8, 0xd3a5, 0xf0d7, 0xf0d9,
  0xf5ba, 0xc2b9, 0xf7e4, 0xf7e5, 0xf7e6, 0xf7e7, 0xf7e8, 0xc2b4,
  0xf7ea, 0xf7eb, 0xc2f3, 0xf4f0, 0xf4ef, 0xc2e9, 0xf7e1, 0xf7e2,
  0xbbc6, 0xd9e4, 0xcaf2, 0xc0e8, 0xf0a4, 0xbada, 0xc7ad, 0xc4ac,
  0xf7ec, 0xf7ed, 0xf7ee, 0xf7f0, 0xf7ef, 0xf7f1, 0xf7f4, 0xf7f3,
  0xf7f2, 0xf7f5, 0xf7f6, 0xede9, 0xedea, 0xedeb, 0xf6bc, 0xf6bd,
  0xf6be, 0xb6a6, 0xd8be, 0xb9c4, 0xd8bb, 0xdcb1, 0xcaf3, 0xf7f7,
  0xf7f8, 0xf7f9, 0xf7fb, 0xf7fa, 0xb1c7, 0xf7fc, 0xf7fd, 0xf7fe,
  0xc6eb, 0xecb4, 0xb3dd, 0xf6b3, 0xf6b4, 0xc1e4, 0xf6b5, 0xf6b6,
  0xf6b7, 0xf6b8, 0xf6b9, 0xf6ba, 0xc8a3, 0xf6bb, 0xc1fa, 0xb9a8,
  0xede8, 0xb9ea, 0xd9df, 0xa3a1, 0xa3a2, 0xa3a3, 0xa1e7, 0xa3a5,
  0xa3a6, 0xa3a7, 0xa3a8, 0xa3a9, 0xa3aa, 0xa3ab, 0xa3ac, 0xa3ad,
  0xa3ae, 0xa3af, 0xa3b0, 0xa3b1, 0xa3b2, 0xa3b3, 0xa3b4, 0xa3b5,
  0xa3b6, 0xa3b7, 0xa3b8, 0xa3b9, 0xa3ba, 0xa3bb, 0xa3bc, 0xa3bd,
  0xa3be, 0xa3bf, 0xa3c0, 0xa3c1, 0xa3c2, 0xa3c3, 0xa3c4, 0xa3c5,
  0xa3c6, 0xa3c7, 0xa3c8, 0xa3c9, 0xa3ca, 0xa3cb, 0xa3cc, 0xa3cd,
  0xa3ce, 0xa3cf, 0xa3d0, 0xa3d1, 0xa3d2, 0xa3d3, 0xa3d4, 0xa3d5,
  0xa3d6, 0xa3d7, 0xa3d8, 0xa3d9, 0xa3da, 0xa3db, 0xa3dc, 0xa3dd,
  0xa3de, 0xa3df, 0xa3e0, 0xa3e1, 0xa3e2, 0xa3e3, 0xa3e4, 0xa3e5,
  0xa3e6, 0xa3e7, 0xa3e8, 0xa3e9, 0xa3ea, 0xa3eb, 0xa3ec, 0xa3ed,
  0xa3ee, 0xa3ef, 0xa3f0, 0xa3f1, 0xa3f2, 0xa3f3, 0xa3f4, 0xa3f5,
  0xa3f6, 0xa3f7, 0xa3f8, 0xa3f9, 0xa3fa, 0xa3fb, 0xa3fc, 0xa3fd,
  0xa1ab, 0xa1e9, 0xa1ea, 0xa3fe, 0xa3a4,
};

typedef struct {
  unsigned short indx; /* index into big table */
  unsigned short used; /* bitmask of used entries */
} Summary16;

static const Summary16 gb2312_uni2indx_page00[70] = {
  /* 0x0000 */
  {    0, 0x0000 }, {    0, 0x0000 }, {    0, 0x0000 }, {    0, 0x0000 },
  {    0, 0x0000 }, {    0, 0x0000 }, {    0, 0x0000 }, {    0, 0x0000 },
  {    0, 0x0000 }, {    0, 0x0000 }, {    0, 0x0190 }, {    3, 0x0003 },
  {    5, 0x0000 }, {    5, 0x0080 }, {    6, 0x3703 }, {   13, 0x168c },
  /* 0x0100 */
  {   19, 0x0002 }, {   20, 0x0808 }, {   22, 0x0800 }, {   23, 0x0000 },
  {   23, 0x2000 }, {   24, 0x0000 }, {   24, 0x0800 }, {   25, 0x0000 },
  {   25, 0x0000 }, {   25, 0x0000 }, {   25, 0x0000 }, {   25, 0x0000 },
  {   25, 0x4000 }, {   26, 0x1555 }, {   33, 0x0000 }, {   33, 0x0000 },
  /* 0x0200 */
  {   33, 0x0000 }, {   33, 0x0000 }, {   33, 0x0000 }, {   33, 0x0000 },
  {   33, 0x0000 }, {   33, 0x0000 }, {   33, 0x0000 }, {   33, 0x0000 },
  {   33, 0x0000 }, {   33, 0x0000 }, {   33, 0x0000 }, {   33, 0x0000 },
  {   33, 0x0280 }, {   35, 0x0000 }, {   35, 0x0000 }, {   35, 0x0000 },
  /* 0x0300 */
  {   35, 0x0000 }, {   35, 0x0000 }, {   35, 0x0000 }, {   35, 0x0000 },
  {   35, 0x0000 }, {   35, 0x0000 }, {   35, 0x0000 }, {   35, 0x0000 },
  {   35, 0x0000 }, {   35, 0xfffe }, {   50, 0x03fb }, {   59, 0xfffe },
  {   74, 0x03fb }, {   83, 0x0000 }, {   83, 0x0000 }, {   83, 0x0000 },
  /* 0x0400 */
  {   83, 0x0002 }, {   84, 0xffff }, {  100, 0xffff }, {  116, 0xffff },
  {  132, 0xffff }, {  148, 0x0002 },
};
static const Summary16 gb2312_uni2indx_page20[101] = {
  /* 0x2000 */
  {  149, 0x0000 }, {  149, 0x3360 }, {  155, 0x0040 }, {  156, 0x080d },
  {  160, 0x0000 }, {  160, 0x0000 }, {  160, 0x0000 }, {  160, 0x0000 },
  {  160, 0x0000 }, {  160, 0x0000 }, {  160, 0x0000 }, {  160, 0x0000 },
  {  160, 0x0000 }, {  160, 0x0000 }, {  160, 0x0000 }, {  160, 0x0000 },
  /* 0x2100 */
  {  160, 0x0008 }, {  161, 0x0040 }, {  162, 0x0000 }, {  162, 0x0000 },
  {  162, 0x0000 }, {  162, 0x0000 }, {  162, 0x0fff }, {  174, 0x0000 },
  {  174, 0x0000 }, {  174, 0x000f }, {  178, 0x0000 }, {  178, 0x0000 },
  {  178, 0x0000 }, {  178, 0x0000 }, {  178, 0x0000 }, {  178, 0x0000 },
  /* 0x2200 */
  {  178, 0x8100 }, {  180, 0x6402 }, {  184, 0x4fa1 }, {  192, 0x20f0 },
  {  197, 0x1100 }, {  199, 0x0000 }, {  199, 0xc033 }, {  205, 0x0000 },
  {  205, 0x0000 }, {  205, 0x0200 }, {  206, 0x0020 }, {  207, 0x0000 },
  {  207, 0x0000 }, {  207, 0x0000 }, {  207, 0x0000 }, {  207, 0x0000 },
  /* 0x2300 */
  {  207, 0x0000 }, {  207, 0x0004 }, {  208, 0x0000 }, {  208, 0x0000 },
  {  208, 0x0000 }, {  208, 0x0000 }, {  208, 0x0000 }, {  208, 0x0000 },
  {  208, 0x0000 }, {  208, 0x0000 }, {  208, 0x0000 }, {  208, 0x0000 },
  {  208, 0x0000 }, {  208, 0x0000 }, {  208, 0x0000 }, {  208, 0x0000 },
  /* 0x2400 */
  {  208, 0x0000 }, {  208, 0x0000 }, {  208, 0x0000 }, {  208, 0x0000 },
  {  208, 0x0000 }, {  208, 0x0000 }, {  208, 0x03ff }, {  218, 0xfff0 },
  {  230, 0xffff }, {  246, 0x0fff }, {  258, 0x0000 }, {  258, 0x0000 },
  {  258, 0x0000 }, {  258, 0x0000 }, {  258, 0x0000 }, {  258, 0x0000 },
  /* 0x2500 */
  {  258, 0xffff }, {  274, 0xffff }, {  290, 0xffff }, {  306, 0xffff },
  {  322, 0x0fff }, {  334, 0x0000 }, {  334, 0x0000 }, {  334, 0x0000 },
  {  334, 0x0000 }, {  334, 0x0000 }, {  334, 0x0003 }, {  336, 0x000c },
  {  338, 0xc8c0 }, {  343, 0x0000 }, {  343, 0x0000 }, {  343, 0x0000 },
  /* 0x2600 */
  {  343, 0x0060 }, {  345, 0x0000 }, {  345, 0x0000 }, {  345, 0x0000 },
  {  345, 0x0005 },
};
static const Summary16 gb2312_uni2indx_page30[35] = {
  /* 0x3000 */
  {  347, 0xff2f }, {  360, 0x00fb }, {  367, 0x0000 }, {  367, 0x0000 },
  {  367, 0xfffe }, {  382, 0xffff }, {  398, 0xffff }, {  414, 0xffff },
  {  430, 0xffff }, {  446, 0x000f }, {  450, 0xfffe }, {  465, 0xffff },
  {  481, 0xffff }, {  497, 0xffff }, {  513, 0xffff }, {  529, 0x087f },
  /* 0x3100 */
  {  537, 0xffe0 }, {  548, 0xffff }, {  564, 0x03ff }, {  574, 0x0000 },
  {  574, 0x0000 }, {  574, 0x0000 }, {  574, 0x0000 }, {  574, 0x0000 },
  {  574, 0x0000 }, {  574, 0x0000 }, {  574, 0x0000 }, {  574, 0x0000 },
  {  574, 0x0000 }, {  574, 0x0000 }, {  574, 0x0000 }, {  574, 0x0000 },
  /* 0x3200 */
  {  574, 0x0000 }, {  574, 0x0000 }, {  574, 0x03ff },
};
static const Summary16 gb2312_uni2indx_page4e[1263] = {
  /* 0x4e00 */
  {  584, 0x7f8b }, {  595, 0x7f7b }, {  608, 0x3db4 }, {  617, 0xef55 },
  {  628, 0xfba8 }, {  638, 0xf35d }, {  649, 0x0243 }, {  653, 0x400b },
  {  657, 0xfb40 }, {  665, 0x8d3e }, {  674, 0x7bf7 }, {  687, 0x8c2c },
  {  693, 0x6eff }, {  706, 0xe3fa }, {  717, 0x1d3a }, {  725, 0xa8ed },
  /* 0x4f00 */
  {  734, 0xe602 }, {  740, 0xcf83 }, {  749, 0x8cf5 }, {  758, 0x3555 },
  {  766, 0xe048 }, {  771, 0xffab }, {  784, 0x92b9 }, {  792, 0xd859 },
  {  800, 0xab18 }, {  807, 0x2892 }, {  812, 0xd7e9 }, {  823, 0x8020 },
  {  825, 0xc438 }, {  831, 0xf583 }, {  840, 0xe74a }, {  849, 0x450a },
  /* 0x5000 */
  {  854, 0xb000 }, {  857, 0x9714 }, {  864, 0x7762 }, {  873, 0x5400 },
  {  876, 0xd188 }, {  882, 0x1420 }, {  885, 0x1020 }, {  887, 0xc8c0 },
  {  892, 0x2121 }, {  896, 0x0000 }, {  896, 0x13a8 }, {  902, 0x0c04 },
  {  905, 0x8000 }, {  906, 0x0440 }, {  908, 0x70c0 }, {  913, 0x0828 },
  /* 0x5100 */
  {  916, 0x08c0 }, {  919, 0x0004 }, {  920, 0x0002 }, {  921, 0x8000 },
  {  922, 0x2b7b }, {  932, 0x1472 }, {  938, 0x7924 }, {  945, 0x3bfb },
  {  957, 0x3327 }, {  965, 0x1ae4 }, {  972, 0x9835 }, {  979, 0x38ef },
  {  989, 0x9ad1 }, {  997, 0x2802 }, { 1000, 0xa813 }, { 1006, 0xbf69 },
  /* 0x5200 */
  { 1017, 0x65cf }, { 1027, 0x2fc6 }, { 1036, 0x6b11 }, { 1043, 0xafc9 },
  { 1053, 0x340f }, { 1060, 0x5053 }, { 1066, 0x86a2 }, { 1072, 0xa004 },
  { 1075, 0x0106 }, { 1078, 0xe809 }, { 1084, 0x3f0f }, { 1094, 0xc00e },
  { 1099, 0x0a88 }, { 1103, 0x8145 }, { 1108, 0x0010 }, { 1109, 0xc601 },
  /* 0x5300 */
  { 1114, 0xa161 }, { 1120, 0x26e1 }, { 1127, 0x444b }, { 1133, 0xce00 },
  { 1138, 0xc7aa }, { 1147, 0xd4ee }, { 1157, 0xcadf }, { 1168, 0x85bb },
  { 1177, 0x3a74 }, { 1185, 0xa520 }, { 1190, 0x436c }, { 1197, 0x8840 },
  { 1200, 0x3f06 }, { 1208, 0x8bd2 }, { 1216, 0xff79 }, { 1229, 0x3bef },
  /* 0x5400 */
  { 1241, 0xf75a }, { 1252, 0xe8ef }, { 1263, 0xfbcb }, { 1275, 0x5b36 },
  { 1284, 0x0d49 }, { 1290, 0x1bfd }, { 1301, 0x0154 }, { 1305, 0x39ee },
  { 1315, 0xd855 }, { 1323, 0x2e75 }, { 1332, 0xbfd8 }, { 1343, 0xa91a },
  { 1350, 0xf3d7 }, { 1362, 0xf6bf }, { 1375, 0x67e0 }, { 1383, 0xb40c },
  /* 0x5500 */
  { 1389, 0x82c2 }, { 1394, 0x0813 }, { 1398, 0xd49d }, { 1407, 0xd08b },
  { 1414, 0x065a }, { 1420, 0x1061 }, { 1424, 0x74f2 }, { 1433, 0x59e0 },
  { 1440, 0x8f9f }, { 1451, 0xb312 }, { 1458, 0x0080 }, { 1459, 0x6aaa },
  { 1467, 0x3230 }, { 1472, 0xb05e }, { 1480, 0x9d7a }, { 1490, 0x60ac },
  /* 0x5600 */
  { 1496, 0xd303 }, { 1503, 0xc900 }, { 1507, 0x3098 }, { 1512, 0x8a56 },
  { 1519, 0x7000 }, { 1522, 0x1390 }, { 1527, 0x1f14 }, { 1534, 0x1842 },
  { 1538, 0xc060 }, { 1542, 0x0008 }, { 1543, 0x8008 }, { 1545, 0x1080 },
  { 1547, 0x0400 }, { 1548, 0xec90 }, { 1555, 0x2817 }, { 1561, 0xe633 },
  /* 0x5700 */
  { 1570, 0x0758 }, { 1576, 0x9000 }, { 1578, 0xf708 }, { 1586, 0x4e09 },
  { 1592, 0xf485 }, { 1600, 0xfc83 }, { 1609, 0xaf53 }, { 1619, 0x18c8 },
  { 1624, 0x187c }, { 1631, 0x080c }, { 1634, 0x6adf }, { 1645, 0x0114 },
  { 1648, 0xc80c }, { 1653, 0xa734 }, { 1661, 0xa011 }, { 1665, 0x2710 },
  /* 0x5800 */
  { 1670, 0x28c5 }, { 1676, 0x4222 }, { 1680, 0x0413 }, { 1684, 0x0021 },
  { 1686, 0x3010 }, { 1689, 0x4112 }, { 1693, 0x1820 }, { 1696, 0x4000 },
  { 1697, 0x022b }, { 1702, 0xc60c }, { 1708, 0x0300 }, { 1710, 0x1000 },
  { 1711, 0x0022 }, { 1713, 0x0022 }, { 1715, 0x5810 }, { 1719, 0x0249 },
  /* 0x5900 */
  { 1723, 0xa094 }, { 1728, 0x9670 }, { 1735, 0xeeb0 }, { 1744, 0x1792 },
  { 1751, 0xcb96 }, { 1760, 0x05f2 }, { 1767, 0x0025 }, { 1770, 0x2358 },
  { 1776, 0x25de }, { 1785, 0x42cc }, { 1791, 0xcf38 }, { 1800, 0x4a04 },
  { 1804, 0x0c40 }, { 1807, 0x359f }, { 1817, 0x1128 }, { 1821, 0x8a00 },
  /* 0x5a00 */
  { 1824, 0x13fa }, { 1833, 0x910a }, { 1838, 0x0229 }, { 1842, 0x1056 },
  { 1847, 0x0641 }, { 1851, 0x0420 }, { 1853, 0x0484 }, { 1856, 0x84f0 },
  { 1862, 0x0000 }, { 1862, 0x0c04 }, { 1865, 0x0400 }, { 1866, 0x412c },
  { 1871, 0x1206 }, { 1875, 0x1154 }, { 1880, 0x0a4b }, { 1886, 0x0002 },
  /* 0x5b00 */
  { 1887, 0x0200 }, { 1888, 0x00c0 }, { 1890, 0x0000 }, { 1890, 0x0094 },
  { 1893, 0x0001 }, { 1894, 0xbfbb }, { 1907, 0x167c }, { 1915, 0x242b },
  { 1921, 0x9bbb }, { 1932, 0x7fa8 }, { 1942, 0x0c7f }, { 1951, 0xe379 },
  { 1961, 0x10f4 }, { 1967, 0xe00d }, { 1973, 0x4132 }, { 1978, 0x9f01 },
  /* 0x5c00 */
  { 1985, 0x8652 }, { 1991, 0x3572 }, { 1999, 0x10b4 }, { 2004, 0xff12 },
  { 2014, 0xcf27 }, { 2024, 0x4223 }, { 2029, 0xc06b }, { 2036, 0x8602 },
  { 2040, 0x3106 }, { 2045, 0x1fd3 }, { 2055, 0x3a0c }, { 2061, 0xa1aa },
  { 2068, 0x0812 }, { 2071, 0x0204 }, { 2073, 0x2572 }, { 2080, 0x0801 },
  /* 0x5d00 */
  { 2082, 0x40cc }, { 2087, 0x4850 }, { 2091, 0x62d0 }, { 2097, 0x6010 },
  { 2100, 0x1c80 }, { 2104, 0x2900 }, { 2107, 0x9a00 }, { 2111, 0x0010 },
  { 2112, 0x0004 }, { 2113, 0x2200 }, { 2115, 0x0000 }, { 2115, 0x0080 },
  { 2116, 0x2020 }, { 2118, 0x6800 }, { 2121, 0xcbe6 }, { 2131, 0x609e },
  /* 0x5e00 */
  { 2138, 0x916e }, { 2146, 0x3f73 }, { 2157, 0x60c0 }, { 2161, 0x3982 },
  { 2167, 0x1034 }, { 2171, 0x4830 }, { 2175, 0x0006 }, { 2177, 0xbd5c },
  { 2187, 0x8cd1 }, { 2194, 0xd6fb }, { 2206, 0x20e1 }, { 2211, 0x43e8 },
  { 2218, 0x0600 }, { 2220, 0x084e }, { 2225, 0x0500 }, { 2227, 0xc4d0 },
  /* 0x5f00 */
  { 2233, 0x8d1f }, { 2242, 0x89aa }, { 2249, 0xa6e1 }, { 2257, 0x1602 },
  { 2261, 0x0001 }, { 2262, 0x21ed }, { 2270, 0x3656 }, { 2278, 0x1a8b },
  { 2285, 0x1fb7 }, { 2296, 0x13a5 }, { 2303, 0x6502 }, { 2308, 0x30a0 },
  { 2312, 0xb278 }, { 2320, 0x23c7 }, { 2328, 0x6c93 }, { 2336, 0xe922 },
  /* 0x6000 */
  { 2343, 0xe47f }, { 2354, 0x3a74 }, { 2362, 0x8fe3 }, { 2372, 0x9820 },
  { 2376, 0x280e }, { 2381, 0x2625 }, { 2387, 0xbf9c }, { 2398, 0xbf49 },
  { 2408, 0x3218 }, { 2413, 0xac54 }, { 2420, 0xb949 }, { 2428, 0x1916 },
  { 2434, 0x0c60 }, { 2438, 0xb522 }, { 2445, 0xfbc1 }, { 2455, 0x0659 },
  /* 0x6100 */
  { 2461, 0xe343 }, { 2469, 0x8420 }, { 2472, 0x08d9 }, { 2478, 0x8000 },
  { 2479, 0x5500 }, { 2483, 0x2022 }, { 2486, 0x0184 }, { 2489, 0x00a1 },
  { 2492, 0x4800 }, { 2494, 0x2010 }, { 2496, 0x1380 }, { 2500, 0x4080 },
  { 2502, 0x0d04 }, { 2506, 0x0016 }, { 2509, 0x0040 }, { 2510, 0x8020 },
  /* 0x6200 */
  { 2512, 0xfd40 }, { 2520, 0x8de7 }, { 2530, 0x5436 }, { 2537, 0xe098 },
  { 2543, 0x7b8b }, { 2553, 0x091e }, { 2559, 0xfec8 }, { 2569, 0xd249 },
  { 2576, 0x0611 }, { 2580, 0x8dee }, { 2590, 0x1937 }, { 2598, 0xba22 },
  { 2605, 0x77f4 }, { 2616, 0x9fdd }, { 2628, 0xf3ec }, { 2639, 0xf0da },
  /* 0x6300 */
  { 2648, 0x4386 }, { 2654, 0xec42 }, { 2661, 0x8d3f }, { 2671, 0x2604 },
  { 2675, 0xfa6c }, { 2685, 0xc021 }, { 2689, 0x628e }, { 2696, 0x0cc2 },
  { 2701, 0xd785 }, { 2710, 0x0145 }, { 2714, 0x77ad }, { 2725, 0x5599 },
  { 2733, 0xe250 }, { 2739, 0x4045 }, { 2743, 0x260b }, { 2749, 0xa154 },
  /* 0x6400 */
  { 2755, 0x9827 }, { 2762, 0x5819 }, { 2768, 0x3443 }, { 2774, 0xa410 },
  { 2778, 0x05f2 }, { 2785, 0x4114 }, { 2789, 0x2280 }, { 2792, 0x0700 },
  { 2795, 0x00b4 }, { 2799, 0x4266 }, { 2805, 0x7210 }, { 2810, 0x15a1 },
  { 2816, 0x6025 }, { 2821, 0x4185 }, { 2826, 0x0054 }, { 2829, 0x0000 },
  /* 0x6500 */
  { 2829, 0x0201 }, { 2831, 0x0104 }, { 2833, 0xc820 }, { 2837, 0xcb70 },
  { 2845, 0x9320 }, { 2850, 0x6a62 }, { 2857, 0x184c }, { 2862, 0x0095 },
  { 2866, 0x1880 }, { 2869, 0x9a8b }, { 2877, 0xaab2 }, { 2885, 0x3201 },
  { 2889, 0xd87a }, { 2898, 0x00c4 }, { 2901, 0xf3e5 }, { 2912, 0x04c3 },
  /* 0x6600 */
  { 2917, 0xd44d }, { 2925, 0xa238 }, { 2931, 0xa1a1 }, { 2937, 0x5072 },
  { 2943, 0x980a }, { 2948, 0x84fc }, { 2956, 0xc152 }, { 2962, 0x44d1 },
  { 2968, 0x1094 }, { 2972, 0x20c2 }, { 2976, 0x4180 }, { 2979, 0x4210 },
  { 2982, 0x0000 }, { 2982, 0x3a00 }, { 2986, 0x0240 }, { 2988, 0xd29d },
  /* 0x6700 */
  { 2997, 0x2f01 }, { 3003, 0xa8b1 }, { 3010, 0xbd40 }, { 3017, 0x2432 },
  { 3022, 0xd34d }, { 3031, 0xd04b }, { 3038, 0xa723 }, { 3046, 0xd0ad },
  { 3054, 0x0a92 }, { 3059, 0x75a1 }, { 3067, 0xadac }, { 3076, 0x01e9 },
  { 3082, 0x801a }, { 3086, 0x771f }, { 3097, 0x9225 }, { 3103, 0xa01b },
  /* 0x6800 */
  { 3109, 0xdfa1 }, { 3119, 0x20ca }, { 3124, 0x0602 }, { 3127, 0x738c },
  { 3135, 0x577f }, { 3147, 0x003b }, { 3152, 0x0bff }, { 3163, 0x00d0 },
  { 3166, 0x806a }, { 3171, 0x0088 }, { 3173, 0xa1c4 }, { 3179, 0x0029 },
  { 3182, 0x2a05 }, { 3187, 0x0524 }, { 3191, 0x4009 }, { 3194, 0x1623 },
  /* 0x6900 */
  { 3200, 0x6822 }, { 3205, 0x8005 }, { 3208, 0x2011 }, { 3211, 0xa211 },
  { 3216, 0x0004 }, { 3217, 0x6490 }, { 3222, 0x4849 }, { 3227, 0x1382 },
  { 3232, 0x23d5 }, { 3240, 0x1930 }, { 3245, 0x2980 }, { 3249, 0x0892 },
  { 3253, 0x5402 }, { 3257, 0x8811 }, { 3261, 0x2001 }, { 3263, 0xa004 },
  /* 0x6a00 */
  { 3266, 0x0400 }, { 3267, 0x8180 }, { 3270, 0x8502 }, { 3274, 0x6022 },
  { 3278, 0x0090 }, { 3280, 0x0b01 }, { 3284, 0x0022 }, { 3286, 0x1202 },
  { 3289, 0x4011 }, { 3292, 0x0083 }, { 3295, 0x1a01 }, { 3299, 0x0000 },
  { 3299, 0x0000 }, { 3299, 0x0000 }, { 3299, 0x0000 }, { 3299, 0x0000 },
  /* 0x6b00 */
  { 3299, 0x0000 }, { 3299, 0x0000 }, { 3299, 0x009f }, { 3305, 0x4684 },
  { 3310, 0x12c8 }, { 3315, 0x0200 }, { 3316, 0x04fc }, { 3323, 0x1a00 },
  { 3326, 0x2ede }, { 3336, 0x0c4c }, { 3341, 0x0402 }, { 3343, 0x80b8 },
  { 3348, 0xa826 }, { 3354, 0x0afc }, { 3362, 0x8c02 }, { 3366, 0x2228 },
  /* 0x6c00 */
  { 3370, 0xa0e0 }, { 3375, 0x8f7b }, { 3386, 0xc7d6 }, { 3396, 0x2135 },
  { 3402, 0x06c7 }, { 3409, 0xf8b1 }, { 3418, 0x0713 }, { 3424, 0x6255 },
  { 3431, 0x936e }, { 3440, 0x8a19 }, { 3446, 0x6efa }, { 3457, 0xfb0e },
  { 3467, 0x1630 }, { 3472, 0x48f9 }, { 3480, 0xcd2f }, { 3490, 0x7deb },
  /* 0x6d00 */
  { 3502, 0x5892 }, { 3508, 0x4e84 }, { 3514, 0x4ca0 }, { 3519, 0x7a2e },
  { 3528, 0xedea }, { 3539, 0x561e }, { 3547, 0xc649 }, { 3554, 0x1190 },
  { 3558, 0x5324 }, { 3564, 0xe83a }, { 3572, 0xcfdb }, { 3584, 0x8124 },
  { 3588, 0x18f1 }, { 3595, 0x6342 }, { 3601, 0x5853 }, { 3608, 0x1a8a },
  /* 0x6e00 */
  { 3614, 0x7420 }, { 3619, 0x24d3 }, { 3626, 0xaa3b }, { 3635, 0x0514 },
  { 3639, 0x6018 }, { 3643, 0x8958 }, { 3649, 0x4800 }, { 3651, 0xc000 },
  { 3653, 0x8268 }, { 3658, 0x9101 }, { 3662, 0x84a4 }, { 3667, 0x2cd6 },
  { 3675, 0x8886 }, { 3680, 0xc4ba }, { 3688, 0x0377 }, { 3696, 0x0210 },
  /* 0x6f00 */
  { 3698, 0x8244 }, { 3702, 0x0038 }, { 3705, 0xae11 }, { 3712, 0x404a },
  { 3716, 0x28c0 }, { 3720, 0x5100 }, { 3723, 0x6044 }, { 3727, 0x1514 },
  { 3732, 0x7310 }, { 3738, 0x1000 }, { 3739, 0x0082 }, { 3741, 0x0248 },
  { 3744, 0x0205 }, { 3747, 0x4006 }, { 3750, 0xc003 }, { 3754, 0x0000 },
  /* 0x7000 */
  { 3754, 0x0000 }, { 3754, 0x0c02 }, { 3757, 0x0008 }, { 3758, 0x0220 },
  { 3760, 0x9000 }, { 3762, 0x4000 }, { 3763, 0xb800 }, { 3767, 0xd161 },
  { 3774, 0x4621 }, { 3779, 0x3274 }, { 3786, 0xf800 }, { 3791, 0x3b8a },
  { 3799, 0x050f }, { 3805, 0x8b00 }, { 3809, 0xbbd0 }, { 3818, 0x2280 },
  /* 0x7100 */
  { 3821, 0x0600 }, { 3823, 0x0769 }, { 3830, 0x8040 }, { 3832, 0x0043 },
  { 3835, 0x5420 }, { 3839, 0x5000 }, { 3841, 0x41d0 }, { 3846, 0x250c },
  { 3851, 0x8410 }, { 3854, 0x8310 }, { 3858, 0x1101 }, { 3861, 0x0228 },
  { 3864, 0x4008 }, { 3866, 0x0030 }, { 3868, 0x40a1 }, { 3872, 0x0200 },
  /* 0x7200 */
  { 3873, 0x0040 }, { 3874, 0x2000 }, { 3875, 0x1500 }, { 3878, 0xabe3 },
  { 3888, 0x3180 }, { 3892, 0xaa44 }, { 3898, 0xc2c6 }, { 3905, 0xc624 },
  { 3911, 0xac13 }, { 3918, 0x8004 }, { 3920, 0xb000 }, { 3923, 0x03d1 },
  { 3929, 0x611e }, { 3936, 0x4285 }, { 3941, 0xf303 }, { 3949, 0x1d9f },
  /* 0x7300 */
  { 3959, 0x440a }, { 3963, 0x78e8 }, { 3971, 0x5e26 }, { 3979, 0xc392 },
  { 3986, 0x2000 }, { 3987, 0x0085 }, { 3990, 0xb001 }, { 3994, 0x4000 },
  { 3995, 0x4a90 }, { 4000, 0x8842 }, { 4004, 0xca04 }, { 4009, 0x0c8d },
  { 4015, 0xa705 }, { 4022, 0x4203 }, { 4026, 0x22a1 }, { 4031, 0x0004 },
  /* 0x7400 */
  { 4032, 0x8668 }, { 4038, 0x0c01 }, { 4041, 0x5564 }, { 4048, 0x1079 },
  { 4054, 0x0002 }, { 4055, 0xdea0 }, { 4063, 0x2000 }, { 4064, 0x40c1 },
  { 4068, 0x488b }, { 4074, 0x5001 }, { 4077, 0x0380 }, { 4080, 0x0400 },
  { 4081, 0x0000 }, { 4081, 0x5004 }, { 4084, 0xc05d }, { 4091, 0x80d0 },
  /* 0x7500 */
  { 4095, 0xa010 }, { 4098, 0x970a }, { 4105, 0xbb20 }, { 4112, 0x4daf },
  { 4122, 0xd921 }, { 4129, 0x1e10 }, { 4134, 0x0460 }, { 4137, 0x8314 },
  { 4142, 0x8848 }, { 4146, 0xa6d6 }, { 4155, 0xd83b }, { 4164, 0x733f },
  { 4175, 0x27bc }, { 4184, 0x4974 }, { 4191, 0x0ddc }, { 4199, 0x9213 },
  /* 0x7600 */
  { 4205, 0x142b }, { 4211, 0x8ba1 }, { 4218, 0x2e75 }, { 4227, 0xd139 },
  { 4235, 0x3009 }, { 4239, 0x5050 }, { 4243, 0x8808 }, { 4246, 0x6900 },
  { 4250, 0x49d4 }, { 4257, 0x024a }, { 4261, 0x4010 }, { 4263, 0x8016 },
  { 4267, 0xe564 }, { 4275, 0x89d7 }, { 4284, 0xc020 }, { 4287, 0x5316 },
  /* 0x7700 */
  { 4294, 0x2b92 }, { 4301, 0x8600 }, { 4304, 0xa345 }, { 4311, 0x15e0 },
  { 4317, 0x008b }, { 4321, 0x0c03 }, { 4325, 0x196e }, { 4333, 0xe200 },
  { 4337, 0x7031 }, { 4343, 0x8006 }, { 4346, 0x16a5 }, { 4353, 0xa829 },
  { 4359, 0x2000 }, { 4360, 0x1880 }, { 4363, 0x7aac }, { 4372, 0xe148 },
  /* 0x7800 */
  { 4378, 0x3207 }, { 4384, 0xb5d6 }, { 4394, 0x32e8 }, { 4401, 0x5f91 },
  { 4410, 0x50a1 }, { 4415, 0x20e5 }, { 4421, 0x7c00 }, { 4426, 0x1080 },
  { 4428, 0x7280 }, { 4433, 0x9d8a }, { 4441, 0x00aa }, { 4445, 0x421f },
  { 4452, 0x0e22 }, { 4457, 0x0231 }, { 4461, 0x1100 }, { 4463, 0x0494 },
  /* 0x7900 */
  { 4467, 0x0022 }, { 4469, 0x4008 }, { 4471, 0x0010 }, { 4472, 0x5c10 },
  { 4477, 0x0343 }, { 4482, 0xfcc8 }, { 4491, 0xa1a5 }, { 4498, 0x0580 },
  { 4501, 0x8433 }, { 4507, 0x0400 }, { 4508, 0x0080 }, { 4509, 0x6e08 },
  { 4515, 0x2a4b }, { 4522, 0x8126 }, { 4527, 0xaad8 }, { 4535, 0x2901 },
  /* 0x7a00 */
  { 4539, 0x684d }, { 4546, 0x4490 }, { 4550, 0x0009 }, { 4552, 0xba88 },
  { 4559, 0x0040 }, { 4560, 0x0082 }, { 4562, 0x0000 }, { 4562, 0x87d1 },
  { 4570, 0x215b }, { 4577, 0xb1e6 }, { 4586, 0x3161 }, { 4592, 0x8008 },
  { 4594, 0x0800 }, { 4595, 0xc240 }, { 4599, 0xa069 }, { 4605, 0xa600 },
  /* 0x7b00 */
  { 4609, 0x8d58 }, { 4616, 0x4a32 }, { 4622, 0x5d71 }, { 4631, 0x550a },
  { 4637, 0x9aa0 }, { 4643, 0x2d57 }, { 4652, 0x4005 }, { 4655, 0x4aa6 },
  { 4662, 0x2021 }, { 4665, 0x30b1 }, { 4671, 0x3fc6 }, { 4681, 0x0112 },
  { 4684, 0x10c2 }, { 4688, 0x260a }, { 4693, 0x4462 }, { 4698, 0x5082 },
  /* 0x7c00 */
  { 4702, 0x9880 }, { 4706, 0x8040 }, { 4708, 0x04c0 }, { 4711, 0x8100 },
  { 4713, 0x2003 }, { 4716, 0x0000 }, { 4716, 0x0000 }, { 4716, 0x3818 },
  { 4721, 0x0200 }, { 4722, 0xf1a6 }, { 4731, 0x4434 }, { 4736, 0x720e },
  { 4743, 0x35a2 }, { 4750, 0x92e0 }, { 4756, 0x8101 }, { 4759, 0x0900 },
  /* 0x7d00 */
  { 4761, 0x0400 }, { 4762, 0x0000 }, { 4762, 0x8885 }, { 4767, 0x0000 },
  { 4767, 0x0000 }, { 4767, 0x0000 }, { 4767, 0x4000 }, { 4768, 0x0080 },
  { 4769, 0x0000 }, { 4769, 0x0000 }, { 4769, 0x4040 }, { 4771, 0x0000 },
  { 4771, 0x0000 }, { 4771, 0x0000 }, { 4771, 0x0000 }, { 4771, 0x0000 },
  /* 0x7e00 */
  { 4771, 0x0000 }, { 4771, 0x0000 }, { 4771, 0x0000 }, { 4771, 0x0800 },
  { 4772, 0x0082 }, { 4774, 0x0000 }, { 4774, 0x0000 }, { 4774, 0x0000 },
  { 4774, 0x0004 }, { 4775, 0x8800 }, { 4777, 0xbfff }, { 4792, 0xe7ef },
  { 4805, 0xffff }, { 4821, 0xffbf }, { 4836, 0xefef }, { 4850, 0xfdff },
  /* 0x7f00 */
  { 4865, 0xfbff }, { 4880, 0xbffe }, { 4894, 0xffff }, { 4910, 0x057f },
  { 4919, 0x0034 }, { 4922, 0x85b3 }, { 4930, 0x4706 }, { 4936, 0x4216 },
  { 4941, 0x5402 }, { 4945, 0xe410 }, { 4950, 0x8092 }, { 4954, 0xb305 },
  { 4961, 0x5422 }, { 4966, 0x8130 }, { 4970, 0x4263 }, { 4976, 0x180b },
  /* 0x8000 */
  { 4981, 0x387b }, { 4990, 0x13f5 }, { 4999, 0x07e5 }, { 5007, 0xa9ea },
  { 5016, 0x3c4c }, { 5023, 0x0514 }, { 5027, 0x0600 }, { 5029, 0x8002 },
  { 5031, 0x1ad9 }, { 5039, 0xbd48 }, { 5047, 0xee37 }, { 5058, 0xf496 },
  { 5067, 0x705f }, { 5076, 0x7ec0 }, { 5084, 0xbfb2 }, { 5095, 0x355f },
  /* 0x8100 */
  { 5105, 0xe644 }, { 5112, 0x455f }, { 5121, 0x9000 }, { 5123, 0x4146 },
  { 5128, 0x1d40 }, { 5133, 0x063b }, { 5140, 0x62a1 }, { 5146, 0xfe13 },
  { 5156, 0x8505 }, { 5161, 0x3902 }, { 5166, 0x0548 }, { 5170, 0x0c08 },
  { 5173, 0x144f }, { 5180, 0x0000 }, { 5180, 0x3488 }, { 5185, 0x5818 },
  /* 0x8200 */
  { 5190, 0x3077 }, { 5198, 0xd815 }, { 5205, 0xbd0e }, { 5214, 0x4bfb },
  { 5225, 0x8a90 }, { 5230, 0x8500 }, { 5233, 0xc100 }, { 5236, 0xe61d },
  { 5245, 0xed14 }, { 5253, 0xb386 }, { 5261, 0xff72 }, { 5273, 0x639b },
  { 5282, 0xfd92 }, { 5292, 0xd9be }, { 5303, 0x887b }, { 5311, 0x0a92 },
  /* 0x8300 */
  { 5316, 0xd3fe }, { 5328, 0x1cb2 }, { 5335, 0xb980 }, { 5341, 0x177a },
  { 5350, 0x82c9 }, { 5356, 0xdc17 }, { 5365, 0xfffb }, { 5380, 0x3980 },
  { 5385, 0x4260 }, { 5389, 0x590c }, { 5395, 0x0f01 }, { 5400, 0x37df },
  { 5412, 0x94a3 }, { 5419, 0xb150 }, { 5425, 0x0623 }, { 5430, 0x2307 },
  /* 0x8400 */
  { 5436, 0xf85a }, { 5445, 0x3102 }, { 5449, 0x01f0 }, { 5454, 0x3102 },
  { 5458, 0x0040 }, { 5459, 0x1e82 }, { 5465, 0x3a0a }, { 5471, 0x056a },
  { 5477, 0x5b84 }, { 5484, 0x1280 }, { 5487, 0x8002 }, { 5489, 0xa714 },
  { 5496, 0x2612 }, { 5501, 0xa04b }, { 5507, 0x1069 }, { 5512, 0x9001 },
  /* 0x8500 */
  { 5515, 0x1000 }, { 5516, 0x848a }, { 5521, 0x1802 }, { 5524, 0x3f80 },
  { 5531, 0x0708 }, { 5535, 0x4240 }, { 5538, 0x0110 }, { 5540, 0x4e14 },
  { 5546, 0x80b0 }, { 5550, 0x1800 }, { 5552, 0xc510 }, { 5557, 0x0281 },
  { 5560, 0x8202 }, { 5563, 0x1029 }, { 5567, 0x0210 }, { 5569, 0x8800 },
  /* 0x8600 */
  { 5571, 0x0020 }, { 5572, 0x0042 }, { 5574, 0x0280 }, { 5576, 0x1100 },
  { 5578, 0xe000 }, { 5581, 0x4413 }, { 5586, 0x5804 }, { 5590, 0xfe02 },
  { 5598, 0x3c07 }, { 5605, 0x3028 }, { 5609, 0x9798 }, { 5617, 0x0473 },
  { 5623, 0xced1 }, { 5632, 0xcb13 }, { 5640, 0x6210 }, { 5644, 0x431f },
  /* 0x8700 */
  { 5652, 0x278d }, { 5660, 0x55ac }, { 5668, 0x422e }, { 5674, 0xc892 },
  { 5680, 0x5380 }, { 5685, 0x0288 }, { 5688, 0x4039 }, { 5693, 0x7851 },
  { 5700, 0x292c }, { 5706, 0x8088 }, { 5709, 0xb900 }, { 5714, 0x2428 },
  { 5718, 0x0c41 }, { 5722, 0x080e }, { 5726, 0x4421 }, { 5730, 0x4200 },
  /* 0x8800 */
  { 5732, 0x0408 }, { 5734, 0x0868 }, { 5738, 0x0006 }, { 5740, 0x1204 },
  { 5743, 0x3031 }, { 5748, 0x0290 }, { 5751, 0x5b3e }, { 5761, 0xe085 },
  { 5767, 0x2936 }, { 5774, 0x1044 }, { 5777, 0x2814 }, { 5781, 0x1082 },
  { 5784, 0x4266 }, { 5790, 0x8334 }, { 5796, 0x013c }, { 5801, 0x531b },
  /* 0x8900 */
  { 5809, 0x0404 }, { 5811, 0x0e0d }, { 5817, 0x0c22 }, { 5821, 0x0051 },
  { 5824, 0x0012 }, { 5826, 0xc000 }, { 5828, 0x0040 }, { 5829, 0x8800 },
  { 5831, 0x004a }, { 5834, 0x0000 }, { 5834, 0x0000 }, { 5834, 0x0000 },
  { 5834, 0xdff6 }, { 5847, 0x5447 }, { 5854, 0x8868 }, { 5859, 0x0008 },
  /* 0x8a00 */
  { 5860, 0x0081 }, { 5862, 0x0000 }, { 5862, 0x0000 }, { 5862, 0x4000 },
  { 5863, 0x0100 }, { 5864, 0x0000 }, { 5864, 0x0000 }, { 5864, 0x0200 },
  { 5865, 0x0600 }, { 5867, 0x0008 }, { 5868, 0x0000 }, { 5868, 0x0000 },
  { 5868, 0x0000 }, { 5868, 0x0000 }, { 5868, 0x0000 }, { 5868, 0x0000 },
  /* 0x8b00 */
  { 5868, 0x0080 }, { 5869, 0x0000 }, { 5869, 0x0040 }, { 5870, 0x0000 },
  { 5870, 0x0000 }, { 5870, 0x0000 }, { 5870, 0x1040 }, { 5872, 0x0000 },
  { 5872, 0x0000 }, { 5872, 0x0000 }, { 5872, 0xefff }, { 5887, 0xf7fd },
  { 5901, 0xff7f }, { 5916, 0xfffe }, { 5931, 0xfbff }, { 5946, 0xffff },
  /* 0x8c00 */
  { 5962, 0xfdff }, { 5977, 0xbfff }, { 5992, 0xffff }, { 6008, 0x00ff },
  { 6016, 0x12c2 }, { 6021, 0x0420 }, { 6023, 0x0c06 }, { 6027, 0x0708 },
  { 6031, 0x1624 }, { 6036, 0x0110 }, { 6038, 0x0000 }, { 6038, 0x0000 },
  { 6038, 0x0000 }, { 6038, 0x0000 }, { 6038, 0x0000 }, { 6038, 0x0000 },
  /* 0x8d00 */
  { 6038, 0x0000 }, { 6038, 0xe000 }, { 6041, 0xfffe }, { 6056, 0xffff },
  { 6072, 0xffff }, { 6088, 0x7f79 }, { 6100, 0x28df }, { 6109, 0x00f9 },
  { 6115, 0x0c32 }, { 6120, 0x8012 }, { 6123, 0x0008 }, { 6124, 0xd53a },
  { 6133, 0xd858 }, { 6140, 0xecc2 }, { 6148, 0x9d18 }, { 6155, 0x2fa8 },
  /* 0x8e00 */
  { 6163, 0x9620 }, { 6168, 0xe010 }, { 6172, 0xd60c }, { 6179, 0x2622 },
  { 6184, 0x0f97 }, { 6193, 0x0206 }, { 6196, 0xb240 }, { 6201, 0x9055 },
  { 6207, 0x80a2 }, { 6211, 0x5011 }, { 6215, 0x9800 }, { 6218, 0x0404 },
  { 6220, 0x4000 }, { 6221, 0x0000 }, { 6221, 0x0000 }, { 6221, 0x0000 },
  /* 0x8f00 */
  { 6221, 0x0000 }, { 6221, 0x0000 }, { 6221, 0x0000 }, { 6221, 0x0000 },
  { 6221, 0x0000 }, { 6221, 0x0000 }, { 6221, 0xfbc0 }, { 6230, 0xffff },
  { 6246, 0xeffe }, { 6260, 0xdffb }, { 6274, 0x0b08 }, { 6278, 0x6243 },
  { 6284, 0x41b6 }, { 6291, 0xfb3b }, { 6303, 0x6f74 }, { 6313, 0x2389 },
  /* 0x9000 */
  { 6319, 0xae7f }, { 6331, 0xecd7 }, { 6342, 0xe047 }, { 6349, 0x5960 },
  { 6355, 0xa096 }, { 6361, 0x098f }, { 6368, 0x612c }, { 6374, 0xa030 },
  { 6378, 0x090d }, { 6383, 0x2aaa }, { 6390, 0xd44e }, { 6398, 0x4f7b },
  { 6409, 0xc4b2 }, { 6416, 0x388b }, { 6423, 0xa9c6 }, { 6431, 0x6110 },
  /* 0x9100 */
  { 6435, 0x0014 }, { 6437, 0x4200 }, { 6439, 0x800c }, { 6442, 0x0202 },
  { 6444, 0xfe48 }, { 6453, 0x6485 }, { 6459, 0xd63e }, { 6469, 0xe3f7 },
  { 6481, 0x3aa0 }, { 6487, 0x0c07 }, { 6492, 0xe40c }, { 6498, 0x0430 },
  { 6501, 0xf680 }, { 6508, 0x1002 }, { 6510, 0x0000 }, { 6510, 0x0000 },
  /* 0x9200 */
  { 6510, 0x0000 }, { 6510, 0x0000 }, { 6510, 0x0000 }, { 6510, 0x0000 },
  { 6510, 0x0000 }, { 6510, 0x0000 }, { 6510, 0x0000 }, { 6510, 0x0010 },
  { 6511, 0x4000 }, { 6512, 0x0000 }, { 6512, 0x4000 }, { 6513, 0x0000 },
  { 6513, 0x0100 }, { 6514, 0x0000 }, { 6514, 0x0000 }, { 6514, 0x0000 },
  /* 0x9300 */
  { 6514, 0x0000 }, { 6514, 0x0000 }, { 6514, 0x0000 }, { 6514, 0x4000 },
  { 6515, 0x0000 }, { 6515, 0x0000 }, { 6515, 0x0400 }, { 6516, 0x0000 },
  { 6516, 0x8000 }, { 6517, 0x0000 }, { 6517, 0x0000 }, { 6517, 0x0000 },
  { 6517, 0x0400 }, { 6518, 0x0040 }, { 6519, 0x0000 }, { 6519, 0x0000 },
  /* 0x9400 */
  { 6519, 0x0000 }, { 6519, 0x0000 }, { 6519, 0x0000 }, { 6519, 0x4000 },
  { 6520, 0x0000 }, { 6520, 0x0000 }, { 6520, 0x0800 }, { 6521, 0x0000 },
  { 6521, 0xffe0 }, { 6532, 0xfebd }, { 6545, 0xffff }, { 6561, 0xffff },
  { 6577, 0x7f7f }, { 6591, 0xfbe7 }, { 6604, 0xffbf }, { 6619, 0xf7ff },
  /* 0x9500 */
  { 6634, 0xffff }, { 6650, 0xefff }, { 6665, 0xff7e }, { 6679, 0xdff7 },
  { 6693, 0xf6f7 }, { 6706, 0xfbdf }, { 6720, 0xbffe }, { 6734, 0x804f },
  { 6740, 0x0000 }, { 6740, 0x0000 }, { 6740, 0x0000 }, { 6740, 0x0000 },
  { 6740, 0x0000 }, { 6740, 0x0000 }, { 6740, 0xef00 }, { 6747, 0x7fff },
  /* 0x9600 */
  { 6762, 0xff7f }, { 6777, 0xb6f7 }, { 6789, 0x4406 }, { 6793, 0xb87e },
  { 6803, 0x3bf5 }, { 6814, 0x8831 }, { 6819, 0x1796 }, { 6827, 0x00f4 },
  { 6832, 0xa960 }, { 6838, 0x1391 }, { 6844, 0x0080 }, { 6845, 0x7249 },
  { 6852, 0xf2f3 }, { 6863, 0x0024 }, { 6865, 0x8701 }, { 6870, 0x42c8 },
  /* 0x9700 */
  { 6875, 0xe3d3 }, { 6885, 0x5048 }, { 6889, 0x2400 }, { 6891, 0x4305 },
  { 6896, 0x0000 }, { 6896, 0x4a4c }, { 6902, 0x0227 }, { 6907, 0x1058 },
  { 6911, 0x2820 }, { 6914, 0x0116 }, { 6918, 0xa809 }, { 6923, 0x0014 },
  { 6925, 0x0000 }, { 6925, 0x0000 }, { 6925, 0x3ec0 }, { 6932, 0x0068 },
  /* 0x9800 */
  { 6935, 0x0000 }, { 6935, 0x0000 }, { 6935, 0x0000 }, { 6935, 0x0000 },
  { 6935, 0x0000 }, { 6935, 0x0000 }, { 6935, 0x0000 }, { 6935, 0xffe0 },
  { 6946, 0xb7ff }, { 6960, 0xfddb }, { 6973, 0x00f7 }, { 6980, 0x0000 },
  { 6980, 0x4000 }, { 6981, 0xc72e }, { 6990, 0x0180 }, { 6992, 0x0000 },
  /* 0x9900 */
  { 6992, 0x2000 }, { 6993, 0x0001 }, { 6994, 0x4000 }, { 6995, 0x0000 },
  { 6995, 0x0000 }, { 6995, 0x0030 }, { 6997, 0xffa8 }, { 7008, 0xb4f7 },
  { 7019, 0xadf3 }, { 7030, 0x03ff }, { 7040, 0x0120 }, { 7042, 0x0000 },
  { 7042, 0x0000 }, { 7042, 0x0000 }, { 7042, 0x0000 }, { 7042, 0x0000 },
  /* 0x9a00 */
  { 7042, 0x0000 }, { 7042, 0x0000 }, { 7042, 0x0000 }, { 7042, 0x0000 },
  { 7042, 0x0000 }, { 7042, 0x0000 }, { 7042, 0xf000 }, { 7046, 0xfffb },
  { 7061, 0x9df7 }, { 7073, 0xfdcf }, { 7086, 0x01bf }, { 7094, 0x15c3 },
  { 7101, 0x1827 }, { 7107, 0x810a }, { 7111, 0xa842 }, { 7116, 0x0a00 },
  /* 0x9b00 */
  { 7118, 0x8108 }, { 7121, 0x8008 }, { 7123, 0x8008 }, { 7125, 0x1804 },
  { 7128, 0xa3be }, { 7138, 0x0012 }, { 7140, 0x0000 }, { 7140, 0x0000 },
  { 7140, 0x0000 }, { 7140, 0x0000 }, { 7140, 0x0000 }, { 7140, 0x0000 },
  { 7140, 0x0000 }, { 7140, 0x0000 }, { 7140, 0x0000 }, { 7140, 0x0000 },
  /* 0x9c00 */
  { 7140, 0x0000 }, { 7140, 0x0000 }, { 7140, 0x0000 }, { 7140, 0x0000 },
  { 7140, 0x0000 }, { 7140, 0x0000 }, { 7140, 0x0000 }, { 7140, 0x9000 },
  { 7142, 0x69e6 }, { 7151, 0xdc37 }, { 7161, 0x6bff }, { 7174, 0x3dff },
  { 7187, 0xfcf8 }, { 7198, 0xf3f9 }, { 7210, 0x0004 },
};
static const Summary16 gb2312_uni2indx_page9e[27] = {
  /* 0x9e00 */
  { 7211, 0x0000 }, { 7211, 0x8000 }, { 7212, 0xbf6f }, { 7225, 0xe7ee },
  { 7237, 0xdffe }, { 7251, 0x5da2 }, { 7259, 0x3fd8 }, { 7269, 0xc00b },
  { 7274, 0x0984 }, { 7278, 0xa00c }, { 7282, 0x0040 }, { 7283, 0x6910 },
  { 7288, 0xe210 }, { 7293, 0xb912 }, { 7300, 0x86a5 }, { 7307, 0x5a00 },
  /* 0x9f00 */
  { 7311, 0x6800 }, { 7314, 0x0289 }, { 7318, 0x9005 }, { 7322, 0x6a80 },
  { 7327, 0x0010 }, { 7328, 0x0003 }, { 7330, 0x0000 }, { 7330, 0x8000 },
  { 7331, 0x1ff9 }, { 7342, 0x8e00 }, { 7346, 0x0001 },
};
static const Summary16 gb2312_uni2indx_pageff[15] = {
  /* 0xff00 */
  { 7347, 0xfffe }, { 7362, 0xffff }, { 7378, 0xffff }, { 7394, 0xffff },
  { 7410, 0xffff }, { 7426, 0x7fff }, { 7441, 0x0000 }, { 7441, 0x0000 },
  { 7441, 0x0000 }, { 7441, 0x0000 }, { 7441, 0x0000 }, { 7441, 0x0000 },
  { 7441, 0x0000 }, { 7441, 0x0000 }, { 7441, 0x002b },
};

INTERNAL int gb2312_wctomb_zint(unsigned int* r, unsigned int wc) {
    const Summary16 *summary = NULL;
    if (wc < 0x0460) {
        if (wc == 0x00b7) { /* ZINT: Patched to duplicate map to 0xA1A4 */
            *r = 0xA1A4;
            return 2;
        }
        summary = &gb2312_uni2indx_page00[(wc>>4)];
    } else if (wc >= 0x2000 && wc < 0x2650) {
        if (wc == 0x2014) { /* ZINT: Patched to duplicate map to 0xA1AA */
            *r = 0xA1AA;
            return 2;
        }
        summary = &gb2312_uni2indx_page20[(wc>>4)-0x200];
    } else if (wc >= 0x3000 && wc < 0x3230) {
        summary = &gb2312_uni2indx_page30[(wc>>4)-0x300];
    } else if (wc >= 0x4e00 && wc < 0x9cf0) {
        summary = &gb2312_uni2indx_page4e[(wc>>4)-0x4e0];
    } else if (wc >= 0x9e00 && wc < 0x9fb0) {
        summary = &gb2312_uni2indx_page9e[(wc>>4)-0x9e0];
    } else if (wc >= 0xff00 && wc < 0xfff0) {
        summary = &gb2312_uni2indx_pageff[(wc>>4)-0xff0];
    }
    if (summary) {
        unsigned short used = summary->used;
        unsigned int i = wc & 0x0f;
        if (used & ((unsigned short) 1 << i)) {
            /* Keep in 'used' only the bits 0..i-1. */
            used &= ((unsigned short) 1 << i) - 1;
            /* Add 'summary->indx' and the number of bits set in 'used'. */
            used = (used & 0x5555) + ((used & 0xaaaa) >> 1);
            used = (used & 0x3333) + ((used & 0xcccc) >> 2);
            used = (used & 0x0f0f) + ((used & 0xf0f0) >> 4);
            used = (used & 0x00ff) + (used >> 8);
            *r = gb2312_2charset[summary->indx + used];
            return 2;
        }
    }
    return 0;
}

/* Convert UTF-8 string to GB 2312 (EUC-CN) and place in array of ints */
INTERNAL int gb2312_utf8tomb(struct zint_symbol *symbol, const unsigned char source[], size_t* p_length, unsigned int* gbdata) {
    int error_number;
    unsigned int i, length;
#ifndef _MSC_VER
    unsigned int utfdata[*p_length + 1];
#else
    unsigned int* utfdata = (unsigned int*) _alloca((*p_length + 1) * sizeof(unsigned int));
#endif

    error_number = utf8_to_unicode(symbol, source, utfdata, p_length, 1 /*disallow_4byte*/);
    if (error_number != 0) {
        return error_number;
    }

    for (i = 0, length = *p_length; i < length; i++) {
        if (utfdata[i] < 0x80) {
            gbdata[i] = utfdata[i];
        } else {
            if (!gb2312_wctomb_zint(gbdata + i, utfdata[i])) {
                strcpy(symbol->errtxt, "810: Invalid character in input data");
                return ZINT_ERROR_INVALID_DATA;
            }
        }
    }

    return 0;
}

/* Convert UTF-8 string to single byte ECI and place in array of ints */
INTERNAL int gb2312_utf8tosb(int eci, const unsigned char source[], size_t* p_length, unsigned int* gbdata, int full_multibyte) {
    int error_number;
#ifndef _MSC_VER
    unsigned char single_byte[*p_length + 1];
#else
    unsigned char* single_byte = (unsigned char*) _alloca(*p_length + 1);
#endif

    error_number = utf_to_eci(eci, source, single_byte, p_length);
    if (error_number != 0) {
        /* Note not setting `symbol->errtxt`, up to caller */
        return error_number;
    }

    gb2312_cpy(single_byte, p_length, gbdata, full_multibyte);

    return 0;
}

/* If `full_multibyte` set, copy byte input stream to array of ints, putting double-bytes that match GRIDMATRIX Chinese mode in a single entry.
 * If `full_multibyte` not set, do a straight copy */
INTERNAL void gb2312_cpy(const unsigned char source[], size_t* p_length, unsigned int* gbdata, int full_multibyte) {
    unsigned int i, j, length;
    unsigned char c1, c2;

    if (full_multibyte) {
        for (i = 0, j = 0, length = *p_length; i < length; i++, j++) {
            if (length - i >= 2) {
                c1 = source[i];
                c2 = source[i + 1];
                if (((c1 >= 0xA1 && c1 <= 0xA9) || (c1 >= 0xB0 && c1 <= 0xF7)) && c2 >= 0xA1 && c2 <= 0xFE) {
                    /* This may or may not be valid GB 2312 (EUC-CN), but don't care as long as it can be encoded in GRIDMATRIX Chinese mode */
                    gbdata[j] = (c1 << 8) | c2;
                    i++;
                } else {
                    gbdata[j] = c1;
                }
            } else {
                gbdata[j] = source[i];
            }
        }
        *p_length = j;
    } else {
        /* Straight copy */
        for (i = 0, length = *p_length; i < length; i++) {
            gbdata[i] = source[i];
        }
    }
}
