package Zim::Repository::Man;

use strict;
use File::Spec;
use Zim::Repository;

our $VERSION = '0.18';
our @ISA     = 'Zim::Repository';

=head1 NAME

Zim::Repository::Man - Man page repository for zim

=head1 DESCRIPTION

This module can be used to read man pages in L<zim>.
It assumes that you have the GNU man program, if not it fails silently.

It derives from L<Zim::Repository>.

=head1 METHODS

=over 4

=cut

sub init {
	my $self = shift;
	open MAN, "man -w |";
	my $path = join '', <MAN>;
	close MAN;
	$self->{path} = [grep length($_), split /:+/, $path];
	$self->{no_show_in_sidepane} = 1; # temp HACK
	#warn "MANPATH: @{$self->{path}}\n";
}

=item C<list_pages(NAMESPACE)>

Lists all manpages in a certain section.

=cut

sub list_pages {
	my ($self, $namespace) = @_;
	$namespace =~ s/^\Q$self->{namespace}\E:*//;
	return $self->list_sections unless length $namespace;
	return unless $namespace =~ /^(\d+\w*):*$/;
	my $section = $1;
	my @pages;
	for (@{$self->{path}}) {
		my $dir = File::Spec->catdir($_, "man$section");
		next unless -d $dir;
		#warn "Listing man pages in $dir\n";
		push @pages, map {s/\..*$//; $_} Zim::File->list_dir($dir);
	}
	$self->wipe_array(\@pages);
	return @pages;
}

=item C<list_sections()>

Used by C<list_pages()> when no section is given.

=cut

sub list_sections {
	my $self = shift;
	my @sections;
	for my $dir (@{$self->{path}}) {
		next unless -d $dir;
		#warn "Listing man sections in $dir\n";
		push @sections, grep s/^man(\d+\w*)/$1/, Zim::File->list_dir($dir);
	}
	$self->wipe_array(\@sections);
	return map "$_:", @sections;
}

=item C<get_page(NAME)>

Returns a page object for man page NAME.

=cut

sub get_page {
	my ($self, $name) = @_;
	#warn "Get man page: $name\n";
	
	my $n = $name;
	$n =~ s/^\Q$self->{namespace}\E:*//;
	$n =~ s/^(\d+\w*):+//;
	my $sect = $1 || '';
	
	open MAN, "man -w $sect $n |";
	my $path = join '', <MAN>;
	close MAN;
	$path = undef unless $path =~ /\S/;
	return unless $path;

	my $page = Zim::Man::Page->new($self, $name);
	$page->{format} = '_man'; # to force formatted interface
	$page->{_manpage} = [$n, $sect];
	
	return $page;
}

=item C<resolve_name(NAME, REF, EXIST)>

Case in-sensitive check whether a page exists or not.
REF is ignored since man pages don't have relative links.

=cut

sub resolve_name {
	my ($self, $name, undef, $exist) = @_;
	$name = ':'.$name;
	$name =~ s/^:*\Q$self->{namespace}\E:*(?:(\d+\w*):+)?//i;
	$name =~ s/^://;
	my $sect = lc($1) || '';
	$sect = lc($1) if $name =~ s/\((\d+\w*)\):*$//;

	my @try = ($name);
	push @try, lc $name if $name =~ /[[:upper:]]/;
	if ($name =~ /:/) {
		my $n = $name;
		$n =~ s/:+/::/; # perl modules / namespaces
		push @try, $n, lc $n;
	}

	my ($path, $n);
	for (@try) {
		$n = $_;
		#warn "Resolving $name in section $sect\n";
		open MAN, "man -w $sect $n |";
		$path = join '', <MAN>;
		close MAN;
		$path = undef unless $path =~ /\S/;
		last if $path;
	}
	$sect = lc($1) if ! $sect and $path =~ /man(\d+\w*)\W/;
	$name = $self->{namespace}.($sect ? $sect.':' : '').($n || $name);

	return $name if $path;
	return $exist ? undef : $name ;
}

package Zim::Man::Page;

use Zim::Page;

our @ISA = qw/Zim::Page/;

=item C<get_parse_tree()>

Returns parse tree for man page.

=cut

sub get_parse_tree {
	my ($self) = @_;
	return $self->{parse_tree} if defined $self->{parse_tree};
	my ($name, $sect) = @{$self->{_manpage}};
	
	$ENV{MANWIDTH} = 80; # FIXME get window size (via Env ?)
	open MAN, "man -c $sect $name |" or return undef;
	my ($block, @data);
	while (<MAN>) {
		# FIXME implement parsing algo like in Zim.pm
		# include bold and head2
		#s/((\S\cH\S)+)/<b>$1<\/b>/g;
		chomp;
		s/.\cH//g;
		if (/^[A-Z]+[A-Z\s]*$/) { # heading
			push @data, $block if length $block;
			push @data, ['head1', {}, $_];
			$block = '';
		}
		elsif (/\b[\w\:\.\-]+\(\w+\)/) { # links
			# FIXME namespace links per man section
			push @data, $block if length $block;
			while (s/(.*?)\b([\w\:\.\-]+\(\d+\w*\))//) {
				push @data, $1 if length($1);
				push @data, ['link', {to => $2}, $2];
			}
			$block = $_ . "\n";
		}
		else { $block .= $_ . "\n" }
	}
	push @data, $block if length $block;
	close MAN;
	
	$self->{parse_tree} = ['Page', {}, @data];
	return $self->{parse_tree};
}

=item C<set_parse_tree(TREE)>

Man pages can not be saved, thus it throws away all data.

=cut

sub set_parse_tree { warn "Man pages are not writable.\n" }

=item C<parse_link(LINK)>

Man pages contain no relative links, only absolute in same namespace.

=cut

sub parse_link {
	my $self = shift;
	my ($t, $l) = Zim::Formats->parse_link(@_);
	$l =~ s/^:*/$self->{namespace}/ if $t eq 'page';
	return ($t, $l);
}


1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<Zim::Repository>

=cut

