/*
 * YICS: Connect a FICS interface to the Yahoo! Chess server.
 * Copyright (C) 2004  Chris Howie
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifndef _TYPES_H
#define _TYPES_H

#include "platform.h"
#include <sys/types.h>
#if defined(__GNUC__)
#include <sys/time.h>
#else
/* This is where "struct timeval" is defined in Windows. */
#include <winsock.h>
#include <sys/timeb.h>
#endif

#define tv2ll(tv) (((i64_t)((tv).tv_sec) * 1000) + ((tv).tv_usec / 1000))

#define PLAYER_MAX 300
#define TABLE_MAX 300

#ifndef true
#define true 1
#endif

#ifndef false
#define false 0
#endif

#ifndef NULL
#define NULL ((void *)0)
#endif

typedef unsigned char uchar;
typedef uchar bool;

/*
 * From the public FICS sources.
 * Copyright (C) 1993  Richard V. Nash
 * Modifications Copyright (C) 2004  Chris Howie
 */
typedef enum {
	WHITE = 0x00,
	BLACK = 0x80
} Color;

typedef enum {
	EMPTY = 0,
	PAWN,
	KNIGHT,
	BISHOP,
	ROOK,
	QUEEN,
	KING
} Piece;

#define W_PAWN (Piece)(PAWN | WHITE)
#define W_KNIGHT (Piece)(KNIGHT | WHITE)
#define W_BISHOP (Piece)(BISHOP | WHITE)
#define W_ROOK (Piece)(ROOK | WHITE)
#define W_QUEEN (Piece)(QUEEN | WHITE)
#define W_KING (Piece)(KING | WHITE)

#define B_PAWN (Piece)(PAWN | BLACK)
#define B_KNIGHT (Piece)(KNIGHT | BLACK)
#define B_BISHOP (Piece)(BISHOP | BLACK)
#define B_ROOK (Piece)(ROOK | BLACK)
#define B_QUEEN (Piece)(QUEEN | BLACK)
#define B_KING (Piece)(KING | BLACK)

#define isblack(p) ((p) & BLACK)
#define iswhite(p) (!isblack(p))
#define iscolor(p,color) (((p) & BLACK) == (color))
#define piecetype(p) ((p) & 0x7f)
#define colorval(p) ((p) & 0x80)

#define CToggle(c) (((c) == BLACK) ? WHITE : BLACK)
/* End FICS code */

typedef Piece Board[8][8];

typedef char AlgMove[8];
typedef char DescMove[10];

typedef struct {
	uchar x1, x2, y1, y2;
	Piece promoteTo;
	AlgMove alg;
	DescMove desc;
	i64_t timeTaken;
} Move;

typedef struct _Movelist {
	Move move;	/* We don't use desc.  Much. */
	struct _Movelist *next;	/* linked list */
} Movelist;

typedef struct _Repetition {
	Board position;
	uchar times;
	Color tomove;
	struct _Repetition *next;	/* linked list */
} Repetition;

typedef struct {
	Board board;
	uchar castlewa, castlewh;
	uchar castleba, castlebh;
	char dpush;
	uchar captured;
	Movelist *movelist;
	Repetition *repetitions;
	int lastirrev;
	int halfmoves;
	Color turn;
	Move *move;
	i64_t movetime;
	struct {
		bool present;
		Move move;
	} lastmove;
} Game;

typedef struct {
	char *string;
	int length;
} String;

typedef struct {
	char *handle;
	char *lhandle;
	short rating;
	i64_t ping;
} Player;

typedef struct _Option {
	char *key;
	char *value;
	struct _Option *next;	/* linked list */
} Option;

typedef struct {
	uchar number;
	Option *options;
	uchar protection;
	Player *players[2];
	Player *host;
	Player *observers[PLAYER_MAX];
	Game *game;
	i64_t clock[2];
	bool start[2];
	bool finished;
	char *result;
	bool inprogress;
} Table;

typedef enum {
	INV_INVITE = 0,
	INV_DRAW,
	INV_ABORT,
	INV_ADJOURN
} InviteType;

typedef struct _Invite {
	Player *who;
	uchar number;
	InviteType type;
	struct _Invite *next, *last;	/* doubly-linked list */
} Invite;

typedef enum {
	VAR_NUMERIC = 0,
	VAR_BOOLEAN,
	VAR_STRING,

	VAR_END = -1
} VariableType;

typedef struct _Variable {
	const char *name;
	bool (*validator)(struct _Variable *, void *new);
	void (*postset)(struct _Variable *);
	VariableType type;
	bool locks;
	String *string;
	int number;
} Variable;

typedef void (*OpcodeHandler)(void);
typedef void (*RoomOpcodeHandler)(String *);
typedef void (*TableOpcodeHandler)(Table *, String *);
typedef void (*CommandHandler)(String *[], int);

typedef void (*StyleHandler)(Table *);

typedef struct {
	short opcode;
	OpcodeHandler handler;
} Opcode;

typedef struct {
	short opcode;
	RoomOpcodeHandler handler;
} RoomOpcode;

typedef struct {
	short opcode;
	TableOpcodeHandler handler;
} TableOpcode;

typedef struct {
	char *command;
	CommandHandler handler;
	int maxparams;
} Command;

typedef struct {
	char *alias;
	char *mapping;
} Alias;

typedef struct {
	int style;
	StyleHandler handler;
} Style;

typedef struct {
	char *name;
	int size;
	char **contents;
	void (*on_add)(const char *);
	void (*on_sub)(const char *);
} List;

/* Tables */
extern char gameType(Table *);
extern char *strGameType(Table *);

/* Options */
extern Option *createOption(const char *, const char *);
extern Option *setOption(Option *, const char *, const char *);
extern Option *destroyOption(Option *);
extern void destroyOptions(Option *);
extern Option *findOption(Option *, char *);
extern long findOptionLong(Option *, char *);

/* Strings */
extern String *StringNew(const char *, int);
extern void StringFree(String *);
extern String *StringSet(String *, const char *, int);
extern String *StringCat(String *, const char *, int);
#define StringCopy(dest,src) StringSet((dest), (src)->string, (src)->length)
#define StringNull() StringNew("", 0)
#define StringDup(src) StringNew((src)->string, (src)->length)

/* Boards */
extern void addMove(Game *, Move *);
extern uchar addRepetition(Game *);
extern void destroyMovelist(Game *);
extern void destroyRepetitions(Game *);
extern void initGame(Game *);
extern void debugGame(Game *, int, char *, Move *);

#endif
