/*
 * YH - Console Chinese Environment -
 * Copyright (C) 1999 Red Flag Linux (office@sonata.iscas.ac.cn)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY TAKASHI MANABE ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */


/*****************---impi.c----*********************/
/*
 * Input Method Platform Interface
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include "imcommon.h"
#include "impi.h"

#define OK		"+OK"
#define ERROR		"-ERR"

struct inaddr
{
	BYTE	b1;
	BYTE	b2;
	BYTE	b3;
	BYTE	b4;
};

int	maxeclen = 8;
int	eclen;
char	extcode[64];
char	candstr[256];
char	result[128];	

static int			skt;
static int			sktListen;
static struct sockaddr_in	sockAddr;
static pid_t			pid;
static int			commpipe[2];

static int	Listen();
static void	NotifyPlatform(int portListen, char *host, int portPlatform);
static void	Serve();
static void	catch_term(int signo);
static void catch_exception(int signo);


int	main(argc, argv)
int		argc;
char	*argv[];
{
	int		c;
	char	s[128];
	FILE	*pf;
	char	*host;
	int		portPlatform, portListen;

	signal(SIGTERM, catch_term);

	if (argc == 3)
	{
		host = argv[1];
		portPlatform = atoi(argv[2]);
	}
	else
	{
		host = "127.0.0.1";
		sprintf(s, "/tmp%s", ttyname(0));
		if ((pf = fopen(s, "r")) == NULL)
		{
			puts("Can't connect to YanHuang Chinese Platform.");
			exit(1);
		}

		fscanf(pf, "%d", &portPlatform);
		fclose(pf);
	}

	pipe(commpipe);

	if ((pid = fork()) != 0)
	{
		char	key[32];
		int		n;

		n = read(commpipe[0], key, sizeof(key));
		if (n > 0)
		{
			char	*p;

			key[n] = '\0';
			p = strchr(key, '\r');
			if (p != NULL)
				*p = '\0';
			strcat(imdscrpt, ":");
			printf("\033[1m%-20s%sл\033[0m\n", imdscrpt, key);
		}
		exit(0);
	}
	else
	{
		int	i;

		if (IMInit() < 0)
		{
			kill(getppid(), SIGTERM);
			exit(1);
		}

		for (i = SIGHUP; i <= SIGPROF; i++)
			signal(i, catch_exception);

		setpgrp();
		umask(0);
		close(0);

		NotifyPlatform(Listen(), host, portPlatform);
		Serve();
	}
}


static int	Listen()
{
#ifdef OPENSERVER
	int	len;
#else
	size_t	len;
#endif

	sktListen = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
	sockAddr.sin_family = PF_INET;
	sockAddr.sin_addr.s_addr = INADDR_ANY;
	sockAddr.sin_port = htons(0);
	bind(sktListen, (struct sockaddr *) &sockAddr, sizeof(sockAddr));
	listen(sktListen, 1);
	len = sizeof(sockAddr);
	getsockname(sktListen, (struct sockaddr *) &sockAddr, &len);
	return	ntohs(sockAddr.sin_port);
}


static void	NotifyPlatform(portListen, hostPlatform, portPlatform)
int		portListen;
char		*hostPlatform;
int		portPlatform;
{
#ifdef OPENSERVER
	int		len;
#else
	size_t		len;
#endif

	struct inaddr	ia;
	char		outbuf[256];

	skt = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
	sockAddr.sin_family = PF_INET;
	sockAddr.sin_port = htons(portPlatform);
	sockAddr.sin_addr.s_addr = inet_addr(hostPlatform);

#if 0
	/*
	 * Don't call gethostbyname() can reduce the size of the a.out
	 * file from 160K to 80K
	 */
	if (sockAddr.sin_addr.s_addr == INADDR_NONE)
	{
		struct hostent		*hp;

		hp = gethostbyname(hostPlatform);
		if (hp == NULL)
		{
			printf("Invalid host %s\n", hostPlatform);
			goto	onerror;
		}
		sockAddr.sin_addr = *((struct in_addr *) *hp->h_addr_list);
	}
#endif
	
	if (connect(skt, (struct sockaddr *) &sockAddr, sizeof(sockAddr)) < 0)
	{
		printf("Can't connect to Chinese platform at %s, %d\n", 
			hostPlatform, portPlatform);
		goto	onerror;
	}

	len = sizeof(sockAddr);
	getsockname(skt, (struct sockaddr *) &sockAddr, &len);
	memcpy(&ia, &sockAddr.sin_addr, sizeof(ia));
	sprintf(outbuf, "%s %d %d %d %d %d %d%s",
		imname, 
		ia.b1, ia.b2, ia.b3, ia.b4,
		portListen >> 8,
		portListen & 0xff,
		CMD_END);
	if (send(skt, outbuf, strlen(outbuf), 0) < 0)
		goto	onerror;

	close(skt);
	return;

onerror:
	close(skt);
	kill(getppid(), SIGTERM);
	exit(1);
}


static void	Serve()
{
	struct sockaddr_in	sockAddr;

#ifdef OPENSERVER
	int			len;
#else
	size_t			len;
#endif

	char			*inbuf;

	len = sizeof(sockAddr);
	skt = accept(sktListen, (struct sockaddr *) &sockAddr, &len);
	close(sktListen);

	inbuf = getresponse(skt, CMD_END);
	write(commpipe[1], inbuf, strlen(inbuf));
	close(commpipe[1]);

	while (1)
	{
		char	*buf;
		char	cmd[64];
		int		param;
		int		rv;
		char	outbuf[512];

		buf = getresponse(skt, CMD_END);
		if (buf == NULL)
			break;
		cmd[0] = '\0';
		sscanf(buf, "%s%d", cmd, &param);

		if (strcmp(cmd, "QUIT") == 0)
		{
			close(skt);
			break;
		}
		else if (strcmp(cmd, "SELECT") == 0)
			rv = IMSelect(param);
		else if (strcmp(cmd, "FILTER") == 0)
			rv = IMFilter(param);
		else
			rv = -1;

		if (rv < 0)
			sprintf(outbuf, "-ERR%s", RESPONSE_END);
		else
			/*sprintf(outbuf, "+OK\r\n%-*s\r\n%s\r\n%s%s",
				maxeclen + 1, extcode, candstr, result, RESPONSE_END);song*/
			sprintf(outbuf,"+OK\r\n%-8.12s\r\n%s\r\n%s%s",
				extcode,candstr,result,RESPONSE_END);

		if (send(skt, outbuf, strlen(outbuf), 0) < 0)
			break;
	}

	IMClearup();
	exit(0);
}


static void catch_exception(signo)
int	signo;
{
	signal(signo, catch_exception);
	printf("Received signal #%d\n", signo);
	IMClearup();
	close(skt);
	exit(1);
}


static void	catch_term(int signo)
{
	signal(signo, catch_term);
	exit(1);
}



void	*shmloadfile(path, plen, magic)
char	*path;
int		*plen;
int		magic;
{
	FILE			*pf;
	void			*buf;
	int				shm_id;
	int				readfile;

	if ((pf = fopen(path, "r")) == NULL)
	{
		*plen = 0;
		return	NULL;
	}

	fseek(pf, 0, SEEK_END);
	*plen = ftell(pf);

#if 0
	shm_id = shmgetLlL(magic, *plen, 0);
	if (shm_id > 0)
		readfile = 0;
	else
	{
		readfile = 1;
		shm_id = shmget(magic, *plen, IPC_CREAT | 0666);
		if (shm_id < 0)
		{
			*plen = 0;
			buf = NULL;
			goto	quit;
		}
	}

	buf = shmat(shm_id, NULL, 0);
	if (buf == NULL)
	{
		*plen = 0;
		goto	quit;
	}

	if (readfile)
	{
		fseek(pf, 0, SEEK_SET);
		fread(buf, 1, *plen, pf);
	}

quit:
	fclose(pf);
#else
	buf = malloc(*plen);
	fseek(pf, 0, SEEK_SET);
	fread(buf, 1, *plen, pf);
	fclose(pf);
#endif
	
	return	buf;
}


void	handle_punct(ch)
int	ch;
{
	static int	state = 1;

	switch (ch)
	{
	case '!':
		strcpy(result, "");
		break;

	case '\\':
		strcpy(result, "");
		break;

	case ';':
		strcpy(result, "");
		break;

	case ':':
		strcpy(result, "");
		break;

	case '\'':
		strcpy(result, state ? "" : "");
		state = ! state;
		break;
		
	case '"':
		strcpy(result, state ? "" : "");
		state = ! state;
		break;

	case ',':
		strcpy(result, "");
		break;

	case '.':
		strcpy(result, "");
		break;

	case '?':
		strcpy(result, "");
		break;

	case '(':
		strcpy(result, "");
		break;

	case ')':
		strcpy(result, "");
		break;

	case '[':
		strcpy(result, "");
		break;

	case ']':
		strcpy(result, "");
		break;

	case '{':
		strcpy(result, "");
		break;

	case '}':
		strcpy(result, "");
		break;

	default:
		result[0] = ch;
		result[1] = '\0';
	}
}

