/**
 * @file    callbacks.c
 * @brief   Callbacks for the various interface widgets.
 *
 *          Routines for handling interaction with the ygraph interface.
 *
 * @author  Denis Pollney
 * @date    1 Oct 2001
 *
 * @par Copyright (C) 2001-2002 Denis Pollney
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 * @par
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details
 * @par
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gdk/gdkkeysyms.h>

#include <stdlib.h>
#include <stdio.h>

#include "ygraph.h"

extern Plot* plot_data_init(GArray*);
extern void draw_frame(Plot*, gint);
extern void axis_draw(Plot*);
extern void legend_draw(Plot*);
extern void current_time_display_value();
extern void current_frame_increment(gint);
extern void current_frame_set(gint);
extern gboolean current_time_set(GtkEditable*, gpointer);
extern void draw_all_frames(gint);
extern void plot_area_draw(Plot*);
extern void all_windows_draw(void);
extern gint frame_draw_next(GtkObject*);
extern gint dataset_read_from_file(gchar*, gint);
extern void plot_data_append(Plot*, gint);
extern void plot_window_expunge(Plot*);
extern void plot_window_draw_all(Plot*);
extern void plot_window_display_all(Plot*);
extern void plot_window_show(void);
extern void plot_axes_create(Plot*);
extern void plot_window_reconfigure(Plot*);
extern void plot_window_draw_data(Plot*);
extern void plot_window_draw_time(Plot*);
extern void plot_area_zoom_start(Plot*, gint, gint);
extern void plot_area_zoom_finish(Plot*, gint, gint);
extern void plot_area_zoom_rectangle_draw(Plot*, gint, gint);
extern void plot_window_display_data(Plot*);
extern void plot_free(Plot*);
extern void data_set_free(DataSet*);
extern DataSet* plot_get_data_index(Plot*, gint);
extern void dataset_recalc(DataSet*);
extern DataSet* dataset_derivative(gint);
extern void subtract_select_dialog_create(Plot*);
extern void convergence_select_dialog_create(Plot*);
extern gint global_data_set_list_append(DataSet*);
extern void  export_to_file(Plot*, gchar*);
extern void  export_to_dir(Plot*, gchar*);
extern void  yg_export_to_mpeg(Plot*, gchar*);

guint motion_id;
GtkWidget* file_entry_A = NULL;

/*
 * PLOT WINDOW CALLBACKS.
 */

/**
 * @brief    Configure the plot window.
 *
 * @param    plot_area  The plot area widget to be drawn.
 * @param    event      The calling event.
 * @param    plot       The Plot to be drawn.
 * @returns  TRUE if successful, crash if not.
 */
gboolean
plot_window_configure(GtkWidget* plot_area, GdkEventConfigure* event,
		      Plot* plot)
{
  if (plot->pixmap)
    gdk_pixmap_unref(plot->pixmap);

  plot->pixmap = gdk_pixmap_new(plot_area->window,
				plot_area->allocation.width,
				plot_area->allocation.height,
				-1);
  plot_window_expunge(plot);
  plot_window_reconfigure(plot);

  return TRUE;
}

/**
 * @brief    Expose the plot window.
 *
 * @param    plot_area  The plot area widget to be drawn.
 * @param    event      The calling event.
 * @param    plot       The Plot to be drawn.
 * @returns  TRUE if successful, crash if not.
 */
gboolean
plot_window_expose(GtkWidget* plot_area, GdkEventExpose* event, Plot* plot)
{
  axis_draw(plot);
  legend_draw(plot);
  plot_window_draw_time(plot);
  plot_area_draw(plot);

  gdk_draw_pixmap(plot_area->window,
		  plot_area->style->fg_gc[GTK_WIDGET_STATE(plot_area)],
		  plot->pixmap,
		  event->area.x, event->area.y,
		  event->area.x, event->area.y,
		  event->area.width, event->area.height);
  return TRUE;
}

/**
 * @brief    Destroy the plot window.
 *
 * @param    plot_area  The plot area widget to be destroyed.
 * @param    event      The calling event.
 * @param    plot       The Plot in question.
 * @returns  TRUE if successful, crash if not.
 */
gboolean
plot_window_destroy(GtkWidget* plot_area, GdkEventConfigure* event,
		    Plot* plot)
{
  Plot* check_plot;
  gint i;

  for (i=0; i<global_plot_window->len; ++i)
    {
      check_plot = g_array_index(global_plot_window, Plot*, i);
      if (check_plot->plot_nbr == plot->plot_nbr)
	{
	  g_array_remove_index(global_plot_window, i);
	  break;
	}
    }
  gtk_widget_destroy(plot->window);
  plot_free(plot);

  return TRUE;
}


void
plot_window_close(Plot* plot, gint action, GtkItem* file_select_button)
{
  plot_window_destroy(NULL, NULL, plot);
}

/**
 * @brief    Close the plot window.
 *
 * @param    plot_area  The plot area widget to be closed.
 * @param    event      The calling event.
 * @param    plot       The Plot in question.
 * @returns  TRUE if successful, crash if not.
 */
gboolean
plot_window_key_press_event(GtkWidget* plot_area, GdkEventKey* event,
			  Plot* plot)
{
  if (event->keyval == GDK_p)
    gdk_window_raise(global_control_panel->window->window);

  return TRUE;
}

/**
 * @brief    Determine the location of the pointer.
 *
 *           This routine determines the location of the pointer whenever
 *           it has been moved. In the case that the right-button has
 *           been clicked (in which case plot->zoom_x_start will have
 *           been set) then it calls a routine to zoom the axes according
 *           to the pointer location.
 *
 * @param    plot_area  The plot area widget.
 * @param    event      The calling event.
 * @param    plot       The Plot in question.
 * @returns  TRUE if successful, crash if not.
 */
gboolean
plot_area_motion_notify_event(GtkWidget* plot_area, GdkEventMotion* event,
			      Plot* plot)
{
  GdkModifierType state;
  gint x;
  gint y;
  
  if (event->is_hint)
    gdk_window_get_pointer(event->window, &x, &y, &state);
  else
    {
      x = event->x;
      y = event->y;
      state = event->state;
    }
  
  if ((state & GDK_BUTTON1_MASK)  && (plot->zoom_x_start != NO_ZOOM))
    plot_area_zoom_rectangle_draw(plot, x, y);

  return TRUE;
}

/**
 * @brief    Handle a button release in the plot area.
 *
 *           If the program is not ready for a zoom (zoom_x_start has
 *           not been set) then just return. Otherwise call the zoom
 *           function with the pointer location.
 *
 * @param    plot_area  The plot area widget to be closed.
 * @param    event      The calling event.
 * @param    plot       The Plot in question.
 * @returns  TRUE if successful, crash if not.
 */
gboolean
plot_area_button_release_event(GtkWidget* plot_area, GdkEventButton* event,
			       Plot* plot)
{
  if (plot->zoom_x_start == NO_ZOOM)
    return TRUE;

  if (event->button == 1)
    plot_area_zoom_finish(plot, event->x, event->y);

  return TRUE;
}

/**
 * @brief    Handle a button press in the plot area.
 *
 *           When the button has been pressed, call the function to
 *           record the upper left-hand corner of the zoom box.
 *
 * @param    plot_area  The plot area widget to be closed.
 * @param    event      The calling event.
 * @param    plot       The Plot in question.
 * @returns  TRUE if successful, crash if not.
 */
gboolean
plot_area_button_press_event(GtkWidget* plot_area, GdkEventButton* event,
			     Plot* plot)
{
  if (event->button == 1)
      plot_area_zoom_start(plot, event->x, event->y);

  return TRUE;
}


/**
 * @brief    Handle an unzoom event.
 *
 *           If the unzoom event is called, then reset the axis ranges
 *           to their original value and redraw the plot.
 *
 * @param    plot       The Plot in question.
 * @returns  TRUE if successful, crash if not.
 */
gboolean
plot_window_unzoom(Plot* plot)
{
  plot->fixed_range = FALSE;
  plot_window_reconfigure(plot);
  plot_window_display_all(plot);

  return TRUE;
}

/**
 * @brief    Open a blank new plot window.
 */
void
plot_window_empty_new(void)
{
  GArray* data;
  Plot* plot;

  data = g_array_new(FALSE, FALSE, sizeof(gint));
  plot = plot_data_init(data);

  gtk_widget_show(plot->window);
}

/*
 * FILE SELECTION CALLBACKS.
 */

/**
 * @brief    Act on the result of a file selection.
 *
 *           Gets the filename returned by a file selection dialog
 *           and reads the specified file. Any new datasets are
 *           appended to the existing list and the plot is redrawn.
 *
 * @param    ok_button  The button which initiated the callback.
 * @param    fs         The file selection dialog.
 * @returns  TRUE if successful.
 */ 
gboolean
file_select_read(GtkObject* ok_button, GtkFileSelection* fs)
{
  GArray* new_data_array;
  Plot* plot;
  gint new_data_set_idx;
  gchar* filename;

  filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs));

  new_data_set_idx = dataset_read_from_file(filename, option_read_skip_step);
  if (new_data_set_idx == FAIL)
    return TRUE;

  plot = gtk_object_get_data(GTK_OBJECT(fs), "plot");
  
  if (plot == NULL)
    {
      new_data_array = g_array_new(FALSE, FALSE, sizeof(gint));
      g_array_append_val(new_data_array, new_data_set_idx);
      plot = plot_data_init(new_data_array);
      plot_window_show();
    }
  else
    plot_data_append(plot, new_data_set_idx);

  if (plot->current_directory)
    g_free(plot->current_directory);
  plot->current_directory = g_strdup_printf("%s/", g_dirname(filename));

  plot_window_display_all(plot);
  return TRUE;
}

/**
 * @brief    Creates a file selection dialog box for reading data into the
 *           existing window.
 *
 * @param    plot                The Plot to which any new data should be
 *                                added.
 * @param    action              The action initiating the call.
 * @param    file_select_button  The button initiating the call.
 */
void
file_select(Plot* plot, gint action, GtkItem* file_select_button)
{
  GtkWidget* fs;

  fs = gtk_file_selection_new(FILE_SELECTION_TITLE);
  gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs),
				  plot->current_directory);

  gtk_object_set_data(GTK_OBJECT(fs), "plot", plot);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
		     "clicked", GTK_SIGNAL_FUNC(file_select_read), fs);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    (gpointer) fs);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    (gpointer) fs);

  gtk_widget_show(fs);
}

/**
 * @brief    Creates a file selection dialog box for reading data into the
 *           new window.
 *
 * @param    plot                The Plot in the window which called the
 *                               file selector.
 * @param    action              The action initiating the call.
 * @param    file_select_button  The button initiating the call.
 */
void
file_select_new_window(Plot* plot, gint action, GtkItem* file_select_button)
{
  GtkWidget* fs;

  fs = gtk_file_selection_new(FILE_SELECTION_TITLE);
  gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs),
				  plot->current_directory);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
		     "clicked", GTK_SIGNAL_FUNC(file_select_read), fs);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    (gpointer) fs);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    (gpointer) fs);
  gtk_widget_show(fs);
}


/**
 * @brief    Turn on/off the drawing of dots at each data point.
 *
 * @param    plot                The Plot to be toggled.
 * @param    action              The action initiating the call.
 * @param    draw_points_toggle  The button initiating the call.
 */
void
plot_window_draw_points_toggle (Plot* plot, gint action,
				GtkItem* draw_points_toggle)
{
  plot->draw_points = !(plot->draw_points);
  option_draw_points = plot->draw_points;

  if (plot->pixmap != NULL)
    {
      plot_area_draw(plot);
      plot_window_display_data(plot);
    }
}

/**
 * @brief    Turn on/off the drawing of grid lines on the plot.
 *
 * @param    plot              The Plot to be toggled.
 * @param    action            The action initiating the call.
 * @param    draw_grid_toggle  The button initiating the call.
 */
void
plot_window_draw_grid_toggle (Plot* plot, gint action,
			      GtkItem* draw_grid_toggle)
{
  plot->draw_grid = !(plot->draw_grid);
  option_draw_grid = plot->draw_grid;

  if (plot->pixmap != NULL)
    {
      plot_area_draw(plot);
      plot_window_display_data(plot);
    }
}

/**
 * @brief    Turn on/off the drawing of range values at corners of the plot.
 *
 * @param    plot               The Plot to be toggled.
 * @param    action             The action initiating the call.
 * @param    draw_range_toggle  The button initiating the call.
 */
void
plot_window_draw_range_toggle (Plot* plot, gint action,
			       GtkItem* draw_range_toggle)
{
  plot->draw_range = !(plot->draw_range);
  option_draw_range = plot->draw_range;

  if (plot->pixmap != NULL)
    {
      plot_area_draw(plot);
      plot_window_display_data(plot);
    }
}

/**
 * @brief    Create a new plot containing the derivative of the current plot.
 *
 * @param    plot    The Plot whose derivative should be taken.
 * @param    action  The action initiating the call.
 * @param    button  The button initiating the call.
 */
void
plot_window_derivative(Plot* plot, gint action, GtkItem* button)
{
  DataSet* data_set;
  DataSet* d_data_set;
  Plot* d_plot;
  GArray* data;
  GArray* d_data;
  gint i;
  gint idx;
  gint d_data_set_idx;

  data = plot->data;
  d_data = g_array_new(FALSE, FALSE, sizeof(gint));

  for (i=0; i<data->len; ++i)
    {
      idx = g_array_index(data, gint, i);
      data_set = g_array_index(global_data_set_list, DataSet*, idx);

      d_data_set = dataset_derivative(idx);
      d_data_set_idx = global_data_set_list_append(d_data_set);

      g_array_append_val(d_data, d_data_set_idx);
    }
  
  d_plot = plot_data_init(d_data);

  gtk_widget_show(d_plot->window);
}

/*
 * CONTROL PANEL CALLBACKS.
 */

/**
 * @brief    Switch the display mode based on "play" button clicks.
 *
 * @param    play_button  The play button widget.
 * @param    event        The calling event.
 * @param    data         Data attached to the event (unused).
 * @returns  TRUE if successful, otherwise something else.
 */
gboolean
pause_play_button_click(GtkWidget* play_button, GdkEventExpose* event,
			gpointer* data)
{
  if (global_display_mode == ANIMATE_MODE)
    {
      global_display_mode = PAUSE_MODE;
      gtk_object_set(GTK_OBJECT(play_button), "GtkButton::label",
		     (gchar*) PLAY_BUTTON_LABEL, NULL);
      return TRUE;
    }

  gtk_object_set(GTK_OBJECT(play_button), "GtkButton::label",
		 (gchar*) PAUSE_BUTTON_LABEL, NULL);
  
  if ((global_display_mode == SHOW_ALL_MODE) ||
      (global_current_frame == global_last_frame))
    global_current_frame = FIRST_FRAME-1;

  global_display_mode = ANIMATE_MODE;

  gtk_timeout_add (option_animate_delay, (GtkFunction) frame_draw_next,
		   play_button);

  return TRUE;
}

/**
 * @brief    Go to the first frame of an animation due to a "start" button
 *           click, and toggle the display mode to PAUSE_MODE.
 *
 * @param    start_button  The start button widget.
 * @param    event         The calling event.
 * @param    data          Data attached to the event (unused).
 * @returns  TRUE if successful, otherwise something else.
 */
gboolean
start_button_click(GtkWidget* start_button, GdkEventExpose* event, 
		   gpointer* data)
{
  if (global_display_mode == PAUSE_MODE && global_current_frame == FIRST_FRAME)
    return TRUE;

  global_display_mode = PAUSE_MODE;

  current_frame_set(FIRST_FRAME);
  current_time_display_value();

  all_windows_draw();

  return TRUE;
}

/**
 * @brief    Go to the last frame of an animation due to a "end" button
 *           click, and toggle the display mode to PAUSE_MODE.
 *
 * @param    end_button  The end button widget.
 * @param    event       The calling event.
 * @param    data        Data attached to the event (unused).
 * @returns  TRUE if successful, otherwise something else.
 */
gboolean
end_button_click(GtkWidget* end_button, GdkEventExpose* event, gpointer* data)
{
  if ((global_display_mode == PAUSE_MODE) &&
      (global_current_frame == global_last_frame))
    return TRUE;

  global_display_mode = PAUSE_MODE;

  current_frame_set(global_last_frame);
  current_time_display_value();

  all_windows_draw();

  return TRUE;
}

/**
 * @brief    Move an animation one frame forward due to a "fwd" button click.
 *           Toggle to PAUSE_MODE and increment the frame number.
 *
 * @param    fwd_button  The forward button widget.
 * @param    event       The calling event.
 * @param    data        Data attached to the event (unused).
 * @returns  TRUE if successful, otherwise something else.
 */
gboolean
step_fwd_button_click(GtkWidget* fwd_button, GdkEventExpose* event,
		      gpointer* data)
{
  if ((global_display_mode == PAUSE_MODE) &&
      (global_current_frame == global_last_frame))
    return TRUE;

  if (global_display_mode == SHOW_ALL_MODE)
    return start_button_click(fwd_button, event, data);

  global_display_mode = PAUSE_MODE;

  current_frame_increment(1);
  current_time_display_value();

  all_windows_draw();

  return TRUE;
}

/**
 * @brief    Move an animation one frame backward due to a "back" button click.
 *           Toggle to PAUSE_MODE and increment the frame number.
 *
 * @param    back_button  The back button widget.
 * @param    event        The calling event.
 * @param    data         Data attached to the event (unused).
 * @returns  TRUE if successful, otherwise something else.
 */
gboolean
step_back_button_click(GtkWidget* back_button, GdkEventExpose* event,
		      gpointer* data)
{
  if (global_display_mode == PAUSE_MODE && global_current_frame == FIRST_FRAME)
    return TRUE;

  if (global_display_mode == SHOW_ALL_MODE)
    return end_button_click(back_button, event, data);

  global_display_mode = PAUSE_MODE;

  current_frame_increment(-1);
  current_time_display_value();

  all_windows_draw();

  return TRUE;
}

/**
 * @brief    Toggle between SHOW_ALL and single frame display modes.
 *
 * @param    showall_button  The show-all button widget.
 * @param    event           The calling event.
 * @param    data            Data attached to the event (unused).
 * @returns  TRUE if successful, otherwise something else.
 */
gboolean
show_all_button_click(GtkWidget* showall_button, GdkEventExpose* event,
		      gpointer* data)
{
  if (global_display_mode == SHOW_ALL_MODE)
    return TRUE;

  global_display_mode = SHOW_ALL_MODE;

  current_frame_set(FIRST_FRAME);

  all_windows_draw();

  return TRUE;
}

/**
 * @brief    Set the animation delay based on updated contents of the "delay"
 *           input field.
 *
 * @param    delay_entry     The delay entry field widget.
 * @param    data            Data attached to the event (unused).
 * @returns  TRUE if successful, otherwise something else.
 */
gboolean
delay_set(GtkEditable* delay_entry, gpointer data)
{
  gchar* delay_str;
  gint requested_delay;

  delay_str = gtk_editable_get_chars(GTK_EDITABLE(delay_entry), 0, -1);
  requested_delay = strtol(delay_str, NULL, 10);

  if (requested_delay >= 0)
    option_animate_delay = requested_delay;

  return TRUE;
}

/**
 * @brief    Set the frame time.
 *
 *           Set the current frame time based on updated contents of the
 *           "time" input field. The data frames whose times are closest
 *           to the requested times are displayed.
 *
 * @param    time_entry     The time entry field widget.
 * @param    data           Data attached to the event (unused).
 * @returns  TRUE if successful, otherwise something else.
 */
gboolean
current_time_set(GtkEditable* time_entry, gpointer data)
{
  gchar* time_str;
  gdouble requested_time;
  gdouble available_time;
  gint frame_nbr;
  
  if (global_time_list == NULL) 
    return TRUE;

  time_str = gtk_editable_get_chars(GTK_EDITABLE(time_entry), 0, -1);

  frame_nbr = 0;

  requested_time = g_strtod(time_str, NULL);
  if (!requested_time)
      return TRUE;

  available_time = g_array_index(global_time_list, gdouble, frame_nbr);
  while ((available_time < requested_time) && 
	 (frame_nbr < global_time_list->len))
    {
      ++frame_nbr;
      available_time = g_array_index(global_time_list, gdouble, frame_nbr);
    }
  
  if (frame_nbr == global_time_list->len)
    --frame_nbr;

  global_display_mode = PAUSE_MODE;

  current_frame_set(frame_nbr);
  current_time_display_value();

  all_windows_draw();

  return TRUE;
}

/**
 * @brief     Reload all of the currently loaded data.
 * 
 * @param     reload_button  The "reload" button.
 * @param     event          The event initiating the callback.
 * @returns   TRUE if successful, otherwise crash.
 */
gboolean
data_reload_all(GtkWidget* reload_button, GdkEvent* event)
{
  Plot* plot;
  DataSet* data_set;
  gint i;

  for(i=0; i<global_data_set_list->len; ++i)
    {
      data_set = g_array_index(global_data_set_list, DataSet*, i);
      dataset_recalc(data_set);
    }

  for (i=0; i<global_plot_window->len; ++i)
    {
      plot = g_array_index(global_plot_window, Plot*, i);
      plot_window_reconfigure(plot);
      plot_window_display_all(plot);
    }

  return TRUE;
}

/**
 * @brief     Reload all of the data in a given plot.
 * 
 * @param     plot           The Plot whose data is to be reloaded.
 * @param     action         The calling action.
 * @param     reload_button  The "reload" button.
 * @returns   TRUE if successful, otherwise crash.
 */
void
data_reload_plot(Plot* plot, gint action, GtkItem* reload_button)
{
  DataSet* data_set;
  gint i;
  
  if (plot->data == NULL)
    return;
  
  for (i=0; i<plot->data->len; ++i)
    {
      data_set = plot_get_data_index(plot, i);
      dataset_read_from_file(data_set->fname, option_read_skip_step);
    }

  plot_window_reconfigure(plot);
  plot_window_display_all(plot);
}


/**
 * @brief     Read the contents of an export file selection dialog.
 * 
 * @param     ok_button  The okay button of the file selector.
 * @param     fs         The file selection widget.
 * @returns   TRUE if successful, otherwise crash.
 */
gboolean
file_select_export(GtkObject* ok_button, GtkFileSelection* fs)
{
  gchar* filename;

  filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs));
  gtk_entry_set_text(GTK_ENTRY(file_entry_A), filename);

  return TRUE;
}


/**
 * @brief     Export a plot to a specified file.
 * 
 * @param     file_entry  The file entry widget.
 * @param     plot        The Plot to be exported.
 * @returns   TRUE if successful, otherwise crash.
 */
gboolean
export_file_set(GtkEditable* file_entry, Plot* plot)
{
  gchar* filename;

  filename = gtk_entry_get_text(GTK_ENTRY(file_entry_A));
  export_to_file(plot, filename);

  return TRUE;
}

/**
 * @brief     Export a plot to a specified directory as a series of frames.
 * 
 * @param     file_entry  The file entry widget.
 * @param     plot        The Plot to be exported.
 * @returns   TRUE if successful, otherwise crash.
 */
gboolean
export_dir_set(GtkEditable* file_entry, Plot* plot)
{
  gchar* dirname;

  dirname = gtk_entry_get_text(GTK_ENTRY(file_entry_A));
  export_to_dir(plot, dirname);

  return TRUE;
}

/**
 * @brief     Create a file selection dialog for exporting plots.
 * 
 * @param     select_button  The button calling for the file selector.
 * @param     event          The initiating event.
 * @param     plot           The Plot to be exported.
 * @returns   TRUE if successful, otherwise crash.
 */
void
image_export_file_select(GtkWidget* select_button, GdkEvent* event, Plot* plot)
{
  GtkWidget* fs;
  gchar* filename;

  fs = gtk_file_selection_new(EXPORT_TO_FILE_TITLE);
  filename = gtk_entry_get_text(GTK_ENTRY(file_entry_A));
  if (filename != NULL)
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs), filename);

  gtk_object_set_data(GTK_OBJECT(fs), "plot", plot);

  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
		     "clicked", GTK_SIGNAL_FUNC(file_select_export), fs);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    (gpointer) fs);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    (gpointer) fs);
  gtk_widget_show(fs);
}

/**
 * @brief     Create a dialog box for exporting an image to a file.
 * 
 * @param     plot           The Plot to be exported.
 * @param     action         The calling action.
 * @param     export_button  The calling button.
 * @returns   TRUE if successful, otherwise crash.
 */
void
image_export_dialog(Plot* plot, gint action, GtkItem* export_button)
{
  GtkWidget* dialog;
  GtkWidget* label;
  GtkWidget* hbox_A;
  GtkWidget* file_label_A;
  GtkWidget* file_button_A;
  GtkWidget* okay_button;
  GtkWidget* cancel_button;

  dialog = gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW(dialog), EXPORT_TO_FILE_TITLE);
  gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, FALSE, TRUE);

  label = gtk_label_new(EXPORT_FILE_SELECT_MESSAGE);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), label);

  hbox_A = gtk_hbox_new(FALSE, 0);
  gtk_widget_show(hbox_A);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox_A, FALSE,
		     FALSE, 0);

  file_label_A = gtk_label_new(EXPORT_FILE_LABEL);
  gtk_widget_show(file_label_A);
  gtk_box_pack_start(GTK_BOX(hbox_A), file_label_A, FALSE, FALSE, 0);

  file_entry_A = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(file_entry_A), FILE_STR_SIZE);
  gtk_entry_set_text(GTK_ENTRY(file_entry_A), "");

  gtk_widget_show(file_entry_A);
  gtk_box_pack_start(GTK_BOX(hbox_A), file_entry_A, FALSE, FALSE, 0);
  gtk_widget_set_usize(file_entry_A, FILE_ENTRY_LENGTH, -2);
  gtk_signal_connect(GTK_OBJECT(file_entry_A), "activate",
		     GTK_SIGNAL_FUNC(export_file_set), plot);
  gtk_signal_connect_object(GTK_OBJECT(file_entry_A), "activate",
			    GTK_SIGNAL_FUNC (gtk_widget_destroy),
			    GTK_OBJECT(dialog));


  file_button_A = gtk_button_new_with_label(SELECT_LABEL);
  gtk_signal_connect(GTK_OBJECT(file_button_A), "clicked",
		     GTK_SIGNAL_FUNC(image_export_file_select), plot);
  gtk_widget_show(file_button_A);
  gtk_box_pack_start(GTK_BOX(hbox_A), file_button_A, FALSE, FALSE, 0);

  okay_button = gtk_button_new_with_label(OKAY_BUTTON_LABEL);
  gtk_signal_connect(GTK_OBJECT(okay_button),
		     "clicked", GTK_SIGNAL_FUNC(export_file_set), plot);
  gtk_signal_connect_object (GTK_OBJECT (okay_button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT(dialog));
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
		     okay_button);
  gtk_widget_show(okay_button);

  cancel_button = gtk_button_new_with_label(CANCEL_BUTTON_LABEL);
  gtk_signal_connect_object (GTK_OBJECT (cancel_button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT(dialog));
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
		     cancel_button);
  gtk_widget_show(cancel_button);

  gtk_widget_show_all (dialog);
}

/**
 * @brief     Create a dialog box for exporting a series of images to a
 *            directory.
 * 
 * @param     plot           The Plot to be exported.
 * @param     action         The calling action.
 * @param     export_button  The calling button.
 * @returns   TRUE if successful, otherwise crash.
 */
void
image_dir_export_dialog(Plot* plot, gint action, GtkItem* export_button)
{
  GtkWidget* dialog;
  GtkWidget* label;
  GtkWidget* hbox_A;
  GtkWidget* file_label_A;
  GtkWidget* file_button_A;
  GtkWidget* okay_button;
  GtkWidget* cancel_button;

  dialog = gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW(dialog), EXPORT_TO_DIR_TITLE);
  gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, FALSE, TRUE);

  label = gtk_label_new(EXPORT_DIR_SELECT_MESSAGE);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), label);

  hbox_A = gtk_hbox_new(FALSE, 0);
  gtk_widget_show(hbox_A);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox_A, FALSE,
		     FALSE, 0);

  file_label_A = gtk_label_new(EXPORT_DIR_LABEL);
  gtk_widget_show(file_label_A);
  gtk_box_pack_start(GTK_BOX(hbox_A), file_label_A, FALSE, FALSE, 0);

  file_entry_A = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(file_entry_A), FILE_STR_SIZE);
  gtk_entry_set_text(GTK_ENTRY(file_entry_A), "");

  gtk_widget_show(file_entry_A);
  gtk_box_pack_start(GTK_BOX(hbox_A), file_entry_A, FALSE, FALSE, 0);
  gtk_widget_set_usize(file_entry_A, FILE_ENTRY_LENGTH, -2);
  gtk_signal_connect(GTK_OBJECT(file_entry_A), "activate",
		     GTK_SIGNAL_FUNC(export_dir_set), plot);
  gtk_signal_connect_object(GTK_OBJECT(file_entry_A), "activate",
			    GTK_SIGNAL_FUNC (gtk_widget_destroy),
			    GTK_OBJECT(dialog));


  file_button_A = gtk_button_new_with_label(SELECT_LABEL);
  gtk_signal_connect(GTK_OBJECT(file_button_A), "clicked",
		     GTK_SIGNAL_FUNC(image_export_file_select), plot);
  gtk_widget_show(file_button_A);
  gtk_box_pack_start(GTK_BOX(hbox_A), file_button_A, FALSE, FALSE, 0);

  okay_button = gtk_button_new_with_label(OKAY_BUTTON_LABEL);
  gtk_signal_connect(GTK_OBJECT(okay_button),
		     "clicked", GTK_SIGNAL_FUNC(export_dir_set), plot);
  gtk_signal_connect_object (GTK_OBJECT (okay_button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT(dialog));
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
		     okay_button);
  gtk_widget_show(okay_button);

  cancel_button = gtk_button_new_with_label(CANCEL_BUTTON_LABEL);
  gtk_signal_connect_object (GTK_OBJECT (cancel_button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT(dialog));
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
		     cancel_button);
  gtk_widget_show(cancel_button);

  gtk_widget_show_all (dialog);
}

/*
 * ABOUT BOX CALLBACKS.
 */
/**
 * @brief     Expose the "About" dialog box.
 * 
 * @param     drawing_area  The About box drawing area.
 * @param     event         The calling event.
 * @param     data          Data attached to the event (unused).
 * @returns   TRUE if successful, otherwise crash.
 */
gboolean
about_expose(GtkWidget* drawing_area, GdkEventExpose* event, gpointer data)
{
  gdk_draw_pixmap(drawing_area->window,
		  drawing_area->style->fg_gc[GTK_WIDGET_STATE(drawing_area)],
		  global_about_pixmap,
		  event->area.x, event->area.y,
		  event->area.x, event->area.y,
		  event->area.width, event->area.height);

  return TRUE;
}
