/* dtp.h
 * Definitions for Cisco's Dynamic Trunking Protocol
 *
 * $Id: dtp.h,v 1.4 2005/07/29 20:37:21 t0mac Exp $ 
 *
 * Yersinia
 * By David Barroso <tomac@wasahero.org> and Alfredo Andres <slay@wasahero.org>
 * Copyright 2005 Alfredo Andres and David Barroso
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifndef __DTP_H__
#define __DTP_H__

#include <libnet.h>

#include "terminal-defs.h"
#include "interfaces.h"


#define DTP_TYPE_DOMAIN   0x01
#define DTP_TYPE_STATUS   0x02
#define DTP_TYPE_TYPE     0x03
#define DTP_TYPE_NEIGHBOR 0x04


/* Status TOS/TAS */
#define DTP_ACCESS    0x00
#define DTP_TRUNK     0x80
#define DTP_ON        0x01
#define DTP_OFF       0x02
#define DTP_DESIRABLE 0x03
#define DTP_AUTO      0x04
#define DTP_UNKNOWN   0x05

/* Type TOT/TAT */
#define DTP_TOT_NATIVE     0x20
#define DTP_TOT_ISL        0x40
#define DTP_TOT_802_1Q     0xa0
#define DTP_TAT_NEGOTIATED 0x00
#define DTP_TAT_NATIVE     0x01
#define DTP_TAT_ISL        0x02
#define DTP_TAT_802_1Q     0x05

/* Default values */
#define DTP_DFL_VERSION 0x01
#define DTP_DFL_DOMAIN  "\x0\x0\x0\x0\x0\x0\x0\x0"
#define DTP_DFL_DOM_LEN 0x08
#define DTP_DFL_STATUS  (DTP_ACCESS | DTP_DESIRABLE)
#define DTP_DFL_TYPE    (DTP_TOT_802_1Q | DTP_TAT_802_1Q)

#define DTP_DOMAIN_SIZE 32

static const struct tuple_type_desc dtp_status[] = {
    { DTP_ACCESS|DTP_DESIRABLE, "(ACCESS/DESIRABLE)" },
    { DTP_ACCESS|DTP_ON,        "(ACCESS/ON)"        },
    { DTP_ACCESS|DTP_OFF,       "(ACCESS/OFF)"       },
    { DTP_ACCESS|DTP_AUTO,      "(ACCESS/AUTO)"      },
    { DTP_TRUNK|DTP_DESIRABLE,  "(TRUNK/DESIRABLE)"  },
    { DTP_TRUNK|DTP_ON,         "(TRUNK/ON)"         },
    { DTP_TRUNK|DTP_OFF,        "(TRUNK/OFF)"        },
    { DTP_TRUNK|DTP_AUTO,       "(TRUNK/AUTO)"       },
    { DTP_UNKNOWN, "(UNKNOWN)" },
    { 0, NULL }
};

static const struct tuple_type_desc dtp_type[] = {
    { DTP_TOT_802_1Q|DTP_TAT_802_1Q,     "(802.1Q/802.1Q)"     },
    { DTP_TOT_802_1Q|DTP_TAT_ISL,        "(802.1Q/ISL)"        },    
    { DTP_TOT_802_1Q|DTP_TAT_NATIVE,     "(802.1Q/NATIVE)"     },
    { DTP_TOT_802_1Q|DTP_TAT_NEGOTIATED, "(802.1Q/NEGOTIATED)" },
    { DTP_TOT_ISL|DTP_TAT_ISL,           "(ISL/ISL)"           },
    { DTP_TOT_ISL|DTP_TAT_802_1Q,        "(ISL/802.1Q)"        },
    { DTP_TOT_ISL|DTP_TAT_NATIVE,        "(ISL/NATIVE)"        },
    { DTP_TOT_ISL|DTP_TAT_NEGOTIATED,    "(ISL/NEGOTIATED)"    },
    { DTP_TOT_NATIVE|DTP_TAT_802_1Q,     "(NATIVE/802.1Q)"     },
    { DTP_TOT_NATIVE|DTP_TAT_ISL,        "(NATIVE/ISL)"        },
    { DTP_TOT_NATIVE|DTP_TAT_NATIVE,     "(NATIVE/NATIVE)"     },
    { DTP_TOT_NATIVE|DTP_TAT_NEGOTIATED, "(NATIVE/NEGOTIATED)" },
    { 0, NULL }
};

static struct proto_features dtp_features[] = {
    { F_LLC_CISCO, 0x2004 },
    { -1, 0}
};

#define DTP_TOTAL_FIELDS 8

#define DTP_SMAC     0
#define DTP_DMAC     1
#define DTP_VERSION  2 
#define DTP_DOMAIN   3
#define DTP_STATUS   4
#define DTP_TYPE     5
#define DTP_NEIGH    6

static struct position_fields dtp_fields[] = {
   { DTP_SMAC,    "Source MAC",      1,  1, 12, 17, 0, 0,  FIELD_MAC,  FIELD_FIRST,  NULL },
   { DTP_DMAC,    "Destination MAC", 1, 31, 47, 17, 0, 0,  FIELD_MAC,  FIELD_NORMAL, NULL },
   { DTP_VERSION, "Version",         2,  1,  9, 2 , 0, 0,  FIELD_HEX,  FIELD_NORMAL, NULL },
   { DTP_DOMAIN,  "Domain",          2, 13, 20, DTP_DOMAIN_SIZE, 1, 37, FIELD_STR, FIELD_NORMAL, NULL },
   { DTP_STATUS,  "Status",          3,  1,  8, 2 , 1, 15, FIELD_HEX,  FIELD_NORMAL, dtp_status },
   { DTP_TYPE,    "Type",            3, 13, 18, 2 , 0, 0,  FIELD_HEX,  FIELD_NORMAL, dtp_type },
   { DTP_NEIGH,   "Neighbor-ID",     3, 23, 35, 12, 1, 2,  FIELD_HEX,  FIELD_LAST,   NULL },
   { 0,           NULL,              0,  0,  0,  0, 0, 0,  FIELD_NONE, FIELD_NORMAL, NULL }
};

/* DTP mode stuff */
struct dtp_data { 
     u_int8_t  mac_source[ETHER_ADDR_LEN];
     u_int8_t  mac_dest[ETHER_ADDR_LEN];
     u_int8_t  version;
     char      domain[DTP_DOMAIN_SIZE+1];
     u_int16_t dom_len;
     u_int8_t  status;
     u_int8_t  type;
     u_int8_t  neighbor[ETHER_ADDR_LEN];
     u_int8_t  state;
};

#define DTP_ATTACK_SEND     0
#define DTP_ATTACK_DO_TRUNK 1

void dtp_th_send(void *);
void dtp_th_send_exit(struct attacks *);
void dtp_th_nondos_do_trunk(void *);
void dtp_th_nondos_do_trunk_exit(struct attacks *);

static struct attack dtp_attack[] = {
    { DTP_ATTACK_SEND,     "sending DTP packet", NONDOS, SINGLE,    dtp_th_send, NULL, 0            },
    { DTP_ATTACK_DO_TRUNK, "enabling trunking",  NONDOS, CONTINOUS, dtp_th_nondos_do_trunk, NULL, 0 },
    { 0,                   NULL,                 0,      0,         NULL, NULL, 0                   }
};


void     dtp_register(void);
int8_t   dtp_send(struct attacks *);
int8_t   dtp_init_attribs(struct term_node *);
int8_t   dtp_learn_packet(int8_t, u_int8_t *, void *, struct pcap_pkthdr *);
char     **dtp_get_printable_packet(struct pcap_data *);
char     **dtp_get_printable_store(struct term_node *);
int8_t   dtp_load_values(struct pcap_data *, void *);
void     dtp_send_negotiate(void *);
int8_t   dtp_update_data(int8_t, int8_t, int8_t, struct term_node *);
char     *dtp_get_info(u_int8_t, char *);
int8_t   dtp_update_field(int8_t, struct term_node *, void *);
void     dtp_help(void);
int8_t   dtp_parser( struct term_node *, int8_t, char **, void *);


extern void   thread_libnet_error( char *, libnet_t *);
extern int8_t thread_create(pthread_t *, void *, void *);
extern void   write_log( u_int16_t mode, char *msg, ... );
extern int8_t attack_th_exit(struct attacks *);
extern void   attack_gen_mac(u_int8_t *);
extern int8_t interfaces_get_packet(u_int8_t *, struct pcap_pkthdr *, u_int8_t *, u_int16_t, time_t);
extern int8_t parser_vrfy_mac(char *, u_int8_t *);

extern int8_t parser_command2index(register const struct attack *, register int8_t);
extern struct interface_data interfaces[];
extern struct terminals *terms;

extern int8_t bin_data[];

#endif
/* vim:set tabstop=4:set expandtab:set shiftwidth=4:set textwidth=78: */
