/*  
  Copyright 2002, Andreas Rottmann

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/
extern "C" {
#include <libguile.h>
}

#include <dirent.h>
#include <sys/stat.h>

#include <yehia/plugin.h>

#include "guile-loader.h"
#include "guile-script.h"

using namespace SigC;
using namespace SigCX;

namespace Yehia
{

using namespace Script;

guilePluginLoader::guilePluginLoader(PluginManager& mgr)
    : PluginLoader(&mgr), Plugin(mgr)
{
}

guilePluginLoader::~guilePluginLoader()
{
}

struct LoadInfo
{
    const char *path; // path of file to load
    PluginManager *plugin_manager;
    bool exception; // exception during load
};

static SCM load_catcher(void *data, SCM tag, SCM args)
{
  LoadInfo *linfo = (LoadInfo *)data;
  SCM p 	= scm_open_output_string();

  linfo->exception = true;
  if (scm_ilength(args) >= 3)
  {
    SCM stack   = scm_fluid_ref(
            scm_variable_ref(scm_the_last_stack_fluid_var));
    SCM subr    = SCM_CAR(args);
    SCM message = SCM_CADR(args);
    SCM parts   = SCM_CADDR(args);
    SCM rest    = SCM_CDDDR(args);

    if (SCM_BACKTRACE_P && SCM_NFALSEP(stack))
    {
      scm_puts("Backtrace:\n", p);
      scm_display_backtrace(stack, p, SCM_UNDEFINED, SCM_UNDEFINED);
      scm_newline(p);
    }
    scm_i_display_error(stack, p, subr, message, parts, rest);
  }
  else
    scm_simple_format(p, scm_makfrom0str("~A: ~A"), scm_list_2(tag, args));
  
  SCM msg = scm_get_output_string(p);
  linfo->plugin_manager->set_error(std::string("Guile exception:\n") + 
                                   SCM_STRING_CHARS(msg));

  return SCM_UNSPECIFIED;
}

typedef guileObjectFactory::SmallInfo SmallInfo;

static void swap_port(void *data)
{
  SCM *save_port = (SCM *)data, tmp = scm_cur_loadp;
  scm_cur_loadp = *save_port;
  *save_port = tmp;
}

static SCM load(void *data)
{
  SCM port = SCM_PACK(data);
  SCM result = SCM_UNSPECIFIED;
  
  while (1)
  {
    SCM form = scm_read (port);
    if (SCM_EOF_OBJECT_P(form))
      break;
    result = scm_primitive_eval_x(form);
  }
  return result;
}

static SCM my_primitive_load(void *data)
{
  LoadInfo *linfo = (LoadInfo *)data;
  PluginManager *pm = linfo->plugin_manager;
  
  // this is a adapted version of primitive-load that returns the
  // result of the last form evaluated
  SCM fname = scm_makfrom0str(linfo->path);
  SCM port = scm_open_file(fname, scm_mem2string("r", sizeof(char)));
  SCM save_port = port;
  SCM result = scm_internal_dynamic_wind(swap_port,
                                         load,
                                         swap_port,
                                         (void *)SCM_UNPACK (port),
                                         &save_port);
  scm_close_port(port);

  // TODO: do some more error checking
  
  ObjectFactory& factory = 
    LanguageManager::instance().language("guile")->factory();
  Object *plugin_class = factory.find_class(typeid(Plugin));
  SCM splugin_class = dynamic_cast<guileObject *>(plugin_class)->sobj();
  if (SCM_CLASSP(result) && SCM_SUBCLASSP(result, splugin_class))
  {
    Object *pm_class = factory.find_class(typeid(PluginManager));
    SCM spm = 
      dynamic_cast<guileObject&>(
              factory.wrap_instance(*pm_class, *pm)).sobj();
    SCM make = scm_variable_ref(
            scm_c_module_lookup(scm_module_goops, "make"));
    return scm_apply(make, scm_list_2(result, spm), SCM_EOL);
  }
  else
    pm->set_error(std::string("file evaluation of '") + linfo->path + 
                  "' did not return class derived from Plugin");
  
  return result;
}

static Plugin *do_load(PluginManager *pm, const std::string& path)
{
  LoadInfo linfo = { path.c_str(), pm, false };

  SCM splugin = scm_internal_stack_catch(SCM_BOOL_T, 
                                         &my_primitive_load, &linfo,
                                         &load_catcher, &linfo);
  
  if (linfo.exception)
    return 0;
  
  SCM sinst = scm_slot_ref(splugin, scm_str2symbol("%ucxx-instance"));
    
  if (SCM_NFALSEP(sinst))
  {
    Plugin *plugin = dynamic_cast<Plugin *>(
            (SigC::Object *)GINT_TO_POINTER(SCM_SMOB_DATA(sinst)));
    if (plugin)
      plugin->reference();
    
    return plugin;
  }  
  return 0;
}

Plugin *guilePluginLoader::load(PluginManager& mgr, const std::string& name)
{
  // remove "guile." prefix
  if (name.substr(0, 6) != "guile.")
    return 0;
  std::string realname = std::string("guile/") + name.substr(6) + ".scm";

  // get language object
  Script::Language *lang = LanguageManager::instance().language("guile");
  
  if (!lang)
    return 0;

  std::list<std::string>::const_iterator it;
  const std::list<std::string>& indep_paths = mgr.arch_indep_paths();
  
  for (it = indep_paths.begin(); it != indep_paths.end(); ++it)
  {
    std::string path = (*it) + "/" + realname;
    
    struct stat st;
    if (stat(path.c_str(), &st) == 0)
    {
      return tunnel<Plugin *, PluginManager *, const std::string&>(
              slot(&do_load), &mgr, path, lang->tunnel(), true);
    }
    else
      mgr.set_error("cannot stat '" + path + "'");
  }
  
  return 0;
}

namespace
{

void do_scan(PluginManager& mgr, const std::string& path, 
             const std::string& basename)
{
  DIR *dir;
  struct dirent *file;

  if ((dir = opendir(path.c_str())) != NULL)
  {
    while ((file = readdir(dir)) != NULL)
    {
      int len;
      std::string fullname = path + file->d_name;
      struct stat statinfo;
      
      if (file->d_name[0] != '.' &&
          stat(fullname.c_str(), &statinfo) == 0 && S_ISDIR(statinfo.st_mode))
        do_scan(mgr,
                path + file->d_name + "/", basename + file->d_name + ".");
      
      len = strlen(file->d_name);
      if (len > 3 && strcmp(file->d_name + len - 3, ".scm") == 0)
        mgr.plugin_found(basename + 
                          std::string(file->d_name).substr(0, len - 4));
    }
  }
}

}

void guilePluginLoader::scan(PluginManager& mgr) const
{
  const std::list<std::string>& paths = mgr.arch_indep_paths();
  std::list<std::string>::const_iterator it;

  for (it = paths.begin(); it != paths.end(); ++it)
    do_scan(mgr, (*it) + "/guile/", "guile.");
}

}
