/****************************************************************************
 *
 * 			texture.cc: Texture and modulation implementation
 *      This is part of the yafray package
 *      Copyright (C) 2002 Alejandro Conty Estevez
 *
 *      This library is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU Lesser General Public
 *      License as published by the Free Software Foundation; either
 *      version 2.1 of the License, or (at your option) any later version.
 *
 *      This library is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *      Lesser General Public License for more details.
 *
 *      You should have received a copy of the GNU Lesser General Public
 *      License along with this library; if not, write to the Free Software
 *      Foundation,Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include "basictex.h"
#include "object3d.h"
#include <iostream>

using namespace std;

__BEGIN_YAFRAY

//-----------------------------------------------------------------------------------------
// Clouds Texture
//-----------------------------------------------------------------------------------------

CFLOAT textureClouds_t::getFloat(const point3d_t &p) const
{
	return noiseGen.turbulence(p, depth);
}

colorA_t textureClouds_t::getColor(const point3d_t &p) const
{
	return color1 + getFloat(p)*(color2 - color1);
}


texture_t *textureClouds_t::factory(paramMap_t &params,
		renderEnvironment_t &render)
{
	color_t color1(0.0),color2(1.0);
	int depth = 2;
	params.getParam("color1", color1);
	params.getParam("color2", color2);
	params.getParam("depth", depth);
	return new textureClouds_t(depth, color1, color2);
}

//-----------------------------------------------------------------------------------------
// Simple Marble Texture
//-----------------------------------------------------------------------------------------

CFLOAT textureMarble_t::getFloat(const point3d_t &p) const
{
  return pow((float)(0.5+0.5*sin(p.y + turbulence*noiseGen.turbulence(p, octaves, hard))), sharpness);
}


colorA_t textureMarble_t::getColor(const point3d_t &p) const
{
	return color1 + getFloat(p)*(color2 - color1);
}


texture_t *textureMarble_t::factory(paramMap_t &params,
		renderEnvironment_t &render)
{
	color_t col1(0.0), col2(1.0);
	int oct = 2;
	CFLOAT turb=1.0, shp=1.0;
	bool hrd = false;
	params.getParam("color1", col1);
	params.getParam("color2", col2);
	params.getParam("depth", oct);
	params.getParam("turbulence", turb);
	params.getParam("sharpness", shp);
	params.getParam("hard", hrd);
	return new textureMarble_t(oct, col1, col2, turb, shp, hrd);
}

//-----------------------------------------------------------------------------------------
// Simple Wood Texture
//-----------------------------------------------------------------------------------------

CFLOAT textureWood_t::getFloat(const point3d_t &p) const
{
  return fabs(sin(sqrt(ringscale_X*p.x*p.x + ringscale_Y*p.y*p.y) + turbulence*noiseGen.turbulence(p, octaves, hard)));
}


colorA_t textureWood_t::getColor(const point3d_t &p) const
{
	return color1 + getFloat(p)*(color2 - color1);
}


texture_t *textureWood_t::factory(paramMap_t &params,
		renderEnvironment_t &render)
{
	color_t col1(0.0), col2(1.0);
	int oct = 2;
	CFLOAT turb = 1.0;
	PFLOAT rx=1, ry=1;
	bool hrd = false;
	params.getParam("color1", col1);
	params.getParam("color2", col2);
	params.getParam("depth", oct);
	params.getParam("turbulence", turb);
	params.getParam("ringscale_x", rx);
	params.getParam("ringscale_y", ry);
	params.getParam("hard", hrd);
	return new textureWood_t(oct, col1, col2, turb, rx, ry, hrd);
}

//-----------------------------------------------------------------------------------------
// Image Texture
//-----------------------------------------------------------------------------------------


extern gBuf_t<unsigned char, 4> * load_jpeg(const char *name);

textureImage_t::textureImage_t(const char *filename)
{
  // try to determine from extensions first
  char *ext = strrchr(filename, '.');
  bool jpg_tried = false;
  bool tga_tried = false;

  image = NULL;
  tga_img = NULL;

  cout << "Loading image file " << filename << endl;

  // try loading image using extension as indication of imagetype
  if (ext) {
    //jpeg
    if(((!strcmp(ext, ".jpg")) || (!strcmp(ext, ".jpeg"))) ||
       ((!strcmp(ext, ".JPG")) || (!strcmp(ext, ".JPEG"))))
#ifdef HAVE_JPEG
    {
      image = load_jpeg(filename);
      jpg_tried = true;
    }
#else
    cout << "Warning, yafray was compiled without jpeg support, cannot load image.\n";
#endif

    // targa, apparently, according to ps description, on mac tga extension can be .tpic
    if(((!strcmp(ext, ".tga")) || (!strcmp(ext, ".tpic"))) ||
       ((!strcmp(ext, ".TGA")) || (!strcmp(ext, ".TPIC")))) 
	{
      //have to change this to direct cBuffer load, not this detour
      tga_img = new targaImg_t();
      if(! tga_img->Load(filename))
			{
				cerr<<tga_img->getErrorString();
				delete tga_img;
				tga_img=NULL;
			}
			else
	      image = tga_img->imageData();
      tga_tried = true;
    }
  }

  // if none was able to load (or no extension), try every type until one or none succeeds
  if (!image) {

    // jpeg first (if supported), targa last (targa has no ID)

#ifdef HAVE_JPEG
    if (!jpg_tried) image = load_jpeg(filename);
#endif

    if ((!image) && (!tga_tried)) {
      tga_img = new targaImg_t();
      if(! tga_img->Load(filename))
			{
				cerr<<tga_img->getErrorString();
				delete tga_img;
				tga_img=NULL;
			}
			else
      	image = tga_img->imageData();
    }

    // later more types heres

  }

  if (image) {
    cout << "OK\n";
    failed = false;
  }
  else {
    cout << "Could not load image\n";
    failed = true;
  }

}


textureImage_t::~textureImage_t()
{
	if (tga_img) {
		// image already deleted above, but is handled in tga_img
		// nevertheless, better change this
		delete tga_img;
		tga_img = NULL;
	}
	else
	if (image) {
		delete image;
		image = NULL;
	}
}


colorA_t textureImage_t::getColor(const point3d_t &p) const
{
	if (!image) return color_t(0.0);

	// p->x/y == u, v
	CFLOAT xinf = fmod(p.x, 1.0f);
	CFLOAT yinf = fmod(p.y, 1.0f);
	// negative uv-coord correction
	if (xinf<0) xinf += 1.0;
	if (yinf<0) yinf += 1.0;

	CFLOAT xf = (CFLOAT)image->resx() * xinf;
	CFLOAT yf = (CFLOAT)image->resy() * yinf;
	int x = (int)xf;
	int y = (int)yf;

	if (x<0) x = 0;
	if (y<0) y = 0;
	if (x>=image->resx()) x = image->resx();
	if (y>=image->resy()) y = image->resy();

	int x2, y2;
	if ((x2 = x+1) >= image->resx()) x2 = image->resx()-1;
	if ((y2 = y+1) >= image->resy()) y2 = image->resy()-1;

	colorA_t c1, c2, c3, c4;
	(*image)(x ,y ) >> c1;
	(*image)(x2,y ) >> c2;
	(*image)(x ,y2) >> c3;
	(*image)(x2,y2) >> c4;

	return BilerpWeight(xf, yf, c1, c2, c3, c4);
}


CFLOAT textureImage_t::getFloat(const point3d_t &p) const
{
	return getColor(p).energy();
}


texture_t *textureImage_t::factory(paramMap_t &params,
		renderEnvironment_t &render)
{
	string _name;
	const string *name=&_name;
	params.getParam("filename", name);
	if (*name=="")
		cerr << "Required argument filename not found for image texture\n";
	else
		return new textureImage_t(name->c_str());
	return NULL;
}

__END_YAFRAY
//-----------------------------------------------------------------------------------------
