#ifndef __BASICBLOCKS_H
#define __BASICBLOCKS_H

#include "metashader.h"
#include "basictex.h"
#include "params.h"

#ifdef HAVE_CONFIG_H
#include<config.h>
#endif

__BEGIN_YAFRAY

class floatToColor_t : public shaderNode_t
{
	public:
		floatToColor_t(const shader_t *in):input(in) {};
		virtual colorA_t stdoutColor(renderState_t &state,const surfacePoint_t &sp,const vector3d_t &eye,
				const scene_t *scene=NULL)const;
		static shader_t * factory(paramMap_t &,std::list<paramMap_t> &,
				        renderEnvironment_t &);
	protected:
		const shader_t *input;
};

class colorToFloat_t : public shaderNode_t
{
	public:
		colorToFloat_t(const shader_t *in):input(in) {};
		virtual CFLOAT stdoutFloat(renderState_t &state,const surfacePoint_t &sp,const vector3d_t &eye,
				const scene_t *scene=NULL)const
		{return input->stdoutColor(state,sp,eye,scene).energy();};
		static shader_t * factory(paramMap_t &,std::list<paramMap_t> &,
				        renderEnvironment_t &);
	protected:
		const shader_t *input;
};


class cloudsNode_t : public shaderNode_t
{
	public:
		cloudsNode_t(PFLOAT s, int dep,
			const shader_t *in1, const shader_t *in2);
		virtual CFLOAT stdoutFloat(renderState_t &state,const surfacePoint_t &sp, const vector3d_t &eye,
				const scene_t *scene=NULL) const;
		virtual colorA_t stdoutColor(renderState_t &state,const surfacePoint_t &sp, const vector3d_t &eye,
				const scene_t *scene=NULL) const;
		virtual bool isRGB() const { return (input1!=NULL) && (input2!=NULL); }
		virtual ~cloudsNode_t() {};
		static shader_t * factory(paramMap_t &,std::list<paramMap_t> &,
				        renderEnvironment_t &);
	protected:
		textureClouds_t tex;
		PFLOAT size;
		const shader_t *input1, *input2;
};


class marbleNode_t : public shaderNode_t
{
	public:
		marbleNode_t(PFLOAT sz, int dep, CFLOAT turb, CFLOAT shp, bool hrd,
				const shader_t *in1, const shader_t *in2);
		virtual CFLOAT stdoutFloat(renderState_t &state,const surfacePoint_t &sp, const vector3d_t &eye,
				const scene_t *scene=NULL) const;
		virtual colorA_t stdoutColor(renderState_t &state,const surfacePoint_t &sp, const vector3d_t &eye,
				const scene_t *scene=NULL) const;
		virtual bool isRGB() const { return (input1!=NULL) && (input2!=NULL); }
		virtual ~marbleNode_t() {};
		static shader_t * factory(paramMap_t &,std::list<paramMap_t> &,
				        renderEnvironment_t &);
	protected:
		textureMarble_t tex;
		PFLOAT size;
		const shader_t *input1, *input2;
};


class woodNode_t : public shaderNode_t
{
	public:
		woodNode_t(PFLOAT sz, int dep, CFLOAT turb, PFLOAT rx, PFLOAT ry, bool hrd,
				const shader_t *in1, const shader_t *in2);
		virtual CFLOAT stdoutFloat(renderState_t &state,const surfacePoint_t &sp, const vector3d_t &eye,
				const scene_t *scene=NULL) const;
		virtual colorA_t stdoutColor(renderState_t &state,const surfacePoint_t &sp, const vector3d_t &eye,
				const scene_t *scene=NULL) const;
		virtual bool isRGB() const { return (input1!=NULL) && (input2!=NULL); }
		virtual ~woodNode_t() {};
		static shader_t * factory(paramMap_t &,std::list<paramMap_t> &,
				        renderEnvironment_t &);
	protected:
		textureWood_t tex;
		PFLOAT size;
		const shader_t *input1, *input2;
};


class colorBandNode_t : public shaderNode_t
{
	public:
		colorBandNode_t(std::vector<std::pair<CFLOAT,colorA_t> > &b,const shader_t *in):
			band(b),input(in) {};
		virtual ~colorBandNode_t() {};
		virtual CFLOAT stdoutFloat(renderState_t &state,const surfacePoint_t &sp, const vector3d_t &eye,
				const scene_t *scene=NULL) const;
		virtual colorA_t stdoutColor(renderState_t &state,const surfacePoint_t &sp,const vector3d_t &eye,
				const scene_t *scene=NULL)const;
		static shader_t * factory(paramMap_t &,std::list<paramMap_t> &,
				        renderEnvironment_t &);

	protected:
		std::vector<std::pair<CFLOAT,colorA_t> > band;
		const shader_t *input;
};


class coordsNode_t : public shaderNode_t
{
	public:
		coordsNode_t(int c):coord(c) {};
		virtual CFLOAT stdoutFloat(renderState_t &state,const surfacePoint_t &sp,const vector3d_t &eye,
				const scene_t *scene=NULL)const
		{
			switch(coord)
			{
				case 0: return sp.P().x;
				case 1: return sp.P().y;
				default: return sp.P().z;
			}
		};
		static shader_t * factory(paramMap_t &,std::list<paramMap_t> &,
				        renderEnvironment_t &);

	protected:
		int coord;
};

class mulNode_t : public shaderNode_t
{
	public:
		mulNode_t(const shader_t *in1,const shader_t *in2,CFLOAT v):
			input1(in1),input2(in2),value(v) {};
		virtual CFLOAT stdoutFloat(renderState_t &state,const surfacePoint_t &sp,const vector3d_t &eye,
				const scene_t *scene=NULL)const
		{
			CFLOAT res=value;
			if(input1!=NULL) res*=input1->stdoutFloat(state,sp,eye,scene);
			if(input2!=NULL) res*=input2->stdoutFloat(state,sp,eye,scene);
			return res;
		};
		static shader_t * factory(paramMap_t &,std::list<paramMap_t> &,
				        renderEnvironment_t &);
	protected:
		const shader_t *input1,*input2;
		CFLOAT value;
};

class sinNode_t : public shaderNode_t
{
	public:
		sinNode_t(const shader_t *in):
			input(in) {};
		virtual CFLOAT stdoutFloat(renderState_t &state,const surfacePoint_t &sp,const vector3d_t &eye,
				const scene_t *scene=NULL)const
		{
			return 0.5*sin(input->stdoutFloat(state,sp,eye,scene))+0.5;
		};
		static shader_t * factory(paramMap_t &,std::list<paramMap_t> &,
				        renderEnvironment_t &);
	protected:
		const shader_t *input;
};

class phongNode_t : public shader_t
{
	public:
		phongNode_t(const shader_t *c,const shader_t *s,const shader_t *e,
								const shader_t *cr,const shader_t *ct,CFLOAT h,PFLOAT ior):
			color(c),specular(s),env(e),caus_rcolor(cr),caus_tcolor(ct),hard(h),
			IOR(ior) {};
		/// @see shader_t
		virtual color_t fromRadiosity(renderState_t &state,const surfacePoint_t &sp,const energy_t &ene,
															const vector3d_t &eye)const;
		/// @see shader_t
		virtual color_t fromLight(renderState_t &state,const surfacePoint_t &sp,const energy_t &energy,
															const vector3d_t &eye)const;
		/// @see shader_t
		virtual color_t fromWorld(renderState_t &state,const surfacePoint_t &sp,const scene_t &scene,
															const vector3d_t &eye)const;
		/// @see shader_t
		virtual const color_t getDiffuse(renderState_t &state,
							const surfacePoint_t &sp, const vector3d_t &eye)const;

		virtual void getCaustics(renderState_t &state,const surfacePoint_t &sp,const vector3d_t &eye,
														color_t &ref,color_t &trans,
														PFLOAT &ior)const;

		/// Destructor
		virtual ~phongNode_t() {};

		static shader_t * factory(paramMap_t &,std::list<paramMap_t> &,
				        renderEnvironment_t &);
	protected:
		const shader_t *color,*specular,*env,*caus_rcolor,*caus_tcolor;
		CFLOAT hard;
		PFLOAT IOR;
};

class rgbNode_t : public shaderNode_t
{
	public:
		rgbNode_t(const shader_t *in1, const shader_t *in2, const shader_t *in3, 
				const color_t &c);

		virtual colorA_t stdoutColor(renderState_t &state,const surfacePoint_t &sp,const vector3d_t &eye,
				const scene_t *scene)const;
		virtual ~rgbNode_t() {};
		static shader_t * factory(paramMap_t &,std::list<paramMap_t> &,
				        renderEnvironment_t &);

	protected:
		const shader_t *inputred;
		const shader_t *inputgreen;
		const shader_t *inputblue;
		color_t color;
};

class hsvNode_t : public shaderNode_t
{
	public:
		hsvNode_t(const shader_t *in1, const shader_t *in2, const shader_t *in3, 
				CFLOAT h, CFLOAT s, CFLOAT v);

		virtual colorA_t stdoutColor(renderState_t &state,const surfacePoint_t &sp,const vector3d_t &eye,
				const scene_t *scene)const;
		static shader_t * factory(paramMap_t &,std::list<paramMap_t> &,
				        renderEnvironment_t &);

	protected:
		const shader_t *inputhue;
		const shader_t *inputsaturation;
		const shader_t *inputvalue;

		CFLOAT hue, saturation, value;
};

class coneTraceNode_t : public shaderNode_t
{
	public:
		coneTraceNode_t(const color_t &c,PFLOAT angle,int s,PFLOAT ior,bool r);

		virtual colorA_t stdoutColor(renderState_t &state,const surfacePoint_t &sp,const vector3d_t &eye,
				const scene_t *scene)const;

		virtual ~coneTraceNode_t() {};
		static shader_t * factory(paramMap_t &,std::list<paramMap_t> &,
				        renderEnvironment_t &);
	protected:
		bool ref;
		color_t color;
		PFLOAT cosa,IOR,sqrdiv;
		CFLOAT div;
		int samples,sqr;
};

class fresnelNode_t : public shaderNode_t
{
	public:
		fresnelNode_t(const shader_t *r,const shader_t *t,PFLOAT ior,CFLOAT minr):trans(t),
			ref(r),minref(minr)
			{
				IOR = (ior - 1.0) / (ior + 1.0);
				IOR *= IOR;
			};

		virtual colorA_t stdoutColor(renderState_t &state,const surfacePoint_t &sp,const vector3d_t &eye,
				const scene_t *scene)const;
		static shader_t * factory(paramMap_t &,std::list<paramMap_t> &,
				        renderEnvironment_t &);

	protected:
			const shader_t *trans,*ref;
			PFLOAT IOR;
			CFLOAT minref;
};

class imageNode_t : public shaderNode_t
{
	public:
		imageNode_t(const char *filename):tex(filename) {};
		virtual CFLOAT stdoutFloat(renderState_t &state,const surfacePoint_t &sp, const vector3d_t &eye,
				const scene_t *scene=NULL) const
		{ return tex.getFloat(sp.P()); };
		virtual colorA_t stdoutColor(renderState_t &state,const surfacePoint_t &sp, const vector3d_t &eye,
				const scene_t *scene=NULL) const
		{ return tex.getColor(sp.P()); };
		virtual bool discrete()const {return true;};
		virtual ~imageNode_t() {};
		static shader_t * factory(paramMap_t &,std::list<paramMap_t> &,
				        renderEnvironment_t &);
	protected:
		textureImage_t tex;
};

class goboNode_t : public shaderNode_t
{
	public:
		goboNode_t(const shader_t *in1, const shader_t *in2, const shader_t *inGoboColor, 
			const shader_t *inGoboFloat,const bool inHardEdge,const CFLOAT inEdgeVal ):
			input1(in1),input2(in2),goboColor(inGoboColor),goboFloat(inGoboFloat),
			hardEdge(inHardEdge),edgeVal(inEdgeVal){};
		virtual ~goboNode_t() {};

		virtual colorA_t stdoutColor(renderState_t &state,const surfacePoint_t &sp,const vector3d_t &eye,
				const scene_t *scene)const;

		static shader_t * factory(paramMap_t &,std::list<paramMap_t> &,
				        renderEnvironment_t &);

	protected:
		const shader_t *input1;
		const shader_t *input2;
		const shader_t *goboColor;
		const shader_t *goboFloat;
		bool hardEdge;
		CFLOAT edgeVal;
};

__END_YAFRAY
#endif
