/*******************************************************************
 * Routines for displaying/logging debug information.
 *
 * Licensed under the dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: xsup_debug.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: xsup_debug.c,v 1.19 2005/08/09 01:39:14 chessing Exp $
 * $Date: 2005/08/09 01:39:14 $
 * $Log: xsup_debug.c,v $
 * Revision 1.19  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <stdio.h>
#include <stdarg.h>
#include <strings.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <netinet/in.h>
#include <syslog.h>
#include "xsup_debug.h"
#include "config.h"

unsigned char debug_level = 0;     // By default just show the "normal" stuff.
int isdaemon = 0;
int syslogging = 0;
FILE *logfile = NULL;

/***************************************************
 *
 * Convert a string to be all lowercase.
 *
 ***************************************************/
void lowercase(char *instr)
{
  int i;

  for (i=0;i<strlen(instr);i++)
    {
      instr[i] = tolower(instr[i]);
    }
}

/***************************************************
 *
 * Return the logging facility the user asked us to use.
 *
 ***************************************************/
int xsup_debug_get_facility(char *facility_str)
{
  int facility_num = LOG_DAEMON;

  if (strcmp("cron", facility_str) == 0) facility_num = LOG_CRON;
  if (strcmp("daemon", facility_str) == 0) facility_num = LOG_DAEMON;
  if (strcmp("ftp", facility_str) == 0) facility_num = LOG_FTP;
  if (strcmp("kern", facility_str) == 0) facility_num = LOG_KERN;
  if (strcmp("local0", facility_str) == 0) facility_num = LOG_LOCAL0;
  if (strcmp("local1", facility_str) == 0) facility_num = LOG_LOCAL1;
  if (strcmp("local2", facility_str) == 0) facility_num = LOG_LOCAL2;
  if (strcmp("local3", facility_str) == 0) facility_num = LOG_LOCAL3;
  if (strcmp("local4", facility_str) == 0) facility_num = LOG_LOCAL4;
  if (strcmp("local5", facility_str) == 0) facility_num = LOG_LOCAL5;
  if (strcmp("local6", facility_str) == 0) facility_num = LOG_LOCAL6;
  if (strcmp("local7", facility_str) == 0) facility_num = LOG_LOCAL7;
  if (strcmp("lpr", facility_str) == 0) facility_num = LOG_LPR;
  if (strcmp("mail", facility_str) == 0) facility_num = LOG_MAIL;
  if (strcmp("news", facility_str) == 0) facility_num = LOG_NEWS;
  if (strcmp("user", facility_str) == 0) facility_num = LOG_USER;
  if (strcmp("uucp", facility_str) == 0) facility_num = LOG_UUCP;

  return facility_num;
}

/***************************************************
 *
 * Remove an old logfile, and create a new one.
 *
 ***************************************************/
void logfile_setup(char *logfilename)
{
  char *tempstr;
  int facility;

  if (logfilename != NULL)
    {
      if (isdaemon != 2)
	{
	  tempstr = (char *)malloc(strlen(logfilename)+1);
	  if (tempstr == NULL)
	    {
	      printf("Couldn't allocate memory for temporary string! (%s:%d)\n",
		     __FUNCTION__, __LINE__);
	      return;
	    }

	  strcpy(tempstr, logfilename);
	  lowercase(tempstr);
	  if (strcmp("syslog", tempstr) == 0)
	    {
	      free(tempstr);

	      tempstr = config_get_log_facility();
	      lowercase(tempstr);

	      facility = xsup_debug_get_facility(tempstr);

	      openlog("Xsupplicant", LOG_NDELAY, facility);
	      syslogging = 1;
	    } else {
	      unlink(logfilename);

	      logfile = fopen(logfilename, "w+");
	    }
	}
    }
}

/**************************************************
 *
 * Clean up our old logfile.
 *
 **************************************************/
void logfile_cleanup()
{
  if (logfile != NULL)
    {
      fclose(logfile);
    }
  if (syslogging == 1)
    {
      closelog();
    }
}

/*************************************************
 *
 * Set flags based on the numeric value that was passed in.
 *
 *************************************************/
void debug_set_flags(int new_flags)
{
  if (new_flags >= 1) debug_level |= DEBUG_CONFIG;
  if (new_flags >= 2) debug_level |= DEBUG_STATE;
  if (new_flags >= 3) debug_level |= DEBUG_AUTHTYPES;
  if (new_flags >= 4) debug_level |= DEBUG_INT;
  if (new_flags >= 5) debug_level |= DEBUG_SNMP;
  if (new_flags >= 6) debug_level |= DEBUG_EVERYTHING;
  if (new_flags >= 7) debug_level |= DEBUG_EXCESSIVE;
}

/*************************************************
 *
 * Set flags based on an ASCII string that was passed in.
 *
 *************************************************/
void debug_alpha_set_flags(char *new_flags)
{
  int i;

  debug_level = 0;

  for (i=0;i<strlen(new_flags);i++)
    {
      switch (new_flags[i])
	{
	case 'c':
	  debug_level |= DEBUG_CONFIG;
	  break;

	case 's':
	  debug_level |= DEBUG_STATE;
	  break;

	case 'a':
	  debug_level |= DEBUG_AUTHTYPES;
	  break;

	case 'i':
	  debug_level |= DEBUG_INT;
	  break;

	case 'n':
	  debug_level |= DEBUG_SNMP;
	  break;

	case 'e':
	  debug_level |= DEBUG_EVERYTHING;
	  break;

	case 'x':
	  debug_level |= DEBUG_EXCESSIVE;
	  break;

	case 'A':
	  debug_level |= 0xff;   // Set all flags.
	  break;
	}
    }
}

/*************************************************
 *
 * Depending on the value of fh, we will either print to the screen, or
 * a log file.
 *
 *************************************************/
void ufprintf(FILE *fh, char *instr)
{
  if (((isdaemon == 2) || (fh == NULL)) && (syslogging != 1))
    {
      printf("%s", instr);
    } else if (syslogging ==1) {
      // XXX Consider ways of using other log levels.
      syslog(LOG_ALERT, "%s", instr);
    } else {
      fprintf(fh, "%s", instr);
      fflush(fh);
    }
}

/*************************************************
 *
 * Set the debug level.  This is a global value, and shouldn't be set per
 * interface.
 *
 *************************************************/
void debug_setdaemon(int xdaemon)
{
  isdaemon = xdaemon;

  if (xdaemon == TRUE)
    {
      close(0);
      close(1);
      close(2); 
    }
}

/*************************************************
 *
 * Get the debug level for debug situations where we can't use debug_printf
 * easily.
 *
 *************************************************/
int debug_getlevel()
{
  return debug_level;
}

/*************************************************
 *
 * Dump hex values, without the ascii versions.
 *
 *************************************************/
void debug_hex_printf(unsigned char level, u_char *hextodump, int size)
{
  int i;
  char chrstr[1024];

  if ((!(debug_level & level)) && (level != 0)) return;

  if (hextodump == NULL) return;

  for (i=0;i<size;i++)
    {
      bzero((char *)&chrstr, 1024);
      sprintf((char *)&chrstr, "%02X ", hextodump[i]);
      ufprintf(logfile, (char *)&chrstr);
    }
  ufprintf(logfile, "\n");
}

/*************************************************
 *
 * dump some hex values -- also
 * show the ascii version of the dump.
 *
 *************************************************/
void debug_hex_dump(unsigned char level, u_char *hextodump, int size)
{
  int i,x,s,t;
  char chrstr[1024];

  if ((!(debug_level & level)) && (level != 0)) return;

  if (hextodump == NULL) return;

  s=0;
  for (i=0;i<size;i++)
    {
      bzero((char *)&chrstr, 1024);
      sprintf((char *)&chrstr, "%02X ",hextodump[i]);
      ufprintf(logfile, (char *)&chrstr);

      if ((i>0) && (((i+1) % 8) == 0) && (((i+1) % 16) != 0)) 
	{
	  ufprintf(logfile, "- ");
	}

      if ((i>0) && (((i+1) % 16) == 0))
	{
	  if (i<17) 
	    {
	      t=i+1;
	    } else {
	      t=i+1;
	      s=s+1;
	    }

	  for (x=s;x<t;x++)
	    {
	      if ((hextodump[x] < 0x21) || (hextodump[x] > 0x7e))
		{
		  ufprintf(logfile, ".");
		} else {
		  bzero((char *)&chrstr, 1024);
		  sprintf((char *)&chrstr, "%c", hextodump[x]);
		  ufprintf(logfile, (char *)&chrstr);
		}
	    }

	  s = i;
	  ufprintf(logfile, "\n");
	}
    }

  if ((size % 16) > 0)
    {
      i = (16 - (size % 16));

      if (i>8) ufprintf(logfile, "  ");

      for (x=0;x<i;x++)
	{
	  ufprintf(logfile, "   ");
	}
      
      for (x=(s+1);x<size;x++)
	{
	  if ((hextodump[x] < 0x21) || (hextodump[x] > 0x7e))
	    {
	      ufprintf(logfile, ".");
	    } else {
	      bzero((char *)&chrstr, 1024);
	      sprintf((char *)&chrstr, "%c", hextodump[x]);
	      ufprintf(logfile, (char *)&chrstr);
	    }
	}
    }
  ufprintf(logfile, "\n");
}

/*************************************************
 *
 * Display some information.  But only if we are at a debug level that
 * should display it.
 *
 *************************************************/
void debug_printf(unsigned char level, char *fmt, ...)
{
  char dumpstr[2048], temp[2048];

  if (((level & debug_level) || (level == 0)) && (fmt != NULL))
    {
      va_list ap;
      va_start(ap, fmt);

      bzero((char *)&dumpstr, 2048);
      bzero((char *)&temp, 2048);

      // Print out a tag that identifies the type of debug message being used.
      switch (level)
	{
	case DEBUG_NORMAL:
	  break;   
	  
	case DEBUG_CONFIG:
	  strcpy((char *)&dumpstr, "[CONFIG] ");
	  break;

	case DEBUG_STATE:
	  strcpy((char *)&dumpstr, "[STATE] ");
	  break;

	case DEBUG_AUTHTYPES:
	  strcpy((char *)&dumpstr, "[AUTH TYPE] ");
	  break;
	  
	case DEBUG_INT:
	  strcpy((char *)&dumpstr, "[INT] ");
	  break;

	case DEBUG_EVERYTHING:
	  strcpy((char *)&dumpstr, "[ALL] ");
	  break;
	}

      vsnprintf((char *)&temp, 2048, fmt, ap);
      
      strcat((char *)&dumpstr, (char *)&temp);

      ufprintf(logfile, dumpstr);

      va_end(ap);
    }
}

/*************************************************
 *
 * Display some information.  But only if we are at a debug level that
 * should display it.
 *
 *************************************************/
void debug_printf_nl(unsigned char level, char *fmt, ...)
{
  char temp[2048];

  if ((((level & debug_level) != 0x00) || (level == 0)) && (fmt != NULL))
    {
      va_list ap;
      va_start(ap, fmt);

      vsnprintf((char *)&temp, 2048, fmt, ap);

      ufprintf(logfile, temp);
	  
      va_end(ap);
    }
}

