/*******************************************************************
 * File: wpa2.c
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: wpa2.c,v 1.9 2005/08/09 01:39:14 chessing Exp $
 * $Date: 2005/08/09 01:39:14 $
 * $Log: wpa2.c,v $
 * Revision 1.9  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <string.h>
#include <inttypes.h>
#include <openssl/hmac.h>
#include <openssl/aes.h>

#include "config.h"
#include "profile.h"
#include "xsup_debug.h"
#include "wpa.h"
#include "wpa_common.h"
#include "wpa2.h"
#include "cardif/cardif.h"

const char wpa2oui[3] = {0x00, 0x0f, 0xac};
uint16_t wpa2_ver = 1;


void wpa2_gen_ie(struct interface_data *thisint, char *iedata, int *ielen)
{
  struct config_network *network_data;

  network_data = config_get_network_config();
  
  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration struct! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return;
    }

  // The first byte is the Element ID for WPA2, which is 0x30.
  iedata[0] = WPA2_EID;

  // A simple IE with capabilities should be 20 bytes long.
  iedata[1] = 20;

  // Set the version #
  iedata[2] = 0x01;
  iedata[3] = 0x00;

  // The group key cipher suite.
  memcpy(&iedata[4], wpa2oui, 3);

  iedata[7] = network_data->wpa_group_crypt;
  debug_printf(DEBUG_CONFIG, "Using Group Cipher Suite : ");
  wpa_print_cipher_suite(DEBUG_CONFIG, iedata[7]);

  // We can only have 1 pairwise cipher suite!
  iedata[8] = 0x01;
  iedata[9] = 0x00;

  // The pairwise cipher suite.
  memcpy(&iedata[10], wpa2oui, 3);
  
  iedata[13] = network_data->wpa_pairwise_crypt;
  debug_printf(DEBUG_CONFIG, "Using Pairwise Cipher Suite : ");
  wpa_print_cipher_suite(DEBUG_CONFIG, iedata[13]);

  if ((network_data->wpa_group_crypt == CIPHER_TKIP) &&
      ((network_data->wpa_pairwise_crypt == CIPHER_WRAP) ||
       (network_data->wpa_pairwise_crypt == CIPHER_CCMP)))
    {
      if (config_get_display_friendly_warnings())
	{
	  debug_printf(DEBUG_NORMAL, "WARNING : Group cipher is TKIP and "
		       "pairwise cipher is using AES.  Many wireless cards "
		       "have problems with this combination!\n");
	}
    }

  if (((network_data->wpa_group_crypt == CIPHER_WEP40) ||
       (network_data->wpa_group_crypt == CIPHER_WEP104)) &&
      ((network_data->wpa_pairwise_crypt == CIPHER_WRAP) ||
       (network_data->wpa_pairwise_crypt == CIPHER_CCMP)))
    {
      if (config_get_display_friendly_warnings())
	{
	  debug_printf(DEBUG_NORMAL, "WARNING : Group cipher is WEP and "
		       "pairwise cipher is using AES.  Many wireless cards "
		       "have problems with this combination!\n");
	}
    }

  // For the authenticated key management suite, we can also only have 1.
  iedata[14] = 0x01;
  iedata[15] = 0x00;

  // The authenticated key management suite.
  memcpy(&iedata[16], wpa2oui, 3);

  if (network_data->methods->method_num == WPA_PSK)
    {
      iedata[19] = 2;  // PSK
      debug_printf(DEBUG_CONFIG, "Using PSK.\n");
    } else {
      iedata[19] = 1;  // 802.1X
      debug_printf(DEBUG_CONFIG, "Using 802.1X\n");
    }

  // We don't support capabilities yet.
  iedata[20] = 0x00;
  iedata[21] = 0x00;

  *ielen = 22;
}
 
void wpa2_gen_ie_caps(struct interface_data *thisint, char *iedata)
{
  wpa_gen_ie(thisint, iedata);

  iedata[1] = 24;
  iedata[24] = 0x00;
  iedata[25] = 0x00;
} 
  
void wpa2_parse_ie(char *iedata)
{
  struct wpa2_ie_struct *ie_struct;
  char suite_id[4];
  int i, ieptr;
  uint16_t value16;

  ie_struct = (struct wpa2_ie_struct *)iedata;

  if (ie_struct->wpaid != WPA2_EID)
    {
      debug_printf(DEBUG_NORMAL, "IE is not a valid WPA2/802.11i IE! (Invalid vendor value!)\n");
      return;
    }

  debug_printf(DEBUG_INT, "--- WPA2/802.11i Data ---\n");

  debug_printf(DEBUG_INT, "WPA2/802.11i Version : %d\n", 
	       ie_struct->rsn_ver);

  if (ie_struct->wpalen <= 2)
    {
      debug_printf(DEBUG_NORMAL, "Short IE.  Should assume TKIP/TKIP for "
		   "ciphers!\n");
      return;
    }

  debug_printf(DEBUG_INT, "Group Key Cipher Suite : ");
  wpa_print_cipher_suite(DEBUG_CONFIG, ie_struct->group_cipher[3]);

  if (ie_struct->wpalen <= 6)
    {
      debug_printf(DEBUG_NORMAL, "Short IE.  Should assume TKIP for "
		   "pairwise cipher.\n");
      return;
    }

  debug_printf(DEBUG_INT, "Pairwise Key Cipher Suite Count : %d\n",
	       ie_struct->pk_suite_cnt);

  ieptr = sizeof(struct wpa2_ie_struct);

  for (i=0;i<ie_struct->pk_suite_cnt;i++)
    {
      if (ie_struct->wpalen < (ieptr-2) + 4)
	{
	  debug_printf(DEBUG_NORMAL, "Invalid IE!  The length specified by the"
		       " IE isn't long enough to cover the number of "
		       "pairwise ciphers the IE claims it lists!\n");
	  return;
	}

      debug_printf(DEBUG_INT, "Cipher Suite : ");
      memcpy((char *)&suite_id, (char *)&iedata[ieptr], 4);

      if (memcmp(suite_id, wpa2oui, 3) != 0)
	{
	  debug_printf(DEBUG_NORMAL, "\nInvalid WPA2/802.11i OUI!\n");
	} else {
	  wpa_print_cipher_suite(DEBUG_CONFIG, suite_id[3]);
	}

      ieptr+=4;
    }

  if (ie_struct->wpalen < (ieptr - 2) + 2)
    {
      debug_printf(DEBUG_NORMAL, "Short IE.  Should assume an AKM of EAP!\n");
      return;
    }

  memcpy((char *)&value16, (char *)&iedata[ieptr], 2);
  ieptr+=2;
  debug_printf(DEBUG_INT, "Authenticated Key Management Suite Count : %d\n",
	       value16);

  for (i=0;i<value16;i++)
    {
      if (ie_struct->wpalen < (ieptr-2) + 4)
	{
	  debug_printf(DEBUG_NORMAL, "Invalid IE!  The length claimed by the "
		       "IE isn't long enough to cover the number of "
		       "Authenticated Key Management suites it claims!\n");
	  return;
	}

      debug_printf(DEBUG_INT, "Authentication Suite : ");
      memcpy((char *)&suite_id, (char *)&iedata[ieptr], 4);

      if (memcmp(suite_id, wpa2oui, 3) != 0)
	{
	  debug_printf(DEBUG_INT, "Invalid WPA2/802.11i OUI!\n");
	} else {
	  wpa_print_auth_suite(DEBUG_CONFIG, suite_id[3]);
	}
      ieptr+=4;
    }

  if ((ieptr-2) < ie_struct->wpalen)
    {
      memcpy((char *)&value16, (char *)&iedata[ieptr], 2);
      debug_printf(DEBUG_INT, "RSN Capabilities : %04X\n\n", value16);
    }
}

