/*******************************************************************
 * File: wpa.c
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: wpa.c,v 1.14 2005/08/09 01:39:14 chessing Exp $
 * $Date: 2005/08/09 01:39:14 $
 * $Log: wpa.c,v $
 * Revision 1.14  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <string.h>
#include <inttypes.h>
#include <unistd.h>
#include <sys/types.h>

#include "config.h"
#include "profile.h"
#include "xsup_debug.h"
#include "wpa.h"
#include "cardif/cardif.h"
#include "wpa_common.h"

const char wpaoui[3] = {0x00, 0x50, 0xf2};

void wpa_gen_ie(struct interface_data *thisint, char *iedata)
{
  struct config_network *network_data;

  network_data = config_get_network_config();
  
  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network data! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return;
    }

  // The first byte is the Element ID for WPA, which is 0xdd.
  iedata[0] = WPA_EID;

  // For WPA (without capabilities), the length value will always be 22.
  iedata[1] = 22;

  // For WPA, we need to add the "special" OUI before the version #.
  memcpy(&iedata[2], wpaoui, 3);
  iedata[5] = 0x01;

  // Set the version #
  iedata[6] = 0x01;
  iedata[7] = 0x00;

  // The group key cipher suite.
  memcpy(&iedata[8], wpaoui, 3);

  iedata[11] = network_data->wpa_group_crypt;
  debug_printf(DEBUG_CONFIG, "Using Group Cipher Suite : ");
  wpa_print_cipher_suite(DEBUG_CONFIG, iedata[11]);

  // We can only have 1 pairwise cipher suite!
  iedata[12] = 0x01;
  iedata[13] = 0x00;

  // The pairwise cipher suite.
  memcpy(&iedata[14], wpaoui, 3);
  
  iedata[17] = network_data->wpa_pairwise_crypt;
  debug_printf(DEBUG_CONFIG, "Using Pairwise Cipher Suite : ");
  wpa_print_cipher_suite(DEBUG_CONFIG, iedata[17]);

  if ((network_data->wpa_group_crypt == CIPHER_TKIP) &&
      ((network_data->wpa_pairwise_crypt == CIPHER_WRAP) ||
       (network_data->wpa_pairwise_crypt == CIPHER_CCMP)))
    {

      if (config_get_display_friendly_warnings())
	{
	  debug_printf(DEBUG_NORMAL, "WARNING : Group cipher is TKIP and "
		       "pairwise cipher is using AES.  Many wireless cards "
		       "have problems with this combination!\n");
	}
    }

  if (((network_data->wpa_group_crypt == CIPHER_WEP40) ||
       (network_data->wpa_group_crypt == CIPHER_WEP104)) &&
      ((network_data->wpa_pairwise_crypt == CIPHER_WRAP) ||
       (network_data->wpa_pairwise_crypt == CIPHER_CCMP)))
    {
      
      if (config_get_display_friendly_warnings())
	{
	  debug_printf(DEBUG_NORMAL, "WARNING : Group cipher is WEP and "
		       "pairwise cipher is using AES.  Many wireless cards "
		       "have problems with this combination!\n");
	}
    }

  // For the authenticated key management suite, we can also only have 1.
  iedata[18] = 0x01;
  iedata[19] = 0x00;

  // The authenticated key management suite.
  memcpy(&iedata[20], wpaoui, 3);

  if (network_data->methods->method_num == WPA_PSK)
    {
      iedata[23] = 2;  // WPA-PSK
      debug_printf(DEBUG_CONFIG, "Using WPA-PSK.\n");
    } else {
      iedata[23] = 1;  // WPA with 802.1X
      debug_printf(DEBUG_CONFIG, "Using WPA with 802.1X\n");
    }
}
 
void wpa_gen_ie_caps(struct interface_data *thisint, char *iedata)
{
  wpa_gen_ie(thisint, iedata);

  iedata[1] = 24;
  iedata[24] = 0x00;
  iedata[25] = 0x00;
} 
  
void wpa_parse_ie(char *iedata)
{
  struct wpa_ie_struct *ie_struct;
  char wpa_oui[3] = {0x00, 0x50, 0xf2};
  char suite_id[4];
  int i, ieptr;
  uint16_t value16;

  ie_struct = (struct wpa_ie_struct *)iedata;

  if (ie_struct->wpaid != 0xdd)
    {
      debug_printf(DEBUG_NORMAL, "IE is not a valid WPA IE! (Invalid vendor value!)\n");
      return;
    }

  debug_printf(DEBUG_INT, "--- WPA Data ---\n");
  
  if ((memcmp(ie_struct->oui, wpa_oui, 3) != 0) && (ie_struct->oui[4] != 0x01))
    {
      debug_printf(DEBUG_NORMAL, "IE is not a valid WPA IE! (Invalid OUI value!\n");
      return;
    }

  debug_printf(DEBUG_INT, "WPA Version : %d\n", ie_struct->wpa_ver);

  // From here, everything else is technically optional.
  if (ie_struct->wpalen <= 6)
    {
      // Nothing after the version #.
      debug_printf(DEBUG_NORMAL, "Short WPA IE.  Should assume TKIP/TKIP for "
		   "keying!\n");
      return;
    }  // Otherwise, we have a group cipher suite.

  debug_printf(DEBUG_INT, "Group Key Cipher Suite : ");
  wpa_print_cipher_suite(DEBUG_INT, ie_struct->group_cipher[3]);

  if (ie_struct->wpalen <= 10)
    {
      debug_printf(DEBUG_NORMAL, "Short WPA IE.  Should assume TKIP for "
		   "pairwise cipher.\n");
      return;
    }

  debug_printf(DEBUG_INT, "Pairwise Key Cipher Suite Count : %d\n",
	       ie_struct->pk_suite_cnt);

  ieptr = sizeof(struct wpa_ie_struct);

  for (i=0;i<ie_struct->pk_suite_cnt;i++)
    {
      if (ie_struct->wpalen < (ieptr-2) + 4)
	{
	  debug_printf(DEBUG_NORMAL, "Invalid WPA IE!  The number of "
		       "cipher suites is too high for the length of the IE!"
		       "\n");
	  return;
	}

      debug_printf(DEBUG_INT, "Cipher Suite : ");
      memcpy((char *)&suite_id, (char *)&iedata[ieptr], 4);

      if (memcmp(suite_id, wpa_oui, 3) != 0)
	{
	  debug_printf(DEBUG_NORMAL, "Invalid WPA OUI!\n");
	} else {
	  wpa_print_cipher_suite(DEBUG_INT, suite_id[3]);
	}

      ieptr+=4;
    }

  if (ie_struct->wpalen < (ieptr-2) + 2)
    {
      debug_printf(DEBUG_NORMAL, "Short IE.  Should assume an AKM of EAP.\n");
      return;
    }

  memcpy((char *)&value16, (char *)&iedata[ieptr], 2);
  ieptr+=2;
  debug_printf(DEBUG_INT, "Authenticated Key Management Suite Count : %d\n",
	       value16);

  for (i=0;i<value16;i++)
    {
      if (ie_struct->wpalen < (ieptr-2) + 4)
	{
	  debug_printf(DEBUG_NORMAL, "Truncated IE!  The length provided in "
		       "the IE isn't long enough to include the number of "
		       "Authenticated Key Management Suites claimed!\n");
	  return;
	}

      debug_printf(DEBUG_INT, "Authentication Suite : ");
      memcpy((char *)&suite_id, (char *)&iedata[ieptr], 4);

      if (memcmp(suite_id, wpa_oui, 3) != 0)
	{
	  debug_printf(DEBUG_NORMAL, "Invalid WPA OUI!\n");
	} else {
	  wpa_print_auth_suite(DEBUG_INT, suite_id[3]);
	}
      ieptr+=4;
    }

  if ((ieptr-2) < ie_struct->wpalen)
    {
      memcpy((char *)&value16, (char *)&iedata[ieptr], 2);
      debug_printf(DEBUG_INT, "RSN Capabilities : %04X\n\n", value16);
    }
}

/* These functions are defined by 802.11i-D3.0 */
void wpa_STADisconnect()
{

}

void wpa_RemoveGTK()
{

}

