/*******************************************************************
 * Handle state machine related functions.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 * File: statemachine.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: statemachine.c,v 1.48 2005/08/20 19:06:53 chessing Exp $
 * $Date: 2005/08/20 19:06:53 $
 * $Log: statemachine.c,v $
 * Revision 1.48  2005/08/20 19:06:53  chessing
 * Patch from Carsten Grohmann to fix a few things in xsup_get_state.c.  Also added the ability to define an empty network clause, that will set the card in to encryption disabled mode.  From there, anything short of changing the SSID will be ignored by Xsupplicant.
 *
 * Revision 1.47  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <netinet/in.h>

#include "profile.h"
#include "config.h"
#include "snmp.h"
#include "statemachine.h"
#include "xsup_debug.h"
#include "frame_structs.h"
#include "eap.h"
#include "eapol.h"
#include "xsup_err.h"
#include "xsup_ipc.h"
#include "wpa.h"
#include "backend_sm.h"
#include "cardif/cardif.h"
#include "eap_types/leap/eapleap.h"
#include "timer.h"

// This is needed to keep the compiler from complaining about not knowing where
// global_deinit() is.  We should probably change this to something a little
// more "correct".
extern void global_deinit();

/******************************************
 *
 * Decrement a value, as long as it is greater than 0.
 *
 ******************************************/
void dec_if_nz(char *decval)
{
  if (!decval) return;

  if (*decval > 0) (*decval)--;
}

/*******************************************
 *
 * Delete all keys that are currently in the driver.
 *
 *******************************************/
void statemachine_clear_keys(struct interface_data *intdata)
{
  int i;

  debug_printf(DEBUG_INT, "Clearing keys!\n");
  
  // Clear the PTK.
  cardif_delete_key(intdata, 0, 1);

  for (i=0;i<4;i++)
    {
      cardif_delete_key(intdata, i, 0);
    }

  // Accept unencrypted frames again.
  cardif_drop_unencrypted(intdata, 0);
}

/******************************************
 *
 * Initalize the state machine
 *
 ******************************************/
int statemachine_init(struct interface_data *newint)
{
  if (!newint) 
    {
      debug_printf(DEBUG_NORMAL, "newint == NULL in %s:%d!\n", __FUNCTION__,
		   __LINE__);
      return XEMALLOC;
    }

  newint->statemachine = (struct dot1x_state *)malloc(sizeof(struct dot1x_state));
  if (newint->statemachine == NULL) return XEMALLOC;

  statemachine_reinit(newint);

  backend_sm_init(newint);

  statemachine_clear_keys(newint);

  return XENONE;
}

/******************************************
 *
 * Reinitalize the state machine
 *
 ******************************************/
int statemachine_reinit(struct interface_data *newint)
{
  if (!newint) 
    {
      debug_printf(DEBUG_NORMAL, "newint == NULL in %s:%d!\n", __FUNCTION__,
		   __LINE__);
      return XEMALLOC;
    }

  debug_printf(DEBUG_STATE, "Reinit state machine\n");

  newint->statemachine->initialize = 0;

  // Now, we want to set up a few defaults as per the 802.1x doc, and
  // initalize a few other statemachine variables that we will be needing.
  newint->statemachine->authPeriod = 30;
  newint->statemachine->authWhile = newint->statemachine->authPeriod;

  newint->statemachine->heldPeriod = 60;
  newint->statemachine->heldWhile = newint->statemachine->heldPeriod;

  newint->statemachine->startPeriod = 30;
  newint->statemachine->startWhen = 0;     // Trigger sending an EAPOL-Start
  newint->statemachine->maxStart = 3;

  // Set up our inital state.
  newint->statemachine->userLogoff = FALSE;
  newint->statemachine->logoffSent = FALSE;
  newint->statemachine->eapFail = FALSE;
  newint->statemachine->startCount = 0;

  newint->statemachine->tick = FALSE;

  /* Some new variables defined by 802.1X-REV-d11 (2004) */
  newint->statemachine->eapFail = FALSE;
  newint->statemachine->eapolEap = FALSE;
  newint->statemachine->eapSuccess = FALSE;
  newint->statemachine->keyAvailable = FALSE;
  newint->statemachine->keyDone = FALSE;
  newint->statemachine->keyRun = FALSE;
  newint->statemachine->keyTxEnabled = FALSE;
  newint->statemachine->portControl = AUTO;
  newint->statemachine->portEnabled = FALSE;
  newint->statemachine->portValid = TRUE;
  newint->statemachine->suppAbort = FALSE;
  newint->statemachine->suppFail = FALSE;
  newint->statemachine->suppPortStatus = UNAUTHORIZED;
  newint->statemachine->suppStart = FALSE;
  newint->statemachine->suppSuccess = FALSE;
  newint->statemachine->suppTimeout = FALSE;

  newint->statemachine->eapRestart = FALSE;
  newint->statemachine->logoffSent = FALSE;
  newint->statemachine->sPortMode = AUTO;
  newint->statemachine->startCount = 0;

  /* Init 802.11i-D3.0 vars. */
  newint->statemachine->DeauthenticationRequest = FALSE;
  newint->statemachine->AuthenticationRequest = FALSE;
  newint->statemachine->AuthenticationFailed = FALSE;
  newint->statemachine->EAPOLKeyReceived = FALSE;
  newint->statemachine->IntegrityFailed = FALSE;
  newint->statemachine->MICVerified = FALSE;
  newint->statemachine->Counter = 0;
  newint->statemachine->SNonce = NULL;
  newint->statemachine->PTK = NULL;
  newint->statemachine->TPTK = NULL;
  newint->statemachine->GTK = NULL;
  newint->statemachine->PMK = NULL;

  newint->statemachine->MICfailures = 0;

  // The 0 below is the DISCONNECTED state.
  newint->statemachine->wpaCurState = 0;
  newint->statemachine->wpaLastState = -1;

  newint->statemachine->curState = DISCONNECTED;
  newint->statemachine->lastState = DISCONNECTED;

  newint->statemachine->unicastKeyLen = 0;
  newint->statemachine->broadcastKeyLen = 0;

  backend_sm_reinit(newint);

  return XENONE;
}

/******************************************
 *
 * Process the state machine, send a frame if we need to.  Returns >0 if
 * there is a frame to be send.
 *
 ******************************************/
int statemachine_run(struct interface_data *thisint)
{
  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface struct passed in to "
		   "%s:%d!\n", __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  if (thisint->statemachine == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Statemachine is not set up correctly in "
		   "%s:%d!\n", __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  if (thisint->statemachine->tick == TRUE)
    {
      // The clock ticked -- Update all of the needed counters.
      dec_if_nz(&thisint->statemachine->authWhile);
      dec_if_nz(&thisint->statemachine->heldWhile);
      dec_if_nz(&thisint->statemachine->startWhen);

      thisint->statemachine->tick = FALSE;
      debug_printf(DEBUG_EVERYTHING, "Clock tick! authWhile=%d heldWhile=%d "
		   "startWhen=%d curState=",
		   thisint->statemachine->authWhile,
		   thisint->statemachine->heldWhile,
		   thisint->statemachine->startWhen,
		   thisint->statemachine->curState);
      
      switch (thisint->statemachine->curState)
	{
	case DISCONNECTED:
	  debug_printf_nl(DEBUG_EVERYTHING, "DISCONNECTED\n");
	  break;
	case LOGOFF:
	  debug_printf_nl(DEBUG_EVERYTHING, "LOGOFF\n");
	  break;
	case ACQUIRED:   // No longer part of 802.1X in 802.1X-REV-d11.
	  debug_printf_nl(DEBUG_EVERYTHING, "ACQUIRED\n");
	  break;
	case AUTHENTICATING:
	  debug_printf_nl(DEBUG_EVERYTHING, "AUTHENTICATING\n");
	  break;
	case AUTHENTICATED:
	  debug_printf_nl(DEBUG_EVERYTHING, "AUTHENTICATED\n");
	  break;
	case CONNECTING:
	  debug_printf_nl(DEBUG_EVERYTHING, "CONNECTING\n");
	  break;
	case HELD:
	  debug_printf_nl(DEBUG_EVERYTHING, "HELD\n");
	  break;
	case RESTART:
	  debug_printf_nl(DEBUG_EVERYTHING, "RESTART\n");
	  break;
	case S_FORCE_AUTH:
	  debug_printf_nl(DEBUG_EVERYTHING, "S_FORCE_AUTH\n");
	  break;
	case S_FORCE_UNAUTH:
	  debug_printf_nl(DEBUG_EVERYTHING, "S_FORCE_UNAUTH\n");
	  break;
	default:
	  debug_printf_nl(DEBUG_EVERYTHING, "UNKNOWN!\n");
	  break;
	}
    }

  thisint->statemachine->portEnabled = cardif_get_if_state(thisint);

  if ((thisint->statemachine->userLogoff && !thisint->statemachine->logoffSent)
      && !(thisint->statemachine->initialize || 
	   !thisint->statemachine->portEnabled))
    {
      debug_printf(DEBUG_STATE, "(global) -> LOGOFF\n");
      thisint->statemachine->curState = LOGOFF;
      if (thisint->flags & IS_WIRELESS) cardif_roam(thisint);
      xsup_ipc_send_auth_state(thisint);
    }

  if ((thisint->statemachine->portControl == FORCE_AUTHORIZED) &&
      (thisint->statemachine->sPortMode != thisint->statemachine->portControl)
      && !(thisint->statemachine->initialize || 
	   !thisint->statemachine->portEnabled))
    {
      debug_printf(DEBUG_STATE, "(global) -> S_FORCE_AUTH\n");
      thisint->statemachine->curState = S_FORCE_AUTH;
      xsup_ipc_send_auth_state(thisint);
    }

  if ((thisint->statemachine->portControl == FORCE_UNAUTHORIZED) &&
      (thisint->statemachine->sPortMode != thisint->statemachine->portControl)
      && !(thisint->statemachine->initialize ||
	   !thisint->statemachine->portEnabled))
    {
      debug_printf(DEBUG_STATE, "(global) -> S_FORCE_UNAUTH\n");
      thisint->statemachine->curState = S_FORCE_UNAUTH;
      xsup_ipc_send_auth_state(thisint);
    }

  if ((thisint->statemachine->portControl == AUTO) &&
      ((thisint->statemachine->sPortMode != thisint->statemachine->portControl)
      || thisint->statemachine->initialize || 
       !thisint->statemachine->portEnabled))
    {
      debug_printf(DEBUG_STATE, "(global) -> DISCONNECTED\n");
      thisint->statemachine->curState = DISCONNECTED;
      statemachine_clear_keys(thisint);
      xsup_ipc_send_auth_state(thisint);
    }

  switch(thisint->statemachine->curState)
    {
    case S_FORCE_AUTH:
      if (thisint->statemachine->lastState != thisint->statemachine->curState)
	{
	  thisint->statemachine->suppPortStatus = AUTHORIZED;
	  thisint->statemachine->sPortMode = FORCE_AUTHORIZED;
	  thisint->statemachine->lastState = thisint->statemachine->curState;
	}
      break;

    case S_FORCE_UNAUTH:
      if (thisint->statemachine->lastState != thisint->statemachine->curState)
	{
	  thisint->statemachine->suppPortStatus = UNAUTHORIZED;
	  thisint->statemachine->sPortMode = FORCE_UNAUTHORIZED;
	  txLogoff(thisint);
	  thisint->statemachine->lastState = thisint->statemachine->curState;
	}
      break;

    case LOGOFF:
      if (thisint->statemachine->lastState != thisint->statemachine->curState)
	{
	  txLogoff(thisint);
	  thisint->statemachine->logoffSent = TRUE;
	  thisint->statemachine->suppPortStatus = UNAUTHORIZED;
	  thisint->statemachine->lastState = thisint->statemachine->curState;
	}

      if (!thisint->statemachine->userLogoff)
	{
	  debug_printf(DEBUG_STATE, "LOGOFF -> DISCONNECTED\n");
	  thisint->statemachine->curState = DISCONNECTED;
	  xsup_ipc_send_auth_state(thisint);
	}
      break;

    case DISCONNECTED:
      if (thisint->statemachine->lastState != thisint->statemachine->curState)
	{
	  thisint->statemachine->sPortMode = AUTO;
	  thisint->statemachine->startCount = 0;
	  thisint->statemachine->logoffSent = FALSE;
	  thisint->statemachine->suppPortStatus = UNAUTHORIZED;
	  thisint->statemachine->suppAbort = TRUE;
	  thisint->statemachine->lastState = thisint->statemachine->curState;
	}

      if ((!thisint->statemachine->initialize) &&
	  (thisint->statemachine->portEnabled))
	{
	  debug_printf(DEBUG_STATE, "DISCONNECTED -> CONNECTING\n");
	  thisint->statemachine->curState = CONNECTING;
	  xsup_ipc_send_auth_state(thisint);
	}
      break;

    case HELD:
      if (thisint->statemachine->lastState != thisint->statemachine->curState)
	{
	  thisint->statemachine->heldWhile = thisint->statemachine->heldPeriod;
	  thisint->statemachine->suppPortStatus = UNAUTHORIZED;
	  thisint->statemachine->lastState = thisint->statemachine->curState;
	}

      if (thisint->statemachine->heldWhile == 0)
	{
	  debug_printf(DEBUG_STATE, "HELD -> CONNECTING\n");
	  thisint->statemachine->curState = CONNECTING;
	  xsup_ipc_send_auth_state(thisint);
	}
      
      if (thisint->statemachine->eapolEap)
	{
	  debug_printf(DEBUG_STATE, "HELD -> RESTART\n");
	  thisint->statemachine->curState = RESTART;
	  xsup_ipc_send_auth_state(thisint);
	}
      break;

    case CONNECTING:
      if (thisint->statemachine->lastState != thisint->statemachine->curState)
	{
	  thisint->statemachine->startWhen = thisint->statemachine->startPeriod;
	  thisint->statemachine->startCount = thisint->statemachine->startCount + 1;
	  thisint->statemachine->eapolEap = FALSE;
	  thisint->statemachine->lastState = thisint->statemachine->curState;
	  txStart(thisint);
	}

      if ((thisint->statemachine->startWhen == 0) &&
	  (thisint->statemachine->startCount < thisint->statemachine->maxStart))
	{
	  // Need to fake the state we were previously in, so that we will
	  // redo the base connecting state.
	  debug_printf(DEBUG_STATE, "CONNECTING -> CONNECTING\n");
	  thisint->statemachine->lastState = DISCONNECTED;
	  xsup_ipc_send_auth_state(thisint);
	}

      if (((thisint->statemachine->startWhen == 0) &&
	   (thisint->statemachine->startCount >= thisint->statemachine->maxStart)
	   && thisint->statemachine->portValid))
	{
	  // Here, we have defaulted in to authenticated state.  We should
	  // display a message to the user to avoid confusion, and terminate
	  // if we were asked to.
	  debug_printf(DEBUG_STATE, "CONNECTING -> AUTHENTICATED\n");
	  thisint->statemachine->curState = AUTHENTICATED;
	  xsup_ipc_send_auth_state(thisint);
	  
	  debug_printf(DEBUG_NORMAL, "Xsupplicant has defaulted to "
		       "authenticated state, due to the inability to\n"
		       "successfully start/complete an EAP conversation.  It "
		       "is likely that this\nauthenticator doesn't support "
		       "802.1X, or that 802.1X isn't configured correctly\n"
		       "on the authenticator or RADIUS server.\n");

	  if (thisint->flags & TERM_ON_FAIL)
	    {
	      // We have been instructed to terminate.  So, do it cleanly.
	      global_deinit();
	    }
	}

      if (thisint->statemachine->eapSuccess || thisint->statemachine->eapFail)
	{
	  debug_printf(DEBUG_STATE, "CONNECTING -> AUTHENTICATING\n");
	  thisint->statemachine->curState = AUTHENTICATING;
	  xsup_ipc_send_auth_state(thisint);
	}

      if (thisint->statemachine->eapolEap)
	{
	  debug_printf(DEBUG_STATE, "CONNECTING -> RESTART\n");
	  thisint->statemachine->curState = RESTART;
	  xsup_ipc_send_auth_state(thisint);
	}

      if ((thisint->statemachine->startWhen == 0) &&
	  (thisint->statemachine->startCount >= thisint->statemachine->maxStart)
	  && !thisint->statemachine->portValid)
	{
	  debug_printf(DEBUG_STATE, "CONNECTING -> HELD\n");
	  thisint->statemachine->curState = HELD;
	  xsup_ipc_send_auth_state(thisint);
	}
      break;

    case RESTART:
      if (thisint->statemachine->lastState != thisint->statemachine->curState)
	{
	  thisint->statemachine->eapRestart = TRUE;
	  thisint->statemachine->lastState = thisint->statemachine->curState;
	}

      if (!thisint->statemachine->eapRestart)
	{
	  thisint->statemachine->curState = AUTHENTICATING;
	  debug_printf(DEBUG_STATE, "RESTART -> AUTHENTICATING\n");
	  xsup_ipc_send_auth_state(thisint);
	}
      break;

    case AUTHENTICATING:
      if (thisint->statemachine->curState != thisint->statemachine->lastState)
	{
	  thisint->statemachine->startCount = 0;
	  thisint->statemachine->suppSuccess = FALSE;
	  thisint->statemachine->suppFail = FALSE;
	  thisint->statemachine->suppTimeout = FALSE;
	  thisint->statemachine->keyRun = FALSE;
	  thisint->statemachine->keyDone = FALSE;
	  thisint->statemachine->suppStart = TRUE;
	  thisint->statemachine->lastState = thisint->statemachine->curState;
	}

      if (thisint->statemachine->suppSuccess &&
	  thisint->statemachine->portValid)
	{
	  thisint->statemachine->curState = AUTHENTICATED;
	  debug_printf(DEBUG_STATE, "AUTHENTICATING -> AUTHENTICATED\n");
	  timer_cancel(AUTHENTICATION_TIMER);
	  timer_cancel(ASSOCIATION_TIMER);
	  xsup_ipc_send_auth_state(thisint);
	}

      if (thisint->statemachine->suppFail || (thisint->statemachine->keyDone &&
					      !thisint->statemachine->portValid))
	{
	  thisint->statemachine->curState = HELD;
	  debug_printf(DEBUG_STATE, "AUTHENTICATING -> HELD\n");
	  xsup_ipc_send_auth_state(thisint);
	}

      if (thisint->statemachine->suppTimeout)
	{
	  thisint->statemachine->curState = CONNECTING;
	  debug_printf(DEBUG_STATE, "AUTHENTICATING -> CONNECTING\n");
	  xsup_ipc_send_auth_state(thisint);
	}
      break;

    case AUTHENTICATED:
      if (thisint->statemachine->curState != thisint->statemachine->lastState)
	{
	  thisint->statemachine->suppPortStatus = AUTHORIZED;
	  thisint->statemachine->lastState = thisint->statemachine->curState;
	}

      if (thisint->statemachine->eapolEap && 
	  thisint->statemachine->portValid)
	{
	  thisint->statemachine->curState = RESTART;
	  debug_printf(DEBUG_STATE, "AUTHENTICATED -> RESTART\n");
	  xsup_ipc_send_auth_state(thisint);
	}

      if (!thisint->statemachine->portValid)
	{
	  thisint->statemachine->curState = DISCONNECTED;
	  debug_printf(DEBUG_STATE, "AUTHENTICATED -> DISCONNECTED\n");
	  xsup_ipc_send_auth_state(thisint);
	}
      break;
    }

  do 
    {
      backend_sm_run(thisint);
    } while (thisint->statemachine->beLastState != thisint->statemachine->beCurState);
  return backend_sm_run(thisint);
}

/*****************************************
 *
 * Clean up our state machine.
 *
 *****************************************/
int statemachine_cleanup(struct interface_data *thisint)
{
  debug_printf(DEBUG_EVERYTHING, "Doing statemachine cleanup!\n");

  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed in to statemachine_cleanup()!\n");
      return XEMALLOC;
    }

  if (thisint->statemachine != NULL)
    {
      free(thisint->statemachine);
      thisint->statemachine = NULL;
    }

  backend_sm_deinit(thisint);
  
  return XENONE;
}

/*****************************************
 *
 * Create a logoff frame to be sent out to the network.
 *
 *****************************************/
int txLogoff(struct interface_data *thisint)
{
  int eapolver;
  struct config_network *network_data;

  // Pick up our current network configuration.
  network_data = config_get_network_config();

  if (network_data == NULL) 
    {
      debug_printf(DEBUG_NORMAL, "No network information available.  Not "
		   "sending a logoff!\n");
      return XENONE;      // Not having data here is NOT an error!
    }

  // If we aren't using 802.1X don't send logoffs.
  if (network_data->methods == NULL) return XENONE;
  if (network_data->methods->method_num == STATIC_WEP_METHOD) return XENONE;

  // If we are using WPA-PSK, don't send logoffs.
  if (network_data->methods->method_num == WPA_PSK) return XENONE;

  debug_printf(DEBUG_STATE, "Sending EAPOL-Logoff Frame.\n");

  eapolver = network_data->force_eapol_ver;

  // If we have not hard set the eapol version, then use the last
  // version that the authenticator gave us.  This isn't in line with
  // the 802.1X-REV-2004-d11 standard, but many authenticators are checking
  // version #s, and breaking when it isn't 1. :-/
  if ((eapolver < 1) || (eapolver > MAX_EAPOL_VER))
    {
      eapolver = snmp_get_dot1xSuppLastEapolFrameVersion();
      if (eapolver == 0)
	{
	  eapolver = MAX_EAPOL_VER;
	} 
    }

  eapol_build_header(EAPOL_LOGOFF, 0, eapolver, thisint->sendframe);
  thisint->send_size = OFFSET_TO_EAP;

  snmp_dot1xSuppEapolLogoffFramesTx();

  cardif_sendframe(thisint);

  return XENONE;
}


/*********************************************
 *
 * Build an EAPoL Start frame to be sent out to the network.
 *
 *********************************************/
int txStart(struct interface_data *thisint)
{
  int eapolver;
  struct config_network *network_data;

  network_data = config_get_network_config();

  // If we are using WPA-PSK, don't send starts.
  if (network_data->methods->method_num == WPA_PSK) return XENONE;

  debug_printf(DEBUG_STATE, "Sending EAPOL-Start Frame.\n");

  eapolver = network_data->force_eapol_ver;

  // If we have not hard set the eapol version, then use the last
  // version that the authenticator gave us.  This isn't in line with
  // the 802.1X-REV-2004-d11 standard, but many authenticators are checking
  // version #s, and breaking when it isn't 1. :-/
  if ((eapolver < 1) || (eapolver > MAX_EAPOL_VER))
    {
      eapolver = snmp_get_dot1xSuppLastEapolFrameVersion();

      if (eapolver == 0)
	{
	  eapolver = MAX_EAPOL_VER;
	} 
    }
  
  eapol_build_header(EAPOL_START, 0, network_data->force_eapol_ver,
		     thisint->sendframe);
  thisint->send_size = OFFSET_TO_EAP;

  snmp_dot1xSuppEapolStartFramesTx();

  cardif_sendframe(thisint);

  return XENONE; 
}
