/*******************************************************************
 * Handler for profile related things.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: profile.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: profile.c,v 1.31 2005/08/09 01:39:14 chessing Exp $
 * $Date: 2005/08/09 01:39:14 $
 * $Log: profile.c,v $
 * Revision 1.31  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <stdlib.h>
#include <strings.h>
#include <string.h>

#include "snmp.h"
#include "profile.h"
#include "xsup_err.h"
#include "cardif/cardif.h"
#include "xsup_debug.h"
#include "eapol.h"

/*******************************************
 *
 * Initalize the default values for the structure.  In general, state machine
 * and user configuration variables won't need to be set here.  We should
 * set up variables that are in the root of the structure.
 *
 *******************************************/
int init_interface_struct(struct interface_data *work, char *intname)
{
  char dot1x_default_dest[6] = {0x01, 0x80, 0xc2, 0x00, 0x00, 0x03};

  if (!work)
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed in to init_interface_struct()!\n");
      return XEMALLOC;
    }

  if (!intname)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface name passed in to init_interface_struct()!\n");
      return XEMALLOC;
    }

  work->intName = (char *)malloc(strlen(intname)+1);
  if (work->intName == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory in %s at line %d!\n", __FUNCTION__, __LINE__);
      return XEMALLOC;
    }
  bzero(work->intName, strlen(intname)+1);
  strncpy(work->intName, intname, strlen(intname));

  bzero(work->source_mac, 6);
  bzero(work->dest_mac, 6);

  work->keyingMaterial = NULL;

  work->tick = FALSE;

  work->rsn_ie = NULL;
  work->wpa_ie = NULL;

  work->flags = WAS_DOWN;

  // The default MAC specified by the IEEE 802.1x spec.
  memcpy(&work->dest_mac[0], &dot1x_default_dest, 6);
	
  work->cur_essid = NULL;

  eapol_init(work);

  work->tempPassword = NULL;

  work->send_size = 0;
  work->recv_size = 0;

  return XENONE;
}

/**********************************************
 *
 * Clean out a structure.  Clear out everything but return a pointer to "next"
 * the pointer to next should be used in a successive call.
 *
 **********************************************/
void destroy_interface_struct(struct interface_data *intdata)
{
  if (!intdata) return;

  if (intdata->intName) free(intdata->intName);

  if (intdata->cur_essid) free(intdata->cur_essid);
  if (intdata->keyingMaterial) free(intdata->keyingMaterial);
}
