/*******************************************************************
 * Handle MIC routines.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE for more info.)
 *
 * File: mic.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: mic.c,v 1.12 2005/08/09 01:39:14 chessing Exp $
 * $Date: 2005/08/09 01:39:14 $
 * $Log: mic.c,v $
 * Revision 1.12  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <openssl/hmac.h>
#include <string.h>

#include "xsup_debug.h"
#include "profile.h"
#include "eapol_key_type254.h"
#include "frame_structs.h"
#include "timer.h"
#include "cardif/cardif.h"
#include "mic.h"

/************************************************************************
 *
 * Calculate the MIC value for a key packet, and return it.
 *
 ************************************************************************/
void mic_process(char *key, int keylen, char *datain, int insize, int version, 
		 char *mic)
{
  char sha_hmac[20];
  int i;

  debug_printf(DEBUG_INT, "Calculating MIC for Version %d!\n", version);

  switch (version)
    {
    case 1:
      // Do an MD5 HMAC
      HMAC(EVP_md5(), key, keylen, datain, insize, mic, &i);
      break;

    case 2:
      // Do an SHA1 HMAC  
      // Since the HMAC will be 20 bytes, and we only need 16, we must use
      // a temporary variable.
      HMAC(EVP_sha1(), key, keylen, datain, insize, (char *)&sha_hmac, &i);
      memcpy(mic, &sha_hmac, 16);
      break;
    }
}

/*******************************************************************
 *
 * Given a frame, pull the MIC out, and check it.  If it matches,
 * return TRUE.  Otherwise, return FALSE.
 *
 *******************************************************************/
int mic_wpa_validate(char *inframe, int framesize, char *key, int keylen)
{
  struct wpa_key_packet *keydata;
  char *tempframe;
  char oldmic[16], newmic[16];
  uint16_t value16;
  int rc = FALSE;

  if (inframe == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid frame passed to %s!\n", __FUNCTION__);
      return FALSE;
    }

  tempframe = (char *)malloc(framesize);
  if (tempframe == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory at %s line %d!\n",
		   __FUNCTION__, __LINE__);
      return FALSE;
    }

  memcpy(tempframe, inframe, framesize);

  // First, get the keydata struct pointing to the correct place.
  keydata = (struct wpa_key_packet *)&tempframe[OFFSET_TO_EAPOL+4];
  
  memcpy(oldmic, keydata->key_mic, 16);
  bzero(keydata->key_mic, 16);

  memcpy(&value16, keydata->key_information, 2);
  value16 = ntohs(value16);
  value16 &= WPA_KEYTYPE_MASK;

  mic_process(key, keylen, (char *)&tempframe[OFFSET_TO_EAPOL],
	      framesize - OFFSET_TO_EAPOL, value16, newmic);

  if (memcmp(oldmic, newmic, 16) == 0) rc = TRUE;

  free(tempframe);
  tempframe = NULL;

  return rc;
}

/*******************************************************************
 *
 * Given a frame, calculate the MIC, and stick it in the frame.
 *
 *******************************************************************/
void mic_wpa_populate(char *inframe, int framesize, char *key, int keylen)
{
  struct wpa_key_packet *keydata;
  char newmic[16];
  uint16_t value16;

  if (inframe == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid frame passed to %s!\n", __FUNCTION__);
      return;
    }

  // First, get the keydata struct pointing to the correct place.
  keydata = (struct wpa_key_packet *)&inframe[OFFSET_TO_EAPOL+4];
  
  bzero(keydata->key_mic, 16);

  memcpy(&value16, &keydata->key_information, 2);
  value16 = ntohs(value16);
  value16 &= WPA_KEYTYPE_MASK;

  mic_process(key, keylen, (char *)&inframe[OFFSET_TO_EAPOL],
	      framesize - OFFSET_TO_EAPOL -4, value16, newmic);

  memcpy(keydata->key_mic, newmic, 16);
}


/*************************************
 *
 * When countermeasures have been enabled, this is the function that will
 * be called once they are ready to be disabled.
 *
 *************************************/
void mic_disable_countermeasures(struct interface_data *intdata)
{
  debug_printf(DEBUG_NORMAL, "MIC countermeasures disabled.\n");
  
  cardif_countermeasures(intdata, FALSE);
  timer_cancel(COUNTERMEASURE_TIMER);
}
