/*******************************************************************
 * The core linux event loop.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: linux_core.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: linux_core.c,v 1.21 2005/08/20 19:06:54 chessing Exp $
 * $Date: 2005/08/20 19:06:54 $
 * $Log: linux_core.c,v $
 * Revision 1.21  2005/08/20 19:06:54  chessing
 * Patch from Carsten Grohmann to fix a few things in xsup_get_state.c.  Also added the ability to define an empty network clause, that will set the card in to encryption disabled mode.  From there, anything short of changing the SSID will be ignored by Xsupplicant.
 *
 * Revision 1.20  2005/08/18 03:19:04  chessing
 * Added the ability to define an SSID with static WEP keys.  When we switch to a network that has this type of configuration we will set the keys, and stop the various association timers.
 *
 * Revision 1.19  2005/08/09 01:39:15  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#ifdef LINUX_FRAMER

#include <sys/select.h>
#include <sys/socket.h>
#include <asm/types.h>
#include <linux/netlink.h>
#include <linux/rtnetlink.h>
#include <unistd.h>
#include <string.h>

#include "profile.h"
#include "config.h"
#include "eap.h"
#include "statemachine.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "eapol.h"
#include "cardif/cardif.h"
#include "cardif/core.h"
#include "cardif/linux/cardif_linux_rtnetlink.h"
#include "timer.h"

char unassociated_warning = 0;

/*****************************************************************
 *
 * Do any setup that is needed for our event loop.  This is called only
 * once!  (And not once per interface.)  If you need to initalize something
 * per interface, is should either be included in cardif_init(), or using
 * some checks in event_core()!
 *
 *****************************************************************/
void event_core_setup()
{
  cardif_linux_rtnetlink_init();
}

// This is in mschapv2.c.  We need it for the static WEP piece.
extern void process_hex(char *, int, char *);

/*****************************************************************
 *
 * Check to see if the ESSID we are attached to uses static WEP.  If so,
 * then we need to set the keys, and set the DONT_KEY flag.  If not, and
 * the config is empty, then turn off encryption, and set the DONT_KEY
 * flag.
 *
 *****************************************************************/
void linux_core_check_non1x(struct interface_data *intcur)
{
  struct config_network *netdata;
  struct config_static_wep *wepdata;
  char key[26];                        // 26 chars should be the max we hold.
  int klen, keyidx, t;

  // If we are set not to key, then don't do anything, so we don't get in
  // the way of the user manually setting keys.
  if (TEST_FLAG(intcur->flags, DONT_KEY)) return;

  netdata = config_get_network_config();

  if (netdata->methods == NULL)
    {
      debug_printf(DEBUG_INT, "We have no 802.1X config for this network! "
		   "Clearing encryption.\n");

      // We don't have a config for this network, so make sure we turn off
      // keying so we don't have any problems.
      cardif_enc_disable(intcur);
      timer_cancel(AUTHENTICATION_TIMER);
      timer_cancel(ASSOCIATION_TIMER);

      SET_FLAG(intcur->flags, DONT_KEY);

      return;
    }

  if (netdata->methods->method_num == STATIC_WEP_METHOD)
    {
      debug_printf(DEBUG_INT, "We appear to be using static WEP.\n");
      // Since we are doing static WEP, we need to cancel any timers
      // that would mess with our state.
      timer_cancel(AUTHENTICATION_TIMER);
      timer_cancel(ASSOCIATION_TIMER);

      debug_printf(DEBUG_INT, "Setting static WEP key(s)!\n");
      wepdata = (struct config_static_wep *)netdata->methods->method_data;

      SET_FLAG(intcur->flags, DONT_KEY);
      
      for (keyidx=1;keyidx<5;keyidx++)
	{
	  if (wepdata->key[keyidx] != NULL)
	    {
	      klen = strlen(wepdata->key[keyidx]);
	      if (((klen/2) == 5) || ((klen/2) == 13))
		{
		  // We have a valid length key.  So, convert it, and set it.
		  process_hex(wepdata->key[keyidx], klen, key);

		  // Calculate the proper key index.
		  t = keyidx-1;

		  if (keyidx == wepdata->tx_key)
		    {
		      debug_printf(DEBUG_INT, "Setting TX key! [%d]\n",
				   wepdata->tx_key);
		      t |= 0x80;
		      cardif_set_wep_key(intcur, key, (klen/2), t);
		    } else {
		      cardif_set_wep_key(intcur, key, (klen/2), t);
		    }
		}
	    }
	}
    }
}

/*****************************************************************
 *
 * Do any event processing that needs to happen.  This *MUST* include
 * executing the statemachine for each interface we know of, and sending
 * and receiving of frames.
 *
 *****************************************************************/
void event_core(struct interface_data *intcur)
{
  struct timeval tv;
  fd_set rfds;
  int maxfd = 0, framewaiting = 0;

  if ((TEST_FLAG(intcur->flags, IS_WIRELESS) && (!(intcur->flags & DONT_KEY))))
	linux_core_check_non1x(intcur);
  
  // If the interface is wired, then move on.  Otherwise, only move on if
  // we are associated.
  if ((TEST_FLAG(intcur->flags, IS_WIRELESS)) 
      && (!(intcur->flags & ASSOCIATED)))
    {
      // If we aren't associated, then just check events until we are 
      // associated.
      if (unassociated_warning == 0)
	{
	  debug_printf(DEBUG_NORMAL, "Interface is currently not associated "
		       "to a wireless network.\n");
	  unassociated_warning = 1;
	}
      cardif_linux_rtnetlink_check_event(intcur);

      if (!(intcur->flags & DONT_KEY))
	linux_core_check_non1x(intcur);

      usleep(500000);

      // We want to have this timer tick, even if we aren't running the
      // state machine.
      if (intcur->tick) timer_tick(intcur);
      return;
    }

  // If we are associated, and using static WEP, then just return.
  if (TEST_FLAG(intcur->flags, DONT_KEY)) 
    {
      // We need to make sure that we process events, so we can get in to
      // dot1x state if it is needed.
      cardif_linux_rtnetlink_check_event(intcur);
      return;
    }

  unassociated_warning = 0;
  
  cardif_linux_rtnetlink_check_event(intcur);

  FD_ZERO(&rfds);
  
  eapol_execute(intcur);
  FD_SET(cardif_get_socket(intcur), &rfds);
  if (cardif_get_socket(intcur) > maxfd)
    maxfd = cardif_get_socket(intcur);
  
  // If we have a frame available on any interface, we won't
  // sleep.  Instead we will keep looping through to keep things
  // moving as fast as possible.
  if (cardif_frameavail(intcur) == TRUE) framewaiting = 1;
  
  if (framewaiting == 0)
    {
      tv.tv_sec = 0;
      tv.tv_usec = 500000;
      select(maxfd + 1, &rfds, NULL, NULL, &tv);
    }

  if (intcur->tick) timer_tick(intcur);
  framewaiting = 0;
}

/******************************************************************
 *
 * Clean up anything that we initialized in event_core_setup().
 *
 ******************************************************************/
void event_core_cleanup()
{
  debug_printf(DEBUG_INT, "Called event_core_cleanup()!\n");
  cardif_linux_rtnetlink_cleanup();
}

#endif
