/*******************************************************************
 * File: cardif_ipw_driver.c
 *
 * A significant portion of this code is either based on, or taken directly
 * from WPA Supplicant.  The copyright and license information for WPA
 * Supplicant is reproduced below.  Code taken from WPA Supplicant is notated
 * in comments above the function definitions.
 *
 * All other code is licensed under a dual GPL/BSD license.  (See LICENSE file 
 * for more info.)
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: cardif_ipw_driver.c,v 1.6 2005/08/20 19:06:54 chessing Exp $
 * $Date: 2005/08/20 19:06:54 $
 * $Log: cardif_ipw_driver.c,v $
 * Revision 1.6  2005/08/20 19:06:54  chessing
 * Patch from Carsten Grohmann to fix a few things in xsup_get_state.c.  Also added the ability to define an empty network clause, that will set the card in to encryption disabled mode.  From there, anything short of changing the SSID will be ignored by Xsupplicant.
 *
 * Revision 1.5  2005/08/09 01:39:15  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

/*
 * WPA Supplicant - driver interaction with Linux ipw2100/2200 drivers
 * Copyright (c) 2005 Zhu Yi <yi.zhu@intel.com>
 * Copyright (c) 2004 Lubomir Gelo <lgelo@cnc.sk>
 * Copyright (c) 2003-2004, Jouni Malinen <jkmaline@cc.hut.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * Alternatively, this software may be distributed under the terms of BSD
 * license.
 *
 * See README and COPYING for more details.
 */

#ifdef LINUX_FRAMER

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/if.h>
#include <linux/wireless.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <stdint.h>

typedef uint64_t u64;
typedef uint32_t u32;
typedef uint16_t u16;
typedef uint8_t u8;
typedef int64_t s64;
typedef int32_t s32;
typedef int8_t s8;

#include "xsup_debug.h"
#include "xsup_err.h"
#include "config.h"
#include "profile.h"
#include "cardif/cardif.h"
#include "cardif/linux/cardif_linux_wext.h"
#include "cardif/linux/cardif_ipw_driver.h"
#include "wpa.h"
#include "wpa2.h"
#include "config_ssid.h"

/* following definitions must be kept in sync with ipw_main.c */

#define IPW_IOCTL_WPA_SUPPLICANT		SIOCIWFIRSTPRIV+30

#define IPW_CMD_SET_WPA_PARAM		1
#define	IPW_CMD_SET_WPA_IE		2
#define IPW_CMD_SET_ENCRYPTION		3
#define IPW_CMD_MLME			4

#define IPW_PARAM_WPA_ENABLED		1
#define IPW_PARAM_TKIP_COUNTERMEASURES	2
#define IPW_PARAM_DROP_UNENCRYPTED	3
#define IPW_PARAM_PRIVACY_INVOKED	4
#define IPW_PARAM_AUTH_ALGS		5
#define IPW_PARAM_IEEE_802_1X		6

#define IPW_MLME_STA_DEAUTH		1
#define IPW_MLME_STA_DISASSOC		2

#define IPW_CRYPT_ERR_UNKNOWN_ALG	2
#define IPW_CRYPT_ERR_UNKNOWN_ADDR	3
#define IPW_CRYPT_ERR_CRYPT_INIT_FAILED	4
#define IPW_CRYPT_ERR_KEY_SET_FAILED	5
#define IPW_CRYPT_ERR_TX_KEY_SET_FAILED	6
#define IPW_CRYPT_ERR_CARD_CONF_FAILED	7

#define	IPW_CRYPT_ALG_NAME_LEN		16

struct ipw_param {
	u32 cmd;
	u8 sta_addr[6];
        union {
		struct {
			u8 name;
			u32 value;
		} wpa_param;
		struct {
			u32 len;
			u8 *data;
		} wpa_ie;
	        struct{
			int command;
    			int reason_code;
		} mlme;
		struct {
			u8 alg[IPW_CRYPT_ALG_NAME_LEN];
			u8 set_tx;
			u32 err;
			u8 idx;
			u8 seq[8];
			u16 key_len;
			u8 key[0];
		} crypt;

	} u;
};

/* end of ipw_main.c code */

// Take verbatim from the ipw driver patch for wpa_supplicant.
int ipw_ioctl(const char *dev, struct ipw_param *param,
		  int len, int show_err)
{
	int s;
	struct iwreq iwr;

	s = socket(PF_INET, SOCK_DGRAM, 0);
	if (s < 0) {
		debug_printf(DEBUG_NORMAL, "Error getting socket! (%s:%d)\n",
			     __FUNCTION__, __LINE__);
		return -1;
	}

	memset(&iwr, 0, sizeof(iwr));
	strncpy(iwr.ifr_name, dev, IFNAMSIZ);
	iwr.u.data.pointer = (caddr_t) param;
	iwr.u.data.length = len;

	if (ioctl(s, IPW_IOCTL_WPA_SUPPLICANT, &iwr) != 0) {
		int ret;
		close(s);
		ret = errno;
		if (show_err) 
		  {
			debug_printf(DEBUG_NORMAL, "Error setting WPA value!\n");
			debug_printf(DEBUG_NORMAL, "Error : %s\n", strerror(ret));
		  }
		return ret;
	}
	close(s);

	return 0;
}

// Taken verbatim from the ipw driver patch to wpa_supplicant.
int ipw_set_wpa_param(const char *ifname, u8 name, u32 value)
{
	struct ipw_param param;

	memset(&param, 0, sizeof(param));
	param.cmd = IPW_CMD_SET_WPA_PARAM;
	param.u.wpa_param.name = name;
	param.u.wpa_param.value = value;

	return ipw_ioctl(ifname, &param, sizeof(param), 1);
}

// Taken verbatim from the ipw driver patch to wpa_supplicant.
void ipw_show_set_key_error(struct ipw_param *param)
{
	switch (param->u.crypt.err) {
	case IPW_CRYPT_ERR_UNKNOWN_ALG:
		debug_printf(DEBUG_NORMAL, "Unknown algorithm '%s'.",
			     param->u.crypt.alg);
		debug_printf(DEBUG_NORMAL, "You may need to load kernel module to "
			     "register that algorithm.");
		debug_printf(DEBUG_NORMAL, "E.g., 'modprobe ieee80211_crypt_wep' for "
			   "WEP.");
		break;
	case IPW_CRYPT_ERR_UNKNOWN_ADDR:
	  debug_printf(DEBUG_NORMAL, "Unknown address ");
	  debug_hex_printf(DEBUG_NORMAL, param->sta_addr, 6);
	  debug_printf(DEBUG_NORMAL, "\n");
		break;
	case IPW_CRYPT_ERR_CRYPT_INIT_FAILED:
		debug_printf(DEBUG_NORMAL, "Crypt algorithm initialization failed.");
		break;
	case IPW_CRYPT_ERR_KEY_SET_FAILED:
		debug_printf(DEBUG_NORMAL, "Key setting failed.");
		break;
	case IPW_CRYPT_ERR_TX_KEY_SET_FAILED:
		debug_printf(DEBUG_NORMAL, "TX key index setting failed.");
		break;
	case IPW_CRYPT_ERR_CARD_CONF_FAILED:
		debug_printf(DEBUG_NORMAL, "Card configuration failed.");
		break;
	}
}

// Taken from the ipw patch to wpa_supplicant.
int ipw_mlme(const char *ifname, u8 *addr, int cmd, int reason)
{
	struct ipw_param param;

	memset(&param, 0, sizeof(param));
	memcpy(param.sta_addr, addr, 6);	
	param.cmd = IPW_CMD_MLME;
	param.u.mlme.command = cmd;
	param.u.mlme.reason_code = reason;

	return ipw_ioctl(ifname, &param, sizeof(param), 1);
}

int cardif_ipw_driver_get_wpa_ie(struct interface_data *intdata,
				     char *iedata, int *ielen)
{
  wpa_gen_ie(intdata, iedata);
  *ielen = 24;
  return XENONE;
}

int cardif_ipw_driver_get_wpa2_ie(struct interface_data *intdata,
				      char *iedata, int *ielen)
{
  wpa2_gen_ie(intdata, iedata, ielen);

  debug_printf(DEBUG_INT, "Setting WPA2 IE : ");
  debug_hex_printf(DEBUG_INT, iedata, *ielen);
  debug_printf(DEBUG_INT, "\n");

  return XENONE;
}


// Derived from WPA supplicant
int cardif_ipw_driver_set_wpa_ie(char *dev, char *wpa_ie, int wpa_ie_len)
{
  struct ipw_param *param;
  int ret;
  size_t blen = sizeof(*param) + wpa_ie_len;

  if ((wpa_ie_len < 0) || (wpa_ie_len > 255))
    {
      debug_printf(DEBUG_NORMAL, "Invalid WPA IE length of %d!  WPA will not"
		   " work!\n");
      return -1;
    }
  
  param = (struct ipw_param *) malloc(blen);
  if (param == NULL)
    return -1;
  
  memset(param, 0, blen);
  param->cmd = IPW_CMD_SET_WPA_IE;
  param->u.wpa_ie.len = wpa_ie_len;
  
  if (wpa_ie_len){
    param->u.wpa_ie.data = (u8 *)(param+1);
    memcpy(param->u.wpa_ie.data, wpa_ie, wpa_ie_len);
  } else {
    param->u.wpa_ie.data = NULL;
  }
  
  ret = ipw_ioctl(dev, param, blen, 1);
  
  free(param);
  return XENONE;
}

int cardif_ipw_driver_wpa(struct interface_data *intdata,  char endis)
{
  int retval = XENONE;

  if (!endis && cardif_ipw_driver_set_wpa_ie(intdata->intName, NULL, 0) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't set WPA IE!\n");
      retval = -1;
    }

  if (ipw_set_wpa_param(intdata->intName, IPW_PARAM_WPA_ENABLED, endis)
      != 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't enable WPA!\n");
      retval = -1;
    }

  return retval;
}

int cardif_ipw_driver_wpa_state(struct interface_data *thisint,  char endis)
{
  char wpaie[24];
  int retval = XENONE, len;

  cardif_ipw_driver_get_wpa_ie(thisint, (char *)&wpaie, &len);

  if (cardif_ipw_driver_set_wpa_ie(thisint->intName, wpaie, len) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't set WPA IE! On device %s!\n",
		   thisint->intName);
      retval = -1;
    }

  if (ipw_set_wpa_param(thisint->intName, IPW_PARAM_WPA_ENABLED, endis)
      != 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't enable WPA on device %s!\n",
		   thisint->intName);
      retval = -1;
    }

  return XENONE;
}

int cardif_ipw_driver_wpa_disable(char *intname)
{
  // XXX Finish!
  return XENONE;
}

int cardif_ipw_driver_set_key(char *dev, int alg, unsigned char *addr,
			      int key_idx, int set_tx, char *seq, int seq_len,
			      char *key, int key_len)
{
  struct ipw_param *param;
  u8 *buf;
  size_t blen;
  int ret = 0;
  char *alg_name;
  
  switch (alg) {
  case WPA_NONE:
    alg_name = "none";
    break;
  case WPA_WEP:
    alg_name = "WEP";
    break;
  case WPA_TKIP:
    alg_name = "TKIP";
    break;
  case WPA_CCMP:
    alg_name = "CCMP";
    break;
  default:
    return -1;
  }
  
  if (seq_len > 8)
    return -2;
  
  blen = sizeof(*param) + key_len;
  buf = malloc(blen);
  if (buf == NULL)
    return -1;
  memset(buf, 0, blen);
  
  param = (struct ipw_param *) buf;
  param->cmd = IPW_CMD_SET_ENCRYPTION;
  memset(param->sta_addr, 0xff, 6);
  strncpy(param->u.crypt.alg, alg_name, IPW_CRYPT_ALG_NAME_LEN);
  param->u.crypt.set_tx = set_tx ? 1 : 0;
  param->u.crypt.idx = key_idx;
  memcpy(param->u.crypt.seq, seq, seq_len);
  param->u.crypt.key_len = key_len;
  memcpy((u8 *) (param + 1), key, key_len);
  
  if (ipw_ioctl(dev, param, blen, 1)) {
    debug_printf(DEBUG_NORMAL, "Failed to set encryption.\n");
    ipw_show_set_key_error(param);
    ret = -1;
  }
  free(buf);

  return ret;
}

int cardif_ipw_driver_set_tkip(struct interface_data *intdata, 
				  unsigned char *addr, int key_idx, int set_tx,
				  char *seq, int seq_len, char *key, 
				  int key_len)
{
  return cardif_ipw_driver_set_key(intdata->intName, WPA_TKIP, addr, 
				      key_idx, set_tx, seq, seq_len, key, 
				      key_len);
}

int cardif_ipw_driver_set_ccmp(struct interface_data *intdata,
				   unsigned char *addr, int key_idx, 
				   int set_tx, char *seq, int seq_len,
				   char *key, int key_len)
{
  return cardif_ipw_driver_set_key(intdata->intName, WPA_CCMP, addr,
				       key_idx, set_tx, seq, seq_len, key,
				       key_len);
}

int cardif_ipw_driver_delete_key(struct interface_data *intdata, 
				     int key_idx, int settx)
{
  return cardif_ipw_driver_set_key(intdata->intName, WPA_NONE, NULL, 
				       key_idx, settx, NULL, 0, NULL, 0);
}

// Derived from WPA supplicant code.
int cardif_ipw_driver_disassociate(struct interface_data *intdata,
				      int reason_code)
{
  int ret;

  ret = ipw_mlme(intdata->intName, intdata->source_mac, 
		     IPW_MLME_STA_DISASSOC, reason_code);

  if (ret <0) return ret;

  return XENONE;
}

int cardif_ipw_driver_countermeasures(struct interface_data *intdata,
					  char endis)
{
  return ipw_set_wpa_param(intdata->intName, IPW_PARAM_TKIP_COUNTERMEASURES, endis);
}

int cardif_ipw_driver_drop_unencrypted(struct interface_data *intdata,
					   char endis)
{
  return ipw_set_wpa_param(intdata->intName, IPW_PARAM_DROP_UNENCRYPTED, endis);
}

void cardif_ipw_driver_associate(struct interface_data *intdata, 
				    char *newessid)
{
  int len;
  u_char wpaie[100];

  // Association for the IPW doesn't require anything too fancy.  We can
  // assume that the keys have all been cleared out before we get here, so
  // we should only need to set the SSID, and tweak any encryption settings.

  intdata->flags |= DONTSCAN;

  config_ssid_dump();

  if (config_ssid_get_ssid_abilities() & RSN_IE)
    {
      cardif_ipw_driver_get_wpa2_ie(intdata, wpaie, &len);
    } else if (config_ssid_get_ssid_abilities() & WPA_IE)
      {
	cardif_ipw_driver_get_wpa_ie(intdata, wpaie, &len);
      }

  // Length needs to be > 0 in order to indicate that we have an IE to set.
  if ((len < 255) && (len >= 0))
    {
      if (cardif_ipw_driver_set_wpa_ie(intdata->intName, wpaie, len) < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't set WPA/RSN IE on device %s!\n",
		       intdata->intName);
	}
    }

  cardif_linux_wext_set_ssid(intdata, newessid);
  cardif_linux_wext_enc_open(intdata);
}


struct cardif_funcs cardif_ipw_driver = {
  .scan = cardif_linux_wext_scan,
  .disassociate = cardif_ipw_driver_disassociate,
  .set_wep_key = cardif_linux_wext_set_WEP_key,
  .set_tkip_key = cardif_ipw_driver_set_tkip,
  .set_ccmp_key = cardif_ipw_driver_set_ccmp,
  .delete_key = cardif_ipw_driver_delete_key,
  .set_ssid = cardif_linux_wext_set_ssid,
  .associate = cardif_ipw_driver_associate,
  .get_ssid = cardif_linux_wext_get_ssid,
  .get_bssid = cardif_linux_wext_get_bssid,
  .wpa_state = cardif_ipw_driver_wpa_state,
  .wpa = cardif_ipw_driver_wpa,
  .roam = cardif_linux_wext_roam,
  .countermeasures = cardif_ipw_driver_countermeasures,
  .drop_unencrypted = cardif_ipw_driver_drop_unencrypted,
  .get_wpa_ie = cardif_ipw_driver_get_wpa_ie,
  .get_wpa2_ie = cardif_ipw_driver_get_wpa2_ie,
  .enc_disable = cardif_linux_wext_enc_disable
};

#endif
