%{
/* rec_lex.l -- Lexical anaylzer for xstroke recognition control file

   Copyright 2001 Carl Worth

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "rec.h"
#include "rec_mode.h"
#include "gesture.h"
#include "action.h"
#include "feature.h"
#include "option.h"

#include "rec_parse.h"
#include "rec_lex.h"
#include "sprintf_alloc.h"

struct include_stack
{
    char *filename;
    FILE *file;
    int line_num;
    struct include_stack *prev;
    YY_BUFFER_STATE yy_buffer;
};
typedef struct include_stack include_stack_t;

/* XXX: static data is so bad.... */
static include_stack_t *include_stack;
static int newlines_wanted = 0;

#define YY_DECL int yylex(YYSTYPE *lvalp)

static int escaped_char(int c);
static char *escaped_string_alloc(char *yytext, int yyleng);

static int include_stack_init(include_stack_t *data, char *filename);
static void include_stack_deinit(include_stack_t *data);
static void push_include(include_stack_t **stack, char *filename);
static int pop_include(include_stack_t **stack);

%}

%option noyywrap nounput
%x INCLUDE
                        
%%

^[ \t]*#.*			;

"include"		BEGIN(INCLUDE);

"Key"			{ lvalp->ival = KEY; return KEY; }
"Button"		{ lvalp->ival = BUTTON; return BUTTON; }
"ModeShift"		{ lvalp->ival = MODE_SHIFT; return MODE_SHIFT; }
"ModeLock"		{ lvalp->ival = MODE_LOCK; return MODE_LOCK; }
"Exec"			{ lvalp->ival = EXEC; return EXEC; }
"OrientationCorrection"	{
			  lvalp->ival = ORIENTATION_CORRECTION;
			  return ORIENTATION_CORRECTION;
			}
"Mode"			{ lvalp->ival = MODE; return MODE; }
"Engine"		{ lvalp->ival = ENGINE; return ENGINE; }
"Option"		{ lvalp->ival = OPTION; return OPTION; }

\"([^\n\"]|\\\")*\"	{
			  lvalp->sval = escaped_string_alloc(yytext, yyleng);
			  return STRING;
			}
[a-zA-Z][a-zA-Z0-9_]*	{
			  lvalp->sval = strdup(yytext);
			  return IDENTIFIER;
			}
-?[0-9]+		{ lvalp->ival = atoi(yytext); return INTEGER; }
-?[0-9]+\.[0-9]*	{ lvalp->dval = atof(yytext); return DOUBLE; }

"("			{ lvalp->ival = '('; return '('; }
")"			{ lvalp->ival = ')'; return ')'; }
"{"			{ lvalp->ival = '{'; return '{'; }
"}"			{ lvalp->ival = '}'; return '}'; }
"["			{ lvalp->ival = '['; return '['; }
"]"			{ lvalp->ival = ']'; return ']'; }
"="			{ lvalp->ival = '='; return '='; }
":"			{ lvalp->ival = ':'; return ':'; }
";"			{ lvalp->ival = ';'; return ';'; }
","			{ lvalp->ival = ','; return ','; }
" "			;
"\t"			;
"\n"			{
			  include_stack->line_num++;
			  if (newlines_wanted) {
			      lvalp->ival = '\n'; return '\n';
			  }
			}

<INCLUDE>{
  [[:space:]]*		;
  \"([^\n\"]|\\\")*\"	{
      			  char *str = escaped_string_alloc(yytext, yyleng);
  			  push_include(&include_stack, str);
			  free(str);
			  BEGIN(INITIAL);
			}
  [^[:space:]]+		{
			  push_include(&include_stack, yytext);
			  BEGIN(INITIAL);
			}
}

<<EOF>>			{
			  int done = pop_include(&include_stack);
			  if (done) yyterminate();
			}

.			{
			  fprintf(stderr, "Syntax error: Unexpected character "
					  "in input stream '%s'\n",
					  yytext);
			  lvalp->sval = strdup(yytext);
			  return UNKNOWN_CHARACTER;
			}

%%

int rec_lex_initial_file(char *filename)
{
    int err;

    include_stack = malloc(sizeof(include_stack_t));
    if (include_stack == NULL) {
	fprintf(stderr, "%s: out of memory\n", __FUNCTION__);
	return ENOMEM;
    }

    err = include_stack_init(include_stack, filename);
    if (err) {
	return err;
    }
    yyin = include_stack->file;
    yy_switch_to_buffer(include_stack->yy_buffer);

    return 0;
}

char *rec_lex_location_alloc(void)
{
    char *loc = NULL;
    if (include_stack == NULL) {
	return strdup("<EOF>");
    } 
    sprintf_alloc(&loc, "%s:%d", include_stack->filename, include_stack->line_num);
    return loc;
}

void rec_lex_newlines_wanted(int wanted)
{
    newlines_wanted = wanted;
}

static int include_stack_init(include_stack_t *stack, char *filename)
{
    stack->filename = NULL;

    if (filename == NULL || filename[0] == '\0') {
	fprintf(stderr, "%s: Cannot open NULL filename\n", __FUNCTION__);
	return EINVAL;
    }

    if (filename[0] == '/' || include_stack->filename == NULL) {
	stack->filename = strdup(filename);
    } else {
	int dirlen = strrchr(include_stack->filename, '/') - include_stack->filename;
	stack->filename = malloc(dirlen + 1 + strlen(filename) + 1);
	if (stack->filename == NULL) {
	    fprintf(stderr, "%s: out of memory\n", __FUNCTION__);
	    return ENOMEM;
	}
	strncpy(stack->filename, include_stack->filename, dirlen);
	strcat(stack->filename, "/");
	strcat(stack->filename, filename);
    }

    stack->line_num = 1;
    stack->yy_buffer = NULL;
    stack->prev = NULL;

    stack->file = fopen(stack->filename, "r");
    if (stack->file == NULL) {
	fprintf(stderr, "%s: failed to open file %s: %s\n",
		__FUNCTION__, stack->filename, strerror(errno));
	return EINVAL;
    }
    stack->yy_buffer = yy_create_buffer(stack->file, YY_BUF_SIZE);

    return 0;
}

static void include_stack_deinit(include_stack_t *stack)
{
    yy_delete_buffer(stack->yy_buffer);
    stack->yy_buffer = NULL;
    fclose(stack->file);
    stack->file = NULL;
    stack->prev = NULL;
    stack->line_num = 0;
    free(stack->filename);
}

static void push_include(include_stack_t **stack, char *filename)
{
    int err;
    include_stack_t *new_top;

    new_top = malloc(sizeof(include_stack_t));
    if (new_top == NULL) {
	fprintf(stderr, "%s: out of memory\n", __FUNCTION__);
	return;
    }

    err = include_stack_init(new_top, filename);
    if (err) {
	free(new_top);
	return;
    }
    yyin = new_top->file;
    yy_switch_to_buffer(new_top->yy_buffer);

    new_top->prev = *stack;
    *stack = new_top;
}

static int pop_include(include_stack_t **stack)
{
    include_stack_t *current = *stack;
    *stack = (*stack)->prev;

    include_stack_deinit(current);
    free(current);

    if (*stack == NULL) {
	return 1;
    } else {
	yyin=(*stack)->file;
	yy_switch_to_buffer((*stack)->yy_buffer);
	return 0;
    }
}

static int escaped_char(int c)
{
    switch (c) {
    case '0':
	return '\0';
    case 'b':
	return '\b';
	break;
    case 'n':
	return '\n';
	break;
    case 'r':
	return '\r';
	break;
    case 't':
	return '\t';
	break;
    case 'f':
	return '\f';
	break;
    default:
	return c;
    }
}

static char *escaped_string_alloc(char *yytext, int yyleng)
{
    char *c, *s;
    char *new_str;

    yytext[yyleng - 1] = '\0';
    new_str = malloc(yyleng-1);
    if (new_str == NULL) {
	fprintf(stderr, "%s: out of memory\n", __FUNCTION__);
	return NULL;
    }
    c = new_str;
    s = yytext + 1;
    while (*s) {
	if (*s == '\\')
	    *c++ = escaped_char(*++s);
	else
	    *c++ = *s;
	++s;
    }
    *c = '\0';

    return new_str;
}

void rec_lex_set_yyin(FILE *file)
{
    yyin = file;
}
