// file      : xsde/cxx/parser/type-processor.cxx
// author    : Boris Kolpackov <boris@codesynthesis.com>
// copyright : Copyright (c) 2006-2007 Code Synthesis Tools CC
// license   : GNU GPL v2 + exceptions; see accompanying LICENSE file

#include <boost/regex.hpp>

#include <cult/containers/set.hxx>

#include <cxx/parser/elements.hxx>
#include <cxx/parser/type-processor.hxx>

#include <xsd-frontend/semantic-graph.hxx>
#include <xsd-frontend/traversal.hxx>

namespace CXX
{
  namespace Parser
  {
    namespace
    {
      //
      //
      struct Type: Traversal::Type
      {
        Type (SemanticGraph::Schema& schema, TypeMap::Namespaces& type_map)
            : schema_ (schema), type_map_ (type_map)
        {
        }

        virtual Void
        traverse (SemanticGraph::Type& type)
        {

          using TypeMap::Namespace;
          using TypeMap::Namespaces;

          if (type.context ().count ("ret-type"))
            return;

          SemanticGraph::Namespace& ns (
            dynamic_cast<SemanticGraph::Namespace&> (type.scope ()));

          String ns_name (ns.name ());
          String t_name (type.name ());

          // std::wcerr << "traversing: " << ns_name << "#" << t_name << endl;

          for (Namespaces::ConstIterator n (type_map_.begin ());
               n != type_map_.end (); ++n)
          {
            // Check if the namespace matches.
            //
            //@@ Should probably store precompiled regex somewhere
            //   instead of doing it every time.
            //
            boost::basic_regex<WideChar> ns_expr;

            Boolean ns_match;

            if (n->xsd_name ())
            {
              ns_expr = n->xsd_name ();
              ns_match = regex_match (ns_name, ns_expr, boost::match_default);
            }
            else
              ns_match = ns_name.empty ();

            // std::wcerr << "considering ns expr: " << n->xsd_name () << endl;

            if (ns_match)
            {
              // Namespace matched. See if there is a type that matches.
              //
              for (Namespace::TypesIterator t (n->types_begin ());
                   t != n->types_end (); ++t)
              {
                boost::basic_regex<WideChar> t_expr (t->xsd_name ());

                if (regex_match (t_name, t_expr, boost::match_default))
                {
                  // Got a match. See if the namespace has the C++
                  // namespace mapping.
                  //

                  String cxx_ns;

                  if (n->has_cxx_name ())
                  {
                    if (n->xsd_name ())
                    {
                      cxx_ns = regex_merge (
                        ns_name,
                        ns_expr,
                        n->cxx_name (),
                        boost::match_default
                        | boost::format_all
                        | boost::format_first_only);
                    }
                    else
                      cxx_ns = n->cxx_name ();

                    cxx_ns += L"::";
                  }

                  // Figure out ret and arg type names.
                  //
                  String ret_type (cxx_ns);

                  ret_type += regex_merge (
                    t_name,
                    t_expr,
                    t->cxx_ret_name (),
                    boost::match_default
                    | boost::format_all
                    | boost::format_first_only);

                  String arg_type;

                  if (t->cxx_arg_name ())
                  {
                    arg_type = cxx_ns;
                    arg_type += regex_merge (
                      t_name,
                      t_expr,
                      t->cxx_arg_name (),
                      boost::match_default
                      | boost::format_all
                      | boost::format_first_only);
                  }
                  else
                  {
                    if (ret_type == L"void")
                      arg_type = ret_type;
                    else
                      arg_type = L"const " + ret_type + L"&";
                  }

                  type.context ().set ("ret-type", ret_type);
                  type.context ().set ("arg-type", arg_type);

                  //std::wcerr << t_name << " -> " << ret_type << endl;

                  // See of we need to add any includes to the translations
                  // unit.
                  //
                  if (n->includes_begin () != n->includes_end ())
                  {
                    typedef Cult::Containers::Set<String> Includes;

                    if (!schema_.context ().count ("includes"))
                      schema_.context ().set ("includes", Includes ());

                    Includes& is (
                      schema_.context ().get<Includes> ("includes"));

                    for (Namespace::IncludesIterator i (n->includes_begin ());
                         i != n->includes_end (); ++i)
                    {
                      is.insert (*i);
                    }
                  }

                  return;
                }
              }
            }
          }
        }

      private:
        SemanticGraph::Schema& schema_;
        TypeMap::Namespaces& type_map_;
      };


      //
      //
      struct GlobalType: Traversal::Type,
                         Traversal::List,
                         Traversal::Complex,
                         Traversal::Enumeration
      {
        GlobalType (SemanticGraph::Schema& schema,
                    TypeMap::Namespaces& type_map)
            : type_ (schema, type_map)
        {
          inherits_ >> type_;
          names_ >> instance_ >> belongs_ >> type_;
          argumented_ >> type_;
        }

        virtual Void
        traverse (SemanticGraph::Type& t)
        {
          type_.traverse (t);
        }

        virtual Void
        traverse (SemanticGraph::List& l)
        {
          type_.traverse (l);
          Traversal::List::argumented (l, argumented_);
        }

        virtual Void
        traverse (SemanticGraph::Complex& c)
        {
          type_.traverse (c);
          Complex::inherits (c, inherits_);
          Complex::names (c, names_);
        }

        virtual Void
        traverse (SemanticGraph::Enumeration& e)
        {
          type_.traverse (e);
          Complex::inherits (e, inherits_);
        }

      private:
        Parser::Type type_;
        Traversal::Names names_;
        Traversal::Instance instance_;
        Traversal::Inherits inherits_;
        Traversal::Belongs belongs_;
        Traversal::Argumented argumented_;
      };

      Void
      process_impl (XSDFrontend::SemanticGraph::Schema& tu,
                    TypeMap::Namespaces& type_map)
      {
        Traversal::Schema schema;
        Traversal::Sources sources;
        Traversal::Implies implies;

        schema >> sources >> schema;
        schema >> implies >> schema;

        Traversal::Names schema_names;
        Traversal::Namespace ns;
        Traversal::Names ns_names;

        schema >> schema_names >> ns >> ns_names;

        GlobalType global_type (tu, type_map);

        ns_names >> global_type;

        schema.dispatch (tu);
      }
    }

    Void TypeProcessor::
    process (XSDFrontend::SemanticGraph::Schema& s, TypeMap::Namespaces& tm)
    {
      process_impl (s, tm);
    }
  }
}
