/* -*-mode: C; style: K&R; c-basic-offset: 4 ; -*- */

/*
 *  xreverse.c - Implementation.
 *
 *  XReverse  - A simple reversi type game.
 *
 * Homepage:
 *   http://www.steve.org.uk/Software/
 *
 * Author:
 *  Steve Kemp <steve@steve.org.uk>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  Steve Kemp
 *  ---
 *
 */



#include <X11/Xlib.h>
#include <X11/Xutil.h>

/*
 * These two lines define the keyboard codes which we use for
 * 'n', 'q', 'u', etc.
 */
#define XK_LATIN1
#include <X11/keysymdef.h>

#include <pwd.h>
#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "xreverse.h"



/**
 * Global variables.
 **/

Display* g_display;		/* pointer to X Display structure.           */
Window   g_win;			/* pointer to the main window.      */
GC       g_gc;			/* GC (graphics context) used for drawing    */


unsigned int g_win_width;	/* height and width for the new window.      */
unsigned int g_win_height;




/*
 * Create our main window.
 *
 */
void createMainWindow()
{
  int screen_num = DefaultScreen( g_display );
  int win_border_width = 2;

  /* Used for setting the window title. */
  XTextProperty title;
  char* titleText = "XReverse v" VERSION;
  Status status;

  /* create a simple window, as a direct child of the screen's */
  /* root window. Use the screen's black and white colors as   */
  /* the foreground and background colors of the window,       */
  /* respectively. Place the new window's top-left corner at   */
  /* the given 'x,y' coordinates.                              */
  g_win = XCreateSimpleWindow(g_display, RootWindow(g_display, screen_num),
                            0, 0, g_win_width, g_win_height, win_border_width,
                            BlackPixel(g_display, screen_num),
                            WhitePixel(g_display, screen_num));


  status = XStringListToTextProperty(&titleText, 1, &title);
  XSetWMName(g_display, g_win, &title);

  /* make the window actually appear on the screen. */
  XMapWindow( g_display, g_win);

  /* flush all pending requests to the X server. */
  XFlush( g_display);
}

GC
create_gc(Display* display, Window win, int reverse_video)
{
  GC gc;				/* handle of newly created GC.  */
  unsigned long valuemask = 0;		/* which values in 'values' to  */
					/* check when creating the GC.  */
  XGCValues values;			/* initial values for the GC.   */
  unsigned int line_width = 2;		/* line width for the GC.       */
  int line_style = LineSolid;		/* style for lines drawing and  */
  int cap_style = CapButt;		/* style of the line's edje and */
  int join_style = JoinBevel;		/*  joined lines.		*/
  int screen_num = DefaultScreen(display);

  gc = XCreateGC(display, win, valuemask, &values);
  if (gc < 0) {
	fprintf(stderr, "XCreateGC: \n");
  }

  /* allocate foreground and background colors for this GC. */
  if (reverse_video) {
    XSetForeground(display, gc, WhitePixel(display, screen_num));
    XSetBackground(display, gc, BlackPixel(display, screen_num));
  }
  else {
    XSetForeground(display, gc, BlackPixel(display, screen_num));
    XSetBackground(display, gc, WhitePixel(display, screen_num));
  }

  /* define the style of lines that will be drawn using this GC. */
  XSetLineAttributes(display, gc,
                     line_width, line_style, cap_style, join_style);

  /* define the fill style for the GC. to be 'solid filling'. */
  XSetFillStyle(display, gc, FillSolid);

  return gc;
}


/*
 * Initialize the board, this means setting all squares to be empty,
 * and setting up the initial four pieces.
 */
void initBoard()
{
    int x, y;
    for( y = 0; y < 8; y++ )
    {
	for(  x = 0; x < 8; x++ )
	{
	    g_board[x][y] = EMPTY;
	}
    }

    /* White hasn't moved yet. */
    white_x_last = -1;
    white_y_last = -1;

    /* Initial starting places. */
    g_board[3][3] = BLACK;
    g_board[3][4] = WHITE;
    g_board[4][3] = WHITE;
    g_board[4][4] = BLACK;

}


/*
 * Count the number of tiles of a given colour on the board.
 */
int countColour( int color )
{
    int x, y;
    int count = 0;

    for( y = 0; y < 8; y++ )
    {
	for(  x = 0; x < 8; x++ )
	{
	    if ( g_board[x][y] == color )
	    {
		count++;
	    }
	}
    }

    return count;
}



/*
 * Return 1 if the game is over now.
 */
int isGameOver()
{
    int empty = countColour( EMPTY );
    int black = countColour( BLACK );
    int white = countColour( WHITE );

    /*
     * If there are no empty squares nobody can move = game over.
     * if there are no white squares black has won, so game over.
     * ditto for black.
     */
    if  ( ( empty == 0 ) ||
	  ( black == 0 ) ||
	  ( white == 0 ) )
    {
	return 1;
    }

    /* Game still in progress. */
    return 0;
}


/*
 * Show the end game text.
 */
void gameOver()
{
    int white = countWhite();
    int black = countBlack();
    
    if ( white == black )
    {
	setStatus( "Game Over : Draw" );
    }
    else if ( white < black )
    {
	setStatus( "Game Over : Black Wins" );
    }
    else if ( white > black )
    {
	setStatus("Game Over : White Wins" );
    }
}


/*
 * Count, and return, the number of black tiles on the board.
 */
int countBlack()
{
    return( countColour( BLACK ) );
}


/*
 * Count, and return, the number of white tiles on the board.
 */
int countWhite()
{
    return( countColour( WHITE ) );
}

/*
 * Count, and return the number of empty tiles.
 */
int countEmpty()
{
    return( countColour( EMPTY ) );
}


/*
 * Draw the board.
 *
 * First we draw the gridlines, then erase the center of each
 * grid.
 *
 * Finally we fill the board with tiles representing the occupied
 * places.
 */
void drawBoard()
{
  int i = 0;
  int x, y;
  int step = g_win_width / 8;

  int screen_num = DefaultScreen(g_display);

  /* Draw the grid lines for the board. */
  for( i = 0; i <= g_win_width; i += step )
  {
    XDrawLine( g_display, g_win, g_gc, i, 0, i, g_win_width );
    XDrawLine( g_display, g_win, g_gc, 0, i, g_win_width, i );
  }


  /* Draw the pieces - small circles, centered in each box.
   * Black pieces are filled circles, white ones are just circles.
   */
  for( y = 0; y < 8; y++ )
  {
    for( x = 0; x < 8; x++ )
    {
      int state = g_board[x][y];

      int xoff = ( (x) * step ) + (step/4);
      int yoff = ( (y) * step ) + (step/4);

      /*
       * first blank the square, we have to do this because otherwise
       * a piece which had previously been black and had been taken
       * by white wouldn'bt be updated properly.
       */
      XSetForeground( g_display, g_gc, WhitePixel(g_display, screen_num));
      XFillRectangle( g_display, g_win, g_gc, (x*step)+1, (y*step)+1, 
		      step-2,step-2);
      XSetForeground( g_display, g_gc, BlackPixel( g_display, screen_num));

      if ( state == BLACK )
      {
	  /* Solid circle */
	  XFillArc( g_display, g_win, g_gc, xoff , yoff,
		    (step/2), (step/2), 0, 360*64);
      }
      else if ( state == WHITE )
      {
	  /* Circle - not filled. */
	  XDrawArc( g_display, g_win, g_gc, xoff , yoff,
		   (step/2), (step/2), 0, 360*64);
      }
    }
  }

  /*
   * Draw the status text - after first erasing that area of the
   * screen, to prevent overwriting problems.
   */
  XSetForeground( g_display, g_gc, WhitePixel( g_display, screen_num));
  XFillRectangle( g_display, g_win, g_gc, 0, g_win_width+1,
		  g_win_width+step, g_win_width);

  XSetForeground( g_display, g_gc, BlackPixel( g_display, screen_num));
  XDrawString( g_display, g_win, g_gc, 10, g_win_width + (step/2),
	       g_statusMsg, strlen( g_statusMsg ) );


  /*
   * If the white piece has made at least one move draw that move
   * differently.
   */
  if ( ( white_x_last != -1 ) &&
       ( white_y_last != -1 ) )
  {
      int lastx = ( (white_x_last) * step ) + (step/4);
      int lasty = ( (white_y_last) * step ) + (step/4);

      /* Circle - not filled. */
      XDrawArc( g_display, g_win, g_gc, lastx , lasty,
		(step/2), (step/2), 0, 360*64);
      /* Solid circle */
      XFillArc( g_display, g_win, g_gc, lastx+(step/8), lasty+(step/8),
		    (step/4), (step/4), 0, 360*64);

  }

  /*
   * Flush all pending requests to the X server.
   */
  XFlush( g_display);

}


/*
 *  Draw small circles in the squares which the given player
 * could move.
 */
int drawHints( int player, int justTest )
{
    int x, y;
    int step   = g_win_width / 8;

    /* Number of valid moves black could make. */
    int nValid = 0;

    drawBoard();

    /* Draw the pieces - small circles, centered in each box.
     * Black pieces are filled circles, white ones are just circles.
     */
    for( y = 0; y < 8; y++ )
    {
	for( x = 0; x < 8; x++ )
	{
	    int xoff = ( (x) * step ) + (step/4);
	    int yoff = ( (y) * step ) + (step/4);

	    if ( legal_move( player, x, y ) )
	    {
		if ( justTest != 0 )
		{
		    /* Solid circle */
		    XFillArc( g_display, g_win, g_gc, xoff , yoff,
			      (step/4), (step/4), 0, 360*64);
		}

		/* We've found another valid move for the player */
		nValid +=1;
	    }
	}
    }

    /*
     * Flush all pending requests to the X server.
     */
    XFlush( g_display);

    /* Return the number of valid moves which could be made. */
    return( nValid );
}


/*
 *  Set the text which is at the bottom of the screen, in the
 * 'status' area.
 *
 *  This also causes a redraw.
 */
void setStatus( char *text )
{
  memset( g_statusMsg, '\0', sizeof( g_statusMsg ) );
  snprintf( g_statusMsg, sizeof( g_statusMsg ) -1 ,
	    "%s  [ B:%d - W:%d ]", text, countBlack(), countWhite() );

  drawBoard( );
}


/*
 *  Check the number of pieces which would be captured in a given
 * direction.
 */
int check_direction(int player, int x, int y, int bx, int by)
{
    int c = 0;

    while( 1 )
    {
	/*  forward to next position */
	x+=bx;
	y+=by;
   
	/* error if out of bounds or empty */
	if((x<0) || (y<0) || (x>=8) || (y>=8) || ( g_board[x][y]==EMPTY)) 
	{
	    return 0;
	}

	/* if one more captured */
	if(player == WHITE) 
	{
	    if(g_board[x][y] == BLACK) 
	    {
		c++;
	    }
	}

	if(player == BLACK)
	{
	    if(g_board[x][y] == WHITE) 
	    {
		c++;
	    }
	}
	
	/* if finished counting */
	if(player == WHITE) 
	{
	    if(g_board[x][y] == WHITE) 
	    {
		return c;
	    }
	}

	if(player == BLACK)
	{
	    if(g_board[x][y] == BLACK) 
	    {
		return c;
	    }
	}
    }
}


/*
 *  Turn all the pieces over in the given direction.
 */
void turn_direction(int player, int x, int y, int bx, int by)
{
    /* 
     * Get the number of pieces that have been captured in
     * this direction.
     */
    int c = check_direction(player, x, y, bx, by);

    while( c &&
	   ( x >= 0 ) && ( x < 8 ) &&
	   ( y >= 0 ) && ( y < 8 ) )
    {
	/* forward to next position */
	x+=bx;
	y+=by;
	c--;

	/*
	 * Only flip the ones which aren't already our colour.
	 */
	if ( g_board[x][y] != player )
	{
	    g_board[x][y] = player ;
	}
    }
}


/*
 *  Turn all the captured bricks over, for the given
 * move.
 */
void turn_bricks(int player, int x, int y)
{
    /* northwest */
    if(check_direction(player, x, y, -1, -1))
	turn_direction(player, x, y, -1, -1);

    /* north */
    if(check_direction(player, x, y,  0, -1))
	turn_direction(player, x, y, 0, -1);

    /* northeast */
    if(check_direction(player, x, y, +1, -1))
	turn_direction(player, x, y, +1, -1);

    /* west */
    if(check_direction(player, x, y, -1,  0))
	turn_direction(player, x, y, -1, 0);

    /* east */
    if(check_direction(player, x, y, +1,  0))
	turn_direction(player, x, y, +1, 0);

    /* southwest */
    if(check_direction(player, x, y, -1, +1))
	turn_direction(player, x, y, -1, +1);

    /* south */
    if(check_direction(player, x, y,  0, +1))
	turn_direction(player, x, y, 0, +1);

    /* southeast */
    if(check_direction(player, x, y, +1, +1))
	turn_direction(player, x,y, +1, +1);

}


/*
 *  Actually perform a move.
 *
 *  This will flip all the pieces which have been captured.
 */
void make_move(int player, int x, int y)
{
    if ( player == BLACK )
    {
	int i, j;
	for( i = 0; i < 8; i++ )
	{
	    for( j = 0; j < 8; j++ )
	    {
		g_backupBoard[i][j] = g_board[i][j];
	    }
	}
    }

    /* 
     *  Save the last white move - so that it can be painted
     * differently.
     */
    if ( player == WHITE )
    {
	white_x_last = x;
	white_y_last = y;
    }

    /* Mark endpoint. */
    g_board[x][y] = player;

    /* Flip the interim pieces. */
    turn_bricks(player, x, y);
    
    /* Update the display. */
    drawBoard( );
}


/*
 *  Undo the last human players move.
 *
 *  We keep a backup of the grid before making each move
 * so we simply revert to that - losing the last computer + human
 * move.
 */
void undo_move()
{
    int x, y;
    for( y = 0; y < 8; y++ )
    {
	for( x = 0; x < 8; x++ )
	{
	    g_board[x][y] = g_backupBoard[x][y];
	}
    }
    
    /* Updated the display. */
    drawBoard( );
}


/*
 * Will the given players move capture any pieces?
 */
int will_capture(int player, int x, int y)
{
    int captured = 0;

    /* calculate amount of captured pieces */

    /* northwest */
    captured += check_direction(player, x, y, -1, -1);

    /* north */
    captured += check_direction(player, x, y,  0, -1);

    /* northeast */
    captured += check_direction(player, x, y, +1, -1);

    /* west */
    captured += check_direction(player, x, y, -1,  0);

    /* east */
    captured += check_direction(player, x, y, +1,  0);

    /* southwest */
    captured += check_direction(player, x, y, -1, +1);

    /* south */
    captured += check_direction(player, x, y,  0, +1);

    /* southeast */
    captured += check_direction(player, x, y, +1, +1);
    
    return captured;
}


/*
 * Is the given players move legal?
 */
int legal_move(int player, int x, int y)
{
    /* occupied? */
    if(g_board[x][y] != EMPTY)
    {
	return 0;
    }

    /* will the move capture at least one of the bricks? */
    if(will_capture(player, x, y))
    {
	return 1;
    }
    else
    {
	return 0;
    }
}


/*
 *  Perform the computer players move.
 *
 *  Iterate over all possible moves, and choose one which has the
 * greatest fitness.
 *
 */
void performComputerMove( )
{
    int i     = 0;
    int max   = -1000;
    int maxX  = -1;
    int maxY  = -1;

    /* Get the moves. */
    struct sMove *moves = generateMoves( WHITE );

    /*
     *  Iterate over each move, choose the move with the highest fitness.
     *  Note: The score isn't just the number of tiles captured, it is
     * weighted.
     */
    for( i = 0; i < 64; i++ )
    {
	struct sMove *move = &moves[i];
	if ( move->score > max )
	{
	    maxX = move->x;
	    maxY = move->y;
	    max = move->score;
	}
    }

    /* Did we have a valid move? */
    if  ( ( maxX != -1 ) && 
	  ( maxY != -1 ) &&
	  ( g_board[maxX][maxY] == EMPTY ) )
    {
	make_move( WHITE, maxX, maxY );
    }
    else
    {
	printf("White Can't move\n" );
    }

    /* Free the generated moves */
    free( moves );


    /* End of game? */
    if ( isGameOver() )
    {
	gameOver();
    }
    sleep(.5);
}



int
main(int argc, char* argv[])
{
  int screen_num;		/* number of screen to place the window on.  */

  char *display_name = getenv("DISPLAY");  /* address of the X display.      */

  unsigned int display_height;
  int playing;

  XFontStruct* font_info;
  char* font_name = "*-helvetica-*-12-*";


  XEvent an_event;
  int x;
  int y;

  /**
   * Make sure root isn't playing.
   */
  if ( 0 ==   geteuid())
  {
      fprintf( stderr, "Root shouldn't play this game.\n" );
      exit( -1 );
  }

  /* open connection with the X server. */
  g_display = XOpenDisplay(display_name);
  if (g_display == NULL) 
  {
      fprintf(stderr, "%s: cannot connect to X server '%s'\n",
	      argv[0], display_name);
      exit(1);
  }

  /* get the geometry of the default screen for our display. */
  screen_num     = DefaultScreen(g_display);
  display_height = DisplayHeight(g_display, screen_num);

  /*
   * Set the size of the main window, originally this is defined
   * as a square the size of a third of the screens height - then
   * it is made bigger to make it a size divisible by 8.
   *
   * Finally we add on room for one extra row at the bottom to
   * hold our status text.
   */
  g_win_height = (display_height / 3);
  while( g_win_height % 8 )
  {
      g_win_height += 1;
  }
  g_win_width  = g_win_height;

  /* Row of text */
  g_win_height += g_win_height/8;


  /* create a simple window, as a direct child of the screen's   */
  /* root window. Use the screen's white color as the background */
  /* color of the window. Place the new window's top-left corner */
  /* at the given 'x,y' coordinates.                             */
  createMainWindow();

  /* allocate a new GC (graphics context) for drawing in the window. */
  g_gc = create_gc(g_display, g_win, 0);
  XSync(g_display, False);

  /* try to load the given font. */
  font_info = XLoadQueryFont(g_display, font_name);
  if (!font_info) 
  {
      fprintf( stderr, "XLoadQueryFont: failed loading font '%s'\n", 
	       font_name );
      exit( -1 );
  }

  /* assign the given font to our GC. */
  XSetFont( g_display, g_gc, font_info->fid );

  /* Setup the startup message. */
  snprintf( g_statusMsg, sizeof(g_statusMsg)-1,
	    "XReverse v%s - By <steve@steve.org.uk>",
	    VERSION );

  /* Initialize the board. */
  initBoard();

  playing = 1;

  /* Draw the board */
  drawBoard( );

  /* flush all pending requests to the X server. */
  XFlush( g_display );

  /* select the input events which we care about. */
  XSelectInput( g_display, g_win, 
		ExposureMask | 
		ButtonPressMask |
		KeyRelease | 
		StructureNotifyMask 
		);

  /* our event loop. */
  while (1) 
  {
      if ( isGameOver() )
      {
	  gameOver();
	  playing = 0;
      }

      /* Get the next event. */
      XNextEvent( g_display, &an_event);

      switch (an_event.type) 
      {
          case DestroyNotify:
	  {
	      printf("Aiii\n" );
	  }
	  break;
          case KeyPress:
	  {
	      /* translate the key code to a key symbol. */
	      KeySym key_symbol = XKeycodeToKeysym( g_display, 
						    an_event.xkey.keycode, 
						    0);
	      switch (key_symbol) 
	      {
	           case XK_N: /* FALL THRU */
	           case XK_n:
		   {
		       playing = 1;
		       initBoard();
		       /* Draw the board */
		       drawBoard( );
		   }
		   break;

	           case XK_H: /* FALL THRU */
	           case XK_h:
		   {
		       drawHints( BLACK, 1 );
		   }
		   break;

	           case XK_Q: /* FALL THRU */
	           case XK_q:
		   {
		       /* close the connection to the X server. */
		       XCloseDisplay( g_display );
		       exit(0);

		   }
		   break;

	           case XK_U: /* FALL THRU */
	           case XK_u:
		   {
		       if ( playing )
		       {
			   undo_move();
		       }
		   }
		   break;
		   
	      }
 	      default:
		  break;
	  }
	  break;

          case Expose:
	  {
	      if (an_event.xexpose.count > 0)
		  break;
	      /* just redraw the whole board. */
	      drawBoard( );
	  }
	  break;

	  case ButtonPress:
	  {
	      if ( !playing )
	      {
		  break;
	      }

	      if ( isGameOver() )
	      {
		  gameOver();
		  playing = 0;
		  break;
	      }

	      /* The event co-ords */
	      x = an_event.xbutton.x;
	      y = an_event.xbutton.y;

	      /* convert to grid co-ords */
	      x = x / (g_win_width/8);
	      y = y / (g_win_width/8);

	      /* make sure its a valid board position. */
	      if ( ( x > -1 ) && ( x < 8 )  &&
		   ( y > -1 ) && ( y < 8 ) )
	      {
		  if ( isGameOver() )
		  {
		      gameOver();
		      playing = 0;
		      break;
		  }

		  if ( legal_move( BLACK, x, y ) )
		  {
		
		      make_move( BLACK, x, y );
		      sleep( 0.5 );
		      setStatus( "Computers Turn" );

		      /* Computers move */
		      performComputerMove();

		      /* Keep moving the computer if the 
		       * human can't make a move */
		      while ( ( drawHints( BLACK, 0 ) == 0 ) &&
			      ( ! isGameOver() ) )
		      { 
			  setStatus( "You can't move" );
			  performComputerMove();
		      }
		
		      setStatus( "Your Move" );
		  }
	      }
	  }
	  break;
    }
  }

  /* close the connection to the X server. */
  XCloseDisplay( g_display );

  return 0;
}


/*
 *  Return a 64 element array containing all potential moves.
 *
 *  Each element of the array contains an x,y coordinate pair,
 * and a fitness score - the highest scoring move is that which
 * should be made.
 *
 */
struct sMove* generateMoves( int player )
{
    int i;
    int x, y;

    /*
     * Allocate moves.
     */
    struct sMove *results = (struct sMove *)malloc( sizeof( struct sMove ) * 64 );
    
    i = 0;
    for( y = 0; y < 8; y++ )
    {
	for(  x = 0; x < 8; x++ )
	{
	    struct sMove *move = &results[i];
	    move->x = x;
	    move->y = y;
	    if ( legal_move( player, x, y ) )
	    {
		move->score = will_capture( player, x, y );

		/*
		 * Up the score of the corners by a small amount.
		 */
		if ( ( x == 0 ) && ( y == 0 ) )
		    move->score += 5;
		if ( ( x == 0 ) && ( y == 7 ) )
		    move->score += 5;
		if ( ( x == 7 ) && ( y == 0 ) )
		    move->score += 5;
		if ( ( x == 7 ) && ( y == 7 ) )
		    move->score += 5;

		/*
		 * Down the scores of moves adjacent to corner pieces.
		 */
		/* top left */
		if ( ( x == 0 ) && ( y == 1 ) )
		    move->score -= 5;
		if ( ( x == 1 ) && ( y == 0 ) )
		    move->score -= 5;
		if ( ( x == 1 ) && ( y == 1 ) )
		    move->score -= 5;

		/* top right. */
		if ( ( x == 6 ) && ( y == 0 ) )
		    move->score -= 5;
		if ( ( x == 7 ) && ( y == 1 ) )
		    move->score -= 5;
		if ( ( x == 6 ) && ( y == 1 ) )
		    move->score -= 5;

		/* bottom left */
		if ( ( x == 0 ) && ( y == 6 ) )
		    move->score -= 5;
		if ( ( x == 1 ) && ( y == 7 ) )
		    move->score -= 5;
		if ( ( x == 1 ) && ( y == 6 ) )
		    move->score -= 5;

		/* bottom right */
		if ( ( x == 6 ) && ( y == 6 ) )
		    move->score -= 5;
		if ( ( x == 6 ) && ( y == 7 ) )
		    move->score -= 5;
		if ( ( x == 7 ) && ( y == 6 ) )
		    move->score -= 5;
	    }
	    else
	    {
		move->score = -100;
	    }

	    i += 1;
	}
    }
    return( results );
}

