/*
 * @(#)xrubik.c
 *
 * Copyright 1993 - 2009  David A. Bagley, bagleyd@tux.org
 *
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the author not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * This program is distributed in the hope that it will be "useful",
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

/* Driver file for Rubik */

#ifndef WINVER
static const char aboutHelp[] = {
"Rubik Version 7.5\n"
"Send bugs (reports or fixes) to the author: "
"David Bagley <bagleyd@tux.org>\n"
"The latest version is at: "
"http://www.tux.org/~bagleyd/puzzles.html"
};

static const char optionsHelp[] = {
"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n"
"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
"[-{foreground|fg} {color}] [-{background|bg} {color}]\n"
"[-face{0|1|2|3|4|5} {color}] [-{border|bd} {color}]\n"
"[-delay msecs] [-[no]sound] [-moveSound {filename}]\n"
"[-{font|fn} {fontname}] [-view {int}] [-size{x|y|z} {int}]\n"
"[-[no]orient] [-[no]practice] [-userName {string}]\n"
"[-scoreFile {filename}] [-scores] [-version]"
};
#endif

#if defined(HAVE_MOTIF) || defined(WINVER)
static const char descriptionHelp[] = {
"The original puzzle  has 9 squares per face  (size = 3).  "
"The puzzle was  designed by Erno Rubik and called the\n"
"Rubik's Cube.  This has 8!*12!*3^8*2^12/12 or 4.3 * 10^19 "
"different combinations.\n"
"The Pocket Cube  has 4 squares per face  (size = 2) also "
"designed by  Erno Rubik.  This has 7!*3^6 or 3,674,160\n"
"different combinations.\n"
"Rubik's Revenge has 16 squares per face (size = 4) also "
"designed by Erno Rubik.  This has 7!*3^6*24!*24!/(4!)^6\n"
"or 7.4 * 10^46 different combinations.\n"
"5x5x5 Cube.  This has 8!*12!*3^7*2^10*(24!)^3/(4!)^12 "
"or 2.83 * 10^74 different combinations.\n"
"There is also the Magic Domino 3x3x2 cube which has "
"(8!)^2/4 or 406,425,600 combinations.\n"
"A physical 6x6x6 cube is possible but to my knowledge no "
"one has been too successful in building one.  7x7x7 is\n"
"also possible, but here one  must make the center most "
"cubes smaller than the  outside cubes, so the corners do\n"
"not fall off when turned."
};

static const char featuresHelp[] = {
"Press \"mouse-left\" button to move a piece.  Release "
"\"mouse-left\" button on a piece on the same face and\n"
"in the same row.  The pieces will then turn towards "
"where the mouse button was released.\n"
"\n"
"Click \"mouse-center\" button, or press \"P\" or \"p\" "
"keys, to toggle the practice mode  (in practice mode\n"
"the record should say \"practice\").  This is good for "
"learning moves and experimenting.\n"
"\n"
"Click \"mouse-right\" button, or press \"Z\" or \"z\" "
"keys, to randomize the puzzle (this must be done first\n"
"to set a new record).\n"
"\n"
"Press \"G\" or \"g\" keys to get a saved puzzle.\n"
"\n"
"Press \"W\" or \"w\" keys to save (write) a puzzle.\n"
"\n"
"Press \"U\" or \"u\" keys to undo a move.\n"
"\n"
"Press \"R\" or \"r\" keys to redo a move.\n"
"\n"
"Press \"C\" or \"c\" keys to clear the puzzle.\n"
"\n"
"Press \"S\" or \"s\" keys to start auto-solver.  Only works "
"on 1x1x1, 2x2x2, and 3x3x3 cubes.\n"
"\n"
"Press \"O\" or \"o\" keys to toggle  the orient mode.  One "
"has to  orient the faces in orient mode, besides\n"
"getting all  the faces to be the  same color.  To do this "
"one  has to get the  lines to be  oriented in\n"
"the same direction,  this only matters  with center "
"piece,  if at all (i.e.  those that are not on a\n"
"corner or edge).  This does add complexity so there are "
"2 sets of records.\n"
"\n"
"Press \"I\" or \"i\" keys to increase the number of pieces.\n"
"\n"
"Press \"D\" or \"d\" keys to decrease the number of pieces.\n"
"\n"
"Press \"x\" key to increase the number of pieces along "
"the x axis.\n"
"\n"
"Press \"X\" key to decrease the number of pieces along "
"the x axis.\n"
"\n"
"Press \"y\" key to increase the number of pieces along "
"the y axis.\n"
"\n"
"Press \"Y\" key to decrease the number of pieces along "
"the y axis.\n"
"\n"
"Press \"f\" key to increase the number of pieces along "
"the z axis.\n"
"\n"
"Press \"F\" key to decrease the number of pieces along "
"the z axis.\n"
"\n"
"Press \"V\" or \"v\" keys to change the view of the cube.\n"
"\n"
"Press \">\" or \".\" keys to speed up the movement of pieces.\n"
"\n"
"Press \"<\" or \",\" keys to slow down the movement of pieces.\n"
"\n"
"Press \"@\" key to toggle the sound.\n"
"\n"
"Press \"Esc\" key to hide program.\n"
"\n"
"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
"\n"
"Use the key pad or arrow keys to move without the mouse.\n"
"Key pad is defined for Rubik2d as:\n"
"  /    Counterclockwise\n"
"  8    Up\n"
"  ^\n"
"4<5>6  Left, Clockwise, Right\n"
"  v\n"
"  2    Down\n"
"Key pad for Rubik3d, use must use your intuition (is this "
"a cop out or what?).  The key pad  is defined\n"
"differently  depending on which side  of the cube your mouse "
"is pointing at.  One thing  that stays the\n"
"same is \"5\" is Clockwise and \"/\" is Counterclockwise.\n"
"\n"
"Use the control key and the left mouse button, keypad, or "
"arrow keys to move the whole cube.  This is\n"
"not recorded as a turn."
};

static const char referencesHelp[] = {
"Inside Rubik's Cube and Beyond by Christoph Bandelow, "
"Birkhauser, 1982 pp 44, 45, 88, 89.\n"
"Magic Cubes 1996 Catalog of Dr. Christoph Bandelow.\n"
"The Simple Solution To Rubik's Cube, James G. Nourse, "
"June 1981.\n"
"Rubik's Cube Newsletter by Ideal Aug 1982 Vol.1 No. 2.\n"
"Rubik's Cube The Solution, Ideal Toy Corporation, 1981.\n"
"Rubik's Revenge Puzzle The Solution, Ideal Toy "
"Corporation, 1982."
};
#endif

static const char solveHelp[] = {
"Auto-solver: sorry, only implemented for size nxnxn where "
"n < 4."
};

#include "file.h"
#ifdef WINVER
#include "RubikP.h"
#define TITLE "wrubik"

static RubikRec widget;

#ifndef SCOREPATH
#ifdef UNIXDELIM
#define SCOREPATH "c:/WINDOWS"
#else
#define SCOREPATH "c:\\WINDOWS"
#endif
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(widget.core.hWnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->rubik.started = b
#else
#include "xwin.h"
#include <X11/Shell.h>
#include <X11/cursorfont.h>
#ifdef HAVE_MOTIF
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/MessageB.h>
#include <Xm/PushBG.h>
#include <Xm/CascadeB.h>
#include <Xm/Scale.h>
#include <Xm/ToggleB.h>
#ifdef MOUSEBITMAPS
#include "pixmaps/mouse-l.xbm"
#include "pixmaps/mouse-r.xbm"
#endif
#define PRINT_MESSAGE(b) printState(message, b)
#else
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#define SET_STARTED(w,b) XtVaSetValues(w, XtNstart, b, NULL)
#include "Rubik.h"
#include "Rubik2d.h"
#include "Rubik3d.h"
#ifdef HAVE_OPENGL
#include "RubikGL.h"
#endif
#ifdef HAVE_XPM
#include <X11/xpm.h>
#ifdef CONSTPIXMAPS
#include "rubik.t.xpm"
#include "rubik.p.xpm"
#include "rubik.s.xpm"
#include "rubik.m.xpm"
#include "rubik.l.xpm"
#include "rubik.xpm"
#else
#include "pixmaps/rubik.t.xpm"
#include "pixmaps/rubik.p.xpm"
#include "pixmaps/rubik.s.xpm"
#include "pixmaps/rubik.m.xpm"
#include "pixmaps/rubik.l.xpm"
#include "pixmaps/rubik.xpm"
#endif
#define RESIZE_XPM(s) ((char **) (((s)<=32)?\
(((s)<=22)?(((s)<=16)?rubik_t_xpm:rubik_p_xpm):\
(((s)<=24)?rubik_s_xpm:rubik_m_xpm)):\
(((s)<=48)?rubik_l_xpm:rubik_xpm)))
#endif
#include "pixmaps/rubik.xbm"
#define DEFINE_XBM (char *) rubik_bits, rubik_width, rubik_height
#ifndef SCOREPATH
#ifdef VMS
#define SCOREPATH "SYS$LOGIN:"
#else
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif
#endif

#ifndef SCOREFILE
#define SCOREFILE "rubik.scores"
#endif

#define MAX_FACETS 6
#define NEVER (-1)
#define FILE_NAME_LENGTH 1024
#define USER_NAME_LENGTH 120
#define MESSAGE_LENGTH (USER_NAME_LENGTH+64)
#define TITLE_LENGTH 2048
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int score;
	char name[USER_NAME_LENGTH];
} PuzzleRecord;

static PuzzleRecord puzzleRecord[2][MAX_FACETS - MIN_FACETS + 1]
[MAX_FACETS - MIN_FACETS + 1][MAX_FACETS - MIN_FACETS + 1];
static int movesDsp = 0;
static char messageDsp[MESSAGE_LENGTH] = "Welcome";
static char recordDsp[MESSAGE_LENGTH] = "NOT RECORDED";
#ifndef HAVE_MOTIF
static char titleDsp[TITLE_LENGTH] = "";
#endif
static char scoreFileName[FILE_NAME_LENGTH] = SCOREFILE;
static char fileName[FILE_NAME_LENGTH];
static Boolean randomized = False;
#ifdef WINVER
#define PROGRAM_NAME_LENGTH 80
static char progDsp[PROGRAM_NAME_LENGTH] = TITLE;
static char userNameDsp[USER_NAME_LENGTH] = "Guest";
#else
#ifdef HAVE_MOTIF
#define MIN_SPEED 1
#define MAX_SPEED 50
static Widget movesText, recordText, message;
static Widget cubesXSlider, cubesYSlider, cubesZSlider, speedSlider;
static Widget orientizeSwitch, practiceSwitch;
static char buff[21];
static Widget descriptionDialog, featuresDialog;
static Widget optionsDialog, referencesDialog, aboutDialog;
static Widget solveDialog, practiceDialog, randomizeDialog;
static Arg arg[3];
#else
static Widget shell;
#ifdef HAVE_OPENGL
static Widget shellGL;
#endif
#endif
static Pixmap pixmap = None;
static Widget topLevel, puzzle2d, puzzle3d;
#ifdef HAVE_OPENGL
static Widget puzzleGL;
#endif
static char *progDsp;
static char userNameDsp[USER_NAME_LENGTH] = "";

#ifdef HAVE_MOTIF
static void
printState(Widget w, char *msg)
{
	XmString xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("printState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtVaSetValues(w, XmNlabelString, xmstr, NULL);
}
#endif

static void
printRecords(void)
{
	int i, j, k, orient;

	(void) printf("                RUBIK  HALL OF FAME\n\n");
	(void) printf("ORIENT   X   Y   Z USER            MOVES\n");
	for (orient = 0; orient < 2; orient++)
		for (i = 0; i < MAX_FACETS - MIN_FACETS + 1; i++)
			for (j = i; j < MAX_FACETS - MIN_FACETS + 1; j++)
				for (k = j; k < MAX_FACETS - MIN_FACETS + 1; k++) {
					if (puzzleRecord[orient][i][j][k].score > 0)
						(void) printf("%6d%4d%4d%4d %-12s%9d\n",
							orient, i + 1, j + 1, k + 1,
							puzzleRecord[orient][i][j][k].name,
							puzzleRecord[orient][i][j][k].score);
				}
}
#endif

static void
initRecords(void)
{
	int i, j, k, orient;

	for (orient = 0; orient < 2; orient++)
		for (i = 0; i < MAX_FACETS - MIN_FACETS + 1; i++)
			for (j = i; j < MAX_FACETS - MIN_FACETS + 1; j++)
				for (k = j; k < MAX_FACETS - MIN_FACETS + 1; k++) {
					puzzleRecord[orient][k][j][i].score = puzzleRecord[orient][k][i][j].score =
						puzzleRecord[orient][j][k][i].score = puzzleRecord[orient][j][i][k].score =
						puzzleRecord[orient][i][k][j].score = puzzleRecord[orient][i][j][k].score = NEVER;
					(void) strncpy(puzzleRecord[orient][k][j][i].name,
						NOACCESS, USER_NAME_LENGTH);
					(void) strncpy(puzzleRecord[orient][k][i][j].name,
						NOACCESS, USER_NAME_LENGTH);
					(void) strncpy(puzzleRecord[orient][j][k][i].name,
						NOACCESS, USER_NAME_LENGTH);
					(void) strncpy(puzzleRecord[orient][j][i][k].name,
						NOACCESS, USER_NAME_LENGTH);
					(void) strncpy(puzzleRecord[orient][i][k][j].name,
						NOACCESS, USER_NAME_LENGTH);
					(void) strncpy(puzzleRecord[orient][i][j][k].name,
						NOACCESS, USER_NAME_LENGTH);
				}
}

static void
readRecords(void)
{
	FILE *fp;
	int i, j, k, n, orient;
	char userName[USER_NAME_LENGTH];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname;

	stringCat(&buf1, CURRENTDELIM, scoreFileName);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	(void) strncpy(fileName, lname, USER_NAME_LENGTH);
	if ((fp = fopen(fileName, "r")) == NULL) {
		(void) strncpy(fileName, fname, USER_NAME_LENGTH);
		/* Try installed directory. */
		if ((fp = fopen(fileName, "r")) == NULL) {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (orient = 0; orient < 2; orient++)
		for (i = 0; i < MAX_FACETS - MIN_FACETS + 1; i++)
			for (j = i; j < MAX_FACETS - MIN_FACETS + 1; j++)
				for (k = j; k < MAX_FACETS - MIN_FACETS + 1; k++) {
					(void) fscanf(fp, "%d %s\n", &n, userName);
					if (n <= puzzleRecord[orient][i][j][k].score ||
							puzzleRecord[orient][i][j][k].score <= NEVER) {
						puzzleRecord[orient][k][j][i].score = puzzleRecord[orient][k][i][j].score =
							puzzleRecord[orient][j][k][i].score = puzzleRecord[orient][j][i][k].score =
							puzzleRecord[orient][i][k][j].score = puzzleRecord[orient][i][j][k].score = n;
						(void) strncpy(puzzleRecord[orient][k][j][i].name,
							userName, USER_NAME_LENGTH);
						(void) strncpy(puzzleRecord[orient][k][i][j].name,
							userName, USER_NAME_LENGTH);
						(void) strncpy(puzzleRecord[orient][j][k][i].name,
							userName, USER_NAME_LENGTH);
						(void) strncpy(puzzleRecord[orient][j][i][k].name,
							userName, USER_NAME_LENGTH);
						(void) strncpy(puzzleRecord[orient][i][k][j].name,
							userName, USER_NAME_LENGTH);
						(void) strncpy(puzzleRecord[orient][i][j][k].name,
							userName, USER_NAME_LENGTH);
					}
				}
	(void) fclose(fp);
}

static void
writeRecords(void)
{
	FILE *fp;
	int i, j, k, orient;
	char *buf1 = NULL;

	if ((fp = fopen(fileName, "w")) == NULL) {
		stringCat(&buf1, "Can not write to ", fileName);
		PRINT_MESSAGE(buf1);
		free(buf1);
		return;
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockFile[FILE_NAME_LENGTH];

		(void) strncpy(lockFile, fileName, FILE_NAME_LENGTH - 6);
		(void) strcat(lockFile, ".lock");
		while (((lfd = open(lockFile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
				"Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
				"Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (orient = 0; orient < 2; orient++) {
			for (i = 0; i < MAX_FACETS - MIN_FACETS + 1; i++) {
				for (j = i; j < MAX_FACETS - MIN_FACETS + 1; j++) {
					for (k = j; k < MAX_FACETS - MIN_FACETS + 1; k++)
						(void) fprintf(fp, "%d %s\n",
							puzzleRecord[orient][i][j][k].score,
							puzzleRecord[orient][i][j][k].name);
					(void) fprintf(fp, "\n");
				}
				(void) fprintf(fp, "\n");
			}

			(void) fprintf(fp, "\n");
		}
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockFile);
#endif
		(void) fclose(fp);
	}
}

static void
printRecord(int sizex, int sizey, int sizez, Boolean orient, Boolean practice)
{
	int i = sizex - MIN_FACETS, j = sizey - MIN_FACETS, k = sizez - MIN_FACETS;
	int l = (orient) ? 1 : 0;

	if (practice) {
		(void) strncpy(recordDsp, "practice", MESSAGE_LENGTH);
	} else if (sizex > MAX_FACETS ||
			sizey > MAX_FACETS || sizez > MAX_FACETS) {
		(void) strncpy(recordDsp, "NOT RECORDED", MESSAGE_LENGTH);
	} else if (puzzleRecord[l][i][j][k].score <= NEVER) {
		(void) sprintf(recordDsp, "NEVER %s", NOACCESS);
	} else {
		(void) sprintf(recordDsp, "%d %s",
			puzzleRecord[l][i][j][k].score,
			puzzleRecord[l][i][j][k].name);
	}
#ifdef HAVE_MOTIF
	printState(recordText, recordDsp);
#endif
}

#ifndef WINVER
/* There is probably a better way to assure that they are the same
 * but I do not know it off hand. */
static void
makeEquivalent(String *userName, String *scoreFile,
		int *sizex, int *sizey, int *sizez,
		Boolean *orient, Boolean *practice, int *delay)
{
	Boolean mono, reverse;
	Boolean scoreOnly, versionOnly;
	Pixel foreground, background, pieceBorder;
	String faceColor[MAX_FACES];

	XtVaGetValues(puzzle2d,
		XtNuserName, userName,
		XtNscoreFile, scoreFile,
		XtNsizex, sizex,
		XtNsizey, sizey,
		XtNsizez, sizez,
		XtNorient, orient,
		XtNpractice, practice,
		XtNmono, &mono,
		XtNreverseVideo, &reverse,
		XtNdelay, delay,
		XtNforeground, &foreground,
		XtNbackground, &background,
		XtNpieceBorder, &pieceBorder,
		XtNfaceColor0, &(faceColor[0]),
		XtNfaceColor1, &(faceColor[1]),
		XtNfaceColor2, &(faceColor[2]),
		XtNfaceColor3, &(faceColor[3]),
		XtNfaceColor4, &(faceColor[4]),
		XtNfaceColor5, &(faceColor[5]),
		XtNscoreOnly, &scoreOnly,
		XtNversionOnly, &versionOnly, NULL);
	if (versionOnly) {
		(void) printf("%s\n", aboutHelp);
		exit(0);
	}
	if (strcmp(*scoreFile, ""))
		(void) strncpy(scoreFileName, *scoreFile, FILE_NAME_LENGTH);
	if (scoreOnly) {
		initRecords();
		readRecords();
		printRecords();
		exit(0);
	}
	XtVaSetValues(puzzle2d,
		XtNdirection, IGNORE_DIR,
		XtNstart, False, NULL);
	XtVaSetValues(puzzle3d,
		XtNuserName, *userName,
		XtNsizex, *sizex,
		XtNsizey, *sizey,
		XtNsizez, *sizez,
		XtNorient, *orient,
		XtNpractice, *practice,
		XtNmono, mono,
		XtNreverseVideo, reverse,
		XtNdelay, *delay,
		XtNdirection, IGNORE_DIR,
		XtNstart, False,
		XtNforeground, foreground,
		XtNbackground, background,
		XtNpieceBorder, pieceBorder,
		XtNfaceColor0, faceColor[0],
		XtNfaceColor1, faceColor[1],
		XtNfaceColor2, faceColor[2],
		XtNfaceColor3, faceColor[3],
		XtNfaceColor4, faceColor[4],
		XtNfaceColor5, faceColor[5], NULL);
#ifdef HAVE_OPENGL
	XtVaSetValues(puzzleGL,
		XtNuserName, *userName,
		XtNsizex, *sizex,
		XtNsizey, *sizey,
		XtNsizez, *sizez,
		XtNorient, *orient,
		XtNpractice, *practice,
		XtNmono, mono,
		XtNreverseVideo, reverse,
		XtNdelay, *delay,
		XtNdirection, IGNORE_DIR,
		XtNstart, False,
		XtNforeground, foreground,
		XtNbackground, background,
		XtNpieceBorder, pieceBorder,
		XtNfaceColor0, faceColor[0],
		XtNfaceColor1, faceColor[1],
		XtNfaceColor2, faceColor[2],
		XtNfaceColor3, faceColor[3],
		XtNfaceColor4, faceColor[4],
		XtNfaceColor5, faceColor[5], NULL);
#endif
}
#endif

static void
printStatus(char *msg, int nMoves
#ifndef HAVE_MOTIF
		, int dim, int sizeX, int sizeY, int sizeZ
#endif
		)
{
#ifdef HAVE_MOTIF
	printState(message, msg);
	(void) sprintf(buff, "%d", nMoves);
	printState(movesText, buff);
#else
#ifdef HAVE_OPENGL
	if (dim == 4)
		(void) sprintf(titleDsp, "%sGL: %dx%dx%d@ (%d/%s) - %s",
			progDsp, sizeX, sizeY, sizeZ,
			nMoves, recordDsp, msg);
	else
#endif
		(void) sprintf(titleDsp, "%s%dd: %dx%dx%d@ (%d/%s) - %s",
			progDsp, dim, sizeX, sizeY, sizeZ,
			nMoves, recordDsp, msg);
#ifdef WINVER
	SetWindowText(widget.core.hWnd, (LPSTR) titleDsp);
#else
#ifdef HAVE_OPENGL
	if (dim == 4)
		XtVaSetValues(XtParent(puzzleGL), XtNtitle, titleDsp, NULL);
	else
#endif
	XtVaSetValues(XtParent((dim == 2) ? puzzle2d : puzzle3d),
		XtNtitle, titleDsp, NULL);
#endif
#endif
}

static Boolean
handleSolved(int counter, int sizex, int sizey, int sizez, Boolean orient)
{
	int i = sizex - MIN_FACETS, j = sizey - MIN_FACETS, k = sizez - MIN_FACETS;
	int l = (orient) ? 1 : 0;

	if (sizex <= MAX_FACETS && sizey <= MAX_FACETS && sizez <= MAX_FACETS &&
			(counter < puzzleRecord[l][i][j][k].score ||
			puzzleRecord[l][i][j][k].score <= NEVER)) {
		readRecords();	/* Maybe its been updated by another */
		puzzleRecord[l][i][j][k].score = counter;
		(void) strncpy(puzzleRecord[l][i][j][k].name, userNameDsp,
			USER_NAME_LENGTH);
		if ((sizex <= 2 && sizey <= 2) || (sizey <= 2 && sizez <= 2) ||
				(sizez <= 2 && sizex <= 2) || (orient &&
				(counter < puzzleRecord[!l][i][j][k].score ||
				puzzleRecord[!l][i][j][k].score <= NEVER))) {
			puzzleRecord[!l][i][j][k].score = counter;
			(void) strncpy(puzzleRecord[!l][i][j][k].name,
				userNameDsp, USER_NAME_LENGTH);
		}
		writeRecords();
		printRecord(sizex, sizey, sizez, orient, False);
		return True;
	}
	return False;
}

static void
initialize(
#ifdef WINVER
RubikWidget w, HBRUSH brush
#else
void
#endif
)
{
	int sizex, sizey, sizez;
	Boolean orient, practice;
	char *userName, *scoreFile;
#ifdef WINVER
	int dim;

	initializePuzzle(w, brush);

	sizex = w->rubik.sizex;
	sizey = w->rubik.sizey;
	sizez = w->rubik.sizez;
	orient = w->rubik.orient;
	practice = w->rubik.practice;
	userName = w->rubik.userName;
	scoreFile = w->rubik.scoreFile;
	dim = w->rubik.dim;
	SET_STARTED(w, False);
	if (strcmp(scoreFile, ""))
		(void) strncpy(scoreFileName, scoreFile, FILE_NAME_LENGTH);
#else
	int delay;

	makeEquivalent(&userName, &scoreFile, &sizex, &sizey, &sizez,
		&orient, &practice, &delay);
#ifdef HAVE_MOTIF
	if (sizex > MAX_FACETS)
		XtVaSetValues(cubesXSlider, XmNmaximum, sizex, NULL);
	XmScaleSetValue(cubesXSlider, sizex);
	if (sizey > MAX_FACETS)
		XtVaSetValues(cubesYSlider, XmNmaximum, sizey, NULL);
	XmScaleSetValue(cubesYSlider, sizey);
	if (sizez > MAX_FACETS)
		XtVaSetValues(cubesZSlider, XmNmaximum, sizez, NULL);
	XmScaleSetValue(cubesZSlider, sizez);
	XmToggleButtonSetState(orientizeSwitch, orient, True);
	XmToggleButtonSetState(practiceSwitch, practice, True);
	XmScaleSetValue(speedSlider, MAX_SPEED + MIN_SPEED - delay - 1);
#endif
#endif
	initRecords();
	readRecords();
#ifndef WINVER
	(void) strncpy(userNameDsp, userName, USER_NAME_LENGTH);
#endif
	if (!strcmp(userName, "") || !strcmp(userName, "(null)") ||
			!strcmp(userName, NOACCESS) ||
			!strcmp(userName, NOBODY)) {
#ifdef WINVER
		(void) strncpy(userNameDsp, userName, USER_NAME_LENGTH);
#else
		char *login = getlogin();

		if (login == NULL) {
			(void) strcpy(userNameDsp, "");
		} else {
			(void) sprintf(userNameDsp, "%s", login);
		}
		if (!strcmp(userNameDsp, "") ||
				!strcmp(userNameDsp, "(null)") ||
				!strcmp(userNameDsp, NOACCESS) ||
				!strcmp(userNameDsp, NOBODY))
			/* It really IS nobody */
			(void) sprintf(userNameDsp, "%s", "guest");
#endif
	}
	printRecord(sizex, sizey, sizez, orient, practice);
#ifdef WINVER
	printStatus(messageDsp, movesDsp, dim, sizex, sizey, sizez);
#else
#ifdef HAVE_MOTIF
	printStatus(messageDsp, movesDsp);
#else
	printStatus(messageDsp, movesDsp, 2, sizex, sizey, sizez);
	printStatus(messageDsp, movesDsp, 3, sizex, sizey, sizez);
#ifdef HAVE_OPENGL
	printStatus(messageDsp, movesDsp, 4, sizex, sizey, sizez);
#endif
#endif
#endif
}

#ifdef WINVER
void
setPuzzle(RubikWidget w, int reason)
#else
static void
puzzleListener(Widget w, caddr_t clientData, rubikCallbackStruct *callData)
#endif
{
	int sizex, sizey, sizez;
	Boolean orient, practice, start, cheat;
#ifdef WINVER
	int dim = 0;
#else
	int reason = callData->reason;
	Widget otherw1 = (Widget) NULL;
#ifdef HAVE_OPENGL
	Widget otherw2 = (Widget) NULL;
#endif
#ifndef HAVE_MOTIF
	int dim = 0, otherdim1 = 0;
#ifdef HAVE_OPENGL
	int otherdim2 = 0;
#endif
#endif

	if (w == puzzle2d) {
		otherw1 = puzzle3d;
#ifdef HAVE_OPENGL
		otherw2 = puzzleGL;
#endif
#ifndef HAVE_MOTIF
		dim = 2;
		otherdim1 = 3;
#ifdef HAVE_OPENGL
		otherdim2 = 4;
#endif
#endif
	} else if (w == puzzle3d) {
		otherw1 = puzzle2d;
#ifdef HAVE_OPENGL
		otherw2 = puzzleGL;
#endif
#ifndef HAVE_MOTIF
		dim = 3;
		otherdim1 = 2;
#ifdef HAVE_OPENGL
		otherdim2 = 4;
#endif
#endif
	}
#ifdef HAVE_OPENGL
	else if (w == puzzleGL) {
		otherw1 = puzzle2d;
		otherw2 = puzzle3d;
#ifndef HAVE_MOTIF
		dim = 4;
		otherdim1 = 2;
		otherdim2 = 3;
#endif
	}
#endif
#endif
	(void) strcpy(messageDsp, "");
#ifdef WINVER
	sizex = w->rubik.sizex;
	sizey = w->rubik.sizey;
	sizez = w->rubik.sizez;
	orient = w->rubik.orient;
	practice = w->rubik.practice;
	cheat = w->rubik.cheat;
	start = w->rubik.started;
	dim = w->rubik.dim;
#else
	XtVaGetValues(w,
		XtNsizex, &sizex,
		XtNsizey, &sizey,
		XtNsizez, &sizez,
		XtNorient, &orient,
		XtNpractice, &practice,
		XtNstart, &start,
		XtNcheat, &cheat, NULL);
#endif
	switch (reason) {
	case ACTION_HIDE:
#ifdef WINVER
		ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
		(void) XIconifyWindow(XtDisplay(topLevel),
			XtWindow(topLevel),
			XScreenNumberOfScreen(XtScreen(topLevel)));
#ifndef HAVE_MOTIF
		(void) XIconifyWindow(XtDisplay(shell),
			XtWindow(shell),
			XScreenNumberOfScreen(XtScreen(shell)));
#ifdef HAVE_OPENGL
		(void) XIconifyWindow(XtDisplay(shellGL),
			XtWindow(shellGL),
			XScreenNumberOfScreen(XtScreen(shellGL)));
#endif
#endif
#endif
		break;
#ifndef WINVER
	case ACTION_PRACTICE_QUERY:
#ifdef HAVE_MOTIF
		XtManageChild(practiceDialog);
#else
		XtVaSetValues(puzzle2d, XtNmenu, ACTION_PRACTICE, NULL);
#endif
		break;
	case ACTION_RANDOMIZE_QUERY:
#ifdef HAVE_MOTIF
		XtManageChild(randomizeDialog);
#else
		XtVaSetValues(puzzle2d, XtNmenu, ACTION_RANDOMIZE, NULL);
#endif
		break;
#endif
	case ACTION_SOLVE_MESSAGE:
#ifdef WINVER
		(void) MessageBox(w->core.hWnd, solveHelp,
			"Auto-solve", MB_OK);
#else
#ifdef HAVE_MOTIF
		XtManageChild(solveDialog);
#else
		(void) strncpy(messageDsp, solveHelp, MESSAGE_LENGTH);
#endif
#endif
		break;
	case ACTION_RESTORE:
		if (practice) {
			(void) strncpy(recordDsp, "practice",
				MESSAGE_LENGTH);
#ifdef HAVE_MOTIF
			printState(recordText, recordDsp);
#endif
		}
		movesDsp = 0;
		randomized = False;
#ifndef WINVER
#ifdef HAVE_OPENGL
		XtVaSetValues(otherw2,
			XtNdirection, RESTORE_DIR, NULL);
#endif
		XtVaSetValues(otherw1,
			XtNdirection, RESTORE_DIR, NULL);
		XtVaSetValues(w,
			XtNdirection, RESTORE_DIR, NULL);
#endif
		break;
	case ACTION_CLEAR:
		movesDsp = 0;
		randomized = False;
#ifndef WINVER
#ifdef HAVE_OPENGL
		XtVaSetValues(otherw2,
			XtNdirection, CLEAR_DIR, NULL);
#endif
		XtVaSetValues(otherw1,
			XtNdirection, CLEAR_DIR, NULL);
		XtVaSetValues(w,
			XtNdirection, CLEAR_DIR, NULL);
#endif
		break;
	case ACTION_RESET:
		movesDsp = 0;
		randomized = False;
		break;
	case ACTION_AMBIGUOUS:
		(void) strncpy(messageDsp, "Ambiguous move",
			MESSAGE_LENGTH);
		break;
	case ACTION_ILLEGAL:
		if (practice || randomized)
			(void) strncpy(messageDsp, "Illegal move",
				MESSAGE_LENGTH);
		else
			(void) strncpy(messageDsp,
				"Randomize to start", MESSAGE_LENGTH);
		break;
#ifndef WINVER
	case ACTION_MOVED:
		if (callData->control) {
#ifdef HAVE_OPENGL
			XtVaSetValues(otherw2,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNcontrol, callData->control,
				XtNfast, callData->fast,
				XtNstart, True, NULL);
#endif
			XtVaSetValues(otherw1,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNcontrol, callData->control,
				XtNfast, callData->fast,
				XtNstart, True, NULL);
		} else {
			movesDsp++;
#ifdef HAVE_OPENGL
			XtVaSetValues(otherw2,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNcontrol, callData->control,
				XtNfast, callData->fast, NULL);
#endif
			XtVaSetValues(otherw1,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNcontrol, callData->control,
				XtNfast, callData->fast, NULL);
			SET_STARTED(w, True);
		}
		break;
#endif
	case ACTION_SOLVED:
		if (practice)
			movesDsp = 0;
		else if (cheat)
			(void) sprintf(messageDsp,
				"No cheating %s!!", userNameDsp);
		else if (handleSolved(movesDsp, sizex, sizey, sizez,
				orient))
			(void) sprintf(messageDsp,
				"Congratulations %s!!", userNameDsp);
		else
			(void) strncpy(messageDsp, "Solved!",
				MESSAGE_LENGTH);
		SET_STARTED(w, False);
#ifndef WINVER
#ifdef HAVE_OPENGL
		SET_STARTED(otherw2, False);
#endif
		SET_STARTED(otherw1, False);
#endif
		randomized = False;
		break;
	case ACTION_COMPUTED:
#ifdef WINVER
		SET_STARTED(w, False);
#else
		SET_STARTED(w, False);
#ifdef HAVE_OPENGL
		SET_STARTED(otherw2, False);
#endif
		SET_STARTED(otherw1, False);
#endif
		break;
	case ACTION_PRACTICE:
		movesDsp = 0;
		randomized = False;
		practice = !practice;
		if (!practice)
			(void) strncpy(messageDsp, "Randomize to start",
				MESSAGE_LENGTH);
		printRecord(sizex, sizey, sizez, orient, practice);
#ifdef WINVER
		w->rubik.practice = practice;
		w->rubik.started = False;
#else
		XtVaSetValues(w,
			XtNpractice, practice,
			XtNstart, False, NULL);
#ifdef HAVE_OPENGL
		XtVaSetValues(otherw2,
			XtNpractice, practice,
			XtNstart, False, NULL);
#endif
		XtVaSetValues(otherw1,
			XtNpractice, practice,
			XtNstart, False, NULL);
#ifdef HAVE_MOTIF
		XmToggleButtonSetState(practiceSwitch, practice, True);
#endif
#endif
		break;
	case ACTION_RANDOMIZE:
		movesDsp = 0;
		randomized = True;
#ifdef WINVER
		w->rubik.practice = False;
		w->rubik.started = False;
#else
		XtVaSetValues(w,
			XtNpractice, False,
			XtNstart, False, NULL);
#ifdef HAVE_OPENGL
		XtVaSetValues(otherw2,
			XtNpractice, False,
			XtNstart, False, NULL);
#endif
		XtVaSetValues(otherw1,
			XtNpractice, False,
			XtNstart, False, NULL);
#endif
		break;
#ifdef HAVE_OPENGL
	case ACTION_VIEW:
		{
			int view;

#ifdef WINVER
			view = w->rubik.view;
#else
			XtVaGetValues(puzzleGL,
				XtNview, &view, NULL);
#endif
			view = (view + 1) % (2 * MAX_VIEWS);
#ifdef WINVER
			w->rubik.view = view;
#else
			XtVaSetValues(puzzleGL,
				XtNview, view, NULL);
#endif
		}
		break;
#endif
	case ACTION_ORIENTIZE:
		movesDsp = 0;
		orient = !orient;
		printRecord(sizex, sizey, sizez, orient, practice);
#ifdef WINVER
		w->rubik.orient = orient;
#else
		XtVaSetValues(w, XtNorient, orient, NULL);
#ifdef HAVE_OPENGL
		XtVaSetValues(otherw2, XtNorient, orient, NULL);
#endif
		XtVaSetValues(otherw1, XtNorient, orient, NULL);
#ifdef HAVE_MOTIF
		XmToggleButtonSetState(orientizeSwitch, orient, True);
#endif
#endif
		break;
	case ACTION_INCX:
		movesDsp = 0;
		sizex++;
		printRecord(sizex, sizey, sizez, orient, practice);
#ifdef WINVER
		w->rubik.sizex = sizex;
#else
		XtVaSetValues(w, XtNsizex, sizex, NULL);
#ifdef HAVE_OPENGL
		XtVaSetValues(otherw2, XtNsizex, sizex, NULL);
#endif
		XtVaSetValues(otherw1, XtNsizex, sizex, NULL);
#ifdef HAVE_MOTIF
		if (sizex > MAX_FACETS)
			XtVaSetValues(cubesXSlider, XmNmaximum, sizex, NULL);
		XmScaleSetValue(cubesXSlider, sizex);
#endif
#endif
		break;
	case ACTION_DECX:
		movesDsp = 0;
		sizex--;
		printRecord(sizex, sizey, sizez, orient, practice);
#ifdef WINVER
		w->rubik.sizex = sizex;
#else
		XtVaSetValues(w, XtNsizex, sizex, NULL);
#ifdef HAVE_OPENGL
		XtVaSetValues(otherw2, XtNsizex, sizex, NULL);
#endif
		XtVaSetValues(otherw1, XtNsizex, sizex, NULL);
#ifdef HAVE_MOTIF
		XmScaleSetValue(cubesXSlider, sizex);
		if (sizex >= MAX_FACETS)
			XtVaSetValues(cubesXSlider, XmNmaximum, sizex, NULL);
#endif
#endif
		break;
	case ACTION_INCY:
		movesDsp = 0;
		sizey++;
		printRecord(sizex, sizey, sizez, orient, practice);
#ifdef WINVER
		w->rubik.sizey = sizey;
#else
		XtVaSetValues(w, XtNsizey, sizey, NULL);
#ifdef HAVE_OPENGL
		XtVaSetValues(otherw2, XtNsizey, sizey, NULL);
#endif
		XtVaSetValues(otherw1, XtNsizey, sizey, NULL);
#ifdef HAVE_MOTIF
		if (sizey > MAX_FACETS)
			XtVaSetValues(cubesYSlider, XmNmaximum, sizey, NULL);
		XmScaleSetValue(cubesYSlider, sizey);
#endif
#endif
		break;
	case ACTION_DECY:
		movesDsp = 0;
		sizey--;
		printRecord(sizex, sizey, sizez, orient, practice);
#ifdef WINVER
		w->rubik.sizey = sizey;
#else
		XtVaSetValues(w, XtNsizey, sizey, NULL);
#ifdef HAVE_OPENGL
		XtVaSetValues(otherw2, XtNsizey, sizey, NULL);
#endif
		XtVaSetValues(otherw1, XtNsizey, sizey, NULL);
#ifdef HAVE_MOTIF
		XmScaleSetValue(cubesYSlider, sizey);
		if (sizey >= MAX_FACETS)
			XtVaSetValues(cubesYSlider, XmNmaximum, sizey, NULL);
#endif
#endif
		break;
	case ACTION_INCZ:
		movesDsp = 0;
		sizez++;
		printRecord(sizex, sizey, sizez, orient, practice);
#ifdef WINVER
		w->rubik.sizez = sizez;
#else
		XtVaSetValues(w, XtNsizez, sizez, NULL);
#ifdef HAVE_OPENGL
		XtVaSetValues(otherw2, XtNsizez, sizez, NULL);
#endif
		XtVaSetValues(otherw1, XtNsizez, sizez, NULL);
#ifdef HAVE_MOTIF
		if (sizez > MAX_FACETS)
			XtVaSetValues(cubesZSlider, XmNmaximum, sizez, NULL);
		XmScaleSetValue(cubesZSlider, sizez);
#endif
#endif
		break;
	case ACTION_DECZ:
		movesDsp = 0;
		sizez--;
		printRecord(sizex, sizey, sizez, orient, practice);
#ifdef WINVER
		w->rubik.sizez = sizez;
#else
		XtVaSetValues(w, XtNsizez, sizez, NULL);
#ifdef HAVE_OPENGL
		XtVaSetValues(otherw2, XtNsizez, sizez, NULL);
#endif
		XtVaSetValues(otherw1, XtNsizez, sizez, NULL);
#ifdef HAVE_MOTIF
		XmScaleSetValue(cubesZSlider, sizez);
		if (sizez >= MAX_FACETS)
			XtVaSetValues(cubesZSlider, XmNmaximum, sizez, NULL);
#endif
#endif
		break;
#ifndef WINVER
	case ACTION_UNDO:
		if (callData->control) {
#ifdef HAVE_OPENGL
			XtVaSetValues(otherw2,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNcontrol, callData->control,
				XtNfast, callData->fast,
				XtNstart, True, NULL);
#endif
			XtVaSetValues(otherw1,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNcontrol, callData->control,
				XtNfast, callData->fast,
				XtNstart, True, NULL);
		} else {
			movesDsp--;
#ifdef HAVE_OPENGL
			XtVaSetValues(otherw2,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNcontrol, callData->control,
				XtNfast, callData->fast, NULL);
#endif
			XtVaSetValues(otherw1,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNcontrol, callData->control,
				XtNfast, callData->fast, NULL);
		}
		break;
#endif
#ifndef WINVER
	case ACTION_REDO:
		if (callData->control) {
#ifdef HAVE_OPENGL
			XtVaSetValues(otherw2,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNcontrol, callData->control,
				XtNfast, callData->fast,
				XtNstart, True, NULL);
#endif
			XtVaSetValues(otherw1,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNcontrol, callData->control,
				XtNfast, callData->fast,
				XtNstart, True, NULL);
		} else {
			movesDsp++;
#ifdef HAVE_OPENGL
			XtVaSetValues(otherw2,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNcontrol, callData->control,
				XtNfast, callData->fast, NULL);
#endif
			XtVaSetValues(otherw1,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNcontrol, callData->control,
				XtNfast, callData->fast, NULL);
		}
		break;
#endif
#ifdef WINVER
	case ACTION_DIM:
		dim++;
#ifdef HAVE_OPENGL
		if (dim > 4)
#else
		if (dim > 3)
#endif
			dim = 2;
		w->rubik.dim = dim;
		break;
#endif
	}
#ifdef WINVER
	printStatus(messageDsp, movesDsp, dim, sizex, sizey, sizez);
#else
#ifdef HAVE_MOTIF
	printStatus(messageDsp, movesDsp);
#else
	printStatus(messageDsp, movesDsp, dim, sizex, sizey, sizez);
	printStatus(messageDsp, movesDsp, otherdim1, sizex, sizey, sizez);
#ifdef HAVE_OPENGL
	printStatus(messageDsp, movesDsp, otherdim2, sizex, sizey, sizez);
#endif
#endif
#endif
}

#ifdef WINVER
void
setPuzzleMove(RubikWidget w, int reason, int face, int position, int direction,
		Boolean control, int fast)
{
	int sizex, sizey, sizez;
	/* Boolean orient, practice, cheat; */
	int dim;

	(void) strcpy(messageDsp, "");
	sizex = w->rubik.sizex;
	sizey = w->rubik.sizey;
	sizez = w->rubik.sizez;
#if 0
	orient = w->rubik.orient;
	practice = w->rubik.practice;
	cheat = w->rubik.cheat;
#endif
	dim = w->rubik.dim;
	switch (reason) {
	case ACTION_MOVED:
		if (!control) {
			movesDsp++;
			SET_STARTED(w, True);
		}
		break;
	case ACTION_UNDO:
		if (!control) {
			movesDsp--;
			SET_STARTED(w, True);
		}
		break;
	case ACTION_REDO:
		if (!control) {
			movesDsp++;
			SET_STARTED(w, True);
		}
		break;
	}
	printStatus(messageDsp, movesDsp, dim, sizex, sizey, sizez);
}

static LRESULT CALLBACK
about(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	if (message == WM_COMMAND && LOWORD(wParam) == IDOK) {
		(void) EndDialog(hDlg, TRUE);
		return TRUE;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;

	widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.rubik.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			enterPuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.rubik.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			leavePuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
	case WM_CREATE:
		initialize(&widget, brush);
		break;
	case WM_DESTROY:
		destroyPuzzle(brush);
		break;
	case WM_SIZE:
		resizePuzzle(&widget);
		(void) InvalidateRect(hWnd, NULL, TRUE);
		break;
	case WM_PAINT:
		widget.core.hDC = BeginPaint(hWnd, &paint);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		exposePuzzle(&widget);
		(void) EndPaint(hWnd, &paint);
		break;
	case WM_RBUTTONDOWN:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		randomizePuzzle(&widget);
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
	case WM_LBUTTONDOWN:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		selectPuzzle(&widget, LOWORD(lParam), HIWORD(lParam),
			(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
	case WM_LBUTTONUP:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		releasePuzzle(&widget, LOWORD(lParam), HIWORD(lParam),
			(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
#if (_WIN32_WINNT >= 0x0400) || (_WIN32_WINDOWS > 0x0400)
	case WM_MOUSEWHEEL:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		{
			int zDelta = ((short) HIWORD(wParam));
			POINT cursor, origin;

			origin.x = 0, origin.y = 0;
			ClientToScreen(hWnd, &origin);
			(void) GetCursorPos(&cursor);
			if (zDelta > (WHEEL_DELTA >> 1)) {
				movePuzzleInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					TOP,
					(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
			} else if (zDelta < -(WHEEL_DELTA >> 1)) {
				movePuzzleInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					BOTTOM,
					(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
			}
		}
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
#endif
	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case ACTION_GET:
			getPuzzle(&widget);
			resizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_WRITE:
			writePuzzle(&widget);
			break;
		case ACTION_EXIT:
			destroyPuzzle(brush);
			break;
		case ACTION_HIDE:
			hidePuzzle(&widget);
			break;
		case ACTION_UNDO:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			undoPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_REDO:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			redoPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_CLEAR:
			clearPuzzle(&widget);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_RANDOMIZE:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			randomizePuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_PRACTICE:
			practicePuzzle(&widget);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_SOLVE:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			solvePuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_ORIENTIZE:
			orientizePuzzle(&widget);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_DIM:
			(void) dimPuzzle(&widget);
			resizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_VIEW:
			(void) viewPuzzle(&widget);
			resizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_TOP:
		case ACTION_TR:
		case ACTION_RIGHT:
		case ACTION_BR:
		case ACTION_BOTTOM:
		case ACTION_BL:
		case ACTION_LEFT:
		case ACTION_TL:
		case ACTION_CW:
		case ACTION_CCW:
			{
				POINT cursor, origin;

				widget.core.hDC = GetDC(hWnd);
				(void) SelectObject(widget.core.hDC,
					GetStockObject(NULL_PEN));
				origin.x = 0, origin.y = 0;
				ClientToScreen(hWnd, &origin);
				(void) GetCursorPos(&cursor);
				(void) movePuzzleInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					(int) LOWORD(wParam) - ACTION_TOP,
					FALSE);
				(void) ReleaseDC(hWnd, widget.core.hDC);
			}
			break;
		case ACTION_CONTROL_TOP:
		case ACTION_CONTROL_TR:
		case ACTION_CONTROL_RIGHT:
		case ACTION_CONTROL_BOTTOM:
		case ACTION_CONTROL_BL:
		case ACTION_CONTROL_LEFT:
		case ACTION_CONTROL_CW:
		case ACTION_CONTROL_CCW:
			{
				POINT cursor, origin;

				widget.core.hDC = GetDC(hWnd);
				(void) SelectObject(widget.core.hDC,
					GetStockObject(NULL_PEN));
				origin.x = 0, origin.y = 0;
				ClientToScreen(hWnd, &origin);
				(void) GetCursorPos(&cursor);
				(void) movePuzzleInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					(int) LOWORD(wParam) - ACTION_CONTROL_TOP,
					TRUE);
				(void) ReleaseDC(hWnd, widget.core.hDC);
			}
			break;
		case ACTION_INCREMENT:
			incrementPuzzle(&widget);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_DECREMENT:
			decrementPuzzle(&widget);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_INCX:
			incrementAxisPuzzle(&widget, 'x');
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_DECX:
			incrementAxisPuzzle(&widget, 'X');
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_INCY:
			incrementAxisPuzzle(&widget, 'y');
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_DECY:
			incrementAxisPuzzle(&widget, 'Y');
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_INCZ:
			incrementAxisPuzzle(&widget, 'z');
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_DECZ:
			incrementAxisPuzzle(&widget, 'Z');
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_SPEED:
			speedUpPuzzle(&widget);
			break;
		case ACTION_SLOW:
			slowDownPuzzle(&widget);
			break;
		case ACTION_SOUND:
			toggleSoundPuzzle(&widget);
			break;
		case ACTION_DESCRIPTION:
			(void) MessageBox(hWnd, descriptionHelp,
				"Description", MB_OK);
			break;
		case ACTION_FEATURES:
			(void) MessageBox(hWnd, featuresHelp,
				"Features", MB_OK);
			break;
		case ACTION_REFERENCES:
			(void) MessageBox(hWnd, referencesHelp,
				"References", MB_OK);
			break;
		case ACTION_ABOUT:
			(void) DialogBox(widget.core.hInstance,
				"About", hWnd, (DLGPROC) about);
			break;
		}
		break;
	default:
		return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(void *) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (msg.wParam);
}

#else
static void
usage(char *programName)
{
	unsigned int i;

	(void) fprintf(stderr, "usage: %s\n", programName);
	for (i = 0; i < strlen(optionsHelp); i++) {
		if (i == 0 || optionsHelp[i - 1] == '\n')
			(void) fprintf(stderr, "\t");
		(void) fprintf(stderr, "%c", (optionsHelp[i]));
	}
	(void) fprintf(stderr, "\n");
	exit(1);
}

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*rubik.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*rubik.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*rubik.reverseVideo", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*rubik.reverseVideo", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*rubik.reverseVideo", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*rubik.reverseVideo", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "rubik.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "rubik.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*rubik.pieceBorder", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*rubik.pieceBorder", XrmoptionSepArg, NULL},
	{(char *) "-face0", (char *) "*rubik.faceColor0", XrmoptionSepArg, NULL},
	{(char *) "-face1", (char *) "*rubik.faceColor1", XrmoptionSepArg, NULL},
	{(char *) "-face2", (char *) "*rubik.faceColor2", XrmoptionSepArg, NULL},
	{(char *) "-face3", (char *) "*rubik.faceColor3", XrmoptionSepArg, NULL},
	{(char *) "-face4", (char *) "*rubik.faceColor4", XrmoptionSepArg, NULL},
	{(char *) "-face5", (char *) "*rubik.faceColor5", XrmoptionSepArg, NULL},
	{(char *) "-delay", (char *) "*rubik.delay", XrmoptionSepArg, NULL},
	{(char *) "-sound", (char *) "*rubik.sound", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nosound", (char *) "*rubik.sound", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fn", (char *) "*rubik.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*rubik.font", XrmoptionSepArg, NULL},
	{(char *) "-view", (char *) "*rubik.view", XrmoptionSepArg, NULL},
	{(char *) "-sizex", (char *) "*rubik.sizex", XrmoptionSepArg, NULL},
	{(char *) "-sizex", (char *) "*rubik.sizex", XrmoptionSepArg, NULL},
	{(char *) "-sizey", (char *) "*rubik.sizey", XrmoptionSepArg, NULL},
	{(char *) "-sizez", (char *) "*rubik.sizez", XrmoptionSepArg, NULL},
	{(char *) "-orient", (char *) "*rubik.orient", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noorient", (char *) "*rubik.orient", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-practice", (char *) "*rubik.practice", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nopractice", (char *) "*rubik.practice", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-userName", (char *) "*rubik.userName", XrmoptionSepArg, NULL},
	{(char *) "-scoreFile", (char *) "*rubik.scoreFile", XrmoptionSepArg, NULL},
	{(char *) "-scores", (char *) "*rubik.scoreOnly", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-version", (char *) "*rubik.versionOnly", XrmoptionNoArg, (char *) "TRUE"}
};

#ifdef HAVE_MOTIF
static void
puzzlePracticeListener(Widget w, XtPointer clientData,
		XmAnyCallbackStruct *cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtVaSetValues(puzzle2d, XtNmenu, ACTION_PRACTICE, NULL);
	}
}

static void
puzzleRandomizeListener(Widget w, XtPointer clientData,
		XmAnyCallbackStruct *cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtVaSetValues(puzzle2d, XtNmenu, ACTION_RANDOMIZE, NULL);
	}
}

static void
cubesXChangeListener(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int sizex = cbs->value, old, sizey, sizez;
	Boolean orient, practice;

	XtVaGetValues(puzzle2d,
		XtNsizex, &old,
		XtNsizey, &sizey,
		XtNsizez, &sizez,
		XtNorient, &orient,
		XtNpractice, &practice, NULL);
	if (old != sizex) {
		XtVaSetValues(puzzle2d,
			XtNsizex, sizex, NULL);
		XtVaSetValues(puzzle3d,
			XtNsizex, sizex, NULL);
#ifdef HAVE_OPENGL
		XtVaSetValues(puzzleGL,
			XtNsizex, sizex, NULL);
#endif
		movesDsp = 0;
		(void) sprintf(buff, "%d", movesDsp);
		printState(movesText, buff);
		printRecord(sizex, sizey, sizez, orient, practice);
		(void) strcpy(messageDsp, "");
		printState(message, messageDsp);
	}
}

static void
cubesYChangeListener(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int sizey = cbs->value, old, sizex, sizez;
	Boolean orient, practice;

	XtVaGetValues(puzzle2d,
		XtNsizex, &sizex,
		XtNsizey, &old,
		XtNsizez, &sizez,
		XtNorient, &orient,
		XtNpractice, &practice, NULL);
	if (old != sizey) {
		XtVaSetValues(puzzle2d,
			XtNsizey, sizey, NULL);
		XtVaSetValues(puzzle3d,
			XtNsizey, sizey, NULL);
#ifdef HAVE_OPENGL
		XtVaSetValues(puzzleGL,
			XtNsizey, sizey, NULL);
#endif
		movesDsp = 0;
		(void) sprintf(buff, "%d", movesDsp);
		printState(movesText, buff);
		printRecord(sizex, sizey, sizez, orient, practice);
		(void) strcpy(messageDsp, "");
		printState(message, messageDsp);
	}
}

static void
cubesZChangeListener(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int sizez = cbs->value, old, sizex, sizey;
	Boolean orient, practice;

	XtVaGetValues(puzzle2d,
		XtNsizex, &sizex,
		XtNsizey, &sizey,
		XtNsizez, &old,
		XtNorient, &orient,
		XtNpractice, &practice, NULL);
	if (old != sizez) {
		XtVaSetValues(puzzle2d,
			XtNsizez, sizez, NULL);
		XtVaSetValues(puzzle3d,
			XtNsizez, sizez, NULL);
#ifdef HAVE_OPENGL
		XtVaSetValues(puzzleGL,
			XtNsizez, sizez, NULL);
#endif
		movesDsp = 0;
		(void) sprintf(buff, "%d", movesDsp);
		printState(movesText, buff);
		printRecord(sizex, sizey, sizez, orient, practice);
		(void) strcpy(messageDsp, "");
		printState(message, messageDsp);
	}
}

static void
orientToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct *cbs)
{
	int sizex, sizey, sizez;
	Boolean orient = cbs->set, practice;

	XtVaGetValues(puzzle2d,
		XtNsizex, &sizex,
		XtNsizey, &sizey,
		XtNsizez, &sizez,
		XtNpractice, &practice, NULL);
	XtVaSetValues(puzzle2d,
		XtNorient, orient, NULL);
	XtVaSetValues(puzzle3d,
		XtNorient, orient, NULL);
#ifdef HAVE_OPENGL
	XtVaSetValues(puzzleGL,
		XtNorient, orient, NULL);
#endif
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	printState(movesText, buff);
	printRecord(sizex, sizey, sizez, orient, practice);
	(void) strcpy(messageDsp, "");
	printState(message, messageDsp);
}

static void
practiceToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct *cbs)
{
	int sizex, sizey, sizez;
	Boolean orient, practice = cbs->set;

	XtVaSetValues(puzzle2d,
		XtNpractice, practice,
		XtNstart, False, NULL);
	XtVaSetValues(puzzle3d,
		XtNpractice, practice,
		XtNstart, False, NULL);
#ifdef HAVE_OPENGL
	XtVaSetValues(puzzleGL,
		XtNpractice, practice,
		XtNstart, False, NULL);
#endif
	XtVaGetValues(puzzle2d,
		XtNsizex, &sizex,
		XtNsizey, &sizey,
		XtNsizez, &sizez,
		XtNpractice, &orient, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	printState(movesText, buff);
	printRecord(sizex, sizey, sizez, orient, practice);
	if (practice)
		(void) strcpy(messageDsp, "");
	else
		(void) strncpy(messageDsp, "Randomize to start", MESSAGE_LENGTH);
	printState(message, messageDsp);
}

static void
speedChangeListener(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int delay = MAX_SPEED + MIN_SPEED - cbs->value - 1, oldDelay;

	XtVaGetValues(puzzle2d,
		XtNdelay, &oldDelay, NULL);
	if (oldDelay != delay) {
#ifdef HAVE_OPENGL
		XtVaSetValues(puzzleGL, XtNdelay, delay, NULL);
#endif
		XtVaSetValues(puzzle3d, XtNdelay, delay, NULL);
		XtVaSetValues(puzzle2d, XtNdelay, delay, NULL);
	}
}

static void
fileMenuListener(Widget w, void *value, void *clientData)
{
	int val = (int) value + ACTION_GET;

	if (val == ACTION_EXIT)
		exit(0);
	XtVaSetValues(puzzle2d, XtNmenu, val, NULL);
}

static void
playCB(Widget w, void *value, void *clientData)
{
	int val = (int) value + ACTION_UNDO;

	XtVaSetValues(puzzle2d, XtNmenu, val, NULL);
}

static Widget
createQuery(Widget w, char *text, char *title, XtCallbackProc callback)
{
	Widget button, messageBox;
	char titleDsp[FILE_NAME_LENGTH + 8];
	XmString titleString = NULL, messageString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNmessageString, messageString);
	messageBox = XmCreateWarningDialog(w, (char *) "queryBox",
		arg, 2);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(messageString);
	XtAddCallback(messageBox, XmNokCallback, callback, (XtPointer) NULL);
	XtAddCallback(messageBox, XmNcancelCallback, callback,
		(XtPointer) NULL);
	return messageBox;
}

static Widget
createHelp(Widget w, char *text, char *title)
{
	Widget button, messageBox;
	char titleDsp[FILE_NAME_LENGTH + 8];
	XmString titleString = NULL, messageString = NULL, buttonString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	buttonString = XmStringCreateSimple((char *) "OK");
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNokLabelString, buttonString);
	XtSetArg(arg[2], XmNmessageString, messageString);
	messageBox = XmCreateInformationDialog(w, (char *) "helpBox",
		arg, 3);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(buttonString);
	XmStringFree(messageString);
	return messageBox;
}

static void
helpMenuListener(Widget w, XtPointer value, XtPointer clientData)
{
	int val = (int) value;

	switch (val) {
	case 0:
		XtManageChild(descriptionDialog);
		break;
	case 1:
		XtManageChild(featuresDialog);
		break;
	case 2:
		XtManageChild(optionsDialog);
		break;
	case 3:
		XtManageChild(referencesDialog);
		break;
	case 4:
		XtManageChild(aboutDialog);
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpMenuListener: %d", val);
			XtWarning(buf);
			free(buf);
		}
	}
}
#endif

int
main(int argc, char **argv)
{
	int pixmapSize;
#ifdef HAVE_MOTIF
	Widget menuBar, pullDownMenu, widget;
	Widget menuBarPanel, mainPanel, controlPanel;
	Widget movesRowCol, cubesRowCol, switchRowCol, messageRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString undoString, redoString;
	XmString clearString, randomizeString, practiceString, solveString;
	XmString incrementString, decrementString, orientizeString;
	XmString speedString, slowString, soundString, viewString;
#endif

	progDsp = argv[0];
	topLevel = XtInitialize(argv[0], "Rubik",
		options, XtNumber(options), &argc, argv);
	if (argc != 1)
		usage(argv[0]);

#ifdef HAVE_MOTIF
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P',
		NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Exit");
	(void) XmVaCreateSimplePulldownMenu(menuBar, (char *) "file_menu",
		0, fileMenuListener,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'x', NULL, NULL,
		NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	undoString = XmStringCreateSimple((char *) "Undo");
	redoString = XmStringCreateSimple((char *) "Redo");
	clearString = XmStringCreateSimple((char *) "Clear");
	practiceString = XmStringCreateSimple((char *) "Practice");
	randomizeString = XmStringCreateSimple((char *) "Randomize");
	solveString = XmStringCreateSimple((char *) "Auto-solve");
	incrementString = XmStringCreateSimple((char *) "Increment Size");
	decrementString = XmStringCreateSimple((char *) "Decrement Size");
	orientizeString = XmStringCreateSimple((char *) "Orientize");
	speedString = XmStringCreateSimple((char *) "Speed >");
	slowString = XmStringCreateSimple((char *) "Slow <");
	soundString = XmStringCreateSimple((char *) "Sound @");
	viewString = XmStringCreateSimple((char *) "View");
	(void) XmVaCreateSimplePulldownMenu(menuBar, (char *) "play_menu",
		1, playCB,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, redoString, 'R', NULL, NULL,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
		XmVaPUSHBUTTON, practiceString, 'P', NULL, NULL,
		XmVaPUSHBUTTON, randomizeString, 'z', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 's', NULL, NULL,
		XmVaPUSHBUTTON, incrementString, 'I', NULL, NULL,
		XmVaPUSHBUTTON, decrementString, 'D', NULL, NULL,
		XmVaPUSHBUTTON, orientizeString, 'O', NULL, NULL,
		XmVaPUSHBUTTON, speedString, '>', NULL, NULL,
		XmVaPUSHBUTTON, slowString, '<', NULL, NULL,
		XmVaPUSHBUTTON, soundString, '@', NULL, NULL,
		XmVaPUSHBUTTON, viewString, 'V', NULL, NULL,
		NULL);
	XmStringFree(undoString);
	XmStringFree(redoString);
	XmStringFree(clearString);
	XmStringFree(practiceString);
	XmStringFree(randomizeString);
	XmStringFree(solveString);
	XmStringFree(incrementString);
	XmStringFree(decrementString);
	XmStringFree(orientizeString);
	XmStringFree(speedString);
	XmStringFree(slowString);
	XmStringFree(soundString);
	XmStringFree(viewString);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL);
	XtVaSetValues(menuBar, XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuListener, (char *) 0);
	widget = XtVaCreateManagedWidget("Features",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuListener, (char *) 1);
	widget = XtVaCreateManagedWidget("Options",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'O', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuListener, (char *) 2);
	widget = XtVaCreateManagedWidget("References",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuListener, (char *) 3);
	widget = XtVaCreateManagedWidget("About",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuListener, (char *) 4);
	XtManageChild(menuBar);
	descriptionDialog = createHelp(menuBar, (char *) descriptionHelp,
		(char *) "Description");
	featuresDialog = createHelp(menuBar, (char *) featuresHelp,
		(char *) "Features");
	optionsDialog = createHelp(menuBar, (char *) optionsHelp,
		(char *) "Options");
	referencesDialog = createHelp(menuBar, (char *) referencesHelp,
		(char *) "References");
	aboutDialog = createHelp(menuBar, (char *) aboutHelp,
		(char *) "About");
	solveDialog = createHelp(menuBar, (char *) solveHelp,
		(char *) "Solve");
	practiceDialog = createQuery(topLevel,
		(char *) "Are you sure you want to toggle the practice mode?",
		(char *) "Practice Query",
		(XtCallbackProc) puzzlePracticeListener);
	randomizeDialog = createQuery(topLevel,
		(char *) "Are you sure you want to randomize?",
		(char *) "Randomize Query",
		(XtCallbackProc) puzzleRandomizeListener);
	mainPanel = XtCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel,
		NULL, 0);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move", 5, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Randomize", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	movesText = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	recordText = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);

	cubesRowCol = XtVaCreateManagedWidget("cubesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	cubesXSlider = XtVaCreateManagedWidget("cubesXSlider",
		xmScaleWidgetClass, cubesRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Size X", 7,
		XmNminimum, MIN_FACETS,
		XmNmaximum, MAX_FACETS,
		XmNvalue, MIN_FACETS,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(cubesXSlider, XmNvalueChangedCallback,
		(XtCallbackProc) cubesXChangeListener, (XtPointer) NULL);
	cubesYSlider = XtVaCreateManagedWidget("cubesYSlider",
		xmScaleWidgetClass, cubesRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Size Y", 7,
		XmNminimum, MIN_FACETS,
		XmNmaximum, MAX_FACETS,
		XmNvalue, MIN_FACETS,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(cubesYSlider, XmNvalueChangedCallback,
		(XtCallbackProc) cubesYChangeListener, (XtPointer) NULL);
	cubesZSlider = XtVaCreateManagedWidget("cubesZSlider",
		xmScaleWidgetClass, cubesRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Size Z", 7,
		XmNminimum, MIN_FACETS,
		XmNmaximum, MAX_FACETS,
		XmNvalue, MIN_FACETS,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(cubesZSlider, XmNvalueChangedCallback,
		(XtCallbackProc) cubesZChangeListener, (XtPointer) NULL);
	switchRowCol = XtVaCreateManagedWidget("switchRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	orientizeSwitch = XtVaCreateManagedWidget("Oriented",
		xmToggleButtonWidgetClass, switchRowCol,
		XmNset, DEFAULT_ORIENT, NULL);
	XtAddCallback(orientizeSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) orientToggle, (XtPointer) NULL);
	practiceSwitch = XtVaCreateManagedWidget("Practice",
		xmToggleButtonWidgetClass, switchRowCol,
		XmNset, DEFAULT_PRACTICE, NULL);
	XtAddCallback(practiceSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) practiceToggle, (XtPointer) NULL);
	speedSlider = XtVaCreateManagedWidget("speedSlider",
		xmScaleWidgetClass, switchRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Animation Speed", 16,
		XmNminimum, MIN_SPEED,
		XmNmaximum, MAX_SPEED,
		XmNvalue, MAX_SPEED - 10,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(speedSlider, XmNvalueChangedCallback,
		(XtCallbackProc) speedChangeListener, (XtPointer) NULL);
	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	message = XtVaCreateManagedWidget("Play Rubik's Cube! (use mouse and keypad)",
		xmLabelWidgetClass, messageRowCol, NULL);
	puzzle2d = XtCreateManagedWidget("rubik",
		rubik2dWidgetClass, mainPanel, NULL, 0);
	XtVaSetValues(puzzle2d,
		XtNheight, 200, NULL);
	XtAddCallback(puzzle2d, XtNselectCallback,
		(XtCallbackProc) puzzleListener, (XtPointer) NULL);
	puzzle3d = XtCreateManagedWidget("rubik",
		rubik3dWidgetClass, mainPanel, NULL, 0);
	XtVaSetValues(puzzle3d,
		XtNheight, 200, NULL);
#ifdef HAVE_OPENGL
	puzzleGL = XtCreateManagedWidget("rubik",
		rubikGLWidgetClass, mainPanel, NULL, 0);
	XtVaSetValues(puzzleGL,
		XtNheight, 200, NULL);
#endif
#else
	shell = XtCreateApplicationShell(argv[0],
		topLevelShellWidgetClass, NULL, 0);
#ifdef HAVE_OPENGL
	shellGL = XtCreateApplicationShell(argv[0],
		topLevelShellWidgetClass, NULL, 0);
#endif
	puzzle2d = XtCreateManagedWidget("rubik",
		rubik2dWidgetClass, topLevel, NULL, 0);
	puzzle3d = XtCreateManagedWidget("rubik",
		rubik3dWidgetClass, shell, NULL, 0);
#ifdef HAVE_OPENGL
	puzzleGL = XtCreateManagedWidget("rubik",
		rubikGLWidgetClass, shellGL, NULL, 0);
#endif
#endif
	XtVaGetValues(puzzle2d,
		XtNpixmapSize, &pixmapSize, NULL);
#ifdef HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel, XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			RESIZE_XPM(pixmapSize), &pixmap, NULL,
			&xpmAttributes);
	}
	if (pixmap == (Pixmap) NULL)
#endif
		pixmap = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			DEFINE_XBM);
#ifdef HAVE_MOTIF
	XtVaSetValues(topLevel,
		XmNkeyboardFocusPolicy, XmPOINTER, /* not XmEXPLICIT */
		XtNiconPixmap, pixmap, NULL);
#else
	XtVaSetValues(topLevel,
		XtNinput, True,
		XtNiconPixmap, pixmap, NULL);
	XtVaSetValues(shell,
		XtNinput, True,
		XtNiconPixmap, pixmap, NULL);
#endif
	XtAddCallback(puzzle3d, XtNselectCallback,
		(XtCallbackProc) puzzleListener, (XtPointer) NULL);
#ifdef HAVE_OPENGL
#ifndef HAVE_MOTIF
	XtVaSetValues(shellGL,
		XtNinput, True,
		XtNiconPixmap, pixmap, NULL);
#endif
	XtAddCallback(puzzleGL, XtNselectCallback,
		(XtCallbackProc) puzzleListener, (XtPointer) NULL);
#endif
	initialize();
	XtRealizeWidget(topLevel);
#ifndef HAVE_MOTIF
	XtRealizeWidget(shell);
#ifdef HAVE_OPENGL
	XtRealizeWidget(shellGL);
#endif
#endif
	XGrabButton(XtDisplay(puzzle2d), (unsigned int) AnyButton, AnyModifier,
		XtWindow(puzzle2d), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(puzzle2d),
		XCreateFontCursor(XtDisplay(puzzle2d), XC_hand2));
	XGrabButton(XtDisplay(puzzle3d), (unsigned int) AnyButton, AnyModifier,
		XtWindow(puzzle3d), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(puzzle3d),
		XCreateFontCursor(XtDisplay(puzzle3d), XC_hand2));
#ifdef HAVE_OPENGL
	XGrabButton(XtDisplay(puzzleGL), (unsigned int) AnyButton, AnyModifier,
		XtWindow(puzzleGL), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(puzzleGL),
		XCreateFontCursor(XtDisplay(puzzleGL), XC_crosshair));
#endif
	XtMainLoop();

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}
#endif
