/*-
# X-BASED PYRAMINX(tm)
#
#  PyraminxS.c
###
#
#  Taken from the algorithm in The Simple Solutions to Cubic Puzzles
#  by James G. Nourse (also looked at Puzzle It Out: Cubes, Groups
#  and Puzzles by John Ewing & Czes Kosniowski)
#  Break ability taken from the X puzzle by Don Bennett, HP Labs
#
#  Copyright (c) 1999 - 2007	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Solver file for Pyraminx */

#include "rngs.h"
#define JMP
#ifdef JMP
#include <setjmp.h> /* longjmp ... interrupt */
#endif
#include "PyraminxP.h"

/* Moving Nourse's large corners translates as:
 * move the whole puzzle
 * undo the face move */
#define RotateLittleCornerCW(w,f) MovePuzzlePiece(w,f,0,(f%2)?TR:BL,PERIOD3,FALSE)
#define RotateLittleCornerCCW(w,f) MovePuzzlePiece(w,f,0,(f%2)?BL:TR,PERIOD3,FALSE)
#define RotateCornerCW(w,f) MovePuzzlePiece(w,f,1,(f%2)?TR:BL,PERIOD3,TRUE); \
	MovePuzzlePiece(w,f,6,(f%2)?BL:TR,PERIOD3,FALSE)
#define RotateCornerCCW(w,f) MovePuzzlePiece(w,f,1,(f%2)?BL:TR,PERIOD3,TRUE); \
	MovePuzzlePiece(w,f,6,(f%2)?TR:BL,PERIOD3,FALSE)

#define topFaceTrans(tf,f) ((tf==0)?f:((tf==1)?MAXFACES-1-(f+MAXFACES/2)%MAXFACES:((tf==2)?(f+MAXFACES/2)%MAXFACES:MAXFACES-1-f)))

#define P_PLUS RotateCornerCW(w,topFaceTrans(topFace,0)) /* Posterior Corner CW */
#define B_PLUS RotateCornerCW(w,topFaceTrans(topFace,1)) /* Bottom Corner CW */
#define L_PLUS RotateCornerCW(w,topFaceTrans(topFace,2)) /* Left Corner CW */
#define R_PLUS RotateCornerCW(w,topFaceTrans(topFace,3)) /* Right Corner CW */
#define P_MINUS RotateCornerCCW(w,topFaceTrans(topFace,0)) /* Posterior Corner CCW */
#define B_MINUS RotateCornerCCW(w,topFaceTrans(topFace,1)) /* Bottom Corner CCW */
#define L_MINUS RotateCornerCCW(w,topFaceTrans(topFace,2)) /* Left Corner CCW */
#define R_MINUS RotateCornerCCW(w,topFaceTrans(topFace,3)) /* Right Corner CCW */

#define RotateWholeCW(w,f) MovePuzzlePiece(w,f,1,(f%2)?TR:BL,PERIOD3,TRUE)
#define RotateWholeCCW(w,f) MovePuzzlePiece(w,f,1,(f%2)?BL:TR,PERIOD3,TRUE)
#define B_PLUS_WHOLE RotateWholeCW(w,topFaceTrans(topFace,1)) /* Bottom Corner CW */
#define B_MINUS_WHOLE RotateWholeCCW(w,topFaceTrans(topFace,1)) /* Bottom Corner CCW */


typedef struct _PyraminxSLoc {
	int face, position;
} PyraminxSLoc;

static int IndexToEdges[3] =
{ 1, 3, 6};

static int EdgesToIndex[9] =
{ 3, 0, 3, 1, 3, 3, 2, 3, 3};

static PyraminxSLoc EdgeMate[MAXFACES][3] =
{
	{
		{2, 1},
		{3, 3},
		{1, 6},
	},
	{
		{3, 1},
		{2, 3},
		{0, 6},
	},
	{
		{0, 1},
		{1, 3},
		{3, 6},
	},
	{
		{1, 1},
		{0, 3},
		{2, 6},
	}
};

static Boolean SolvingFlag = False;
/* TODO Interruptability for Windows */
#ifdef JMP
static Boolean AbortSolvingFlag = False;
static jmp_buf solve_env;

static void
AbortSolving(void)
{
	if (SolvingFlag)
		AbortSolvingFlag = True;
}

#ifdef WINVER
static Boolean
ProcessMessage(UINT msg)
{
	switch (msg) {
	case WM_KEYDOWN:
	case WM_CLOSE:
	case WM_LBUTTONDOWN:
	case WM_RBUTTONDOWN:
		AbortSolving();
		return True;
	default:
		return False;
	}
}
#else
static void
ProcessButton(void /*XButtonEvent *event*/)
{
	AbortSolving();
}

static void
ProcessVisibility(XVisibilityEvent *event)
{
	if (event->state != VisibilityUnobscured)
		AbortSolving();
}

static void
GetNextEvent(PyraminxWidget w, XEvent *event)
{
	if (!XCheckMaskEvent(XtDisplay(w), VisibilityChangeMask, event))
		(void) XNextEvent(XtDisplay(w), event);
}

static void
ProcessEvent(XEvent *event)
{
	switch(event->type) {
	case KeyPress:
	case ButtonPress:
		ProcessButton(/*&event->xbutton*/);
		break;
	case VisibilityNotify:
		ProcessVisibility(&event->xvisibility);
		break;
	default:
		break;
	}
}

static void
ProcessEvents(PyraminxWidget w)
{
	XEvent event;

	while (XPending(XtDisplay(w))) {
		GetNextEvent(w, &event);
		ProcessEvent(&event);
	}
}
#endif
#endif

static void
MovePuzzlePiece(PyraminxWidget w, int face, int position,
	int direction, int style, int control)
{
#ifdef JMP
#ifdef WINVER
	MSG msg;

	if (PeekMessage(&msg, NULL, 0, 0, 0)) {
		if (!ProcessMessage(msg.message)) {
			if (GetMessage(&msg, NULL, 0, 0))
				DispatchMessage(&msg);
		}
	}
#else
	ProcessEvents(w);
#endif
	if (SolvingFlag && AbortSolvingFlag)
		longjmp(solve_env, 1);
#endif
	MovePuzzleDelay(w, face, position, direction, style, control);
}

/* This is impossible to do with Drag and Drop as implemented.
 * Use the keypad.  Turn the little corners so that the
 * colors match the adjacent pieces all around. */
static void
OrientLittleCorners(PyraminxWidget w)
{
	int face, currentColor;

	for (face = 0; face < MAXFACES; face++) {
		currentColor = w->pyraminx.facetLoc[face][0].face;
		if (currentColor != w->pyraminx.facetLoc[face][2].face) {
			if (currentColor == w->pyraminx.facetLoc[MAXFACES - face - 1][w->pyraminx.sizeSize - 2].face) {
				RotateLittleCornerCCW(w,face);
			} else {
				RotateLittleCornerCW(w,face);
			}
		}
	}
}

static int
FindFaceColor(PyraminxWidget w, int face)
{
	int colors[3], color, cornerFace;

	switch (face) {
		case 0:
			colors[0] = w->pyraminx.facetLoc[1][2].face;
			colors[1] = w->pyraminx.facetLoc[3][5].face;
			colors[2] = w->pyraminx.facetLoc[2][7].face;
			break;
		case 1:
			colors[0] = w->pyraminx.facetLoc[0][2].face;
			colors[1] = w->pyraminx.facetLoc[2][5].face;
			colors[2] = w->pyraminx.facetLoc[3][7].face;
			break;
		case 2:
			colors[0] = w->pyraminx.facetLoc[3][2].face;
			colors[1] = w->pyraminx.facetLoc[1][5].face;
			colors[2] = w->pyraminx.facetLoc[0][7].face;
			break;
		case 3:
			colors[0] = w->pyraminx.facetLoc[2][2].face;
			colors[1] = w->pyraminx.facetLoc[0][5].face;
			colors[2] = w->pyraminx.facetLoc[1][7].face;
			break;
		default:
			(void) printf("Wrong face %d.\n", face);
	}
	for (color = 0; color < MAXFACES; color++) {
		for (cornerFace = 0; cornerFace < 3; cornerFace++) {
			if (color == colors[cornerFace])
				break;
		}
		if (cornerFace == 3)
			return color;
	}
	(void) printf("No opposite color found!\n");
	return 0;
}

static Boolean
CheckPiece(PyraminxWidget w, int color, int face, int position)
{
	int newFace, newPosition, positionIndex;

	positionIndex = EdgesToIndex[position];
	if (positionIndex == 3) {
		positionIndex = 0;
		(void) printf("position %d incorrect\n", position);
	}
	newFace = EdgeMate[face][positionIndex].face;
	newPosition = EdgeMate[face][positionIndex].position;
	if (w->pyraminx.facetLoc[newFace][newPosition].face == color) {
		return True;
	}
	return False;
}

static void
FindPiece(PyraminxWidget w, int color0, int color1, int *face, int *position)
{
	int faceSkip, positionIndex;

	/* Check starting face first */
	for (positionIndex = 0; positionIndex < 3; positionIndex++) {
		*position = IndexToEdges[positionIndex];
		if (w->pyraminx.facetLoc[*face][*position].face == color0) {
			if (CheckPiece(w, color1, *face, *position))
				return;
		}
		if (w->pyraminx.facetLoc[*face][*position].face == color1) {
			if (CheckPiece(w, color0, *face, *position))
				return;
		}
	}
	faceSkip = *face;
	for (*face = 0; *face < MAXFACES; (*face)++) {
		if (*face != faceSkip) {
			for (positionIndex = 0; positionIndex < 3; positionIndex++) {
				*position = IndexToEdges[positionIndex];
				if (w->pyraminx.facetLoc[*face][*position].face == color0) {
					if (CheckPiece(w, color1, *face, *position))
						return;
				}
				if (w->pyraminx.facetLoc[*face][*position].face == color1) {
					if (CheckPiece(w, color0, *face, *position))
						return;
				}
			}
		}
	}
	(void) printf("Piece %d %d not found!\n", color0, color1);
}

static void
FaceCorners(PyraminxWidget w, int face, int faceColor)
{
	int otherFace;

	if (faceColor != w->pyraminx.facetLoc[face][2].face) {
		if (faceColor == w->pyraminx.facetLoc[MAXFACES - 1 - face][7].face) {
			RotateCornerCW(w,face);
		} else {
			RotateCornerCCW(w,face);
		}
	}
	otherFace = (face + 2) % 4;
	if (faceColor != w->pyraminx.facetLoc[face][5].face) {
		if (faceColor == w->pyraminx.facetLoc[otherFace][2].face) {
			RotateCornerCW(w,otherFace);
		} else {
			RotateCornerCCW(w,otherFace);
		}
	}
	otherFace = MAXFACES - 1 - face;
	if (faceColor != w->pyraminx.facetLoc[face][7].face) {
		if (faceColor == w->pyraminx.facetLoc[otherFace][2].face) {
			RotateCornerCCW(w,otherFace);
		} else {
			RotateCornerCW(w,otherFace);
		}
	}
}

#if 0
/* This is a general approach that seems unwieldly right now. */
static void
FaceEdge(PyraminxWidget w, int topFace, int position, int faceColor)
{
	int color0, color1, newFace, newPosition;

	color0 = w->pyraminx.facetLoc[topFace][2].face;
	switch (position) {
		case 1:
			newFace = (topFace + 2) % 4;
			newPosition = position + 1;
			break;
		case 3:
			newFace = MAXFACES - 1 - topFace;
			newPosition = position - 1;
			break;
		case 6:
			newFace = MAXFACES - 1 - ((topFace + 2) % 4);
			newPosition = position - 1;
			break;
		default:
			(void) printf("Wrong position %d.\n", position);
	}
	color1 = w->pyraminx.facetLoc[newFace][newPosition].face;
	/* newFace and newPosition variables are reused here */
	newFace = topFace; /* This makes narrows down cases */
	FindPiece(w, color0, color1, &newFace, &newPosition);
	(void) printf("topFace %d, position %d, newFace %d, newPosition %d, color0 %d, color1 %d\n", topFace, position, newFace, newPosition, color0, color1);
}
#else
static void
FaceEdge6(PyraminxWidget w, int topFace, int faceColor)
{
	int color0, color1, newFace, newPosition, position;

	position = 6;
	color0 = w->pyraminx.facetLoc[topFace][2].face;
	newFace = MAXFACES - 1 - ((topFace + 2) % 4);
	newPosition = position - 1;
	color1 = w->pyraminx.facetLoc[newFace][newPosition].face;
	/* newFace and newPosition variables are reused here */
	newFace = topFace; /* This makes narrows down cases below */
	FindPiece(w, color0, color1, &newFace, &newPosition);
	if (newFace == topFace) {
		switch (newPosition) {
			case 1: /* Move LT to FT */
				L_MINUS;
				B_PLUS;
				L_MINUS;
				B_MINUS;
				L_MINUS;
				break;
			case 3: /* Move FR to FT */
				R_PLUS;
				B_MINUS;
				R_PLUS;
				B_PLUS;
				R_PLUS;
				break;
			case 6: /* Move FT to FT */
				/* No move :) */
				break;
			default:
				(void) printf("Wrong new position %d.\n", newPosition);
		}
	} else { /* Little more figuring... */
		if ((newFace == topFaceTrans(topFace, 1) || newFace == topFaceTrans(topFace, 2)) && newPosition == 3) { /* Move FL to FT */
			R_MINUS;
			B_PLUS;
			R_PLUS;
		} else if ((newFace == topFaceTrans(topFace, 1) || newFace == topFaceTrans(topFace, 3)) && newPosition == 1) { /* Move FR to FT */
			L_PLUS;
			B_MINUS;
			L_MINUS;
		} else if ((newFace == topFaceTrans(topFace, 2) || newFace == topFaceTrans(topFace, 3)) && newPosition == 6) { /* Move LR to FT */
			R_MINUS;
			B_MINUS;
			R_PLUS;
		} else
			(void) printf("Wrong new face %d or new position %d.\n", newFace, newPosition);
	}
		/* Not correctly positioned... need to flip it */
	if (faceColor != w->pyraminx.facetLoc[topFace][position].face) {
		R_MINUS;
		B_PLUS;
		R_PLUS;
		L_PLUS;
		B_PLUS;
		L_MINUS;
	}
}
#endif

static int
TopFace(PyraminxWidget w)
{
	/* Pick a face (0) and determine which color it is by looking at its
	 * opposite corner.  Whatever color is not there is what this
	 * face should be.  Turn big corners to match */
	int topFace, faceColor, aBottomFace;

	/* Could have hard coded at 0 but this way is more challenging */
	topFace = NRAND(4);
	faceColor = FindFaceColor(w, topFace);
	FaceCorners(w, topFace, faceColor);
#if 0
	FaceEdge(w, topFace, 1, faceColor);
	FaceEdge(w, topFace, 3, faceColor);
	FaceEdge(w, topFace, 6, faceColor);
#else
	FaceEdge6(w, topFace, faceColor);
	aBottomFace = MAXFACES - 1 - ((topFace + 2) % 4);
	MovePuzzlePiece(w, aBottomFace, 6, (topFace % 2) ? TR : BL, PERIOD3, TRUE);
	FaceEdge6(w, topFace, faceColor);
	MovePuzzlePiece(w, aBottomFace, 6, (topFace % 2) ? TR : BL, PERIOD3, TRUE);
	FaceEdge6(w, topFace, faceColor);
#endif
	return topFace;
}

static void
AlignBigCorner(PyraminxWidget w, int topFace)
{
	int frontColor, leftColor, rightColor;

	frontColor = w->pyraminx.facetLoc[topFaceTrans(topFace, 1)][7].face;
	leftColor = w->pyraminx.facetLoc[topFaceTrans(topFace, 2)][5].face;
	rightColor = w->pyraminx.facetLoc[topFaceTrans(topFace, 3)][2].face;

	if (w->pyraminx.facetLoc[topFaceTrans(topFace, 1)][3].face == frontColor &&
			w->pyraminx.facetLoc[topFaceTrans(topFace, 2)][3].face == leftColor) {
		return;
	}
	if (w->pyraminx.facetLoc[topFaceTrans(topFace, 2)][6].face == leftColor &&
			w->pyraminx.facetLoc[topFaceTrans(topFace, 3)][6].face == rightColor) {
		B_PLUS_WHOLE;
		return;
	}
	if (w->pyraminx.facetLoc[topFaceTrans(topFace, 3)][1].face == rightColor &&
			w->pyraminx.facetLoc[topFaceTrans(topFace, 1)][1].face == frontColor) {
		B_MINUS_WHOLE;
		return;
	}

	/* CW */
	if (w->pyraminx.facetLoc[topFaceTrans(topFace, 1)][3].face == rightColor &&
			w->pyraminx.facetLoc[topFaceTrans(topFace, 2)][3].face == frontColor) {
		B_PLUS;
		B_MINUS_WHOLE;
		return;
	}
	if (w->pyraminx.facetLoc[topFaceTrans(topFace, 2)][6].face == frontColor &&
			w->pyraminx.facetLoc[topFaceTrans(topFace, 3)][6].face == leftColor) {
		B_PLUS;
		return;
	}
	if (w->pyraminx.facetLoc[topFaceTrans(topFace, 3)][1].face == leftColor &&
			w->pyraminx.facetLoc[topFaceTrans(topFace, 1)][1].face == rightColor) {
		B_PLUS;
		B_PLUS_WHOLE;
		return;
	}

	/* CCW */
	if (w->pyraminx.facetLoc[topFaceTrans(topFace, 1)][3].face == leftColor &&
			w->pyraminx.facetLoc[topFaceTrans(topFace, 2)][3].face == rightColor) {
		B_MINUS;
		B_PLUS_WHOLE;
		return;
	}
	if (w->pyraminx.facetLoc[topFaceTrans(topFace, 2)][6].face == rightColor &&
			w->pyraminx.facetLoc[topFaceTrans(topFace, 3)][6].face == frontColor) {
		B_MINUS;
		B_MINUS_WHOLE;
		return;
	}
	if (w->pyraminx.facetLoc[topFaceTrans(topFace, 3)][1].face == frontColor &&
			w->pyraminx.facetLoc[topFaceTrans(topFace, 1)][1].face == leftColor) {
		B_MINUS;
		return;
	}
	(void) printf("Could not align big corner!\n");
	return;
}

static void
BottomBigCorner(PyraminxWidget w, int topFace)
{
	int i = 0;

	AlignBigCorner(w, topFace);
	while (!CheckSolved(w)) {
		if (i++ > 2) {
			(void) printf("Infinite Loop detected!\n");
			return;
		}
		if (w->pyraminx.facetLoc[topFaceTrans(topFace, 1)][7].face ==
				w->pyraminx.facetLoc[topFaceTrans(topFace, 1)][1].face) {
			if (w->pyraminx.facetLoc[topFaceTrans(topFace, 1)][2].face ==
					w->pyraminx.facetLoc[topFaceTrans(topFace, 2)][3].face) {
				R_MINUS;
				B_MINUS;
				R_PLUS;
				B_MINUS;
				R_MINUS;
				B_MINUS;
				R_PLUS;
			} else {
			/* Short cut: Alternate sequence */
			/* Could be solved by using above sequence but adds extra loops. */
				R_MINUS;
				B_PLUS;
				R_PLUS;
				B_PLUS;
				R_MINUS;
				B_PLUS;
				R_PLUS;
			}
		} else {
			L_PLUS;
			B_MINUS;
			L_MINUS;
			R_MINUS;
			B_MINUS;
			R_PLUS;

			L_PLUS;
			B_MINUS;
			L_MINUS;
			R_MINUS;
			B_MINUS;
			R_PLUS;
		}
		AlignBigCorner(w, topFace);
	}
#ifdef DEBUG
	(void) printf("loops %d\n", i);
#endif
}

#if 0
From Puzzle It Out: Cubes, Groups and Puzzles
/* Upper(U) = Posterior (P) */
/*      P       */
/*   e3    e4   */
/* L    e1    B */
/*   e2    e5   */
/*      R       */
#define e1_e2_e3 P_MINUS; R_PLUS; P_PLUS; R_MINUS
#define e1_e3_e2 R_PLUS; P_MINUS; R_MINUS; P_PLUS
#define e1_e3_e4 L_PLUS; P_PLUS; R_PLUS; P_MINUS; R_MINUS; L_MINUS
#define e1_e4_e3 L_PLUS; R_PLUS; P_PLUS; R_MINUS; P_MINUS; L_MINUS
#define e2_e1_e4 R_PLUS; P_PLUS; R_MINUS; P_MINUS
#define e4_e1_e2 P_PLUS; R_PLUS; P_MINUS; R_MINUS
#define e5_e1_e3 R_MINUS; P_MINUS; R_PLUS; P_PLUS
#define e3_e1_e5 P_MINUS; R_MINUS; P_PLUS; R_PLUS
#define e1_e3 R_PLUS; P_MINUS; R_MINUS; P_PLUS;\
 R_MINUS; L_PLUS; R_PLUS; L_MINUS
#define e3_e5 P_MINUS; R_MINUS; P_PLUS;\
 R_MINUS; L_PLUS; R_PLUS; L_MINUS; R_PLUS
#endif

/* This procedure coordinates the solution process. */
void
SolveSomePieces(PyraminxWidget w)
{
	setPuzzle(w, PUZZLE_RESET);
	if (SolvingFlag)
		return;
#ifdef JMP
	if (!setjmp(solve_env))
#endif
	{
		SolvingFlag = True;

		if (!CheckSolved(w)) {
			OrientLittleCorners(w);
			if (w->pyraminx.size > 2) {
				BottomBigCorner(w, TopFace(w));
			}
		}
	}
#ifdef JMP
	AbortSolvingFlag = False;
#endif
	SolvingFlag = False;
	w->pyraminx.cheat = True; /* Assume the worst. */
	setPuzzle(w, PUZZLE_COMPUTED);
}
