/*-
# X-BASED SKEWB
#
#  Skewb2d.c
#
###
#
#  Copyright (c) 1994 - 2007	David A. Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Skewb2d */

#include "SkewbP.h"
#include "Skewb2dP.h"

#ifndef WINVER
static Boolean SetValuesPuzzle2D(Widget current, Widget request, Widget renew);
static void ResizePuzzle2D(Skewb2DWidget w);
static void InitializePuzzle2D(Widget request, Widget renew);
static void ExposePuzzle2D(Widget renew, XEvent *event, Region region);
static void MovePuzzle2DTl(Skewb2DWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzle2DTop(Skewb2DWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzle2DTr(Skewb2DWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzle2DLeft(Skewb2DWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzle2dRight(Skewb2DWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzle2DBl(Skewb2DWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzle2DBottom(Skewb2DWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzle2DBr(Skewb2DWidget w,
	XEvent *event, char **args, int nArgs);

static char defaultTranslationsPuzzle2D[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <KeyPress>0x2E: Speed()\n\
 <KeyPress>0x3E: Speed()\n\
 <KeyPress>0x3C: Slow()\n\
 <KeyPress>0x2C: Slow()\n\
 Shift<KeyPress>2: Sound()\n\
 <KeyPress>KP_Divide: MoveCcw()\n\
 <KeyPress>R5: MoveCcw()\n\
 <KeyPress>Home: MoveTl()\n\
 <KeyPress>KP_7: MoveTl()\n\
 <KeyPress>R7: MoveTl()\n\
 <KeyPress>Up: MoveTop()\n\
 <KeyPress>osfUp: MoveTop()\n\
 <KeyPress>KP_Up: MoveTop()\n\
 <KeyPress>KP_8: MoveTop()\n\
 <KeyPress>R8: MoveTop()\n\
 <KeyPress>Prior: MoveTr()\n\
 <KeyPress>KP_9: MoveTr()\n\
 <KeyPress>R9: MoveTr()\n\
 <KeyPress>Left: MoveLeft()\n\
 <KeyPress>osfLeft: MoveLeft()\n\
 <KeyPress>KP_Left: MoveLeft()\n\
 <KeyPress>KP_4: MoveLeft()\n\
 <KeyPress>R10: MoveLeft()\n\
 <KeyPress>Begin: MoveCw()\n\
 <KeyPress>KP_5: MoveCw()\n\
 <KeyPress>R11: MoveCw()\n\
 <KeyPress>Right: MoveRight()\n\
 <KeyPress>osfRight: MoveRight()\n\
 <KeyPress>KP_Right: MoveRight()\n\
 <KeyPress>KP_6: MoveRight()\n\
 <KeyPress>R12: MoveRight()\n\
 <KeyPress>End: MoveBl()\n\
 <KeyPress>KP_1: MoveBl()\n\
 <KeyPress>R1: MoveBl()\n\
 <KeyPress>Down: MoveBottom()\n\
 <KeyPress>osfDown: MoveBottom()\n\
 <KeyPress>KP_Down: MoveBottom()\n\
 <KeyPress>KP_2: MoveBottom()\n\
 <KeyPress>R14: MoveBottom()\n\
 <KeyPress>Next: MoveBr()\n\
 <KeyPress>KP_3: MoveBr()\n\
 <KeyPress>R3: MoveBr()\n\
 <Btn1Down>: Select()\n\
 <Btn1Up>: Release()\n\
 <Btn2Down>: PracticeMaybe()\n\
 <Btn2Down>(2+): Practice2()\n\
 <Btn3Down>: RandomizeMaybe()\n\
 <Btn3Down>(2+): Randomize2()\n\
 <Btn4Down>: MoveTop()\n\
 <Btn5Down>: MoveBottom()\n\
 <KeyPress>g: Get()\n\
 <KeyPress>w: Write()\n\
 <KeyPress>u: Undo()\n\
 <KeyPress>r: Redo()\n\
 <KeyPress>c: Clear()\n\
 <KeyPress>z: Randomize()\n\
 <KeyPress>s: Solve()\n\
 <KeyPress>p: Practice()\n\
 <KeyPress>o: Orientize()\n\
 <KeyPress>v: View()\n\
 <EnterWindow>: Enter()\n\
 <LeaveWindow>: Leave()";

static XtActionsRec actionsListPuzzle2D[] =
{
	{(char *) "Quit", (XtActionProc) QuitPuzzle},
	{(char *) "Hide", (XtActionProc) HidePuzzle},
	{(char *) "MoveCcw", (XtActionProc) MovePuzzleCcw},
	{(char *) "MoveTl", (XtActionProc) MovePuzzle2DTl},
	{(char *) "MoveTop", (XtActionProc) MovePuzzle2DTop},
	{(char *) "MoveTr", (XtActionProc) MovePuzzle2DTr},
	{(char *) "MoveLeft", (XtActionProc) MovePuzzle2DLeft},
	{(char *) "MoveCw", (XtActionProc) MovePuzzleCw},
	{(char *) "MoveRight", (XtActionProc) MovePuzzle2dRight},
	{(char *) "MoveBl", (XtActionProc) MovePuzzle2DBl},
	{(char *) "MoveBottom", (XtActionProc) MovePuzzle2DBottom},
	{(char *) "MoveBr", (XtActionProc) MovePuzzle2DBr},
	{(char *) "Select", (XtActionProc) SelectPuzzle},
	{(char *) "Release", (XtActionProc) ReleasePuzzle},
	{(char *) "PracticeMaybe", (XtActionProc) PracticePuzzleMaybe},
	{(char *) "Practice2", (XtActionProc) PracticePuzzle2},
	{(char *) "RandomizeMaybe", (XtActionProc) RandomizePuzzleMaybe},
	{(char *) "Randomize2", (XtActionProc) RandomizePuzzle2},
	{(char *) "Get", (XtActionProc) GetPuzzle},
	{(char *) "Write", (XtActionProc) WritePuzzle},
	{(char *) "Undo", (XtActionProc) UndoPuzzle},
	{(char *) "Redo", (XtActionProc) RedoPuzzle},
	{(char *) "Clear", (XtActionProc) ClearPuzzle},
	{(char *) "Randomize", (XtActionProc) RandomizePuzzle},
	{(char *) "Solve", (XtActionProc) SolvePuzzle},
	{(char *) "Practice", (XtActionProc) PracticePuzzle},
	{(char *) "Orientize", (XtActionProc) OrientizePuzzle},
	{(char *) "View", (XtActionProc) ViewPuzzle},
	{(char *) "Speed", (XtActionProc) SpeedPuzzle},
	{(char *) "Slow", (XtActionProc) SlowPuzzle},
	{(char *) "Sound", (XtActionProc) SoundPuzzle},
	{(char *) "Enter", (XtActionProc) EnterPuzzle},
	{(char *) "Leave", (XtActionProc) LeavePuzzle}
};

static XtResource resourcesPuzzle2D[] =
{
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(SkewbWidget, core.width),
	 XtRString, (caddr_t) "300"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(SkewbWidget, core.height),
	 XtRString, (caddr_t) "400"},
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverseVideo, XtCReverseVideo, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(SkewbWidget, skewb.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNbackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(SkewbWidget, skewb.background),
	 XtRString, (caddr_t) XtDefaultBackground},
	{XtNframeColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(SkewbWidget, skewb.frameColor),
	 XtRString, (caddr_t) "cyan" /*XtDefaultForeground*/},
	{XtNfaceColor0, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[0]),
	 XtRString, (caddr_t) "red"},
	{XtNfaceColor1, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[1]),
	 XtRString, (caddr_t) "blue"},
	{XtNfaceColor2, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[2]),
	 XtRString, (caddr_t) "white"},
	{XtNfaceColor3, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[3]),
	 XtRString, (caddr_t) "green"},
	{XtNfaceColor4, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[4]),
	 XtRString, (caddr_t) "pink"},
	{XtNfaceColor5, XtCLabel, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.faceName[5]),
	 XtRString, (caddr_t) "yellow"},
	{XtNpieceBorder, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(SkewbWidget, skewb.borderColor),
	 XtRString, (caddr_t) "gray25" /*XtDefaultForeground*/},
	{XtNdelay, XtCDelay, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.delay),
	 XtRString, (caddr_t) "10"},
	{XtNsound, XtCSound, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.sound),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmoveSound, XtCMoveSound, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.moveSound),
	 XtRString, (caddr_t) MOVESOUND},
	{XtNfont, XtCFont, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.font),
	 XtRString, (caddr_t) "9x15bold"},
	{XtNorient, XtCOrient, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.orient),
	 XtRString, (caddr_t) "FALSE"},	/* DEFAULT_ORIENT */
	{XtNpractice, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.practice),
	 XtRString, (caddr_t) "TRUE"}, /* DEFAULT_PRACTICE */
	{XtNuserName, XtCUserName, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.userName),
	 XtRString, (caddr_t) ""},
	{XtNscoreFile, XtCScoreFile, XtRString, sizeof (String),
	 XtOffset(SkewbWidget, skewb.scoreFile),
	 XtRString, (caddr_t) ""},
	{XtNscoreOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.scoreOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNversionOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.versionOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmenu, XtCMenu, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.menu),
	 XtRString, (caddr_t) "999"}, /* ACTION_IGNORE */
	{XtNstart, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.started),
	 XtRString, (caddr_t) "FALSE"},
	{XtNcheat, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(SkewbWidget, skewb.cheat),
	 XtRString, (caddr_t) "FALSE"},
	{XtNface, XtCFace, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.currentFace),
	 XtRString, (caddr_t) "-1"},
	{XtNpos, XtCPos, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.currentPosition),
	 XtRString, (caddr_t) "-1"},
	{XtNdirection, XtCDirection, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.currentDirection),
	 XtRString, (caddr_t) "-1"},
	{XtNpixmapSize, XtCPixmapSize, XtRInt, sizeof (int),
	 XtOffset(SkewbWidget, skewb.pixmapSize),
	 XtRString, (caddr_t) "64"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(SkewbWidget, skewb.select),
	 XtRCallback, (caddr_t) NULL}
};

Skewb2DClassRec skewb2dClassRec =
{
	{
		(WidgetClass) & skewbClassRec,	/* superclass */
		(char *) "Skewb2D",	/* class name */
		sizeof (Skewb2DRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializePuzzle2D,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListPuzzle2D,	/* actions */
		XtNumber(actionsListPuzzle2D),	/* num actions */
		resourcesPuzzle2D,	/* resources */
		XtNumber(resourcesPuzzle2D),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		NULL,		/* destroy */
		(XtWidgetProc) ResizePuzzle2D,	/* resize */
		(XtExposeProc) ExposePuzzle2D,	/* expose */
		(XtSetValuesFunc) SetValuesPuzzle2D,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		XtInheritAcceptFocus,	/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		defaultTranslationsPuzzle2D,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	},
	{
		0		/* ignore */
	}
};

WidgetClass skewb2dWidgetClass = (WidgetClass) & skewb2dClassRec;
#endif

static int planeToCube[MAX_RECT] =
{6, 0, 6, 1, 2, 3, 6, 4, 6, 6, 5, 6};
static int cubeToPlane[MAX_FACES] =
{1, 3, 4, 5, 7, 10};
static Point diamondUnit[MAX_ORIENT + 1] =
{
	{0, 1},
	{1, -1},
	{1, 1},
	{-1, 1},
	{-1, -1}
};
static Point triangleUnit[MAX_ORIENT][4] =
{
	{
		{2, 0},
		{1, 0},
		{0, 1},
		{-1, -1}
	},
	{
		{3, 2},
		{0, 1},
		{-1, 0},
		{1, -1}
	},
	{
		{1, 3},
		{-1, 0},
		{0, -1},
		{1, 1}
	},
	{
		{0, 1},
		{0, -1},
		{1, 0},
		{-1, 1}
	}
};
static Point letterUnit[MAX_CUBES] =
{
	{2, 0},
	{2, 2},
	{0, 2},
	{0, 0},
	{1, 1}
};
static Point diamondList[MAX_ORIENT + 1], triangleList[MAX_ORIENT][4];
static Point letterList[MAX_CUBES], offsetList[MAX_CUBES];

static Pixmap dr = 0; /* dummy for future double buffering */

static void
DrawOrientLine(Skewb2DWidget w, int cube, int orient, int dx, int dy,
		GC borderGC)
{
	char *buf;

	if (cube == MAX_ORIENT) {
		switch (orient) {
		case TR:
			DRAWLINE(w, dr, borderGC,
				dx + w->skewb2d.diamondLength / 2 - 2,
				dy + w->skewb.delta - 2,
				dx + w->skewb2d.diamondLength / 2 - 2,
				dy + w->skewb.delta +
				w->skewb.orientLineLength);
			return;
		case BR:
			DRAWLINE(w, dr, borderGC,
				dx + w->skewb2d.diamondLength - 7,
				dy + w->skewb2d.diamondLength / 2 - 2,
				dx + w->skewb2d.diamondLength -
				w->skewb.orientLineLength - 7,
				dy + w->skewb2d.diamondLength / 2 - 2);
				return;
		case BL:
			DRAWLINE(w, dr, borderGC,
				dx + w->skewb2d.diamondLength / 2 - 2,
				dy + w->skewb2d.diamondLength - 7,
				dx + w->skewb2d.diamondLength / 2 - 2,
				dy + w->skewb2d.diamondLength -
				w->skewb.orientLineLength - 7);
				return;
		case TL:
			DRAWLINE(w, dr, borderGC,
				dx + w->skewb.delta - 2,
				dy + w->skewb2d.diamondLength / 2 - 2,
				dx + w->skewb.delta +
				w->skewb.orientLineLength,
				dy + w->skewb2d.diamondLength / 2 - 2);
				return;
			default:
				intCat(&buf, "DrawOrientLine: orient ",
					orient);
				DISPLAY_WARNING(buf);
				free(buf);
		}
	} else {	/* cube != MAX_ORIENT */
		switch (orient) {
		case TR:
			DRAWLINE(w, dr, borderGC,
				dx,
				dy - w->skewb.delta,
				dx,
				dy - w->skewb.delta -
				w->skewb.orientLineLength / 2);
			return;
		case BR:
			DRAWLINE(w, dr, borderGC,
				dx + w->skewb.delta,
				dy,
				dx + w->skewb.delta +
				w->skewb.orientLineLength / 2,
				dy);
			return;
		case BL:
			DRAWLINE(w, dr, borderGC,
				dx,
				dy + w->skewb.delta,
				dx,
				dy + w->skewb.delta +
				w->skewb.orientLineLength / 2);
			return;
		case TL:
			DRAWLINE(w, dr, borderGC,
				dx - w->skewb.delta,
				dy,
				dx - w->skewb.delta -
				w->skewb.orientLineLength / 2,
				dy);
			return;
		default:
			intCat(&buf, "DrawOrientLine: orient ",
				orient);
			DISPLAY_WARNING(buf);
			free(buf);
		}
	}
}

void
DrawDiamond2D(Skewb2DWidget w, int face, int offset)
{
	GC faceGC, borderGC;
	int dx, dy, orient;

	orient = w->skewb.cubeLoc[face][MAX_ORIENT].rotation;
	if (w->skewb.vertical || face != MAX_FACES - 1) {
		dx = (cubeToPlane[face] % MAXX) * w->skewb2d.viewLength +
			w->skewb.delta;
		dy = (cubeToPlane[face] / MAXX) * w->skewb2d.viewLength +
			w->skewb.delta;
	} else {
		dx = (cubeToPlane[face] / MAXX) * w->skewb2d.viewLength +
			w->skewb.delta;
		dy = (cubeToPlane[face] % MAXX) * w->skewb2d.viewLength +
			w->skewb.delta;
		orient = (orient + HALF) % STRT;
	}
	dx += w->skewb.puzzleOffset.x + w->skewb.delta;
	dy += w->skewb.puzzleOffset.y + w->skewb.delta;
	diamondList[0].x = offsetList[MAX_ORIENT].x + dx;
	diamondList[0].y = offsetList[MAX_ORIENT].y + dy;
	if (offset) {
		borderGC = w->skewb.faceGC[(int) w->skewb.cubeLoc[face][MAX_ORIENT].face];
		if (w->skewb.mono) {
			faceGC = w->skewb.inverseGC;
		} else {
			faceGC = w->skewb.borderGC;
		}
	} else {
		faceGC = w->skewb.faceGC[(int) w->skewb.cubeLoc[face][MAX_ORIENT].face];
		borderGC = w->skewb.borderGC;
	}
	POLYGON((SkewbWidget) w, dr, faceGC, borderGC,
		diamondList, 4, True, False);
	if (w->skewb.mono) {
		int letterX, letterY;
		char buf[2];

		buf[0] =
#ifdef WINVER
			w->skewb.faceChar[w->skewb.cubeLoc
				[face][MAX_ORIENT].face];
#else
			w->skewb.faceName[w->skewb.cubeLoc
				[face][MAX_ORIENT].face][0];
#endif
		buf[1] = '\0';
		letterX = dx + letterList[MAX_ORIENT].x;
		letterY = dy + letterList[MAX_ORIENT].y;
		if (offset) {
			borderGC = w->skewb.borderGC;
		} else {
			borderGC = w->skewb.inverseGC;
		}
		DRAWTEXT(w, dr, borderGC,
			letterX, letterY, buf, 1);
	}
	if (w->skewb.orient)
		DrawOrientLine(w, MAX_ORIENT, orient, dx, dy, borderGC);
}

void
DrawTriangle2D(Skewb2DWidget w, int face, int position, int offset)
{
	GC faceGC, borderGC;
	int dx, dy, letterX, letterY, orient, newCorner;

	orient = w->skewb.cubeLoc[face][position].rotation;
	if (w->skewb.vertical || face != MAX_FACES - 1) {
		dx = (cubeToPlane[face] % MAXX) * w->skewb2d.viewLength +
			w->skewb.delta - 1;
		dy = (cubeToPlane[face] / MAXX) * w->skewb2d.viewLength +
			w->skewb.delta - 1;
		newCorner = position;
	} else {
		dx = (cubeToPlane[face] / MAXX) * w->skewb2d.viewLength +
			w->skewb.delta - 1;
		dy = (cubeToPlane[face] % MAXX) * w->skewb2d.viewLength +
			w->skewb.delta - 1;
		newCorner = (position + HALF) % STRT;
		orient = (orient + HALF) % STRT;
	}
	dx += w->skewb.puzzleOffset.x + w->skewb.delta;
	dy += w->skewb.puzzleOffset.y + w->skewb.delta;
	letterX = dx + letterList[newCorner].x;
	letterY = dy + letterList[newCorner].y;
	triangleList[newCorner][0].x = offsetList[newCorner].x + dx;
	triangleList[newCorner][0].y = offsetList[newCorner].y + dy;
	if (offset) {
		borderGC = w->skewb.faceGC[(int) w->skewb.cubeLoc[face][position].face];
		if (w->skewb.mono) {
			faceGC = w->skewb.inverseGC;
		} else {
			faceGC = w->skewb.borderGC;
		}
	} else {
		faceGC = w->skewb.faceGC[(int) w->skewb.cubeLoc[face][position].face];
		borderGC = w->skewb.borderGC;
	}
	POLYGON((SkewbWidget) w, dr, faceGC, borderGC,
		triangleList[newCorner], 3, True, False);
	if (w->skewb.mono) {
		char buf[2];

		buf[0] =
#ifdef WINVER
			w->skewb.faceChar[w->skewb.cubeLoc
				[face][position].face];
#else
			w->skewb.faceName[w->skewb.cubeLoc
				[face][position].face][0];
#endif
		buf[1] = '\0';
		if (offset) {
			borderGC = w->skewb.borderGC;
		} else {
			borderGC = w->skewb.inverseGC;
		}
		DRAWTEXT(w, dr, borderGC, letterX, letterY, buf, 1);
	}
	if (w->skewb.orient)
		DrawOrientLine(w, newCorner, orient,
			letterX - w->skewb.letterOffset.x,
			letterY - w->skewb.letterOffset.y,
			borderGC);
}

void
DrawFrame2D(Skewb2DWidget w, Boolean focus)
{
	int i;
	Point pos[MAXXY + 1], letters;
	GC gc = (focus) ? w->skewb.frameGC : w->skewb.borderGC;

	for (i = 0; i <= MAXXY; i++) {
		pos[i].x = i * w->skewb2d.viewLength + w->skewb.puzzleOffset.x;
		pos[i].y = i * w->skewb2d.viewLength + w->skewb.puzzleOffset.y;
	}
	DRAWLINE(w, dr, gc, pos[1].x, pos[0].y, pos[2].x, pos[0].y);
	DRAWLINE(w, dr, gc, pos[3].x, pos[1].y, pos[3].x, pos[2].y);
	DRAWLINE(w, dr, gc, pos[1].x, pos[3].y, pos[2].x, pos[3].y);
	DRAWLINE(w, dr, gc, pos[0].x, pos[1].y, pos[0].x, pos[2].y);
	letters.x = pos[0].x + w->skewb2d.viewLength / 2 - w->skewb.delta;
	letters.y = pos[0].y + w->skewb2d.viewLength / 2;
	DRAWTEXT(w, dr, w->skewb.borderGC,
		(int) (letters.x + 5 * w->skewb.letterOffset.x),
		(int) (letters.y + w->skewb.letterOffset.y), "Front", 5);
	letters.x = pos[2].x + w->skewb2d.viewLength / 2 - w->skewb.delta;
	letters.y = pos[2].y + w->skewb2d.viewLength / 2;
	DRAWTEXT(w, dr, w->skewb.borderGC,
		(int) (letters.x + 4 * w->skewb.letterOffset.x),
		(int) (letters.y + w->skewb.letterOffset.y), "Back", 4);
	if (w->skewb.vertical) {
		DRAWLINE(w, dr, gc, pos[1].x, pos[0].y, pos[1].x, pos[4].y);
		DRAWLINE(w, dr, gc, pos[2].x, pos[0].y, pos[2].x, pos[4].y);
		DRAWLINE(w, dr, gc, pos[0].x, pos[1].y, pos[3].x, pos[1].y);
		DRAWLINE(w, dr, gc, pos[0].x, pos[2].y, pos[3].x, pos[2].y);
		DRAWLINE(w, dr, gc, pos[1].x, pos[4].y, pos[2].x, pos[4].y);
	} else {
		DRAWLINE(w, dr, gc, pos[0].x, pos[1].y, pos[4].x, pos[1].y);
		DRAWLINE(w, dr, gc, pos[0].x, pos[2].y, pos[4].x, pos[2].y);
		DRAWLINE(w, dr, gc, pos[1].x, pos[0].y, pos[1].x, pos[3].y);
		DRAWLINE(w, dr, gc, pos[2].x, pos[0].y, pos[2].x, pos[3].y);
		DRAWLINE(w, dr, gc, pos[4].x, pos[1].y, pos[4].x, pos[2].y);
	}
}

static void
ResizePieces(Skewb2DWidget w)
{
	int i, j;

	w->skewb.orientLineLength = w->skewb2d.diamondLength / 8;
	w->skewb.letterOffset.x = -2;
	w->skewb.letterOffset.y = 4;
	for (i = 0; i <= MAX_ORIENT; i++) {
		diamondList[i].x = diamondUnit[i].x *
			(w->skewb2d.diamondLength / 2 - w->skewb.delta);
		diamondList[i].y = diamondUnit[i].y *
			(w->skewb2d.diamondLength / 2 - w->skewb.delta);
	}
	for (i = 0; i < MAX_ORIENT; i++) {
		for (j = 0; j <= 3; j++) {
			triangleList[i][j].x = triangleUnit[i][j].x *
				(w->skewb2d.diamondLength / 2 - 3 *
				w->skewb.delta);
			triangleList[i][j].y = triangleUnit[i][j].y *
				(w->skewb2d.diamondLength / 2 - 3 *
				w->skewb.delta);
		}
	}

	for (i = 0; i < MAX_ORIENT; i++) {
		if (letterUnit[i].x == 0)
			letterList[i].x = w->skewb2d.diamondLength / 8 +
				w->skewb.letterOffset.x;
		else if (letterUnit[i].x == 2)
			letterList[i].x = 7 * w->skewb2d.diamondLength / 8 -
				2 + w->skewb.letterOffset.x;
		if (letterUnit[i].y == 0)
			letterList[i].y = w->skewb2d.diamondLength / 8 +
				2 + w->skewb.letterOffset.y;
		else if (letterUnit[i].y == 2)
			letterList[i].y = 7 * w->skewb2d.diamondLength / 8 -
				3 + w->skewb.letterOffset.y;

		if (triangleUnit[i][0].x == 0)
			offsetList[i].x = w->skewb.delta - 1;
		else if (triangleUnit[i][0].x == 1)
			offsetList[i].x = w->skewb2d.diamondLength / 2 -
				2 * w->skewb.delta - 1;
		else if (triangleUnit[i][0].x == 2)
			offsetList[i].x = w->skewb2d.diamondLength / 2 +
				2 * w->skewb.delta;
		else if (triangleUnit[i][0].x == 3)
			offsetList[i].x = w->skewb2d.diamondLength -
				w->skewb.delta - 1;
		if (triangleUnit[i][0].y == 0)
			offsetList[i].y = w->skewb.delta - 1;
		else if (triangleUnit[i][0].y == 1)
			offsetList[i].y = w->skewb2d.diamondLength / 2 -
				2 * w->skewb.delta - 1;
		else if (triangleUnit[i][0].y == 2)
			offsetList[i].y = w->skewb2d.diamondLength / 2 +
				2 * w->skewb.delta - 1;
		else if (triangleUnit[i][0].y == 3)
			offsetList[i].y = w->skewb2d.diamondLength -
				w->skewb.delta - 2;
	}
	if (diamondUnit[0].x == 0)
		offsetList[MAX_ORIENT].x = w->skewb.delta - 2;
	else if (diamondUnit[0].x == 1)
		offsetList[MAX_ORIENT].x = w->skewb2d.diamondLength / 2 - 1;
	if (diamondUnit[0].y == 0)
		offsetList[MAX_ORIENT].y = w->skewb.delta - 2;
	else if (diamondUnit[0].y == 1)
		offsetList[MAX_ORIENT].y = w->skewb2d.diamondLength / 2 - 2;
	if (letterUnit[MAX_ORIENT].x == 1)
		letterList[MAX_ORIENT].x = w->skewb2d.diamondLength / 2 - 2 +
			w->skewb.letterOffset.x;
	if (letterUnit[MAX_ORIENT].y == 1)
		letterList[MAX_ORIENT].y = w->skewb2d.diamondLength / 2 - 2 +
			w->skewb.letterOffset.y;
}

Boolean
SelectPieces2D(Skewb2DWidget w, int x, int y, int *face, int *position)
{
	int faceX, faceY, i, j;

	x -= w->skewb.puzzleOffset.x;
	y -= w->skewb.puzzleOffset.y;
	faceX = x / w->skewb2d.viewLength;
	faceY = y / w->skewb2d.viewLength;
	i = x - faceX * w->skewb2d.viewLength;
	j = y - faceY * w->skewb2d.viewLength;
	if (i - j > w->skewb2d.viewLength / 2 - 3)
		*position = TR;
	else if (i + j > 3 * w->skewb2d.viewLength / 2)
		*position = BR;
	else if (j - i > w->skewb2d.viewLength / 2 - 2)
		*position = BL;
	else if (i + j < w->skewb2d.viewLength / 2 + 7)
		*position = TL;
	else
		*position = MAX_ORIENT;
	if ((faceX != 1 && faceY != 1) ||
			(faceX >= 3 && w->skewb.vertical) ||
			(faceY >= 3 && !w->skewb.vertical))
		return False;
	*face = planeToCube[faceX + faceY * MAXX];
	if (faceX == 3) {
		*face = MAX_FACES - 1;
		if (*position != MAX_ORIENT)
			*position = (*position + HALF) % MAX_ORIENT;
	}
	return True;
}

Boolean
NarrowSelection2D(Skewb2DWidget w, int *face, int *position, int *direction)
{
	if (*face == MAX_FACES - 1 && *direction < MAX_ORIENT &&
			!w->skewb.vertical) {
		if (*direction < MAX_ORIENT)
			*direction = (*direction + HALF) % MAX_ORIENT;
		else if (*direction >= 2 * MAX_ORIENT)
			*direction = 2 * MAX_ORIENT +
				(*direction + HALF) % MAX_ORIENT;
	}
	if (*position != MAX_ORIENT) {
		if (*direction == CW)
			*direction = (*position + 3) % MAX_ORIENT;
		else if (*direction == CCW)
			*direction = (*position + 1) % MAX_ORIENT;
		else if (*direction < MAX_ORIENT &&
				!((*direction + *position) % 2))
			return False;
	}
	return True;
}

#ifndef WINVER
static Boolean
SetValuesPuzzle2D(Widget current, Widget request, Widget renew)
{
	Skewb2DWidget c = (Skewb2DWidget) current, w = (Skewb2DWidget) renew;
	Boolean redraw = False;

	if (w->skewb2d.diamondLength != c->skewb2d.diamondLength) {
		ResizePuzzle2D(w);
		redraw = True;
	}
	return (redraw);
}
#endif

#ifndef WINVER
static
#endif
void
ResizePuzzle2D(Skewb2DWidget w)
{
#ifdef WINVER
	RECT rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif
	w->skewb.delta = 3;
	w->skewb.vertical = (w->core.height >= w->core.width);
	if (w->skewb.vertical)
		w->skewb2d.faceLength = MIN(w->core.height / MAXY,
			w->core.width / MAXX);
	else
		w->skewb2d.faceLength = MIN(w->core.height / MAXX,
			w->core.width / MAXY);
	w->skewb2d.faceLength = MAX(w->skewb2d.faceLength - w->skewb.delta - 1,
		0);
	w->skewb2d.diamondLength = w->skewb2d.faceLength - w->skewb.delta;
	w->skewb2d.viewLength = w->skewb2d.faceLength + w->skewb.delta;
	if (w->skewb.vertical) {
		w->skewb.puzzleSize.x = MAXX * (w->skewb2d.viewLength - 1) +
			w->skewb.delta;
		w->skewb.puzzleSize.y = MAXY * (w->skewb2d.viewLength - 1) +
			w->skewb.delta;
	} else {
		w->skewb.puzzleSize.x = MAXY * (w->skewb2d.viewLength - 1) +
			w->skewb.delta;
		w->skewb.puzzleSize.y = MAXX * (w->skewb2d.viewLength - 1) +
			w->skewb.delta;
	}
	w->skewb.puzzleOffset.x = ((int) w->core.width - w->skewb.puzzleSize.x)
		/ 2;
	w->skewb.puzzleOffset.y = ((int) w->core.height - w->skewb.puzzleSize.y)
		/ 2;
	ResizePieces(w);
}

#ifndef WINVER
static
#endif
void
InitializePuzzle2D(
#ifdef WINVER
Skewb2DWidget w
#else
Widget request, Widget renew
#endif
)
{
#ifndef WINVER
	Skewb2DWidget w = (Skewb2DWidget) renew;

	SetAllColors((SkewbWidget) w);
	w->skewb.dim = 2;
	ResizePuzzle2D(w);
#endif
}

#ifndef WINVER
static
#endif
void
ExposePuzzle2D(
#ifdef WINVER
Skewb2DWidget w
#else
Widget renew, XEvent *event, Region region
#endif
)
{
#ifndef WINVER
	Skewb2DWidget w = (Skewb2DWidget) renew;

	if (!w->core.visible)
		return;
#endif
	FILLRECTANGLE(w, dr, w->skewb.inverseGC,
		0, 0, w->core.width, w->core.height);
	DrawFrame2D(w, w->skewb.focus);
	DrawAllPieces((SkewbWidget) w);
}

#ifndef WINVER
static void
MovePuzzle2DTl(Skewb2DWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		TL, (int) (event->xkey.state & ControlMask), FALSE);
}

static void
MovePuzzle2DTop(Skewb2DWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		TOP,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzle2DTr(Skewb2DWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		TR, (int) (event->xkey.state & ControlMask), FALSE);
}

static void
MovePuzzle2DLeft(Skewb2DWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		LEFT,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzle2dRight(Skewb2DWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		RIGHT,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzle2DBl(Skewb2DWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		BL, (int) (event->xkey.state & ControlMask), FALSE);
}

static void
MovePuzzle2DBottom(Skewb2DWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		BOTTOM,
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzle2DBr(Skewb2DWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((SkewbWidget) w, event->xbutton.x, event->xbutton.y,
		BR, (int) (event->xkey.state & ControlMask), FALSE);
}
#endif
