/*-
# X-BASED MISSING LINK(tm)
#
#  Mlink.c
#
###
#
#  Copyright (c) 1994 - 2008	David A. Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Mlink */

#include "file.h"
#include "rngs.h"
#include "sound.h"
#include "MlinkP.h"

#ifndef PICTURE
#if 1
#define PICTURE ""
#else
#ifdef WINVER
#define PICTURE "picture"
#else
#ifdef HAVE_XPM
#define PICTURE "./mandrill.xpm"
#else
#define PICTURE "./mandrill.xbm"
#endif
#endif
#endif
#endif

#ifdef WINVER
#ifndef LOGPATH
#define LOGPATH "/usr/tmp"
#endif
#ifndef INIFILE
#define INIFILE "wmlink.ini"
#endif
#define SECTION "setup"

static const char *faceColorString[MAX_FACES] =
{
	"255 255 255",
	"255 255 0",
	"0 255 0",
	"255 0 0",
	"0 0 255",
	"255 0 255",
	"0 255 255",
	"255 165 0"
};

static char faceColorChar[MAX_FACES] =
{'W', 'Y', 'G', 'R', 'B', 'M', 'C', 'O'};
#else
#include "picture.h"

#ifndef LOGPATH
#ifdef VMS
#define LOGPATH "SYS$SCRATCH:"
#else
#define LOGPATH "/usr/tmp"
#endif
#endif

static Boolean SetValuesPuzzle(Widget current, Widget request, Widget renew);
static void QuitPuzzle(MlinkWidget w, XEvent *event, char **args, int nArgs);
static void DestroyPuzzle(Widget old);
static void ResizePuzzle(MlinkWidget w);
static void SizePuzzle(MlinkWidget w);
static void InitializePuzzle(Widget request, Widget renew);
static void ExposePuzzle(Widget renew, XEvent *event, Region region);
static void HidePuzzle(MlinkWidget w, XEvent *event, char **args, int nArgs);
static void SelectPuzzle(MlinkWidget w,
	XEvent *event, char **args, int nArgs);
static void MotionPuzzle(MlinkWidget w,
	XEvent *event, char **args, int nArgs);
static void ReleasePuzzle(MlinkWidget w,
	XEvent *event, char **args, int nArgs);
static void PracticePuzzleMaybe(MlinkWidget w,
	XEvent *event, char **args, int nArgs);
static void PracticePuzzle2(MlinkWidget w,
	XEvent *event, char **args, int nArgs);
static void RandomizePuzzleMaybe(MlinkWidget w,
	XEvent *event, char **args, int nArgs);
static void RandomizePuzzle2(MlinkWidget w,
	XEvent *event, char **args, int nArgs);
static void GetPuzzle(MlinkWidget w, XEvent *event, char **args, int nArgs);
static void WritePuzzle(MlinkWidget w, XEvent *event, char **args, int nArgs);
static void UndoPuzzle(MlinkWidget w, XEvent *event, char **args, int nArgs);
static void RedoPuzzle(MlinkWidget w, XEvent *event, char **args, int nArgs);
static void ClearPuzzle(MlinkWidget w, XEvent *event, char **args, int nArgs);
static void RandomizePuzzle(MlinkWidget w,
	XEvent *event, char **args, int nArgs);
static void SolvePuzzle(MlinkWidget w, XEvent *event, char **args, int nArgs);
static void PracticePuzzle(MlinkWidget w,
	XEvent *event, char **args, int nArgs);
static void OrientizePuzzle(MlinkWidget w,
	XEvent *event, char **args, int nArgs);
static void MiddlePuzzle(MlinkWidget w,
	XEvent *event, char **args, int nArgs);
static void SpeedPuzzle(MlinkWidget w, XEvent *event, char **args, int nArgs);
static void SlowPuzzle(MlinkWidget w, XEvent *event, char **args, int nArgs);
static void SoundPuzzle(MlinkWidget w, XEvent *event, char **args, int nArgs);
static void EnterPuzzle(MlinkWidget w, XEvent *event, char **args, int nArgs);
static void LeavePuzzle(MlinkWidget w, XEvent *event, char **args, int nArgs);
static void MovePuzzleTop(MlinkWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzleBottom(MlinkWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzleLeft(MlinkWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzleRight(MlinkWidget w,
	XEvent *event, char **args, int nArgs);

static char defaultTranslationsPuzzle[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <KeyPress>0x2E: Speed()\n\
 <KeyPress>0x3E: Speed()\n\
 <KeyPress>0x3C: Slow()\n\
 <KeyPress>0x2C: Slow()\n\
 Shift<KeyPress>2: Sound()\n\
 <KeyPress>Up: MoveTop()\n\
 <KeyPress>osfUp: MoveTop()\n\
 <KeyPress>KP_Up: MoveTop()\n\
 <KeyPress>KP_8: MoveTop()\n\
 <KeyPress>R8: MoveTop()\n\
 <KeyPress>Left: MoveLeft()\n\
 <KeyPress>osfLeft: MoveLeft()\n\
 <KeyPress>KP_Left: MoveLeft()\n\
 <KeyPress>KP_4: MoveLeft()\n\
 <KeyPress>R10: MoveLeft()\n\
 <KeyPress>Right: MoveRight()\n\
 <KeyPress>osfRight: MoveRight()\n\
 <KeyPress>KP_Right: MoveRight()\n\
 <KeyPress>KP_6: MoveRight()\n\
 <KeyPress>R12: MoveRight()\n\
 <KeyPress>Down: MoveBottom()\n\
 <KeyPress>osfDown: MoveBottom()\n\
 <KeyPress>KP_Down: MoveBottom()\n\
 <KeyPress>KP_2: MoveBottom()\n\
 <KeyPress>R14: MoveBottom()\n\
 <Btn1Down>: Select()\n\
 <Btn1Motion>: Motion()\n\
 <Btn1Up>: Release()\n\
 <Btn2Down>: PracticeMaybe()\n\
 <Btn2Down>(2+): Practice2()\n\
 <Btn3Down>: RandomizeMaybe()\n\
 <Btn3Down>(2+): Randomize2()\n\
 <Btn4Down>: MoveTop()\n\
 <Btn5Down>: MoveBottom()\n\
 <KeyPress>g: Get()\n\
 <KeyPress>w: Write()\n\
 <KeyPress>u: Undo()\n\
 <KeyPress>r: Redo()\n\
 <KeyPress>c: Clear()\n\
 <KeyPress>z: Randomize()\n\
 <KeyPress>s: Solve()\n\
 <KeyPress>p: Practice()\n\
 <KeyPress>o: Orientize()\n\
 <KeyPress>m: Middle()\n\
 <EnterWindow>: Enter()\n\
 <LeaveWindow>: Leave()";

static XtActionsRec actionsListPuzzle[] =
{
	{(char *) "Quit", (XtActionProc) QuitPuzzle},
	{(char *) "Hide", (XtActionProc) HidePuzzle},
	{(char *) "MoveTop", (XtActionProc) MovePuzzleTop},
	{(char *) "MoveLeft", (XtActionProc) MovePuzzleLeft},
	{(char *) "MoveRight", (XtActionProc) MovePuzzleRight},
	{(char *) "MoveBottom", (XtActionProc) MovePuzzleBottom},
	{(char *) "Select", (XtActionProc) SelectPuzzle},
	{(char *) "Motion", (XtActionProc) MotionPuzzle},
	{(char *) "Release", (XtActionProc) ReleasePuzzle},
	{(char *) "PracticeMaybe", (XtActionProc) PracticePuzzleMaybe},
	{(char *) "Practice2", (XtActionProc) PracticePuzzle2},
	{(char *) "RandomizeMaybe", (XtActionProc) RandomizePuzzleMaybe},
	{(char *) "Randomize2", (XtActionProc) RandomizePuzzle2},
	{(char *) "Get", (XtActionProc) GetPuzzle},
	{(char *) "Write", (XtActionProc) WritePuzzle},
	{(char *) "Undo", (XtActionProc) UndoPuzzle},
	{(char *) "Redo", (XtActionProc) RedoPuzzle},
	{(char *) "Clear", (XtActionProc) ClearPuzzle},
	{(char *) "Randomize", (XtActionProc) RandomizePuzzle},
	{(char *) "Solve", (XtActionProc) SolvePuzzle},
	{(char *) "Practice", (XtActionProc) PracticePuzzle},
	{(char *) "Orientize", (XtActionProc) OrientizePuzzle},
	{(char *) "Middle", (XtActionProc) MiddlePuzzle},
	{(char *) "Speed", (XtActionProc) SpeedPuzzle},
	{(char *) "Slow", (XtActionProc) SlowPuzzle},
	{(char *) "Sound", (XtActionProc) SoundPuzzle},
	{(char *) "Enter", (XtActionProc) EnterPuzzle},
	{(char *) "Leave", (XtActionProc) LeavePuzzle}
};

static XtResource resourcesPuzzle[] =
{
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(MlinkWidget, core.width),
	 XtRString, (caddr_t) "512"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(MlinkWidget, core.height),
	 XtRString, (caddr_t) "512"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(MlinkWidget, mlink.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNbackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(MlinkWidget, mlink.background),
	 XtRString, (caddr_t) XtDefaultBackground},
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(MlinkWidget, mlink.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverseVideo, XtCReverseVideo, XtRBoolean, sizeof (Boolean),
	 XtOffset(MlinkWidget, mlink.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNframeColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(MlinkWidget, mlink.frameColor),
	 XtRString, (caddr_t) "cyan" /*XtDefaultForeground*/},
	{XtNtileColor, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(MlinkWidget, mlink.tileColor),
	 XtRString, (caddr_t) "black" /*XtDefaultForeground*/},
	{XtNfaceColor0, XtCLabel, XtRString, sizeof (String),
	 XtOffset(MlinkWidget, mlink.faceName[0]),
	 XtRString, (caddr_t) "white"},
	{XtNfaceColor1, XtCLabel, XtRString, sizeof (String),
	 XtOffset(MlinkWidget, mlink.faceName[1]),
	 XtRString, (caddr_t) "yellow"},
	{XtNfaceColor2, XtCLabel, XtRString, sizeof (String),
	 XtOffset(MlinkWidget, mlink.faceName[2]),
	 XtRString, (caddr_t) "green"},
	{XtNfaceColor3, XtCLabel, XtRString, sizeof (String),
	 XtOffset(MlinkWidget, mlink.faceName[3]),
	 XtRString, (caddr_t) "red"},
	{XtNfaceColor4, XtCLabel, XtRString, sizeof (String),
	 XtOffset(MlinkWidget, mlink.faceName[4]),
	 XtRString, (caddr_t) "blue"},
	{XtNfaceColor5, XtCLabel, XtRString, sizeof (String),
	 XtOffset(MlinkWidget, mlink.faceName[5]),
	 XtRString, (caddr_t) "magenta"},
	{XtNfaceColor6, XtCLabel, XtRString, sizeof (String),
	 XtOffset(MlinkWidget, mlink.faceName[6]),
	 XtRString, (caddr_t) "cyan"},
	{XtNfaceColor7, XtCLabel, XtRString, sizeof (String),
	 XtOffset(MlinkWidget, mlink.faceName[7]),
	 XtRString, (caddr_t) "orange"},
	{XtNtileBorder, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(MlinkWidget, mlink.borderColor),
	 XtRString, (caddr_t) "gray25" /*XtDefaultBackground*/},
	{XtNinstall, XtCInstall, XtRBoolean, sizeof (Boolean),
	 XtOffset(MlinkWidget, mlink.install),
	 XtRString, (caddr_t) "FALSE"},
	{XtNpicture, XtCPicture, XtRString, sizeof (String),
	 XtOffset(MlinkWidget, mlink.picture),
	 XtRString, (caddr_t) PICTURE},
	{XtNdelay, XtCDelay, XtRInt, sizeof (int),
	 XtOffset(MlinkWidget, mlink.delay),
	 XtRString, (caddr_t) "10"}, /* DEFAULT_DELAY */
	{XtNsound, XtCSound, XtRBoolean, sizeof (Boolean),
	 XtOffset(MlinkWidget, mlink.sound),
	 XtRString, (caddr_t) "FALSE"},
	{XtNbumpSound, XtCBumpSound, XtRString, sizeof (String),
	 XtOffset(MlinkWidget, mlink.bumpSound),
	 XtRString, (caddr_t) BUMPSOUND},
	{XtNmoveSound, XtCMoveSound, XtRString, sizeof (String),
	 XtOffset(MlinkWidget, mlink.moveSound),
	 XtRString, (caddr_t) MOVESOUND},
	{XtNfont, XtCFont, XtRString, sizeof (String),
	 XtOffset(MlinkWidget, mlink.font),
	 XtRString, (caddr_t) "9x15bold"},
	{XtNtiles, XtCTiles, XtRInt, sizeof (int),
	 XtOffset(MlinkWidget, mlink.tiles),
	 XtRString, (caddr_t) "4"}, /* DEFAULT_TILES */
	{XtNfaces, XtCFaces, XtRInt, sizeof (int),
	 XtOffset(MlinkWidget, mlink.faces),
	 XtRString, (caddr_t) "4"}, /* DEFAULT_FACES */
	{XtNorient, XtCOrient, XtRBoolean, sizeof (Boolean),
	 XtOffset(MlinkWidget, mlink.orient),
	 XtRString, (caddr_t) "FALSE"}, /* DEFAULT_ORIENT */
	{XtNmiddle, XtCMiddle, XtRBoolean, sizeof (Boolean),
	 XtOffset(MlinkWidget, mlink.middle),
	 XtRString, (caddr_t) "TRUE"}, /* DEFAULT_MIDDLE */
	{XtNpractice, XtCPractice, XtRBoolean, sizeof (Boolean),
	 XtOffset(MlinkWidget, mlink.practice),
	 XtRString, (caddr_t) "TRUE"}, /* DEFAULT_PRACTICE */
	{XtNbase, XtCBase, XtRInt, sizeof (int),
	 XtOffset(MlinkWidget, mlink.base),
	 XtRString, (caddr_t) "10"}, /* DEFAULT_BASE */
	{XtNuserName, XtCUserName, XtRString, sizeof (String),
	 XtOffset(MlinkWidget, mlink.userName),
	 XtRString, (caddr_t) ""},
	{XtNscoreFile, XtCScoreFile, XtRString, sizeof (String),
	 XtOffset(MlinkWidget, mlink.scoreFile),
	 XtRString, (caddr_t) ""},
	{XtNscoreOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(MlinkWidget, mlink.scoreOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNversionOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(MlinkWidget, mlink.versionOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmenu, XtCMenu, XtRInt, sizeof (int),
	 XtOffset(MlinkWidget, mlink.menu),
	 XtRString, (caddr_t) "999"}, /* ACTION_IGNORE */
	{XtNstart, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(MlinkWidget, mlink.started),
	 XtRString, (caddr_t) "FALSE"},
	{XtNcheat, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(MlinkWidget, mlink.cheat),
	 XtRString, (caddr_t) "FALSE"},
	{XtNpixmapSize, XtCPixmapSize, XtRInt, sizeof (int),
	 XtOffset(MlinkWidget, mlink.pixmapSize),
	 XtRString, (caddr_t) "64"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(MlinkWidget, mlink.select),
	 XtRCallback, (caddr_t) NULL}
};

MlinkClassRec mlinkClassRec =
{
	{
		(WidgetClass) & widgetClassRec,		/* superclass */
		(char *) "Mlink",	/* class name */
		sizeof (MlinkRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializePuzzle,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListPuzzle,	/* actions */
		XtNumber(actionsListPuzzle),	/* num actions */
		resourcesPuzzle,	/* resources */
		XtNumber(resourcesPuzzle),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		(XtWidgetProc) DestroyPuzzle,	/* destroy */
		(XtWidgetProc) ResizePuzzle,	/* resize */
		(XtExposeProc) ExposePuzzle,	/* expose */
		(XtSetValuesFunc) SetValuesPuzzle,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		NULL,		/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		defaultTranslationsPuzzle,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	}
};

WidgetClass mlinkWidgetClass = (WidgetClass) & mlinkClassRec;

void
setPuzzle(MlinkWidget w, int reason)
{
	mlinkCallbackStruct cb;

	cb.reason = reason;
	XtCallCallbacks((Widget) w, (char *) XtNselectCallback, &cb);
}
#endif

static void
loadFont(MlinkWidget w)
{
#ifndef WINVER
	Display *display = XtDisplay(w);
	const char *altfontname = "-*-times-*-r-*-*-*-180-*";
	char buf[512];

	if (w->mlink.fontInfo) {
		XUnloadFont(display, w->mlink.fontInfo->fid);
		XFreeFont(display, w->mlink.fontInfo);
	}
	if (w->mlink.font && (w->mlink.fontInfo =
			XLoadQueryFont(display, w->mlink.font)) == NULL) {
		(void) sprintf(buf,
			"Can not open %s font.\nAttempting %s font as alternate\n",
			w->mlink.font, altfontname);
		DISPLAY_WARNING(buf);
		if ((w->mlink.fontInfo = XLoadQueryFont(display,
				altfontname)) == NULL) {
			(void) sprintf(buf,
				"Can not open %s alternate font.\nUse the -font option to specify a font to use.\n",
				altfontname);
			DISPLAY_WARNING(buf);
		}
	}
	if (w->mlink.fontInfo) {
		w->mlink.digitOffset.x = XTextWidth(w->mlink.fontInfo, "8", 1)
			/ 2;
		w->mlink.digitOffset.y = w->mlink.fontInfo->max_bounds.ascent
			/ 2;
	} else
#endif
	{
		w->mlink.digitOffset.x = 3;
		w->mlink.digitOffset.y = 4;
	}
}

#ifndef LOGFILE
#define LOGFILE "mlink.log"
#endif

static MlinkStack undo = {NULL, NULL, NULL, 0};
static MlinkStack redo = {NULL, NULL, NULL, 0};

static void
CheckTiles(MlinkWidget w)
{
	char *buf1 = NULL, *buf2 = NULL;

	if (w->mlink.tiles < MIN_TILES) {
		intCat(&buf1,
			"Number of tiles in X direction out of bounds, use at least ",
			MIN_TILES);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, DEFAULT_TILES);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->mlink.tiles = DEFAULT_TILES;
	}
	if (w->mlink.faces < MIN_FACES || w->mlink.faces > MAX_FACES) {
		intCat(&buf1,
			"Number of faces out of bounds, use ",
			MIN_FACES);
		stringCat(&buf2, buf1, "..");
		free(buf1);
		intCat(&buf1, buf2, MAX_FACES);
		free(buf2);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, DEFAULT_FACES);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->mlink.faces = DEFAULT_FACES;
	}
	if (w->mlink.base < MIN_BASE || w->mlink.base > MAX_BASE) {
		intCat(&buf1, "Base out of bounds, use ", MIN_BASE);
		stringCat(&buf2, buf1, "..");
		free(buf1);
		intCat(&buf1, buf2, MAX_BASE);
		free(buf2);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, DEFAULT_BASE);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->mlink.base = DEFAULT_BASE;
	}
}

static Boolean
PositionTile(MlinkWidget w, int x, int *r)
{
	*r = (x - w->mlink.delta.x / 2 - w->mlink.puzzleOffset.x) /
		w->mlink.offset.x;
	if (*r < 0 || *r >= w->mlink.tiles)
		return False;
	return True;
}

static int
Column(MlinkWidget w, int pos)
{
	return (pos % w->mlink.tiles);
}

static int
Row(MlinkWidget w, int pos)
{
	return (pos / w->mlink.tiles);
}

static int
CartesianX(MlinkWidget w, int pos)
{
	return Column(w, pos) * w->mlink.offset.x + w->mlink.delta.x +
		w->mlink.puzzleOffset.x;
}

static int
CartesianY(MlinkWidget w, int pos)
{
	return Row(w, pos) * (w->mlink.offset.y + 1) + w->mlink.delta.y +
		w->mlink.puzzleOffset.y;
}

static int
TileNFromSpace(MlinkWidget w, int n, int direction)
{
	int pos = w->mlink.spacePosition;

	if (direction == LEFT || direction == RIGHT)
		return (pos + ((direction == RIGHT) ? -n : n));
	else /* direction == TOP || direction == BOTTOM */
		return (pos + (w->mlink.tiles * ((direction == BOTTOM) ?
			-n : n)));
}

static int
SolvedPosition(MlinkWidget w, int tile) {
	return (tile + 1) % w->mlink.tileFaces;
}

Boolean
CheckSolved(const MlinkWidget w)
{
	int i, j, firstTile, lastTile, midTile;

	if (w->mlink.tiles < 2)
		return True;
	if (w->mlink.orient) {
		firstTile = w->mlink.tileOfPosition[0];
		if (firstTile != 0 && (firstTile - 1) % w->mlink.tiles == 0) {
			for (i = 1; i < w->mlink.tileFaces; i++) {
				midTile = w->mlink.tileOfPosition[i];
				if (midTile && (midTile - firstTile + w->mlink.tileFaces) %
						w->mlink.tileFaces != i)
					return False;
			}
		} else
			return False;
	} else {
		for (i = 0; i < w->mlink.faces; i++) {
			firstTile = w->mlink.tileOfPosition[i * w->mlink.tiles];
			if (firstTile == 0)
				firstTile = w->mlink.tileOfPosition[i * w->mlink.tiles + 1];
			lastTile = w->mlink.tileOfPosition[(i + 1) * w->mlink.tiles - 1];
			if (lastTile == 0)
				lastTile = w->mlink.tileOfPosition[(i + 1) * w->mlink.tiles - 2];
			if (firstTile % w->mlink.tiles != 1 ||
					(lastTile % w->mlink.tiles != 0 &&
					lastTile != w->mlink.tileFaces - 1) ||
					(lastTile - 1) / w->mlink.tiles !=
					(firstTile - 1) / w->mlink.tiles)
				return False;
			for (j = 1; j < w->mlink.tiles - 1; j++) {
				midTile = w->mlink.tileOfPosition[i * w->mlink.tiles + j];
				if ((midTile - 1) / w->mlink.tiles !=
						(firstTile - 1) /
						w->mlink.tiles || midTile == 0)
					return False;
			}
		}
	}
	return True;
}

static int
int2String(MlinkWidget w, char *buf, int number, int base, Boolean capital)
{
	int digit, mult = base, last, position;
	int a, i, j, s, r;

	if (capital) {
		a = 'A', i = 'I', j = 'J', s = 'S', r = 'R';
	} else {
		a = 'a', i = 'i', j = 'j', s = 's', r = 'r';
	}
	if (number < 0) {
		char *buff;

		intCat(&buff, "int2String: 0 > ", number);
		DISPLAY_WARNING(buff);
		free(buff);
		return 0;
	}
	last = 1;
	while (number >= mult) {
		last++;
		mult *= base;
	}
	for (position = 0; position < last; position++) {
		mult /= base;
		digit = number / mult;
		number -= digit * mult;
		buf[position] = digit + '0';
		if (buf[position] > '9') {	/* ASCII */
			buf[position] += (a - '9' - 1);
		} else if (buf[position] < '0') {	/* EBCDIC */
			buf[position] += (a - '9' - 1);
			if (buf[position] > i)
				buf[position] += (j - i - 1);
			if (buf[position] > r)
				buf[position] += (s - r - 1);
		}
	}
	buf[last] = '\0';
	return last;
}

static void
fill3DRect(MlinkWidget w, Pixmap dr, GC gc, GC darkerGC, GC brighterGC,
		int x, int y, int width, int height, Boolean raised)
{
	GC currentGC = (raised) ? gc : darkerGC;

	if (width > 2 && height > 2) {
		FILLRECTANGLE(w, dr, currentGC,
			x + 1, y + 1, width - 2, height - 2);
	}
	currentGC = (raised) ? brighterGC : darkerGC;
	FILLRECTANGLE(w, dr, currentGC,
		x, y, 1, height - 1);
	if (width > 1) {
		FILLRECTANGLE(w, dr, currentGC,
			x + 1, y, width - 2, 1);
	}
	currentGC = (raised) ? darkerGC : gc;
	if (width > 1 && height > 1) {
		FILLRECTANGLE(w, dr, currentGC,
			x, y + height - 1, width, 1);
	}
	if (width > 1 && height > 1) {
		FILLRECTANGLE(w, dr, currentGC,
			x + width - 1, y, 1, height - 1);
	}
}

#define THICKNESS 7
static void
DrawLink(MlinkWidget w, Pixmap dr, GC gc, int pos, int offsetX, int offsetY)
{
	int dx, dy, sizex, sizey, thick;
	int tile = SolvedPosition(w, pos);
	int angle1, angle2;

	sizex = w->mlink.offset.x * 3 / 2 - 6;
	sizey = w->mlink.offset.y * 3 / 4 - 6;
	dx = CartesianX(w, pos) - sizex / 2 + offsetX + 2;
	dy = CartesianY(w, pos) + w->mlink.tileSize.y / 2 - sizey / 2 -
		1 + offsetY;
	thick = MIN(sizey, sizex) / 3;
	thick = MIN(thick, THICKNESS);
	if (thick > 0) {
		if (tile % w->mlink.tiles == 0 || tile == w->mlink.tileFaces - 1) {

#ifdef WINVER
			angle1 = 87, angle2 = -174;
#else
			angle1 = 89, angle2 = -178;
#endif
			DRAWARC(w, dr, gc, thick, dx, dy,
				sizex, sizey, angle1, angle2);
		} else if (tile % w->mlink.tiles == 1) {
#ifdef WINVER
			angle1 = 93, angle2 = 174;
#else
			angle1 = 90, angle2 = 180;
#endif
			DRAWARC(w, dr, gc, thick, dx + w->mlink.tileSize.x - 5, dy,
				sizex, sizey, angle1, angle2);
		} else {
#ifdef WINVER
			angle1 = 87, angle2 = -120;
#else
			angle1 = 89, angle2 = -124;
#endif
			DRAWARC(w, dr, gc, thick, dx, dy,
				sizex, sizey, angle1, angle2);
#ifdef WINVER
			angle1 = -87, angle2 = 25;
#else
			angle1 = -89, angle2 = 29;
#endif
			DRAWARC(w, dr, gc, thick, dx, dy,
				sizex, sizey, angle1, angle2);
			dx += w->mlink.tileSize.x - 5;
#ifdef WINVER
			angle1 = 93, angle2 = 25;
#else
			angle1 = 90, angle2 = 30;
#endif
			DRAWARC(w, dr, gc, thick, dx, dy,
				sizex, sizey, angle1, angle2);
#ifdef WINVER
			angle1 = -93, angle2 = -120;
#else
			angle1 = -90, angle2 = -125;
#endif
			DRAWARC(w, dr, gc, thick, dx, dy,
				sizex, sizey, angle1, angle2);
		}
	}
}

#if 0
#define THICKNESS 8
static void
DrawLink(MlinkWidget w, Pixmap dr, GC gc, int pos, int offsetX, int offsetY)
{
	int dx, dy, sizex, sizey, i;
	int tile = SolvedPosition(w, pos);

	for (i = 0; i < THICKNESS; i++) {
		sizex = w->mlink.offset.x * 3 / 2 - i;
		sizey = w->mlink.offset.y * 3 / 4 - i;
		dx = CartesianX(w, pos) - sizex / 2 + offsetX;
		dy = CartesianY(w, pos) + w->mlink.tileSize.y / 2 - sizey / 2 +
			offsetY;
		if (tile % w->mlink.tiles == 0 ||
				tile == w->mlink.tileFaces - 1) {
			DRAWARC(w, dr, gc, 1, gc, dx, dy,
				sizex, sizey, 89 * MULT, -179 * MULT);
		} else if (tile % w->mlink.tiles == 1) {
			DRAWARC(w, dr, gc, 1, gc,
				dx + w->mlink.tileSize.x - 1, dy,
				sizex, sizey, 90 * MULT, 180 * MULT);
		} else {
			DRAWARC(w, dr, gc, 1, gc, dx, dy,
				sizex, sizey, 89 * MULT, -32 * MULT);
			DRAWARC(w, dr, gc, 1, gc, dx, dy,
				sizex, sizey, -90 * MULT, 128 * MULT);
			dx += w->mlink.tileSize.x - 1;
			DRAWARC(w, dr, gc, 1, gc, dx, dy,
				sizex, sizey, 90 * MULT, 128 * MULT);
			DRAWARC(w, dr, gc, 1, gc, dx, dy,
				sizex, sizey, -90 * MULT, -33 * MULT);
		}
	}
}
#endif

static void
drawShadow(MlinkWidget w, Pixmap dr, GC gc, int startX, int startY,
		int sizeX, int sizeY)
{
	FILLRECTANGLE(w, dr, gc,
		startX, startY, sizeX + 1, 1);
	FILLRECTANGLE(w, dr, gc,
		startX, startY, 1, sizeY + 1);
}

static void
DrawTile(MlinkWidget w, int pos, Boolean blank, Boolean erase,
		int pressedOffset, int offsetX, int offsetY)
{
	int dx, dy, sx, sy;
	int tile = (w->mlink.tileOfPosition[pos] + w->mlink.tileFaces - 1) %
		w->mlink.tileFaces;
	Pixmap dr = 0;

	dx = CartesianX(w, pos) + pressedOffset + offsetX;
	dy = CartesianY(w, pos) + pressedOffset + offsetY;
	sx = CartesianX(w, tile);
	sy = CartesianY(w, tile);
	if (blank) {
		FILLRECTANGLE(w, dr,
			(erase) ? w->mlink.inverseGC : w->mlink.tileGC[1],
			dx, dy,
			w->mlink.tileSize.x, w->mlink.tileSize.y);
		return;
	}
#ifdef WINVER
	w->core.hOldBitmap = (HBITMAP) SelectObject(w->core.memDC,
		w->mlink.bufferTiles[pressedOffset]);
	BitBlt(w->core.hDC,
		dx, dy,
		w->mlink.tileSize.x, w->mlink.tileSize.y,
		w->core.memDC,
		sx, sy,
		SRCCOPY);
	(void) SelectObject(w->core.memDC, w->core.hOldBitmap);
#else
	XSetGraphicsExposures(XtDisplay(w), w->mlink.tileGC[1], False);
	XCopyArea(XtDisplay(w),
		w->mlink.bufferTiles[pressedOffset],
		XtWindow(w),
		w->mlink.tileGC[1],
		sx, sy,
		w->mlink.tileSize.x, w->mlink.tileSize.y,
		dx, dy);
#endif
}

static void
DrawBufferedTile(MlinkWidget w, int pos, int pressedOffset)
{
	Pixmap *dr;
	GC linkGC, tileGC, borderGC;
	int dx, dy, tile = SolvedPosition(w, pos);
	int i = 0, digitOffsetX = 0, t = tile;
	char buf[66];

	dr = &(w->mlink.bufferTiles[pressedOffset]);
	tileGC = w->mlink.tileGC[1];
	borderGC = w->mlink.borderGC;
	if (w->mlink.mono && pressedOffset) {
		linkGC = borderGC;
	} else {
		int face = ((tile - 1) / w->mlink.tiles + 1) % w->mlink.faces;

		linkGC = w->mlink.faceGC[face];
#ifndef WINVER
		if (!w->mlink.mono) {
			if (w->mlink.faceColor[face] == w->mlink.tileColor) {
				linkGC = (pressedOffset) ? tileGC : borderGC;
			}
		}
#endif
	}
	dx = CartesianX(w, pos) + pressedOffset;
	dy = CartesianY(w, pos) + pressedOffset;
	if (pressedOffset != 0) {
		drawShadow(w, *dr, w->mlink.tileGC[2],
			dx - pressedOffset, dy - pressedOffset,
			w->mlink.tileSize.x - 1, w->mlink.tileSize.y - 1);
	}
	fill3DRect(w, *dr, tileGC,
		w->mlink.tileGC[2], w->mlink.tileGC[0],
		dx, dy, w->mlink.tileSize.x, w->mlink.tileSize.y,
		pressedOffset == 0);
	buf[0] = '\0';
	if (w->mlink.orient || w->mlink.mono) {
		if (w->mlink.mono) {
			if (w->mlink.orient) {
				int last;

				last = int2String(w, buf, tile, w->mlink.base, False);
				buf[last] =
#ifdef WINVER
					w->mlink.faceChar[((tile - 1) / w->mlink.tiles + 1) %
						w->mlink.faces];
#else
					w->mlink.faceName[((tile - 1) / w->mlink.tiles + 1) %
						w->mlink.faces][0];
#endif
					buf[last + 1] = '\0';
					t *= w->mlink.base;
			} else {
				buf[0] =
#ifdef WINVER
					w->mlink.faceChar[((tile - 1) / w->mlink.tiles + 1) %
						w->mlink.faces];
#else
					w->mlink.faceName[((tile - 1) / w->mlink.tiles + 1) %
							w->mlink.faces][0];
#endif
				buf[1] = '\0';
				t = 1;
			}
		} else {
			(void) int2String(w, buf, tile, w->mlink.base, True);
		}
		while (t >= 1) {
			t /= w->mlink.base;
			digitOffsetX += w->mlink.digitOffset.x;
			i++;
		}
		DRAWTEXT(w, *dr, linkGC,
			dx + w->mlink.tileSize.x / 2 - digitOffsetX,
			dy + w->mlink.tileSize.y / 2 + w->mlink.digitOffset.y,
			buf, i);
	}
	if (tile != 0 && ((tile != w->mlink.tileFaces - 1 &&
			w->mlink.tiles > 1) || w->mlink.tiles > 2))
		DrawLink(w, *dr, linkGC, pos, pressedOffset, pressedOffset);
#if 0
	DRAWRECTANGLE(w, *dr, borderGC, dx, dy,
		w->mlink.tileSize.x - 1, w->mlink.tileSize.y - 1);
#endif
}

static void
DrawAllBufferedTiles(const MlinkWidget w)
{
	int k, l;

	for (k = 0; k < w->mlink.tileFaces; k++)
		for (l = 0; l < 2; l++)
			DrawBufferedTile(w, k, l);
}

void
DrawAllTiles(const MlinkWidget w)
{
	int k;

	for (k = 0; k < w->mlink.tileFaces; k++)
		DrawTile(w, k, (w->mlink.tileOfPosition[k] <= 0),
			(w->mlink.tileOfPosition[k] <= 0), FALSE, 0, 0);
}

static void
AnimateSlide(MlinkWidget w, int numTiles, int dir, int fast, Boolean logMoves)
{
	int inc, aTile;
	int fillBeginPos, fillBeginI = 0, gapI = 0, moveI = 0;
	int dx, dy, pos, posNext;
	int ix = 0, iy = 0;
	Pixmap dr = 0;

	fillBeginPos = TileNFromSpace(w, numTiles, dir);
	fillBeginI = CartesianX(w, fillBeginPos);
	dy = CartesianY(w, fillBeginPos);
	gapI = w->mlink.tileSize.x * fast / w->mlink.numSlices;
	moveI = w->mlink.tileSize.x + w->mlink.delta.x;
	if (gapI == 0)
		gapI++;
	if (numTiles < 0)
		numTiles = -numTiles;
	FLUSH(w);
	initTimer(w->mlink.oldTime);
	for (inc = 0; inc < moveI + gapI; inc += gapI) {
	  if (inc > moveI)
	    inc = moveI;
	  for (aTile = 0; aTile < numTiles; aTile++) {
	    posNext = TileNFromSpace(w, aTile + 1, dir);
	    if (logMoves && inc == 0) {
	      setPuzzle(w, ACTION_MOVED);
	      setMove(&undo, dir, posNext, 0);
	      flushMoves(w, &redo, FALSE);
	    }
	    /* Calculate deltas */
	    dx = CartesianX(w, posNext);
	    dy = CartesianY(w, posNext);
	    ix = ((dir == RIGHT) ? inc : -inc);
	    iy = 0;
	    DrawTile(w, posNext, False, False, FALSE, ix, iy);
	    /* Erase old slivers */
	    ix += dx;
	    iy += dy;
	    if (aTile < numTiles - 1) {
	      switch (dir) {
	      case RIGHT:
		FILLRECTANGLE(w, dr,
		  w->mlink.inverseGC,
		  ix - gapI, iy, gapI, w->mlink.tileSize.y);
		break;
	      case LEFT:
		FILLRECTANGLE(w, dr,
		  w->mlink.inverseGC,
		  ix + w->mlink.tileSize.x, iy,
		  gapI, w->mlink.tileSize.y);
		break;
	      }
	    } else {
	      switch (dir) {
	      case RIGHT:
		FILLRECTANGLE(w, dr,
		  w->mlink.inverseGC,
		  fillBeginI, iy,
		  ix - fillBeginI, w->mlink.tileSize.y);
		break;
	      case LEFT:
		FILLRECTANGLE(w, dr,
		  w->mlink.inverseGC,
		  ix + w->mlink.tileSize.x, iy,
		  fillBeginI - ix, w->mlink.tileSize.y);
		break;
	      }
	    }
	  }
	  FLUSH(w);
	  useTimer(&(w->mlink.oldTime), w->mlink.delay / fast);
	}
	pos = w->mlink.spacePosition;
	for (aTile = 0; aTile < numTiles; aTile++) {
	  posNext = TileNFromSpace(w, aTile + 1, dir);
	  w->mlink.tileOfPosition[pos] =
	    w->mlink.tileOfPosition[posNext];
	  pos = posNext;
	}
	w->mlink.spacePosition = fillBeginPos;
	w->mlink.tileOfPosition[fillBeginPos] = 0;
}

static void
DrawAllTilesForColumns(MlinkWidget w, int start, int finish)
{
	int i, j, k;

	for (i = start; i <= finish; i++) {
		for (j = 0; j < w->mlink.faces; j++) {
			k = j * w->mlink.tiles + i;
			if (w->mlink.tileOfPosition[k] > 0)
				DrawTile(w, k,
					(w->mlink.tileOfPosition[k] <= 0),
					(w->mlink.tileOfPosition[k] <= 0),
					FALSE, 0, 0);
		}
	}
}

static void
AnimateRotate(MlinkWidget w, int start, int finish, int direction, int fast)
{
#ifdef WINVER
	HDC memDC;
	HBITMAP hBitmap, hOldBitmap;
#else
	Pixmap bufferBox;
	XWindowAttributes xgwa;
#endif
	int size = finish - start + 1;
	int moveI, gapJ, moveJ, inc, i, offset = 0;

	moveI = w->mlink.tileSize.x + w->mlink.delta.x;
	gapJ = w->mlink.tileSize.y * fast / w->mlink.numSlices;
	moveJ = w->mlink.tileSize.y + w->mlink.delta.y + 1;
	if (gapJ == 0)
		gapJ++;

#ifdef WINVER
	memDC = CreateCompatibleDC(w->core.hDC);
	hBitmap = CreateCompatibleBitmap(w->core.hDC, size * moveI + 1,
			w->mlink.puzzleSize.y);
	hOldBitmap = SelectObject(memDC, hBitmap);
	BitBlt(memDC, 0, 0,
		size * moveI,
		w->mlink.puzzleSize.y,
		w->core.hDC,
		start * moveI + w->mlink.puzzleOffset.x + 1,
		w->mlink.puzzleOffset.y,
		SRCCOPY);
	(void) SelectObject(memDC, hOldBitmap);
#else
	(void) XGetWindowAttributes(XtDisplay(w), XtWindow(w), &xgwa);
	if ((bufferBox = XCreatePixmap(XtDisplay(w), XtWindow(w),
			size * moveI,
			w->mlink.puzzleSize.y,
			xgwa.depth)) == None) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	XCopyArea(XtDisplay(w), XtWindow(w), bufferBox,
		w->mlink.frameGC,
		start * moveI + w->mlink.puzzleOffset.x + 1,
		w->mlink.puzzleOffset.y,
		size * moveI,
		w->mlink.puzzleSize.y,
		0, 0);
	XSetGraphicsExposures(XtDisplay(w), w->mlink.frameGC, False);
#endif
	FLUSH(w);
	initTimer(w->mlink.oldTime);
	for (inc = 0; inc < moveJ + gapJ; inc += gapJ) {
		if (inc > moveJ)
			inc = moveJ;
		if (direction == TOP) {
			i = inc;
		} else {
			i = w->mlink.puzzleSize.y - inc + 1;
			offset = 3;
		}
#ifdef WINVER
		hOldBitmap = SelectObject(memDC, hBitmap);
		BitBlt(w->core.hDC,
			start * moveI + w->mlink.puzzleOffset.x + 1,
			w->mlink.puzzleOffset.y + offset,
			size * moveI,
			w->mlink.puzzleSize.y - i - 1,
			memDC,
			0, i,
			SRCCOPY);
		BitBlt(w->core.hDC,
			start * moveI + w->mlink.puzzleOffset.x + 1,
			w->mlink.puzzleSize.y - i + w->mlink.puzzleOffset.y -
			w->mlink.delta.y - 1 + offset,
			size * moveI,
			i - w->mlink.delta.y - 1,
			memDC,
			0, 0,
			SRCCOPY);
		(void) SelectObject(memDC, hOldBitmap);
#else
		XCopyArea(XtDisplay(w), bufferBox, XtWindow(w),
			w->mlink.frameGC,
			0, i,
			size * moveI,
			w->mlink.puzzleSize.y - i - 1,
			start * moveI + w->mlink.puzzleOffset.x + 1,
			w->mlink.puzzleOffset.y + offset);
		XCopyArea(XtDisplay(w), bufferBox, XtWindow(w),
			w->mlink.frameGC,
			0, 0,
			size * moveI,
			i - w->mlink.delta.y - 1,
			start * moveI + w->mlink.puzzleOffset.x + 1,
			w->mlink.puzzleSize.y - i + w->mlink.puzzleOffset.y -
			w->mlink.delta.y - 1 + offset);
#endif
		FLUSH(w);
		useTimer(&(w->mlink.oldTime), w->mlink.delay / fast);
	}
#ifdef WINVER
	DeleteObject(hBitmap);
	DeleteDC(memDC);
#else
	XFreePixmap(XtDisplay(w), bufferBox);
#endif
}

static void
DrawFrame(MlinkWidget w, Pixmap dr, Boolean focus)
{
	int sumX, sumY, offsetX, offsetY, k;
	GC gc = (focus) ? w->mlink.frameGC : w->mlink.borderGC;

	offsetX = w->mlink.puzzleOffset.x;
	offsetY = w->mlink.puzzleOffset.y;
	FILLRECTANGLE(w, dr, w->mlink.borderGC,
		0, 0, w->core.width, offsetY);
	FILLRECTANGLE(w, dr, w->mlink.borderGC,
		0, w->core.height - offsetY,
		w->core.width, offsetY);
	FILLRECTANGLE(w, dr, w->mlink.borderGC,
		0, 0, offsetX, w->core.height);
	FILLRECTANGLE(w, dr, w->mlink.borderGC,
		w->core.width - offsetX, 0,
		offsetX, w->core.height);


	sumX = w->mlink.tiles * w->mlink.offset.x + 1;
	sumY = w->mlink.offset.y + 1;
	for (k = 0; k < w->mlink.faces; k++) {
		FILLRECTANGLE(w, dr, gc,
			offsetX, offsetY, 1, sumY);
		FILLRECTANGLE(w, dr, gc,
			offsetX, offsetY, sumX, 1);
		FILLRECTANGLE(w, dr, gc,
			sumX + offsetX, offsetY, 1, sumY + 1);
		offsetY += sumY;
		FILLRECTANGLE(w, dr, gc,
			offsetX, offsetY, sumX + 1, 1);
	}
}

static void
EraseFrame(const MlinkWidget w, Pixmap dr)
{
	FILLRECTANGLE(w, dr, w->mlink.inverseGC,
		0, 0, w->core.width, w->core.height);
}

static void
MoveNoTiles(MlinkWidget w)
{
	setPuzzle(w, ACTION_IGNORE);
}

static void
MoveTiles(MlinkWidget w, int from, int fast)
{
	int i =	w->mlink.currentTile = Column(w, from);
	int j = w->mlink.currentFace = Row(w, from);
	int l = i + w->mlink.tiles * j - w->mlink.spacePosition;

	w->mlink.currentRef = from;
	if (l / w->mlink.tiles == 0 &&
			j == Row(w, w->mlink.spacePosition)) {
		if (fast != INSTANT && w->mlink.delay > 0) {

			if (l < 0) {
				AnimateSlide(w, -l, RIGHT, fast, False);
			} else { /* (l > 0) */
				AnimateSlide(w, l, LEFT, fast, False);
			}
		} else {
			int pos, posNext, aTile, dir;

			dir = (l < 0) ? RIGHT : LEFT;
			if (l < 0)
				l = -l;
			pos = w->mlink.spacePosition;
			for (aTile = 0; aTile < l; aTile++) {
				posNext = TileNFromSpace(w, aTile + 1, dir);
				w->mlink.tileOfPosition[pos] =
					w->mlink.tileOfPosition[posNext];
				DrawTile(w, pos, False, False, FALSE, 0, 0);
				pos = posNext;
			}
			w->mlink.spacePosition = from;
			w->mlink.tileOfPosition[from] = 0;
			DrawTile(w, w->mlink.spacePosition, True, True,
				FALSE, 0, 0);
		}
	}
	w->mlink.currentRef = -w->mlink.tileFaces;
#ifdef USE_SOUND
	if (w->mlink.sound) {
		playSound((char *) BUMPSOUND);
	}
#endif
}

static void
RotateTiles(MlinkWidget w, int start, int finish, int direction, int fast)
{
	int newR, i, k, tempTile, currTile, pos, newPos;
	int r = 0, space = -1;
	Boolean	animate = (fast != INSTANT && w->mlink.delay > 0);

	w->mlink.currentRef = start;
	if (animate)
		AnimateRotate(w, start, finish, direction, fast);
	for (i = start; i <= finish; i++) {
		currTile = w->mlink.tileOfPosition[r * w->mlink.tiles + i];
		for (k = 1; k <= w->mlink.faces; k++) {
			newR = (direction == TOP) ?
				(r + w->mlink.faces - 1) % w->mlink.faces :
				(r + 1) % w->mlink.faces;
			pos = r * w->mlink.tiles + i;
			newPos = newR * w->mlink.tiles + i;
			tempTile = w->mlink.tileOfPosition[newPos];
			w->mlink.tileOfPosition[newPos] = currTile;
			currTile = tempTile;
			if (w->mlink.spacePosition == pos) {
				if (!animate)
					DrawTile(w, newPos, True, True,
						FALSE, 0, 0);
				space = newPos;
			} else if (!animate)
				DrawTile(w, newPos, False, False,
					FALSE, 0, 0);
			r = newR;
		}
		if (space >= 0)
			w->mlink.spacePosition = space;
	}
	/* Needed in case a window is on top. */
	if (animate) {
		int moveI = w->mlink.tileSize.x + w->mlink.delta.x;
		Pixmap dr = 0;

		FILLRECTANGLE(w, dr, w->mlink.inverseGC,
			start * moveI + w->mlink.puzzleOffset.x + 1, 0,
			(finish - start + 1) * moveI, w->core.height);
		DrawFrame(w, dr, w->mlink.focus);
		DrawAllTilesForColumns(w, start, finish);
	}
	w->mlink.currentRef = -w->mlink.tileFaces;
#ifdef USE_SOUND
	if (w->mlink.sound) {
		playSound((char *) MOVESOUND);
	}
#endif
}

static Boolean
MoveTilesDir(MlinkWidget w, int direction, int tile, Boolean all, int fast)
{
	int start, finish;

	start = tile;
	finish = tile;
	if (all) {
		start = 0;
		finish = w->mlink.tiles - 1;
	} else if (w->mlink.middle && tile > 0 && tile < w->mlink.tiles - 1) {
		start = 1;
		finish = w->mlink.tiles - 2;
	}
	switch (direction) {
	case TOP:
		RotateTiles(w, start, finish, TOP, fast);
		return True;
	case RIGHT:
		if (tile >= 0 && Row(w, w->mlink.spacePosition) ==
				Row(w, tile)) {
			MoveTiles(w, tile, fast);
			return True;
		}
		break;
	case BOTTOM:
		RotateTiles(w, start, finish, BOTTOM, fast);
		return True;
	case LEFT:
		if (Row(w, w->mlink.spacePosition) == Row(w, tile)) {
			MoveTiles(w, tile, fast);
			return True;
		}
		break;
	default:
		{
			char *buf;
			intCat(&buf, "MoveTilesDir: direction ", direction);
			DISPLAY_WARNING(buf);
			free(buf);
		}
	}
	return False;
}

static int
ExchangeTiles(MlinkWidget w, int pos1, int pos2)
{
	int tempTile;

	if (w->mlink.tileOfPosition[pos1] <= 0)
		return FALSE;
	else if (w->mlink.tileOfPosition[pos2] <= 0)
		return FALSE;
	tempTile = w->mlink.tileOfPosition[pos1];
	w->mlink.tileOfPosition[pos1] = w->mlink.tileOfPosition[pos2];
	w->mlink.tileOfPosition[pos2] = tempTile;
	return TRUE;
}

static void
DiscreteMoves(MlinkWidget w, int from, int dir)
{
	MoveTiles(w, from, NORMAL);
	setPuzzle(w, ACTION_MOVED);
	setMove(&undo, dir, from, 0);
	flushMoves(w, &redo, FALSE);
}

static void
MoveShift(MlinkWidget w, int direction, int fast)
{
	(void) MoveTilesDir(w, direction, 0, True, fast);
	setPuzzle(w, ACTION_CONTROL);
}

static void
ControlPuzzle(MlinkWidget w, int direction)
{
	switch (direction) {
	case TOP:
		if (w->mlink.faces <= MIN_FACES)
			return;
		setPuzzle(w, ACTION_DECY);
		break;
	case RIGHT:
		setPuzzle(w, ACTION_INCX);
		break;
	case BOTTOM:
		if (w->mlink.faces >= MAX_FACES)
			return;
		setPuzzle(w, ACTION_INCY);
		break;
	case LEFT:
		if (w->mlink.tiles <= MIN_TILES)
			return;
		setPuzzle(w, ACTION_DECX);
		break;
	default:
		{
			char *buf;

			intCat(&buf, "ControlMlink: direction ",
				direction);
			DISPLAY_WARNING(buf);
			free(buf);
			setPuzzle(w, ACTION_IGNORE);
		}
	}
}

Boolean
MovePuzzle(MlinkWidget w, const int direction, const int tile, const int shift,
		const Boolean motion, const int fast)
{
	if (shift != 0 && (direction == TOP || direction == BOTTOM)) {
		MoveShift(w, direction, fast);
		if (!motion) {
			setMove(&undo, direction, tile, 1);
			flushMoves(w, &redo, FALSE);
		}
		return True;
	} else if (MoveTilesDir(w, direction, tile, False, fast)) {
		if (!motion) {
			setPuzzle(w, ACTION_MOVED);
			setMove(&undo, direction, tile, 0);
			flushMoves(w, &redo, FALSE);
		}
		return True;
	}
	return False;
}

static void
DiffMove(MlinkWidget w, int diff, int j, int shift)
{
	int k;

	if (diff > w->mlink.faces / 2 ||
			(diff == (w->mlink.faces + 1) / 2 &&
			j > w->mlink.currentFace))
		for (k = 0; k < w->mlink.faces - diff; k++)
			(void) MovePuzzle(w, BOTTOM, w->mlink.currentTile,
				shift, True, NORMAL);
	else
		for (k = 0; k < diff; k++)
			(void) MovePuzzle(w, TOP, w->mlink.currentTile,
				shift, True, NORMAL);
}

static void
MovePosition(MlinkWidget w, const int direction, const int tile, const int shift)
{
	if (shift != 0 && (direction == TOP || direction == BOTTOM)) {
		setMove(&undo, direction, tile, 1);
	} else {
		setPuzzle(w, ACTION_MOVED);
		setMove(&undo, direction, tile, 0);
	}
	flushMoves(w, &redo, FALSE);
}

static void
DiffPosition(MlinkWidget w, int diff, int j, int shift)
{
	int k;

	if (diff > w->mlink.faces / 2 ||
			(diff == (w->mlink.faces + 1) / 2 &&
			j > w->mlink.currentFace))
		for (k = 0; k < w->mlink.faces - diff; k++)
			(void) MovePosition(w, BOTTOM,
				w->mlink.currentTile, shift);
	else
		for (k = 0; k < diff; k++)
			(void) MovePosition(w, TOP,
				w->mlink.currentTile, shift);
}

void
MovePuzzleDelay(MlinkWidget w,
		const int direction, const int tile, const Boolean all)
{
	int n, pos;

	if (direction == RIGHT) {
		pos = tile - w->mlink.spacePosition;
		for (n = (-pos) % w->mlink.tiles - 1; n >= 0; n--) {
			setPuzzle(w, ACTION_MOVED);
			setMove(&undo,
				direction, tile + n, False);
		}
	} else if (direction == LEFT) {
		pos = tile - w->mlink.spacePosition;
		for (n = pos % w->mlink.tiles - 1; n >= 0; n--) {
			setPuzzle(w, ACTION_MOVED);
			setMove(&undo,
				direction, tile - n, False);
		}
	} else {
		setPuzzle(w, ACTION_MOVED);
		setMove(&undo, direction, tile, all);
	}
	flushMoves(w, &redo, FALSE);
	(void) MoveTilesDir(w, direction, tile, all, NORMAL);
	Sleep((unsigned int) w->mlink.delay);
}

static void
SlidePuzzle(MlinkWidget w, int direction, int pos)
{
	if (CheckSolved(w) && !w->mlink.practice) {
		MoveNoTiles(w);
		return;
	}
	if (!MovePuzzle(w, direction, pos, FALSE, False, NORMAL)) {
		setPuzzle(w, ACTION_BLOCKED);
		return;
	}
	if (CheckSolved(w)) {
		setPuzzle(w, ACTION_SOLVED);
	}
	return;
}

static void
RotatePuzzle(MlinkWidget w, int direction, int tile)
{
	if (CheckSolved(w) && !w->mlink.practice) {
		MoveNoTiles(w);
		return;
	}
	(void) MovePuzzle(w, direction, tile, FALSE, False, NORMAL);
	if (CheckSolved(w)) {
		setPuzzle(w, ACTION_SOLVED);
	}
	return;
}

#ifndef WINVER
static
#endif
void
MovePuzzleInput(MlinkWidget w, int x, int direction, int shift, int control)
{
	int r;

	if (control != 0)
		ControlPuzzle(w, direction);
	else if (shift != 0 && (direction == TOP || direction == BOTTOM)) {
		MoveShift(w, direction, NORMAL);
		setMove(&undo, direction, 0, 1);
		flushMoves(w, &redo, FALSE);
	} else if (CheckSolved(w) && !w->mlink.practice) {
		MoveNoTiles(w);
		return;
	} else if (direction == LEFT || direction == RIGHT) {
		SlidePuzzle(w, direction,
			w->mlink.spacePosition + direction - BOTTOM);
		if (CheckSolved(w)) {
			setPuzzle(w, ACTION_SOLVED);
		}
	} else {
		if (!PositionTile(w, x, &r))
			return;
		RotatePuzzle(w, direction, r);
		if (CheckSolved(w)) {
			setPuzzle(w, ACTION_SOLVED);
		}
	}
}

static int
SelectTiles(MlinkWidget w, int x, int y, int *i, int *j)
{
	int temp = w->mlink.faces * w->mlink.offset.y + w->mlink.delta.y - 1;

	*i = (x - w->mlink.delta.x / 2 - w->mlink.puzzleOffset.x) /
		w->mlink.offset.x;
	*j = ((y - w->mlink.delta.y / 2 - w->mlink.puzzleOffset.y + temp) %
		temp) / w->mlink.offset.y;
	if (*i >= 0 && *j >= 0 &&
			*i < w->mlink.tiles && *j < w->mlink.faces)
		return (*i + w->mlink.tiles * *j -
			w->mlink.spacePosition % w->mlink.tileFaces);
	return -w->mlink.tileFaces;
}

static void
SelectSlideTiles(MlinkWidget w, int pos)
{
	int n;

	if (pos < 0) {
		if (w->mlink.delay > 0) {
			AnimateSlide(w, -pos, RIGHT, NORMAL, True);
#ifdef USE_SOUND
			if (w->mlink.sound) {
				playSound((char *) BUMPSOUND);
			}
#endif
		} else {
			for (n = 1; n <= (-pos) % w->mlink.tiles; n++) {
				DiscreteMoves(w, w->mlink.spacePosition - 1,
					RIGHT);
			}
		}
	} else if (pos > 0) {
		if (w->mlink.delay > 0) {
			AnimateSlide(w, pos, LEFT, NORMAL, True);
#ifdef USE_SOUND
			if (w->mlink.sound) {
				playSound((char *) BUMPSOUND);
			}
#endif
		} else {
			for (n = 1; n <= pos % w->mlink.tiles; n++) {
				DiscreteMoves(w, w->mlink.spacePosition + 1,
					LEFT);
			}
		}
	}
}

static void
ResetTiles(MlinkWidget w)
{
	int i;

	w->mlink.tileFaces = w->mlink.tiles * w->mlink.faces;
	if (w->mlink.tileOfPosition)
		free(w->mlink.tileOfPosition);
	if (!(w->mlink.tileOfPosition = (int *)
			malloc(sizeof (int) * w->mlink.tileFaces))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}

	if (startPosition)
		free(startPosition);
	if (!(startPosition = (int *)
			malloc(sizeof (int) * w->mlink.tileFaces))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}

	w->mlink.spacePosition = w->mlink.tileFaces - 1;
	w->mlink.tileOfPosition[w->mlink.tileFaces - 1] = 0;
	for (i = 1; i < w->mlink.tileFaces; i++)
		w->mlink.tileOfPosition[i - 1] = i;
	flushMoves(w, &undo, TRUE);
	flushMoves(w, &redo, FALSE);
	w->mlink.currentRef = -w->mlink.tileFaces;
	w->mlink.started = False;
}

static void
GetTiles(MlinkWidget w)
{
	FILE *fp;
	int c, i, tiles, faces, middle, orient, practice, moves;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "r")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "r")) == NULL) {
			stringCat(&buf1, "Can not read (get) ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read (get) ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
#endif
	}
	flushMoves(w, &undo, TRUE);
	flushMoves(w, &redo, FALSE);
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &tiles);
	if (tiles >= MIN_TILES) {
		for (i = w->mlink.tiles; i < tiles; i++) {
			setPuzzle(w, ACTION_INCX);
		}
		for (i = w->mlink.tiles; i > tiles; i--) {
			setPuzzle(w, ACTION_DECX);
		}
	} else {
		stringCat(&buf1, name, " corrupted: tiles ");
		intCat(&buf2, buf1, tiles);
		free(buf1);
		stringCat(&buf1, buf2, " should be between ");
		free(buf2);
		intCat(&buf2, buf1, MIN_TILES);
		free(buf1);
		stringCat(&buf1, buf2, " and MAXINT");
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &faces);
	if (faces >= MIN_FACES && faces <= MAX_FACES) {
		for (i = w->mlink.faces; i < faces; i++) {
			setPuzzle(w, ACTION_INCY);
		}
		for (i = w->mlink.faces; i > faces; i--) {
			setPuzzle(w, ACTION_DECY);
		}
	} else {
		stringCat(&buf1, name, " corrupted: faces ");
		intCat(&buf2, buf1, faces);
		free(buf1);
		stringCat(&buf1, buf2, " should be between ");
		free(buf2);
		intCat(&buf2, buf1, MIN_FACES);
		free(buf1);
		stringCat(&buf1, buf2, " and ");
		free(buf2);
		intCat(&buf2, buf1, MAX_FACES);
		free(buf1);
		DISPLAY_WARNING(buf2);
		free(buf2);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &middle);
	if (w->mlink.middle != (Boolean) middle) {
		setPuzzle(w, ACTION_MIDDLE);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &orient);
	if (w->mlink.orient != (Boolean) orient) {
		setPuzzle(w, ACTION_ORIENTIZE);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &practice);
	if (w->mlink.practice != (Boolean) practice) {
		setPuzzle(w, ACTION_PRACTICE);
	}
#ifdef WINVER
	ResetTiles(w);
#endif
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &moves);
	scanStartPosition(fp, w);
	setPuzzle(w, ACTION_RESTORE);
	setStartPosition(w);
	scanMoves(fp, w, moves);
	(void) fclose(fp);
	(void) printf("%s: tiles %d, faces %d, middle %s, orient %s, ",
		name, tiles, faces, BOOL_STRING(middle), BOOL_STRING(orient));
	(void) printf("practice %s, moves %d.\n",
		BOOL_STRING(practice), moves);
	free(lname);
	free(fname);
	w->mlink.cheat = True; /* Assume the worst. */
}

static void
WriteTiles(MlinkWidget w)
{
	FILE *fp;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "w")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "w")) == NULL) {
			stringCat(&buf1, "Can not write to ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not write to ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
#endif
	}
	(void) fprintf(fp, "tiles%c %d\n", SYMBOL, w->mlink.tiles);
	(void) fprintf(fp, "faces%c %d\n", SYMBOL, w->mlink.faces);
	(void) fprintf(fp, "middle%c %d\n", SYMBOL,
		(w->mlink.middle) ? 1 : 0);
	(void) fprintf(fp, "orient%c %d\n", SYMBOL,
		(w->mlink.orient) ? 1 : 0);
	(void) fprintf(fp, "practice%c %d\n", SYMBOL,
		(w->mlink.practice) ? 1 : 0);
	(void) fprintf(fp, "moves%c %d\n", SYMBOL,
		numMoves(&undo));
	printStartPosition(fp, w);
	printMoves(fp, &undo);
	(void) fclose(fp);
	(void) printf("Saved to %s.\n", name);
	free(lname);
	free(fname);
}

static void
UndoTiles(MlinkWidget w)
{
	if (madeMoves(&undo) &&
			w->mlink.currentRef == -w->mlink.tileFaces) {
		int direction, tile, shift;

		getMove(&undo, &direction, &tile, &shift);
		setMove(&redo, direction, tile, shift);
		direction = (direction + COORD / 2) % COORD;
		if (shift != 0 && (direction == TOP || direction == BOTTOM)) {
			MoveShift(w, direction, DOUBLE);
		} else {
			if (direction == LEFT)
				tile = w->mlink.spacePosition + 1;
			else if (direction == RIGHT)
				tile = w->mlink.spacePosition - 1;
			if (MoveTilesDir(w, direction, tile, False, DOUBLE)) {
				setPuzzle(w, ACTION_UNDO);
				if (CheckSolved(w)) {
					setPuzzle(w, ACTION_SOLVED);
				}
			} else {
				char *buf1, *buf2;

				intCat(&buf1, "Move ", direction);
				stringCat(&buf2, buf1, " can not be made");
				free(buf1);
				DISPLAY_WARNING(buf2);
				free(buf2);
			}
		}
	}
}

static void
RedoTiles(MlinkWidget w)
{
	if (madeMoves(&redo) &&
			w->mlink.currentRef == -w->mlink.tileFaces) {
		int direction, tile, shift;

		getMove(&redo, &direction, &tile, &shift);
		setMove(&undo, direction, tile, shift);
		if (shift != 0 && (direction == TOP || direction == BOTTOM)) {
			MoveShift(w, direction, DOUBLE);
		} else {
			if (direction == LEFT)
				tile = w->mlink.spacePosition + 1;
			else if (direction == RIGHT)
				tile = w->mlink.spacePosition - 1;
			if (MoveTilesDir(w, direction, tile, False, DOUBLE)) {
				setPuzzle(w, ACTION_REDO);
				if (CheckSolved(w)) {
					setPuzzle(w, ACTION_SOLVED);
				}
			} else {
				char *buf1, *buf2;

				intCat(&buf1, "Move ", direction);
				stringCat(&buf2, buf1, " can not be made");
				free(buf1);
				DISPLAY_WARNING(buf2);
				free(buf2);
			}
		}
	}
}

static void
ClearTiles(MlinkWidget w)
{
	if (w->mlink.currentRef != -w->mlink.tileFaces)
		return;
	ResetTiles(w);
	DrawAllTiles(w);
	setPuzzle(w, ACTION_RESET);
}

static void
PracticeTiles(MlinkWidget w)
{
	setPuzzle(w, ACTION_PRACTICE);
}

static void
RandomizeTiles(MlinkWidget w)
{
	if (w->mlink.currentRef != -w->mlink.tileFaces)
		return;
	if (w->mlink.practice)
		PracticeTiles(w);
	setPuzzle(w, ACTION_RESET);
	w->mlink.cheat = False;
	/* First interchange tiles an even number of times */
	if (w->mlink.tiles > 1 && w->mlink.faces > 1 && w->mlink.tileFaces > 4) {
		int pos, count = 0;

		for (pos = 0; pos < w->mlink.tileFaces; pos++) {
			int randomPos = pos;

			while (randomPos == pos)
				randomPos = NRAND(w->mlink.tileFaces);
			count += ExchangeTiles(w, pos, randomPos);
		}
		if (count % 2 &&
				!ExchangeTiles(w, 0, 1) &&
				!ExchangeTiles(w, w->mlink.tileFaces - 2,
				w->mlink.tileFaces - 1)) {
			DISPLAY_WARNING("RandomizeTiles: should not get here");
		}
		DrawAllTiles(w);
		FLUSH(w);
	}
	/* randomly position space */
	/* Idea for this came from "puzzle" by Don Bennett, HP Labs */
	{
		int n, s, e, c;

		/* order matters */
		s = Row(w, w->mlink.spacePosition);
		c = Column(w, w->mlink.spacePosition);
		e = NRAND(w->mlink.faces);
		for (n = 0; n < e - s; n++)
			(void) MoveTilesDir(w, BOTTOM, c, False, INSTANT);
		for (n = 0; n < s - e; n++)
			(void) MoveTilesDir(w, TOP, c, False, INSTANT);
		s = Column(w, w->mlink.spacePosition);
		e = NRAND(w->mlink.tiles);
		for (n = 0; n < e - s; n++)
			(void) MoveTilesDir(w, LEFT,
				w->mlink.spacePosition + 1, False, INSTANT);
		for (n = 0; n < s - e; n++)
			(void) MoveTilesDir(w, RIGHT,
				w->mlink.spacePosition - 1, False, INSTANT);
		flushMoves(w, &undo, TRUE);
		flushMoves(w, &redo, FALSE);
		setPuzzle(w, ACTION_RANDOMIZE);
	}
#if 0
	/* Now move the space around randomly */
	if (w->mlink.tiles > 1 || w->mlink.faces > 1) {
		int big = w->mlink.tileFaces + NRAND(2);
		int lastDirection = -1;
		int randomDirection;

		setPuzzle(w, ACTION_RESET);

#ifdef DEBUG
		big = 3;
#endif
		if (big > 100)
			big = 100;
		while (big--) {
			randomDirection = NRAND(COORD);

#ifdef DEBUG
			sleep(1);
#endif
			if ((randomDirection + COORD / 2) % COORD != lastDirection ||
					w->mlink.tiles == 1 || w->mlink.faces == 1) {
				if (MovePuzzle(w, randomDirection,
						NRAND(w->mlink.tiles),
						FALSE, False, INSTANT))
					lastDirection = randomDirection;
				else
					big++;
			}
		}
		flushMoves(w, &undo, TRUE);
		flushMoves(w, &redo, FALSE);
		setPuzzle(w, ACTION_RANDOMIZE);
	}
#endif
	if (CheckSolved(w)) {
		setPuzzle(w, ACTION_SOLVED);
	}
}

static void
SolveTiles(MlinkWidget w)
{
	if (CheckSolved(w) || w->mlink.currentRef != -w->mlink.tileFaces)
		return;
	if (w->mlink.tiles == 4 && w->mlink.faces == 4) {
		SolveSomeTiles(w);
	} else {
		setPuzzle(w, ACTION_SOLVE_MESSAGE);
	}
}

static void
OrientizeTiles(MlinkWidget w)
{
	setPuzzle(w, ACTION_ORIENTIZE);
}

static void
MiddleTiles(MlinkWidget w)
{
	setPuzzle(w, ACTION_MIDDLE);
}

static void
SpeedTiles(MlinkWidget w)
{
	w->mlink.delay -= 5;
	if (w->mlink.delay < 0)
		w->mlink.delay = 0;
#ifdef HAVE_MOTIF
	setPuzzle(w, ACTION_SPEED);
#endif
}

static void
SlowTiles(MlinkWidget w)
{
	w->mlink.delay += 5;
#ifdef HAVE_MOTIF
	setPuzzle(w, ACTION_SPEED);
#endif
}

static void
SoundTiles(MlinkWidget w)
{
	w->mlink.sound = !w->mlink.sound;
}

#define BRIGHT_FACTOR 0.8
#define DARK_FACTOR 0.75

#ifdef WINVER
#define MAX_INTENSITY 0xFF
static int
brighter(const int light)
{
	int i = (int) ((1 - BRIGHT_FACTOR) * MAX_INTENSITY);
	int temp = light;

	if (temp < i)
		temp = i;
	return MIN(temp / BRIGHT_FACTOR, MAX_INTENSITY);
}

static int
darker(const int light)
{
	return (int) (light * DARK_FACTOR);
}

static void
SetValuesPuzzle(MlinkWidget w)
{
	struct tagColor {
		int red, green, blue;
	} color;
	char szBuf[80], buf[20], charbuf[2];
	int face;

	w->mlink.tiles = GetPrivateProfileInt(SECTION, "tiles",
		DEFAULT_TILES, INIFILE);
	w->mlink.faces = GetPrivateProfileInt(SECTION, "faces",
		DEFAULT_FACES, INIFILE);
	w->mlink.orient = (BOOL) GetPrivateProfileInt(SECTION, "orient",
		DEFAULT_ORIENT, INIFILE);
	w->mlink.middle = (BOOL) GetPrivateProfileInt(SECTION, "middle",
		DEFAULT_MIDDLE, INIFILE);
	w->mlink.practice = (BOOL) GetPrivateProfileInt(SECTION, "practice",
		DEFAULT_PRACTICE, INIFILE);
	w->mlink.base = GetPrivateProfileInt(SECTION, "base",
		DEFAULT_BASE, INIFILE);
	w->mlink.mono = (BOOL) GetPrivateProfileInt(SECTION, "mono",
		DEFAULT_MONO, INIFILE);
	w->mlink.reverse = (BOOL) GetPrivateProfileInt(SECTION, "reverseVideo",
		DEFAULT_REVERSE, INIFILE);
	/* cyan */
	(void) GetPrivateProfileString(SECTION, "frameColor", "0 255 255",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->mlink.frameGC = RGB(color.red, color.green, color.blue);
	/* black */
	(void) GetPrivateProfileString(SECTION, "tileColor", "0 0 0",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->mlink.tileGC[1] = RGB(color.red, color.green, color.blue);
	/* gray25 */
	(void) GetPrivateProfileString(SECTION, "tileBorder", "64 64 64",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->mlink.tileGC[0] = RGB(brighter(color.red),
		brighter(color.green), brighter(color.blue));
	w->mlink.tileGC[2] = RGB(darker(color.red),
		darker(color.green), darker(color.blue));
	w->mlink.borderGC = RGB(color.red, color.green, color.blue);
	/* #AEB2C3 */
	(void) GetPrivateProfileString(SECTION, "background", "174 178 195",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->mlink.inverseGC = RGB(color.red, color.green, color.blue);
	for (face = 0; face < MAX_FACES; face++) {
		(void) sprintf(buf, "faceColor%d", face);
		(void) GetPrivateProfileString(SECTION,
			buf, faceColorString[face],
			szBuf, sizeof (szBuf), INIFILE);
		(void) sscanf(szBuf, "%d %d %d",
			&(color.red), &(color.green), &(color.blue));
		w->mlink.faceGC[face] =
			RGB(color.red, color.green, color.blue);
		(void) sprintf(buf, "faceChar%d", face);
		charbuf[0] = faceColorChar[face];
		charbuf[1] = '\0';
		(void) GetPrivateProfileString(SECTION, buf, charbuf,
			szBuf, sizeof (szBuf), INIFILE);
		w->mlink.faceChar[face] = szBuf[0];
	}
	(void) GetPrivateProfileString(SECTION, "picture", PICTURE,
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->mlink.picture, szBuf);
	w->mlink.picture[80] = 0;
	w->mlink.delay = GetPrivateProfileInt(SECTION, "delay",
		DEFAULT_DELAY, INIFILE);
	w->mlink.sound = (BOOL)
		GetPrivateProfileInt(SECTION, "sound", 0, INIFILE);
	(void) GetPrivateProfileString(SECTION, "bumpSound", BUMPSOUND,
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->mlink.bumpSound, szBuf);
	w->mlink.bumpSound[80] = 0;
	(void) GetPrivateProfileString(SECTION, "moveSound", MOVESOUND,
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->mlink.moveSound, szBuf);
	w->mlink.moveSound[80] = 0;
	(void) GetPrivateProfileString(SECTION, "userName", "Guest",
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->mlink.userName, szBuf);
	w->mlink.userName[80] = 0;
	(void) GetPrivateProfileString(SECTION, "scoreFile", "",
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->mlink.scoreFile, szBuf);
	w->mlink.scoreFile[80] = 0;
}

void
DestroyPuzzle(MlinkWidget w, HBRUSH brush)
{
	if (w->core.memDC) {
		if (w->mlink.bufferTiles != NULL) {
			DeleteObject(w->mlink.bufferTiles);
		}
		DeleteDC(w->core.memDC);
		w->core.memDC = NULL;
	}
	deleteMoves(&undo);
	deleteMoves(&redo);
	(void) DeleteObject(brush);
	PostQuitMessage(0);
}

#else
#define MAX_INTENSITY 0xFFFF

static Pixel
brighter(MlinkWidget w, Pixel pixel)
{
	XColor color;
	Colormap colormap = (w->mlink.colormap == None) ?
		w->mlink.oldColormap : w->mlink.colormap;
	int i = (int) ((1 - BRIGHT_FACTOR) * MAX_INTENSITY);

	color.pixel = pixel;
	VOID XQueryColor(XtDisplay(w), colormap, &color);
	if (color.red < i)
		color.red = i;
	if (color.green < i)
		color.green = i;
	if (color.blue < i)
		color.blue = i;
	color.red = (unsigned short) MIN(color.red / BRIGHT_FACTOR, MAX_INTENSITY);
	color.green = (unsigned short) MIN(color.green / BRIGHT_FACTOR, MAX_INTENSITY);
	color.blue = (unsigned short) MIN(color.blue / BRIGHT_FACTOR, MAX_INTENSITY);
	if (XAllocColor(XtDisplay(w), colormap, &color))
		return color.pixel;
	return pixel;
}

static Pixel
darker(MlinkWidget w, Pixel pixel)
{
	XColor color;
	Colormap colormap = (w->mlink.colormap == None) ?
		w->mlink.oldColormap : w->mlink.colormap;

	color.pixel = pixel;
	VOID XQueryColor(XtDisplay(w), colormap, &color);
	color.red = (unsigned short) (color.red * DARK_FACTOR);
	color.green = (unsigned short) (color.green * DARK_FACTOR);
	color.blue = (unsigned short) (color.blue * DARK_FACTOR);
	if (XAllocColor(XtDisplay(w), colormap, &color))
		return color.pixel;
	return pixel;
}

static void
GetColor(MlinkWidget w, int face)
{
	XGCValues values;
	XtGCMask valueMask;
	XColor colorCell, rgb;
	Colormap colormap = (w->mlink.colormap == None) ?
		w->mlink.oldColormap : w->mlink.colormap;

	valueMask = GCForeground | GCBackground;
	if (w->mlink.reverse) {
		values.background = w->mlink.foreground;
	} else {
		values.background = w->mlink.background;
	}
	if (!w->mlink.mono) {
		if (XAllocNamedColor(XtDisplay(w), colormap,
				w->mlink.faceName[face], &colorCell, &rgb)) {
			values.foreground = w->mlink.faceColor[face] = colorCell.pixel;
			if (w->mlink.faceGC[face])
				XtReleaseGC((Widget) w, w->mlink.faceGC[face]);
			w->mlink.faceGC[face] = XtGetGC((Widget) w, valueMask, &values);
			if (w->mlink.fontInfo)
				XSetFont(XtDisplay(w), w->mlink.faceGC[face],
					w->mlink.fontInfo->fid);
			return;
		} else {
			char *buf1, *buf2;

			stringCat(&buf1, "Color name \"",
				w->mlink.faceName[face]);
			stringCat(&buf2, buf1, "\" is not defined for face ");
			free(buf1);
			intCat(&buf1, buf2, face);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
	}
	if (w->mlink.reverse) {
		values.background = w->mlink.background;
		values.foreground = w->mlink.foreground;
	} else {
		values.background = w->mlink.foreground;
		values.foreground = w->mlink.background;
	}
	if (w->mlink.faceGC[face])
		XtReleaseGC((Widget) w, w->mlink.faceGC[face]);
	w->mlink.faceGC[face] = XtGetGC((Widget) w, valueMask, &values);
	if (w->mlink.fontInfo)
		XSetFont(XtDisplay(w), w->mlink.faceGC[face],
			w->mlink.fontInfo->fid);
}

static void
SetAllColors(MlinkWidget w)
{
	XGCValues values;
	XtGCMask valueMask;
	int face;

	valueMask = GCForeground | GCBackground;

	if (w->mlink.reverse) {
		values.foreground = w->mlink.foreground;
		values.background = w->mlink.background;
	} else {
		values.foreground = w->mlink.background;
		values.background = w->mlink.foreground;
	}
	if (w->mlink.inverseGC)
		XtReleaseGC((Widget) w, w->mlink.inverseGC);
	w->mlink.inverseGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->mlink.mono) {
		if (w->mlink.reverse) {
			values.foreground = w->mlink.background;
			values.background = w->mlink.foreground;
		} else {
			values.foreground = w->mlink.foreground;
			values.background = w->mlink.background;
		}
	} else {
		values.foreground = w->mlink.frameColor;
		values.background = w->mlink.borderColor;
	}
	if (w->mlink.frameGC)
		XtReleaseGC((Widget) w, w->mlink.frameGC);
	w->mlink.frameGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->mlink.mono) {
		if (w->mlink.reverse) {
			values.background = w->mlink.foreground;
			values.foreground = w->mlink.background;
		} else {
			values.foreground = w->mlink.foreground;
			values.background = w->mlink.background;
		}
	} else {
		values.foreground = w->mlink.tileColor;
		values.background = w->mlink.borderColor;
	}
	if (w->mlink.tileGC)
		XtReleaseGC((Widget) w, w->mlink.tileGC[1]);
	w->mlink.tileGC[1] = XtGetGC((Widget) w, valueMask, &values);
	if (!w->mlink.mono) {
		values.foreground = darker(w, w->mlink.tileColor);
	}
	if (w->mlink.tileGC[0])
		XtReleaseGC((Widget) w, w->mlink.tileGC[0]);
	w->mlink.tileGC[0] = XtGetGC((Widget) w, valueMask, &values);
	if (!w->mlink.mono) {
		values.foreground = brighter(w, w->mlink.tileColor);
	}
	if (w->mlink.tileGC[2])
		XtReleaseGC((Widget) w, w->mlink.tileGC[2]);
	w->mlink.tileGC[2] = XtGetGC((Widget) w, valueMask, &values);
	if (w->mlink.mono) {
		if (w->mlink.reverse) {
			values.foreground = w->mlink.foreground;
			values.background = w->mlink.background;
		} else {
			values.background = w->mlink.foreground;
			values.foreground = w->mlink.background;
		}
	} else {
		values.foreground = w->mlink.borderColor;
		values.background = w->mlink.tileColor;
	}
	if (w->mlink.borderGC)
		XtReleaseGC((Widget) w, w->mlink.borderGC);
	w->mlink.borderGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->mlink.fontInfo)
		XSetFont(XtDisplay(w), w->mlink.borderGC,
			w->mlink.fontInfo->fid);
	for (face = 0; face < MAX_FACES; face++)
		GetColor(w, face);
}

static Boolean
SetValuesPuzzle(Widget current, Widget request, Widget renew)
{
	MlinkWidget c = (MlinkWidget) current, w = (MlinkWidget) renew;
	Boolean redraw = False, setColors = False;
	Boolean redrawTiles = False;
	int face;

	CheckTiles(w);
	for (face = 0; face < MAX_FACES; face++) {
		if (strcmp(w->mlink.faceName[face], c->mlink.faceName[face])) {
			setColors = True;
			break;
		}
	}
	if (w->mlink.font != c->mlink.font ||
			w->mlink.borderColor != c->mlink.borderColor ||
			w->mlink.reverse != c->mlink.reverse ||
			w->mlink.mono != c->mlink.mono ||
			setColors) {
		loadFont(w);
		SetAllColors(w);
		redrawTiles = True;
	} else if (w->mlink.background != c->mlink.background ||
			w->mlink.foreground != c->mlink.foreground ||
			w->mlink.tileColor != c->mlink.tileColor) {
		SetAllColors(w);
		redrawTiles = True;
	}
	if (w->mlink.tiles != c->mlink.tiles ||
			w->mlink.faces != c->mlink.faces ||
			w->mlink.orient != c->mlink.orient ||
			w->mlink.middle != c->mlink.middle ||
			w->mlink.practice != c->mlink.practice ||
			w->mlink.base != c->mlink.base) {
		SizePuzzle(w);
		redraw = True;
	} else if (w->mlink.offset.x != c->mlink.offset.x ||
			w->mlink.offset.y != c->mlink.offset.y) {
		ResizePuzzle(w);
		redraw = True;
	}
	if (w->mlink.delay != c->mlink.delay) {
		w->mlink.numSlices = ((w->mlink.delay < MAX_SLICES) ?
			w->mlink.delay + 1 : MAX_SLICES);
	}
	if (w->mlink.menu != ACTION_IGNORE) {
		int menu = w->mlink.menu;

		w->mlink.menu = ACTION_IGNORE;
		switch (menu) {
		case ACTION_GET:
			GetTiles(w);
			break;
		case ACTION_WRITE:
			WriteTiles(w);
			break;
		case ACTION_UNDO:
			UndoTiles(w);
			break;
		case ACTION_REDO:
			RedoTiles(w);
			break;
		case ACTION_CLEAR:
			ClearTiles(w);
			break;
		case ACTION_PRACTICE:
			PracticeTiles(w);
			break;
		case ACTION_RANDOMIZE:
			RandomizeTiles(w);
			break;
		case ACTION_SOLVE:
			SolveTiles(w);
			break;
		case ACTION_ORIENTIZE:
			OrientizeTiles(w);
			break;
		case ACTION_MIDDLE:
			MiddleTiles(w);
			break;
		case ACTION_SPEED:
			SpeedTiles(w);
			break;
		case ACTION_SLOW:
			SlowTiles(w);
			break;
		case ACTION_SOUND:
			SoundTiles(w);
			break;
		default:
			break;
		}
	}
	if (redrawTiles && !redraw && XtIsRealized(renew) && renew->core.visible) {
		EraseFrame(c, 0);
		if (w->mlink.focus)
			DrawFrame(w, 0, True);
		DrawAllTiles(w);
	}
	return (redraw);
}

static void
DestroyPuzzle(Widget old)
{
	MlinkWidget w = (MlinkWidget) old;
	Display *display = XtDisplay(w);
	int i;

	for (i = 0; i < MAX_FACES; i++)
		XtReleaseGC(old, w->mlink.faceGC[i]);
	XtReleaseGC(old, w->mlink.borderGC);
	for (i = 0; i < FG_SHADES; i++)
		XtReleaseGC(old, w->mlink.tileGC[i]);
	XtReleaseGC(old, w->mlink.frameGC);
	XtReleaseGC(old, w->mlink.inverseGC);
	XtRemoveCallbacks(old, XtNselectCallback, w->mlink.select);
	if (w->mlink.colormap != None) {
		XInstallColormap(display, w->mlink.oldColormap);
		XFreeColormap(display, w->mlink.colormap);
	}
	for (i = 0; i < 2; i++)
		if (w->mlink.bufferTiles[i] != None)
			XFreePixmap(display, w->mlink.bufferTiles[i]);
	if (w->mlink.fontInfo) {
		XUnloadFont(display, w->mlink.fontInfo->fid);
		XFreeFont(display, w->mlink.fontInfo);
	}
	deleteMoves(&undo);
	deleteMoves(&redo);
}

static void
QuitPuzzle(MlinkWidget w, XEvent *event, char **args, int nArgs)
{
	XtCloseDisplay(XtDisplay(w));
	exit(0);
}
#endif

static void
ResizeTiles(MlinkWidget w)
{
	int sel;
#ifdef WINVER
	if (w->core.memDC == NULL) {
		w->core.memDC = CreateCompatibleDC(w->core.hDC);
		if (w->core.memDC == NULL) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
	}
	for (sel = 0; sel < 2; sel++) {
		if (w->mlink.bufferTiles[sel] != NULL) {
			DeleteObject(w->mlink.bufferTiles[sel]);
				w->mlink.bufferTiles[sel] = NULL;
		}
		if (!(w->mlink.picture && *(w->mlink.picture))) {
			if ((w->mlink.bufferTiles[sel] =
					CreateCompatibleBitmap(w->core.hDC,
					w->core.width,
					w->core.height)) == NULL) {
				DISPLAY_ERROR("Not enough memory, exiting.");
			}
		}
	}
#else
	Display *display = XtDisplay(w);
	Window window = XtWindow(w);
	XWindowAttributes xgwa;

	(void) XGetWindowAttributes(display, window, &xgwa);
	if (w->mlink.oldColormap == None) {
		w->mlink.mono = (xgwa.depth < 2 || w->mlink.mono);
		w->mlink.oldColormap = xgwa.colormap;
	}
	for (sel = 0; sel < 2; sel++) {
		if (w->mlink.bufferTiles[sel] != None) {
			XFreePixmap(display, w->mlink.bufferTiles[sel]);
			w->mlink.bufferTiles[sel] = None;
		}
		if ((w->mlink.bufferTiles[sel] = XCreatePixmap(display,
			window, w->core.width, w->core.height,
			xgwa.depth)) == None) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
	}
#endif
	if (w->mlink.picture && *(w->mlink.picture)) {
#ifdef WINVER
		for (sel = 0; sel < 2; sel++) {
			w->mlink.bufferTiles[sel] =
				LoadBitmap(w->core.hInstance,
				w->mlink.picture);
		}
#else
		if (w->mlink.image != NULL) {
			destroyImage(&(w->mlink.image),
				&(w->mlink.graphicsFormat));
		}
		if (!getImage(display, window,
				xgwa.visual, w->mlink.oldColormap, xgwa.depth,
				&(w->mlink.image), w->mlink.picture,
				w->mlink.install, &(w->mlink.graphicsFormat),
				&(w->mlink.colormap))) {
			w->mlink.picture = NULL;
		} else if (w->mlink.image == NULL) {
			w->mlink.picture = NULL;
		}
#endif
	}
#ifndef WINVER
	if (!(w->mlink.picture && *(w->mlink.picture)) &&
			!fixedColors(xgwa.visual, xgwa.depth, w->mlink.install) &&
			w->mlink.colormap == None) {
		w->mlink.colormap = XCreateColormap(display, window,
			xgwa.visual, AllocNone);
	}
	SetAllColors(w);
	for (sel = 0; sel < 2; sel++) {
		FILLRECTANGLE(w, w->mlink.bufferTiles[sel],
			w->mlink.inverseGC,
			0, 0, w->core.width, w->core.height);
		if ((w->mlink.picture && *(w->mlink.picture))) {

			(void) XPutImage(display, w->mlink.bufferTiles[sel],
				w->mlink.inverseGC, w->mlink.image, 0, 0,
				0, 0,
				MIN(w->mlink.image->width, w->core.width),
				MIN(w->mlink.image->height, w->core.height));
		}
	}
#endif
	if (!(w->mlink.picture && *(w->mlink.picture))) {
		DrawAllBufferedTiles(w);
	} else if (!w->mlink.orient) {
		setPuzzle(w, ACTION_ORIENTIZE);
	}
}

void
ResizePuzzle(MlinkWidget w)
{
#ifdef WINVER
	RECT rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif

	w->mlink.delta.x = 1;
	w->mlink.delta.y = 1;
	w->mlink.offset.x = MAX(((int) w->core.width - 2) /
		w->mlink.tiles, 0);
	w->mlink.offset.y = MAX((int) w->core.height /
		w->mlink.faces - 2, 0);
	w->mlink.faceSize.x = w->mlink.offset.x * w->mlink.tiles +
		w->mlink.delta.x + 2;
	w->mlink.faceSize.y = w->mlink.offset.y + w->mlink.delta.y + 2;
	w->mlink.puzzleSize.x = w->mlink.faceSize.x;
	w->mlink.puzzleSize.y = (w->mlink.faceSize.y - 2) *
		w->mlink.faces + 2;
	w->mlink.puzzleOffset.x = ((int) w->core.width -
		w->mlink.puzzleSize.x + 2) / 2;
	w->mlink.puzzleOffset.y = ((int) w->core.height -
		w->mlink.puzzleSize.y + 2) / 2;
	w->mlink.tileSize.x = MAX(w->mlink.offset.x - w->mlink.delta.x, 0);
	w->mlink.tileSize.y = MAX(w->mlink.offset.y - w->mlink.delta.y, 0);
}

#ifndef WINVER
static
#endif
void
SizePuzzle(MlinkWidget w)
{
	ResetTiles(w);
	ResizePuzzle(w);
}

#ifndef WINVER
static
#endif
void
InitializePuzzle(
#ifdef WINVER
MlinkWidget w, HBRUSH brush
#else
Widget request, Widget renew
#endif
)
{
#ifdef WINVER
	SetValuesPuzzle(w);
	brush = CreateSolidBrush(w->mlink.inverseGC);
	SETBACK(w->core.hWnd, brush);
	(void) SRAND(time(NULL));
	w->mlink.bufferTiles[0] = NULL;
	w->mlink.bufferTiles[1] = NULL;
#else
	MlinkWidget w = (MlinkWidget) renew;
	int i;

	(void) SRAND(getpid());
	w->mlink.bufferTiles[0] = None;
	w->mlink.bufferTiles[1] = None;
	w->mlink.colormap = None;
	w->mlink.oldColormap = None;
	w->mlink.fontInfo = NULL;
	for (i = 0; i < MAX_FACES; i++)
		w->mlink.faceGC[i] = NULL;
	for (i = 0; i < FG_SHADES; i++)
		w->mlink.tileGC[i] = NULL;
	w->mlink.borderGC = NULL;
	w->mlink.frameGC = NULL;
	w->mlink.inverseGC = NULL;
	w->mlink.image = NULL;
#endif
	w->mlink.focus = False;
	loadFont(w);
	w->mlink.tileOfPosition = NULL;
	CheckTiles(w);
	newMoves(&undo);
	newMoves(&redo);
	w->mlink.numSlices = ((w->mlink.delay < MAX_SLICES) ?
		w->mlink.delay + 1 : MAX_SLICES);
	w->mlink.cheat = False;
	SizePuzzle(w);
}

#ifndef WINVER
static
#endif
void
ExposePuzzle(
#ifdef WINVER
MlinkWidget w
#else
Widget renew, XEvent *event, Region region
#endif
)
{
#ifndef WINVER
	MlinkWidget w = (MlinkWidget) renew;

	if (!w->core.visible)
		return;
#endif
	ResizeTiles(w);
	EraseFrame(w, 0);
	DrawFrame(w, 0, w->mlink.focus);
	DrawAllTiles(w);
}

#ifndef WINVER
static
#endif
void
HidePuzzle(MlinkWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	setPuzzle(w, ACTION_HIDE);
}

#ifndef WINVER
static
#endif
void
SelectPuzzle(MlinkWidget w
#ifdef WINVER
, const int x, const int y, const int shift
#else
, XEvent *event, char **args, int nArgs
#endif
)
{
	int i, j, pos;
#ifndef WINVER
	int x = event->xbutton.x, y = event->xbutton.y;
	int shift = (int) (event->xkey.state & (ShiftMask | LockMask));
#endif

	w->mlink.motion = False;
	w->mlink.motionDiff = 0;
	w->mlink.shiftDiff = 0;
	pos = SelectTiles(w, x, y, &i, &j);
	if (-w->mlink.tileFaces != pos) {
		w->mlink.currentTile = i;
		w->mlink.currentFace = j;
		w->mlink.currentRef = pos;
		if (shift != 0 || w->mlink.practice || !CheckSolved(w)) {
			pos = w->mlink.currentTile +
				w->mlink.currentFace * w->mlink.tiles;
			DrawTile(w, pos, (w->mlink.tileOfPosition[i +
				j * w->mlink.tiles] <= 0),
				False, TRUE, 0, 0);
		}
	} else
		w->mlink.currentRef = -w->mlink.tileFaces;
}

#ifndef WINVER
static
#endif
void
MotionPuzzle(MlinkWidget w
#ifdef WINVER
, const int x, const int y, const int shift
#else
, XEvent *event, char **args, int nArgs
#endif
)
{
	int i, j, pos, diff;
#ifndef WINVER
	int x = event->xbutton.x, y = event->xbutton.y;
	int shift = (int) (event->xkey.state & (ShiftMask | LockMask));
#endif

	if (w->mlink.currentRef == -w->mlink.tileFaces)
		return;
	if (shift == 0 && !w->mlink.practice && CheckSolved(w)) {
		MoveNoTiles(w);
		w->mlink.currentRef = -w->mlink.tileFaces;
		return;
	}
	pos = SelectTiles(w, x, y, &i, &j);
	if (w->mlink.currentTile != i) {
		int k = w->mlink.currentTile +
			w->mlink.currentFace * w->mlink.tiles;

		DrawTile(w, k, True, True, TRUE, 0, 0);
		DrawTile(w, k, (w->mlink.tileOfPosition[k] <= 0),
			(w->mlink.tileOfPosition[k] <= 0), FALSE, 0, 0);
		DiffPosition(w, w->mlink.motionDiff, j, FALSE);
		DiffPosition(w, w->mlink.shiftDiff, j, TRUE);
		if (shift == 0 && CheckSolved(w)) {
			setPuzzle(w, ACTION_SOLVED);
		}
		w->mlink.currentTile = i;
		w->mlink.currentFace = j;
		w->mlink.currentRef = pos;
		w->mlink.motionDiff = 0;
		w->mlink.shiftDiff = 0;
	}
	if (-w->mlink.tileFaces != pos) {
		if (j != w->mlink.currentFace) {
			diff = (w->mlink.currentFace - j + w->mlink.faces) %
				w->mlink.faces;
			DiffMove(w, diff, j, shift);
			if (shift == 0 && CheckSolved(w)) {
				setPuzzle(w, ACTION_SOLVED);
			}
			w->mlink.motion = True;
			if (shift == 0)
				w->mlink.motionDiff = (diff +
					w->mlink.motionDiff) % w->mlink.faces;
			else
				w->mlink.shiftDiff = (diff +
					w->mlink.shiftDiff) % w->mlink.faces;
		}
		w->mlink.currentTile = i;
		w->mlink.currentFace = j;
		w->mlink.currentRef = pos;
		if (shift != 0 || w->mlink.practice || !CheckSolved(w)) {
			pos = w->mlink.currentTile +
				w->mlink.currentFace * w->mlink.tiles;
			DrawTile(w, pos, (w->mlink.tileOfPosition[i +
				j * w->mlink.tiles] <= 0),
				False, TRUE, 0, 0);
		}
	} else
		w->mlink.currentRef = -w->mlink.tileFaces;
}

#ifndef WINVER
static
#endif
void
ReleasePuzzle(MlinkWidget w
#ifdef WINVER
, const int x, const int y, const int shift
#else
, XEvent *event, char **args, int nArgs
#endif
)
{
	int i, j, pos, diff;
#ifndef WINVER
	int x = event->xbutton.x, y = event->xbutton.y;
	int shift = (int) (event->xkey.state & (ShiftMask | LockMask));
#endif

	if (w->mlink.currentRef == -w->mlink.tileFaces)
		return;
	pos = w->mlink.currentTile + w->mlink.currentFace * w->mlink.tiles;
	DrawTile(w, pos, True, True, TRUE, 0, 0);
	if (w->mlink.tileOfPosition[pos] > 0) {
		DrawTile(w, pos, False, False, FALSE, 0, 0);
	}
	if (shift == 0 && !w->mlink.practice && CheckSolved(w)) {
		MoveNoTiles(w);
		w->mlink.currentRef = -w->mlink.tileFaces;
		return;
	}
	pos = SelectTiles(w, x, y, &i, &j);
	if (-w->mlink.tileFaces != pos) {
		if (j == w->mlink.currentFace && !w->mlink.motion) {
			pos = w->mlink.currentRef;
			if (pos / w->mlink.tiles == 0 &&
					j == Row(w, w->mlink.spacePosition) &&
					pos != 0) {
				if (shift != 0 && CheckSolved(w))
					MoveNoTiles(w);
				else {
					SelectSlideTiles(w, pos);
					w->mlink.currentTile = w->mlink.tiles;
					if (CheckSolved(w)) {
						setPuzzle(w, ACTION_SOLVED);
					}
				}
			}
		} else {
			diff = (w->mlink.currentFace - j + w->mlink.faces) %
				w->mlink.faces;
			DiffMove(w, diff, j, shift);
			if (shift == 0)
				w->mlink.motionDiff = (diff +
					w->mlink.motionDiff) %
					w->mlink.faces;
			else
				w->mlink.shiftDiff = (diff +
					w->mlink.shiftDiff) % w->mlink.faces;
			DiffPosition(w, w->mlink.motionDiff, j, FALSE);
			DiffPosition(w, w->mlink.shiftDiff, j, TRUE);
			if (shift == 0 && CheckSolved(w)) {
				setPuzzle(w, ACTION_SOLVED);
			}
		}
	}
	w->mlink.currentRef = -w->mlink.tileFaces;
}

#ifndef WINVER
static void
PracticePuzzleMaybe(MlinkWidget w
, XEvent *event, char **args, int nArgs
)
{
	if (!w->mlink.started)
		PracticeTiles(w);
#ifdef HAVE_MOTIF
	else {
		setPuzzle(w, ACTION_PRACTICE_QUERY);
	}
#endif
}

static void
PracticePuzzle2(MlinkWidget w
, XEvent *event, char **args, int nArgs
)
{
#ifdef HAVE_MOTIF
	if (!w->mlink.started)
#endif
		PracticeTiles(w);
}
#endif

#ifndef WINVER
static void
RandomizePuzzleMaybe(MlinkWidget w
, XEvent *event, char **args, int nArgs
)
{
	if (!w->mlink.started)
		RandomizeTiles(w);
#ifdef HAVE_MOTIF
	else {
		setPuzzle(w, ACTION_RANDOMIZE_QUERY);
	}
#endif
}

static void
RandomizePuzzle2(MlinkWidget w
, XEvent *event, char **args, int nArgs
)
{
#ifdef HAVE_MOTIF
	if (!w->mlink.started)
#endif
		RandomizeTiles(w);
}
#endif

#ifndef WINVER
static
#endif
void
GetPuzzle(MlinkWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	GetTiles(w);
}

#ifndef WINVER
static
#endif
void
WritePuzzle(MlinkWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	WriteTiles(w);
}

#ifndef WINVER
static
#endif
void
UndoPuzzle(MlinkWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	UndoTiles(w);
}

#ifndef WINVER
static
#endif
void
RedoPuzzle(MlinkWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	RedoTiles(w);
}

#ifndef WINVER
static
#endif
void
ClearPuzzle(MlinkWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	ClearTiles(w);
}

#ifndef WINVER
static
#endif
void
RandomizePuzzle(MlinkWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	RandomizeTiles(w);
}

#ifndef WINVER
static
#endif
void
SolvePuzzle(MlinkWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	SolveTiles(w);
}

#ifndef WINVER
static
#endif
void
PracticePuzzle(MlinkWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	PracticeTiles(w);
}

#ifndef WINVER
static
#endif
void
OrientizePuzzle(MlinkWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	OrientizeTiles(w);
}

#ifndef WINVER
static
#endif
void
MiddlePuzzle(MlinkWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	MiddleTiles(w);
}

#ifndef WINVER
static
#endif
void
SpeedPuzzle(MlinkWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	SpeedTiles(w);
}

#ifndef WINVER
static
#endif
void
SlowPuzzle(MlinkWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	SlowTiles(w);
}

#ifndef WINVER
static
#endif
void
SoundPuzzle(MlinkWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	SoundTiles(w);
}

#ifndef WINVER
static
#endif
void
EnterPuzzle(MlinkWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	w->mlink.focus = True;
	DrawFrame(w, 0, w->mlink.focus);
#ifndef WINVER
	if (w->mlink.colormap != None)
		XInstallColormap(XtDisplay(w), w->mlink.colormap);
#endif
}

#ifndef WINVER
static
#endif
void
LeavePuzzle(MlinkWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	w->mlink.focus = False;
	DrawFrame(w, 0, w->mlink.focus);
#ifndef WINVER
	if (w->mlink.colormap != None)
		XInstallColormap(XtDisplay(w), w->mlink.oldColormap);
#endif

}

#ifndef WINVER
static void
MovePuzzleTop(MlinkWidget w , XEvent *event, char **args, int nArgs
)
{
	int x = event->xbutton.x;
	int shift = (int) (event->xkey.state & (ShiftMask | LockMask));
	int control = (int) (event->xkey.state & ControlMask);

	MovePuzzleInput(w, x, TOP, shift, control);
}

static void
MovePuzzleBottom(MlinkWidget w , XEvent *event, char **args, int nArgs
)
{
	int x = event->xbutton.x;
	int shift = (int) (event->xkey.state & (ShiftMask | LockMask));
	int control = (int) (event->xkey.state & ControlMask);

	MovePuzzleInput(w, x, BOTTOM, shift, control);
}

static void
MovePuzzleLeft(MlinkWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput(w, event->xbutton.x, LEFT,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}

static void
MovePuzzleRight(MlinkWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput(w, event->xbutton.x, RIGHT,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}
#endif
