/*-
# X-BASED DINOSAUR CUBE
#
#  Dino2d.c
#
###
#
#  Copyright (c) 1995 - 2006	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Dino2d */

#include "DinoP.h"
#include "Dino2dP.h"

#ifndef WINVER
static Boolean SetValuesDino2D(Widget current, Widget request, Widget renew);
static void ResizeDino2D(Dino2DWidget w);
static void InitializeDino2D(Widget request, Widget renew);
static void ExposeDino2D(Widget renew, XEvent * event, Region region);
static void MoveDino2DTl(Dino2DWidget w, XEvent * event, char **args, int nArgs);
static void MoveDino2DTop(Dino2DWidget w, XEvent * event, char **args, int nArgs);
static void MoveDino2DTr(Dino2DWidget w, XEvent * event, char **args, int nArgs);
static void MoveDino2DLeft(Dino2DWidget w, XEvent * event, char **args, int nArgs);
static void MoveDino2DRight(Dino2DWidget w, XEvent * event, char **args, int nArgs);
static void MoveDino2DBl(Dino2DWidget w, XEvent * event, char **args, int nArgs);
static void MoveDino2DBottom(Dino2DWidget w, XEvent * event, char **args, int nArgs);
static void MoveDino2DBr(Dino2DWidget w, XEvent * event, char **args, int nArgs);

static char defaultTranslationsDino2D[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <KeyPress>Home: MoveTl()\n\
 <KeyPress>KP_7: MoveTl()\n\
 <KeyPress>R7: MoveTl()\n\
 <KeyPress>Up: MoveTop()\n\
 <KeyPress>osfUp: MoveTop()\n\
 <KeyPress>KP_Up: MoveTop()\n\
 <KeyPress>KP_8: MoveTop()\n\
 <KeyPress>R8: MoveTop()\n\
 <KeyPress>Prior: MoveTr()\n\
 <KeyPress>KP_9: MoveTr()\n\
 <KeyPress>R9: MoveTr()\n\
 <KeyPress>Left: MoveLeft()\n\
 <KeyPress>osfLeft: MoveLeft()\n\
 <KeyPress>KP_Left: MoveLeft()\n\
 <KeyPress>KP_4: MoveLeft()\n\
 <KeyPress>R10: MoveLeft()\n\
 <KeyPress>Right: MoveRight()\n\
 <KeyPress>osfRight: MoveRight()\n\
 <KeyPress>KP_Right: MoveRight()\n\
 <KeyPress>KP_6: MoveRight()\n\
 <KeyPress>R12: MoveRight()\n\
 <KeyPress>End: MoveBl()\n\
 <KeyPress>KP_1: MoveBl()\n\
 <KeyPress>R1: MoveBl()\n\
 <KeyPress>Down: MoveBottom()\n\
 <KeyPress>osfDown: MoveBottom()\n\
 <KeyPress>KP_Down: MoveBottom()\n\
 <KeyPress>KP_2: MoveBottom()\n\
 <KeyPress>R14: MoveBottom()\n\
 <KeyPress>Next: MoveBr()\n\
 <KeyPress>KP_3: MoveBr()\n\
 <KeyPress>R3: MoveBr()\n\
 <Btn1Down>: Select()\n\
 <Btn1Up>: Release()\n\
 <KeyPress>p: Practice()\n\
 <Btn2Down>: PracticeMaybe()\n\
 <Btn2Down>(2+): Practice2()\n\
 <KeyPress>r: Randomize()\n\
 <Btn3Down>: RandomizeMaybe()\n\
 <Btn3Down>(2+): Randomize2()\n\
 <Btn4Down>: MoveTop()\n\
 <Btn5Down>: MoveBottom()\n\
 <KeyPress>g: Get()\n\
 <KeyPress>w: Write()\n\
 <KeyPress>c: Clear()\n\
 <KeyPress>u: Undo()\n\
 <KeyPress>s: Solve()\n\
 <KeyPress>o: Orientize()\n\
 <KeyPress>2: Period2()\n\
 <KeyPress>3: Period3()\n\
 <KeyPress>b: Both()\n\
 <EnterWindow>: Enter()\n\
 <LeaveWindow>: Leave()";

static XtActionsRec actionsListDino2D[] =
{
	{(char *) "Quit", (XtActionProc) QuitDino},
	{(char *) "Hide", (XtActionProc) HideDino},
	{(char *) "MoveTl", (XtActionProc) MoveDino2DTl},
	{(char *) "MoveTop", (XtActionProc) MoveDino2DTop},
	{(char *) "MoveTr", (XtActionProc) MoveDino2DTr},
	{(char *) "MoveLeft", (XtActionProc) MoveDino2DLeft},
	{(char *) "MoveRight", (XtActionProc) MoveDino2DRight},
	{(char *) "MoveBl", (XtActionProc) MoveDino2DBl},
	{(char *) "MoveBottom", (XtActionProc) MoveDino2DBottom},
	{(char *) "MoveBr", (XtActionProc) MoveDino2DBr},
	{(char *) "Select", (XtActionProc) SelectDino},
	{(char *) "Release", (XtActionProc) ReleaseDino},
	{(char *) "Practice", (XtActionProc) PracticeDino},
	{(char *) "PracticeMaybe", (XtActionProc) PracticeDinoMaybe},
	{(char *) "Practice2", (XtActionProc) PracticeDino2},
	{(char *) "Randomize", (XtActionProc) RandomizeDino},
	{(char *) "RandomizeMaybe", (XtActionProc) RandomizeDinoMaybe},
	{(char *) "Randomize2", (XtActionProc) RandomizeDino2},
	{(char *) "Get", (XtActionProc) GetDino},
	{(char *) "Write", (XtActionProc) WriteDino},
	{(char *) "Clear", (XtActionProc) ClearDino},
	{(char *) "Undo", (XtActionProc) UndoDino},
	{(char *) "Solve", (XtActionProc) SolveDino},
	{(char *) "Orientize", (XtActionProc) OrientizeDino},
	{(char *) "Period2", (XtActionProc) Period2ModeDino},
	{(char *) "Period3", (XtActionProc) Period3ModeDino},
	{(char *) "Both", (XtActionProc) BothModeDino},
	{(char *) "Enter", (XtActionProc) EnterDino},
	{(char *) "Leave", (XtActionProc) LeaveDino}
};

static XtResource resourcesDino2D[] =
{
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(DinoWidget, core.width),
	 XtRString, (caddr_t) "300"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(DinoWidget, core.height),
	 XtRString, (caddr_t) "400"},
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverse, XtCReverse, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(DinoWidget, dino.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNbackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(DinoWidget, dino.background),
	 XtRString, (caddr_t) XtDefaultBackground},
	{XtNframeColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(DinoWidget, dino.frameColor),
	 XtRString, (caddr_t) "cyan" /*XtDefaultForeground*/},
	{XtNfaceColor0, XtCLabel, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.faceName[0]),
	 XtRString, (caddr_t) "Red"},
	{XtNfaceColor1, XtCLabel, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.faceName[1]),
	 XtRString, (caddr_t) "Blue"},
	{XtNfaceColor2, XtCLabel, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.faceName[2]),
	 XtRString, (caddr_t) "Yellow"},
	{XtNfaceColor3, XtCLabel, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.faceName[3]),
	 XtRString, (caddr_t) "Green"},
	{XtNfaceColor4, XtCLabel, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.faceName[4]),
	 XtRString, (caddr_t) "White"},
	{XtNfaceColor5, XtCLabel, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.faceName[5]),
	 XtRString, (caddr_t) "Orange"},
	{XtNpieceBorder, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(DinoWidget, dino.borderColor),
	 XtRString, (caddr_t) "gray25" /*XtDefaultForeground*/},
	{XtNfont, XtCFont, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.font),
	 XtRString, (caddr_t) "9x15bold"},
	{XtNmode, XtCMode, XtRInt, sizeof (int),
	 XtOffset(DinoWidget, dino.mode),
	 XtRString, (caddr_t) "3"},	/*DEFAULTMODE */
	{XtNorient, XtCOrient, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.orient),
	 XtRString, (caddr_t) "FALSE"},	/* DEFAULTORIENT */
	{XtNpractice, XtCPractice, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.practice),
	 XtRString, (caddr_t) "FALSE"}, /* DEFAULTPRACTICE */
	{XtNuserName, XtCUserName, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.userName),
	 XtRString, (caddr_t) ""},
	{XtNscoreFile, XtCScoreFile, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.scoreFile),
	 XtRString, (caddr_t) ""},
	{XtNscoreOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.scoreOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNversionOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.versionOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmenu, XtCMenu, XtRInt, sizeof (int),
	 XtOffset(DinoWidget, dino.menu),
	 XtRString, (caddr_t) "-1"},
	{XtNstart, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.started),
	 XtRString, (caddr_t) "FALSE"},
	{XtNcheat, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.cheat),
	 XtRString, (caddr_t) "FALSE"},
	{XtNface, XtCFace, XtRInt, sizeof (int),
	 XtOffset(DinoWidget, dino.currentFace),
	 XtRString, (caddr_t) "-1"},
	{XtNpos, XtCPos, XtRInt, sizeof (int),
	 XtOffset(DinoWidget, dino.currentPosition),
	 XtRString, (caddr_t) "-1"},
	{XtNdirection, XtCDirection, XtRInt, sizeof (int),
	 XtOffset(DinoWidget, dino.currentDirection),
	 XtRString, (caddr_t) "-1"},
	{XtNstyle, XtCStyle, XtRInt, sizeof (int),
	 XtOffset(DinoWidget, dino.style),
	 XtRString, (caddr_t) "-1"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(DinoWidget, dino.select),
	 XtRCallback, (caddr_t) NULL}
};

Dino2DClassRec dino2dClassRec =
{
	{
		(WidgetClass) & dinoClassRec,	/* superclass */
		(char *) "Dino2D",	/* class name */
		sizeof (Dino2DRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializeDino2D,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListDino2D,	/* actions */
		XtNumber(actionsListDino2D),	/* num actions */
		resourcesDino2D,	/* resources */
		XtNumber(resourcesDino2D),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		NULL,		/* destroy */
		(XtWidgetProc) ResizeDino2D,	/* resize */
		(XtExposeProc) ExposeDino2D,	/* expose */
		(XtSetValuesFunc) SetValuesDino2D,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		XtInheritAcceptFocus,	/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		defaultTranslationsDino2D,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	},
	{
		0		/* ignore */
	}
};

WidgetClass dino2dWidgetClass = (WidgetClass) & dino2dClassRec;
#endif

static int  planeToCube[MAXRECT] =
{6, 0, 6, 1, 2, 3, 6, 4, 6, 6, 5, 6};
static int  cubeToPlane[MAXFACES] =
{1, 3, 4, 5, 7, 10};
static Point triangleUnit[MAXORIENT][4] =
{
	{
		{2, 0},
		{-1, 1},
		{-1, -1},
		{2, 0}},
	{
		{3, 2},
		{-1, -1},
		{1, -1},
		{0, 2}},
	{
		{1, 3},
		{1, -1},
		{1, 1},
		{-2, 0}},
	{
		{0, 1},
		{1, 1},
		{-1, 1},
		{0, -2}}
},          letterUnit[MAXORIENT] =
{
	{1, 0},
	{2, 1},
	{1, 2},
	{0, 1}};
static Point triangleList[MAXORIENT][4], letterList[MAXORIENT];
static Point offsetList[MAXORIENT];

static Pixmap dr = 0; /* dummy for future double buffering */

static void
DrawOrientLine(Dino2DWidget w, int orient, int dx, int dy, GC borderGC)
{
	switch (orient) {
		case TOP:
			DRAWLINE(w, dr, borderGC,
				  dx,
				  dy - w->dino.delta,
				  dx,
			  dy - w->dino.delta - w->dino.orientLineLength / 2);
			return;
		case RIGHT:
			DRAWLINE(w, dr, borderGC,
				  dx + w->dino.delta,
				  dy,
			   dx + w->dino.delta + w->dino.orientLineLength / 2,
				  dy);
			return;
		case BOTTOM:
			DRAWLINE(w, dr, borderGC,
				  dx,
				  dy + w->dino.delta,
				  dx,
			  dy + w->dino.delta + w->dino.orientLineLength / 2);
			return;
		case LEFT:
			DRAWLINE(w, dr, borderGC,
				  dx - w->dino.delta,
				  dy,
			   dx - w->dino.delta - w->dino.orientLineLength / 2,
				  dy);
			return;
		default:
			{
				char *buf;

				intCat(&buf, "DrawOrientLine: orient ",
					 orient);
				DISPLAY_WARNING(buf);
				free(buf);
			}
	}
}

void
DrawTriangle2D(Dino2DWidget w, int face, int position, int offset)
{
	GC          faceGC, borderGC;
	int         dx, dy, letterX, letterY, newSide, orient;

	orient = w->dino.cubeLoc[face][position].rotation;
	if (w->dino.vertical || face != MAXFACES - 1) {
		dx = (cubeToPlane[face] % MAXX) * w->dino2d.viewLength;
		dy = (cubeToPlane[face] / MAXX) * w->dino2d.viewLength;
		newSide = position;
	} else {
		dx = (cubeToPlane[face] / MAXX) * w->dino2d.viewLength;
		dy = (cubeToPlane[face] % MAXX) * w->dino2d.viewLength;
		newSide = (position + HALF) % STRT;
		orient = (orient + HALF) % STRT;
	}
	dx += w->dino.puzzleOffset.x + 2 * w->dino.delta;
	dy += w->dino.puzzleOffset.y + 2 * w->dino.delta;
	letterX = dx + letterList[newSide].x - 1;
	letterY = dy + letterList[newSide].y;
	triangleList[newSide][0].x = offsetList[newSide].x + dx;
	triangleList[newSide][0].y = offsetList[newSide].y + dy;
	if (offset) {
		borderGC = w->dino.faceGC[(int) w->dino.cubeLoc[face][position].face];
		if (w->dino.mono) {
			faceGC = w->dino.inverseGC;
		} else {
			faceGC = w->dino.borderGC;
		}
	} else {
		faceGC = w->dino.faceGC[(int) w->dino.cubeLoc[face][position].face];
		borderGC = w->dino.borderGC;
	}
	POLYGON((DinoWidget) w, dr, faceGC, borderGC,
		triangleList[newSide], 3, True, False);
	if (w->dino.mono) {
		char        buf[2];

		buf[0] =
#ifdef WINVER
		 	w->dino.faceChar[w->dino.cubeLoc
			 	[face][position].face];
#else
			w->dino.faceName[w->dino.cubeLoc
				[face][position].face][0];
#endif
		if (offset) {
			borderGC = w->dino.borderGC;
		} else {
			borderGC = w->dino.inverseGC;
		}
		DRAWTEXT(w, dr, borderGC, letterX, letterY, buf, 1);
	}
	if (w->dino.orient)
		DrawOrientLine(w, orient + 2 * MAXORIENT,
			       letterX - w->dino.letterOffset.x, letterY - w->dino.letterOffset.y,
			       borderGC);
}

void
DrawFrame2D(Dino2DWidget w, Boolean focus)
{
	int         i;
	Point       pos[MAXXY + 1], letters;
	GC gc = (focus) ? w->dino.frameGC : w->dino.borderGC;

	for (i = 0; i <= MAXXY; i++) {
		pos[i].x = i * w->dino2d.viewLength + w->dino.puzzleOffset.x;
		pos[i].y = i * w->dino2d.viewLength + w->dino.puzzleOffset.y;
	}
	DRAWLINE(w, dr, gc, pos[1].x, pos[0].y, pos[2].x, pos[0].y);
	DRAWLINE(w, dr, gc, pos[3].x, pos[1].y, pos[3].x, pos[2].y);
	DRAWLINE(w, dr, gc, pos[1].x, pos[3].y, pos[2].x, pos[3].y);
	DRAWLINE(w, dr, gc, pos[0].x, pos[1].y, pos[0].x, pos[2].y);
	letters.x = pos[0].x + w->dino2d.viewLength / 2 - w->dino.delta;
	letters.y = pos[0].y + w->dino2d.viewLength / 2;
	DRAWTEXT(w, dr, w->dino.borderGC,
		    (int) (letters.x + 5 * w->dino.letterOffset.x),
		    (int) (letters.y + w->dino.letterOffset.y), "Front", 5);
	letters.x = pos[2].x + w->dino2d.viewLength / 2 - w->dino.delta;
	letters.y = pos[2].y + w->dino2d.viewLength / 2;
	DRAWTEXT(w, dr, w->dino.borderGC,
		    (int) (letters.x + 4 * w->dino.letterOffset.x),
		    (int) (letters.y + w->dino.letterOffset.y), "Back", 4);
	if (w->dino.vertical) {
		DRAWLINE(w, dr, gc, pos[1].x, pos[0].y, pos[1].x, pos[4].y);
		DRAWLINE(w, dr, gc, pos[2].x, pos[0].y, pos[2].x, pos[4].y);
		DRAWLINE(w, dr, gc, pos[0].x, pos[1].y, pos[3].x, pos[1].y);
		DRAWLINE(w, dr, gc, pos[0].x, pos[2].y, pos[3].x, pos[2].y);
		DRAWLINE(w, dr, gc, pos[1].x, pos[4].y, pos[2].x, pos[4].y);
	} else {
		DRAWLINE(w, dr, gc, pos[0].x, pos[1].y, pos[4].x, pos[1].y);
		DRAWLINE(w, dr, gc, pos[0].x, pos[2].y, pos[4].x, pos[2].y);
		DRAWLINE(w, dr, gc, pos[1].x, pos[0].y, pos[1].x, pos[3].y);
		DRAWLINE(w, dr, gc, pos[2].x, pos[0].y, pos[2].x, pos[3].y);
		DRAWLINE(w, dr, gc, pos[4].x, pos[1].y, pos[4].x, pos[2].y);
	}
}

static void
ResizePieces(Dino2DWidget w)
{
	int         i, j;

	w->dino.orientLineLength = w->dino2d.diamondLength / 3;
	w->dino.letterOffset.x = -2;
	w->dino.letterOffset.y = 4;
	for (i = 0; i < MAXORIENT; i++) {
		for (j = 0; j <= 3; j++) {
			triangleList[i][j].x = triangleUnit[i][j].x *
				(w->dino2d.diamondLength / 2 - 2 * w->dino.delta);
			triangleList[i][j].y = triangleUnit[i][j].y *
				(w->dino2d.diamondLength / 2 - 2 * w->dino.delta);
		}
		if (letterUnit[i].x == 0)
			letterList[i].x = w->dino2d.diamondLength / 5 +
				w->dino.letterOffset.x;
		else if (letterUnit[i].x == 1)
			letterList[i].x = w->dino2d.diamondLength / 2 - 1 +
				w->dino.letterOffset.x;
		else if (letterUnit[i].x == 2)
			letterList[i].x = 4 * w->dino2d.diamondLength / 5 - 2 +
				w->dino.letterOffset.x;
		if (letterUnit[i].y == 0)
			letterList[i].y = w->dino2d.diamondLength / 5 + 2 +
				w->dino.letterOffset.y;
		else if (letterUnit[i].y == 1)
			letterList[i].y = w->dino2d.diamondLength / 2 - 1 +
				w->dino.letterOffset.y;
		else if (letterUnit[i].y == 2)
			letterList[i].y = 4 * w->dino2d.diamondLength / 5 - 3 +
				w->dino.letterOffset.y;

		if (triangleUnit[i][0].x == 0)
			offsetList[i].x = w->dino.delta - 1;
		else if (triangleUnit[i][0].x == 1)
			offsetList[i].x = 2 * w->dino.delta - 1;
		else if (triangleUnit[i][0].x == 2)
			offsetList[i].x = w->dino2d.diamondLength -
				2 * w->dino.delta - 2;
		else if (triangleUnit[i][0].x == 3)
			offsetList[i].x = w->dino2d.diamondLength - w->dino.delta - 2;
		if (triangleUnit[i][0].y == 0)
			offsetList[i].y = w->dino.delta - 2;
		else if (triangleUnit[i][0].y == 1)
			offsetList[i].y = 2 * w->dino.delta - 1;
		else if (triangleUnit[i][0].y == 2)
			offsetList[i].y = w->dino2d.diamondLength -
				2 * w->dino.delta - 2;
		else if (triangleUnit[i][0].y == 3)
			offsetList[i].y = w->dino2d.diamondLength - w->dino.delta - 1;
	}
}

Boolean
SelectPieces2D(Dino2DWidget w, int x, int y, int *face, int *position)
{
	int         faceX, faceY, i, j;

	x -= w->dino.puzzleOffset.x;
	y -= w->dino.puzzleOffset.y;
	faceX = x / w->dino2d.viewLength;
	faceY = y / w->dino2d.viewLength;
	i = x - faceX * w->dino2d.viewLength;
	j = y - faceY * w->dino2d.viewLength;
	if (i + j < w->dino2d.viewLength && i - j > 0)
		*position = TOP;
	else if (i - j > 0 && i + j > w->dino2d.viewLength)
		*position = RIGHT;
	else if (i + j > w->dino2d.viewLength && j - i > 0)
		*position = BOTTOM;
	else if (j - i > 0 && i + j < w->dino2d.viewLength)
		*position = LEFT;
	else
		*position = MAXORIENT;
	if ((faceX != 1 && faceY != 1) ||
	    (faceX >= 3 && w->dino.vertical) ||
	    (faceY >= 3 && !w->dino.vertical))
		return FALSE;
	*face = planeToCube[faceX + faceY * MAXX];
	if (*position != MAXORIENT && faceX == 3) {
		*face = MAXFACES - 1;
		*position = (*position + HALF) % MAXORIENT;
	} else
		*position %= MAXORIENT;
	return TRUE;
}

Boolean
NarrowSelection2D(Dino2DWidget w, int *face, int *direction)
{
	if (*face == MAXFACES - 1 && *direction < MAXORIENT && !w->dino.vertical) {
		if (*direction < MAXORIENT)
			*direction = (*direction + HALF) % MAXORIENT;
		else if (*direction >= 2 * MAXORIENT)
			*direction = 2 * MAXORIENT + (*direction + HALF) % MAXORIENT;
	}
	return TRUE;
}

#ifndef WINVER
static      Boolean
SetValuesDino2D(Widget current, Widget request, Widget renew)
{
	Dino2DWidget c = (Dino2DWidget) current, w = (Dino2DWidget) renew;
	Boolean     redraw = False;

	if (w->dino2d.diamondLength != c->dino2d.diamondLength) {
		ResizeDino2D(w);
		redraw = TRUE;
	}
	return (redraw);
}
#endif

#ifndef WINVER
static
#endif
void
ResizeDino2D(Dino2DWidget w)
{
#ifdef WINVER
	RECT        rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif
	w->dino.delta = 2;
	w->dino.vertical = (w->core.height >= w->core.width);
	if (w->dino.vertical)
		w->dino2d.faceLength = MIN(w->core.height / MAXY, w->core.width / MAXX);
	else
		w->dino2d.faceLength = MIN(w->core.height / MAXX, w->core.width / MAXY);
	w->dino2d.faceLength = MAX(w->dino2d.faceLength - w->dino.delta - 1, 0);
	w->dino2d.diamondLength = w->dino2d.faceLength - w->dino.delta;
	w->dino2d.viewLength = w->dino2d.faceLength + w->dino.delta;
	if (w->dino.vertical) {
		w->dino.puzzleSize.x = MAXX * (w->dino2d.viewLength - 1) +
			w->dino.delta;
		w->dino.puzzleSize.y = MAXY * (w->dino2d.viewLength - 1) +
			w->dino.delta;
	} else {
		w->dino.puzzleSize.x = MAXY * (w->dino2d.viewLength - 1) +
			w->dino.delta;
		w->dino.puzzleSize.y = MAXX * (w->dino2d.viewLength - 1) +
			w->dino.delta;
	}
	w->dino.puzzleOffset.x = ((int) w->core.width - w->dino.puzzleSize.x)
		/ 2;
	w->dino.puzzleOffset.y = ((int) w->core.height - w->dino.puzzleSize.y)
		/ 2;
	ResizePieces(w);
}

#ifndef WINVER
static
#endif
void
InitializeDino2D(
#ifdef WINVER
Dino2DWidget w
#else
Widget request, Widget renew
#endif
)
{
#ifndef WINVER
	Dino2DWidget w = (Dino2DWidget) renew;

	SetAllColors((DinoWidget) w);
#endif
	w->dino.dim = 2;
	ResizeDino2D(w);
}

#ifndef WINVER
static
#endif
void
ExposeDino2D(
#ifdef WINVER
Dino2DWidget w
#else
Widget renew, XEvent * event, Region region
#endif
)
{
#ifndef WINVER
	Dino2DWidget w = (Dino2DWidget) renew;

	if (!w->core.visible)
		return;
#endif
	FILLRECTANGLE(w, dr, w->dino.inverseGC,
		0, 0, w->core.width, w->core.height);
	DrawFrame2D(w, w->dino.focus);
	DrawAllPieces((DinoWidget) w);
}

#ifndef WINVER
static void
MoveDino2DTl(Dino2DWidget w, XEvent * event, char **args, int nArgs)
{
	MoveDinoInput((DinoWidget) w, event->xbutton.x, event->xbutton.y, TL,
		      (int) (event->xkey.state & (ShiftMask | LockMask)),
		      (int) (event->xkey.state & ControlMask),
		      (int) (event->xkey.state &
		    (Mod1Mask | Mod2Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
MoveDino2DTop(Dino2DWidget w, XEvent * event, char **args, int nArgs)
{
	MoveDinoInput((DinoWidget) w, event->xbutton.x, event->xbutton.y, TOP,
		      (int) (event->xkey.state & (ShiftMask | LockMask)),
		      (int) (event->xkey.state & ControlMask),
		      (int) (event->xkey.state &
		    (Mod1Mask | Mod2Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
MoveDino2DTr(Dino2DWidget w, XEvent * event, char **args, int nArgs)
{
	MoveDinoInput((DinoWidget) w, event->xbutton.x, event->xbutton.y, TR,
		      (int) (event->xkey.state & (ShiftMask | LockMask)),
		      (int) (event->xkey.state & ControlMask),
		      (int) (event->xkey.state &
		    (Mod1Mask | Mod2Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
MoveDino2DLeft(Dino2DWidget w, XEvent * event, char **args, int nArgs)
{
	MoveDinoInput((DinoWidget) w, event->xbutton.x, event->xbutton.y, LEFT,
		      (int) (event->xkey.state & (ShiftMask | LockMask)),
		      (int) (event->xkey.state & ControlMask),
		      (int) (event->xkey.state &
		    (Mod1Mask | Mod2Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
MoveDino2DRight(Dino2DWidget w, XEvent * event, char **args, int nArgs)
{
	MoveDinoInput((DinoWidget) w, event->xbutton.x, event->xbutton.y, RIGHT,
		      (int) (event->xkey.state & (ShiftMask | LockMask)),
		      (int) (event->xkey.state & ControlMask),
		      (int) (event->xkey.state &
		    (Mod1Mask | Mod2Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
MoveDino2DBl(Dino2DWidget w, XEvent * event, char **args, int nArgs)
{
	MoveDinoInput((DinoWidget) w, event->xbutton.x, event->xbutton.y, BL,
		      (int) (event->xkey.state & (ShiftMask | LockMask)),
		      (int) (event->xkey.state & ControlMask),
		      (int) (event->xkey.state &
		    (Mod1Mask | Mod2Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
MoveDino2DBottom(Dino2DWidget w, XEvent * event, char **args, int nArgs)
{
	MoveDinoInput((DinoWidget) w, event->xbutton.x, event->xbutton.y, BOTTOM,
		      (int) (event->xkey.state & (ShiftMask | LockMask)),
		      (int) (event->xkey.state & ControlMask),
		      (int) (event->xkey.state &
		    (Mod1Mask | Mod2Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}

static void
MoveDino2DBr(Dino2DWidget w, XEvent * event, char **args, int nArgs)
{
	MoveDinoInput((DinoWidget) w, event->xbutton.x, event->xbutton.y, BR,
		      (int) (event->xkey.state & (ShiftMask | LockMask)),
		      (int) (event->xkey.state & ControlMask),
		      (int) (event->xkey.state &
		    (Mod1Mask | Mod2Mask | Mod3Mask | Mod4Mask | Mod5Mask)));
}
#endif
