/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <stdlib.h>
#include <stdio.h>

#include <xmms/xmmsctrl.h>
#include <xmms/util.h>

#include <SDL/SDL_types.h>
#include <SDL/SDL_syswm.h>
#include <SDL/SDL_ttf.h>

#include "singit/displayer_plugin.h"
#include "singit/control.h"

#include "displayer_test_sdl.h"
#include "displayer_test_sdl_status.h"

#include "test_sdl_handlers.h"
#include "test_sdl_lyrics.h"
#include "sdl_helpers.h"


static void displayer_test_sdl_init(void);
static void displayer_test_sdl_finish(void);

static void displayer_test_sdl_set_song(SingitSong *cur_song);
static void displayer_test_sdl_set_time(guint time, GList *real_next);

static void displayer_test_sdl_playback_start(void);
static void displayer_test_sdl_playback_stop(void);

static void displayer_test_sdl_config_update(void);

DisplayerPlugin test_sdl_dp =
{
	NULL,
	NULL,
	0,
	0,
	NULL,

	"Test SDL displayer",

	displayer_test_sdl_init,
	displayer_test_sdl_finish,

	displayer_test_sdl_config_update,

	displayer_test_sdl_set_song,
	displayer_test_sdl_set_time,

	NULL,
	NULL,
	displayer_test_sdl_playback_start,
	displayer_test_sdl_playback_stop,

	NULL,
	NULL
};

DIPSPLAYER_SYMBOL(libdisplayer_test_sdl, test_sdl_dp)

gint displayer_test_sdl_disable_func(gpointer data)
{
	GDK_THREADS_ENTER();
	test_sdl_dp.disable(&test_sdl_dp);
	GDK_THREADS_LEAVE();
	return (FALSE);
}

static void clear_screen()
{
	SDL_Rect rect;

	if (displayer_test_sdl_status.screen != SDL_GetVideoSurface())
		{ return; }

	rect.x = 0;
	rect.y = 0;
	rect.w = displayer_test_sdl_status.screen->w;
	rect.h = displayer_test_sdl_status.screen->h;

	SDL_FillRect(displayer_test_sdl_status.screen, &rect, 0);
}

static void displayer_test_sdl_process_events(guint time)
{
	SDL_Event sdl_event;

	if (displayer_test_sdl_status.sdl_inuse == TRUE)
		{ return; }

	if (displayer_test_sdl_status.screen != SDL_GetVideoSurface())
		{ return; }

	clear_screen();
	lyrics_render_text(time);
	lyrics_draw_solid(displayer_test_sdl_status.screen);

	if (displayer_test_sdl_status.screen == SDL_GetVideoSurface())
		SDL_UpdateRect(displayer_test_sdl_status.screen, 0, 0, 0, 0);

	if (!displayer_test_sdl_status.kill_render_thread && (SDL_PollEvent(&sdl_event) > 0)) {
		switch (sdl_event.type) {
		case SDL_KEYDOWN:
			tsnh_key_down(&sdl_event);
			break;
		case SDL_QUIT:
			// Prevent double shutdown
			sdl_helper_blanking(BLANK_RESET);
			if (displayer_test_sdl_status.kill_render_thread == FALSE) {
				gtk_idle_add(displayer_test_sdl_disable_func, NULL);
			}
			break;
		case SDL_VIDEORESIZE:
			tsnh_resize(&sdl_event);
			break;
		case SDL_ACTIVEEVENT: // Application visibility event structure
			tsnh_active(&sdl_event);
			break;
		case SDL_VIDEOEXPOSE:
			tsnh_expose_video();
			break;
		}
	}
}

static void *displayer_test_sdl_event_pthread_func(void *args)
{
	while (!displayer_test_sdl_status.kill_render_thread)
	{
		displayer_test_sdl_process_events(0);
	}

#ifdef CODEDEBUG
	DEBUG(("  Quit render thread\n"));
#endif

	displayer_test_sdl_status.kill_render_thread = FALSE;
	displayer_test_sdl_status.render_thread = 0;

	pthread_exit(0);
}

static gboolean sdl_render_pthread_quit()
{
	if (displayer_test_sdl_status.render_thread != 0)
	{
		displayer_test_sdl_status.kill_render_thread = TRUE;
		pthread_join(displayer_test_sdl_status.render_thread, NULL);
		displayer_test_sdl_status.kill_render_thread = FALSE;
		displayer_test_sdl_status.render_thread = 0;
		return TRUE;
	}
	return FALSE;
}

static gboolean set_sdl_render_pthread()
{
	gboolean result = FALSE;

	gboolean need_thread =
		((displayer_test_sdl_status.xmms_playing == FALSE) ||
		 (displayer_test_sdl_status.singit_playing == FALSE) ||
		 (xmms_remote_is_paused(test_sdl_dp.xmms_session) == TRUE));

	if ((displayer_test_sdl_status.render_thread == 0) && (need_thread == TRUE))
	{
#ifdef CODEDEBUG
		DEBUG(("  Start render thread\n"));
#endif
		if (pthread_create(&displayer_test_sdl_status.render_thread, NULL,
			displayer_test_sdl_event_pthread_func, NULL) != 0)
		{
			displayer_test_sdl_status.render_thread = 0;
			g_print("Unable to create sdl render thread\n");
		}
		else { result = TRUE; }
	}
	else if (need_thread == FALSE)
		{ result = sdl_render_pthread_quit(); }

	return result;
}

static void displayer_test_sdl_init(void)
{
	SDL_Color col[3];

#ifdef CODEDEBUG
	DEBUG(("displayer_test_sdl.c [displayer_test_sdl_init]\n"));
#endif

	displayer_test_sdl_status_init();

	if (tsnh_init() == FALSE) {
		gtk_idle_add (displayer_test_sdl_disable_func, NULL);
		return;
	}
	lyrics_init();
	lyrics_set_optimize_font(TRUE);

	col[0].r = 255;
	col[0].g = 255;
	col[0].b = 255;
	col[1].r = 128;
	col[1].g = 128;
	col[1].b = 128;
	col[2].r = 0;
	col[2].g = 0;
	col[2].b = 0;
	lyrics_set_text_colors(col[0], col[1], col[2]);
	lyrics_set_font("/usr/X11R6/lib/X11/fonts/truetype/luxisr.ttf", 32, TRUE);

//	if (!((scd->lyricFontName != NULL) && lyrics_set_font(scd->lyricFontName, scd->lyricFontSize, TRUE)))
//		{ lyrics_set_font(SINGIT_DATA_DIR "/vixar.ttf", 32, TRUE); }

	displayer_test_sdl_status.xmms_playing = FALSE;
}

static void displayer_test_sdl_finish(void)
{
#ifdef CODEDEBUG
	DEBUG(("displayer_test_sdl.c [displayer_test_sdl_finish]\n"));
#endif

	sdl_render_pthread_quit();

	lyrics_finish();

	displayer_test_sdl_status_finish();
}

static void displayer_test_sdl_set_time(guint time, GList *real_next)
{
	set_sdl_render_pthread();

	if (displayer_test_sdl_status.screen == SDL_GetVideoSurface())
	{
		displayer_test_sdl_status.time = time;
		if (displayer_test_sdl_status.render_thread == 0)
			{ displayer_test_sdl_process_events(time); }
	}
}

static void displayer_test_sdl_config_update(void)
{
#ifdef CODEDEBUG
	DEBUG(("displayer_test_sdl.c [displayer_test_sdl_config_update]\n"));
#endif

	displayer_test_sdl_status.toggle_yz = singit_remote_cfg_toggle_yz
		(test_sdl_dp.singit_session);
}

static void displayer_test_sdl_playback_start(void)
{
#ifdef CODEDEBUG
	DEBUG(("displayer_test_sdl.c [displayer_test_sdl_playback_start]\n"));
#endif

	displayer_test_sdl_status.xmms_playing = TRUE;
}

static void displayer_test_sdl_playback_stop(void)
{
#ifdef CODEDEBUG
	DEBUG(("displayer_test_sdl.c [displayer_test_sdl_playback_stop]\n"));
#endif

	displayer_test_sdl_status.xmms_playing = FALSE;
	set_sdl_render_pthread();
}

static void displayer_test_sdl_set_song(SingitSong *new_song)
{
	SingitSong *tmp_song;

	tmp_song = displayer_test_sdl_status.song;
	displayer_test_sdl_status.song = singit_song_attach(new_song);
	singit_song_detach(&tmp_song);

	displayer_test_sdl_status.singit_playing =
		singit_song_lyrics_found(new_song);

	displayer_test_sdl_status.xmms_playing =
		xmms_remote_is_playing(test_sdl_dp.xmms_session);

	clear_screen();

	lyrics_set_song(new_song);

	if (singit_song_lyrics_found(new_song)) {
		if (!xmms_remote_is_paused(test_sdl_dp.xmms_session) &&
			(set_sdl_render_pthread()))
		{
			sdl_helper_map(TRUE);
		}
	}
	else {
		set_sdl_render_pthread();
	}
}
