

#ifndef __PIXPORT_H__
#define __PIXPORT_H__

#include "VFrameBuffer.h"
#include "FrameBuffer.h"

#include "UtilStr.h"
#include "TempMem.h"
#include "XPtrList.h"
#include "DictValue.h"
#include "Hashtable.h"

#include "CEgFileSpec.h"

#if EG_MAC || WIN_QUICKTIME_PRESENT
#include <QuickDraw.h>
#include <QDOffscreen.h>
#endif

#ifdef EG_SDL
#include <SDL/SDL_video.h>
#endif

class DeltaFieldData {
public:
	long			mX, mY;		// mBytesPerRow is how many bytes in each PixMap row
	char*			mField;
};

class PixTextStyle : public DictValue, public Hashable {

public:
	virtual ~PixTextStyle() {
#if EG_WIN
		::DeleteObject( (HFONT) mOSFontID );
#endif
	}

	UtilStr		mFontName;
	short		mPointSize;
	short		mStyle;
	short		mAscent, mDescent, mLineHeight;
	long		mOSFontID;
	long		mOSStyle;

	long		Hash( long inParam ) const {

		return mFontName.Hash( inParam ) * 37 + mPointSize * 19 + mStyle;
	}

	bool		Equals( const Hashable* inComp, long inParam ) const {

		if ( mPointSize == ( (PixTextStyle*)inComp ) -> mPointSize ) {
			if ( mStyle == ( (PixTextStyle*)inComp ) -> mStyle ) {
				return mFontName.compareTo( &( (PixTextStyle*)inComp ) -> mFontName, inParam ) == 0;
			}
		}
		return false;
	}
};


enum PixDrawMode {
	SRC_OR,
	SRC_BIC,
	SRC_XOR
};

#define PP_DONT_SET_OS_SEED		0xFFEFFEEF

/*  PixPort is a platform indep drawing world/environment.  You set the size of one using
Init() and then execute whatever drawing commands. When the drawing is complete, CopyBits()
copies pixel data from this world to the given destination OS graphics world.  */

#define	PP_PLAIN	0
#define PP_BOLD		0x1
#define	PP_ITALIC	0x2
#define	PP_UNDERLINE	0x4

class PixPort : public VFrameBuffer {
	friend class FrameBuffer;
public:
	PixPort();
	virtual	~PixPort();

	// One or the other must be called before a PixPort is used.
	// Call Deactivate() or Init() to exit fullscreen mode.
	// inWin is passed because PixPort may need the window that's going fullscreen
	void		Init( PP_GrafPtr inPort );
	void		Init( int inWidth, int inHeight, int inDepth );

	//	Sets the background colors (for erase rect and Blur() )
	//  Returns the new pixel entry for the given color and this port
	//  Note:  For 8 bit ports, the red 0-2^16 component maps directly to 0-255 pixel value
	long		SetBackColor( const RGBColor& inColor );
	long		SetBackColor( long inR, long inG, long inB );
	void		SetBackColor( long inDevicePixelValue )							{ mBackColor = inDevicePixelValue;	}

	//	Blurs the rect given in this image, with a given box filter of size (1=no blur)
	//	If the dest is NULL, the blur is applied to itself
	void		GaussBlur( int inBoxWidth, const Rect& inRect, void* inDestBits = NULL );

	//	A different, more primitive blur that doesn't look as good as GaussBlur,
	void		CrossBlur( const Rect& inRect );

	// 	Sets the width of the pen
	void		SetLineWidth( long inWidth );
	int		GetLineWidth()													{ return mLineWidth; }

	//	Draw a line.  Use GetPortColor() to get a device color for a RGB
	void		Line( int sx, int sy, int ex, int ey, long inColor );
	void		Line( int sx, int sy, int ex, int ey, const RGBColor& inS, const RGBColor& inE );

	//	Sets the clip rgn for all drawing operations.  NULL for the cliprect means to remove all clipping.
	void		SetClipRect( const Rect* inRect = NULL );
	void		SetClipRect( long inSX, long inSY, long inEX, long inEY );

	//  Note:  For 8 bit ports, the red 0-2^16 component maps directly to 0-255 pixel value
	inline long	GetPortColor_inline( long inR, long inG, long inB ) {

		if ( mDepth == 16 )
			return __RGBTo16( inR, inG, inB );
		else if ( mDepth == 32 )
			return __RGBTo32( inB, inG, inR );
		else
			return __RGBTo8( inR, inG, inB );
	}

	void		SetTextMode( PixDrawMode inMode );
	void		SetTextColor( PixPalEntry& inColor );
	void		SetTextColor( RGBColor& inColor );
	void		SetPaletteColor( long inEntry );
	void		DrawText( long inX, long inY, UtilStr& inStr )									{ DrawText( inX, inY, inStr.getCStr() ); 	}
	void		DrawText( long inX, long inY, const char* inStr );

	// See how big some text is going to be...
	void		TextRect( const char* inStr, long& outWidth, long& outHeight );

	//	Set the given rect to the current background color.  If no rect is specified, the entire port rect is cleared.
	void		EraseRect( const Rect* inRect = NULL );

	//	Copies the pixels in a rectangle from this pixel rect to the destination.  If the port is in fullscreen, inDestWin and inDest are ignored.
	void		CopyBits( PP_GrafPtr inDestPort, const Rect* inSrce, const Rect* inDest );
	void		CopyBits( PixPort& inDestPort, const Rect* inSrce, const Rect* inDest );

	//  Weather this port is inited or not, this routes drawing to the given frame buffer.  Of course, this must be called whenever you
	//  suspect that your frame buffer is different (ex, of you resize it).  Also, if you make a PixPort call that inits or de-inits it,
	//  your redirect will stop.
	//	Note:  PixPort's current implementation is such that any text drawn will not be redirected
	//void					RedirectDrawing( void* inBits, long inBytesPerRow, long inBitDepth, long inX, long inY );

	//	Gets a port/device color for a given RGB
	//  Note:  For 8 bit ports, the red 0-2^16 component maps directly to 0-255 pixel value
	long		GetPortColor( long inR, long inG, long inB );
	inline long	GetPortColor( const RGBColor& inColor )  		{ return GetPortColor( inColor.red, inColor.green, inColor.blue );  }


	//	When this sprocket is set to 256 colors, you may change the palette it's using any time
	//	Pre: inColors[].rgb is the RGB of palette entry i.
	//	Pre: <inPaletteSeed> corresponds to <inPal>
	//	Post:  The current palette is set to inColors[]
	//	If inSysWindow holds an OS window, SetPalette will set that particular window to the palette give in inColors and force the OS palette to this palette.
	void		SetPalette( PixPalEntry inPal[ 256 ], long inPaletteSeed = PP_DONT_SET_OS_SEED );

	//	Sets the port font and style.
	//	Returns:  a long usable in SetFont( long ) for faster swtiches
	long		SetFont( const UtilStr&	inFontName, long inSize, long inStyleFlags = PP_PLAIN )				{ return SetFont( inFontName.getCStr(), inSize, inStyleFlags );		}
	long		SetFont( const char*	inFontName, long inSize, long inStyleFlags = PP_PLAIN );

	//	Set the port font and a style based on an ID returned from the above version of SetFont()
	void		SetFont( long inFontID );

	// Save this PixPort to disk as a BMP
	void		WriteImage( const CEgFileSpec& inSpec );

	static char*	sTemp;
	static long	sTempSize;

	long		GetCurPaletteSeed()									{ return mCurPaletteSeed;	}

#if EG_MAC
	PixMapHandle	GetPixMap()											{ return mBM; 				}
#endif

#if EG_WIN
	HDC		GetDC()												{ return mWorld;			}
#endif

#define MAX_LINE_WIDTH	32

#if WIN_QUICKTIME_PRESENT || EG_MAC
	GWorldPtr	GetGWorld();
#endif

protected:
	Rect			mClipRect;
	long			mBackColor;
	long			mLineWidth;
	long 			mCurPaletteSeed;

	PP_PixMapHandle		mBM;
	PP_GWorldPtr		mWorld;

	TempMem			mBlurTemp;

	Hashtable		mFontTable;
	long			mCurFontID;

#if EG_WIN
	BITMAPINFO		mInfo;
#endif

#if WIN_QUICKTIME_PRESENT
	GWorldPtr		mQTML_World;
#endif

	void			Init();

	void			EraseRect8 ( const Rect* inRect );
	void			EraseRect16( const Rect* inRect );
	void			EraseRect32( const Rect* inRect );

	static void		BoxBlur8 ( char* inSrce, char* inDest, int inBoxWidth, int inWidth, int inHeight, int inSrceRowSize, int inDestRowSize, unsigned long* temp, unsigned long inBackColor );
	static void		BoxBlur16( char* inSrce, char* inDest, int inBoxWidth, int inWidth, int inHeight, int inSrceRowSize, int inDestRowSize, unsigned long* temp, unsigned long inBackColor );
	static void		BoxBlur32( char* inSrce, char* inDest, int inBoxWidth, int inWidth, int inHeight, int inSrceRowSize, int inDestRowSize, unsigned long* temp, unsigned long inBackColor );

	static void		CrossBlur8 ( char* inSrce, int inWidth, int inHeight, int inBytesPerRow, unsigned char* inRowBuf );
	static void		CrossBlur16( char* inSrce, int inWidth, int inHeight, int inBytesPerRow, unsigned char* inRowBuf );
	static void		CrossBlur32( char* inSrce, int inWidth, int inHeight, int inBytesPerRow, unsigned char* inRowBuf );

	void			Line8 ( int sx, int sy, int ex, int ey, long inColor );
	void			Line16( int sx, int sy, int ex, int ey, long inColor );
	void			Line32( int sx, int sy, int ex, int ey, long inColor );

	void			Line8 ( int sx, int sy, int ex, int ey, long inR, long dR );
	void			Line16( int sx, int sy, int ex, int ey, const RGBColor& inS, long dR, long dG, long dB );
	void			Line32( int sx, int sy, int ex, int ey, const RGBColor& inS, long dR, long dG, long dB );

//	static void		Fade( char* ioPix, long inBytesPerRow, long inX, long inY, DeltaFieldData* inGrad );
};

#endif // __PIXPORT_H__
