/*  
   XMMS-SID - SIDPlay input plugin for X MultiMedia System (XMMS)

   Get song length (from database or by other means)
   
   Written by Matti "ccr" Hamalainen <ccr@tnsp.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#include "xs_length.h"
#include "xs_support.h"
#include "xs_config.h"
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>


/*
 * Pointer to database in memory
 */
static t_xs_sldb_node	*xs_sldb = NULL;
static t_xs_sldb_node	**xs_sldbi = NULL;
static gint		xs_sldbn   = 0;


/*
 * Hash-database handling functions
 */
t_xs_sldb_node *xs_sldb_node_new(void)
{
 t_xs_sldb_node *pResult;
 
 /* Allocate memory for new node */
 pResult = (t_xs_sldb_node *) g_malloc0(sizeof(t_xs_sldb_node));
 if (!pResult) return NULL;
 
 return pResult;
}


void xs_sldb_node_free(t_xs_sldb_node *pNode)
{
 if (pNode) g_free(pNode);
}


/*
 * Insert given node to db linked list
 */
#define LPREV	(pNode->pPrev)
#define LTHIS	(pNode)
#define LNEXT	(pNode->pNext)

void xs_sldb_node_insert(t_xs_sldb_node *pNode)
{
 if (xs_sldb)
 	{
 	/* The first node's pPrev points to last node */
	LPREV = xs_sldb->pPrev;		/* New node's prev = Previous last node */
	xs_sldb->pPrev->pNext = pNode;	/* Previous last node's next = New node */
	xs_sldb->pPrev = pNode;		/* New last node = New node */
	LNEXT = NULL;				/* But next is NULL! */
 	} else {
 	xs_sldb = pNode;			/* First node ... */
 	LPREV = pNode;				/* ... it's also last */
 	LNEXT = NULL;				/* But next is NULL! */
 	}
}


/*
 * Compare two given MD5-hashes.
 * Return: 0 if equal
 *         negative if testHash1 < testHash2
 *         positive if testHash1 > testHash2
 */
gint xs_sldb_cmphash(t_xs_md5hash testHash1, t_xs_md5hash testHash2)
{
 register gint i, res = 0;
  
 /* Compute difference of hashes */
 for (i = 0; (i < XS_MD5HASH_LENGTH) && (!res); i++)
 	res = (testHash1[i] - testHash2[i]);

 return res;
}


/*
 * Get song length from database
 */
t_xs_sldb_node * xs_sldb_get(t_xs_md5hash pHash)
{
 gint iStartNode, iEndNode, iQNode, r, i;
 gboolean iFound;
 t_xs_sldb_node *pResult;

 /* Check the database pointers */
 if (!xs_sldb || !xs_sldbi)
 	return NULL;
 	
 /* Look-up via index using binary search */
 pResult = NULL;
 iStartNode = 0;
 iEndNode = (xs_sldbn - 1);
 iQNode = (iEndNode / 2);
 iFound = FALSE;

 while ((!iFound) && ((iEndNode - iStartNode) > 128))
 	{
 	r = xs_sldb_cmphash(pHash, xs_sldbi[iQNode]->md5Hash);
 	if (r < 0)
 		{
 		/* Hash was in the <- LEFT side */
 		iEndNode = iQNode;
 		iQNode = iStartNode + ((iEndNode - iStartNode) / 2);
 		} else
 	if (r > 0)
 		{
 		/* Hash was in the RIGHT -> side */
 		iStartNode = iQNode;
 		iQNode = iStartNode + ((iEndNode - iStartNode) / 2);
 		} else
 		iFound = TRUE;
 	}

 /* If not found already */
 if (!iFound)
 	{
	/* Search the are linearly */
	iFound = FALSE;
	i = iStartNode;
	while ((i <= iEndNode) && (!iFound))
		{
		if (xs_sldb_cmphash(pHash, xs_sldbi[i]->md5Hash) == 0)
			iFound = TRUE;
			else
			i++;
		}
	
	/* Check the result */
	if (iFound)
		pResult = xs_sldbi[i];

 	} else {
	/* Found via binary search */
 	pResult = xs_sldbi[iQNode];
	}


 return pResult;
}


/*
 * Parses a time-entry in SLDB format
 */
gint32 xs_gettime(gchar *pcStr, int *piPos)
{
 gint32 iResult, iTemp;

 /* Check if it starts with a digit */ 
 if (isdigit(pcStr[*piPos]))
	{
	/* Get minutes-field */
	iResult = 0;
	while (isdigit(pcStr[*piPos]))
		iResult = (iResult * 10) + (pcStr[(*piPos)++] - '0');
		
	iResult *= 60;

	/* Check the field separator char */
	if (pcStr[*piPos] == ':')
		{
		/* Get seconds-field */
		(*piPos)++;
		iTemp = 0;
		while (isdigit(pcStr[*piPos]))
			iTemp = (iTemp * 10) + (pcStr[(*piPos)++] - '0');

		iResult += iTemp;
		} else
		iResult = -2;
	} else
	iResult = -1;

 /* Ignore and skip the possible attributes */
 while (pcStr[*piPos] && !isspace(pcStr[*piPos])) (*piPos)++;
 
 return iResult;
}


/*
 * Read database to memory
 */
gint xs_sldb_read(gchar *dbFilename)
{
 FILE *inFile;
 gchar inLine[XS_BUFSIZE];
 guint lineNum, linePos;
 gboolean iOK;
 t_xs_sldb_node *tmpNode;
 
 /* Try to open the file */
 if ((inFile = fopen(dbFilename, "ra")) == NULL)
 	{
 	XSERR("Could not open SongLengthDB '%s'\n", dbFilename);
 	return -1;
 	}
 
 /* Read and parse the data */
 lineNum = 0;

 while (!feof(inFile))
 {
 fgets(inLine, sizeof(inLine), inFile);
 lineNum++;

 /* Check if it is datafield */
 if (isxdigit(inLine[0])) 
 {
 /* Check the length of the hash */
 linePos = 0;
 while (isxdigit(inLine[linePos])) linePos++;

 if (linePos != XS_MD5HASH_LENGTH_CH)
	{
	XSERR("Invalid hash in SongLengthDB file '%s' line #%d!\n",
	dbFilename, lineNum);
	} else {
	/* Allocate new node */
	if ((tmpNode = xs_sldb_node_new()) == NULL)
		{
		XSERR("Error allocating new node. Fatal error.\n");
		exit(5);
		}
			
	/* Get hash value */
#if (XS_MD5HASH_LENGTH != 16)
#error Mismatch in hashcode length. Fix here.
#endif
	sscanf(&inLine[0], "%2x%2x%2x%2x%2x%2x%2x%2x%2x%2x%2x%2x%2x%2x%2x%2x",
	(guint *) &(tmpNode->md5Hash[0]), (guint *) &(tmpNode->md5Hash[1]),
	(guint *) &(tmpNode->md5Hash[2]), (guint *) &(tmpNode->md5Hash[3]),
	(guint *) &(tmpNode->md5Hash[4]), (guint *) &(tmpNode->md5Hash[5]),
	(guint *) &(tmpNode->md5Hash[6]), (guint *) &(tmpNode->md5Hash[7]),
	(guint *) &(tmpNode->md5Hash[8]), (guint *) &(tmpNode->md5Hash[9]),
	(guint *) &(tmpNode->md5Hash[10]), (guint *) &(tmpNode->md5Hash[11]),
	(guint *) &(tmpNode->md5Hash[12]), (guint *) &(tmpNode->md5Hash[13]),
	(guint *) &(tmpNode->md5Hash[14]), (guint *) &(tmpNode->md5Hash[15]));

	/* Get playtimes */
	if (inLine[linePos] != 0)
	{
	if (inLine[linePos] != '=')
		{
		XSERR("'=' expected in SongLengthDB file '%s' line #%d, column #%d\n",
		dbFilename, lineNum, linePos);

		xs_sldb_node_free(tmpNode);
		} else {
		/* First playtime is after '=' */
		linePos++;
		iOK = TRUE;

		while ((linePos < strlen(inLine)) && iOK)
			{
			xs_findnext(inLine, &linePos);

			if (tmpNode->nLengths < XS_STIL_MAXENTRY)
				{
				tmpNode->sLengths[tmpNode->nLengths] =
				xs_gettime(inLine, &linePos);
				tmpNode->nLengths++;
				} else
				iOK = FALSE;
			}

		/* Add an node to db in memory */
		if (iOK)
			xs_sldb_node_insert(tmpNode);
			else
			xs_sldb_node_free(tmpNode);
		}
	}
	}
 } else
 if ((inLine[0] != ';') && (inLine[0] != '['))
 	{
	XSERR("Invalid line in SongLengthDB file '%s' line #%d\n", 
	dbFilename, lineNum);
	}

 } /* while */

 /* Close the file */  
 fclose(inFile);
 
 return 0;
}


/*
 * Compare two nodes' hashes
 */
gint xs_sldb_cmp(const void *pNode1, const void *pNode2)
{
 /* We assume here that we never ever get NULL-pointers or similar */
 return xs_sldb_cmphash((*(t_xs_sldb_node **) pNode1)->md5Hash,
                      (*(t_xs_sldb_node **) pNode2)->md5Hash);
}


/*
 * Initialize the song-length system
 */
gint xs_songlen_init(void)
{
 t_xs_sldb_node *pCurr;
 gint i;

XSDEBUG("sldb_init()\n");
 
 /* Read the database */
 if (!xs_cfg.songlenDBPath)
 	return -10;

 if (xs_sldb_read(xs_cfg.songlenDBPath) < 0)
 	return -9;

XSDEBUG("indexing...\n");

 /* Get size of db */
 pCurr = xs_sldb;
 xs_sldbn = 0;
 while (pCurr)
 	{
 	xs_sldbn++;
 	pCurr = pCurr->pNext;
 	}

 /* Check number of nodes */
 if (xs_sldbn > 0)
	{
	/* Allocate memory for index-table */
	xs_sldbi = (t_xs_sldb_node **) g_malloc(sizeof(t_xs_sldb_node *) * xs_sldbn);
	if (!xs_sldbi) return -6;

	/* Get node-pointers to table */
	i = 0;
	pCurr = xs_sldb;
	while (pCurr)
		{
		xs_sldbi[i++] = pCurr;
		pCurr = pCurr->pNext;
		}

	/* Sort the indexes */
	qsort(xs_sldbi, xs_sldbn, sizeof(t_xs_sldb_node *), xs_sldb_cmp);
	}

 /* OK */
XSDEBUG("init ok.\n");
 return 0;
}


/*
 * Close song-length system
 */
void xs_songlen_close(void)
{
 t_xs_sldb_node *pCurr, *pNext;
 
 /* Free the memory allocated for database */
XSDEBUG("sldb_close()\n");
 pCurr = xs_sldb;
 while (pCurr)
 	{
 	pNext = pCurr->pNext;
	xs_sldb_node_free(pCurr);
 	pCurr = pNext;
 	}

 xs_sldb = NULL;
 
 /* Free memory allocated for indexes */
 if (xs_sldbi)
 	{
 	g_free(xs_sldbi);
 	xs_sldbi = NULL;
 	}
}


/*
 * Compute md5hash of given SID-file
 */
typedef struct {
	gchar	magicID[4];	/* "PSID" magic identifier */
	guint16	version,	/* Version number */
		dataOffset,	/* Start of actual c64 data in file */
		loadAddress,	/* Loading address */
		initAddress,	/* Initialization address */
		playAddress,	/* Play one frame */
		nSongs,		/* Number of subsongs */
		startSong;	/* Default starting song */
	guint32	speed;		/* Speed */
	gchar	sidName[32];	/* Descriptive text-fields, ASCIIZ */
	gchar	sidAuthor[32];
	gchar	sidCopyright[32];
} t_xs_psidv1_header;


typedef struct {
	guint16	flags;		/* Flags */
	guint8	startPage,
		pageLength;
	guint16 reserved;
} t_xs_psidv2_header;


guint16 rd_be16(FILE *f)
{
 return (((guint16) fgetc(f)) << 8) |
 	 ((guint16) fgetc(f));
}


guint32 rd_be32(FILE *f)
{
 return (((guint32) fgetc(f)) << 24) |
 	(((guint32) fgetc(f)) << 16) |
 	(((guint32) fgetc(f)) << 8) |
 	 ((guint32) fgetc(f));
}


gint rd_str(FILE *f, gchar *s, gint l)
{
 return fread(s, sizeof(gchar), l, f);
}

#define MAX_MEMBUF (80*1024)

gint xs_get_sid_hash(gchar *pcFilename, t_xs_md5hash hash)
{
 FILE *inFile;
 t_xs_md5state inState;
 t_xs_psidv1_header psidH;
 t_xs_psidv2_header psidH2;
 guint8 songData[MAX_MEMBUF+1], ib8[2], i8;
 gint iIndex, iRes;


 /* Try to open the file */
 if ((inFile = fopen(pcFilename, "rb")) == NULL)
 	return -1;

 /* Read PSID header in */
 rd_str(inFile, psidH.magicID, sizeof(psidH.magicID));
 if ((psidH.magicID[0] != 'P') || (psidH.magicID[1] != 'S') || 
     (psidH.magicID[2] != 'I') || (psidH.magicID[3] != 'D'))
     return -2;
 
 psidH.version		= rd_be16(inFile);
 psidH.dataOffset	= rd_be16(inFile);
 psidH.loadAddress	= rd_be16(inFile);
 psidH.initAddress	= rd_be16(inFile);
 psidH.playAddress	= rd_be16(inFile);
 psidH.nSongs		= rd_be16(inFile);
 psidH.startSong	= rd_be16(inFile);
 psidH.speed		= rd_be32(inFile);

 rd_str(inFile, psidH.sidName, sizeof(psidH.sidName));
 rd_str(inFile, psidH.sidAuthor, sizeof(psidH.sidAuthor));
 rd_str(inFile, psidH.sidCopyright, sizeof(psidH.sidCopyright));
 
 /* Check if we need to load PSIDv2NG header ... */
 if (psidH.version == 2)
 	{
 	/* Yes, we need to */
 	psidH2.flags		= rd_be16(inFile);
 	psidH2.startPage	= fgetc(inFile);
 	psidH2.pageLength	= fgetc(inFile);
 	psidH2.reserved		= rd_be16(inFile);
 	}

 /* Read data to buffer */
 iRes = fread(songData, sizeof(guint8), MAX_MEMBUF, inFile);
 fclose(inFile);

 /* Initialize and start MD5-hash calculation */
 xs_md5_init(&inState);
 if (psidH.loadAddress == 0)
 	/* COULD SOMEONE EXPLAIN WHY DO WE NEED THIS +2 STRIP???? */
	xs_md5_append(&inState, &songData[2], iRes-2);
	else
	xs_md5_append(&inState, &songData, iRes);

 /* Append header data to hash */
#define XSADDHASH(QDATAB) { ib8[0] = (QDATAB & 0xff); ib8[1] = (QDATAB >> 8); xs_md5_append(&inState, (guint8 *) &ib8, sizeof(ib8)); }
 
 XSADDHASH(psidH.initAddress)
 XSADDHASH(psidH.playAddress)
 XSADDHASH(psidH.nSongs)

#undef XSADDHASH

 /* Append song speed data to hash */ 
 i8 = 0;
 for (iIndex = 0; (iIndex < psidH.nSongs) && (iIndex < 32); iIndex++)
 	{
 	i8 = (psidH.speed & (1 << iIndex)) ? 60 : 0;
 	xs_md5_append(&inState, &i8, sizeof(i8));
 	}
 
 /* Rest of songs (more than 32) */
 for (iIndex = 32; iIndex < psidH.nSongs; iIndex++)
 	{
 	xs_md5_append(&inState, &i8, sizeof(i8));
 	}


 /* PSIDv2NG specific */ 
 if (psidH.version == 2)
 	{
 	/* SEE SIDPLAY HEADERS FOR INFO */
 	i8 = (psidH2.flags >> 2) & 3;
	if (i8 == 2)
		xs_md5_append(&inState, &i8, sizeof(i8));
	}

 /* Calculate the hash */
 xs_md5_finish(&inState, hash);
 
 return 0; 
}


/*
 * Get song lengths
 */
t_xs_sldb_node * xs_songlen_get(gchar *pcFilename)
{
 t_xs_sldb_node *pResult;
 t_xs_md5hash dbHash;

 pResult = NULL;
 
 if (xs_cfg.songlenDBEnable)
	{
	/* Get the hash and then look up from db */
	if (xs_get_sid_hash(pcFilename, dbHash) == 0)
		{
		pResult = xs_sldb_get(dbHash);
		}
	}

 return pResult;
}
