/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * filename: xerror.c                                                      *
 *                                                                         *
 * UTIL C-source: Medical Image Conversion Utility                         *
 *                                                                         *
 * purpose      : error display routines                                   *
 *                                                                         *
 * project      : (X)MedCon by Erik Nolf                                   *
 *                                                                         *
 * Note         : This code gets linked in (X)MedCon library with X-support*
 *                                                                         *
 * Functions    : XMdcFatalErrorKill()   - Quit program with fatal error   *
 *                XMdcDisplayDialog()    - Display a dialog window         *
 *                XMdcDisplayWarn()      - Display a  warning              *
 *                XMdcDisplayMesg()      - Display a  message              *
 *                XMdcDisplayErr()       - Display an error                *
 *                XMdcDisplayFatalErr()  - Display a  fatal error          *
 *                XMdcLogHandler()       - The Glib log output handler     *
 *                XMdcShowErrorConsole() - Show the error console          *
 *                                                                         *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* $Id$
 */

/*
   Copyright (C) 1997-2018 by Erik Nolf

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
   Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Place - Suite 330, Boston, MA 02111-1307, USA.  */

/****************************************************************************
                              H E A D E R S
****************************************************************************/

/* gtktext was replaced by gtktextview in gtk-2.0, this enables it in 2.0 */
#define GTK_ENABLE_BROKEN

#include <stdio.h>
#include <time.h>
#include <gtk/gtk.h>

#include "xmedcon.h"

/****************************************************************************
                              D E F I N E S
****************************************************************************/

static GtkWidget *wconsole = NULL;
static GtkWidget *wlogs = NULL;

/****************************************************************************
                            F U N C T I O N S
****************************************************************************/

void XMdcFatalErrorKill(GtkWidget *button, int *code)
{
  gtk_exit(-*code);
}

void *XMdcDisplayDialog(int code, char *windowtitle, char *info)
{
  GtkWidget *dialog;
  GtkWidget *label;
  GtkWidget *button;

  /* gdk_beep(); */

  dialog =  gtk_dialog_new(); 

  gtk_container_set_border_width(GTK_CONTAINER(GTK_DIALOG(dialog)->action_area),0);

  if (code != MDC_OK) {
    gtk_signal_connect(GTK_OBJECT(dialog), "destroy",
                       GTK_SIGNAL_FUNC(XMdcFatalErrorKill), &code);
  }else{
    gtk_signal_connect(GTK_OBJECT(dialog),"destroy",
                       GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);
  }
  /* gtk_widget_set_uposition(dialog,100,100); */
  gtk_window_position(GTK_WINDOW(dialog), GTK_WIN_POS_MOUSE);
  gtk_window_set_title(GTK_WINDOW(dialog), windowtitle);
  gtk_container_set_border_width(GTK_CONTAINER(dialog), 0);

  label = gtk_label_new(info);
  gtk_misc_set_padding(GTK_MISC(label), 30, 5);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),
                     label, TRUE, TRUE, 0);
  gtk_widget_show(label);

  button = gtk_button_new_with_label("OK");
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),
                     button, TRUE, TRUE, 0);

  if (code != MDC_OK) {
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
                       GTK_SIGNAL_FUNC(XMdcFatalErrorKill), &code);
  }else{
    gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                              GTK_SIGNAL_FUNC(gtk_widget_destroy),
                              GTK_OBJECT(dialog));
  }

  gtk_widget_show(button);

  gtk_widget_show_now(dialog);

  XMdcUpdateDrawing();

  return(dialog);
}

void *XMdcDisplayWarn(char *fmt, ...)
{
  va_list args;

  va_start(args, fmt);
  vsprintf(errmsg, fmt, args);
  va_end(args);

  return(XMdcDisplayDialog(MDC_OK,"Warning",errmsg));
}


void *XMdcDisplayMesg(char *fmt, ...)
{
  va_list args;

  va_start(args, fmt);
  vsprintf(errmsg, fmt, args);
  va_end(args);
 
  return(XMdcDisplayDialog(MDC_OK,"Message",errmsg));
}


void *XMdcDisplayErr(char *fmt, ...)
{
  va_list args;

  va_start(args, fmt);
  vsprintf(errmsg, fmt, args);
  va_end(args);

  return(XMdcDisplayDialog(MDC_OK,"Error",errmsg));
}

void XMdcDisplayFatalErr(int code, char *fmt, ...)
{
  va_list args;

  va_start(args, fmt);
  vsprintf(errmsg, fmt, args);
  va_end(args);

  XMdcDisplayDialog(code, "Fatal Error", errmsg);

}

void XMdcLogHandler(const gchar *domain, GLogLevelFlags level,
                    const gchar *message, gpointer user_data)
{
  time_t t;
  char *logmsg, timestr[32];

  time(&t);
  strftime(timestr,32,"%b %d %H:%M:%S",localtime(&t));

  switch (level) {
    case G_LOG_LEVEL_DEBUG:
      logmsg = g_strdup_printf("DEBUG   **: %s\n %s\n", timestr, message);
      break;
    case G_LOG_LEVEL_MESSAGE:
      logmsg = g_strdup_printf("MESSAGE **: %s\n %s\n", timestr, message);
      break;
    case G_LOG_LEVEL_WARNING:
      logmsg = g_strdup_printf("WARNING **: %s\n %s\n", timestr, message);
      break;
    case G_LOG_LEVEL_ERROR:
      logmsg = g_strdup_printf("ERROR   **: %s\n %s\n", timestr, message);
      break;
    default:
      logmsg = g_strdup_printf("REMARK  **: %s\n %s\n", timestr, message);
  }

  if (logmsg == NULL) return;

  gtk_text_insert(GTK_TEXT(wlogs),NULL,NULL,NULL,logmsg,-1);
  gtk_text_thaw(GTK_TEXT(wlogs));
  g_free(logmsg);

}


void XMdcCreateLogConsole(void)
{
  GtkWidget *box1, *box2;
  GtkWidget *table;
  GtkWidget *button;
  GtkWidget *separator;
  GtkWidget *vscrollbar;
  char *str;

  if (wconsole != NULL) return;

  wconsole = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_widget_set_usize(wconsole, 506, 100);

  gtk_signal_connect(GTK_OBJECT(wconsole), "destroy",
                     GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);
    gtk_signal_connect(GTK_OBJECT(wconsole), "delete_event",
                     GTK_SIGNAL_FUNC(XMdcHandlerToHide), NULL);

  str = g_strdup_printf("%s Console Logs",MDC_PRGR);
  if (str != NULL) {
    gtk_window_set_title(GTK_WINDOW(wconsole),str);
    g_free(str);
  }
  
  box1 = gtk_vbox_new(FALSE,0);
  gtk_container_add(GTK_CONTAINER(wconsole), box1);
  gtk_widget_show(box1);

  box2 = gtk_vbox_new(FALSE,0);
  gtk_container_set_border_width(GTK_CONTAINER(box2),0);
  gtk_box_pack_start(GTK_BOX(box1),box2,TRUE,TRUE,0);
  gtk_widget_show(box2);

  table = gtk_table_new(2, 2, FALSE);
  gtk_table_set_row_spacing(GTK_TABLE(table), 0, 2);
  gtk_table_set_col_spacing(GTK_TABLE(table), 0, 2);
  gtk_box_pack_start(GTK_BOX(box2),table,TRUE,TRUE,0);
  gtk_widget_show(table);

  wlogs = gtk_text_new(NULL,NULL);
  gtk_text_set_editable(GTK_TEXT(wlogs),FALSE);
  gtk_text_set_word_wrap(GTK_TEXT(wlogs), TRUE);
  gtk_table_attach(GTK_TABLE(table),wlogs, 0, 1, 0, 1,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                   GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
  gtk_widget_show(wlogs);

  vscrollbar = gtk_vscrollbar_new(GTK_TEXT(wlogs)->vadj);
  gtk_table_attach(GTK_TABLE(table), vscrollbar, 1, 2, 0, 1,
                   GTK_FILL, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
  gtk_widget_show(vscrollbar);

  gtk_widget_show(wlogs);

  separator = gtk_hseparator_new();
  gtk_box_pack_start(GTK_BOX(box1),separator,FALSE,TRUE,0);
  gtk_widget_show(separator);

  box2 = gtk_hbox_new(FALSE,0);
  gtk_container_set_border_width(GTK_CONTAINER(box2),0);
  gtk_box_pack_start(GTK_BOX(box1),box2,FALSE,FALSE,0);
  gtk_widget_show(box2);


  button = gtk_button_new_with_label("Clear");
  gtk_signal_connect(GTK_OBJECT(button),"clicked",
                     GTK_SIGNAL_FUNC(XMdcClearLogConsole),NULL);
  gtk_box_pack_start(GTK_BOX(box2),button,TRUE,TRUE,2);
  gtk_widget_show(button);

  button = gtk_button_new_with_label("Close");
  gtk_signal_connect_object(GTK_OBJECT(button),"clicked",
                            GTK_SIGNAL_FUNC(gtk_widget_hide),
                            GTK_OBJECT(wconsole));
  gtk_box_pack_start(GTK_BOX(box2),button,TRUE,TRUE,2);
  gtk_widget_show(button);

  g_log_set_handler(MDC_PRGR,G_LOG_LEVEL_MASK,XMdcLogHandler, NULL);

}

void XMdcShowLogConsole(void)
{
  if (wconsole == NULL) XMdcCreateLogConsole();

  gtk_widget_show(wconsole);

}

void XMdcClearLogConsole(void)
{
  gtk_editable_delete_text(
     GTK_EDITABLE(wlogs), 0, (gint)gtk_text_get_length(GTK_TEXT(wlogs)));
}
