/*
  Copyright (C) 2000-2007

  Code contributed by Greg Collecutt, Joseph Hope and Paul Cochrane

  This file is part of xmds.

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/*
  $Id: xmds_integrate_ark89.cc 1630 2007-12-21 18:51:09Z paultcochrane $
*/

/*! @file xmds_integrate_ark89.cc
  @brief Integrate element parsing classes and methods; fourth order Runge-Kutta

  More detailed explanation...
*/

#include <xmds_common.h>
#include <xmds_integrate_ark89.h>
#include <xmds_simulation.h>
#include <xmds_vector.h>
#include <cstdlib>

// **************************************************************************
// **************************************************************************
//                              xmdsIntegrateARK89 public
// **************************************************************************
// **************************************************************************

extern bool debugFlag;

long nxmdsIntegrateARK89s=0;   //!< Number of xmds integrate ARK89 objects

// **************************************************************************
xmdsIntegrateARK89::xmdsIntegrateARK89(
                                       const xmdsSimulation *const yourSimulation,
                                       const bool& yourVerboseMode) :
  xmdsIntegrate(yourSimulation, yourVerboseMode, false, false) {
  if (debugFlag) {
    nxmdsIntegrateARK89s++;
    printf("xmdsIntegrateARK89::xmdsIntegrateARK89\n");
    printf("nxmdsIntegrateARK89s=%li\n", nxmdsIntegrateARK89s);
  }
}

// **************************************************************************
xmdsIntegrateARK89::~xmdsIntegrateARK89() {
  if (debugFlag) {
    nxmdsIntegrateARK89s--;
    printf("xmdsIntegrateARK89::~xmdsIntegrateARK89\n");
    printf("nxmdsIntegrateARK89s=%li\n", nxmdsIntegrateARK89s);
  }
}

// **************************************************************************
void xmdsIntegrateARK89::processElement(
                                        const Element *const yourElement) {
  if (debugFlag) {
    printf("xmdsIntegrateARK89::processElement\n");
  }

  /* if ((simulation()->parameters()->stochastic)&&(!noNoises())) {
     printf("\n");
     printf("WARNING: ARK89 methods may not always yield correct stochastic convergence.\n");
     printf("\n");
     }*/

  if (simulation()->parameters()->noiseKind == "poissonian") {
    throw xmdsException(yourElement,
        "Adaptive methods are currently not capable of adjusting for "
        "Poissonian noise.  We suggest RK9 algorithms.");
  }

  list<XMLString> myXMLStringList;
  list<unsigned long> myULongList;

  // ************************************
  // find 'tolerance'

  getAssignmentStrings(yourElement, "tolerance", 1, 1, myXMLStringList);

  myTolerance=*myXMLStringList.begin();

  if (verbose()) {
    printf("integration tolerance = %s\n", myTolerance.c_str());
  }

  // ************************************
  // find 'cutoff'

  getAssignmentStrings(yourElement, "cutoff", 0, 1, myXMLStringList);


  if (myXMLStringList.size()==1) {
    myCutoff=*myXMLStringList.begin();

    if (verbose()) {
      printf("cutoff  = %s\n", myCutoff.c_str());

    }
  }
  else {
    printf("cutoff defaulting to 1e-3 \n");
    myCutoff="1e-3";
  }

  // ************************************
  // find 'maximum iterations'

  getAssignmentULongs(yourElement, "max_iterations", 0, 1, myULongList);

  if (myULongList.size()==1) {
    myMaxIterations=*myULongList.begin();

    if (myMaxIterations==0)
      throw xmdsException(yourElement, "Maximum Iterations must be >= 1 !");

    if (verbose()) {
      printf("Maximum iterations  = %li\n", myMaxIterations);

    }
  }
  else {
    if (verbose())
      printf("Maximum iterations = infinity \n");
    myMaxIterations=0;
    // this means the feature is disabled
  }

}



// **************************************************************************
// **************************************************************************
//                              xmdsIntegrateARK89 protected
// **************************************************************************
// **************************************************************************
const XMLString* xmdsIntegrateARK89::tolerance() const {
  if (debugFlag) {
    printf("xmdsIntegrateARK89::tolerance\n");
  }

  return &myTolerance;
}

// **************************************************************************
const XMLString* xmdsIntegrateARK89::cutoff() const {
  if (debugFlag) {
    printf("xmdsIntegrateARK89::cutoff\n");
  }

  return &myCutoff;
}

// **************************************************************************
void xmdsIntegrateARK89::writePrototypes(
                                         FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateARK89::writePrototypes\n");
  }

  const xmdsVector* mainVector;

  if (!simulation()->field()->getVector("main", mainVector)) {
    throw xmdsException("Internal error in xmdsIntegrateARK89::writePrototypes: cannot find 'main' vector");
  }

  const char* typeName="";
  if (mainVector->vectorType()==COMPLEX) {
    typeName="complex";
  }
  else if (mainVector->vectorType()==DOUBLE) {
    typeName="double";
  }
  fprintf(outfile, "\n");
  fprintf(outfile, "// integrate (ARK89) prototypes\n");
  fprintf(outfile, "\n");

  fprintf(outfile, "double _segment%li_timestep_error(%s* _checkfield);\n", segmentNumber, typeName);
  fprintf(outfile, "\n");

  fprintf(outfile, "double _segment%li_setup_sampling(bool* _next_sample_flag, unsigned long* _next_sample_counter);\n", segmentNumber);
  fprintf(outfile, "\n");

  fprintf(outfile, "void _segment%li(unsigned long cycle);\n", segmentNumber);
  fprintf(outfile, "\n");

  if (crossVectorNamesList()->size() > 0) {
    fprintf(outfile, "void _segment%li_calculate_cross_field(", segmentNumber);
    if ((simulation()->parameters()->stochastic)&&(!noNoises())) {
      fprintf(outfile, "const double *const _noise_vector");
    }
    fprintf(outfile, ");\n");
    fprintf(outfile, "\n");
  }

}

// **************************************************************************
void xmdsIntegrateARK89::writeRoutines(
                                       FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateARK89::writeRoutines\n");
  }

  writeTimestepErrorRoutine(outfile);
  writeSetupSamplingRoutine(outfile);
  writeMainIntegrateRoutine(outfile);

  if (crossVectorNamesList()->size() > 0) {
    writeCalculateCrossFieldRoutine(outfile);
  }
}

// **************************************************************************
// **************************************************************************
//                              xmdsIntegrateARK89 private
// **************************************************************************
// **************************************************************************

void xmdsIntegrateARK89::writeTimestepErrorRoutine(
                                                   FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateARK89::writeTimestepErrorRoutine\n");
  }

  const char *const fieldName = simulation()->field()->name()->c_str();

  const xmdsVector* mainVector;

  if (!simulation()->field()->getVector("main", mainVector)) {
    throw xmdsException("Internal error in xmdsIntegrateARK89::writeTimestepErrorRoutine: cannot find 'main' vector");
  }

  const char* typeName="";
  if (mainVector->vectorType()==COMPLEX) {
    typeName="complex";
  }
  else if (mainVector->vectorType()==DOUBLE) {
    typeName="double";
  }
  fprintf(outfile, "/* **************************************************/\n");
  fprintf(outfile, "double _segment%li_timestep_error(%s* _checkfield) {\n", segmentNumber, typeName);
  fprintf(outfile, "\n");
  fprintf(outfile, "double _error=1e-24;\n");
  fprintf(outfile, "double _result[_%s_main_ncomponents];\n", fieldName);

  if (simulation()->field()->geometry()->nDims()>0){
    // finds one peak value for each component of field
    fprintf(outfile, "double _peak[_%s_main_ncomponents];\n", fieldName);
    fprintf(outfile, "for (unsigned long _i0=0; _i0<_%s_main_ncomponents; _i0++){\n", fieldName);
    fprintf(outfile, "     _peak[_i0]=0.0;\n");
    fprintf(outfile, "     _result[_i0]=0.0;\n");
    fprintf(outfile, "     }\n");
    fprintf(outfile, "\n");
  }
  fprintf(outfile, "double _temp_error=0.0;\n");
  fprintf(outfile, "double _temp_mod=0.0;\n");
  fprintf(outfile, "\n");
  fprintf(outfile, "unsigned long _%s_main_index_pointer=0;\n", fieldName);
  fprintf(outfile, "\n");

  if (simulation()->field()->geometry()->nDims()>0){
    if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
      fprintf(outfile, "for (unsigned long _i0=0; _i0<unswapped_local_size; _i0++){\n");
    }
    else {
      fprintf(outfile, "for (unsigned long _i0=0; _i0<_%s_size; _i0++){\n", fieldName);
    }
    fprintf(outfile, "   for (unsigned long _i1=0; _i1<_%s_main_ncomponents; _i1++){\n", fieldName);
    if (strcmp(typeName, "complex") == 0) {
      fprintf(outfile, "    _temp_mod=mod2(_%s_main[_%s_main_index_pointer + _i1]);\n", fieldName, fieldName);
    }
    else {
      fprintf(outfile, "    _temp_mod=fabs(_%s_main[_%s_main_index_pointer + _i1]);\n", fieldName, fieldName);
    }
    // this construction will add zero if field(x)<peak and correct peak to its new value if field(x)>peak

      // this construction was _peak[_i1] += 0.5*(_temp_mod - _peak[_i1] + fabs(_temp_mod - _peak[_i1]));
      // It was a bad idea because it leads to NaNs for the peak value when it should be infinite, because it tries to
      // add values to Inf, leading to NaN.
      // The other problem with this construction is that it requires a store on each loop which will be slower than
      // a branch (particularly when that branch is predicted correctly as it will be by modern CPUs).
    fprintf(outfile, "       if (!isfinite(_temp_mod)) _peak[_i1] = INFINITY;\n");
    fprintf(outfile, "       else if (_peak[_i1] < _temp_mod) _peak[_i1] = _temp_mod;\n");
    fprintf(outfile, "    }\n");

    fprintf(outfile, "       _%s_main_index_pointer+=_%s_main_ncomponents;\n", fieldName, fieldName);
    fprintf(outfile, "    }\n");

    fprintf(outfile, "\n");

    if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
      fprintf(outfile, "MPI_Allreduce(&_peak, &_result, _%s_main_ncomponents, MPI_DOUBLE, MPI_MAX, MPI_COMM_WORLD);\n", fieldName);
      fprintf(outfile, "for (unsigned long _i0=0; _i0<_%s_main_ncomponents; _i0++){\n", fieldName);
      fprintf(outfile, "    _peak[_i0]=_result[_i0];\n");
      fprintf(outfile, "    _result[_i0]=0;\n");
      fprintf(outfile, "}\n");
    }

    // now the peak value is multiplied with the cutoff so that it is now in fact the amplitude-threshold for error determination
    fprintf(outfile, "for (unsigned long _i0=0; _i0<_%s_main_ncomponents; _i0++) {\n", fieldName);
    fprintf(outfile, "    if (!isfinite(_peak[_i0]))\n");
    fprintf(outfile, "    // We return the maximum possible error in this case because the timestep must be reduced\n");
    fprintf(outfile, "    // for the error to be calculated correctly\n");
    fprintf(outfile, "        return 1.0;\n");
    fprintf(outfile, "    _peak[_i0]*=%s;\n", cutoff()->c_str());
    fprintf(outfile, "}\n");

    fprintf(outfile, "\n");
    fprintf(outfile, "_%s_main_index_pointer=0;\n", fieldName);

    if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
      fprintf(outfile, "for (long _i0=0; _i0<unswapped_local_size; _i0++){\n");
    }
    else {
      fprintf(outfile, "for (unsigned long _i0=0; _i0<_%s_size; _i0++){\n", fieldName);
    }
    fprintf(outfile, "   for (unsigned long _i1=0; _i1<_%s_main_ncomponents; _i1++)\n", fieldName);

    if (strcmp(typeName, "complex") == 0) {
      fprintf(outfile, "       if (mod2(_%s_main[_%s_main_index_pointer + _i1])>_peak[_i1]){\n", fieldName, fieldName);
      fprintf(outfile, "            _temp_error=mod(_%s_main[_%s_main_index_pointer + _i1]-_checkfield[_%s_main_index_pointer + _i1])/mod(_%s_main[_%s_main_index_pointer + _i1]);\n", fieldName, fieldName, fieldName, fieldName, fieldName);
    }
    else{
      fprintf(outfile, "       if (fabs(_%s_main[_%s_main_index_pointer + _i1])>_peak[_i1]){\n", fieldName, fieldName);
      fprintf(outfile, "            _temp_error=fabs(_%s_main[_%s_main_index_pointer + _i1]-_checkfield[_%s_main_index_pointer + _i1])/fabs(_%s_main[_%s_main_index_pointer + _i1]);\n", fieldName, fieldName, fieldName, fieldName, fieldName);
    }
    fprintf(outfile, "            if (_error < _temp_error) _error = _temp_error;\n");
    fprintf(outfile, "       }\n");
    fprintf(outfile, "       _%s_main_index_pointer+=_%s_main_ncomponents;\n", fieldName, fieldName);
    fprintf(outfile, "    }\n");

  }
  else{// if ndims==0

    fprintf(outfile, "   for (unsigned long _i1=0; _i1<_%s_main_ncomponents; _i1++){\n", fieldName);

    if (strcmp(typeName, "complex") == 0){
      fprintf(outfile, "       _temp_error=mod(_%s_main[_i1]-_checkfield[_i1])/mod(_%s_main[_i1]);\n", fieldName, fieldName);
    }
    else{
      fprintf(outfile, "       _temp_error=fabs(_%s_main[_i1]-_checkfield[_i1])/fabs(_%s_main[_i1]);\n", fieldName, fieldName);
    }
    fprintf(outfile, "       if (_error < _temp_error) _error = _temp_error;\n");
    fprintf(outfile, "    }\n");
  }

  if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile, "MPI_Allreduce(&_error, &_result, 1, MPI_DOUBLE, MPI_MAX, MPI_COMM_WORLD);\n");
  }
  else {
    fprintf(outfile, "_result[0]=_error;\n");
  }
  fprintf(outfile, "return(_result[0]);\n");
  fprintf(outfile, "}\n");
  fprintf(outfile, "\n");

}

// **************************************************************************

void xmdsIntegrateARK89::writeSetupSamplingRoutine(
                                                   FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateARK89::writeSetupSamplingRoutine\n");
  }

  const char *const propDim = simulation()->parameters()->propDimName.c_str();

  fprintf(outfile, "/* **************************************************/\n");
  fprintf(outfile, "double _segment%li_setup_sampling(bool* _next_sample_flag, unsigned long* _next_sample_counter) {\n", segmentNumber);
  fprintf(outfile, "\n");
  // will contain the numbers of all the moment groups that need to be sampled at the next sampling point. The N+1'th entry means "reached end of integration interval".
  fprintf(outfile, "unsigned long _number_next_mg[%li];\n", simulation()->output()->nMomentGroups()+1);
  // number of mg's that are sampled at the next sampling point
  fprintf(outfile, "unsigned long _number_minima=1;\n");

  // The check if n (T_tot/N_samp) = (or<) m (T_tot/M_samp) for two momentgroups, where n, m are _next_sample_counters
  // T_tot is the integration interval and N_samp (M_samp) are the numbers of sampling points, will be replaced
  // by n M_samp = (or<) m N_samp to avoid floating point precision problems.

  fprintf(outfile, "unsigned long _previous_m=1;\n");
  fprintf(outfile, "unsigned long _previous_M=1;\n");
  fprintf(outfile, "\n");
  fprintf(outfile, "double _%s_break_next=(double)%s;\n", propDim, interval()->c_str());
  fprintf(outfile, "_number_next_mg[0]=%li;\n", simulation()->output()->nMomentGroups());
  // initialize all flags to false
  fprintf(outfile, "for (unsigned long _i0=0; _i0<%li; _i0++)\n", simulation()->output()->nMomentGroups()+1);
  fprintf(outfile, "    _next_sample_flag[_i0]=false;\n");
  fprintf(outfile, "\n");

  // check if moment group needs sampling at the same time as another, already discovered sample (or the final time). If so, add this moment group to the to-be-sampled-list. If moment group demands sampling earlier than all previously noted mg's erase all previous ones from list and set the brekpoint-time to this earlier one.
  for (unsigned long i=0; i<simulation()->output()->nMomentGroups(); i++)
    if (samples(i)!=0){
      fprintf(outfile, "if (_next_sample_counter[%li]*_previous_M==_previous_m*%li){\n", i, samples(i));
      fprintf(outfile, "     _number_next_mg[_number_minima]=%li;\n", i);
      fprintf(outfile, "     _number_minima++;\n");
      fprintf(outfile, "     }\n");
      fprintf(outfile, "else if (_next_sample_counter[%li]*_previous_M<_previous_m*%li){\n", i, samples(i));
    double temp_interval;
    if (interval()->asDouble(temp_interval)) {  // If the user specified a number, than we can compute some stuff now
        fprintf(outfile, "    _%s_break_next=_next_sample_counter[%li]*%.23e;\n", propDim, i, temp_interval/samples(i));
    }
    else {  // If the user specified something else (probably a const double in <globals>), then we need to let the compiler do it or do it at run time...
    fprintf(outfile, "    _%s_break_next=_next_sample_counter[%li]*%s/%lu;\n", propDim, i, interval()->c_str(), samples(i));
    }
      fprintf(outfile, "    _number_minima=1;\n");
      fprintf(outfile, "    _number_next_mg[0]=%li;\n", i);
      fprintf(outfile, "    _previous_M=%li;\n", samples(i));
      fprintf(outfile, "    _previous_m=_next_sample_counter[%li];\n", i);
      fprintf(outfile, "     }\n");
    }
  fprintf(outfile, "\n");
  //Values of _number_next_mg until _number_minima contain now the complete list of mg's that need to be sampled at the next breakpoint. Set their flags to true.
  fprintf(outfile, "for (unsigned long _i0=0; _i0<_number_minima; _i0++)\n");
  fprintf(outfile, "   _next_sample_flag[_number_next_mg[_i0]]=true;\n");
  fprintf(outfile, "\n");
  fprintf(outfile, "return(_%s_break_next);\n", propDim);
  fprintf(outfile, "}\n");
  fprintf(outfile, "\n");

}


// **************************************************************************
void xmdsIntegrateARK89::writeMainIntegrateRoutine(FILE *const outfile) const
{
  if (debugFlag) {
    printf("xmdsIntegrateARK89::writeMainIntegrateRoutine\n");
  }

  const char *const fieldName = simulation()->field()->name()->c_str();
  const char *const propDim = simulation()->parameters()->propDimName.c_str();

  const xmdsVector* mainVector;

  if (!simulation()->field()->getVector("main", mainVector)) {
    throw xmdsException("Internal error in xmdsIntegrateARK89::writeMainIntegrateRoutine: cannot find 'main' vector");
  }

  const char* typeName="";
  if (mainVector->vectorType()==COMPLEX) {
    typeName="complex";
  }
  else if (mainVector->vectorType()==DOUBLE) {
    typeName="double";
  }

  bool max_iter=true;

  if (myMaxIterations == 0) {
    max_iter = false;
  }

  fprintf(outfile, "/* ******************************************** */\n");
  fprintf(outfile, "void _segment%li(unsigned long cycle) {\n", segmentNumber);
  fprintf(outfile, "\n");
  if ((simulation()->parameters()->usempi)&!(simulation()->parameters()->stochastic)){
    fprintf(outfile, "%s *_akafield_%s_main = (%s*)fftw_malloc(sizeof(%s)*total_local_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName);
    fprintf(outfile, "%s *_akbfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*total_local_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName);
    fprintf(outfile, "%s *_akcfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*total_local_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName);
    fprintf(outfile, "%s *_akdfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*total_local_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName);
    fprintf(outfile, "%s *_akefield_%s_main = (%s*)fftw_malloc(sizeof(%s)*total_local_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName);
    fprintf(outfile, "%s *_akffield_%s_main = (%s*)fftw_malloc(sizeof(%s)*total_local_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName);
    fprintf(outfile, "%s *_akgfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*total_local_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName);
    fprintf(outfile, "%s *_akhfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*total_local_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName);
    fprintf(outfile, "%s *_akifield_%s_main = (%s*)fftw_malloc(sizeof(%s)*total_local_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName);
    fprintf(outfile, "%s *_akjfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*total_local_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName);
    fprintf(outfile, "\n");
    fprintf(outfile, "%s *_%s_init = (%s*)fftw_malloc(sizeof(%s)*total_local_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName);
  }
  else {
    fprintf(outfile, "%s *_akafield_%s_main = (%s*)fftw_malloc(sizeof(%s)*_%s_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName, fieldName);
    fprintf(outfile, "%s *_akbfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*_%s_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName, fieldName);
    fprintf(outfile, "%s *_akcfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*_%s_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName, fieldName);
    fprintf(outfile, "%s *_akdfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*_%s_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName, fieldName);
    fprintf(outfile, "%s *_akefield_%s_main = (%s*)fftw_malloc(sizeof(%s)*_%s_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName, fieldName);
    fprintf(outfile, "%s *_akffield_%s_main = (%s*)fftw_malloc(sizeof(%s)*_%s_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName, fieldName);
    fprintf(outfile, "%s *_akgfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*_%s_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName, fieldName);
    fprintf(outfile, "%s *_akhfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*_%s_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName, fieldName);
    fprintf(outfile, "%s *_akifield_%s_main = (%s*)fftw_malloc(sizeof(%s)*_%s_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName, fieldName);
    fprintf(outfile, "%s *_akjfield_%s_main = (%s*)fftw_malloc(sizeof(%s)*_%s_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName, fieldName);
    fprintf(outfile, "\n");
    fprintf(outfile, "%s *_%s_init = (%s*)fftw_malloc(sizeof(%s)*_%s_size*_%s_main_ncomponents);\n", typeName, fieldName, typeName, typeName, fieldName, fieldName);

  }

  fprintf(outfile, "\n");
  fprintf(outfile, "// Runge Kutta method constants \n");
  fprintf(outfile, "double _a_raw[16];\n");
  fprintf(outfile, "double _a[16];\n");
  fprintf(outfile, "double _b[16][16];\n");
  fprintf(outfile, "double _c[16];\n");
  fprintf(outfile, "double _cs[16];\n");
  fprintf(outfile, "\n");

  fprintf(outfile, "for (unsigned long _i0=0; _i0<16; _i0++) {\n");
  fprintf(outfile, "  _a_raw[_i0]=0.0;\n");
  fprintf(outfile, "  _c[_i0]=0.0;\n");
  fprintf(outfile, "  for (unsigned long _i1=0; _i1<16; _i1++)\n");
  fprintf(outfile, "   _b[_i0][_i1]=0.0;\n");
  fprintf(outfile, "}\n");
  fprintf(outfile, "\n");

  fprintf(outfile, "_a_raw[1] = 0.02173913043478260869565217391304347;\n");
  fprintf(outfile, "_a_raw[2] = 0.09629581047800066670113001679819925;\n");
  fprintf(outfile, "_a_raw[3] = 0.14444371571700100005169502519729888;\n");
  fprintf(outfile, "_a_raw[4] = 0.52205882352941176470588235294117647;\n");
  fprintf(outfile, "_a_raw[5] = 0.22842443612863469578031459099794265;\n");
  fprintf(outfile, "_a_raw[6] = 0.54360353589933733219171338103002937;\n");
  fprintf(outfile, "_a_raw[7] = 0.64335664335664335664335664335664335;\n");
  fprintf(outfile, "_a_raw[8] = 0.48251748251748251748251748251748251;\n");
  fprintf(outfile, "_a_raw[9] = 0.06818181818181818181818181818181818;\n");
  fprintf(outfile, "_a_raw[10] = 0.25060827250608272506082725060827250;\n");
  fprintf(outfile, "_a_raw[11] = 0.66736715965600568968278165443304378;\n");
  fprintf(outfile, "_a_raw[12] = 0.85507246376811594202898550724637681;\n");
  fprintf(outfile, "_a_raw[13] = 0.89795918367346938775510204081632653;\n");
  fprintf(outfile, "_a_raw[14] = 1.0;\n");
  fprintf(outfile, "_a_raw[15] = 1.0;\n");
  fprintf(outfile, "\n");

  fprintf(outfile, "_a[0]=0.0;\n");
  fprintf(outfile, "for (unsigned long _i0=1; _i0<16; _i0++)\n");
  fprintf(outfile, "  _a[_i0]=_a_raw[_i0]-_a_raw[_i0-1];\n");
  fprintf(outfile, "\n");

  fprintf(outfile, "_b[1][0] = 1.0/46.0;\n");
  fprintf(outfile, "_b[2][0] =-0.11698050118114486205818241524969622;\n");
  fprintf(outfile, "_b[2][1] = 0.21327631165914552875931243204789548;\n");
  fprintf(outfile, "_b[3][0] = 0.03611092892925025001292375629932472;\n");
  fprintf(outfile, "_b[3][2] = 0.10833278678775075003877126889797416;\n");
  fprintf(outfile, "_b[4][0] = 1.57329743908138605107331820072051125;\n");
  fprintf(outfile, "_b[4][2] =-5.98400943754042002888532938159655553;\n");
  fprintf(outfile, "_b[4][3] = 4.93277082198844574251789353381722074;\n");
  fprintf(outfile, "_b[5][0] = 0.05052046351120380909008334360006234;\n");
  fprintf(outfile, "_b[5][3] = 0.17686653884807108146683657390397612;\n");
  fprintf(outfile, "_b[5][4] = 0.00103743376935980522339467349390418;\n");
  fprintf(outfile, "_b[6][0] = 0.10543148021953768958529340893598138;\n");
  fprintf(outfile, "_b[6][3] =-0.16042415162569842979496486916719383;\n");
  fprintf(outfile, "_b[6][4] = 0.11643956912829316045688724281285250;\n");
  fprintf(outfile, "_b[6][5] = 0.48215663817720491194449759844838932;\n");
  fprintf(outfile, "_b[7][0] = 0.07148407148407148407148407148407148;\n");
  fprintf(outfile, "_b[7][5] = 0.32971116090443908023196389566296464;\n");
  fprintf(outfile, "_b[7][6] = 0.24216141096813279233990867620960722;\n");
  fprintf(outfile, "_b[8][0] = 0.07162368881118881118881118881118881;\n");
  fprintf(outfile, "_b[8][5] = 0.32859867301674234161492268975519694;\n");
  fprintf(outfile, "_b[8][6] = 0.11622213117906185418927311444060725;\n");
  fprintf(outfile, "_b[8][7] =-0.03392701048951048951048951048951048;\n");
  fprintf(outfile, "_b[9][0] = 0.04861540768024729180628870095388582;\n");
  fprintf(outfile, "_b[9][5] = 0.03998502200331629058445317782406268;\n");
  fprintf(outfile, "_b[9][6] = 0.10715724786209388876739304914053506;\n");
  fprintf(outfile, "_b[9][7] =-0.02177735985419485163815426357369818;\n");
  fprintf(outfile, "_b[9][8] =-0.10579849950964443770179884616296721;\n");
  fprintf(outfile, "_b[10][0] =-0.02540141041535143673515871979014924;\n");
  fprintf(outfile, "_b[10][5] = 1.0/30.0;\n");
  fprintf(outfile, "_b[10][6] =-0.16404854760069182073503553020238782;\n");
  fprintf(outfile, "_b[10][7] = 0.03410548898794737788891414566528526;\n");
  fprintf(outfile, "_b[10][8] = 0.15836825014108792658008718465091487;\n");
  fprintf(outfile, "_b[10][9] = 0.21425115805975734472868683695127609;\n");
  fprintf(outfile, "_b[11][0] = 0.00584833331460742801095934302256470;\n");
  fprintf(outfile, "_b[11][5] =-0.53954170547283522916525526480339109;\n");
  fprintf(outfile, "_b[11][6] = 0.20128430845560909506500331018201158;\n");
  fprintf(outfile, "_b[11][7] = 0.04347222773254789483240207937678906;\n");
  fprintf(outfile, "_b[11][8] =-0.00402998571475307250775349983910179;\n");
  fprintf(outfile, "_b[11][9] = 0.16541535721570612771420482097898952;\n");
  fprintf(outfile, "_b[11][10] = 0.79491862412512344573322086551518180;\n");
  fprintf(outfile, "_b[12][0] =-0.39964965968794892497157706711861448;\n");
  fprintf(outfile, "_b[12][5] =-3.79096577568393158554742638116249372;\n");
  fprintf(outfile, "_b[12][6] =-0.40349325653530103387515807815498044;\n");
  fprintf(outfile, "_b[12][7] =-2.82463879530435263378049668286220715;\n");
  fprintf(outfile, "_b[12][8] = 1.04226892772185985533374283289821416;\n");
  fprintf(outfile, "_b[12][9] = 1.12510956420436603974237036536924078;\n");
  fprintf(outfile, "_b[12][10] = 3.32746188718986816186934832571938138;\n");
  fprintf(outfile, "_b[12][11] = 2.77897957186355606325818219255783627;\n");
  fprintf(outfile, "_b[13][0] = 0.39545306350085237157098218205756922;\n");
  fprintf(outfile, "_b[13][5] = 5.82534730759650564865380791881446903;\n");
  fprintf(outfile, "_b[13][6] =-0.36527452339161313311889856846974452;\n");
  fprintf(outfile, "_b[13][7] = 1.18860324058346533283780076203192232;\n");
  fprintf(outfile, "_b[13][8] = 0.57970467638357921347110271762687972;\n");
  fprintf(outfile, "_b[13][9] =-0.86824862589087693262676988867897834;\n");
  fprintf(outfile, "_b[13][10] =-5.20227677296454721392873650976792184;\n");
  fprintf(outfile, "_b[13][11] =-0.79895541420753382543211121058675915;\n");
  fprintf(outfile, "_b[13][12] = 0.14360623206363792632792463778889008;\n");
  fprintf(outfile, "_b[14][0] = 8.49173149061346398013352206978380938;\n");
  fprintf(outfile, "_b[14][5] = 86.32213734729036800877634194386790750;\n");
  fprintf(outfile, "_b[14][6] = 1.02560575501091662034511526187393241;\n");
  fprintf(outfile, "_b[14][7] = 85.77427969817339941806831550695235092;\n");
  fprintf(outfile, "_b[14][8] =-13.98699305104110611795532466113248067;\n");
  fprintf(outfile, "_b[14][9] =-20.71537405501426352265946477613161883;\n");
  fprintf(outfile, "_b[14][10] =-72.16597156619946800281180102605140463;\n");
  fprintf(outfile, "_b[14][11] =-76.71211139107806345587696023064419687;\n");
  fprintf(outfile, "_b[14][12] = 4.22319427707298828839851258893735507;\n");
  fprintf(outfile, "_b[14][13] =-1.25649850482823521641825667745565428;\n");
  fprintf(outfile, "_b[15][0] =-0.42892119881959353241190195318730008;\n");
  fprintf(outfile, "_b[15][5] =-9.16865700950084689999297912545025359;\n");
  fprintf(outfile, "_b[15][6] = 1.08317616770620939241547721530003920;\n");
  fprintf(outfile, "_b[15][7] =-1.23501525358323653198215832293981810;\n");
  fprintf(outfile, "_b[15][8] =-1.21438272617593906232943856422371019;\n");
  fprintf(outfile, "_b[15][9] = 1.37226168507232166621351243731869914;\n");
  fprintf(outfile, "_b[15][10] = 9.15723239697162418155377135344394113;\n");
  fprintf(outfile, "_b[15][11] = 1.30616301842220047563298585480401671;\n");
  fprintf(outfile, "_b[15][12] =-0.25285618808937955976690569433069974;\n");
  fprintf(outfile, "_b[15][13] = 0.38099910799663987066763679926508552;\n");
  fprintf(outfile, "\n");

  fprintf(outfile, "_c[0] = 0.01490902081978461022483617102382552;\n");
  fprintf(outfile, "_c[7] =-0.20408044692054151258349120934134791;\n");
  fprintf(outfile, "_c[8] = 0.22901438600570447264772469337066476;\n");
  fprintf(outfile, "_c[9] = 0.12800558251147375669208211573729202;\n");
  fprintf(outfile, "_c[10] = 0.22380626846054143649770066956485937;\n");
  fprintf(outfile, "_c[11] = 0.39553165293700054420552389156421651;\n");
  fprintf(outfile, "_c[12] = 0.05416646758806981196568364538360743;\n");
  fprintf(outfile, "_c[13] = 0.12691439652445903685643385312168037;\n");
  fprintf(outfile, "_c[14] =-0.00052539244262118876455834655383035;\n");
  fprintf(outfile, "_c[15] = 1.0/31.0;\n");
  fprintf(outfile, "\n");

  fprintf(outfile, "_cs[0] = 0.00653047880643482012034413441159249;\n");
  fprintf(outfile, "_cs[7] =-2.31471038197461347517552506241529830;\n");
  fprintf(outfile, "_cs[8] = 0.43528227238866280799530900822377013;\n");
  fprintf(outfile, "_cs[9] = 0.14907947287101933118545845390618763;\n");
  fprintf(outfile, "_cs[10] = 0.17905535442235532311850533252768020;\n");
  fprintf(outfile, "_cs[11] = 2.53400872222767706921176214508820825;\n");
  fprintf(outfile, "_cs[12] =-0.55430437423209112896721332268159015;\n");
  fprintf(outfile, "_cs[13] = 0.56924788787870083224213506297615260;\n");
  fprintf(outfile, "_cs[14] =-0.03644749690427461198884026816573513;\n");
  fprintf(outfile, "_cs[15] = 1.0/31.0;\n");
  fprintf(outfile, "\n");

  fprintf(outfile, "\n");
  if ((simulation()->parameters()->stochastic)&&(!noNoises())) {
    fprintf(outfile, "const double _var = 1");
    for (unsigned long i=0; i<simulation()->field()->geometry()->nDims(); i++) {
      fprintf(outfile, "/_%s_dx%li", fieldName, i);
    }
    fprintf(outfile, ";\n");
    fprintf(outfile, "double *_noise_vector = NULL;\n");
    fprintf(outfile, "list<_dtdWstore> _noise_list;\n");
    fprintf(outfile, "list<_dtdWstore>::iterator _active_node;\n");
    fprintf(outfile, "\n");

  }

  fprintf(outfile, "double _step = %s/(double)%li;\n", interval()->c_str(), lattice());
  fprintf(outfile, "double _min_step=%s;\n", interval()->c_str());
  fprintf(outfile, "double _max_step=0.0;\n");
  fprintf(outfile, "double _tolerance=%s;\n", tolerance()->c_str());

  if (simulation()->parameters()->errorCheck) {
    fprintf(outfile, "if (_half_step)\n");
    fprintf(outfile, "    _tolerance=_tolerance/16.0;\n");
    fprintf(outfile, "\n");
  }
  fprintf(outfile, "double _error;\n");
  fprintf(outfile, "bool _discard=false;\n");
  fprintf(outfile, "bool _break_next=false;\n");
  fprintf(outfile, "bool _next_sample_flag[%li];\n", simulation()->output()->nMomentGroups()+2);
  fprintf(outfile, "for (unsigned long _i0=0; _i0<%li; _i0++)\n", simulation()->output()->nMomentGroups()+2);
  fprintf(outfile, "    _next_sample_flag[_i0]=false;\n");
  fprintf(outfile, "unsigned long _next_sample_counter[%li];\n", simulation()->output()->nMomentGroups());
  fprintf(outfile, "for (unsigned long _i0=0; _i0<%li; _i0++)\n", simulation()->output()->nMomentGroups());
  fprintf(outfile, "    _next_sample_counter[_i0]=1;\n");
  fprintf(outfile, "\n");
  fprintf(outfile, "const double _%s_ini=%s;\n", propDim, propDim);
  fprintf(outfile, "\n");
  fprintf(outfile, "double _%s_break_next=_segment%li_setup_sampling(_next_sample_flag, _next_sample_counter);\n", propDim, segmentNumber);
  fprintf(outfile, "if ((%s-_%s_ini+_step)>=_%s_break_next){\n", propDim, propDim, propDim);
  fprintf(outfile, "    _break_next=true;\n");
  fprintf(outfile, "    _step=_%s_break_next-%s+_%s_ini;\n", propDim, propDim, propDim);
  fprintf(outfile, "}\n");
  fprintf(outfile, "\n");

  if (max_iter){
    fprintf(outfile, "unsigned long _step_counter=0;\n");
    fprintf(outfile, "unsigned long _max_steps;\n");
    if (simulation()->parameters()->errorCheck) {
      fprintf(outfile, "if (_half_step)\n");
      fprintf(outfile, "    _max_steps=%li;\n", 2*myMaxIterations);
      fprintf(outfile, "  else\n");
      fprintf(outfile, "    _max_steps=%li;\n", myMaxIterations);
    }
    else{
      fprintf(outfile, "_max_steps=%li;\n", myMaxIterations);
    }
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "const double _epsilon = 1e-6; // small number for tolerances on floating-point differences\n");

  fprintf(outfile, "do{\n    do{\n");
  fprintf(outfile, "\n");

  writeRuntimeCheckingCode(outfile);

  if ((simulation()->parameters()->stochastic)&&(!noNoises())) {

    fprintf(outfile, "if (_noise_list.empty()) { \n");
    fprintf(outfile, "   //Noise list empty so start afresh \n");
    fprintf(outfile, "   _noise_vector = new double[_%s_size*_n_noises];\n", fieldName);
    if (simulation()->parameters()->errorCheck) {
      fprintf(outfile, "   _make_noises(_gen1, _var/_step, _noise_vector, _%s_size*_n_noises);\n", fieldName);
    }
    else{
      fprintf(outfile, "   _make_noises(_gen, _var/_step, _noise_vector, _%s_size*_n_noises);\n", fieldName);
    }
    fprintf(outfile, "   _noise_list.push_front(_dtdWstore()); \n");
    fprintf(outfile, "   _active_node = _noise_list.begin(); \n");
    fprintf(outfile, "   _active_node->_step = _step; \n");
    fprintf(outfile, "   _active_node->_noise_vector = _noise_vector; \n");
    fprintf(outfile, "} \n");
    fprintf(outfile, "else { \n");
    fprintf(outfile, "   if (_step*(1.0+_epsilon) < _noise_list.begin()->_step) { \n");
    fprintf(outfile, " _noise_vector = new double[_%s_size*_n_noises];\n", fieldName);
    fprintf(outfile, "     //Create new smallest time step \n");
    if (simulation()->parameters()->errorCheck) {
      fprintf(outfile, "       _make_noises(_gen1, _var*((_noise_list.begin()->_step)-_step)/(_step*_noise_list.begin()->_step), _noise_vector, _main_size*_n_noises);\n");
    }
    else{
      fprintf(outfile, "       _make_noises(_gen, _var*((_noise_list.begin()->_step)-_step)/(_step*_noise_list.begin()->_step), _noise_vector, _main_size*_n_noises);\n");
    }
    fprintf(outfile, "       for (unsigned long _i0=0; _i0<_main_size*_n_noises; _i0+=1) { \n");
    fprintf(outfile, "          _noise_vector[_i0] += _noise_list.begin()->_noise_vector[_i0]; \n");
    fprintf(outfile, "       } \n");
    fprintf(outfile, "    _noise_list.push_front(_dtdWstore()); \n");
    fprintf(outfile, "    _active_node = _noise_list.begin(); \n");
    fprintf(outfile, "    _active_node->_step = _step; \n");
    fprintf(outfile, "    _active_node->_noise_vector = _noise_vector; \n");
    fprintf(outfile, " } \n");
    fprintf(outfile, " else { \n");
    fprintf(outfile, "     //Use time step already attempted \n");
    fprintf(outfile, " for (_active_node=_noise_list.begin(); (_active_node!=_noise_list.end())&&(_active_node->_step <= _step*(1.0+_epsilon)); _active_node++) ;\n \n");
    fprintf(outfile, " _active_node--; \n");
    fprintf(outfile, "   _step = _active_node->_step; \n");
    fprintf(outfile, "   _noise_vector = _active_node->_noise_vector; \n");
    fprintf(outfile, "   if (_break_next && fabs(t -_%s_ini + _step - _t_break_next) > _step*_epsilon)\n",propDim);
    fprintf(outfile, "     _break_next = false;\n");
    fprintf(outfile, " } \n");
    fprintf(outfile, "} \n");

  }
  writeSingleStepCode(outfile, FULLSTEP);

  fprintf(outfile, "\n");

  fprintf(outfile, "    _error=_segment%li_timestep_error(_akafield_%s_main);\n", segmentNumber, fieldName);
  fprintf(outfile, "\n");
  fprintf(outfile, "    if (_error<_tolerance){;\n");
  if ((simulation()->parameters()->stochastic)&&(!noNoises())) {
    fprintf(outfile, "   //Trim tree \n");
    fprintf(outfile, "   _active_node++;\n");
    fprintf(outfile, "        if (_active_node == _noise_list.end()) {\n");
    fprintf(outfile, "    _noise_vector = NULL;\n");
    fprintf(outfile, "    _noise_list.clear();\n");
    fprintf(outfile, "   }\n");
    fprintf(outfile, "        else {\n");
    fprintf(outfile, "    for (list<_dtdWstore>::iterator _temp_iter=_active_node; _temp_iter!=_noise_list.end(); _temp_iter++) {\n");
    fprintf(outfile, "           _temp_iter->_step -= _step;\n");
    fprintf(outfile, "      for (unsigned long _i0=0; _i0<_main_size*_n_noises; _i0+=1) \n");
    fprintf(outfile, "       _temp_iter->_noise_vector[_i0] = (_temp_iter->_noise_vector[_i0]*(_temp_iter->_step+_step) - _noise_vector[_i0]*_step)/(_temp_iter->_step);\n");
    fprintf(outfile, "         }\n");
    fprintf(outfile, "    _noise_vector = NULL;\n");
    fprintf(outfile, "    while (_noise_list.begin()!=_active_node)\n");
    fprintf(outfile, "     _noise_list.pop_front();\n");
    fprintf(outfile, "        }\n");
    fprintf(outfile, "\n");
  }
  fprintf(outfile, "        if (_step>_max_step)\n");
  fprintf(outfile, "             _max_step=_step;\n");
  fprintf(outfile, "        if (!_break_next)\n");
  fprintf(outfile, "             if (_step<_min_step)\n");
  fprintf(outfile, "                 _min_step=_step;\n");
  if ((simulation()->parameters()->stochastic)&&(!noNoises())) {
    fprintf(outfile, "        _step*=0.92*pow(fabs(_tolerance/_error), 0.2);\n");
  }
  else {
    fprintf(outfile, "        _step*=0.92*pow(fabs(_tolerance/_error), 0.111);\n");
  }
  fprintf(outfile, "        _discard=false;\n");
  fprintf(outfile, "        }\n");
  fprintf(outfile, "    else{\n");
  fprintf(outfile, "        %s-=_step;\n", propDim);
  if ((simulation()->parameters()->stochastic)&&(!noNoises())) {
    fprintf(outfile, "        _step*=0.5;\n");
  }
  else {
    fprintf(outfile, "        _step*=0.92*pow(fabs(_tolerance/_error), 0.125);\n");
  }
  fprintf(outfile, "        _discard=true;\n");
  fprintf(outfile, "        _break_next=false;\n");
  fprintf(outfile, "        _segment%li_reset(_%s_init, _step);\n", segmentNumber, fieldName);
  fprintf(outfile, "        }\n");

  if (max_iter){
    fprintf(outfile, "\n");
    fprintf(outfile, "    _step_counter++;\n");
    fprintf(outfile, "    if (_step_counter>= _max_steps){\n");
    fprintf(outfile, "        _discard=false;\n");
    fprintf(outfile, "        _break_next=true;\n");
    fprintf(outfile, "        _next_sample_flag[%li]=true;\n", simulation()->output()->nMomentGroups()+1);
    fprintf(outfile, "    }\n");
  }

  fprintf(outfile, "    }while (_discard);\n");


  fprintf(outfile, "\n");
  fprintf(outfile, "    if (_break_next){\n");
  for (unsigned long i=0; i<simulation()->output()->nMomentGroups(); i++)
    if (samples(i)!=0){
      fprintf(outfile, "        if (_next_sample_flag[%li]){\n", i);
      fprintf(outfile, "          _mg%li_sample();\n", i);
      fprintf(outfile, "          _next_sample_counter[%li]++;\n", i);
      fprintf(outfile, "        }\n");
    }
  fprintf(outfile, "        if (_next_sample_flag[%li])\n", simulation()->output()->nMomentGroups());
  fprintf(outfile, "           _next_sample_flag[%li]=true;\n", simulation()->output()->nMomentGroups()+1);
  fprintf(outfile, "             else{\n");
  fprintf(outfile, "               _break_next=false;\n");
  fprintf(outfile, "               _%s_break_next=_segment%li_setup_sampling(_next_sample_flag, _next_sample_counter);\n", propDim, segmentNumber);
  fprintf(outfile, "                 }\n");
  fprintf(outfile, "    }\n");
  fprintf(outfile, "    if ((%s-_%s_ini+_step)>_%s_break_next){\n", propDim, propDim, propDim);
  fprintf(outfile, "        _break_next=true;\n");

  // calculate the previous timestep which was actually accepted while the current
  // value of _step is only the guess for the next iteration.
  if (simulation()->parameters()->nPaths=="1") {
    if (simulation()->parameters()->usempi)
      fprintf(outfile, "      if (rank==0)\n");
    if ((simulation()->parameters()->stochastic)&&(!noNoises())) {
      fprintf(outfile, "          printf(\"Current timestep: %%e\\n\", _step/(0.92*pow(fabs(_tolerance/_error), 0.2)));\n");
    }
    else {
      fprintf(outfile, "          printf(\"Current timestep: %%e\\n\", _step/(0.92*pow(fabs(_tolerance/_error), 0.125)));\n");
    }
  }
  fprintf(outfile, "        _step=_%s_break_next-%s+_%s_ini;\n", propDim, propDim, propDim);
  fprintf(outfile, "    }\n");
  fprintf(outfile, "}while (!_next_sample_flag[%li]);\n", simulation()->output()->nMomentGroups()+1);
  fprintf(outfile, "\n");

  if (simulation()->parameters()->runTimeLimit > 0.0)
    fprintf(outfile, "_SEGMENT%li_END:\n", segmentNumber);

  if (simulation()->parameters()->nPaths=="1") {
    if (simulation()->parameters()->usempi)
      fprintf(outfile, "if (rank==0)\n");
    fprintf(outfile, "printf(\"Segment %li: minimum timestep: %%e maximum timestep: %%e \\n\", _min_step, _max_step);\n", segmentNumber);
  }
  fprintf(outfile, "\n");

  if (max_iter){
    fprintf(outfile, "if (_step_counter>= _max_steps){\n");
    fprintf(outfile, "    printf(\" \\n \");\n");
    if (simulation()->parameters()->errorCheck) {
      fprintf(outfile, "    if (_half_step)\n");
      fprintf(outfile, "        printf(\"Reached %li iterations, exiting at %s = %%e \\n \", %s);\n", 2*myMaxIterations, propDim, propDim);
      fprintf(outfile, "    else\n");
      fprintf(outfile, "        printf(\"Reached %li iterations, exiting at %s = %%e \\n \", %s);\n", myMaxIterations, propDim, propDim);
    }
    else{
      fprintf(outfile, "    printf(\"Reached %li iterations, exiting at %s = %%e \\n \", %s);\n", myMaxIterations, propDim, propDim);
    }
    fprintf(outfile, "    printf(\"Last error: %%e \\n \", _error);\n");
    fprintf(outfile, "    printf(\"Last planned timestep: %%e \\n \", _step);\n");
    fprintf(outfile, "    printf(\" \\n \");\n");
    fprintf(outfile, "}\n");
  }//end max iter

  if ((simulation()->parameters()->stochastic)&&(!noNoises())) {
    fprintf(outfile, "    _noise_vector=NULL;\n");
    fprintf(outfile, "  _noise_list.clear();\n");
  }
  fprintf(outfile, "    fftw_free(_akafield_%s_main);\n", fieldName);
  fprintf(outfile, "    fftw_free(_akbfield_%s_main);\n", fieldName);
  fprintf(outfile, "    fftw_free(_akcfield_%s_main);\n", fieldName);
  fprintf(outfile, "    fftw_free(_akdfield_%s_main);\n", fieldName);
  fprintf(outfile, "    fftw_free(_akefield_%s_main);\n", fieldName);
  fprintf(outfile, "    fftw_free(_akffield_%s_main);\n", fieldName);
  fprintf(outfile, "    fftw_free(_akgfield_%s_main);\n", fieldName);
  fprintf(outfile, "    fftw_free(_akhfield_%s_main);\n", fieldName);
  fprintf(outfile, "    fftw_free(_akifield_%s_main);\n", fieldName);
  fprintf(outfile, "    fftw_free(_akjfield_%s_main);\n", fieldName);
  fprintf(outfile, "    fftw_free(_%s_init);\n", fieldName);

  fprintf(outfile, "}\n");
  fprintf(outfile, "\n");
}

// **************************************************************************
void xmdsIntegrateARK89::writeCalculateCrossFieldRoutine(
                                                         FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateARK89IP::writeCalculateCrossFieldRoutine\n");
  }

  const unsigned long nDims = simulation()->field()->geometry()->nDims();
  const char *const fieldName = simulation()->field()->name()->c_str();

  fprintf(outfile, "// *************************\n");
  fprintf(outfile, "void _segment%li_calculate_cross_field(", segmentNumber);
  if ((simulation()->parameters()->stochastic)&&(!noNoises())) {
    fprintf(outfile, "const double *const _noise_vector");
  }
  fprintf(outfile, ") {\n");
  fprintf(outfile, "\n");

  if ((simulation()->parameters()->stochastic)&&(!noNoises())) {
    fprintf(outfile, "const double* _noises = _noise_vector;\n");
    fprintf(outfile, "\n");
  }

  if (crossDimNumber()+1<nDims) {
    fprintf(outfile, "const unsigned long _%s_cross_size = _%s_lattice%li", fieldName, fieldName, crossDimNumber()+1);
    for (unsigned long i=crossDimNumber()+2; i<nDims; i++) {
      fprintf(outfile, "*_%s_lattice%li", fieldName, i);
    }
    fprintf(outfile, ";\n");
  }
  else
    fprintf(outfile, "const unsigned long _%s_cross_size = 1;\n", fieldName);
  fprintf(outfile, "\n");

  // need to create a vector list for ONLY main vectors

  list<XMLString> myMainVectorNamesList;

  for (list<XMLString>::const_iterator pXMLString = vectorNamesList()->begin(); pXMLString != vectorNamesList()->end(); pXMLString++) {
    list<XMLString>::const_iterator pXMLString2 = crossVectorNamesList()->begin();
    while ((pXMLString2 != crossVectorNamesList()->end()) && (*pXMLString2 != *pXMLString)) {
      pXMLString2++;
    }
    if (*pXMLString2 != *pXMLString) {
      myMainVectorNamesList.push_back(*pXMLString);
    }
  }

  const char* typeName;
  list<const xmdsVector*> mainVectorList;

  for (list<XMLString>::const_iterator pXMLString = myMainVectorNamesList.begin(); pXMLString != myMainVectorNamesList.end(); pXMLString++) {

    const xmdsVector* mainVector;

    if (!simulation()->field()->getVector(*pXMLString, mainVector)) {
      throw xmdsException("Internal error in xmdsIntegrateARK89::writeCalculateCrossFieldRoutine: cannot find main vector");
    }

    mainVectorList.push_back(mainVector);

    if (mainVector->vectorType()==DOUBLE) {
      typeName="double";
    }
    else {
      typeName="complex";
    }

    fprintf(outfile, "%s *_%s_%s_old = (%s*)fftw_malloc(sizeof(%s)*_%s_cross_size*_%s_%s_ncomponents);\n",
            typeName, fieldName, pXMLString->c_str(), typeName, typeName, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  list<const xmdsVector*> crossVectorList;

  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {

    const xmdsVector* crossVector;

    if (!simulation()->field()->getVector(*pXMLString, crossVector)) {
      throw xmdsException("Internal error in xmdsIntegrateARK89::writeCalculateCrossFieldRoutine: cannot find cross vector");
    }

    crossVectorList.push_back(crossVector);

    if (crossVector->vectorType()==DOUBLE) {
      typeName="double";
    }
    else {
      typeName="complex";
    }

    fprintf(outfile, "%s *_%s_%s_K = (%s*)fftw_malloc(sizeof(%s)*_%s_cross_size*_%s_%s_ncomponents);\n",
            typeName, fieldName, crossVector->name()->c_str(), typeName, typeName, fieldName, fieldName, crossVector->name()->c_str());
    fprintf(outfile, "%s *_%s_%s_I = (%s*)fftw_malloc(sizeof(%s)*_%s_cross_size*_%s_%s_ncomponents);\n",
            typeName, fieldName, crossVector->name()->c_str(), typeName, typeName, fieldName, fieldName, crossVector->name()->c_str());
    fprintf(outfile, "%s *_%s_%s_d = (%s*)fftw_malloc(sizeof(%s)*_%s_cross_size*_%s_%s_ncomponents);\n",
            typeName, fieldName, crossVector->name()->c_str(), typeName, typeName, fieldName, fieldName, crossVector->name()->c_str());
    fprintf(outfile, "\n");

    for (unsigned long i=0; i<crossVector->nComponents(); i++) {
      fprintf(outfile, "%s d%s_d%s;\n",
              typeName, crossVector->componentName(i)->c_str(),
              simulation()->field()->geometry()->dimension(crossDimNumber())->name.c_str());
    }
    fprintf(outfile, "\n");
  }

  // add cross vectors to total vectors to use

  list<XMLString> myTotalVectorsList = myMainVectorNamesList;

  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    myTotalVectorsList.push_back(*pXMLString);
  }


  simulation()->field()->vectors2space(outfile, 0, myTotalVectorsList, "");

  // open outer loops

  for (unsigned long i=0; i<crossDimNumber(); i++) {

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, "double %s = _%s_xmin%li;\n", simulation()->field()->geometry()->dimension(i)->name.c_str(), fieldName, i);
    fprintf(outfile, "\n");

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, "for (unsigned long _i%li=0; _i%li<_%s_lattice%li; _i%li++) {\n", i, i, fieldName, i, i);
    fprintf(outfile, "\n");
  }

  for (unsigned long j=0; j<crossDimNumber(); j++) {
    fprintf(outfile, " ");
  }
  fprintf(outfile, "double %s = _%s_xmin%li;\n",
          simulation()->field()->geometry()->dimension(crossDimNumber())->name.c_str(), fieldName, crossDimNumber());
  fprintf(outfile, "\n");

  for (unsigned long j=0; j<crossDimNumber(); j++) {
    fprintf(outfile, " ");
  }
  fprintf(outfile, "for (unsigned long _i%li=0; _i%li<_%s_lattice%li-1; _i%li++) {\n",
          crossDimNumber(), crossDimNumber(), fieldName, crossDimNumber(), crossDimNumber());
  fprintf(outfile, "\n");

  char indent[64];
  for (unsigned long i=0; i<crossDimNumber()+1; i++) {
    indent[i]=0x09;
  }
  indent[crossDimNumber()+1]=0;

  char indent2[64];
  for (unsigned long i=0; i<nDims; i++) {
    indent2[i]=0x09;
  }
  indent2[nDims]=0;

  for (list<XMLString>::const_iterator pXMLString = myTotalVectorsList.begin(); pXMLString != myTotalVectorsList.end(); pXMLString++) {
    fprintf(outfile, "%sunsigned long _%s_%s_index_pointer_begin=0;\n", indent, fieldName, pXMLString->c_str());
    for (unsigned long i=0; i<crossDimNumber()+1; i++) {
      fprintf(outfile, "%s_%s_%s_index_pointer_begin += _i%li", indent, fieldName, pXMLString->c_str(), i);
      for (unsigned long j=i+1; j<nDims; j++) {
        fprintf(outfile, "*_%s_lattice%li", fieldName, j);
      }
      fprintf(outfile, "*_%s_%s_ncomponents;\n", fieldName, pXMLString->c_str());
    }
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s// copy cross vectors into K and I vectors\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++) {\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s_K[_j] = _%s_%s[_%s_%s_index_pointer_begin + _j];\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s_I[_j] = _%s_%s[_%s_%s_index_pointer_begin + _j];\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "%s }\n", indent);
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s// store main vectors into old\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = myMainVectorNamesList.begin(); pXMLString != myMainVectorNamesList.end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s_old[_j] = _active_%s_%s[_%s_%s_index_pointer_begin + _j];\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  // ****************************************************************
  fprintf(outfile, "%s// ********** step 1 ***************\n", indent);
  fprintf(outfile, "\n");
  fprintf(outfile, "%s {\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = myTotalVectorsList.begin(); pXMLString != myTotalVectorsList.end(); pXMLString++) {
    fprintf(outfile, "%s unsigned long _%s_%s_index_pointer=_%s_%s_index_pointer_begin;\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s // calculate k1\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%s unsigned long _%s_%s_index_pointer_local=0;\n", indent, fieldName, pXMLString->c_str());
  }
  fprintf(outfile, "\n");

  // open inner loops
  for (unsigned long i=crossDimNumber()+1; i<nDims; i++) {

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " double %s = _%s_xmin%li;\n", simulation()->field()->geometry()->dimension(i)->name.c_str(), fieldName, i);
    fprintf(outfile, "\n");

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " for (unsigned long _i%li=0; _i%li<_%s_lattice%li; _i%li++) {\n", i, i, fieldName, i, i);
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "// *********** cross_propagation code ***********\n");
  fprintf(outfile, "%s\n", crossPropagationCode()->c_str());
  fprintf(outfile, "// **********************************************\n");
  fprintf(outfile, "\n");
  for (list<const xmdsVector*>::const_iterator pxmdsVector = crossVectorList.begin(); pxmdsVector != crossVectorList.end(); pxmdsVector++) {
    for (unsigned long i=0; i<(*pxmdsVector)->nComponents(); i++) {
      fprintf(outfile, "%s _%s_%s_d[_%s_%s_index_pointer_local + %li] = d%s_d%s*_%s_dx%li;\n",
              indent2, fieldName, (*pxmdsVector)->name()->c_str(), fieldName, (*pxmdsVector)->name()->c_str(), i,
              (*pxmdsVector)->componentName(i)->c_str(),
              simulation()->field()->geometry()->dimension(crossDimNumber())->name.c_str(), fieldName, crossDimNumber());
    }
    fprintf(outfile, "\n");
  }

  //close inner loops
  if (crossDimNumber()+1<nDims) {
    for (list<XMLString>::const_iterator pXMLString = myTotalVectorsList.begin(); pXMLString != myTotalVectorsList.end(); pXMLString++) {
      fprintf(outfile, "%s _%s_%s_index_pointer += _%s_%s_ncomponents;\n",
              indent2, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    }
    fprintf(outfile, "\n");
    for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
      fprintf(outfile, "%s _%s_%s_index_pointer_local += _%s_%s_ncomponents;\n",
              indent2, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    }
    fprintf(outfile, "\n");
  }

  for (unsigned long i=nDims; i>crossDimNumber()+1; i--) {

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " %s += _%s_dx%li;\n", simulation()->field()->geometry()->dimension(i-1)->name.c_str(), fieldName, i-1);
    fprintf(outfile, "\n");
    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " }\n");
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s }\n", indent);
  fprintf(outfile, "\n");

  fprintf(outfile, "%s// add to K cross vector\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s_K[_j] += _%s_%s_d[_j]/6;\n", indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s// create next cross vectors\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s[_%s_%s_index_pointer_begin + _j] = _%s_%s_I[_j] + _%s_%s_d[_j]/2;\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s// create midpoint main vectors for steps 2 and 3\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = myMainVectorNamesList.begin(); pXMLString != myMainVectorNamesList.end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=_%s_%s_index_pointer_begin; _j < _%s_%s_index_pointer_begin + _%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _active_%s_%s[_j] = (_active_%s_%s[_j]+_active_%s_%s[_%s_cross_size*_%s_%s_ncomponents + _j])/2;\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(),
            fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }
  fprintf(outfile, "%s// move cross dim to lattice midpoint for steps 2 and 3\n", indent);
  fprintf(outfile, "\n");
  fprintf(outfile, "%s%s += _%s_dx%li/2;\n",
          indent, simulation()->field()->geometry()->dimension(crossDimNumber())->name.c_str(), fieldName, crossDimNumber());
  fprintf(outfile, "\n");

  // ****************************************************************
  fprintf(outfile, "%s// ********** step 2 ***************\n", indent);
  fprintf(outfile, "\n");
  fprintf(outfile, "%s {\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = myTotalVectorsList.begin(); pXMLString != myTotalVectorsList.end(); pXMLString++) {
    fprintf(outfile, "%s unsigned long _%s_%s_index_pointer=_%s_%s_index_pointer_begin;\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s // calculate k1\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%s unsigned long _%s_%s_index_pointer_local=0;\n", indent, fieldName, pXMLString->c_str());
  }
  fprintf(outfile, "\n");

  // open inner loops
  for (unsigned long i=crossDimNumber()+1; i<nDims; i++) {

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " double %s = _%s_xmin%li;\n", simulation()->field()->geometry()->dimension(i)->name.c_str(), fieldName, i);
    fprintf(outfile, "\n");

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " for (unsigned long _i%li=0; _i%li<_%s_lattice%li; _i%li++) {\n", i, i, fieldName, i, i);
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "// *********** cross_propagation code ***********\n");
  fprintf(outfile, "%s\n", crossPropagationCode()->c_str());
  fprintf(outfile, "// **********************************************\n");
  fprintf(outfile, "\n");
  for (list<const xmdsVector*>::const_iterator pxmdsVector = crossVectorList.begin(); pxmdsVector != crossVectorList.end(); pxmdsVector++) {
    for (unsigned long i=0; i<(*pxmdsVector)->nComponents(); i++) {
      fprintf(outfile, "%s _%s_%s_d[_%s_%s_index_pointer_local + %li] = d%s_d%s*_%s_dx%li;\n",
              indent2, fieldName, (*pxmdsVector)->name()->c_str(), fieldName, (*pxmdsVector)->name()->c_str(), i,
              (*pxmdsVector)->componentName(i)->c_str(),
              simulation()->field()->geometry()->dimension(crossDimNumber())->name.c_str(), fieldName, crossDimNumber());
    }
    fprintf(outfile, "\n");
  }

  //close inner loops
  if (crossDimNumber()+1<nDims) {
    for (list<XMLString>::const_iterator pXMLString = myTotalVectorsList.begin(); pXMLString != myTotalVectorsList.end(); pXMLString++) {
      fprintf(outfile, "%s _%s_%s_index_pointer += _%s_%s_ncomponents;\n",
              indent2, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    }
    fprintf(outfile, "\n");
    for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
      fprintf(outfile, "%s _%s_%s_index_pointer_local += _%s_%s_ncomponents;\n",
              indent2, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    }
    fprintf(outfile, "\n");
  }

  for (unsigned long i=nDims; i>crossDimNumber()+1; i--) {

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " %s += _%s_dx%li;\n", simulation()->field()->geometry()->dimension(i-1)->name.c_str(), fieldName, i-1);
    fprintf(outfile, "\n");
    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " }\n");
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s }\n", indent);
  fprintf(outfile, "\n");

  fprintf(outfile, "%s// add to K cross vector\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s_K[_j] += _%s_%s_d[_j]/3;\n", indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s// create next cross vectors\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s[_%s_%s_index_pointer_begin + _j] = _%s_%s_I[_j] + _%s_%s_d[_j]/2;\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  // ****************************************************************
  fprintf(outfile, "%s// ********** step 3 ***************\n", indent);
  fprintf(outfile, "\n");
  fprintf(outfile, "%s {\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = myTotalVectorsList.begin(); pXMLString != myTotalVectorsList.end(); pXMLString++) {
    fprintf(outfile, "%s unsigned long _%s_%s_index_pointer=_%s_%s_index_pointer_begin;\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s // calculate k1\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%s unsigned long _%s_%s_index_pointer_local=0;\n", indent, fieldName, pXMLString->c_str());
  }
  fprintf(outfile, "\n");

  // open inner loops
  for (unsigned long i=crossDimNumber()+1; i<nDims; i++) {

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " double %s = _%s_xmin%li;\n", simulation()->field()->geometry()->dimension(i)->name.c_str(), fieldName, i);
    fprintf(outfile, "\n");

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " for (unsigned long _i%li=0; _i%li<_%s_lattice%li; _i%li++) {\n", i, i, fieldName, i, i);
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "// *********** cross_propagation code ***********\n");
  fprintf(outfile, "%s\n", crossPropagationCode()->c_str());
  fprintf(outfile, "// **********************************************\n");
  fprintf(outfile, "\n");
  for (list<const xmdsVector*>::const_iterator pxmdsVector = crossVectorList.begin(); pxmdsVector != crossVectorList.end(); pxmdsVector++) {
    for (unsigned long i=0; i<(*pxmdsVector)->nComponents(); i++) {
      fprintf(outfile, "%s _%s_%s_d[_%s_%s_index_pointer_local + %li] = d%s_d%s*_%s_dx%li;\n",
              indent2, fieldName, (*pxmdsVector)->name()->c_str(), fieldName, (*pxmdsVector)->name()->c_str(), i,
              (*pxmdsVector)->componentName(i)->c_str(),
              simulation()->field()->geometry()->dimension(crossDimNumber())->name.c_str(), fieldName, crossDimNumber());
    }
    fprintf(outfile, "\n");
  }

  //close inner loops
  if (crossDimNumber()+1<nDims) {
    for (list<XMLString>::const_iterator pXMLString = myTotalVectorsList.begin(); pXMLString != myTotalVectorsList.end(); pXMLString++) {
      fprintf(outfile, "%s _%s_%s_index_pointer += _%s_%s_ncomponents;\n",
              indent2, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    }
    fprintf(outfile, "\n");
    for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
      fprintf(outfile, "%s _%s_%s_index_pointer_local += _%s_%s_ncomponents;\n",
              indent2, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    }
    fprintf(outfile, "\n");
  }

  for (unsigned long i=nDims; i>crossDimNumber()+1; i--) {

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " %s += _%s_dx%li;\n", simulation()->field()->geometry()->dimension(i-1)->name.c_str(), fieldName, i-1);
    fprintf(outfile, "\n");
    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " }\n");
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s }\n", indent);
  fprintf(outfile, "\n");

  fprintf(outfile, "%s// add to K cross vector\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s_K[_j] += _%s_%s_d[_j]/3;\n", indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s// create next cross vectors\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s[_%s_%s_index_pointer_begin + _j] = _%s_%s_I[_j] + _%s_%s_d[_j];\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s// move main vectors to next lattice point for step 4\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = myMainVectorNamesList.begin(); pXMLString != myMainVectorNamesList.end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=_%s_%s_index_pointer_begin; _j < _%s_%s_index_pointer_begin + _%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _active_%s_%s[_j] = _active_%s_%s[_%s_cross_size*_%s_%s_ncomponents + _j];\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(),
            fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }
  fprintf(outfile, "%s// move cross dim to next lattice point for step 4\n", indent);
  fprintf(outfile, "\n");
  fprintf(outfile, "%s%s += _%s_dx%li/2;\n",
          indent, simulation()->field()->geometry()->dimension(crossDimNumber())->name.c_str(), fieldName, crossDimNumber());
  fprintf(outfile, "\n");

  // ****************************************************************
  fprintf(outfile, "%s// ********** step 4 ***************\n", indent);
  fprintf(outfile, "\n");
  fprintf(outfile, "%s {\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = myTotalVectorsList.begin(); pXMLString != myTotalVectorsList.end(); pXMLString++) {
    fprintf(outfile, "%s unsigned long _%s_%s_index_pointer=_%s_%s_index_pointer_begin;\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s // calculate k1\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%s unsigned long _%s_%s_index_pointer_local=0;\n", indent, fieldName, pXMLString->c_str());
  }
  fprintf(outfile, "\n");

  // open inner loops
  for (unsigned long i=crossDimNumber()+1; i<nDims; i++) {

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " double %s = _%s_xmin%li;\n", simulation()->field()->geometry()->dimension(i)->name.c_str(), fieldName, i);
    fprintf(outfile, "\n");

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " for (unsigned long _i%li=0; _i%li<_%s_lattice%li; _i%li++) {\n", i, i, fieldName, i, i);
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "// *********** cross_propagation code ***********\n");
  fprintf(outfile, "%s\n", crossPropagationCode()->c_str());
  fprintf(outfile, "// **********************************************\n");
  fprintf(outfile, "\n");
  for (list<const xmdsVector*>::const_iterator pxmdsVector = crossVectorList.begin(); pxmdsVector != crossVectorList.end(); pxmdsVector++) {
    for (unsigned long i=0; i<(*pxmdsVector)->nComponents(); i++) {
      fprintf(outfile, "%s _%s_%s_d[_%s_%s_index_pointer_local + %li] = d%s_d%s*_%s_dx%li;\n",
              indent2, fieldName, (*pxmdsVector)->name()->c_str(), fieldName, (*pxmdsVector)->name()->c_str(), i,
              (*pxmdsVector)->componentName(i)->c_str(),
              simulation()->field()->geometry()->dimension(crossDimNumber())->name.c_str(), fieldName, crossDimNumber());
    }
    fprintf(outfile, "\n");
  }

  //close inner loops
  if (crossDimNumber()+1<nDims) {
    for (list<XMLString>::const_iterator pXMLString = myTotalVectorsList.begin(); pXMLString != myTotalVectorsList.end(); pXMLString++) {
      fprintf(outfile, "%s _%s_%s_index_pointer += _%s_%s_ncomponents;\n",
              indent2, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    }
    fprintf(outfile, "\n");
    for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
      fprintf(outfile, "%s _%s_%s_index_pointer_local += _%s_%s_ncomponents;\n",
              indent2, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    }
    fprintf(outfile, "\n");
  }

  for (unsigned long i=nDims; i>crossDimNumber()+1; i--) {

    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " %s += _%s_dx%li;\n", simulation()->field()->geometry()->dimension(i-1)->name.c_str(), fieldName, i-1);
    fprintf(outfile, "\n");
    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, " }\n");
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s }\n", indent);
  fprintf(outfile, "\n");

  fprintf(outfile, "%s// add to K cross vector\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s_K[_j] += _%s_%s_d[_j]/6;\n", indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s// copy I cross vector back into old main cross vector\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s[_%s_%s_index_pointer_begin + _j] = _%s_%s_I[_j];\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s// copy K cross vector into next main cross vector\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _%s_%s[_%s_%s_index_pointer_begin + _%s_cross_size*_%s_%s_ncomponents + _j] = _%s_%s_K[_j];\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s// copy old main vectors back into last lattice point\n", indent);
  fprintf(outfile, "\n");
  for (list<XMLString>::const_iterator pXMLString = myMainVectorNamesList.begin(); pXMLString != myMainVectorNamesList.end(); pXMLString++) {
    fprintf(outfile, "%sfor (unsigned long _j=0; _j<_%s_cross_size*_%s_%s_ncomponents; _j++)\n",
            indent, fieldName, fieldName, pXMLString->c_str());
    fprintf(outfile, "%s _active_%s_%s[_%s_%s_index_pointer_begin + _j] = _%s_%s_old[_j];\n",
            indent, fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str(), fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  // close outer loops

  fprintf(outfile, "%s}\n", indent);

  for (unsigned long i=crossDimNumber(); i>0; i--) {

    fprintf(outfile, "\n");
    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, "%s += _%s_dx%li;\n", simulation()->field()->geometry()->dimension(i-1)->name.c_str(), fieldName, i-1);
    fprintf(outfile, "\n");
    for (unsigned long j=0; j<i; j++) {
      fprintf(outfile, " ");
    }
    fprintf(outfile, "}\n");
  }


  for (list<XMLString>::const_iterator pXMLString = myMainVectorNamesList.begin(); pXMLString != myMainVectorNamesList.end(); pXMLString++) {

    fprintf(outfile, "    fftw_free(_%s_%s_old);\n", fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");
  }

  for (list<XMLString>::const_iterator pXMLString = crossVectorNamesList()->begin(); pXMLString != crossVectorNamesList()->end(); pXMLString++) {

    fprintf(outfile, "fftw_free(_%s_%s_K);\n", fieldName, pXMLString->c_str());
    fprintf(outfile, "fftw_free(_%s_%s_I);\n", fieldName, pXMLString->c_str());
    fprintf(outfile, "fftw_free(_%s_%s_d);\n", fieldName, pXMLString->c_str());
    fprintf(outfile, "\n");

  }

  fprintf(outfile, "\n");
  fprintf(outfile, "}\n");
  fprintf(outfile, "\n");
}


/*
 * Local variables:
 * c-indentation-style: bsd
 * c-basic-offset: 2
 * indent-tabs-mode: nil
 * End:
 *
 * vim: tabstop=2 expandtab shiftwidth=2:
 */
