/*
 Copyright (C) 2000-2006

 Code contributed by Greg Collecutt, Joseph Hope and Paul Cochrane

 This file is part of xmds.
 
 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/*
  $Id: kissdom.h 1194 2006-12-18 12:59:46Z paultcochrane $
*/

/*! @file kissdom.h
  @brief Greg Collecutt's implementation of DOM3

  Greg Collecutt's implementation of the XML Document Object Model level 3
*/

/* This header file provides the definition of my implementation
   DOM3, called "kiss", which of course is the acronym for
   "Keep It Simple Stupid". */

#ifndef LIST
#include<list>
#define LIST
#endif

#ifndef DOM3
#include<dom3.h>
#endif

// *****************************************************************************
// *****************************************************************************
//                       KissDOMImplementation
// *****************************************************************************
// *****************************************************************************

//! Implementation of the KISS version of DOM
class KissDOMImplementation : public DOMImplementation {

public :

  //! Constructor of a KissDOMImplementation object
  KissDOMImplementation();

  //! Destructor
  virtual ~KissDOMImplementation();

  //! Determines if feature exists in this version
  bool hasFeature(
		  DOMString& feature,
		  DOMString& version) const;

  // Introduced in DOM Level 2:

  //! Creates a document type
  DocumentType* createDocumentType(
				   const DOMString& qualifiedName,
				   const DOMString& publicId,
				   const DOMString& systemId);

  //! Creates a document
  Document* createDocument(
			   const DOMString& namespaceURI,
			   const DOMString& qualifiedName,
			   DocumentType* doctype);

  // Introduced in DOM Level 3:

  //! Makes available DOMImplementation's specialised interface
  DOMImplementation* getAs(
			   DOMString& feature);

  //! Prints all nodes
  void printAll();

  //! Prints a node
  void printNode(
		 const unsigned long& indent,
		 const Node* node2Print);

private :

  list<Node*> myNodeList;  //!< The node list
};

// ******************************************************************************
// ******************************************************************************
//	KissNodeList
// ******************************************************************************
// ******************************************************************************

//! Node list class in KISS DOM implementation
class KissNodeList : public NodeList {

public :

  //! Constructor of KissNodeList object
  KissNodeList(
	       const list<Node*>* yourPNodeList);

  //! Destructor
  virtual ~KissNodeList();

  //! Obtains item in node list at index
  Node* item(
	     unsigned long	index) const;

  //! Returns the length of the node list
  unsigned long length() const;
	
private :

  const list<Node*>* myPNodeList;  //!< Node list pointer (I think)
};

// ******************************************************************************
// ******************************************************************************
//	KissNamedNodeMap
// ******************************************************************************
// ******************************************************************************

//! The named node map class in the KISS DOM implementation
class KissNamedNodeMap : public NamedNodeMap {

public:

  //! Constructor for KissNamedNodeMap object
  KissNamedNodeMap(
		   const unsigned long& yourNodeType);

  //! Destructor
  virtual ~KissNamedNodeMap();

  //! Gets a named item
  Node* getNamedItem(
		     const DOMString& name) const ;

  //! Sets a named item
  Node* setNamedItem(
		     Node& arg);

  //! Removes a named item
  Node* removeNamedItem(
			const DOMString& name);

  //! Returns the length (of what?)
  unsigned long length() const;

  //! Gets an item as a node object
  Node* item(
	     const unsigned long index) const;

  // Introduced in DOM Level 2:

  //! Gets a named item namespace
  Node* getNamedItemNS(
		       const DOMString& namespaceURI,
		       const DOMString& localname) const;

  //! Sets a named item namespace
  Node* setNamedItemNS(
		       Node& arg);

  //! Removes a named item namespace
  Node* removeNamedItemNS(
			  const DOMString& namespaceURI,
			  const DOMString& localname);

  //! Sets document readonly
  void setReadOnly(
		   const bool& newReadOnly);

  //! Gets the owner document as a Document object
  const Document* ownerDocument() const;

  //! Sets the owner document
  void setOwnerDocument(
			const Document* yourOwnerDocument);

private :

  const Document* myOwnerDocument;            //!< The owner document object
  const unsigned long myNodeType;             //!< The node type
  bool readOnly;                              //!< Whether or not the document is read only
  list<Node*> myNodeList;                     //!< The list of nodes
  list<const DOMString*> myNameList;          //!< The list of names  (of what?)
  list<const DOMString*> myNamespaceURIList;  //!< The list of namespace URIs
	
  //! Whether or not can add this node (but why doesn't this return anything?)
  void canAddThisNode(const Node& arg) const;
};

// ******************************************************************************
// ******************************************************************************
//	KissNode
// ******************************************************************************
// ******************************************************************************

//! Node class in KISS DOM implementation
class KissNode : public virtual Node {

public :

  //! Constructor of KissNode object
  KissNode(
	   const Document *const yourOwnerDocument,
	   Node *const yourParentNode,
	   const DOMString& yourNodeName);

  //! Destructor
  ~KissNode();
	
  //! Returns name of node
  const DOMString* nodeName() const;

  //! Returns value of node
  const DOMString* nodeValue() const;

  //! Sets value of node
  virtual void setNodeValue(
			    const DOMString& newNodeValue);

  //! Returns parent node
  Node* parentNode() const;

  //! Sets parent node
  virtual void setParentNode(
			     Node* newParentNode);

  //! Returns a list of child nodes
  const NodeList* childNodes() const;

  //! Returns the first child node
  Node* firstChild() const;

  //! Returns the last child node
  Node* lastChild() const;

  //! Returns the previous sibling node
  Node* previousSibling() const;

  //! Returns the next sibling node
  Node* nextSibling() const;

  //! Returns the attributes as a NamedNodeMap
  const NamedNodeMap* attributes() const;

  //! Returns the owner document
  const Document* ownerDocument() const;

  //! Sets the owner document
  virtual void setOwnerDocument(
				const Document *const newOwnerDocument);

  //! Inserts a new child node before a reference child node
  Node* insertBefore(
		     Node* newChild,
		     Node* refChild);

  //! Replaces a child node
  Node* replaceChild(
		     Node* newChild,
		     Node* oldChild);

  //! Removes a child node
  Node* removeChild(
		    Node* oldChild);

  //! Appends a child node
  Node* appendChild(
		    Node* newChild);

  //! Determines if object has child nodes
  bool hasChildNodes() const;

  //! Puts nodes into a normal form
  void normalize();

  // Introduced in DOM Level 2:

  //! Determines if feature is supported in current version of xml/dom?
  bool isSupported(
		   DOMString& feature,
		   DOMString& version) const;

  //! Gets the namespace URI as a DOMString
  const DOMString* namespaceURI() const;

  //! Gets some prefix (of what?)
  const DOMString* prefix() const;

  //! Sets the prefix (of what?)
  void setPrefix(
		 const DOMString& newPrefix);

  //! Gets the local name (of what?)
  const DOMString* localName() const;

  // Introduced in DOM Level 3:

  //! Determines if object/element has attributes
  bool hasAttributes() const;

  //! Returns the base URI as a DOMString
  const DOMString* baseURI() const;

  //! Compares the document order with another node
  DocumentOrder compareDocumentOrder(
				     const Node* other) const;

  //! Compares the tree position with another node
  TreePosition compareTreePosition(
				   const Node* other) const;

  //! Returns the text content of this node and its decendants
  const DOMString* textContent(
			       const bool& deep) const;

  //! Sets the text content (of what?)
  virtual void setTextContent(
			      const DOMString& newTextContent);

  //! Determines if the current node is the same as argument node
  bool isSameNode(
		  const Node* other) const;

  //! Looks up namespace prefix and returns as DOMString
  const DOMString* lookupNamespacePrefix(
					 const DOMString& namespaceURI) const;

  //! Looks up namespace URI and returns as DOMString
  const DOMString* lookupNamespaceURI(
				      const DOMString& prefix) const;

  //! Puts the namespace into a normal form
  void normalizeNS();

  //! Returns a key identifying this node
  DOMKey key() const;

  //! Determines if node is equal to argument node
  bool equalsNode(
		  const Node* arg,
		  bool deep) const;

  //! Gets the node's specialised interface
  Node* getAs(
	      DOMString& feature);

  //! Determines if (something) is read only
  bool readOnly() const;

  //! Sets something read only
  void setReadOnly(
		   const bool& newReadOnly,
		   const bool& deep);

  //! Checks the child adding constraints
  void checkChildAddingConstraints1(
				    const Node* newChild) const;

  //! Checks the child adding constraints
  virtual void checkChildAddingConstraints2(
					    const Node* newChild) const;

protected :

  DOMString myNodeName;  //!< The current node as a DOMString
  bool myReadOnly;       //!< The read only status of the node/document?

private :

  const Document* myOwnerDocument;   //!< The owner document
  Node* myParentNode;                //!< The parent of the current node
  DOMKey myDOMKey;                   //!< The DOM key
  list<Node*> myNodeList;            //!< The node list
  KissNodeList myKissNodeList;       //!< The Kiss node list
  mutable DOMString myTextContent;   //!< The text content of the node
};

// ******************************************************************************
// ******************************************************************************
//	KissDocument
// ******************************************************************************
// ******************************************************************************

//! Document class for the KISS DOM implementation
class KissDocument :	public KissNode,
			public Document {

public :

  //! Override of KissNode::nodeType()
  unsigned long nodeType() const;

  //! Override of KissNode::setParentNode()
  void setParentNode(
		     Node* newParentNode);

  //! Override of KissNode::setOwnerDocument()
  void setOwnerDocument(
			const Document *const newOwnerDocument);

  //! Override of KissNode::insertBefore()
  Node* insertBefore(
		     Node* newChild,
		     Node* refChild);

  //! Override of KissNode::replaceChild()
  Node* replaceChild(
		     Node* newChild,
		     Node* oldChild);

  //! Override of KissNode::removeChild()
  Node* removeChild(
		    Node* oldChild);

  //! Override of KissNode::appendChild()
  Node* appendChild(
		    Node* newChild);

  //! Implementation of Node::cloneNode()
  Node* cloneNode(
		  const bool& deep) const;

  //! Override of KissNode::namespaceURI()
  const DOMString* namespaceURI() const;

  //! Override of KissNode::compareDocumentOrder()
  DocumentOrder compareDocumentOrder(
				     const Node* other) const;

  //! Override of KissNode::compareTreePosition()
  TreePosition compareTreePosition(
				   const Node* other) const;

  //! Override of KissNode::setTextContent()
  void setTextContent(
		      const DOMString& newTextContent);

  //! Override of KissNode::normalizeNS()
  void normalizeNS();

  //! Override of KissNode::checkChildAddingConstraints2()
  void checkChildAddingConstraints2(const Node* newChild) const;

  //! Constructor of KissDocument object
  KissDocument(
	       const DOMImplementation* yourDOMImplementation);

  //! Destructor
  ~KissDocument();
	
  //! Gets the DOM key
  DOMKey getDOMKey() const;

  //! Obtains the doctype
  const DocumentType* doctype() const;

  //! Obtains the DOM implementation
  const DOMImplementation* implementation() const;

  //! Obtains a document element
  Element* documentElement() const;

  //! Creates an element
  Element* createElement(
			 const DOMString& tagName);

  //! Creates a document fragment
  DocumentFragment* createDocumentFragment();

  //! Creates a text node
  Text* createTextNode(
		       const DOMString& data);

  //! Creates a comment
  Comment* createComment(
			 const DOMString& data);

  //! Creates a CDATA section
  CDATASection* createCDATASection(
				   const DOMString& data);

  //! Creates a processing instruction
  ProcessingInstruction* createProcessingInstruction(
						     const DOMString& target,
						     const DOMString& data);

  //! Creates an attribute
  Attr* createAttribute(
			const DOMString& name);

  //! Creates an entity reference
  EntityReference* createEntityReference(
					 const DOMString& name);

  //! Gets the elements by tag name
  NodeList* getElementsByTagName(
				 const DOMString& tagName);

  // Introduced in DOM Level 2:

  //! Imports a node
  Node* importNode(
		   const Node* importNode,
		   const bool& deep);

  //! Creates an elements namespace
  Element* createElementNS(
			   const DOMString& namespaceURI,
			   const DOMString& qualifiedName);

  //! Creates an attributes namespace
  Attr* createAttributeNS(
			  const DOMString& namespaceURI,
			  const DOMString& qualifiedName);

  //! Gets elements by tag name's namespace
  NodeList* getElementsByTagNameNS(
				   const DOMString& namespaceURI,
				   const DOMString& localname);

  //! Gets elements by id
  Element* getElementById(
			  const DOMString& elementId);

  // Introduced in DOM Level 3:

  //! Determines actual encoding
  const DOMString* actualEncoding() const;

  //! Sets actual encoding
  void setActualEncoding(
			 const DOMString& newActualEncoding);

  //! Determines encoding
  const DOMString* encoding() const;

  //! Sets encoding
  void setEncoding(
		   const DOMString& newEncoding);

  //! Specifies if the document is standalone
  bool standalone() const;

  //! Sets standalone boolean
  void setStandalone(
		     const bool& newStandalone);

  //! Gets strict error checking flag
  bool strictErrorChecking() const ;

  //! Sets strict error checking flag
  void setStrictErrorChecking(
			      const bool& newStrictErrorChecking);

  //! Gets the version
  const DOMString* version() const;

  //! Sets the version
  void setVersion(
		  const DOMString& newVersion);

  //! Adopts another node
  Node* adoptNode(
		  Node* source);

  //! Sets the base URI
  void setBaseURI(
		  const DOMString *const baseURI);

private :

  Node* myDoctype;                              //!< The document type of the Document
  Node* myDocumentElement;                      //!< The document element
  const DOMImplementation* myDOMImplementation; //!< The DOM implementation
  mutable DOMKey nextKey;                       //!< The next key
  DOMString myNamespaceURI;                     //!< The namespace URI for the document
  list<Node*> myElementList;                    //!< The document element list
  KissNodeList myElementNodeList;               //!< The document element node list
  DOMString myActualEncoding;                   //!< The actual encoding
  DOMString myEncoding;                         //!< The encoding
  DOMString myVersion;                          //!< The document(?) version
  bool myStandalone;                            //!< Whether or not the document is standalone
  bool myStrictErrorChecking;                   //!< Whether or not strict error checking is on
};

// ******************************************************************************
// ******************************************************************************
//	KissElement
// ******************************************************************************
// ******************************************************************************

//! Element class in KISS DOM implementation
class KissElement : public KissNode,
		    public Element {

public :

  //! Override of KissNode::nodeType()
  unsigned long nodeType() const;

  //! Override of KissNode::attributes()
  const NamedNodeMap* attributes() const;

  //! Override of KissNode::setOwnerDocument()
  void setOwnerDocument(
			const Document *const newOwnerDocument);

  //! Implementation of Node::cloneNode()
  Node* cloneNode(
		  const bool& deep) const;

  //! Override of KissNode::namespaceURI()
  const DOMString* namespaceURI() const;

  //! Override of KissNode::prefix()
  const DOMString* prefix() const;

  //! Override of KissNode::setPrefix()
  void setPrefix(
		 const DOMString& newPrefix);

  //! Override of KissNode::localName()
  const DOMString* localName() const;

  //! Override of KissNode::hasAttributes()
  bool hasAttributes() const;

  //! Override of KissNode::normalizeNS()
  void normalizeNS();

  //! Constructor for KissElement object
  KissElement(
	      const Document *const yourOwnerDocument,
	      Node *const yourParentNode,
	      const DOMString& yourNamespaceURI,
	      const DOMString& yourTagName);

  //! Destructor
  ~KissElement();

  //! Gets the tag name
  const DOMString* tagName() const;

  //! Gets the attributes of a tag
  const DOMString* getAttribute(
				const DOMString& name) const;

  //! Sets the attributes of a tag
  void setAttribute(
		    const DOMString& name,
		    const DOMString& value);

  //! Removes the attributes of a tag
  void removeAttribute(
		       const DOMString& name);

  //! Gets the attribute's node (I think)
  Attr* getAttributeNode(
			 const DOMString& name);

  //! Sets the attribute's node (I think)
  Attr* setAttributeNode(
			 Attr& newAttr);

  //! Removes the attribute's node (I think)
  Attr* removeAttributeNode(
			    Attr& oldAttr) ;

  //! Gets the elements by tag name
  const NodeList* getElementsByTagName(
				       const DOMString& name,
				       const bool& deep) const;

  // Introduced in DOM Level 2:

  //! Gets the attribute's namespace
  const DOMString* getAttributeNS(
				  const DOMString& namespaceURI,
				  const DOMString& localname) const;

  //! Sets the attribute's namespace
  void setAttributeNS(
		      const DOMString& namespaceURI,
		      const DOMString& qualifiedName,
		      const DOMString& value);

  //! Removes the attribute's namespace
  void removeAttributeNS(
			 const DOMString& namespaceURI,
			 const DOMString& localname);

  //! Gets the attribute node's namespace
  Attr* getAttributeNodeNS(
			   const DOMString& namespaceURI,
			   const DOMString& localname);

  //! Sets the attribute node's namespace
  Attr* setAttributeNodeNS(
			   Attr& newAttr);

  //! Gets the elements by the tag's namespace
  NodeList* getElementsByTagNameNS(
				   const DOMString& namespaceURI,
				   const DOMString& localname);

  //! Returns true if element/tag/object/thing has attribute
  bool hasAttribute(
		    const DOMString& name) const;

  //! Returns true if element/tag/object/thing has attribute namespace
  bool hasAttributeNS(
		      const DOMString& namespaceURI,
		      const DOMString& localname) const;	

private :

  KissNamedNodeMap myAttributesMap;  //!< The node map of attributes of the element
  KissNodeList myElementNodeList;    //!< The element node list
  const DOMString myNamespaceURI;    //!< The namespace URI of the element

  DOMString myPrefix;                //!< The prefix of the current element
  DOMString myLocalname;             //!< The local name of the current element
  mutable list<Node*> myElementList; //!< The list of (sub?)elements

  //! Performs a non-default namespace prefix lookup
  const DOMString* lookupNamespacePrefixNonDefault(
						   const DOMString& namespaceURI) const;

};

// ******************************************************************************
// ******************************************************************************
//	KissAttr
// ******************************************************************************
// ******************************************************************************

//! Attribute class for the KISS DOM implementation
class KissAttr :	public KissNode,
			public Attr {

public :


  //! Override of KissNode::nodeValue()
  const DOMString* nodeValue() const;

  //! Override of Node::setNodeValue()
  void setNodeValue(
		    const DOMString& newNodeValue);

  //! Override of KissNode::nodeType()
  unsigned long nodeType() const;

  //! Override of KissNode::setParentNode()
  void setParentNode(
		     Node* newParentNode);

  //! Implementation of Node::cloneNode()
  Node* cloneNode(
		  const bool& deep) const;

  //! Override of KissNode::namespaceURI()
  const DOMString* namespaceURI() const;

  //! Override of KissNode::prefix()
  const DOMString* prefix() const;

  //! Override of KissNode::setPrefix()
  void setPrefix(
		 const DOMString& newPrefix);

  //! Override of KissNode::localName()
  const DOMString* localName() const;

  //! Override of KissNode::compareDocumentOrder()
  Node::DocumentOrder compareDocumentOrder(
					   const Node* other) const;

  //! Override of KissNode::compareTreePosition()
  TreePosition compareTreePosition(
				   const Node* other) const;

  //! Override of KissNode::checkChildAddingConstraints2()
  void checkChildAddingConstraints2(const Node* newChild) const;

  //! Constructor of KissAttr object
  KissAttr(
	   const Document *const yourOwnerDocument,
	   const Element *const yourOwnerElement,			
	   const DOMString& yourNamespaceURI,
	   const DOMString& yourAttributeName,
	   const bool& isSpecified);

  //! Destructor
  ~KissAttr();

  //! Obtains name of something
  const DOMString* name() const;

  //! Tells if this attribute is specified in the original document
  bool specified() const;

  //! Sets the specified boolean
  void setSpecified(
		    const bool& newSpecified);

  //! Obtains the value of something
  const DOMString* value() const;

  //! Sets the value of something
  void setValue(
		const DOMString& newValue);

  // Introduced in DOM Level 2:

  //! Obtains the owner element of an xml element
  const Element* ownerElement() const;

  //! Sets the owner element in xml
  void setOwnerElement(
		       const Element* newOwnerElement);

private :

  const Element*  myOwnerElement;   //!< The element owning the attribute
  const DOMString myNamespaceURI;   //!< The attribute namespace URI
  DOMString       myPrefix;         //!< The attribute prefix
  DOMString       myLocalname;      //!< The attribute local name
  DOMString       myAttributeValue; //!< The attribute value
  bool            amSpecified;      //!< Whether or not the attribute is specified
};

// ******************************************************************************
// ******************************************************************************
//	KissCharacterData
// ******************************************************************************
// ******************************************************************************

//! Character data class for KISS DOM implementation
class KissCharacterData :	public KissNode,
				public virtual CharacterData {

public :

  //! Override of KissNode::nodeValue()
  const DOMString* nodeValue() const;

  //! Override of Node::setNodeValue()
  void setNodeValue(
		    const DOMString& newNodeValue);

  //! Override of KissNode::insertBefore()
  Node* insertBefore(
		     Node* newChild,
		     Node* refChild);

  //! Override of KissNode::replaceChild()
  Node* replaceChild(
		     Node* newChild,
		     Node* oldChild);

  //! Override of KissNode::removeChild()
  Node* removeChild(
		    Node* oldChild);

  //! Override of KissNode::appendChild()
  Node* appendChild(
		    Node* newChild);

  //! Override of KissNode::setTextContent()
  void setTextContent(
		      const DOMString& newTextContent);

  //! Constructor of KissCharacterData object
  KissCharacterData(
		    const Document *const yourOwnerDocument,
		    Node *const yourParentNode,
		    const DOMString& yourNodeName,
		    const DOMString& yourData);

  //! Destructor
  ~KissCharacterData();

  //! Obtains the character data string
  const DOMString* data() const;

  //! Returns the length of the data (I believe)
  unsigned long length() const;

  //! Obtains the substring data
  const DOMString* substringData(
				 unsigned long offset,
				 unsigned long count) const;

  //! Appends data to a string
  void appendData(
		  const DOMString& arg);

  //! Inserts data into a string
  void insertData(
		  unsigned long offset,
		  const DOMString& arg);

  //! Deletes data from a string
  void deleteData(
		  unsigned long offset,
		  unsigned long count);

  //! Replaces data in a string
  void replaceData(
		   unsigned long offset,
		   unsigned long count,
		   const DOMString& arg);

  //! Sets a data string
  void setData(
	       const DOMString& newData);

private :

  DOMString myData;                  //!< The data of the character string
  mutable DOMString mySubStringData; //!< The substring data of the character string
};

// ******************************************************************************
// ******************************************************************************
//	KissTextInt
// ******************************************************************************
// ******************************************************************************

//! Text int (?) class in KISS DOM implementation
class KissTextInt : public KissCharacterData,
		    public virtual Text {

public :

  //! Constructor of KissTextInt object
  KissTextInt(
	      const Document *const yourOwnerDocument,
	      Node *const yourParentNode,
	      const DOMString& yourNodeName,
	      const DOMString& yourTextContent);

  //! Destructor
  ~KissTextInt();

  //! Splits text at offset
  Text* splitText(
		  unsigned long offset);

  // Introduced in DOM Level 3:

  //! Determines if whitespace is in the element content
  bool isWhiteSpaceInElementContent() const;

  //! Returns the whole text as a DOMString
  const DOMString* wholeText() const;

  //! Replaces the whole text
  Text* replaceWholeText(
			 const DOMString& content);

private :

  mutable DOMString myWholeTextString;  //!< The whole text string
};

// ******************************************************************************
// ******************************************************************************
//	KissText
// ******************************************************************************
// ******************************************************************************

//! Text class in KISS DOM implementation
class KissText : public KissTextInt {

public :

  //! Override of KissNode::nodeType()
  unsigned long nodeType() const;
	
  //! Implementation of Node::cloneNode()
  Node* cloneNode(
		  const bool& deep) const;

  //! Constructor of KissText object
  KissText(
	   const Document *const yourOwnerDocument,
	   Node *const yourParentNode,
	   const DOMString& yourTextContent);

  //! Destructor
  ~KissText();

};

// ******************************************************************************
// ******************************************************************************
//	KissDocumentFragment
// ******************************************************************************
// ******************************************************************************

//! Document fragment class in KISS DOM implementation
class KissDocumentFragment :	public KissNode,
				public DocumentFragment {

public :

  //! Override of KissNode::nodeType()
  unsigned long nodeType() const;

  //! Implementation of Node::cloneNode()
  Node* cloneNode(
		  const bool& deep) const;

  //! Override of KissNode::normalizeNS()
  void normalizeNS();

  //! Constructor for KissDocumentFragment object
  KissDocumentFragment(
		       const Document *const yourOwnerDocument);

  //! Destructor
  ~KissDocumentFragment();

};

// ******************************************************************************
// ******************************************************************************
//	KissComment
// ******************************************************************************
// ******************************************************************************

//! Comment class in KISS DOM implementation
class KissComment : public KissCharacterData,
		    public Comment {

public :

  //! Override of KissNode::nodeType()
  unsigned long nodeType() const;

  //! Implementation of Node::cloneNode()
  Node* cloneNode(
		  const bool& deep) const;

  //! Constructor of KissComment object
  KissComment(
	      const Document *const yourOwnerDocument,
	      Node *const yourParentNode,
	      const DOMString& yourTextContent);

  //! Destructor
  ~KissComment();

};

// ******************************************************************************
// ******************************************************************************
// ******************************************************************************
//	The Extended Interfaces
// ******************************************************************************
// ******************************************************************************
// ******************************************************************************


// ******************************************************************************
// ******************************************************************************
//	KissEntity
// ******************************************************************************
// ******************************************************************************

//! Entity class in KISS DOM implementation
class KissEntity :	public KissNode,
			public Entity {

public :

  //! Override of KissNode::nodeType()
  unsigned long nodeType() const;

  //! Override of KissNode::nodeValue()
  const DOMString* nodeValue() const;

  //! Override of KissNode::setParentNode()
  void setParentNode(
		     Node* newParentNode);

  //! Implementation of Node::cloneNode()
  Node* cloneNode(
		  const bool& deep) const;

  //! Override of KissNode::compareDocumentOrder()
  Node::DocumentOrder compareDocumentOrder(
					   const Node* other) const;

  //! Override of KissNode::compareTreePosition()
  TreePosition compareTreePosition(
				   const Node* other) const;

  //! Constructor of KissEntity object
  KissEntity(
	     const Document *const yourOwnerDocument,
	     const DOMString& yourEntityName,
	     const DOMString& yourPublicID,
	     const DOMString& yourSystemID,
	     const DOMString& yourNotationName,
	     const DOMString& yourEntityValue);

  //! Desctructor
  ~KissEntity();

  //! Obtains public id string
  const DOMString* publicId() const;

  //! Obtains system id string
  const DOMString* systemId() const;

  //! Obtains notation name (whatever that is)
  const DOMString* notationName() const;

  // Introduced in DOM Level 3:

  //! Obtains actual encoding string
  const DOMString* actualEncoding() const;

  //! Sets actual enconding
  void setActualEncoding(
			 const DOMString& newActualEncoding);

  //! Obtains encoding
  const DOMString* encoding() const;

  //! Sets encoding
  void setEncoding(
		   const DOMString& newEncoding);

  //! Obtains version string
  const DOMString* version() const;

  //! Sets version
  void setVersion(
		  const DOMString& newVersion);

private :

  const DOMString myPublicId;       //!< Entity public id
  const DOMString mySystemId;       //!< Entity system id
  const DOMString myNotationName;   //!< Entity notation name
  const DOMString myEntityValue;    //!< Entity value
  DOMString       myActualEncoding; //!< Actual encoding of entity
  DOMString       myEncoding;       //!< Encoding of entity
  DOMString       myVersion;        //!< Version of entity (?)
};

// ******************************************************************************
// ******************************************************************************
//	KissDocumentType
// ******************************************************************************
// ******************************************************************************

//! Document type class in KISS DOM implementation
class KissDocumentType :	public KissNode,
				public DocumentType {

public :

  //! Override of KissNode::nodeType()
  unsigned long nodeType() const;

  //! Override of KissNode::insertBefore()
  Node* insertBefore(
		     Node* newChild,
		     Node* refChild);

  //! Override of KissNode::replaceChild()
  Node* replaceChild(
		     Node* newChild,
		     Node* oldChild);

  //! Override of KissNode::removeChild()
  Node* removeChild(
		    Node* oldChild);

  //! Override of KissNode::appendChild()
  Node* appendChild(
		    Node* newChild);

  //! Implementation of Node::cloneNode()
  Node* cloneNode(
		  const bool& deep) const;

  //! Override of KissNode::setTextContent()
  void setTextContent(
		      const DOMString& newTextContent);

  //! Constructor for KissDocumentType object
  KissDocumentType(
		   const DOMString& yourDocumentTypeName,
		   const DOMString& yourPublicId,
		   const DOMString& yourSystemId);

  //! Destructor
  ~KissDocumentType();

  //! Gets the name (of what?) as a DOMString
  const DOMString* name() const;

  //! Returns a named node map of general entities both external and internal declared in the DTD
  const NamedNodeMap* entities() const;

  //! Returns a named node mape of notations declared in the DTD
  const NamedNodeMap* notations() const;

  // Introduced in DOM Level 2:

  //! The public identifier of the external subset
  const DOMString* publicId() const;

  //! The system identifier of the external subset
  const DOMString* systemId() const;

  //! The internal subset as a string, or null if there is none
  const DOMString* internalSubset() const;

private :

  DOMString        myPublicId;      //!< The public id of the document type
  DOMString        mySystemId;      //!< The system id of the document type
  KissNamedNodeMap myEntitiesMap;   //!< The entities map of the document type
  KissNamedNodeMap myNotationsMap;  //!< The notation map of the document type
};

// ******************************************************************************
// ******************************************************************************
//	KissProcessingInstruction
// ******************************************************************************
// ******************************************************************************

//! Processing instruction class in KISS DOM implementation
class KissProcessingInstruction :	public KissNode,
					public ProcessingInstruction {

public :

  //! Override of KissNode::nodeValue()
  const DOMString* nodeValue() const;

  //! Override of Node::setNodeValue()
  void setNodeValue(
		    const DOMString& newNodeValue);

  //! Override of KissNode::nodeType()
  unsigned long nodeType() const;

  //! Override of KissNode::insertBefore()
  Node* insertBefore(
		     Node* newChild,
		     Node* refChild);

  //! Override of KissNode::replaceChild()
  Node* replaceChild(
		     Node* newChild,
		     Node* oldChild);

  //! Override of KissNode::removeChild()
  Node* removeChild(
		    Node* oldChild);

  //! Override of KissNode::appendChild()
  Node* appendChild(
		    Node* newChild);

  //! Implementation of Node::cloneNode()
  Node* cloneNode(
		  const bool& deep) const;

  //! Override of KissNode::setTextContent()
  void setTextContent(
		      const DOMString& newTextContent);

  //! Constructor of KissProcessingInstruction object
  KissProcessingInstruction(
			    const Document *const yourOwnerDocument,
			    Node *const yourParentNode,
			    const DOMString& yourTarget,
			    const DOMString& yourContent);

  //! Destructor
  ~KissProcessingInstruction();

  //! Returns target as DOMString
  const DOMString* target() const;

  //! Returns data as DOMString
  const DOMString* data() const;

  //! Sets data
  void setData(
	       const DOMString& newData);

private :

  DOMString myContent;  //!< Processing instruction content
};

// ******************************************************************************
// ******************************************************************************
//	KissNotation
// ******************************************************************************
// ******************************************************************************

//! Notation class in KISS DOM implementation
class KissNotation :	public KissNode,
			public Notation {

public :

  //! Override of KissNode::nodeType()
  unsigned long nodeType() const;

  //! Override of KissNode::setParentNode()
  void setParentNode(
		     Node* newParentNode);

  //! Override of KissNode::insertBefore()
  Node* insertBefore(
		     Node* newChild,
		     Node* refChild);

  //! Override of KissNode::replaceChild()
  Node* replaceChild(
		     Node* newChild,
		     Node* oldChild);

  //! Override of KissNode::removeChild()
  Node* removeChild(
		    Node* oldChild);

  //! Override of KissNode::appendChild()
  Node* appendChild(
		    Node* newChild);

  //! Implementation of Node::cloneNode()
  Node* cloneNode(
		  const bool& deep) const;

  //! Override of KissNode::compareDocumentOrder()
  Node::DocumentOrder compareDocumentOrder(
					   const Node* other) const;

  //! Override of KissNode::compareTreePosition()
  TreePosition compareTreePosition(
				   const Node* other) const;

  //! Override of KissNode::setTextContent()
  void setTextContent(
		      const DOMString& newTextContent);

  //! Constructor of KissNotation object
  KissNotation(
	       const DOMString& yourNotationName,
	       const DOMString& yourPublicId,
	       const DOMString& yourSystemId);

  //! Destructor
  ~KissNotation();

  //! Returns the public id as a DOMString
  const DOMString* publicId() const;

  //! Returns the system id as a DOMString
  const DOMString* systemId() const;
	
private :

  const DOMString myNotationName;   //!< The name of the notation
  const DOMString myPublicId;       //!< Notation public id
  const DOMString mySystemId;       //!< Notation system id
};

// ******************************************************************************
// ******************************************************************************
//	KissEntityReference
// ******************************************************************************
// ******************************************************************************

//! Entity reference class in KISS DOM implementation
class KissEntityReference : public KissNode,
			    public EntityReference {

public :

  //! Override of KissNode::nodeType()
  unsigned long nodeType() const;

  //! Implementation of Node::cloneNode()
  Node* cloneNode(
		  const bool& deep) const;

  //! Constructor for KissEntityReference object
  KissEntityReference(
		      const Document *const yourOwnerDocument,
		      Node *const yourParentNode,
		      const DOMString& yourEntityNameReference);

  //! Destructor
  ~KissEntityReference();
};

// ******************************************************************************
// ******************************************************************************
//	KissCDATASection
// ******************************************************************************
// ******************************************************************************

//! CDATA section class in KISS DOM implementation
class KissCDATASection :	public KissTextInt,
				public CDATASection {

public :

  //! Override of KissNode::nodeType()
  unsigned long nodeType() const;

  //! Implementation of Node::cloneNode()
  Node* cloneNode(
		  const bool& deep) const;

  //! Constructor of the KissCDATASection object
  KissCDATASection(
		   const Document *const yourOwnerDocument,
		   Node *const yourParentNode,
		   const DOMString& yourCDATASectionContent);

  //! Destructor
  ~KissCDATASection();
};

#define KISSDOM

