/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001-2004 Joop Stakenborg <pg4i@amsat.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* callbacks.c
 *
 * Callbacks for the preferences dialog and the dialogs called from here. 
 * For callbacks related to the widgets in the main window, see the files
 * called callbacks_<widgetname>.[c|h].
 */


#include <stdlib.h>
#include <gtk/gtk.h>
#include <string.h>

#if WANT_HAMLIB
#	include <hamlib/rig.h>
#	include "hamlib-utils.h"
#endif

#include "callbacks_preferencesdialog.h"
#include "gui_utils.h"
#include "support.h"
#include "types.h"
#include "log.h"
#include "utils.h"

extern GtkWidget *mainwindow;
extern preferencestype preferences;
extern statetype state;
extern GtkWidget *preferencesdialog;
extern gint clocktimer, savetimer;
extern GList *logwindowlist;


#if WANT_HAMLIB
/* return value for setting widgets */
static gint
whichhamlibwidgets (gboolean frequency, gboolean smeter)
{
	if (frequency && smeter)
		return (4);
	else if (frequency && (!smeter))
		return (2);
	else if ((!frequency) && smeter)
		return (3);
	else
		return (1);
}
#endif

/* check save value, if autosave value larger than 0 start timer */
static void
set_autosave (gint value, gint saving)
{
	if (saving != 1 && preferences.saving == 1)
	{
		g_source_remove (savetimer);
		savetimer = -1;
	}
	else if (saving == 1 && preferences.saving != 1)
	{
		savetimer =
			g_timeout_add (value * 60 * 1000, (GSourceFunc) autosave, NULL);
	}
	else if (value != preferences.autosave)
	{			/* new value, stop and start timer */
		g_source_remove (savetimer);
		savetimer =
			g_timeout_add (value * 60 * 1000, (GSourceFunc) autosave, NULL);
	}
	preferences.autosave = value;
	preferences.saving = saving;
}

/* set the theme if changed */
static void
set_theme (gchar * color, gchar * font, gboolean alsoforlog)
{
	GtkWidget *clocklabel, *qsoframe, *dxccframe, *locatorframe;
	gint i;
	logtype *logwindow;
	gint logcolor;
#if WANT_HAMLIB
	GtkWidget *frequencylabel;
#endif

	logcolor = alsoforlog ? 1 : 0;

	if (g_ascii_strcasecmp (color, preferences.themecolor) ||
			g_ascii_strcasecmp (font, preferences.themefont))
	{
		clocklabel = lookup_widget (mainwindow, "clocklabel");
		setlabelstyle (clocklabel, color, font);
#if WANT_HAMLIB
		frequencylabel = lookup_widget (mainwindow, "frequencylabel");
		setlabelstyle (frequencylabel, color, font);
#endif
		qsoframe = lookup_widget (mainwindow, "qsoframe");
		dxccframe = lookup_widget (mainwindow, "dxccframe");
		locatorframe = lookup_widget (mainwindow, "locatorframe");
		setframestyle (qsoframe, color, font);
		setframestyle (dxccframe, color, font);
		setframestyle (locatorframe, color, font);
		for (i = 0; i < g_list_length (logwindowlist); i++)
		{
			logwindow = g_list_nth_data (logwindowlist, i);
			setlabelstyle (logwindow->label, color, font);
		}
		preferences.themecolor = color;
		preferences.themefont = g_strdup (font);
	}

	preferences.logcolor = logcolor;
}

/* set savepath if changed */
static void
set_path (gchar * pathstr)
{
	if (strlen (pathstr) > 0)
	{
		if (g_ascii_strcasecmp (preferences.savedir, pathstr))
			preferences.savedir = g_strdup (pathstr);
	}
}

/* set logs to load if changed */
static void
set_logstoload (gchar * logs)
{
	gchar **logsplit;

	if (g_ascii_strcasecmp (logs, preferences.logstoload) != 0)
	{
		if (strlen (logs) == 0)
			logs = g_strdup ("*");
		logsplit = g_strsplit (logs, ",", -1);	/* delete spaces */
		deletespaces (logsplit);
		preferences.logstoload = g_strjoinv (",", logsplit);
		g_strfreev (logsplit);
	}
}

/* set QTH locator */
static void
set_qthlocator (gchar * locator)
{
	if (g_ascii_strcasecmp (locator, preferences.locator) != 0)
	{
		if (strlen (locator) == 0)
			locator = g_strdup ("AA00AA");
		preferences.locator = g_strdup (locator);
	}
}

/* set callsign */
static void
set_callsign (gchar * callsign)
{
	if (g_ascii_strcasecmp (callsign, preferences.callsign) != 0)
	{
		if (strlen (callsign) == 0)
			callsign = g_strdup ("N0CALL");
		preferences.callsign = g_strdup (callsign);
	}
}

/* enable/disable clock */
static void
set_clock (gboolean on)
{
	GtkWidget *clockhandlebox;

	clockhandlebox = lookup_widget (mainwindow, "clockhandlebox");
	if (on)
	{			/* yes, we want clock */
		if (preferences.clock == 0)
			clocktimer = g_timeout_add (1000, (GSourceFunc) updateclock, NULL);
		gtk_widget_show (clockhandlebox);
		preferences.clock = 1;
	}
	else
	{			/* no clock */
		if (preferences.clock > 0)
			g_source_remove (clocktimer);
		clocktimer = -1;
		gtk_widget_hide (clockhandlebox);
		preferences.clock = 0;
	}
}

#if WANT_HAMLIB
/* check if hamlib has changed */
static gboolean hamlib_changed (gint hamlibwidgets, gchar *radio, gchar *device, 
	gint polltime, gchar *rigconf)
{

	if ((preferences.hamlib == 0) && (hamlibwidgets > 0))
		return TRUE;
	else if (g_ascii_strcasecmp (preferences.rigconf, rigconf))
		return TRUE;
	else if (g_ascii_strcasecmp (preferences.radio, radio))
		return TRUE;
	else if (g_ascii_strcasecmp (preferences.device, device))
		return TRUE;
	else if (preferences.polltime != polltime)
		return TRUE;
	else 
	/* FALSE is returned when nothing is	 *
	 * changed or hamlib should be stopped */
		return FALSE;
}
#endif

/* ok button clicked */
void
on_preferencesokbutton_clicked (GtkButton * button, gpointer user_data)
{
	GtkWidget *latentry, *longentry, *NScombo, *EWcombo,
		*unitscombo, *bandsentry, *bandoptionmenu, *modesentry,
		*modeoptionmenu, *bandsradiobutton1, *modesradiobutton1,
		*clockcheckbutton, *examplelabel, *autosavecombo,
		*themecheckbutton, *pathentry, *logsentry, *qthlocatorentry,
		*callsignentry, *fontentry, *bandentry, *modeentry,
		*savecheckbutton, *autosavecheckbutton, *typeaheadfindcheckbutton, 
		*remoteradiobutton1;
	gchar *temp, *font;
	gboolean toggletrue, logcoloryesno;
	gint autosavevalue, saving;
#if WANT_HAMLIB
	GtkWidget *hamlibcheckbutton, *radiocombo, *devicecombo,
		*frequencycheckbutton, *digitscombo, *smetercheckbutton,
		*pollcombo, *pollingcheckbutton, *rigsetconfentry;
	gchar *radio = NULL, *device = NULL, *digits = NULL, *pollstr = NULL,
		*rigconf = NULL;
	gboolean hamlibyes, frequencyyesno, smeteryesno, pollingyes;
	gint hamlibwidgets = 0;
	gboolean hamlibresult;
#endif

	/* check autosave value */
	autosavecheckbutton =
		lookup_widget (preferencesdialog, "autosavecheckbutton");
	savecheckbutton =
		lookup_widget (preferencesdialog, "savecheckbutton");
	if (gtk_toggle_button_get_active
			(GTK_TOGGLE_BUTTON (autosavecheckbutton)))
		saving = 1;
	else if (gtk_toggle_button_get_active
		 (GTK_TOGGLE_BUTTON (savecheckbutton)))
		saving = 2;
	else
		saving = 0;
	autosavecombo =
		lookup_widget (preferencesdialog, "autosavecombo");
	temp = gtk_editable_get_chars (GTK_EDITABLE (GTK_BIN(autosavecombo)->child),
		0, -1);
	autosavevalue = atoi (temp);
	set_autosave (autosavevalue, saving);

	/* check type ahead find */
	typeaheadfindcheckbutton = lookup_widget
		(preferencesdialog, "typeaheadfindcheckbutton");
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
						(typeaheadfindcheckbutton)))
		preferences.typeaheadfind = 1;
	else
		preferences.typeaheadfind = 0;

	/* remote logging */
	remoteradiobutton1 =
		lookup_widget (preferencesdialog, "remoteradiobutton1");
	if (gtk_toggle_button_get_active
			(GTK_TOGGLE_BUTTON (remoteradiobutton1)))
		preferences.remoteadding = 1;
	else
		preferences.remoteadding = 2;

	/* apply theme color */
	fontentry = lookup_widget (preferencesdialog, "fontentry");
	examplelabel = lookup_widget (preferencesdialog, "examplelabel");
	themecheckbutton =
		lookup_widget (preferencesdialog, "themecheckbutton");
	logcoloryesno =
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
								(themecheckbutton));
	temp = getlabelcolor (examplelabel);
	font = gtk_editable_get_chars (GTK_EDITABLE (fontentry), 0, -1);
	set_theme (temp, font, logcoloryesno);
	g_free (font);

	/* get path for the logs */
	pathentry = lookup_widget (preferencesdialog, "pathentry");
	temp = gtk_editable_get_chars (GTK_EDITABLE (pathentry), 0, -1);
	set_path (temp);

	/* get logs to load */
	logsentry = lookup_widget (preferencesdialog, "logsentry");
	temp = gtk_editable_get_chars (GTK_EDITABLE (logsentry), 0, -1);
	set_logstoload (temp);

	/* locator */
	qthlocatorentry =
		lookup_widget (preferencesdialog, "qthlocatorentry");
	temp = gtk_editable_get_chars (GTK_EDITABLE (qthlocatorentry), 0, -1);
	set_qthlocator (temp);

	/* callsign */
	callsignentry = lookup_widget (preferencesdialog, "callsignentry");
	temp = gtk_editable_get_chars (GTK_EDITABLE (callsignentry), 0, -1);
	set_callsign (temp);

	/* set latitude, longitude and units */
	latentry = lookup_widget (preferencesdialog, "latentry");
	longentry = lookup_widget (preferencesdialog, "longentry");
	NScombo = lookup_widget (preferencesdialog, "NScombo");
	EWcombo = lookup_widget (preferencesdialog, "EWcombo");
	unitscombo = lookup_widget (preferencesdialog, "unitscombo");
	temp = gtk_editable_get_chars (GTK_EDITABLE (latentry), 0, -1);
	g_strstrip (temp);
	if (strlen (temp) == 0)
		preferences.latitude = 0.0;
	else
		preferences.latitude = atof (temp);
	temp = gtk_editable_get_chars (GTK_EDITABLE (longentry), 0, -1);
	g_strstrip (temp);
	if (strlen (temp) == 0)
		preferences.longitude = 0.0;
	else
		preferences.longitude = atof (temp);

	preferences.NS = gtk_combo_box_get_active (GTK_COMBO_BOX (NScombo)) + 1;
	preferences.EW = gtk_combo_box_get_active (GTK_COMBO_BOX (EWcombo)) + 1;
	preferences.units = gtk_combo_box_get_active 
		(GTK_COMBO_BOX (unitscombo))  == 1 ? 0 : 1;

	state.mylocation = setlocation (preferences.latitude, preferences.NS,
		preferences.longitude, preferences.EW);

	/* get the comma seperated list of modes and assign 
	 * to the optionmenu */
	modesentry = lookup_widget (preferencesdialog, "modesentry");
	temp = gtk_editable_get_chars (GTK_EDITABLE (modesentry), 0, -1);
	makemodeoptionmenu (temp);
	preferences.modes = g_strdup (temp);

	/* get the comma seperated list of bands and assign to the optionmenu */
	bandsentry = lookup_widget (preferencesdialog, "bandsentry");
	temp = gtk_editable_get_chars (GTK_EDITABLE (bandsentry), 0, -1);
	makebandoptionmenu (temp);
	preferences.bands = g_strdup (temp);

	/* set visibility of widgets in QSO frame depending on 
	 * state of toggle buttons */
	bandsradiobutton1 =
		lookup_widget (preferencesdialog, "bandsradiobutton1");
	bandentry = lookup_widget (mainwindow, "bandentry");
	bandoptionmenu = lookup_widget (mainwindow, "bandoptionmenu");
	toggletrue =
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
								(bandsradiobutton1));
	if (toggletrue)
	{
		preferences.bandseditbox = 1;
		gtk_widget_hide (bandoptionmenu);
		gtk_widget_show (bandentry);
	}
	else
	{
		preferences.bandseditbox = 0;
		gtk_widget_hide (bandentry);
		gtk_widget_show (bandoptionmenu);
	}
	modesradiobutton1 =
		lookup_widget (preferencesdialog, "modesradiobutton1");
	modeentry = lookup_widget (mainwindow, "modeentry");
	modeoptionmenu = lookup_widget (mainwindow, "modeoptionmenu");
	toggletrue =
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
								(modesradiobutton1));
	if (toggletrue)
	{
		preferences.modeseditbox = 1;
		gtk_widget_hide (modeoptionmenu);
		gtk_widget_show (modeentry);
	}
	else
	{
		preferences.modeseditbox = 0;
		gtk_widget_hide (modeentry);
		gtk_widget_show (modeoptionmenu);
	}

/* hamlib */
#if WANT_HAMLIB
	hamlibcheckbutton =
		lookup_widget (preferencesdialog, "hamlibcheckbutton");
	frequencycheckbutton =
		lookup_widget (preferencesdialog, "frequencycheckbutton");
	smetercheckbutton =
		lookup_widget (preferencesdialog, "smetercheckbutton");
	pollingcheckbutton =
		lookup_widget (preferencesdialog, "pollingcheckbutton");
	radiocombo = lookup_widget (preferencesdialog, "radiocombo");
	devicecombo = lookup_widget (preferencesdialog, "devicecombo");
	digitscombo = lookup_widget (preferencesdialog, "digitscombo");
	pollcombo = lookup_widget (preferencesdialog, "pollcombo");
	rigsetconfentry = lookup_widget (preferencesdialog, "rigsetconfentry");

	hamlibyes = gtk_toggle_button_get_active
		(GTK_TOGGLE_BUTTON (hamlibcheckbutton));
	frequencyyesno = gtk_toggle_button_get_active
		(GTK_TOGGLE_BUTTON (frequencycheckbutton));
	smeteryesno = gtk_toggle_button_get_active
		(GTK_TOGGLE_BUTTON (smetercheckbutton));
	pollingyes = gtk_toggle_button_get_active
		(GTK_TOGGLE_BUTTON (pollingcheckbutton));
	radio = gtk_editable_get_chars
		(GTK_EDITABLE (GTK_BIN(radiocombo)->child), 0,	-1);
	device = gtk_editable_get_chars
		(GTK_EDITABLE (GTK_BIN(devicecombo)->child), 0, -1);
	digits = gtk_editable_get_chars
		(GTK_EDITABLE (GTK_BIN(digitscombo)->child), 0, -1);
	if (pollingyes)
		pollstr = gtk_editable_get_chars 
			(GTK_EDITABLE (GTK_BIN(pollcombo)->child), 0, -1);
	else
		pollstr = g_strdup ("0");
	rigconf = gtk_editable_get_chars (GTK_EDITABLE (rigsetconfentry), 0, -1);
	if (strlen(rigconf) == 0) rigconf = g_strdup ("?");

	if (hamlibyes)
		hamlibwidgets =	whichhamlibwidgets (frequencyyesno, smeteryesno);
	else
		hamlibwidgets = 0;

	if (smeteryesno)
		sethamlibwidgets (hamlibwidgets, TRUE);
	else
		sethamlibwidgets (hamlibwidgets, FALSE);

	/* hamlib should be started or restarted because settings have changed */
	if (hamlib_changed (hamlibwidgets, radio, device, atoi(pollstr), rigconf))
	{
		if (preferences.hamlib > 0) stop_hamlib ();
		preferences.rigconf = g_strdup (rigconf);
		hamlibresult = start_hamlib (radio, device, RIG_DEBUG_NONE, atoi (pollstr));
		preferences.radio = g_strdup (radio);
		preferences.device = g_strdup (device);
		preferences.polltime = atoi (pollstr);
		preferences.round = atoi (digits);
	}
	else
	{
		if (hamlibwidgets == 0)
		{
			/* hamlib disabled, close rig */
			stop_hamlib ();
			preferences.radio = g_strdup ("Dummy");
			preferences.device = g_strdup ("/dev/ttyS1");
			preferences.rigconf = g_strdup ("?");
			preferences.round = 0;
			preferences.polltime = 0;
		}
	}
	preferences.hamlib = hamlibwidgets;

		g_free (digits);
	g_free (pollstr);
	g_free (radio);
	g_free (device);
	g_free (rigconf);
#endif

	/* clock */
	clockcheckbutton =
		lookup_widget (preferencesdialog, "clockcheckbutton");
	toggletrue = gtk_toggle_button_get_active 
		(GTK_TOGGLE_BUTTON (clockcheckbutton));
	set_clock (toggletrue);

	g_free (temp);
	gtk_widget_set_sensitive (mainwindow, 1);
	gtk_widget_destroy (preferencesdialog);
}

/* toggle hamlib widgets in the preferences dialog */
void
on_hamlibcheckbutton_toggled (GtkToggleButton * togglebutton,
						gpointer user_data)
{
	GtkWidget *hamlibframe, *pollingframe, *frequencyframe;
	gboolean toggletrue;

	hamlibframe = lookup_widget (preferencesdialog, "hamlibframe");
	toggletrue = gtk_toggle_button_get_active (togglebutton);
	if (toggletrue)
	{
		gtk_widget_set_sensitive (hamlibframe, 1);
		if (preferences.hamlib == 0)
		{
			pollingframe = lookup_widget (preferencesdialog, "pollingframe");
			gtk_widget_set_sensitive (pollingframe, 0);
			frequencyframe = lookup_widget (preferencesdialog, "frequencyframe");
			gtk_widget_set_sensitive (frequencyframe, 0);
		}
	}
	else
		gtk_widget_set_sensitive (hamlibframe, 0);
}

/* cancel the preferences dialog */
void
on_preferencescancelbutton_clicked (GtkButton * button, gpointer user_data)
{
	gtk_widget_set_sensitive (mainwindow, 1);
	gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}

/* the preferences dialog is deleted */
gboolean
on_preferencesdialog_delete_event (GtkWidget * widget, GdkEvent * event,
					 gpointer user_data)
{
	gtk_widget_set_sensitive (mainwindow, 1);
	return FALSE;
}

/* toggle sensistivity of band widgets in preferences dialog */
void
on_bandsradiobutton_toggled (GtkToggleButton * togglebutton,
					 gpointer user_data)
{
	GtkWidget *bandslabel, *bandsentry;

	bandslabel = lookup_widget (preferencesdialog, "bandslabel");
	bandsentry = lookup_widget (preferencesdialog, "bandsentry");
	switch (atoi (user_data))
	{
	case 1:
		gtk_widget_set_sensitive (bandslabel, 0);
		gtk_widget_set_sensitive (bandsentry, 0);
		break;
	case 2:
		gtk_widget_set_sensitive (bandslabel, 1);
		gtk_widget_set_sensitive (bandsentry, 1);
		break;
	}
}

/* toggle sensistivity of modes widgets in preferences dialog */
void
on_modesradiobutton_toggled (GtkToggleButton * togglebutton,
					 gpointer user_data)
{
	GtkWidget *modeslabel, *modesentry;

	modeslabel = lookup_widget (preferencesdialog, "modeslabel");
	modesentry = lookup_widget (preferencesdialog, "modesentry");
	switch (atoi (user_data))
	{
	case 1:
		gtk_widget_set_sensitive (modeslabel, 0);
		gtk_widget_set_sensitive (modesentry, 0);
		break;
	case 2:
		gtk_widget_set_sensitive (modeslabel, 1);
		gtk_widget_set_sensitive (modesentry, 1);
		break;
	}

}


/* check autosave */
void
on_autosavecheckbutton_toggled (GtkToggleButton * togglebutton,
				gpointer user_data)
{
	gboolean state;
	GtkWidget *savecheckbutton, *autosaveframe;

	state = gtk_toggle_button_get_active (togglebutton);
	savecheckbutton =
		lookup_widget (preferencesdialog, "savecheckbutton");
	if (gtk_toggle_button_get_active
			(GTK_TOGGLE_BUTTON (savecheckbutton)))
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON
								(savecheckbutton), !state);
	autosaveframe = lookup_widget (preferencesdialog, "autosaveframe");
	gtk_widget_set_sensitive (autosaveframe, state);
}

/* check save */
void
on_savecheckbutton_toggled (GtkToggleButton * togglebutton,
					gpointer user_data)
{
	gboolean state;
	GtkWidget *autosavecheckbutton, *autosaveframe;

	state = gtk_toggle_button_get_active (togglebutton);
	autosavecheckbutton =
		lookup_widget (preferencesdialog, "autosavecheckbutton");
	if (gtk_toggle_button_get_active
			(GTK_TOGGLE_BUTTON (autosavecheckbutton)))
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON
								(autosavecheckbutton), !state);
	autosaveframe = lookup_widget (preferencesdialog, "autosaveframe");
	gtk_widget_set_sensitive (autosaveframe, !state);
}

/* check polling */
void
on_pollingcheckbutton_toggled (GtkToggleButton * togglebutton,
						 gpointer user_data)
{
	gboolean state;
	GtkWidget *pollingframe;

	state = gtk_toggle_button_get_active (togglebutton);
	pollingframe = lookup_widget (preferencesdialog, "pollingframe");
	gtk_widget_set_sensitive (pollingframe, state);
}

/* check frequency meter */
void
on_frequencycheckbutton_toggled (GtkToggleButton * togglebutton,
				 gpointer user_data)
{
	gboolean state;
	GtkWidget *frequencyframe;

	state = gtk_toggle_button_get_active (togglebutton);
	frequencyframe = lookup_widget (preferencesdialog, "frequencyframe");
	gtk_widget_set_sensitive (frequencyframe, state);
}


/* end of file */
