/*
 * Copyright (c) 2003 Benedikt Meurer (benedikt.meurer@unix-ag.uni-siegen.de)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* !HAVE_CONFIG_H */

#include <stdlib.h>

#include <xfprinter.h>

#include "inline-icon.h"


XfPrinter *
xfprinter_new(const gchar *alias, const gchar *name)
{
	XfPrinter *pr;

	g_return_val_if_fail(alias != NULL, NULL);
	g_return_val_if_fail(name != NULL, NULL);

	pr = g_new0(XfPrinter, 1);
	pr->alias = g_strdup(alias);
	pr->name = g_strdup(name);
	pr->postscript = FALSE;

	return(pr);
}

XfPrinter *
xfprinter_new_ps(const gchar *alias, const gchar *name)
{
	XfPrinter *pr;

	if ((pr = xfprinter_new(alias, name)) != NULL)
		pr->postscript = TRUE;

	return(pr);
}

void
xfprinter_free(XfPrinter *pr)
{
	g_return_if_fail(pr != NULL);

	if (pr->alias)		g_free(pr->alias);
	if (pr->name)		g_free(pr->name);

	g_free(pr);
}

void
xfprinter_copy(XfPrinter *t, const XfPrinter *s)
{
	g_return_if_fail(s != NULL);
	g_return_if_fail(t != NULL);

	xfprinter_set_alias(t, s->alias);
	xfprinter_set_name(t, s->name);
	xfprinter_set_postscript(t, s->postscript);
}

XfPrinter *
xfprinter_dup(const XfPrinter *s)
{
	XfPrinter *t;

	g_return_val_if_fail(s != NULL, NULL);

	if (s->postscript)
		t = xfprinter_new_ps(s->alias, s->name);
	else
		t = xfprinter_new(s->alias, s->name);

	return(t);
}

GdkPixbuf *
xfprinter_icon(XfPrinter *pr, gboolean is_default)
{
	GdkPixbuf *pb;

	if (!pr->postscript)
		pb = gdk_pixbuf_new_from_inline(-1, printer_lp, FALSE, NULL);
	else
		pb = gdk_pixbuf_new_from_inline(-1, printer_ps, FALSE, NULL);

	return(pb);
}

void
xfprinter_set_alias(XfPrinter *pr, const gchar *alias)
{
	g_return_if_fail(pr != NULL);

	if (pr->alias)
		g_free(pr->alias);

	pr->alias = alias ? g_strdup(alias) : NULL;
}

void
xfprinter_set_name(XfPrinter *pr, const gchar *name)
{
	g_return_if_fail(pr != NULL);

	if (pr->name)
		g_free(pr->name);

	pr->name = name ? g_strdup(name) : NULL;
}

void
xfprinter_set_postscript(XfPrinter *pr, gboolean postscript)
{
	g_return_if_fail(pr != NULL);

	pr->postscript = postscript;
}


