/*
 *
 * Edscott Wilson Garcia 2001-2004 for xfce project.
 *
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/stat.h>

#include <errno.h>
#include <limits.h>
#include <memory.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include <gtk/gtk.h>
#include <glib.h>
#include <libxml/parser.h>
#include <libxml/tree.h>
#include <gmodule.h>


#ifndef S_ISSOCK 
# ifdef S_IFSOCK
# define S_ISSOCK(x) ((x&S_IFMT)==S_IFSOCK)
# else
# define S_ISSOCK(x) FALSE
# endif
#else
#endif

#ifndef S_ISLNK
# ifdef S_IFLNK
# define S_ISLNK(x) ((x&S_IFMT)==S_IFLNK)
# else
# define S_ISLNK(x) FALSE
# endif
#endif


#include <dbh.h>
#define CACHE_DIR "xffm","modules"

typedef struct magic_t {
	char *mimetype;
	char *type;
	char *value;
	char *mask;
	int offset;
	int last_offset;
	int priority;
}magic_t;

G_MODULE_EXPORT
void mime_add(const gchar *sfx, const gchar *command);
G_MODULE_EXPORT
gboolean is_valid_command(const char *cmd_fmt);

G_MODULE_EXPORT
const gchar *mime_typeinfo(const gchar *type);
G_MODULE_EXPORT
const gchar *mime_key_type (const gchar *file);
G_MODULE_EXPORT
const gchar *mime_command(const gchar *file);
G_MODULE_EXPORT
const gchar **mime_apps(const gchar *file);
G_MODULE_EXPORT
const gchar *mk_command_line(const gchar *command_fmt,const gchar *path,gboolean interm,gboolean hold);
G_MODULE_EXPORT
const gchar *mime_get_type(const gchar *file, gboolean try_magic);

#include "applications-module.h"
#include "applications-module.i"

/****************************************************************************/
G_MODULE_EXPORT
const gchar *g_module_check_init(GModule *module){
    xfmime_fun = g_new0 (xfmime_functions,1);
    if (!xfmime_fun) return ("unable to create function structure");
    
    xfmime_fun->mime_command = mime_command;
    xfmime_fun->mime_typeinfo = mime_typeinfo;
    xfmime_fun->mime_key_type = mime_key_type;
    xfmime_fun->mime_get_type = mime_get_type;
    xfmime_fun->mime_apps = mime_apps;
    xfmime_fun->mime_add = mime_add;
    xfmime_fun->mk_command_line = mk_command_line;
    xfmime_fun->is_valid_command = is_valid_command;
    return NULL;
}

G_MODULE_EXPORT
const char *mime_key_type (const gchar *file){
    const mime_t *mime;
    const gchar *m=mimeable_file(file);
    if (!m || strlen(m)) return NULL;
    mime= locate_mime_t(file);
    if (!mime || !mime->mimetype) return NULL;
    return mime->mimetype;
}
G_MODULE_EXPORT
void mime_add(const gchar *file, const gchar *command)
{
    gchar *sfx=NULL;
    mime_t *mime;
    if (!command || !strlen(command)) return;
    if (!application_hash) mime_build_list();

    if (*file == '\"') sfx = g_path_get_basename(file+1);
    else sfx = g_path_get_basename(file);
    /* strip any remaining quote */
    if (strchr(sfx,'\"')) *strchr(sfx,'\"')=0;


    if (!sfx || !strlen(sfx)){
	g_free(sfx);
	return;
    }
    TRACE("adding suffix %s\n",sfx);
    sfx = g_utf8_strdown (sfx,-1);

    mime = g_hash_table_lookup(application_hash, sfx);

    if (!mime) {
	mime=(mime_t *)malloc(sizeof(mime_t));
	if (!mime) g_assert_not_reached();
	mime->apps = (char **)malloc(2*sizeof(char *));
	*(mime->apps)=g_strdup(command);
	*(mime->apps+1)=NULL;
	mime->key = g_strdup(sfx);
	mime->mimetype = g_strconcat("application/xffm-",sfx,NULL);
    } else {
	int i,j;
	char **tmp=mime->apps;
	for (i=0;mime->apps[i];i++);
	mime->apps = (char **)malloc((i+2)*sizeof(char *));
	*(mime->apps) = g_strdup(command);
	for (j=0; j<i; j++) mime->apps[j+1] = tmp[j];
	*(mime->apps+i+1) = NULL;
	g_free(tmp);
    }
    g_hash_table_replace(application_hash,  (gpointer) (mime->key),(gpointer) mime);
    mime_write(sfx,mime);
    TRACE("mime_write(%s,%s)\n",sfx,mime->mimetype);

    g_free(sfx);
    generate_cache();
    return;
}

G_MODULE_EXPORT
gboolean is_valid_command(const char *cmd_fmt){
    GError *error = NULL;
    int argc;
    gchar *path;
    gchar **argv;
    if (!cmd_fmt) return FALSE;
    if (!g_shell_parse_argv (cmd_fmt, &argc,&argv,&error)){
	gchar *msg = g_strcompress (error->message);
	g_warning("%s: %s",msg,cmd_fmt);
	g_error_free (error);
	g_free (msg);
	return FALSE;
    }
    path=g_find_program_in_path(argv[0]);
    if (!path || access(path,X_OK)!=0){
	g_strfreev (argv);
	if (!path) errno=ENOENT;
	return FALSE;
    }
    g_strfreev (argv);
    return TRUE;
}

G_MODULE_EXPORT
const gchar *mk_command_line(const gchar *command_fmt,const gchar *path,gboolean interm,gboolean hold){
    static gchar *command_line=NULL;
    gchar  *fmt=NULL,*termcmd=NULL;
    gboolean quoted=FALSE;
    gchar *p;

    if (!command_fmt) return NULL;
    if (command_line){
	g_free(command_line); 
	command_line=NULL;
    }
    if (!path) path="";

    if(interm){
	gchar *term=NULL;
	gchar *exec_flag="-e";
	if(getenv("TERMCMD") && strlen( getenv("TERMCMD"))){
	    term = g_strdup(getenv("TERMCMD"));
	} else {
	    term = g_strdup("xterm");
	}
	if (!is_valid_command(term)){
	    g_warning("%s == NULL",term);
	    g_free(term);
	    return NULL;
	}
	if (strstr("gnome-terminal",term) 	|| 
		strstr("gnome2-terminal",term)	||
		strstr("Terminal",term)	||
		strstr("terminal",term)	||
		strstr("xterminal",term) ) exec_flag="-x";

	if (hold && strncmp(term,"xterm",strlen("xterm"))==0) {
	    termcmd=g_strdup_printf("%s -hold %s ", term, exec_flag);
	    /*termcmd=g_strconcat(term," -hold -e ",NULL);*/
	} else {
	    termcmd=g_strdup_printf("%s %s ", term, exec_flag);
	    /*termcmd=g_strconcat(term," -e ",NULL);*/
	}
	g_free(term);
    }

    TRACE("command_fmt=%s\n",command_fmt);

    /* this is to send path as an argument */

    if (strstr(command_fmt,"\%s")){
	fmt = g_strconcat(((termcmd)?termcmd:""),command_fmt,NULL);
    } else {
	fmt = g_strconcat(((termcmd)?termcmd:""),command_fmt," \%s",NULL);
    }
    TRACE("fmt=%s\n",fmt);

    TRACE("path=%s",path);
    if (*path != '"') for (p=(gchar *)path;*p;p++){
	if ((*p>='A' && *p<='Z') || (*p>='a' && *p<='z') || (*p>='0' && *p<='9')) continue;
	quoted=TRUE;
	break;
    }
    if (!quoted) command_line = g_strdup_printf(fmt,path);
    else {
	gchar *quoted_path=g_strdup_printf("\"%s\"",path);
	command_line = g_strdup_printf(fmt,quoted_path);
	g_free(quoted_path);
    }
    g_free(fmt);
    g_free(termcmd);
    return command_line;
}

G_MODULE_EXPORT
const gchar *mime_command(const gchar *file){
    const mime_t *mime;
    int i;
    static gchar *cmd_fmt=NULL;
    g_free(cmd_fmt); cmd_fmt=NULL;
    mime=locate_mime_t (file);
    if (!mime || !mime->apps || !mime->apps[0]) return NULL;
    for (i=0; mime->apps[i]; i++){
	g_free(cmd_fmt);
	cmd_fmt = g_strcompress(mime->apps[i]);
	if (is_valid_command(cmd_fmt)) return cmd_fmt;	  
    }
    g_free(cmd_fmt); cmd_fmt=NULL;
    TRACE("mime_command(%s)=%s\n",file,((cmd_fmt)?cmd_fmt:"null"));
    return (const gchar *)cmd_fmt;
}

G_MODULE_EXPORT
const gchar **mime_apps(const gchar *file){
    const mime_t *mime;
    mime=locate_mime_t (file);
    if (!mime || !mime->apps) return NULL;
    return (const gchar **)mime->apps;
}

G_MODULE_EXPORT
xfmime_functions *module_init(void){
    return xfmime_fun;
}
G_MODULE_EXPORT
const gchar *mime_typeinfo(const gchar *type){
    xmlNodePtr node;
    xmlNodePtr subnode;
    xmlDocPtr doc;
    gchar *mimefile=NULL,*mimetype=NULL;
    static gchar *info;

    mimefile=g_build_filename(FREEDESKTOP_MIME_FILE,NULL);

    TRACE("mimefile=%s\n",mimefile);
    if (access(mimefile,R_OK)!=0){
	g_free(mimefile); mimefile=NULL;
	TRACE("access(mimefile,R_OK)!=0\n");
	return NULL;
    }
    xmlKeepBlanksDefault(0);

    if((doc = xmlParseFile(mimefile)) == NULL)
    {
	TRACE("xffm-modules: invalid xml file: %s\n",mimefile);
	g_free(mimefile); mimefile=NULL;
	return NULL;
    }

    node = xmlDocGetRootElement(doc);
    if(!xmlStrEqual(node->name, (const xmlChar *)"mime-info"))
    {
	TRACE("xffm-modules: invalid xml file %s\n",mimefile);
	g_free(mimefile); mimefile=NULL;
	xmlFreeDoc(doc);
	return NULL;
    }

    /* Now parse the xml tree */
    TRACE("TRACE: parsing %s\n",mimefile);
    TRACE("------------looking for %s\n",type);
    for(node = node->children; node; node = node->next)
    {
	if(xmlStrEqual(node->name, (const xmlChar *)"mime-type")){
	    mimetype = (char *)xmlGetProp(node, (const xmlChar *)"type");
	    TRACE("%s==%s\n",mimetype,type);
	    if (xmlStrEqual(mimetype, (const xmlChar *)type)){
		for(subnode = node->children; subnode; subnode = subnode->next){
		    if(xmlStrEqual(subnode->name, (const xmlChar *)"comment")){
			char *e=xmlNodeListGetString(doc, subnode->children, 1);
			g_free(mimetype);
			xmlFreeDoc(doc);
			g_free(mimefile);
			if (info) g_free(info);
			info =g_strdup(e);
			if (e) g_free(e);
			return (const gchar *)info;	
		    }
		}

	    }
	    g_free(mimetype);
	}
    }
    xmlFreeDoc(doc);
    g_free(mimefile);
    return NULL;
}


G_MODULE_EXPORT
const gchar *mime_get_type(const gchar *file,gboolean try_magic){
	const mime_t *mime;
	const gchar *m=mimeable_file(file);

	if (m && strlen(m)) return m;  
  	
	mime=locate_mime_t(file);
	if (mime) return (const gchar *)(mime->mimetype);

	if (!m) return "undetermined type";
	/* try magic */
	if (try_magic) return mime_magic_type(file);
	else return last_type_determination(file);	
}

G_MODULE_EXPORT
void g_module_unload(GModule *module){
	 GList *tmp;
	if (application_hash){
	   destroy_application_hash(application_hash);
	   application_hash=NULL;
	}
	if (magic_list){
	   for (tmp=magic_list;tmp; tmp=tmp->next){
		   magic_t *magic =(magic_t *)tmp->data;
		   if (magic->type) g_free(magic->type);
		   if (magic->value) g_free(magic->value);
		   if (magic->mask) g_free(magic->mask);
		   g_free(magic);
	   }
	   g_list_free(magic_list);
	   magic_list=NULL;
	}
	if (xfmime_fun) g_free(xfmime_fun);
	xfmime_fun = NULL;
}


