/*
 * Copyright (C) 2002-5 Edscott Wilson Garcia
 * EMail: edscott@imp.mx
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**  miscelaneous */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <regex.h>
#include <dirent.h>
#include <X11/Xlib.h>
#include <X11/Xproto.h>
#include <X11/Xatom.h>
#include <grp.h>
#include <pwd.h>

#ifdef HAVE_STDARG_H
#include <stdarg.h>
#elif HAVE_VARARGS_H
#include <varargs.h>
#endif


#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>


#include "constants.h"
#include "types.h"

#ifdef HAVE_LIBXFCEGUI4
#include <libxfcegui4/libxfcegui4.h>
G_MODULE_EXPORT
SessionClient *session_client;
#endif

#include "primary.h"

G_MODULE_EXPORT
gchar *xffm_iconpath=NULL;

G_MODULE_EXPORT
gboolean diagnostics_is_visible(GtkWidget *diagnostics);

static gboolean diagnostics_disabled=FALSE;
#if 0
typedef struct dostext_t {
	unsigned char readable;
	unsigned char unreadable;
} dostext_t;

/* starts at 0xc0 */
static dostext_t dostext[]={
 {0xc0, 0xb7}, /*  */
 {0xc1, 0xb5}, /*  */
 {0xc2, 0xb6}, /*  */
 {0xc3, 0xc7}, /*  */
 {0xc4, 0x8e}, /*  */
 {0xc5, 0x8f}, /*  */
 {0xc6, 0x92}, /*  */
 {0xc7, 0x80}, /*  */
 {0xc8, 0xd4}, /*  */
 {0xc9, 0x90}, /*  */
 {0xca, 0xd2}, /*  */
 {0xcb, 0xd3}, /*  */
 {0xcc, 0xde}, /*  */
 {0xcd, 0xd6}, /*  */
 {0xce, 0xd7}, /*  */
 {0xcf, 0xd8}, /*  */
 {0xd0, 0xd1}, /*  */
 {0xd1, 0xa5}, /*  */
 {0xd2, 0xe3}, /*  */
 {0xd3, 0xe0}, /*  */
 {0xd4, 0xe2}, /*  */
 {0xd5, 0xe5}, /*  */
 {0xd6, 0x99}, /*  */
 {0xd7, 0x9e}, /*  */
 {0xd8, 0x9d}, /*  */
 {0xd9, 0xeb}, /*  */
 {0xda, 0xe9}, /*  */
 {0xdb, 0xea}, /*  */
 {0xdc, 0x9a}, /*  */
 {0xdd, 0xed}, /*  */
 {0xde, 0xe8}, /*  */
 {0xdf, 0xe1}, /*  */
 {0xe0, 0x85}, /*  */
 {0xe1, 0xa0}, /*  */
 {0xe2, 0x83}, /*  */
 {0xe3, 0xc6}, /*  */
 {0xe4, 0x84}, /*  */
 {0xe5, 0x86}, /*  */
 {0xe6, 0x91}, /*  */
 {0xe7, 0x87}, /*  */
 {0xe8, 0x8a}, /*  */
 {0xe9, 0x82}, /*  */
 {0xea, 0x88}, /*  */
 {0xeb, 0x89}, /*  */
 {0xec, 0x8d}, /*  */
 {0xed, 0xa1}, /*  */
 {0xee, 0x8c}, /*  */
 {0xef, 0x8b}, /*  */
 {0xf0, 0xd0}, /*  */
 {0xf1, 0xa4}, /*  */
 {0xf2, 0x95}, /*  */
 {0xf3, 0xa2}, /*  */
 {0xf4, 0x93}, /*  */
 {0xf5, 0xe4}, /*  */
 {0xf6, 0x94}, /*  */
 {0xf7, 0xf6}, /*  */
 {0xf8, 0x9b}, /*  */
 {0xf9, 0x97}, /*  */
 {0xfa, 0xa3}, /*  */
 {0xfb, 0x96}, /*  */
 {0xfc, 0x81}, /*  */
 {0xfd, 0xec}, /*  */
 {0xfe, 0xe7}, /*  */
 {0,0}
};

#endif
G_MODULE_EXPORT
void place_dialog(GtkWidget *parent, GtkWidget *window){
    int x,y,w,h,X,Y,W,H;
    if (!parent || !window) {
	g_warning("!parent || !window");
	return; 
    }
    gtk_widget_realize(window);
    w=parent->allocation.width;
    h=parent->allocation.height;
    W=window->allocation.width;
    H=window->allocation.height;
    gtk_window_get_position ((GtkWindow *)parent,&x,&y);
    X=(w-W)/2 + x;
    Y=(h-H)/2 + y;
    if (X < 0) X=0;
    if (Y < 0) Y=0;
    TRACE("place_dialog: x=%d, y=%d, w=%d, h=%d, W=%d, H=%d, ->X=%d, ->Y=%d",
	    x,y,w,h,W,H,X,Y);
    gtk_window_move((GtkWindow *)window,X,Y);
}
G_MODULE_EXPORT
void set_restart_command (GtkWidget *widget, int argc, char **argv){
    Atom atomo;
    static gboolean interned=FALSE;
#ifdef HAVE_LIBXFCEGUI4
    int j;
    static gchar **rstart=NULL;
    gchar **saved_rstart=rstart;
	TRACE("set_restart_command argc=%d",argc);
    if (!argc) return;
     
    rstart = (gchar **)malloc((argc+1)*sizeof(gchar *));
    for(j=0;j<argc;j++){
	rstart[j]=g_strdup(argv[j]);
	TRACE("new_rstart %d: %s",j,rstart[j]);
    }
    rstart[argc]=NULL;
    if (session_client) session_client->restart_command=rstart;
    g_free(saved_rstart);

#else
#if 0
/*#ifdef HAVE_LIBSM*/
      int i=0;
       SmProp prop1,*props[1];
       SmPropValue *vals = g_new (SmPropValue, argc);
       
       props[0] = &prop1;

	prop1.name = SmRestartCommand;
	prop1.type = SmLISTofARRAY8;
	prop1.vals = vals;
	for (i=0;i<argc;i++){
	    if (!argv[i]) {
		argc=i;
		break;
	    }
	    vals[i].value = argv[i];
	    vals[i].length = strlen(argv[i]);
	}
	prop1.num_vals = argc;
	
	printf("TRACE:xxx 0x%x\n",session_client);
	printf("TRACE:xxx 0x%x\n",session_client->session_connection);
	SmcSetProperties ((SmcConn) session_client->session_connection, 1, props);
	g_free(vals);
#endif
#endif

      if (interned) {
	/*static char *argv[4];
	argv[0]=xffm_details->argv[0];
	argv[1]=argv[2]=argv[3]=NULL;
	if (argc >1) argv[1]=xffm_details->argv[1];
	if (argc >2) argv[2]=xffm_details->argv[2];*/
	
	XSetCommand(GDK_DISPLAY(), 
		GDK_WINDOW_XID(gtk_widget_get_root_window(widget)), 
		argv, argc);
      }
      else 
      {
	int i;
	interned=TRUE;
	atomo = XInternAtom(GDK_DISPLAY(), "WM_CLASS", FALSE);
	XChangeProperty(GDK_DISPLAY(), 
		GDK_WINDOW_XID(gtk_widget_get_root_window(widget)), 
		atomo, 
		XA_STRING, 
		8, PropModeReplace, 
		(unsigned char *)argv[0], strlen(argv[0]) + 1);
	atomo = XInternAtom(GDK_DISPLAY(), "WM_COMMAND", FALSE);
	XChangeProperty(GDK_DISPLAY(), 
		GDK_WINDOW_XID(gtk_widget_get_root_window(widget)), 
		atomo, XA_STRING, 
		8, PropModeReplace, 
		(unsigned char *)argv[0], strlen(argv[0]) + 1);

	for (i=1; argc>i && argv[i]; i++) {
	    XChangeProperty(GDK_DISPLAY(), 
		    GDK_WINDOW_XID(gtk_widget_get_root_window(widget)), 
		    atomo, XA_STRING,	/* GdkAtom type, */
		    8,		/* bit per data element: gint format */
	    	    PropModeAppend, 
		    (unsigned char *)argv[i], strlen(argv[i])+1);
	}
      }

      return;
}


G_MODULE_EXPORT
gboolean is_number(char *p){
   char *c=p;
   if (!c || !strlen(c)) return FALSE;
   for (;*c;c++) {
	if (*c < '0' || *c > '9') return FALSE;
   }
   return TRUE;   
}

G_MODULE_EXPORT
const char *tod (void){
	time_t t=time(NULL);
	return ctime(&t);
}

G_MODULE_EXPORT
int valid_pasteboard(void)
{
    int ret = 0, len = -1;
    char *b;
    b = XFetchBuffer(GDK_DISPLAY(), &len, 0);
    /*printf("dbg:bytes=%d,buffer0=%s\n",len,b); */

    if((!b) || (!strlen(b)))
	ret = 0;
    else if(strncmp(b, "#xfvalid_buffer:copy", strlen("#xfvalid_buffer:copy")) == 0)
	ret = 1;
    else if(strncmp(b, "#xfvalid_buffer:cut", strlen("#xfvalid_buffer:cut")) == 0)
	ret = 2;

    XFree(b);
    return ret;
}


/* returns 0 if not in pasteboard, 1 if in copy pasteboard or 2 if
 * in cut pasteboard */
G_MODULE_EXPORT
int in_pasteboard(record_entry_t *en)
{
    int len = -1;
    char *b, *search,*path;
    static char *files=NULL;
    gboolean cut;

    if(!en || !en->path) return FALSE;
    if (IS_ROOT_TYPE(en->type)&&!IS_LOCAL_TYPE(en->type)) return FALSE;
    if (__XF_MASK&en->subtype) return FALSE;
    
    b = XFetchBuffer(GDK_DISPLAY(), &len, 0);
    if((!b) || (!strlen(b))) goto not_here;

    if (IS_NETDIR(en->subtype)||IS_NETFILE(en->subtype)){
	char *server,*remote_file;
	if (files) {
		g_free(files);
		files=NULL;
	}
	server=g_strdup(en->path+2);
	strtok(server,"/");
	remote_file=server+strlen(server)+1;
	files=(char *)malloc(strlen(server)+
			strlen(remote_file)+
			strlen(en->tag)+
			strlen("smb://@://\n")+1);
	if (IS_NETDIR(en->subtype))
		sprintf (files, "%s://%s@%s:%s/",
		  IS_SAMBA_SERVER(en->subtype)?"SMB":"smb",
		  (en->tag)?en->tag:"GUEST%%",
		  server,remote_file);
	else
		sprintf (files, "%s://%s@%s:%s",
		  IS_SAMBA_SERVER(en->subtype)?"SMB":"smb",
		  (en->tag)?en->tag:"GUEST%%",
		  server,remote_file);
	
	g_free(server);
	server=NULL;
	path=files;
    }
    else path=en->path;
    
    search = strtok(b, "\n");
    if (!search) return 0;
    if (strncmp(search, "#xfvalid_buffer", strlen("#xfvalid_buffer")) != 0)
	    return 0;
    if (strncmp(search, "#xfvalid_buffer:cut", strlen("#xfvalid_buffer:cut")) == 0) cut = TRUE; else cut = FALSE;
    
    search = strtok(NULL,"\n");
    while(search)
    {
	if(strcmp(search, path) == 0) {
	    XFree(b);
	    return ((cut)?2:1);
	}
	search = strtok(NULL, "\n");
    }
  not_here:
    XFree(b);
    return 0;
}

G_MODULE_EXPORT
const gchar *
sizetag(off_t tama, gint count)
{
    gchar *tag = "B";
    static gchar *buf=NULL;
    long long utama=(long long)tama;

    g_free(buf);buf=NULL;
    if(utama >= 0)
    {
	if(utama >= (long long)1024 * 1024 * 1024)
	{
	    utama /= ((long long)1024 *1024 * 1024);
	    tag = "GB";
	}
	else if(utama >= 1024 * 1024)
	{
	    utama /= (1024 * 1024);
	    tag = "MB";
	}
	else if(utama >= 1024)
	{
	    utama /= 1024;
	    tag = "KB";
	}
	if(count <= 0){
	    /* format for size column of regular files */
	    buf=g_strdup_printf("%lld %s", utama, tag);
	}
	else if(count == 1){
	    /* format for size column of folders (one file) */
	    buf=g_strdup_printf(_("%d file, %lld %s."), count, utama, tag);
	}
	else {
	    /* format for size column of folders (more than one file) */
	    buf=g_strdup_printf(_("%d files, %lld %s."), count, utama, tag);
	}
    }
    else
    {
	if(count < 0){
	    buf=g_strdup_printf(" ");
	}
	else if(count == 1){
	    /* format for count of processed files (single file)*/
	    buf=g_strdup_printf(_("%d file"), count);
	} else {
	    /* format for count of processed files (multiple files)*/
	    buf=g_strdup_printf(_("%d files"), count);
	}
    }
    return (const gchar *)buf;
}

G_MODULE_EXPORT
const gchar *xffm_get_basename(const gchar *path)
{
    static gchar *name=NULL;
    g_free (name);
    if (!path) return "";
    name = g_path_get_basename(path);
    if (!name) return "";
    return (const gchar *)name;
}

static
gchar *
private_valid_utf_pathstring(const gchar *path){
    gchar *dir,*base,*valid,*recursed;
    if (!path) return NULL;
    if (g_utf8_validate (path,-1,NULL)) return g_strdup(path);
    dir=g_path_get_dirname(path);
    base=g_path_get_basename(path);
    recursed=private_valid_utf_pathstring(dir);
    valid=g_strconcat(recursed,G_DIR_SEPARATOR_S,
			my_utf_string(base),NULL);
    g_free(recursed);
    g_free(base);
    g_free(dir);
    return valid;
}

G_MODULE_EXPORT
const gchar *
my_valid_utf_pathstring(const gchar *path){
    static gchar *utf_path=NULL;
    g_free(utf_path);
    utf_path=private_valid_utf_pathstring(path);
    if (strncmp(utf_path,"./",strlen("./")==0) && 
	strncmp(path,"./",strlen("./")!=0)   )
    {
	gchar *g=g_strdup(utf_path+strlen("./"));
	g_free(utf_path);
	utf_path=g;
    }
    return utf_path;
}


G_MODULE_EXPORT
const gchar *my_utf2local_string(const gchar *t)
{
    static gchar *s = NULL;
    GError *error = NULL;
    gsize r_bytes, w_bytes;
    unsigned char *c;
    const char *fc;
    gchar *from_codeset=NULL;
    
    if(!t) {
	TRACE("my_local_string(NULL)");
	return "";
    }
    
    if (!g_utf8_validate (t,-1,NULL)) {
	TRACE("TRACE: valid utf8: %s\n",t);
	return t;
    }
    
    /* so we got a UTF-8 */
    

    	g_get_charset(&fc);
    	if (fc) from_codeset = g_strdup(fc);
    	else from_codeset = g_strdup("ASCII");

    TRACE("TRACE: codeset for system is: %s\n",from_codeset);    
    
    if(s) {
	    g_free(s);
	    s=NULL;
    }
    TRACE("TRACE: un-utfing %s\n",t); 
    s = g_convert (t,strlen(t),from_codeset,"UTF-8",&r_bytes, &w_bytes, &error);

    if(!s)
    {
	s=g_strdup(t);
	for(c = s; *c != 0; c++) if(*c > 128) *c = '?';
    }
    if(error){
        g_warning("%s. Codeset for system is: %s\nUnable to convert from utf-8",
			error->message,from_codeset);
	g_error_free(error);
    }
    g_free(from_codeset);
    from_codeset=NULL;
    return (const gchar *)s;
}


G_MODULE_EXPORT
const gchar *my_utf_string(const gchar *t)
{
    static gchar *s = NULL;
    GError *error = NULL;
    gsize r_bytes, w_bytes;
    unsigned char *c;
    const char *fc;
    gchar *from_codeset=NULL;
    
    if(!t) {
	TRACE("my_utf_string(NULL)");
	return "";
    }
    
    if (g_utf8_validate (t,-1,NULL)) {
	TRACE("TRACE: valid utf8: %s\n",t);
	return t;
    }
    
    /* so we got a non-UTF-8 */

    /* SMB_CODESET now obsolete... */
    /*if (getenv("SMB_CODESET") && strlen( getenv("SMB_CODESET"))){
	    from_codeset=g_strdup(getenv("SMB_CODESET"));
    }
    else */
    {
    	g_get_charset(&fc);
    	if (fc) from_codeset = g_strdup(fc);
    	else from_codeset = g_strdup("ISO-8859-1");
    }
    
    TRACE("TRACE: codeset for system is: %s\n",from_codeset);    
    
    if(s) {
	    g_free(s);
	    s=NULL;
    }
    TRACE("TRACE: utfing %s\n",t); 
    for(c = (unsigned char *)t; *c != 0; c++)
	if(*c < 32 && *c != '\n')
	    *c = ' ';
    s = g_convert (t,strlen(t),"UTF-8",from_codeset,&r_bytes, &w_bytes, &error);

    if(!s)
    {
	s=g_strdup(t);
	for(c = s; *c != 0; c++) if(*c > 128) *c = '?';
    }
    if(error){
        g_warning("%s. Codeset for system is: %s\nunable to convert to utf-8",
			error->message,from_codeset);
	g_error_free(error);
    }
     g_free(from_codeset);
    from_codeset=NULL;
   return (const gchar *)s;
}

G_MODULE_EXPORT
const gchar *
host_name (Window xid){
    static char *name = NULL;
    unsigned char *property_data;
    unsigned long items, remaining;
    int actual_format;
    Atom atomo, actual_atom;

    g_free(name);
    atomo = XInternAtom(GDK_DISPLAY(), "WM_CLIENT_MACHINE", FALSE);
    if(XGetWindowProperty(GDK_DISPLAY(), xid,
		atomo, 0, 255, FALSE, XA_STRING, 
		&actual_atom, &actual_format, 
		&items, &remaining, &property_data) == Success)
    {
	TRACE("property_data=%s",((property_data)?property_data:(unsigned char *)"null")); 
	if (!property_data) name = g_strdup("localhost");
	else {
	    name = g_strdup(property_data);
	    XFree(property_data);
	}
    }
    else name = g_strdup("localhost");
    return (const gchar *)name;
}

G_MODULE_EXPORT
const gchar *
Xour_host_name (widgets_t *widgets_p)
{
    static char *name = NULL;
    Window xid;

    if(name) return name;
    xid = GDK_WINDOW_XID(widgets_p->window->window);
    name = g_strdup(host_name(xid));
    return name;
}

G_MODULE_EXPORT
void fork_function (void *data)
{
    char **argument = (char **)data;

    execvp(argument[0], argument);
    fprintf(stderr, "CHILD could not execvp: this should not happen");
    fprintf(stderr, "Do you have %s in your path?", argument[0]);
    fflush(NULL);
    usleep(500000);
    _exit(123);
}

/***   diagnostics... ***/
static gchar * my_cut_utf_string(gchar *s){
	static  gchar *u=NULL;
	gchar *v=g_strdup(my_utf_string(s));
	int max_len=48;
	if (u) {
		g_free(u);
		u=NULL;
	}
        if (g_utf8_strlen(s,-1) > max_len){
	    u=g_utf8_offset_to_pointer (v,max_len-4);
	    *u=0;
	    u=g_strjoin ("",v,"...",NULL);
	    g_free(v);
	    v=NULL;
       } else u=v;
       return u;
} 

static void insert_string(GtkTextBuffer * buffer, const gchar *s)
{
    GtkTextIter start,end;
    if(s)
    {
	gtk_text_buffer_get_bounds(buffer,&start, &end);
	gtk_text_buffer_insert(buffer, &end, s, -1);
    }
    return;
}


G_MODULE_EXPORT
void print_status(widgets_t *widgets_p,const gchar *id, ...)
{
    va_list var_args;
    char *t;
    GtkTextIter start, end;
    GtkTextBuffer *buffer;
    GdkPixbuf *icono;
    if (!widgets_p) return;

    if (!widgets_p->status) {
#ifdef DEBUG
	g_warning("print_status: status==NULL");
	va_start(var_args, id);			
	do					
	{					
	    t = va_arg(var_args, char *);	
	    if(t && strlen(t)){			
		printf("%s\n",my_utf_string(t));
	    }					
	}					
	while(t);				
	va_end(var_args);
#endif
	return;
    }
    gtk_text_view_set_justification ((GtkTextView *) widgets_p->status,GTK_JUSTIFY_LEFT);
    buffer = gtk_text_view_get_buffer((GtkTextView *)widgets_p->status);
    gtk_text_buffer_set_text(buffer, " ", -1);
    gtk_text_buffer_get_bounds(buffer, &start, &end);
   
    if (!id) id="xfce/unknown";
	
    icono = icon_tell(widgets_p, 0, id);
 
    if(icono){ 
	gtk_text_buffer_insert_pixbuf(buffer, &start, icon_tell(widgets_p, 0, id));
	g_object_unref(icono);
    }

    gtk_text_buffer_get_bounds(buffer, &start, &end);
    gtk_text_buffer_insert(buffer, &end, " ", -1);    
    va_start(var_args, id);
    do
    {
	t = va_arg(var_args, char *);
	if(t && strlen(t)){	
	    insert_string(buffer, my_cut_utf_string(t));
	}
    }
    while(t);
    va_end(var_args);
  gdk_flush();
}

G_MODULE_EXPORT
void print_status_pixbuf(widgets_t *widgets_p,GdkPixbuf * icono, ...)
{
    va_list var_args;
    char *t;
    GtkTextIter start, end;
    GtkTextBuffer *buffer;
    if (!widgets_p) return;

    if (!widgets_p->status) {
	DBG("print_status_pixbuf: status==NULL");
	return;
    }
    gtk_text_view_set_justification ((GtkTextView *) widgets_p->status,GTK_JUSTIFY_LEFT);
    buffer = gtk_text_view_get_buffer((GtkTextView *)widgets_p->status);
    gtk_text_buffer_set_text(buffer, " ", -1);
    gtk_text_buffer_get_bounds(buffer, &start, &end);

    if(icono)
	gtk_text_buffer_insert_pixbuf(buffer, &end, icono);
    gtk_text_buffer_get_bounds(buffer, &start, &end);
    gtk_text_buffer_insert(buffer, &end, "  ", -1);
    va_start(var_args, icono);
    do
    {
	t = va_arg(var_args, char *);
	if(t && strlen(t)){
	    insert_string(buffer, my_cut_utf_string(t));
	}
    }
    while(t);
    va_end(var_args);
  gdk_flush();
}

G_MODULE_EXPORT
void print_diagnostics(widgets_t *widgets_p,const gchar *id, ...)
{
    char *t;
    va_list var_args;
    GtkTextIter start, end;
    GtkTextMark *mark;
    GtkTextBuffer *buffer;
    GdkPixbuf *icono=NULL;
    
    if (!widgets_p) return;

    /* XXX this is a startup hack for treeview. It may no longer be necessary*/
    if(diagnostics_disabled) {
	TRACE("diagnostics_disabled\n");
	return;
    }

    if (getenv("XFFM_FORCE_DIAGNOSTICS") && strlen(getenv("XFFM_FORCE_DIAGNOSTICS"))) 
    {
	if (widgets_p->diagnostics) show_text(widgets_p->diagnostics);
    } else {
	if (getenv("XFFM_DISABLE_DIAGNOSTICS") && strlen(getenv("XFFM_DISABLE_DIAGNOSTICS")))
	    return;
    }
    
    if(!widgets_p->diagnostics ) {
#ifdef DEBUG
	g_warning("diagnostics==NULL");
	va_start(var_args, id);			
	do					
	{					
	    t = va_arg(var_args, char *);	
	    if(t && strlen(t)){			
		printf(my_utf_string(t));	
	    }					
	}					
	while(t);				
	va_end(var_args);			
#endif
	return;
    }
    


    if (id && strcmp(id,"xfce/error")==0) show_text(widgets_p->diagnostics);
    if (!diagnostics_is_visible(widgets_p->diagnostics)) 
    {
	TRACE("!diagnostics_is_visible(diagnostics)\n");
	return;
    }
    
    /****  */
    buffer= gtk_text_view_get_buffer(GTK_TEXT_VIEW(widgets_p->diagnostics));

    /*printf("TRACE:print_diagnostics\n");*/
    gtk_text_buffer_get_bounds(buffer, &start, &end);
    
    if(id) icono = icon_tell(widgets_p, 0, id);
    if(icono){
	gtk_text_buffer_insert_pixbuf(buffer, &end, icono);
	g_object_unref(icono);
    }

    va_start(var_args, id);
    do
    {
	t = va_arg(var_args, char *);
	if(t && strlen(t)){
	    insert_string(buffer, my_utf_string(t));
	}
    }
    while(t);
    va_end(var_args);
    gtk_text_buffer_get_bounds(buffer, &start, &end);
    mark = gtk_text_buffer_create_mark(buffer, "scrollmark", &end, FALSE);
    gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(widgets_p->diagnostics), mark, 0.2,	/*gdouble within_margin, */
				 FALSE, 0.0, 0.0);
    gtk_text_buffer_delete_mark(buffer, mark);
  gdk_flush();
}


G_MODULE_EXPORT
void disable_diagnostics (void){diagnostics_disabled=TRUE;};
G_MODULE_EXPORT
void enable_diagnostics (void){diagnostics_disabled=FALSE;};

G_MODULE_EXPORT
void show_text(GtkWidget *diagnostics)
{
    GtkWidget *vpaned;
    if (!diagnostics) return;
    vpaned=lookup_widget(diagnostics,"vpaned1");
    if (!vpaned) return;
    if (gtk_paned_get_position((GtkPaned *)vpaned) > vpaned->allocation.height * 0.80)
    gtk_paned_set_position(GTK_PANED(vpaned), vpaned->allocation.height * 0.80);
}

G_MODULE_EXPORT 
gboolean diagnostics_is_visible(GtkWidget *diagnostics){
    GtkWidget *vpaned;
    if (!diagnostics) return FALSE;
    vpaned=lookup_widget(diagnostics,"vpaned1");
    if (!vpaned) return FALSE;
    if (gtk_paned_get_position((GtkPaned *)vpaned) > vpaned->allocation.height * 0.95) return FALSE;
    else return TRUE; 
}

G_MODULE_EXPORT
void hide_text(GtkWidget *diagnostics)
{
    GtkWidget *vpaned;
    if (!diagnostics) return;
    vpaned=lookup_widget(diagnostics,"vpaned1");
    if (!vpaned) return;
    gtk_paned_set_position(GTK_PANED(vpaned), vpaned->allocation.height);
}


G_MODULE_EXPORT
void clear_diagnostics(GtkWidget *diagnostics)
{
    GtkTextIter start, end;
    GtkTextBuffer *buffer;
    if(!diagnostics) return;
    buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(diagnostics));
    gtk_text_buffer_get_bounds(buffer, &start, &end);
    gtk_text_buffer_delete(buffer, &start, &end);
    hide_text(diagnostics);
}

static const gchar *plain_sizetag(int tama, int tamal)
{
    char *tag = "KB";
    char *tagl = "KB";
    static gchar *buf=NULL;

    g_free(buf);
    buf=NULL;
    if(tama >= 1024 * 1024)
    {
	tama /= (1024 *1024);
	tag = "GB";
    }
    else if(tama >= 1024)
    {
	tama /= 1024;
	tag = "MB";
    }

    if(tamal >= 1024 * 1024)
    {
	tamal /= (1024 *1024);
	tagl = "GB";
    }
    else if(tamal >= 1024)
    {
	tamal /= 1024;
	tagl = "MB";
    }

    buf=g_strdup_printf("%d %s / %d %s", tama, tag, tamal, tagl);

    return (const gchar *)buf;
}



G_MODULE_EXPORT
void set_progress_generic(widgets_t *widgets_p,int count, int total, int caso)
{
    static int smallcount = 1;
    GtkWidget *progress;
    gfloat fraction;


    if (!widgets_p) return;
    progress=widgets_p->progress;
    if(!progress){
	g_warning("progress==NULL");
	return;
    }
    gtk_widget_show(progress);
    if(count == -1)
    {
	/*printf("TRACE:updating pulse\n"); */
	if(total < 0 || (smallcount++ & (1 << 7)))
	{
	    gtk_progress_bar_pulse((GtkProgressBar *) progress);
	    process_pending_gtk();
	    smallcount = 1;
	}
    }
    else
    {
	char texto[_POSIX_PATH_MAX];
	if(!total)
	    fraction = 0.0;
	else
	    fraction = (float)count / (float)total;
	if(fraction < 0 || fraction > 1)
	    return;
	gtk_progress_bar_set_fraction((GtkProgressBar *) progress, fraction);
	if(count >= total)
	    texto[0] = 0;
	else
	    switch (caso)
	    {
		case 1:
		    sprintf(texto, "%d / %d", count, total);
		    break;
		case 0:
		default:
		    sprintf(texto, "%s", plain_sizetag(count, total));
		    break;
	    }
	gtk_progress_bar_set_text((GtkProgressBar *) progress, texto);
    }
}


G_MODULE_EXPORT
void
set_icon_name(	GtkWidget *widget,const gchar *path){
    gchar *basename;
    gchar *iconname;
    if (!path) {
	basename=g_path_get_basename(xffm_details->argv[0]);
	iconname=g_strdup(my_utf_string(basename));
    }
    else {
	basename=g_path_get_basename(path);
	if (strcmp(basename,path)) {
	    if (g_get_home_dir() && strncmp(path,g_get_home_dir(),strlen(g_get_home_dir()))==0){
		iconname=g_strconcat(my_utf_string(basename)," (~",my_valid_utf_pathstring(path+strlen(g_get_home_dir())),")",NULL);
	    } else {
		iconname=g_strconcat(my_utf_string(basename)," (",my_valid_utf_pathstring(path),")",NULL);
	    }
	}
	else iconname=g_strdup(my_utf_string(basename));
    }
    gdk_window_set_icon_name(
		    gtk_widget_get_toplevel(widget)->window, 
		    iconname);
    gtk_window_set_title(
		    GTK_WINDOW(gtk_widget_get_toplevel(widget)), 
		    iconname);    
    gdk_flush();
    g_free(basename);
    g_free(iconname);
}
    
G_MODULE_EXPORT
void
set_application_icon(widgets_t *widgets_p, record_entry_t *en){
    const gchar *icon_id="xfce/b-iconview";
    GdkPixbuf *icon_pixbuf;
    /* don't change these: */
    if (strstr(xffm_details->argv[0],"xfglob")) return;
    if (!en) icon_id="xfce/stock_system";
    else if (en->module && function_natural("plugins",en->module,en,"module_icon_id")) icon_id=function_natural("plugins",en->module,en,"module_icon_id");
    else if (en->path && g_file_test(en->path,G_FILE_TEST_EXISTS)){
	if (strcmp(en->path,g_get_home_dir())==0)
	    icon_id="xfce/b-home";
	else
	    icon_id="xfce/b-treeview";
    }
    else if (IS_FIND_TYPE(en->type)){
	    icon_id="xfce/b-find";
    }
    icon_pixbuf = icon_tell (widgets_p, REAL_BIG,icon_id);
    if (icon_pixbuf) {
 	gtk_window_set_icon (GTK_WINDOW (widgets_p->window), icon_pixbuf);
      	g_object_unref (G_OBJECT(icon_pixbuf));
    }
}

G_MODULE_EXPORT
gchar *startup_path(const gchar *in_path){
	gchar *path=NULL;
	if (g_path_is_absolute(in_path) && g_file_test(in_path,G_FILE_TEST_EXISTS)){
	    TRACE("%s is dir",in_path);
	    path=g_strdup(in_path);
	}
	else {
	    gchar *dir=g_get_current_dir();
	    gchar *test=g_build_filename(dir,in_path,NULL);
	    g_free(dir);
		TRACE("%s is testdir1",test);
	    if (g_file_test(test,G_FILE_TEST_EXISTS)) path=test;
	    else {
		g_free(test);
		test=g_build_filename(g_get_home_dir(),in_path,NULL);
		TRACE("%s is testdir2",test);
		if (g_file_test(test,G_FILE_TEST_EXISTS)) path=test;
		else g_free(test);
	    }
	}
	if (path && !g_file_test(path,G_FILE_TEST_IS_DIR)){
	    gchar *dir=g_path_get_dirname(path);
	    g_free(path);
	    path=dir;
	}

	return path;
}

G_MODULE_EXPORT
const gchar *
path_info(record_entry_t *en){
    const gchar *mimetype = NULL;
    gchar *s1=NULL,*s2=NULL;
    gchar *info=NULL;
    gchar *vpath=NULL;
    if (!en || !en->path) return NULL;
    g_free(info);
    info=NULL;
    if (IS_ROOT_TYPE(en->type) && !g_file_test(en->path, G_FILE_TEST_EXISTS)) return NULL;
		    
    vpath=g_strdup(my_valid_utf_pathstring(en->path));
    mimetype = MIME_get_type(en->path,IS_PATH(en->type));
    /* this is too slow, and of not much use...
     * typeinfo = MIME_typeinfo(mimetype);  
     * */
    
    /* printout full file information: */
	    
    if (IS_XF_LNK(en->type)){
	char lpath[_POSIX_PATH_MAX+1];
	memset(lpath,0,_POSIX_PATH_MAX+1);
	if (readlink(en->path, lpath, _POSIX_PATH_MAX)>0) {    
	    s1=g_strdup_printf(_("Symbolic link: %s -> %s\n"),vpath,my_valid_utf_pathstring(lpath));
	} 
    } else {
	s1=g_strdup_printf(_("Path: %s\n"),vpath);
    }
			    
    if (g_file_test(en->path, G_FILE_TEST_EXISTS) && en->st) {
      struct group *g;
      struct passwd *p;
      char *grupo,*owner;
      const gchar *tag = sizetag((off_t)en->st->st_size, -1);
      if((g = getgrgid(en->st->st_gid)) != NULL)	grupo = g->gr_name;
      else if ((int)en->st->st_gid < 0) grupo = ""; else grupo = "?";
      if((p = getpwuid(en->st->st_uid)) != NULL) owner = p->pw_name;
      else if ((int)en->st->st_uid < 0) owner = ""; else owner = "?";
      s2=g_strdup_printf(_("Date=%s; Size=%s\nOwner=%s:%s; Protection=%s\nMimetype=%s"),
	      my_utf_string(time_to_string(en->st->st_mtime)),
	      tag, owner,grupo,mode_string(en->st->st_mode),
	      mimetype);
    }

    if (!s1) s1=g_strdup("");
    if (!s2) s2=g_strdup("");
    info=g_strconcat(s1,s2,NULL);
    g_free(s1);
    g_free(s2);
    return (const gchar *)info;
}

G_MODULE_EXPORT
void 
print_path_info (widgets_t *widgets_p,record_entry_t *en)
{
    const gchar *mimetype = NULL;
    if (!en || !en->path) return;
    mimetype = MIME_get_type(en->path,IS_PATH(en->type));
    if (IS_FIND_TYPE(en->type) && !g_file_test(en->path,G_FILE_TEST_EXISTS)) {
	mimetype="xfce/stock_zoom-fit";
    }
    print_diagnostics(widgets_p,mimetype,path_info(en),"\n",NULL);
    return;    
}


/*
 * This function converts a time value specified in seconds since 1970-01-01
 * to a string representation. It shows the date and the time if the point
 * if less then six month in the past and only the date otherwise.
 * The exact format must be provided by a translation string. 
 *
 * The function should be thread-save since there are not used static
 * (or even global) variables if the system provided a localtime_r() function.
 *
 * Arguments:     when:    time in seconds since 1970-01-01
 *                string:  the result char-array in which the string is placed
 *                length:  the length of the string-array
 *                
 * Return value:  string on success, NULL on failure
 */
G_MODULE_EXPORT
const 
gchar *
time_to_string (time_t when) 
{
      time_t          now             = time(NULL);
#ifdef HAVE_LOCALTIME_R
      struct tm       timestruct;
#endif
      struct tm*      timestruct_ptr;
      char	*formatstring;
      static gchar *s=NULL;
      gchar string[64];
      
#ifdef HAVE_MEMSET
      memset (string, 0, 64);
#else
      string[0]=0;
#endif
      
      formatstring = difftime(now, when) > 24*60*60*30*6
            /* strftime format for non-recent files (older than 6 months)  */
          ? _("%b %e  %Y")
            /* strftime format for recent files */
          : _("%b %e %H:%M");

#ifdef HAVE_LOCALTIME_R
      timestruct_ptr = &timestruct;
      localtime_r(&when, timestruct_ptr);
#else
      timestruct_ptr = localtime(&when);
#endif

      if (strftime(string, 64, formatstring, localtime(&when)) == 0) {
          return NULL;
      }
      g_free(s);
      s=g_strdup(my_utf_string(string));
      return (const gchar *)s;
}


G_MODULE_EXPORT
void on_clear_text_window(GtkButton * button, gpointer data)
{
    
    widgets_t *widgets_p=(widgets_t *)data;
    if (!widgets_p->diagnostics) return;
    clear_diagnostics(widgets_p->diagnostics);
}


G_MODULE_EXPORT
void on_stop(GtkButton * button, gpointer data)
{
    widgets_t *widgets_p=(widgets_t *)data;
    
    widgets_p->stop = TRUE;
    if (!widgets_p->stop_button || !widgets_p->clear_button) return;
    gtk_widget_hide(widgets_p->stop_button);
    gtk_widget_show(widgets_p->clear_button);
}

G_MODULE_EXPORT
void 
show_stop(			widgets_t *widgets_p){
    WIDGET_P_CHECK;
    if (!widgets_p->stop_button || !widgets_p->clear_button) return;
    widgets_p->stop=FALSE;
    gtk_widget_show(widgets_p->stop_button);
    gtk_widget_hide(widgets_p->clear_button);
    process_pending_gtk();
}
G_MODULE_EXPORT
void 
hide_stop(			widgets_t *widgets_p){
    WIDGET_P_CHECK;
    if (!widgets_p->stop_button || !widgets_p->clear_button) return;
    gtk_widget_hide(widgets_p->stop_button);
    gtk_widget_show(widgets_p->clear_button);
}

static 
GtkWidget *
dialog_button (widgets_t *widgets_p, const char *icon_id, const char *text)
{
    GtkWidget *button, *align, *image, *hbox, *label;
    GdkPixbuf *pb=icon_tell(widgets_p, SMALL,icon_id);

    button = gtk_button_new ();
    label = gtk_label_new_with_mnemonic (text);
    gtk_label_set_mnemonic_widget (GTK_LABEL (label), button);

    image = gtk_image_new_from_pixbuf (pb);
    g_object_unref(pb);
    /*image = gtk_image_new_from_stock (stock, GTK_ICON_SIZE_BUTTON);*/
    hbox = gtk_hbox_new (FALSE, 2);
    align = gtk_alignment_new (0.5, 0.5, 0.0, 0.0);

    gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, FALSE, 0);
    gtk_box_pack_end (GTK_BOX (hbox), label, FALSE, FALSE, 0);

    gtk_container_add (GTK_CONTAINER (button), align);
    gtk_container_add (GTK_CONTAINER (align), hbox);
    gtk_widget_show_all (align);

    return button;

}

G_MODULE_EXPORT 
GtkWidget *
xffm_confirm_dialog (		widgets_t *widgets_p,
				const gchar *text,
				const gchar *action_false,
				const gchar *action_true){
    GtkWidget *dialog;
    GtkWidget *button;

    if (!widgets_p){
	g_warning("xffm_confirm_dialog requieres widgets_p != NULL");
	return NULL;
    }
	

    if (!action_false || !action_true) {
	g_error("!action_false || !action_true");
    }

    dialog = gtk_message_dialog_new (NULL,
				     GTK_DIALOG_MODAL,
				     GTK_MESSAGE_QUESTION, GTK_BUTTONS_NONE,
				     text);

    button = dialog_button (widgets_p,"xfce/stock_no",action_false);    
    gtk_widget_show (button);
    gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, GTK_RESPONSE_NO);

    button = dialog_button (widgets_p,"xfce/stock_yes",action_true);
    gtk_widget_show (button);
    gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, GTK_RESPONSE_YES);

    gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (widgets_p->window));
    gtk_widget_show (dialog);
    place_dialog(widgets_p->window, dialog);
    return dialog;
}

G_MODULE_EXPORT 
gboolean
xffm_confirm (			widgets_t *widgets_p,
				const gchar *text,
				const gchar *action_false,
				const gchar *action_true){
    int response = GTK_RESPONSE_NONE;
    GtkWidget *dialog = xffm_confirm_dialog (widgets_p, text, action_false, action_true);
    if (!dialog) return FALSE;
    response = gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_hide (dialog);
    gtk_widget_destroy (dialog);

    if (response == GTK_RESPONSE_YES) return TRUE;
    else return FALSE;
}
G_MODULE_EXPORT
const gchar *utf_2_local_string(const gchar *t){
    static gchar *s = NULL;
    GError *error = NULL;
    gsize r_bytes, w_bytes;
    const char *fc;
    gchar *from_codeset=NULL;

    
    g_free(s);
    s=NULL;
    g_get_charset(&fc);
    if (fc) from_codeset = g_strdup(fc);
    else from_codeset = g_strdup("ISO-8859-1");
    
    
    if (strcmp(from_codeset,"ASCII")==0){
	    g_free(from_codeset);
	    from_codeset = g_strdup("ISO-8859-1");
    }    
    TRACE("TRACE: codeset for system is: %s\n",from_codeset);    
    TRACE("TRACE: setting to local codeset-> %s\n",t); 
    if (strcmp(from_codeset,"UTF-8")==0) s = (gchar *)t;
    else {
	s = g_convert (t,strlen(t),from_codeset,"UTF-8",&r_bytes, &w_bytes, &error);
	if(error){
	    g_warning("%s",error->message);
	    g_error_free(error);
	}
    }
    g_free(from_codeset);
    return (const gchar *)s;
}


G_MODULE_EXPORT
gchar *randomTmpName(char *ext)
{
    gchar *fname;
    gchar *rname;
    rname = g_strconcat(g_get_tmp_dir(),G_DIR_SEPARATOR_S,"xffm.XXXXXX",NULL);
    TRACE("TRACE: rname=%s\n",(rname)?rname:"null");
    close(mkstemp(rname));
    if(ext == NULL) fname = g_strdup(rname);
    else fname = g_strconcat(rname,".",ext,NULL);
    TRACE("TRACE: fname=%s\n",(fname)?fname:"null");
    g_free(rname);
    return fname;
}
static 
void purge_cache_files(const gchar *cd){
    DIR *directory;
    char *fullpath;
    struct dirent *d;
	gchar *xdg_dir;
	gchar *cache_dir;
	time_t now=time(NULL);
	xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
	cache_dir = g_build_filename(xdg_dir,"xffm",cd,NULL);  
	g_free(xdg_dir);
	if (!cache_dir) return;
	directory = opendir(cache_dir);
	if(!directory){
	    TRACE("cannot open %s",cache_dir);
	    g_free(cache_dir);
	    return;
	}
	while((d = readdir(directory)) != NULL) 
	{
	    struct stat st;
	    if (strcmp(d->d_name,".")==0 || strcmp(d->d_name,"..")==0)continue;
	    fullpath = g_build_filename((gchar *)cache_dir, d->d_name,NULL);
	    if (strcmp(d->d_name,"..Wastebasket")==0){
		if (fork()==0){
		    execlp("rm","rm","-rf",fullpath,NULL);
		    _exit(123);
		}
	    }
	    if (stat(fullpath,&st) == 0){
		if (now - st.st_mtime > 3600*24*30) {
		    TRACE("removing %s",fullpath);
		    unlink(fullpath);
		}
		else TRACE("not removing %s: age is %ld (%ld days)",fullpath,(long)(now - st.st_mtime),(long)(now - st.st_mtime)/3600/24 );
	    }
	    g_free(fullpath);
        }
	closedir(directory);
        g_free(cache_dir);
	    
}
/* this could also delete tmp files in use by another xffm instance 
 * (slight possibility)
 * done in background*/
G_MODULE_EXPORT
void cleanup_tmpfiles (void)
{
    DIR *directory;
    char *fullpath;
    struct dirent *d;
    const char *path=g_get_tmp_dir();
    pid_t pid=fork();

    if (pid > 0) return;
    
    DBG("cleanup_tmpfiles");

    if (function_void("plugins","xffm_trash","trash_background_purge")){
	    DBG("trash purge done");
    }
    
    directory = opendir(path);
    if(!directory) _exit(1);
    while((d = readdir(directory)) != NULL)
    {
	if(strncmp(d->d_name, "xffm",strlen("xffm"))==0 ||
	   strncmp(d->d_name, "tubopid",strlen("tubopid"))==0 ){
	   fullpath = g_build_filename((gchar *)path, d->d_name,NULL);
	   unlink(fullpath);
	   rmdir(fullpath);
	   g_free(fullpath);
	}
    }
    closedir(directory);
    purge_cache_files("smb");
    purge_cache_files("cache");
    purge_cache_files("thumbnails");
    DBG("cleanup_tmpfiles done");
    _exit(1);
}




