// This is adapted from 'textedit', a demo text viewer found
// in the early versions of the FOX library and written
// by Jeroen van der Zijp.

#include "config.h"
#include "i18n.h"

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <string.h>
#include <unistd.h>

#include <fox-1.4/fx.h>
#include <fox-1.4/fxkeys.h>
#include <fox-1.4/FXPNGIcon.h>

#include "icons.h"
#include "FileDialog.h"
#include "MessageBox.h"
#include "XFileView.h"

FXMainWindow *mainWindow;

// Map
FXDEFMAP(XFileView) XFileViewMap[]={
                                       FXMAPFUNC(SEL_COMMAND,XFileView::ID_ABOUT,XFileView::onCmdAbout),
                                       FXMAPFUNC(SEL_COMMAND,XFileView::ID_OPEN,XFileView::onCmdOpen),
                                       FXMAPFUNC(SEL_UPDATE,XFileView::ID_TITLE,XFileView::onUpdTitle),
                                       FXMAPFUNC(SEL_COMMAND,XFileView::ID_FIND,XFileView::onCmdFind),
                                       FXMAPFUNC(SEL_COMMAND,XFileView::ID_FIND_AGAIN,XFileView::onCmdFind),
                                       FXMAPFUNC(SEL_COMMAND,XFileView::ID_QUIT,XFileView::onCmdQuit),
	                                   FXMAPFUNC(SEL_COMMAND,XFileView::ID_PRINT,XFileView::onCmdPrint),
                                       FXMAPFUNC(SEL_UPDATE,XFileView::ID_FIND_AGAIN,XFileView::onCmdFind),
                                       FXMAPFUNC(SEL_UPDATE,XFileView::ID_FIND,XFileView::onUpdMenu),
                                       FXMAPFUNC(SEL_UPDATE,XFileView::ID_PRINT,XFileView::onUpdMenu),
                                   };


// Object implementation
FXIMPLEMENT(XFileView,FXMainWindow,XFileViewMap,ARRAYNUMBER(XFileViewMap))


// Make some windows
XFileView::XFileView(FXApp* a):FXMainWindow(a,"Xfv ",NULL,NULL,DECOR_ALL,0,0,600,500,0,0)
{
    FXIcon *bigp=new FXPNGIcon(getApp(),xfv);
    setIcon(bigp);
	
    // Create icons
	FXPNGIcon *fileopenicon=new FXPNGIcon(getApp(),fileopen);
	FXPNGIcon *quiticon=new FXPNGIcon(getApp(),quit);
	FXPNGIcon *printicon=new FXPNGIcon(getApp(),print);	
	FXPNGIcon *searchicon=new FXPNGIcon(getApp(),search);		
	FXPNGIcon *findicon=new FXPNGIcon(getApp(),find_again);

	// Make main window; set myself as the target
    setTarget(this);
    setSelector(ID_TITLE);

	// Site where to dock
	FXDockSite *docksite=new FXDockSite(this,LAYOUT_SIDE_TOP|LAYOUT_FILL_X);

    // Make menu bar
  	FXToolBarShell* dragshell=new FXToolBarShell(this,FRAME_RAISED);
    menubar=new FXMenuBar(docksite,dragshell,LAYOUT_DOCK_NEXT|LAYOUT_SIDE_TOP|LAYOUT_FILL_X|FRAME_RAISED);
	new FXToolBarGrip(menubar,menubar,FXMenuBar::ID_TOOLBARGRIP,TOOLBARGRIP_DOUBLE);

    // Tool bar
  	FXToolBarShell* dragshell1=new FXToolBarShell(this,FRAME_RAISED);
  	toolbar=new FXToolBar(docksite,dragshell1,LAYOUT_DOCK_NEXT|LAYOUT_SIDE_TOP|LAYOUT_FILL_X|FRAME_RAISED);
  	new FXToolBarGrip(toolbar,toolbar,FXToolBar::ID_TOOLBARGRIP,TOOLBARGRIP_DOUBLE);
    int style=BUTTON_TOOLBAR;

    // Status bar
    statusbar=new FXStatusBar(this,FRAME_RAISED|LAYOUT_SIDE_BOTTOM|LAYOUT_FILL_X);

    // File menu
    filemenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&File"),NULL,filemenu);

    // Search Menu
    FXMenuPane *searchmenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&Search"),NULL,searchmenu);

    // Preferences Menu
    FXMenuPane *prefsmenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&Preferences"),NULL,prefsmenu);

    // Help menu
    helpmenu=new FXMenuPane(this);
    new FXMenuTitle(menubar,_("&Help"),NULL,helpmenu);

    // Make contents
    contents=new FXHorizontalFrame(this,FRAME_SUNKEN|LAYOUT_FILL_X|LAYOUT_FILL_Y,0,0,0,0, 0,0,0,0, 0,0);

    // Make viewer window
    viewer=new FXText(contents,NULL,0,LAYOUT_FILL_X|LAYOUT_FILL_Y);

    // Show column number in status bar
    new FXTextField(statusbar,4,viewer,FXText::ID_CURSOR_COLUMN,FRAME_SUNKEN|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,0,0,0,0);

    // Caption before number
    new FXLabel(statusbar,_("Col:"),NULL,LAYOUT_RIGHT|LAYOUT_CENTER_Y);

    // Show line number in status bar
    new FXTextField(statusbar,4,viewer,FXText::ID_CURSOR_ROW,FRAME_SUNKEN|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,0,0,0,0);

    // Caption before number
    new FXLabel(statusbar,_("Line:"),NULL,LAYOUT_RIGHT|LAYOUT_CENTER_Y);

    // Toolbar button: File open
    new FXButton(toolbar,_("\tOpen document (Ctrl-O)\tOpen document file. (Ctrl-O)"),fileopenicon,this,ID_OPEN,style|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);

    // Toolbar button: Print
    new FXFrame(toolbar,LAYOUT_TOP|LAYOUT_LEFT|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT,0,0,10,20);
	new FXButton(toolbar,_("\tPrint document (Ctrl-P)\tPrint document file. (Ctrl-P)"),printicon,this,ID_PRINT,style|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);

    // Toolbar buttons: Searching
    new FXFrame(toolbar,LAYOUT_TOP|LAYOUT_LEFT|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT,0,0,10,20);
    new FXButton(toolbar,_("\tFind a string (Ctrl-F)\tFind string in document. (Ctrl-F)"),searchicon,this,ID_FIND,style|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
    new FXButton(toolbar,_("\tFind again (Ctrl-G)\tFind again. (Ctrl-G)"),findicon,this,ID_FIND_AGAIN,style|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);

    // Toolbar button: Quit
    new FXFrame(toolbar,LAYOUT_TOP|LAYOUT_LEFT|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT,0,0,10,20);
    new FXButton(toolbar,_("\tQuit Xfv (Ctrl-Q)\tQuit Xfv. (Ctrl-Q)"),quiticon,this,ID_QUIT,style|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);

    // File Menu entries
    new FXMenuCommand(filemenu,_("&Open file...\tCtrl-O\tOpen document file. (Ctrl-O)"),fileopenicon,this,ID_OPEN);
    new FXMenuCommand(filemenu,_("&Print file...\tCtrl-P\tPrint document file. (Ctrl-P)"),printicon,this,ID_PRINT);
    new FXMenuCommand(filemenu,_("&Quit\tCtrl-Q\tQuit Xfv. (Ctrl Q)"),quiticon,this,ID_QUIT);

    // Search Menu entries
    new FXMenuCommand(searchmenu,_("&Find a string...\tCtrl-F\tFind a string in a document. (Ctrl-F)"),searchicon,this,ID_FIND);
    new FXMenuCommand(searchmenu,_("Find &again\tCtrl-G\tFind again. (Ctrl-G)"),findicon,this,ID_FIND_AGAIN);

    // Preferences menu
    new FXMenuCheck(prefsmenu,_("&Toolbar\t\tDisplay or hide toolbar."),toolbar,FXWindow::ID_TOGGLESHOWN);
    new FXMenuCheck(prefsmenu,_("&Status bar\t\tDisplay or hide status bar."),statusbar,FXWindow::ID_TOGGLESHOWN);

    // Help Menu entries
    new FXMenuCommand(helpmenu,_("&About X File View\tF1\tAbout X File View. (F1)"),NULL,this,ID_ABOUT,0);

    // Make a tool tip
    new FXToolTip(getApp(),0);

    // Initialize file name
    filename="";
}


// About box
long XFileView::onCmdAbout(FXObject*,FXSelector,void*)
{
    FXPNGIcon icon(getApp(),xfv);
    MessageBox about(this,_("About X File View"),_("X File View is a simple text viewer.\n\nCopyright (C) 2002-2005 Roland Baudin (roland65@free.fr)"),&icon,MBOX_OK|DECOR_TITLE|DECOR_BORDER);
    about.execute();
    return 1;
}


// Load file
FXbool XFileView::loadfile(const FXString& file)
{
    struct stat info;
    char *text;
    FILE *fp;
    int n;
    if(stat(file.text(),&info)!=0)
    {
        MessageBox::error(this,MBOX_OK,_("Error Loading File"),_("Non-existing file: %s"),file.text());
        return FALSE;
    }
    fp=fopen(file.text(),"r");
    if(!fp)
    {
        MessageBox::error(this,MBOX_OK,_("Error Loading File"),_("Unable to open file: %s"),file.text());
        return FALSE;
    }
    FXMALLOC(&text,FXchar,info.st_size+1);
    n=fread(text,1,info.st_size,fp);
    if(n!=info.st_size)
    {
        MessageBox::error(this,MBOX_OK,_("Error Reading File"),_("Unable to load entire file: %s"),file.text());
        return FALSE;
    }
    else
    {
        text[info.st_size]='\0';
        viewer->setText(text);
        viewer->setModified(FALSE);
    }
    FXFREE(&text);
    fclose(fp);
    return TRUE;
}


// Open
long XFileView::onCmdOpen(FXObject*,FXSelector,void*)
{
    const FXchar *patterns[]=
        {
            _("All Files"),          "*",
            _("C++ Source Files"),   "*.cc",
            _("C Source Files"),     "*.c",
            _("C/C++ Header Files"),   "*.h",
            _("Text Files"),         "*.txt",
            _("HTML Header Files"),  "*.html",NULL
        };
    FileDialog open(this,_("Open Document"));
	open.setSelectMode(SELECTFILE_EXISTING);
    open.setPatternList(patterns);
  	open.setDirectory(FXFile::directory(filename));
    if(open.execute())
    {
        filename=open.getFilename();
        loadfile(filename);
        viewer->setFocus();
    }
    return 1;
}


// Print the text
long XFileView::onCmdPrint(FXObject*,FXSelector,void*)
{
	FXPrintDialog dlg(this,_("Print File"));
	FXPrinter printer;
	if(dlg.execute())
    	dlg.getPrinter(printer);
  	return 1;
}

// Save configuration when quitting
void XFileView::saveConfig()
{
    // Write new window size and position back to registry
	getApp()->reg().writeUnsignedEntry("SETTINGS","width",(FXuint)getWidth());
	getApp()->reg().writeUnsignedEntry("SETTINGS","height",(FXuint)getHeight());
	getApp()->reg().writeIntEntry("SETTINGS","xpos",getX());
	getApp()->reg().writeIntEntry("SETTINGS","ypos",getY());

    // Toolbar status
    if(toolbar->shown())
        getApp()->reg().writeUnsignedEntry("SETTINGS","showtoolbar",TRUE);
    else
        getApp()->reg().writeUnsignedEntry("SETTINGS","showtoolbar",FALSE);

    // Status bar status
    if(statusbar->shown())
        getApp()->reg().writeUnsignedEntry("SETTINGS","showstatusbar",TRUE);
    else
        getApp()->reg().writeUnsignedEntry("SETTINGS","showstatusbar",FALSE);

	// Write registry settings
	getApp()->reg().write();
}


// Quit
long XFileView::onCmdQuit(FXObject* sender,FXSelector sel,void* ptr)
{
	// Save settings 
    saveConfig();

    // Quit
    getApp()->exit(0);
    return 1;
}


// Update title
long XFileView::onUpdTitle(FXObject* sender,FXSelector,void*)
{
    FXString title="Xfv " + filename;
    sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETSTRINGVALUE),(void*)&title);
    return 1;
}


// Update open, print and find menus
long XFileView::onUpdMenu(FXObject* sender,FXSelector,void*)
{
    if (viewer->getLength()==0)
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
	else
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
    return 1;
}


// Open find dialog
long XFileView::onCmdFind(FXObject* obj,FXSelector sel,void* ptr)
{
    int loc;

    static FXString str="";

    if(FXSELTYPE(sel)==SEL_UPDATE)
    {
        if(str=="")
            ((FXWindow*)obj)->disable();
        else
            ((FXWindow*)obj)->enable();
        return 1;
    }

    if(FXSELID(sel)==ID_FIND)
    {
        FXDialogBox dialog(this,_("Find"),DECOR_TITLE|DECOR_BORDER);
        FXVerticalFrame* content=new FXVerticalFrame(&dialog,LAYOUT_FILL_X|LAYOUT_FILL_Y,0,0,0,0,10,10,10,10,10,10);
        text=new FXTextField(content,30,NULL,0,FRAME_SUNKEN|LAYOUT_FILL_X);
        new FXHorizontalSeparator(content,SEPARATOR_GROOVE|LAYOUT_FILL_X);
        FXHorizontalFrame* buttons=new FXHorizontalFrame(content,LAYOUT_FILL_X|PACK_UNIFORM_WIDTH,0,0,0,0,0,0,0,0);
        new FXButton(buttons,_("&Cancel"),NULL,&dialog,FXDialogBox::ID_CANCEL,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT,0,0,0,0,20,20);
        new FXButton(buttons,_("&Find"),NULL,&dialog,FXDialogBox::ID_ACCEPT,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT);
        text->setText(str);
        text->setSelection(0,str.length());
        text->setFocus();

        if(!dialog.execute())
            return 0;
        str=text->getText();
    }
    loc=find(str,viewer->getCursorPos());
    if(loc==-1)
        MessageBox::information(this,MBOX_OK,_("Not Found"),_("String '%s' not found"),str.text());
    else
    {
        viewer->setCursorPos(loc+str.length());
        viewer->setTopLine(loc);
        viewer->setSelection(loc,str.length());
        viewer->setFocus();
    }
    return 1;
}


// Find string
FXint XFileView::find(const FXString str,FXint pos)
{
    FXString buf=viewer->getText();
    const char* text=buf.text();
    char *p = strstr(&text[pos],str.text());
    if(!p)
        return -1;
    return (p-text);
}


// Start the ball rolling
void XFileView::start(FXString startfile)
{
    filename=startfile;
	if(filename != "")
		loadfile(filename);
}


// Create and show window
void XFileView::create()
{
    // Read the Xfe registry
	FXRegistry* reg_xfe=new FXRegistry("Xfe",NULL);
	reg_xfe->read();
	
	// Set Xfv text font according to the Xfe registry
	FXString fontspec;
	fontspec=reg_xfe->readStringEntry("SETTINGS","textfont","Helvetica,100,normal,regular");
	if(!fontspec.empty())
	{
    	FXFont* font=new FXFont(getApp(),fontspec);
        viewer->setFont(font);
	}
	delete reg_xfe;

	// Wheel scrolling
    FXint wheellines=getApp()->reg().readUnsignedEntry("SETTINGS","wheellines",5);
	getApp()->setWheelLines(wheellines);
	
    // Get size and position
    FXuint width=getApp()->reg().readUnsignedEntry("SETTINGS","width",600);
    FXuint height=getApp()->reg().readUnsignedEntry("SETTINGS","height",400);
    FXint xpos=getApp()->reg().readIntEntry("SETTINGS","xpos",0);
    FXint ypos=getApp()->reg().readIntEntry("SETTINGS","ypos",0);

	// Get toolbar status
    if(getApp()->reg().readUnsignedEntry("SETTINGS","showtoolbar",TRUE)==FALSE)
        toolbar->hide();
   
	// Get status bar status
    if(getApp()->reg().readUnsignedEntry("SETTINGS","showstatusbar",TRUE)==FALSE)
        statusbar->hide();
   
	// Reposition window
    position(xpos,ypos,width,height);

    FXMainWindow::create();
    show();
    viewer->setFocus();
    viewer->handle(this,FXSEL(SEL_COMMAND,FXText::ID_TOGGLE_EDITABLE),NULL);
}



// Usage message
#define USAGE_MSG	_("\
Usage: xfv [options] [file] \n\
\n\
    [options] can be any of the following:\n\
\n\
        -h, --help         Print (this) help screen and exit.\n\
        -v, --version      Print version information and exit.\n\
\n\
    [file] is the path to the file you want to open on start up.\n\
\n")


// Start the whole thing
int main(int argc,char *argv[])
{
	int i;
	FXString startfile="";
	char *appname = "Xfv";
	char *xfename = "Xfe";
	char *vdrname = "XFileExplorer";

#ifdef HAVE_SETLOCALE
  // Set locale via LC_ALL.
  setlocale (LC_ALL, "");
#endif

#if ENABLE_NLS
  // Set the text message domain.
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);
#endif

	// Parse basic arguments
	for(i = 1; i < argc; ++i)
	{
    	if(compare(argv[i],"-v")==0 || compare(argv[i],"--version")==0)
		{
			fprintf(stdout,"%s version %s\n",PACKAGE,VERSION);
			exit(0);
		}
    	else if(compare(argv[i],"-h")==0 || compare(argv[i],"--help")==0)
		{
			fprintf(stdout,USAGE_MSG);
			exit(0);
		}
		else
	    {
			// Start file, if any
			startfile=argv[i];
	    }
	}

    // Make application
    FXApp* application=new FXApp(appname,vdrname);

    // Open display
    application->init(argc,argv);

    // Read the Xfe registry
	FXRegistry* reg=new FXRegistry(xfename,vdrname);
	reg->read();
	
	// Set Xfv normal font according to the Xfe registry
	FXString fontspec;
	fontspec=reg->readStringEntry("SETTINGS","font","Helvetica,100,normal,regular");
	if(!fontspec.empty())
	{
    	FXFont* normalFont=new FXFont(application,fontspec);
        application->setNormalFont(normalFont);
    }

	// Set Xfv colors according to the Xfe registry
	FXColor basecolor,forecolor,backcolor,selforecolor,selbackcolor,bordercolor;
    basecolor=reg->readColorEntry("SETTINGS","basecolor",FXRGB(214,207,198));
    forecolor=reg->readColorEntry("SETTINGS","forecolor",FXRGB(0,0,0));
    backcolor=reg->readColorEntry("SETTINGS","backcolor",FXRGB(255,255,255));
    selforecolor=reg->readColorEntry("SETTINGS","selforecolor",FXRGB(255,255,255));
    selbackcolor=reg->readColorEntry("SETTINGS","selbackcolor",FXRGB(10,36,106));
    bordercolor=reg->readColorEntry("SETTINGS","bordercolor",FXRGB(0,0,0));
	application->setBaseColor(basecolor);
	application->setForeColor(forecolor);
	application->setBackColor(backcolor);
	application->setSelforeColor(selforecolor);
	application->setSelbackColor(selbackcolor);
	application->setBorderColor(bordercolor);

	// Delete the Xfe registry
	delete reg;
    
	// Make window
    XFileView* window=new XFileView(application);
	mainWindow=window;
	
    // Create it
    application->create();

    // Start
    window->start(startfile);

    // Run
    application->run();
}

