// Directory list. Taken from the FOX library and slightly modified.

#include "config.h"
#include "i18n.h"

#include <fox-1.4/fx.h>
#include <fox-1.4/FXPNGIcon.h>
#if defined(linux)
#include <mntent.h>
#endif

#include "xfedefs.h"
#include "icons.h"
#include "DirList.h"
#include "InputDialog.h"
#include "File.h"
#include "FileDict.h"
#include "MessageBox.h"


// Interval between updevices and mtdevices read (ms)
#define UPDEVICES_INTERVAL		65000
#define MTDEVICES_INTERVAL		1000

// Interval between refreshes (ms)
#define REFRESH_INTERVAL     1000

// File systems not supporting mod-time, refresh every nth time
#define REFRESH_FREQUENCY    32

// Time interval before expanding a folder (ms)
#define EXPAND_INTERVAL		500

extern FXbool ask_before_copy;


#if defined(linux)
#define FSTAB_PATH "/etc/fstab"
#define MTAB_PATH "/proc/mounts"
extern FXStringDict* fsdevices;
extern FXStringDict* mtdevices;
extern FXStringDict* updevices;
extern FXbool mount_warn;
#endif


// Object implementation
FXIMPLEMENT(DirItem,FXTreeItem,NULL,0)



// Map
FXDEFMAP(DirList) DirListMap[]={
                                   FXMAPFUNC(SEL_DRAGGED,0,DirList::onDragged),
                                   FXMAPFUNC(SEL_TIMEOUT,DirList::ID_REFRESH_TIMER,DirList::onRefreshTimer),
#if defined(linux)
                                   FXMAPFUNC(SEL_TIMEOUT,DirList::ID_MTDEVICES_REFRESH,DirList::onMtdevicesRefresh),
                                   FXMAPFUNC(SEL_TIMEOUT,DirList::ID_UPDEVICES_REFRESH,DirList::onUpdevicesRefresh),
#endif
                                   FXMAPFUNC(SEL_TIMEOUT,DirList::ID_EXPAND_TIMER,DirList::onExpandTimer),
                                   FXMAPFUNC(SEL_DND_ENTER,0,DirList::onDNDEnter),
                                   FXMAPFUNC(SEL_DND_LEAVE,0,DirList::onDNDLeave),
                                   FXMAPFUNC(SEL_DND_DROP,0,DirList::onDNDDrop),
                                   FXMAPFUNC(SEL_DND_MOTION,0,DirList::onDNDMotion),
                                   FXMAPFUNC(SEL_DND_REQUEST,0,DirList::onDNDRequest),
                                   FXMAPFUNC(SEL_BEGINDRAG,0,DirList::onBeginDrag),
                                   FXMAPFUNC(SEL_ENDDRAG,0,DirList::onEndDrag),
                                   FXMAPFUNC(SEL_OPENED,0,DirList::onOpened),
                                   FXMAPFUNC(SEL_CLOSED,0,DirList::onClosed),
                                   FXMAPFUNC(SEL_EXPANDED,0,DirList::onExpanded),
                                   FXMAPFUNC(SEL_COLLAPSED,0,DirList::onCollapsed),
                                   FXMAPFUNC(SEL_UPDATE,DirList::ID_SHOW_HIDDEN,DirList::onUpdShowHidden),
                                   FXMAPFUNC(SEL_UPDATE,DirList::ID_HIDE_HIDDEN,DirList::onUpdHideHidden),
                                   FXMAPFUNC(SEL_UPDATE,DirList::ID_TOGGLE_HIDDEN,DirList::onUpdToggleHidden),
                                   FXMAPFUNC(SEL_UPDATE,DirList::ID_SHOW_FILES,DirList::onUpdShowFiles),
                                   FXMAPFUNC(SEL_UPDATE,DirList::ID_HIDE_FILES,DirList::onUpdHideFiles),
                                   FXMAPFUNC(SEL_UPDATE,DirList::ID_TOGGLE_FILES,DirList::onUpdToggleFiles),
                                   FXMAPFUNC(SEL_UPDATE,DirList::ID_SET_PATTERN,DirList::onUpdSetPattern),
                                   FXMAPFUNC(SEL_UPDATE,DirList::ID_SORT_REVERSE,DirList::onUpdSortReverse),
                                   FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETVALUE,DirList::onCmdSetValue),
                                   FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETSTRINGVALUE,DirList::onCmdSetStringValue),
                                   FXMAPFUNC(SEL_COMMAND,FXWindow::ID_GETSTRINGVALUE,DirList::onCmdGetStringValue),
                                   FXMAPFUNC(SEL_COMMAND,DirList::ID_SHOW_HIDDEN,DirList::onCmdShowHidden),
                                   FXMAPFUNC(SEL_COMMAND,DirList::ID_HIDE_HIDDEN,DirList::onCmdHideHidden),
                                   FXMAPFUNC(SEL_COMMAND,DirList::ID_TOGGLE_HIDDEN,DirList::onCmdToggleHidden),
                                   FXMAPFUNC(SEL_COMMAND,DirList::ID_SHOW_FILES,DirList::onCmdShowFiles),
                                   FXMAPFUNC(SEL_COMMAND,DirList::ID_HIDE_FILES,DirList::onCmdHideFiles),
                                   FXMAPFUNC(SEL_COMMAND,DirList::ID_TOGGLE_FILES,DirList::onCmdToggleFiles),
                                   FXMAPFUNC(SEL_COMMAND,DirList::ID_SET_PATTERN,DirList::onCmdSetPattern),
                                   FXMAPFUNC(SEL_COMMAND,DirList::ID_SORT_REVERSE,DirList::onCmdSortReverse),
                               };


// Object implementation
FXIMPLEMENT(DirList,FXTreeList,DirListMap,ARRAYNUMBER(DirListMap))


// For serialization
DirList::DirList()
{
    flags|=FLAG_ENABLED|FLAG_DROPTARGET;
    counter=0;
    matchmode=0;
    associations=NULL;
    dropaction=DRAG_MOVE;
    sortfunc=(FXTreeListSortFunc)ascendingCase;
    list=NULL;
}


// Directory List Widget
DirList::DirList(FXComposite *p,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h):
        FXTreeList(p,tgt,sel,opts,x,y,w,h),pattern("*")
{
    flags|=FLAG_ENABLED|FLAG_DROPTARGET;
    open_folder=new FXPNGIcon(getApp(),minifolderopen);
    closed_folder=new FXPNGIcon(getApp(),minifolderclosed);
    mini_doc=new FXPNGIcon(getApp(),minidoc);
    mini_app=new FXPNGIcon(getApp(),miniapp);
    locked_folder=new FXPNGIcon(getApp(),minifolderlocked);
    harddiskicon=new FXPNGIcon(getApp(),harddisk);
    nfsicon=new FXPNGIcon(getApp(),nfsdrive);
    floppyicon=new FXPNGIcon(getApp(),floppy);
    cdromicon=new FXPNGIcon(getApp(),cdrom);
    zipicon=new FXPNGIcon(getApp(),zip);
    linkicon=new FXPNGIcon(getApp(),minilink);
    matchmode=FILEMATCH_FILE_NAME|FILEMATCH_NOESCAPE;
    associations=NULL;
    if(!(options&DIRLIST_NO_OWN_ASSOC))
        associations=new FileDict(getApp());
    list=NULL;
    sortfunc=(FXTreeListSortFunc)ascendingCase;
    dropaction=DRAG_MOVE;
    counter=0;

#if defined(linux)

    // Initialize the fsdevices, mtdevices and updevices lists
    struct mntent *mnt;
    if (fsdevices==NULL)
    {
        // To list file system devices
        fsdevices=new FXStringDict();
        FILE *fstab=setmntent(FSTAB_PATH,"r");
        if(fstab)
        {
            while((mnt=getmntent(fstab)))
            {
                if(!::streq(mnt->mnt_type,MNTTYPE_IGNORE) && !::streq(mnt->mnt_type,MNTTYPE_SWAP) && !::streq(mnt->mnt_type,"proc") && !::streq(mnt->mnt_dir,"/"))
                {
                    if(!strncmp(mnt->mnt_fsname,"/dev/fd",7))
                        fsdevices->insert(mnt->mnt_dir,"floppy");
                    else if (!strncmp(mnt->mnt_type,"iso",3))
                        fsdevices->insert(mnt->mnt_dir,"cdrom");
                    else if (!strncmp(mnt->mnt_fsname,"/dev/zip",8))
                        fsdevices->insert(mnt->mnt_dir,"zip");
                    else if (::streq(mnt->mnt_type,"nfs"))
                        fsdevices->insert(mnt->mnt_dir,"nfsdisk");
                    else if (::streq(mnt->mnt_type,"smbfs"))
                        fsdevices->insert(mnt->mnt_dir,"smbdisk");
                    else
                        fsdevices->insert(mnt->mnt_dir,"harddisk");
                }
            }
            endmntent(fstab);
        }
    }
    if (mtdevices==NULL)
    {
        // To list mounted devices
        mtdevices=new FXStringDict();
        FILE *mtab=setmntent(MTAB_PATH,"r");
        if(mtab)
        {
            while((mnt=getmntent(mtab)))
                mtdevices->insert(mnt->mnt_dir,mnt->mnt_type);
            endmntent(mtab);
        }
    }
	if (updevices==NULL)
    {
        // To mark mount points that are up or down
        updevices=new FXStringDict();
        struct stat statbuf;
        FXString mtstate;
        FILE *mtab=setmntent(MTAB_PATH,"r");
        if(mtab)
        {
            while((mnt=getmntent(mtab)))
            {
                if (mtlstat(mnt->mnt_dir,&statbuf)==-1)
                    mtstate="down";
                else
                    mtstate="up";
                updevices->insert(mnt->mnt_dir,mtstate.text());
            }
            endmntent(mtab);
        }
    }
#endif
}


// Create the directory list
void DirList::create()
{

    FXTreeList::create();
    if(!deleteType)
        deleteType=getApp()->registerDragType(deleteTypeName);
    if(!urilistType)
        urilistType=getApp()->registerDragType(urilistTypeName);
    getApp()->addTimeout(this,ID_REFRESH_TIMER,REFRESH_INTERVAL);
#if defined(linux)
    getApp()->addTimeout(this,ID_MTDEVICES_REFRESH,MTDEVICES_INTERVAL);
    getApp()->addTimeout(this,ID_UPDEVICES_REFRESH,UPDEVICES_INTERVAL);
#endif
    dropEnable();
    open_folder->create();
    closed_folder->create();
    mini_doc->create();
    mini_app->create();
    locked_folder->create();
    harddiskicon->create();
    nfsicon->create();
    floppyicon->create();
    cdromicon->create();
    zipicon->create();
    linkicon->create();

    // Scan root directory
    scan(FALSE);
}


// Expand folder tree when hovering long over a folder
long DirList::onExpandTimer(FXObject* sender,FXSelector sel,void* ptr)
{
    FXint xx,yy;
    FXuint state;
    DirItem *item;

    getCursorPosition(xx,yy,state);
    item=(DirItem*)getItemAt(xx,yy);

    if(!(item->state&DirItem::FOLDER))
        return 0;

    // Expand tree item
    expandTree((TreeItem*)item,TRUE);
    scan(TRUE);

    // Set open timer
    getApp()->addTimeout(this,ID_EXPAND_TIMER,EXPAND_INTERVAL);

    return 1;

}

// Create item
TreeItem* DirList::createItem(const FXString& text,FXIcon* oi,FXIcon* ci,void* ptr)
{
    return (TreeItem*) new DirItem(text,oi,ci,ptr);
}


// Sort ascending order, keeping directories first
FXint DirList::ascending(const TreeItem* pa,const TreeItem* pb)
{
    register const DirItem *a=(DirItem*)pa;
    register const DirItem *b=(DirItem*)pb;
    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    return diff ? diff : compare(a->label,b->label);
}


// Sort descending order, keeping directories first
FXint DirList::descending(const TreeItem* pa,const TreeItem* pb)
{
    register const DirItem *a=(DirItem*)pa;
    register const DirItem *b=(DirItem*)pb;
    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    return diff ? diff : compare(b->label,a->label);
}


// Sort ascending order, case insensitive, keeping directories first
FXint DirList::ascendingCase(const TreeItem* pa,const TreeItem* pb)
{
    register const DirItem *a=(DirItem*)pa;
    register const DirItem *b=(DirItem*)pb;
    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    return diff ? diff : comparecase(a->label,b->label);
}


// Sort descending order, case insensitive, keeping directories first
FXint DirList::descendingCase(const TreeItem* pa,const TreeItem* pb)
{
    register const DirItem *a=(DirItem*)pa;
    register const DirItem *b=(DirItem*)pb;
    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    return diff ? diff : comparecase(b->label,a->label);
}



// Handle drag-and-drop enter
long DirList::onDNDEnter(FXObject* sender,FXSelector sel,void* ptr)
{
    FXTreeList::onDNDEnter(sender,sel,ptr);
    return 1;
}


// Handle drag-and-drop leave
long DirList::onDNDLeave(FXObject* sender,FXSelector sel,void* ptr)
{
    // Cancel open up timer
    getApp()->removeTimeout(this,ID_EXPAND_TIMER);

    stopAutoScroll();
    FXTreeList::onDNDLeave(sender,sel,ptr);
    if(prevSelItem)
    {
        if(!isItemCurrent(prevSelItem))
            closeItem(prevSelItem);
        prevSelItem = NULL;
    }
    return 1;
}


// Handle drag-and-drop motion
long DirList::onDNDMotion(FXObject* sender,FXSelector sel,void* ptr)
{
    FXEvent *event=(FXEvent*)ptr;
    TreeItem *item;

    // Cancel open up timer
    getApp()->removeTimeout(this,ID_EXPAND_TIMER);

    // Start autoscrolling
    if(startAutoScroll(event,FALSE))
        return 1;

    // Give base class a shot
    if(FXTreeList::onDNDMotion(sender,sel,ptr))
        return 1;

    // Dropping list of filenames
    if(offeredDNDType(FROM_DRAGNDROP,urilistType))
    {

        // Locate drop place
        item=(TreeItem*)getItemAt(event->win_x,event->win_y);

        // We can drop in a directory
        if(item && isItemDirectory(item))
        {
            // Get drop directory
            dropdirectory=getItemPathname(item);

            // What is being done (move,copy,link)
            dropaction=inquireDNDAction();

            // Set open up timer
            getApp()->addTimeout(this,ID_EXPAND_TIMER,EXPAND_INTERVAL);

            // See if this is writable
            if(::isWritable(dropdirectory))
            {
                acceptDrop(DRAG_ACCEPT);
                FXint x,y;
                FXuint state;
                getCursorPosition(x,y,state);
                TreeItem* item=(TreeItem*)getItemAt(x,y);

                if(prevSelItem && prevSelItem != item)
                {
                    if(!isItemCurrent(prevSelItem))
                        closeItem(prevSelItem);
                    prevSelItem = NULL;
                }
                if(item && prevSelItem != item)
                {
                    openItem(item);
                    prevSelItem = item;
                }
            }
        }
        return 1;
    }
    return 0;
}


// Handle drag-and-drop drop
long DirList::onDNDDrop(FXObject* sender,FXSelector sel,void* ptr)
{
    FXuchar *data;
    FXuint len;
    FXbool showdialog=TRUE;
    File *f=NULL;

    if(prevSelItem)
    {
        if(!isItemCurrent(prevSelItem))
            closeItem(prevSelItem);
        prevSelItem = NULL;
    }

    // Cancel open up timer
    getApp()->removeTimeout(this,ID_EXPAND_TIMER);

    // Stop scrolling
    stopAutoScroll();

    // Perhaps target wants to deal with it
    if(FXTreeList::onDNDDrop(sender,sel,ptr))
        return 1;

    // Get uri-list of files being dropped
    if(getDNDData(FROM_DRAGNDROP,urilistType,data,len))
    {
        FXRESIZE(&data,FXuchar,len+1);
        data[len]='\0';
        FXchar *p,*q;
        p=q=(FXchar*)data;
        
		// Number of selected items
		FXString buf=p;
        int num=buf.count('\n')+1;
		
		// Eventually correct the number of selected items
		// because sometimes there is another '\n' at the end of the string
		FXint pos=buf.rfind('\n');
		if (pos==buf.length()-1)
			num=num-1;

        // File object
        if (dropaction==DRAG_COPY)
            f=new File(this,_("File copy"),COPY);
        else if (dropaction==DRAG_MOVE)
            f=new File(this,_("File move"),MOVE);
        else if ((dropaction==DRAG_LINK) && (num==1))
            f=new File(this,_("File symlink"),SYMLINK);
        else if ((dropaction==DRAG_LINK) && (num>1))
		{
			MessageBox::error(this,MBOX_OK,_("Error"),_("Can't symlink multiple selection!"));
			return 0;
		}
		else
            return 0;

        while(*p)
        {
            while(*q && *q!='\r')
                q++;
            FXString url(p,q-p);
            FXString source(FXURL::fileFromURL(url));
            FXString target(dropdirectory+PATHSEPSTRING+FXFile::name(source));
            FXString sourcedir=FXFile::directory(source);
            FXString targetdir=FXFile::directory(target);

            // File operation dialog, if needed
            if(ask_before_copy & showdialog)
            {
                FXIcon *icon=NULL;
                FXString title,message;
                if (dropaction==DRAG_COPY)
                {
                    title=_("Copy ");
                    icon = new FXPNGIcon(getApp(),copy_big);
                    if (num==1)
                        message=title+source;
                    else
                        message=_("Copy ")+FXStringVal(num)+_(" files/folders.\nFrom: ")+sourcedir;
                }
                else if (dropaction==DRAG_MOVE)
                {
                    title=_("Move ");
                    icon = new FXPNGIcon(getApp(),move_big);
                    if (num==1)
                        message=title+source;
                    else
                        message=_("Move ")+FXStringVal(num)+_(" files/folders.\nFrom: ")+sourcedir;
                }
                else if ((dropaction==DRAG_LINK) && (num==1))
                {
                    title=_("Symlink ");
                    icon=new FXPNGIcon(getApp(),link_big);
					message=title+source;
                }

                InputDialog* dialog = new InputDialog(this,targetdir,message,title,_("To:"),icon);
                dialog->CursorEnd();
                int rc=1;
                rc=dialog->execute();
                target=dialog->getText();
                target=::filePath(target);
                if (num>1)
                    showdialog=FALSE;
                delete dialog;
                delete icon;
                if (!rc)
                    return 0;
            }

            // Move the source file
            if(dropaction==DRAG_MOVE)
            {
                // Move file
                f->create();
                f->move(source,target);

                // If action is cancelled in progress dialog
                if (f->isCancelled)
                {
                    f->hide();
                    MessageBox::error(this,MBOX_OK,_("Error"),_("Move file operation cancelled!"));
                }
            }
            // Copy the source file
            else if(dropaction==DRAG_COPY)
            {
                // Copy file
                f->create();
                f->copy(source,target);

                // If action is cancelled in progress dialog
                if (f->isCancelled)
                {
                    f->hide();
                    MessageBox::error(this,MBOX_OK,_("Error"),_("Copy file operation cancelled!"));
                }
            }
            // Link the source file (no progress dialog in this case)
            else if(dropaction==DRAG_LINK)
            {
                // Link file
                f->create();
                f->symlink(source,target);
            }
            if(*q=='\r')
                q+=2;
            p=q;
        }
        delete f;
        FXFREE(&data);
        return 1;
    }
    return 0;
}


// Somebody wants our dragged data
long DirList::onDNDRequest(FXObject* sender,FXSelector sel,void* ptr)
{
    FXEvent *event=(FXEvent*)ptr;
    FXuchar *data;
    FXuint len;

    // Perhaps the target wants to supply its own data
    if(FXTreeList::onDNDRequest(sender,sel,ptr))
        return 1;

    // Return list of filenames as a uri-list
    if(event->target==urilistType)
    {
        if(!dragfiles.empty())
        {
            len=dragfiles.length();
            FXMEMDUP(&data,dragfiles.text(),FXuchar,len);
            setDNDData(FROM_DRAGNDROP,event->target,data,len);
        }
        return 1;
    }

    // Delete selected files
    if(event->target==deleteType)
        return 1;

    return 0;
}


// Start a drag operation
long DirList::onBeginDrag(FXObject* sender,FXSelector sel,void* ptr)
{
    register TreeItem *item;
    if(FXTreeList::onBeginDrag(sender,sel,ptr))
        return 1;
    if(beginDrag(&urilistType,1))
    {
        dragfiles=FXString::null;
        item=(TreeItem*)firstitem;
        while(item)
        {
            if(item->isSelected())
            {
                if(!dragfiles.empty())
                    dragfiles+="\r\n";
                dragfiles+=FXURL::fileToURL(getItemPathname(item));
            }
            if(item->first)
                item=(TreeItem*)item->first;
            else
            {
                while(!item->next && item->parent)
                    item=(TreeItem*)item->parent;
                item=(TreeItem*)item->next;
            }
        }
        return 1;
    }
    return 0;
}


// End drag operation
long DirList::onEndDrag(FXObject* sender,FXSelector sel,void* ptr)
{
    if(FXTreeList::onEndDrag(sender,sel,ptr))
        return 1;
    endDrag((didAccept()!=DRAG_REJECT));
    setDragCursor(getDefaultCursor());
    return 1;
}


// Dragged stuff around
long DirList::onDragged(FXObject* sender,FXSelector sel,void* ptr)
{
    FXEvent* event=(FXEvent*)ptr;
    FXDragAction action;
    if(FXTreeList::onDragged(sender,sel,ptr))
        return 1;
    action=DRAG_MOVE;
    if(event->state&CONTROLMASK)
        action=DRAG_COPY;
    if(event->state&SHIFTMASK)
        action=DRAG_MOVE;
	if((event->state&CONTROLMASK) && (event->state&SHIFTMASK))
		action=DRAG_LINK;
    handleDrag(event->root_x,event->root_y,action);
    if(didAccept()!=DRAG_REJECT)
    {
        if(action==DRAG_MOVE)
            setDragCursor(getApp()->getDefaultCursor(DEF_DNDMOVE_CURSOR));
    	else if(action==DRAG_LINK)
			setDragCursor(getApp()->getDefaultCursor(DEF_DNDLINK_CURSOR));
        else
            setDragCursor(getApp()->getDefaultCursor(DEF_DNDCOPY_CURSOR));
    }
    else
        setDragCursor(getApp()->getDefaultCursor(DEF_DNDSTOP_CURSOR));
    return 1;
}


// Open up the path down to the given string
long DirList::onCmdSetValue(FXObject*,FXSelector,void* ptr)
{
    if(ptr)
        setCurrentFile((const FXchar*)ptr);
    return 1;
}


// Open up the path down to the given string
long DirList::onCmdSetStringValue(FXObject*,FXSelector,void* ptr)
{
    if(ptr)
        setCurrentFile(*((FXString*)ptr));
    return 1;
}


// Obtain value of the current item
long DirList::onCmdGetStringValue(FXObject*,FXSelector,void* ptr)
{
    if(ptr)
        *((FXString*)ptr)=getCurrentFile();
    return 1;
}


// Toggle hidden files
long DirList::onCmdToggleHidden(FXObject*,FXSelector,void*)
{
    showHiddenFiles(!showHiddenFiles());
    return 1;
}


// Update toggle hidden files widget
long DirList::onUpdToggleHidden(FXObject* sender,FXSelector,void*)
{
    if(showHiddenFiles())
        sender->handle(this,FXSEL(SEL_COMMAND,ID_CHECK),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Show hidden files
long DirList::onCmdShowHidden(FXObject*,FXSelector,void*)
{
    showHiddenFiles(TRUE);
    return 1;
}


// Update show hidden files widget
long DirList::onUpdShowHidden(FXObject* sender,FXSelector,void*)
{
    if(showHiddenFiles())
        sender->handle(this,FXSEL(SEL_COMMAND,ID_CHECK),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Hide hidden files
long DirList::onCmdHideHidden(FXObject*,FXSelector,void*)
{
    showHiddenFiles(FALSE);
    return 1;
}


// Update hide hidden files widget
long DirList::onUpdHideHidden(FXObject* sender,FXSelector,void*)
{
    if(!showHiddenFiles())
        sender->handle(this,FXSEL(SEL_COMMAND,ID_CHECK),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Toggle files display
long DirList::onCmdToggleFiles(FXObject*,FXSelector,void*)
{
    showFiles(!showFiles());
    return 1;
}


// Update toggle files widget
long DirList::onUpdToggleFiles(FXObject* sender,FXSelector,void*)
{
    if(showFiles())
        sender->handle(this,FXSEL(SEL_COMMAND,ID_CHECK),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Show files
long DirList::onCmdShowFiles(FXObject*,FXSelector,void*)
{
    showFiles(TRUE);
    return 1;
}


// Update show files widget
long DirList::onUpdShowFiles(FXObject* sender,FXSelector,void*)
{
    if(showFiles())
        sender->handle(this,FXSEL(SEL_COMMAND,ID_CHECK),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Hide files
long DirList::onCmdHideFiles(FXObject*,FXSelector,void*)
{
    showFiles(FALSE);
    return 1;
}


// Update hide files widget
long DirList::onUpdHideFiles(FXObject* sender,FXSelector,void*)
{
    if(!showFiles())
        sender->handle(this,FXSEL(SEL_COMMAND,ID_CHECK),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Change pattern
long DirList::onCmdSetPattern(FXObject*,FXSelector,void* ptr)
{
    if(!ptr)
        return 0;
    setPattern((const char*)ptr);
    return 1;
}


// Update pattern
long DirList::onUpdSetPattern(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETVALUE),(void*)pattern.text());
    return 1;
}


// Reverse sort order
long DirList::onCmdSortReverse(FXObject*,FXSelector,void*)
{
    if(sortfunc==(FXTreeListSortFunc)ascending)
        sortfunc=(FXTreeListSortFunc)descending;
    else if(sortfunc==(FXTreeListSortFunc)descending)
        sortfunc=(FXTreeListSortFunc)ascending;
    else if(sortfunc==(FXTreeListSortFunc)ascendingCase)
        sortfunc=(FXTreeListSortFunc)descendingCase;
    else if(sortfunc==(FXTreeListSortFunc)descendingCase)
        sortfunc=(FXTreeListSortFunc)ascendingCase;
    scan(TRUE);
    return 1;
}


// Update sender
long DirList::onUpdSortReverse(FXObject* sender,FXSelector,void* ptr)
{
    sender->handle(this,(sortfunc==(FXTreeListSortFunc)descending || sortfunc==(FXTreeListSortFunc)descendingCase) ? FXSEL(SEL_COMMAND,ID_CHECK) : FXSEL(SEL_COMMAND,ID_UNCHECK),ptr);
    return 1;
}

// Toggle case sensitivity
long DirList::onCmdSortCase(FXObject*,FXSelector,void*)
{
    if(sortfunc==(FXTreeListSortFunc)ascending)
        sortfunc=(FXTreeListSortFunc)ascendingCase;
    else if(sortfunc==(FXTreeListSortFunc)descending)
        sortfunc=(FXTreeListSortFunc)descendingCase;
    else if(sortfunc==(FXTreeListSortFunc)ascendingCase)
        sortfunc=(FXTreeListSortFunc)ascending;
    else if(sortfunc==(FXTreeListSortFunc)descendingCase)
        sortfunc=(FXTreeListSortFunc)descending;
    scan(TRUE);
    return 1;
}


// Check if case sensitive
long DirList::onUpdSortCase(FXObject* sender,FXSelector,void* ptr)
{
    sender->handle(this,(sortfunc==(FXTreeListSortFunc)ascendingCase || sortfunc==(FXTreeListSortFunc)descendingCase) ? FXSEL(SEL_COMMAND,ID_CHECK) : FXSEL(SEL_COMMAND,ID_UNCHECK),ptr);
    return 1;
}


// Close directory
long DirList::onClosed(FXObject*,FXSelector,void* ptr)
{
    DirItem *item=(DirItem*)ptr;
    if(item->state&DirItem::FOLDER)
        return target && target->handle(this,FXSEL(SEL_CLOSED,message),ptr);

    return 1;
}


// Open directory
long DirList::onOpened(FXObject*,FXSelector,void* ptr)
{
    DirItem *item=(DirItem*)ptr;
    if(item->state&DirItem::FOLDER)
        return target && target->handle(this,FXSEL(SEL_OPENED,message),ptr);
    return 1;
}


// Item opened
long DirList::onExpanded(FXObject* sender,FXSelector sel,void* ptr)
{
    DirItem *item=(DirItem*)ptr;

    if(!(item->state&DirItem::FOLDER))
        return 0;

    // Expand tree item
    expandTree((TreeItem*)item,TRUE);
    listChildItems(item);

    // Now we know for sure whether we really have subitems or not
    if(!item->first)
        item->state&=~DirItem::HASITEMS;
    else
        item->state|=DirItem::HASITEMS;

    sortChildItems(item);
    return 1;
}


// Item closed
long DirList::onCollapsed(FXObject* sender,FXSelector sel,void* ptr)
{
    DirItem *item=(DirItem*)ptr;
    if(!(item->state&DirItem::FOLDER))
        return 0;

    // Collapse tree item
    collapseTree((TreeItem*)item,TRUE);

    return 1;
}



// Expand tree
FXbool DirList::expandTree(TreeItem* tree,FXbool notify)
{
    if(FXTreeList::expandTree(tree,notify))
    {
        if(isItemDirectory(tree))
        {
            FXTRACE((10,"expandTree: %s pathname=%s\n",tree->getText().text(),getItemPathname(tree).text()));
            listChildItems((DirItem*)tree);
            sortChildItems(tree);
        }
        return TRUE;
    }
    return FALSE;
}


// Collapse tree
FXbool DirList::collapseTree(TreeItem* tree,FXbool notify)
{
    if(FXTreeList::collapseTree(tree,notify))
    {
        if(isItemDirectory(tree))
        {
            FXTRACE((10,"collapseTree: %s path=%s\n",tree->getText().text(),getItemPathname(tree).text()));
            // As a memory saving feature, all knowledge below this item
            // is deleted; we'll just recreate it when its reexpanded!
            //removeItems(item->first,item->last);
            //recalc();
        }
        return TRUE;
    }
    return FALSE;
}


#if defined(linux)
// To periodically scan /proc/mounts and refresh the mtdevices list
long DirList::onMtdevicesRefresh(FXObject*,FXSelector,void*)
{
    struct mntent *mnt;

    FXStringDict* tmpdict = new FXStringDict();
    FILE *mtab=setmntent(MTAB_PATH,"r");
    if(mtab)
    {
        while((mnt=getmntent(mtab)))
        {
            tmpdict->insert(mnt->mnt_dir,"");
            if (mtdevices->find(mnt->mnt_dir))
                mtdevices->remove
                (mnt->mnt_dir);
            mtdevices->insert(mnt->mnt_dir,mnt->mnt_type);
        }
        endmntent(mtab);
    }

    // Remove mount points that don't exist anymore
    FXint s;
    const FXchar *key, *data;
    for (s = mtdevices->first(); s < mtdevices->size(); s = mtdevices->next(s))
    {
        key = mtdevices->key(s);
        data = mtdevices->data(s);
        if (!tmpdict->find(mtdevices->key(s)))
            mtdevices->remove
            (mtdevices->key(s));
    }
    delete tmpdict;

    // Reset timer again
    getApp()->addTimeout(this,ID_MTDEVICES_REFRESH,MTDEVICES_INTERVAL);
    return 0;
}


// To periodically scan /proc/mounts and detect up and down mounted devices
// NB : the refresh period is much longer than for onMtdevicesRefresh
long DirList::onUpdevicesRefresh(FXObject*,FXSelector,void*)
{
    struct mntent *mnt;
    struct stat statbuf;
    FXString mtstate;

    FXStringDict* tmpdict = new FXStringDict();
    FILE *mtab=setmntent(MTAB_PATH,"r");
    if(mtab)
    {
        while((mnt=getmntent(mtab)))
        {
            tmpdict->insert(mnt->mnt_dir,"");

            if (mtlstat(mnt->mnt_dir,&statbuf)==-1)
            {
                mtstate="down";
                if (mount_warn)
                    MessageBox::warning(this,MBOX_OK,_("Warning"),_("Mount point %s is not responding..."),mnt->mnt_dir);
            }
            else
                mtstate="up";

            if (updevices->find(mnt->mnt_dir))
                updevices->remove
                (mnt->mnt_dir);
            updevices->insert(mnt->mnt_dir,mtstate.text());

        }
        endmntent(mtab);
    }

    // Remove mount points that don't exist anymore
    FXint s;
    const FXchar *key, *data;
    for (s = updevices->first(); s < updevices->size(); s = updevices->next(s))
    {
        key = updevices->key(s);
        data = updevices->data(s);
        if (!tmpdict->find(updevices->key(s)))
            updevices->remove
            (updevices->key(s));

    }
    delete tmpdict;

    // Reset timer again
    getApp()->addTimeout(this,ID_UPDEVICES_REFRESH,UPDEVICES_INTERVAL);
    return 0;
}
#endif


// Refresh with timer
long DirList::onRefreshTimer(FXObject*,FXSelector,void*)
{
    if(flags&FLAG_UPDATE)
    {
        scan(FALSE);
        counter=(counter+1)%REFRESH_FREQUENCY;
    }

    // Reset timer again
    getApp()->addTimeout(this,ID_REFRESH_TIMER,REFRESH_INTERVAL);
    return 0;
}

// Force refresh
long DirList::onRefresh(FXObject*,FXSelector,void*)
{
    scan(TRUE);
    return 0;
}

// Scan items to see if listing is necessary
void DirList::scan(FXbool force)
{
    FXString pathname;
    struct stat info;
    DirItem *item;

    // Do root first time
    if(!firstitem || force)
    {
        listRootItems();
        sortRootItems();
    }

    // Check all items
    item=(DirItem*)firstitem;
    while(item)
    {
        // Is expanded directory?
        if(item->isDirectory() && item->isExpanded())
        {
            // Get the full path of the item
            pathname=getItemPathname((TreeItem*)item);

            // Stat this directory; should not fail as parent has been scanned already
			statout(pathname.text(),&info);

            // Get the mod date of the item
            FXTime newdate=(FXTime)FXMAX(info.st_mtime,info.st_ctime);

            // Forced, date was changed, or failed to get proper date and counter expired
            if(force || (item->date!=newdate) || (counter==0))
            {
                // And do the refresh
                listChildItems(item);
                sortChildItems(item);

                // Remember when we did this
                item->date=newdate;
            }

            // Go deeper
            if(item->first)
            {
                item=(DirItem*)item->first;
                continue;
            }
        }

        // Go up
        while(!item->next && item->parent)
        {
            item=(DirItem*)item->parent;
        }

        // Go to next
        item=(DirItem*)item->next;
    }
}



// List root directories
void DirList::listRootItems()
{
    DirItem   *item=(DirItem*)firstitem;
    FXIcon      *openicon;
    FXIcon      *closedicon;
    FileAssoc *fileassoc;

    // First time, make root node
    if(!item)
        item=list=(DirItem*)appendItem(NULL,PATHSEPSTRING,harddiskicon,harddiskicon,NULL,TRUE);

    // Root is a directory, has items under it, and is searchable
    item->state|=DirItem::FOLDER|DirItem::HASITEMS;
    item->state&=~(DirItem::CHARDEV|DirItem::BLOCKDEV|DirItem::FIFO|DirItem::SOCK|DirItem::SYMLINK|DirItem::EXECUTABLE);

    // Determine associations, icons and type
    fileassoc=NULL;
    openicon=harddiskicon;
    closedicon=harddiskicon;
    if(associations)
        fileassoc=associations->findDirBinding(PATHSEPSTRING);

    // If association is found, use it
    if(fileassoc)
    {
        if(fileassoc->miniicon)
            closedicon=fileassoc->miniicon;
        if(fileassoc->miniiconopen)
            openicon=fileassoc->miniiconopen;
    }

    // Update item information
    item->openIcon=openicon;
    item->closedIcon=closedicon;
    item->size=0L;
    item->assoc=fileassoc;
    item->date=0;

    // Create item
    if(id())
        item->create();

    // Need to layout
    recalc();
}


// List child items
void DirList::listChildItems(DirItem *par)
{
    DirItem     *oldlist,*newlist,**po,**pp,**pn,*item,*link;
    FXIcon        *openicon;
    FXIcon        *closedicon;
    FileAssoc   *fileassoc;
    DIR           *dirp;
    struct dirent *dp;
    struct stat    info;
    FXString       pathname;
    FXString       directory;
    FXString       name;
    FXint          islink;

    // Path to parent node
    directory=getItemPathname((TreeItem*)par);

    // Build new insert-order list
    oldlist=par->list;
    newlist=NULL;

    // Assemble lists
    po=&oldlist;
    pn=&newlist;

    // Get directory stream pointer
    dirp=opendir(directory.text());

    // Managed to open directory
    if(dirp)
    {
        // Process directory entries
#ifdef FOX_THREAD_SAFE
        struct fxdirent dirresult;
        while(!readdir_r(dirp,&dirresult,&dp) && dp)
        {
#else
        while((dp=readdir(dirp))!=NULL)
        {
#endif
            // Get name of entry
            name=dp->d_name;

            // A dot special file?
            if(name[0]=='.' && (name[1]==0 || (name[1]=='.' && name[2]==0)))
                continue;

            // Hidden file or directory normally not shown
            if(name[0]=='.' && !(options&DIRLIST_SHOWHIDDEN))
                continue;

            // Build full pathname of entry
            pathname=directory;
            if(!ISPATHSEP(pathname[pathname.length()-1]))
                pathname+=PATHSEPSTRING;
            pathname+=name;

            // Get file/link info
			if(lstatout(pathname.text(),&info)!=0)
                continue;

            // If its a link, get the info on file itself
            islink=S_ISLNK(info.st_mode);
			if(islink && statout(pathname.text(),&info)!=0)
                continue;

            // If it is not a directory, and not showing files and matching pattern skip it
            if(!S_ISDIR(info.st_mode) && !((options&DIRLIST_SHOWFILES) && FXFile::match(pattern,name,matchmode)))
                continue;

            // Find it, and take it out from the old list if found
            for(pp=po; (item=*pp)!=NULL; pp=&item->link)
            {
                if(compare(item->label,name)==0)
                {
                    *pp=item->link;
                    item->link=NULL;
                    po=pp;
                    goto fnd;
                }
            }

            // Not found; prepend before list
            item=(DirItem*)appendItem(par,name,open_folder,closed_folder,NULL,TRUE);

            // Next gets hung after this one
fnd:
            *pn=item;
            pn=&item->link;

            // Item flags
            if(info.st_mode&(S_IXUSR|S_IXGRP|S_IXOTH))
            {
                item->state|=DirItem::EXECUTABLE;
            }
            else
            {
                item->state&=~DirItem::EXECUTABLE;
            }
            if(S_ISDIR(info.st_mode))
            {
                item->state|=DirItem::FOLDER;
                item->state&=~DirItem::EXECUTABLE;
            }
            else
            {
                item->state&=~(DirItem::FOLDER|DirItem::HASITEMS);
            }
            if(S_ISCHR(info.st_mode))
            {
                item->state|=DirItem::CHARDEV;
                item->state&=~DirItem::EXECUTABLE;
            }
            else
            {
                item->state&=~DirItem::CHARDEV;
            }
            if(S_ISBLK(info.st_mode))
            {
                item->state|=DirItem::BLOCKDEV;
                item->state&=~DirItem::EXECUTABLE;
            }
            else
            {
                item->state&=~DirItem::BLOCKDEV;
            }
            if(S_ISFIFO(info.st_mode))
            {
                item->state|=DirItem::FIFO;
                item->state&=~DirItem::EXECUTABLE;
            }
            else
            {
                item->state&=~DirItem::FIFO;
            }
            if(S_ISSOCK(info.st_mode))
            {
                item->state|=DirItem::SOCK;
                item->state&=~DirItem::EXECUTABLE;
            }
            else
            {
                item->state&=~DirItem::SOCK;
            }
            if(islink)
            {
                item->state|=DirItem::SYMLINK;
            }
            else
            {
                item->state&=~DirItem::SYMLINK;
            }

            // We can drag items
            item->state|=DirItem::DRAGGABLE;

            // Assume no associations
            fileassoc=NULL;

            // Determine icons and type
            if(item->state&DirItem::FOLDER)
            {
                if(!::isReadExecutable(pathname))
                {
                    openicon=locked_folder;
                    closedicon=locked_folder;
                }
                else
                {
                    openicon=open_folder;
                    closedicon=closed_folder;
                }
                if(associations)
                    fileassoc=associations->findDirBinding(pathname.text());
            }
            else if(item->state&DirItem::EXECUTABLE)
            {
                openicon=mini_app;
                closedicon=mini_app;
                if(associations)
                    fileassoc=associations->findExecBinding(pathname.text());
            }
            else
            {
                openicon=mini_doc;
                closedicon=mini_doc;
                if(associations)
                    fileassoc=associations->findFileBinding(pathname.text());
            }

            // If association is found, use it
            if(fileassoc)
            {
                if(fileassoc->miniicon)
                    closedicon=fileassoc->miniicon;
                if(fileassoc->miniiconopen)
                    openicon=fileassoc->miniiconopen;
            }

            // Update item information
            item->openIcon=openicon;
            item->closedIcon=closedicon;
            item->size=(unsigned long)info.st_size;
            item->assoc=fileassoc;
            item->date=info.st_mtime;
            //item->date=filetime;

#if defined(linux)
            // Devices have a specific icon
            if(fsdevices->find(pathname.text()))
            {
                if(::streq(fsdevices->find(pathname.text()),"harddisk"))
                {
                    item->closedIcon=harddiskicon;
                    item->openIcon=harddiskicon;
                }
                else if(::streq(fsdevices->find(pathname.text()),"nfsdisk"))
                {
                    item->closedIcon=nfsicon;
                    item->openIcon=nfsicon;
                }
                else if(::streq(fsdevices->find(pathname.text()),"smbdisk"))
                {
                    item->closedIcon=nfsicon;
                    item->openIcon=nfsicon;
                }
                else if(::streq(fsdevices->find(pathname.text()),"floppy"))
                {
                    item->closedIcon=floppyicon;
                    item->openIcon=floppyicon;
                }
                else if(::streq(fsdevices->find(pathname.text()),"cdrom"))
                {
                    item->closedIcon=cdromicon;
                    item->openIcon=cdromicon;
                }
                else if(::streq(fsdevices->find(pathname.text()),"zip"))
                {
                    item->closedIcon=zipicon;
                    item->openIcon=zipicon;
                }
            }
#endif

            // Symbolic links have a specific icon
            if(islink)
            {
                item->closedIcon=linkicon;
                item->openIcon=linkicon;
            }

            // Create item
            if(id())
                item->create();
        }

        // Close it
        closedir(dirp);
    }

    // Wipe items remaining in list:- they have disappeared!!
    for(item=oldlist; item; item=link)
    {
        link=item->link;
        removeItem(item,TRUE);
    }

    // Now we know for sure whether we really have subitems or not
    if(par->first)
        par->state|=DirItem::HASITEMS;
    else
        par->state&=~DirItem::HASITEMS;

    // Remember new list
    par->list=newlist;

    // Need to layout
    recalc();
}



// Is directory
FXbool DirList::isItemDirectory(const TreeItem* item) const
{
    if(item==NULL)
        fxerror("%s::isItemDirectory: item is NULL.\n",getClassName());
    return (item->state&DirItem::FOLDER)!=0;
}


// Is file
FXbool DirList::isItemFile(const TreeItem* item) const
{
    if(item==NULL)
        fxerror("%s::isItemFile: item is NULL.\n",getClassName());
    return (item->state&(DirItem::FOLDER|DirItem::CHARDEV|DirItem::BLOCKDEV|DirItem::FIFO|DirItem::SOCK))==0;
}


// Is executable
FXbool DirList::isItemExecutable(const TreeItem* item) const
{
    if(item==NULL)
        fxerror("%s::isItemExecutable: item is NULL.\n",getClassName());
    return (item->state&DirItem::EXECUTABLE)!=0;
}


// Return absolute pathname of item
FXString DirList::getItemPathname(const TreeItem* item) const
{
    FXString pathname;
    if(item)
    {
        while(1)
        {
            pathname.prepend(item->getText());
            item=(TreeItem*)item->parent;
            if(!item)
                break;
            if(item->parent)
                pathname.prepend(PATHSEP);
        }
    }
    return pathname;
}


// Return the item from the absolute pathname
TreeItem* DirList::getPathnameItem(const FXString& path)
{
    register TreeItem *item,*it;
    register FXint beg=0,end=0;
    FXString name;
    if(!path.empty())
    {
        if(ISPATHSEP(path[0]))
            end++;
        if(beg<end)
        {
            name=path.mid(beg,end-beg);
            for(it=(TreeItem*)firstitem; it; it=(TreeItem*)it->next)
            {
                if(compare(name,it->getText())==0)
                    goto x;
            }
            listRootItems();
            sortRootItems();
            for(it=(TreeItem*)firstitem; it; it=(TreeItem*)it->next)
            {
                if(compare(name,it->getText())==0)
                    goto x;
            }
            return NULL;
x:
            item=it;
            FXASSERT(item);
            while(end<path.length())
            {
                beg=end;
                while(end<path.length() && !ISPATHSEP(path[end]))
                    end++;
                name=path.mid(beg,end-beg);
                for(it=(TreeItem*)item->first; it; it=(TreeItem*)it->next)
                {
                    if(compare(name,it->getText())==0)
                        goto y;
                }
                listChildItems((DirItem*)item);
                sortChildItems(item);
                for(it=(TreeItem*)item->first; it; it=(TreeItem*)it->next)
                {
                    if(compare(name,it->getText())==0)
                        goto y;
                }
                return item;
y:
                item=it;
                FXASSERT(item);
                if(end<path.length() && ISPATHSEP(path[end]))
                    end++;
            }
            FXASSERT(item);
            return item;
        }
    }
    return NULL;
}


// Obtain item's file name only
FXString DirList::getItemFilename(const TreeItem* item) const
{
    if(item==NULL)
        fxerror("%s::getItemFilename: item is NULL.\n",getClassName());
    return item->label;
}


// Open all intermediate directories down toward given one
void DirList::setDirectory(const FXString& pathname,FXbool notify)
{
    FXTRACE((100,"%s::setDirectory(%s)\n",getClassName(),pathname.text()));
    if(!pathname.empty())
    {
        FXString path=FXFile::absolute(getItemPathname((TreeItem*)currentitem),pathname);
        while(!FXFile::isTopDirectory(path) && !FXFile::isDirectory(path))
        {
            path=FXFile::upLevel(path);
        }
        TreeItem *item=getPathnameItem(path);
        if(id())
            layout();
        makeItemVisible(item);
        setCurrentItem(item,notify);
    }
}


// Return directory part of path to current item
FXString DirList::getDirectory() const
{
    const TreeItem* item=(TreeItem*)currentitem;
    while(item)
    {
        if(item->state&DirItem::FOLDER)
            return getItemPathname(item);
        item=(TreeItem*)item->parent;
    }
    return "";
}


// Set current (dir/file) name path
void DirList::setCurrentFile(const FXString& pathname,FXbool notify)
{
    FXTRACE((100,"%s::setCurrentFile(%s)\n",getClassName(),pathname.text()));
    if(!pathname.empty())
    {
        FXString path=FXFile::absolute(getItemPathname((TreeItem*)currentitem),pathname);
        while(!FXFile::isTopDirectory(path) && !::exists(path))
        {
            path=FXFile::upLevel(path);
        }
        TreeItem *item=getPathnameItem(path);
        if(id())
            layout();
        makeItemVisible(item);
        setCurrentItem(item,notify);
    }
}


// Get current (dir/file) name path
FXString DirList::getCurrentFile() const
{
    return getItemPathname((TreeItem*)currentitem);
}



// Get list style
FXbool DirList::showFiles() const
{
    return (options&DIRLIST_SHOWFILES)!=0;
}


// Change list style
void DirList::showFiles(FXbool showing)
{
    FXuint opts=options;
    if(showing)
        opts|=DIRLIST_SHOWFILES;
    else
        opts&=~DIRLIST_SHOWFILES;
    if(options!=opts)
    {
        options=opts;
        scan(TRUE);
    }
}


// Return TRUE if showing hidden files
FXbool DirList::showHiddenFiles() const
{
    return (options&DIRLIST_SHOWHIDDEN)!=0;
}


// Change show hidden files mode
void DirList::showHiddenFiles(FXbool showing)
{
    FXuint opts=options;
    if(showing)
        opts|=DIRLIST_SHOWHIDDEN;
    else
        opts&=~DIRLIST_SHOWHIDDEN;
    if(opts!=options)
    {
        options=opts;
        scan(TRUE);
    }
}


// Set associations
void DirList::setAssociations(FileDict* assoc)
{
    associations=assoc;
    scan(TRUE);
}


// Set the pattern to filter
void DirList::setPattern(const FXString& ptrn)
{
    if(ptrn.empty())
        return;
    if(pattern!=ptrn)
    {
        pattern=ptrn;
        scan(TRUE);
    }
}


// Change file match mode
void DirList::setMatchMode(FXuint mode)
{
    if(matchmode!=mode)
    {
        matchmode=mode;
        scan(TRUE);
    }
}


// Cleanup
DirList::~DirList()
{
    clearItems();
    getApp()->removeTimeout(this,ID_REFRESH_TIMER);
    getApp()->removeTimeout(this,ID_EXPAND_TIMER);
#if defined(linux)
    getApp()->removeTimeout(this,ID_MTDEVICES_REFRESH);
    getApp()->removeTimeout(this,ID_UPDEVICES_REFRESH);
#endif
    if(!(options&DIRLIST_NO_OWN_ASSOC))
        delete associations;
    delete closed_folder;
    delete open_folder;
    delete mini_doc;
    delete mini_app;
    delete harddiskicon;
    delete nfsicon;
    delete floppyicon;
    delete cdromicon;
    delete zipicon;
    delete linkicon;
    delete locked_folder;
    associations=(FileDict*)-1;
    closed_folder=(FXPNGIcon*)-1;
    open_folder=(FXPNGIcon*)-1;
    mini_doc=(FXPNGIcon*)-1;
    mini_app=(FXPNGIcon*)-1;
    harddiskicon=(FXPNGIcon*)-1;
    nfsicon=(FXPNGIcon*)-1;
    linkicon=(FXPNGIcon*)-1;
    locked_folder=(FXPNGIcon*)-1;
}


