/* $Id: xfce-menu-item.c,v 1.5 2005/01/07 17:32:44 bmeurer Exp $ */
/*-
 * Copyright (c) 2004 os-cillation
 * All rights reserved.
 *
 * Written by Benedikt Meurer <bm@os-cillation.de>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <libxfcegui4/libxfcegui4.h>

#include "xfce-menu-item.h"



static void       xfce_menu_item_class_init   (XfceMenuItemClass *klass);
static void       xfce_menu_item_init         (XfceMenuItem      *item);
static void       xfce_menu_item_finalize     (GObject           *object);
static void       xfce_menu_item_mapped       (XfceMenuItem      *item);
static gboolean   xfce_menu_item_idle_load    (XfceMenuItem      *item);
static void       xfce_menu_item_show_icons   (GtkSettings       *settings,
                                               GParamSpec        *pspec,
                                               XfceMenuItem      *item);

#if GTK_CHECK_VERSION(2,4,0)
static void       xfce_menu_item_icon_theme_changed (GtkIconTheme *icon_theme,
                                                     XfceMenuItem *item);
#endif



static GObjectClass *parent_class;



GType
xfce_menu_item_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      static const GTypeInfo type_info = {
        sizeof (XfceMenuItemClass),
        NULL,
        NULL,
        (GClassInitFunc) xfce_menu_item_class_init,
        NULL,
        NULL,
        sizeof (XfceMenuItem),
        0,
        (GInstanceInitFunc) xfce_menu_item_init,
      };

      type = g_type_register_static (GTK_TYPE_IMAGE_MENU_ITEM,
                                     "XfceMenuItem",
                                     &type_info,
                                     0);
    }
  
  return type;
}



GtkWidget*
xfce_menu_item_new (const gchar *text,
                    const gchar *icon,
                    const gchar *tip)
{
  XfceMenuItem *item;

  item = g_object_new (XFCE_TYPE_MENU_ITEM, NULL);
  xfce_menu_item_set_text (item, text);
  xfce_menu_item_set_icon (item, icon);
  xfce_menu_item_set_tip  (item, tip);

  return GTK_WIDGET (item);
}



const gchar*
xfce_menu_item_get_text (const XfceMenuItem *item)
{
  return gtk_label_get_text (GTK_LABEL (item->label));
}



void
xfce_menu_item_set_text (XfceMenuItem *item,
                         const gchar  *text)
{
  gtk_label_set_text (GTK_LABEL (item->label), text);
}



void
xfce_menu_item_set_icon (XfceMenuItem *item,
                         const gchar  *icon)
{
  if (G_LIKELY (item->icon_name != NULL))
    g_free (item->icon_name);

  if (G_LIKELY (icon != NULL))
    item->icon_name = g_strdup (icon);
  else
    item->icon_name = NULL;

  item->icon_valid = FALSE;

  if (GTK_WIDGET_MAPPED (item))
    xfce_menu_item_mapped (item);
}



void
xfce_menu_item_set_tip (XfceMenuItem *item,
                        const gchar  *tip)
{
  XfceMenuItemClass *klass;

  klass = XFCE_MENU_ITEM_GET_CLASS (item);
  gtk_tooltips_set_tip (klass->tips, GTK_WIDGET (item), tip, NULL);
}



static void
xfce_menu_item_class_init (XfceMenuItemClass *klass)
{
  GObjectClass *gobject_class;

  klass->default_icon = gdk_pixbuf_new (GDK_PIXBUF_ALPHA_BILEVEL,
                                        TRUE, 8, 16, 16);
  gdk_pixbuf_fill (klass->default_icon, 0);

  klass->tips = gtk_tooltips_new ();

  parent_class = g_type_class_peek_parent (klass);

  gobject_class = G_OBJECT_CLASS (klass);
  gobject_class->finalize = xfce_menu_item_finalize;
}



static void
xfce_menu_item_init (XfceMenuItem *item)
{
  XfceMenuItemClass *klass;
  GtkSettings       *settings;
  gboolean           show_icons;

  settings = gtk_settings_get_default ();
  g_object_get (G_OBJECT (settings),
                "xfdesktop-menu-show-icons", &show_icons,
                NULL);
  g_signal_connect (G_OBJECT (settings), "notify::xfdesktop-menu-show-icons",
                    G_CALLBACK (xfce_menu_item_show_icons), item);

#if GTK_CHECK_VERSION(2,4,0)
  g_signal_connect (G_OBJECT (gtk_icon_theme_get_default ()), "changed",
                    G_CALLBACK (xfce_menu_item_icon_theme_changed), item);
#endif

  item->label = g_object_new (GTK_TYPE_LABEL, NULL);
  gtk_misc_set_alignment (GTK_MISC (item->label), 0.0, 0.5);
  gtk_container_add (GTK_CONTAINER (item), item->label);
  gtk_widget_show (item->label);

  item->image = g_object_new (GTK_TYPE_IMAGE, NULL);
  g_object_ref (G_OBJECT (item->image));
  gtk_object_sink (GTK_OBJECT (item->image));
  if (show_icons)
    gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), item->image);
  gtk_widget_show (item->image);

  klass = XFCE_MENU_ITEM_GET_CLASS (item);
  gtk_image_set_from_pixbuf (GTK_IMAGE (item->image), klass->default_icon);

  g_signal_connect (G_OBJECT (item), "map",
                    G_CALLBACK (xfce_menu_item_mapped), NULL);
}



static void
xfce_menu_item_finalize (GObject *object)
{
  XfceMenuItem *item;
  GtkSettings  *settings;

  item = XFCE_MENU_ITEM (object);
  if (G_UNLIKELY (item->idle_id != 0))
    g_source_remove (item->idle_id);
  if (G_LIKELY (item->icon_name != NULL))
    g_free (item->icon_name);

  settings = gtk_settings_get_default ();
  g_signal_handlers_disconnect_by_func (settings, xfce_menu_item_show_icons, item);

   G_OBJECT_CLASS (parent_class)->finalize (object);
}



static void
xfce_menu_item_mapped (XfceMenuItem *item)
{
  GtkSettings *settings;
  gboolean     show_icons;

  settings = gtk_settings_get_default ();
  g_object_get (G_OBJECT (settings),
                "xfdesktop-menu-show-icons", &show_icons,
                NULL);

  if (G_LIKELY (show_icons && !item->icon_valid && item->idle_id == 0))
    {
      item->idle_id = g_idle_add_full (G_PRIORITY_LOW,
                                       (GSourceFunc) xfce_menu_item_idle_load,
                                       item,
                                       NULL);
    }
}



static gboolean
xfce_menu_item_idle_load (XfceMenuItem *item)
{
  XfceMenuItemClass *klass;
  GdkPixbuf         *pixbuf;

  if (G_LIKELY (item->icon_name != NULL))
    pixbuf = xfce_themed_icon_load (item->icon_name, 16);
  else
    pixbuf = NULL;

  if (pixbuf == NULL)
    {
      klass = XFCE_MENU_ITEM_GET_CLASS (item);
      pixbuf = GDK_PIXBUF (g_object_ref (G_OBJECT (klass->default_icon)));
    }

  gtk_image_set_from_pixbuf (GTK_IMAGE (item->image), pixbuf);
  g_object_unref (pixbuf);

  item->icon_valid = TRUE;
  item->idle_id = 0;

  return FALSE;
}



static void
xfce_menu_item_show_icons (GtkSettings       *settings,
                           GParamSpec        *pspec,
                           XfceMenuItem      *item)
{
  gboolean show_icons;

  settings = gtk_settings_get_default ();
  g_object_get (G_OBJECT (settings),
                "xfdesktop-menu-show-icons", &show_icons,
                NULL);

  if (show_icons)
    {
      if (G_LIKELY (item->idle_id == 0))
        {
          item->idle_id = g_idle_add_full (G_PRIORITY_LOW,
                                           (GSourceFunc) xfce_menu_item_idle_load,
                                           item,
                                           NULL);
        }

      gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), item->image);
    }
  else
    {
      gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), NULL);
    }
}



#if GTK_CHECK_VERSION(2,4,0)
static void
xfce_menu_item_icon_theme_changed (GtkIconTheme *icon_theme,
                                   XfceMenuItem *item)
{
  if (item->idle_id == 0 && item->icon_valid)
    {
      item->icon_valid = FALSE;
      item->idle_id = g_idle_add_full (G_PRIORITY_LOW,
                                       (GSourceFunc) xfce_menu_item_idle_load,
                                       item,
                                       NULL);
    }
}
#endif



