/*
 *	Xenophilia GTK+ Theme Engine
 *	
 *	xeno_style_fill.c:
 *		Background drawing routines of XenoStyle.
 *
 *	Copyright  1999-2002 Johan Hanson <misagon@bahnhof.se>
 *	
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Library General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *	
 *	This library is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *	Library General Public License for more details.
 *	
 *	You should have received a copy of the GNU Library General Public
 *	License along with this library; if not, write to the 
 *	Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
 *	Boston, MA  02111-1307  USA.
 */

#include "xeno_rc_style.h"
#include "xeno_style.h"
#include "xeno_style_fill.h"
#include "xeno_color.h"


/*
 *	Types
 */

#if !XENO_CONFIG_DIRTY_GRADIENTS
typedef struct _XenoGradientApp	XenoGradientApp;
struct _XenoGradientApp {
	XenoGradientApp	**ptr, *next;
	XenoGradient	*gradient;
	GtkWidget		*widget;
}; /* 16 or 32 bytes */
#endif

struct _XenoGradient {
	XenoGradient	**ptr, *next;
	GtkStyle		*style;
	
	GdkPixmap		*pixmap[5];
  #if XENO_CONFIG_DIRTY_GRADIENTS
	gint			ref_count;
  #else
	XenoGradientApp	*apps;
  #endif
	guint16			width, height;
};

typedef struct _XenoGradientContext XenoGradientContext;
typedef void (* XenoGradientPixelFunc)(XenoGradientContext *, gfloat, gint, gint);

struct _XenoGradientContext {
	XenoGradientPixelFunc put_pixel;
	GdkImage	*image;
	GdkVisual	*visual;
	GdkColormap	*colormap;
	
	gfloat		alpha;
	gfloat		red_factor;
	gfloat		green_factor;
	gfloat		blue_factor;
	
	XenoColor	top;
	XenoColor	bot;
	XenoColor	bg;
	
	guint		top_pixel;
	guint		bot_pixel;
	guint		bg_pixel;
	
	gboolean	blend;
};


/*
 *	Globals
 */

#if !XENO_CONFIG_DIRTY_GRADIENTS
static GMemChunk *		xeno_gradient_app_chunk	= NULL;
#endif
static GMemChunk *		xeno_gradient_chunk		= NULL;
static GQuark			xeno_gradient_quark		= 0;
static gint				xeno_gradient_ref_count	= 0;


/*
 *	Functions
 */

void
_xeno_style_fill_rectangle	(GtkStyle		*style,
							 GdkDrawable	*drawable,
							 GdkGC			*bg_gc,
							 GdkPixmap		*bg_pixmap,
							 GtkStateType	state_type,
							 GdkRectangle	*area,
							 gint			tile_x,
							 gint			tile_y,
							 gint			window_width,
							 gint			window_height)
{
	GdkRectangle	rect;
	XenoRcData		*rc_data;
	XenoOriginType	origin_type;
	gint			pixmap_width, pixmap_height;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (drawable != NULL);
	g_return_if_fail (area != NULL);
	g_return_if_fail (bg_gc != NULL);
	
	rc_data = XENO_STYLE_RC_DATA(style);
	origin_type = XENO_ORIGIN_DEFAULT;
	if (rc_data && origin_type == XENO_ORIGIN_UNKNOWN)
		origin_type = rc_data->origin[state_type].type;
	
	if (area == NULL) {
		rect.x = rect.y = 0;
		rect.width = window_width;
		rect.height = window_height;
		area = &rect;
	}
	
	if (bg_pixmap != NULL && bg_pixmap != (GdkPixmap *)GDK_PARENT_RELATIVE) {
		if (origin_type == XENO_ORIGIN_ALIGNMENT) {
			XenoRcOrigin *rc_origin;
			rc_origin = &rc_data->origin[state_type];
			gdk_window_get_size (bg_pixmap, &pixmap_width, &pixmap_height);
			tile_x += rc_origin->x * (window_width  % pixmap_width  + (((window_width  / pixmap_width ) & 1) ? 0 : pixmap_width));
			tile_y += rc_origin->y * (window_height % pixmap_height + (((window_height / pixmap_height) & 1) ? 0 : pixmap_height));
		}
		gdk_gc_set_fill (bg_gc, GDK_TILED);
		gdk_gc_set_tile (bg_gc, bg_pixmap);
		gdk_gc_set_ts_origin (bg_gc, tile_x, tile_y);
	}
	gdk_draw_rectangle (drawable, bg_gc, TRUE, area->x, area->y, area->width, area->height);
	if (bg_pixmap)
		gdk_gc_set_fill (bg_gc, GDK_SOLID);
}


/* XenoStyle::draw_background */
void
xeno_style_fill_background	(GtkStyle		*style,
							 GdkDrawable	*drawable,
							 GtkStateType	state_type,
							 GdkRectangle	*area,
							 GtkWidget		*widget,
							 gint			x,
							 gint			y,
							 gint			width,
							 gint			height)
#if XENO_GTK2
{
	g_return_if_fail (XENO_IS_STYLE(style));
	
	XENO_STYLE_GET_CLASS(style)->fill_background (style, drawable, state_type, area,
												  widget, x, y, width, height);
}

void
xeno_style_real_fill_background	(GtkStyle		*style,
								 GdkDrawable	*drawable,
								 GtkStateType	state_type,
								 GdkRectangle	*area,
								 GtkWidget		*widget,
								 gint			x,
								 gint			y,
								 gint			width,
								 gint			height)
#endif
{
	GdkRectangle	rect;
	GdkPixmap		*bg_pixmap;
	XenoRcData		*rc_data;
	XenoOriginType	origin_type;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (drawable != NULL);
	/*
	if (width < 0 || height < 0) {
		gint window_width, window_height;
		gdk_window_get_size (drawable, &window_width, &window_height);
		if (width < 0) width = window_width - x;
		if (height < 0) height = window_height - y;
	}
	*/
	rect.x		= x;
	rect.y		= y;
	rect.width	= width;
	rect.height	= height;
	if (area) {
		GdkRectangle rect2;
		if (!gdk_rectangle_intersect (&rect, area, &rect2))
			return;
		
		rect = rect2;
	}
	
	bg_pixmap	= style->bg_pixmap[state_type];
	origin_type	= XENO_ORIGIN_DEFAULT;
	rc_data		= XENO_STYLE_RC_DATA(style);
	if (rc_data) {
		origin_type = rc_data->origin[state_type].type;
		if (rc_data->gradient[state_type].type != XENO_GRADIENT_NONE && widget) {
			bg_pixmap = xeno_gradient_get (style, drawable, state_type, widget, width, height);
			if (origin_type == XENO_ORIGIN_DEFAULT && (x | y) != 0)
				origin_type = XENO_ORIGIN_UNKNOWN;
		}
	}

	if (   widget && !GTK_WIDGET_NO_WINDOW(widget)
	  #if XENO_GTK2
		&& GDK_IS_WINDOW(drawable) && !GDK_WINDOW_OBJECT(drawable)->input_only
	  #else
		&& gdk_window_get_type (drawable) != GDK_WINDOW_PIXMAP
	  #endif
		&& (bg_pixmap == NULL || origin_type == XENO_ORIGIN_DEFAULT)
		&& !GTK_IS_TEAROFF_MENU_ITEM(widget))
	{
		if (bg_pixmap) {
			if (bg_pixmap == (GdkPixmap *)GDK_PARENT_RELATIVE)
				bg_pixmap = NULL;
			
			gdk_window_set_back_pixmap (drawable, bg_pixmap, bg_pixmap == NULL);
		} else {
			gdk_window_set_background (drawable, &style->bg[state_type]);
		}
		gdk_window_clear_area (drawable, rect.x, rect.y, rect.width, rect.height);
	}
  #if !XENO_GTK2
	else
  #endif
	{
		_xeno_style_fill_rectangle (style, drawable, style->bg_gc[state_type], bg_pixmap, state_type,
								   &rect, x, y, width, height);
	}
}

void
xeno_style_fill_base	(GtkStyle		*style,
						 GdkWindow		*window,
						 GtkStateType	state_type,
						 GdkRectangle	*area,
						 GtkWidget		*widget,
						 gint			x,
						 gint			y,
						 gint			width,
						 gint			height)
#if XENO_GTK2
{
	g_return_if_fail (XENO_IS_STYLE(style));
	
	XENO_STYLE_GET_CLASS(style)->fill_base (style, window, state_type, area, widget,
											x, y, width, height);
}

void
xeno_style_real_fill_base	(GtkStyle		*style,
							 GdkWindow		*window,
							 GtkStateType	state_type,
							 GdkRectangle	*area,
							 GtkWidget		*widget,
							 gint			x,
							 gint			y,
							 gint			width,
							 gint			height)
#endif
{
	GdkGC			*gc;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	if ((width | height) < 0)
		gdk_window_get_size (window, (width < 0) ? &width : NULL, (height < 0) ? &height : NULL);
	
	gc = style->base_gc[state_type];
	
	if (area)
		gdk_gc_set_clip_rectangle (gc, area);
	
	gdk_gc_set_fill(gc, GDK_SOLID);
	gdk_draw_rectangle(window, gc, TRUE, x, y, width, height);
	
	if (area)
		gdk_gc_set_clip_rectangle (gc, NULL);
}


/*
 *	XenoGradient
 */

/* color functions */
static void
xeno_gradient_put_pseudo_color (XenoGradientContext *ctx, gfloat f, gint x, gint y)
{
	guint	pixel;
	
	pixel = ctx->bot_pixel;
	f = f - 1.0;
	if (f < 0.0) {
		pixel = ctx->top_pixel;
		f = -f;
	}
	if (f <= xeno_dither(x, y)) {
		if (ctx->blend)
			return;
		
		pixel = ctx->bg_pixel;
	}
	gdk_image_put_pixel (ctx->image, x, y, pixel);
}

static void
xeno_gradient_put_true_color (XenoGradientContext *ctx, gfloat f, gint x, gint y)
{
	XenoColor	tmp, *ptr;
	GdkColor	color;
	
	ptr = &ctx->bot;
	f = f - 1.0;
	if (f < 0.0) {
		ptr = &ctx->top;
		f = -f;
	}
	xeno_color_blend (&ctx->bg, ptr, f * ctx->alpha, &tmp);
	xeno_color_dither (&tmp, xeno_theme_visual, x, y, &color);
	gdk_colormap_alloc_color (xeno_theme_colormap, &color, FALSE, TRUE);
	gdk_image_put_pixel (ctx->image, x, y, color.pixel);
}

static void
xeno_gradient_blend_true_color (XenoGradientContext *ctx, gfloat f, gint x, gint y)
{
	guint32 pixel;
	
	pixel = gdk_image_get_pixel (ctx->image, x, y);
	xeno_color_init (&ctx->bg,
					 (pixel & ctx->visual->red_mask)   * ctx->red_factor,
					 (pixel & ctx->visual->green_mask) * ctx->green_factor,
					 (pixel & ctx->visual->blue_mask)  * ctx->blue_factor);
	xeno_gradient_put_true_color (ctx, f, x, y);
}

static void
xeno_gradient_context_init (XenoGradientContext *ctx, const GtkStyle *style, GtkStateType state_type,
							GdkVisual *visual, GdkColormap *colormap, GdkImage *image,
							gboolean blend, gboolean realize)
{
	GdkColor			 color;
	XenoColor			 tmp;
	const XenoColor		 *top_color;
	const XenoStyleData	 *style_data;
	const XenoRcData	 *rc_data;
	const XenoRcGradient *rc_gradient;
	gfloat alpha;
	
	style_data	= XENO_STYLE_DATA(style);
	rc_data		= XENO_STYLE_RC_DATA(style);
	rc_gradient	= &rc_data->gradient[state_type];
	
	ctx->visual	= visual;
	ctx->colormap = colormap;
	ctx->image	= image;
	ctx->blend	= blend;
	
	top_color = style_data->white;
	if (rc_gradient->type == XENO_GRADIENT_NONE) {
		alpha = 0.0;
	} else {
		alpha = rc_gradient->factor;
		if (alpha < 1.0) {
			alpha = 1.0 - MAX(alpha, rc_data->shade[state_type]);
			top_color = style_data->black;
		} else {
			alpha = MIN(alpha, rc_data->shine[state_type]) - 1.0;
		}
	}
	ctx->alpha = alpha;
	top_color = &top_color[state_type];
	ctx->top = *top_color;
	
	xeno_color_from_gdk (&ctx->bg, &style->bg[state_type]);
	xeno_color_flip (&ctx->bg, top_color, &ctx->bg, &ctx->bot);
	
	if (xeno_theme_pseudocolor) {
		ctx->put_pixel = xeno_gradient_put_pseudo_color;
		if (realize) {
			ctx->bg_pixel = style->bg[state_type].pixel;
			
			xeno_color_blend (&ctx->bg, &ctx->top, alpha, &tmp);
			xeno_color_to_gdk (&tmp, &color);
			gdk_colormap_alloc_color (colormap, &color, FALSE, TRUE);
			ctx->top_pixel = color.pixel;
			
			xeno_color_blend (&ctx->bg, &ctx->bot, alpha, &tmp);
			xeno_color_to_gdk (&tmp, &color);
			gdk_colormap_alloc_color (colormap, &color, FALSE, TRUE);
			ctx->bot_pixel = color.pixel;
		}
	} else {
		ctx->put_pixel = xeno_gradient_put_true_color;
		if (visual->type == GDK_VISUAL_TRUE_COLOR && blend) {
			ctx->put_pixel    = xeno_gradient_blend_true_color;
			
			ctx->red_factor   = 1.0 / visual->red_mask;
			ctx->green_factor = 1.0 / visual->green_mask;
			ctx->blue_factor  = 1.0 / visual->blue_mask;
		}
	}
}

void
xeno_gradient_color (const GtkStyle *style, GtkStateType state_type, gfloat f, XenoColor *dst)
{
	XenoGradientContext ctx;
	XenoColor	*ptr;
	
	if (   XENO_STYLE_RC_DATA(style)->gradient[state_type].type == XENO_GRADIENT_NONE
		|| xeno_theme_pseudocolor)
	{
		xeno_color_from_gdk (dst, &style->bg[state_type]);
	} else {
		xeno_gradient_context_init (&ctx, style, state_type, xeno_theme_visual, style->colormap,
									NULL, FALSE, FALSE);
		ptr = &ctx.bot;
		f -= 1.0;
		if (f < 0.0) {
			ptr = &ctx.top;
			f = -f;
		}
		xeno_color_blend (&ctx.bg, ptr, f * ctx.alpha, dst);
	}
}

/* realize */
static GdkPixmap *
xeno_gradient_realize  (GtkStyle		 *style,
						GdkWindow		 *window,
						GtkStateType 	 state_type,
						XenoGradient	 *gradient)
{
	GdkVisual			*visual;
	GdkPixmap			*pixmap;
	GdkPixmap			*bg_pixmap;
	GdkImage			*image;
	
	XenoGradientContext	ctx;
	XenoRcGradient		*rc_gradient;
	XenoGradientType	gradient_type;
	XenoGradientSet		*gradient_set;
	
	gint	x, y, width, height, i;
	gint	min_width, min_height;
	gint	pixmap_width, pixmap_height;
	gfloat	f, g, t;
	
	g_return_val_if_fail (style != NULL, NULL);
	g_return_val_if_fail (window != NULL, NULL);
	g_return_val_if_fail (gradient != NULL, NULL);
	
	min_width = 8;
	if (xeno_theme_pseudocolor)
		min_width = 16;
	min_height = min_width;
	bg_pixmap = style->bg_pixmap[state_type];
	if (bg_pixmap) {
		if (bg_pixmap == (GdkPixmap *)GDK_PARENT_RELATIVE) {
			bg_pixmap = NULL;
		} else {
			gdk_window_get_size (bg_pixmap, &pixmap_width, &pixmap_height);
			min_width  = MAX(min_width, pixmap_width);
			min_height = MAX(min_height, pixmap_height);
		}
	}
	
	rc_gradient   = &XENO_STYLE_RC_DATA(style)->gradient[state_type];
	gradient_type = rc_gradient->type;
	switch (gradient_type) {
	  case XENO_GRADIENT_VERTICAL:
		width  = min_width;
		height = gradient->height;
		break;
		
	  case XENO_GRADIENT_HORIZONTAL:
		width  = gradient->width;
		height = min_height;
		if (rc_gradient->interlaced && height == 1)
			height = 2;
		break;
		
	  case XENO_GRADIENT_DIAGONAL:
		width  = gradient->width;
		height = gradient->height;
		break;
		
	  case XENO_GRADIENT_NONE:
	  default:
		return bg_pixmap;
	}
	
	/* Create pixmap and draw */
	visual = gdk_window_get_visual (window);
	if (!bg_pixmap) {
		if ((image = gdk_image_new (GDK_IMAGE_NORMAL, visual, width, height)) == NULL)
			return NULL;
	}
	if ((pixmap = gdk_pixmap_new (window, width, height, visual->depth)) == NULL)
		return bg_pixmap;
	
	if (bg_pixmap) {
		GdkRectangle area;
		area.x = area.y = 0;
		area.width = width;
		area.height = height;
		_xeno_style_fill_rectangle (style, pixmap, style->bg_gc[state_type], bg_pixmap,
								   state_type, &area, 0, 0, width, height);
		if ((image = gdk_image_get (pixmap, 0, 0, width, height)) == NULL) {
			gdk_pixmap_unref (pixmap);
			return bg_pixmap;
		}
	}
	
	xeno_gradient_context_init (&ctx, style, state_type, visual, gdk_window_get_colormap(window),
								image, bg_pixmap != NULL, TRUE);

	if (!rc_gradient->interlaced) {
		if (gradient_type == XENO_GRADIENT_VERTICAL) {
			f = 2.0 / (height - 1);
			for (y = 0; y < height; ++y) {
				t = f * y;
				for (x = 0; x < width; ++x)
					ctx.put_pixel (&ctx, t, x, y);
			}
		} else if (gradient_type == XENO_GRADIENT_HORIZONTAL) {
			f = 2.0 / (width - 1);
			for (y = 0; y < height; ++y) {
				for (x = 0; x < width; ++x)
					ctx.put_pixel (&ctx, f * x, x, y);
			}
		} else {
			f = 1.0 / (width - 1);
			g = 1.0 / (height - 1);
			for (y = 0; y < height; ++y) {
				for (x = 0; x < width; ++x)
					ctx.put_pixel (&ctx, f*x + g*y, x, y);
			}
		}
	} else {
		if (gradient_type == XENO_GRADIENT_VERTICAL) {
			f = 1.0 / (height - 1);
			for (y = 0; y < height; ++y) {
				t = f * y + (y & 0x01);
				for (x = 0; x < width; ++x)
					ctx.put_pixel (&ctx, t, x, y);
			}
		} else if (gradient_type == XENO_GRADIENT_HORIZONTAL) {
			f = 1.0 / (width - 1);
			for (y = 0; y < height; ++y) {
				t = y & 0x01;
				for (x = 0; x < width; ++x) {
					ctx.put_pixel (&ctx, f*x + t, x, y);
				}
			}
		} else {
			f = 0.5 / (width - 1);
			g = 0.5 / (height - 1);
			for (y = 0; y < height; ++y) {
				t = y & 0x01;
				for (x = 0; x < width; ++x)
					ctx.put_pixel (&ctx, f*x + g*y + t, x, y);
			}
		}
	}
	gdk_draw_image (pixmap, style->bg_gc[state_type], image, 0, 0, 0, 0, width, height);
	gdk_image_destroy (image);
	
	/* make pixmap available in more than one state if possible */
	gradient_set = &XENO_STYLE_DATA(style)->gradient_set;
	state_type = gradient_set->redirect[state_type];
	for (i = 0; i < 5; ++i) {
		if (gradient_set->redirect[i] == state_type) {
			if (gradient->pixmap[i] != NULL)
				gdk_pixmap_unref (gradient->pixmap[i]);
			
			gradient->pixmap[i] = pixmap;
			if (state_type != i)
				gdk_pixmap_ref (pixmap);
		}
	}
	return pixmap;
}

static void
xeno_gradient_destroy (XenoGradient *gradient)
{
	XenoGradient **ptr, *next;
	GdkPixmap	 *pixmap;
	gint i;
	
	g_return_if_fail (gradient != NULL);
  #if XENO_CONFIG_DIRTY_GRADIENTS
	g_return_if_fail (gradient->ref_count == 0);
  #else
	g_return_if_fail (gradient->apps == NULL);
  #endif
  #if XENO_CONFIG_COUNT_GRADIENT_STYLE
	gtk_style_unref (gradient->style);
  #endif
	
	if ((ptr = gradient->ptr) != NULL) {
		if ((*ptr = next = gradient->next) != NULL)
			next->ptr = ptr;
	}
	for (i = 0; i < 5; ++i) {
		if ((pixmap = gradient->pixmap[i]) != NULL) {
			gdk_pixmap_unref(pixmap);
		}
	}
	
	g_mem_chunk_free (xeno_gradient_chunk, gradient);
	if (--xeno_gradient_ref_count == 0) {
		g_mem_chunk_destroy (xeno_gradient_chunk);
		xeno_gradient_chunk = NULL;
	}
}

#if XENO_CONFIG_DIRTY_GRADIENTS
static void
xeno_gradient_unref (XenoGradient *gradient)
{
	g_return_if_fail (gradient != NULL);
	
	if (--gradient->ref_count == 0)
		xeno_gradient_destroy (gradient);
}
#else
static void
xeno_gradient_app_destroy (XenoGradientApp *app)
{
	g_return_if_fail (app != NULL);
	
	if ((*app->ptr = app->next) != NULL) {
		app->next->ptr = app->ptr;
	} else if (app->ptr == &app->gradient->apps) {
		xeno_gradient_destroy (app->gradient);
	}
	g_mem_chunk_free (xeno_gradient_app_chunk, app);
}
#endif

GdkPixmap *
xeno_gradient_get	(GtkStyle		*style,
					 GdkWindow		*window,
					 GtkStateType 	state_type,
					 GtkWidget		*widget,
					 gint			width,
					 gint			height)
{
  #if !XENO_CONFIG_DIRTY_GRADIENTS
	XenoGradientApp	 *app;
  #endif
	XenoGradient	 *gradient;
	XenoGradientSet	 *gradient_set;
	XenoRcData		 *rc_data;
	XenoGradientType  gradient_type;
	GdkPixmap		 *pixmap;
	gint			  i;
	
	g_return_val_if_fail (style != NULL, NULL);
	g_return_val_if_fail (window != NULL, NULL);
	g_return_val_if_fail (widget != NULL, NULL);
	g_return_val_if_fail (XENO_STYLE_RC_DATA(style), NULL);
	
	rc_data = XENO_STYLE_RC_DATA(style);
	gradient_type = rc_data->gradient[state_type].type;
	if (gradient_type == XENO_GRADIENT_NONE)
		return NULL;
	
	/* init the gradient system */
	if (xeno_gradient_quark == 0)
		xeno_gradient_quark = g_quark_from_string ("XenoGradientWidgetMapping");
	
  #if !XENO_CONFIG_DIRTY_GRADIENTS
	if (xeno_gradient_app_chunk == NULL)
		xeno_gradient_app_chunk = g_mem_chunk_create (XenoGradientApp, 32, G_ALLOC_AND_FREE);
	
  #endif
	if (xeno_gradient_chunk == NULL)
		xeno_gradient_chunk = g_mem_chunk_create (XenoGradient, 8, G_ALLOC_AND_FREE);
	
	/* Lookup gradient attached to widget */
  #if XENO_CONFIG_DIRTY_GRADIENTS
	if ((gradient = gtk_object_get_data_by_id (GTK_OBJECT(widget), xeno_gradient_quark)) != NULL) {
  #else
	if ((app = gtk_object_get_data_by_id (GTK_OBJECT(widget), xeno_gradient_quark)) != NULL) {
		gradient = app->gradient;
  #endif
		if (   gradient->ptr != NULL
			&& gradient->style == style
			&& (   (gradient_type == XENO_GRADIENT_VERTICAL   && gradient->height == height)
				|| (gradient_type == XENO_GRADIENT_HORIZONTAL && gradient->width == width)
				|| (gradient->width == width && gradient->height == height)))
		{
			goto match;
		}
  #if XENO_CONFIG_DIRTY_GRADIENTS
		gtk_object_remove_data_by_id (GTK_OBJECT(widget), xeno_gradient_quark);
  #else
		if ((*app->ptr = app->next) != NULL) {
			app->next->ptr = app->ptr;
		} else if (gradient->apps == NULL) {
			xeno_gradient_destroy (gradient);
		}
	} else {
		if ((app = g_mem_chunk_alloc (xeno_gradient_app_chunk)) == NULL)
			return NULL;
		
		app->widget = widget;
		gtk_object_set_data_by_id_full (GTK_OBJECT(widget), xeno_gradient_quark,
										app, (GtkDestroyNotify)xeno_gradient_app_destroy);
  #endif
	}
	
	/* Find gradient with the right size */
	gradient_set = &XENO_STYLE_DATA(style)->gradient_set;
	for (gradient = gradient_set->gradients; gradient != NULL; gradient = gradient->next) {
		if (   (gradient_type == XENO_GRADIENT_VERTICAL   && gradient->height == height)
			|| (gradient_type == XENO_GRADIENT_HORIZONTAL && gradient->width == width)
			|| (gradient->width == width && gradient->height == height))
		{
			goto attach;
		}
	}
	
	/* Create new gradient */
	if ((gradient = g_mem_chunk_alloc (xeno_gradient_chunk)) == NULL) {
	  #if !XENO_CONFIG_DIRTY_GRADIENTS
		g_mem_chunk_free (xeno_gradient_app_chunk, app);
	  #endif
		return NULL;
	}
	
	gradient->style = style;
  #if XENO_CONFIG_COUNT_GRADIENT_STYLE
	gtk_style_ref (style);
  #endif
	
	if ((gradient->next = gradient_set->gradients) != NULL)
		gradient->next->ptr = &gradient->next;
	gradient->ptr = &gradient_set->gradients;
	gradient_set->gradients = gradient;
	
	gradient->width  = width;
	gradient->height = height;
	for (i = 0; i < 5; ++i)
		gradient->pixmap[i] = NULL;
	
  #if XENO_CONFIG_DIRTY_GRADIENTS
	gradient->ref_count = 0;
	xeno_gradient_ref_count += 1;
  #else
	gradient->apps = NULL;
  #endif
	
  attach:
	/* Attach gradient to widget */
  #if XENO_CONFIG_DIRTY_GRADIENTS
	gtk_object_set_data_by_id_full (GTK_OBJECT(widget), xeno_gradient_quark,
									gradient, (GtkDestroyNotify)xeno_gradient_unref);
	gradient->ref_count += 1;
  #else
	if ((app->next = gradient->apps) != NULL)
		app->next->ptr = &app->next;
	app->ptr = &gradient->apps;
	gradient->apps = app;
	app->gradient = gradient;
  #endif
	
  match:
	/* Create gradient pixmap if necessary */
	if ((pixmap = gradient->pixmap[state_type]) == NULL)
		pixmap = xeno_gradient_realize (style, window, state_type, gradient);
	
	return pixmap;
}

void
xeno_gradient_set_realize (XenoGradientSet *gradient_set, GtkStyle *style)
{
	XenoRcData		*rc_data;
	XenoRcGradient	*rc_gradient;
	XenoGradient	*gradient;
	gint			i, j;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (XENO_STYLE_IS_XENO(style));
	g_return_if_fail (gradient_set != NULL);
	
	rc_data = XENO_STYLE_RC_DATA(style);
	if (rc_data == NULL)
		return;
	
	/* clear existing gradients, reusing existing structures but forcing the pixmaps
	   to be re-realized
	*/
	for (gradient = gradient_set->gradients; gradient != NULL; gradient = gradient->next) {
		for (i = 0; i < 5; ++i) {
			if (gradient->pixmap[i] != NULL) {
				gdk_pixmap_unref (gradient->pixmap[i]);
				gradient->pixmap[i] = NULL;
			}
		}
	}
	
	/* determine which pixmaps we can use for more than one state */
	for (i = 0; i < 5; ++i) {
		rc_gradient	= &rc_data->gradient[i];
		if (rc_gradient->type != XENO_GRADIENT_NONE) {
			for (j = 0; j < i; ++j) {
				if (   rc_gradient[i].type == rc_data->gradient[j].type
					&& rc_gradient[i].interlaced == rc_data->gradient[i].interlaced
					&& rc_gradient[i].factor == rc_data->gradient[j].factor
					&& rc_data->white[i] == rc_data->white[j]
					&& rc_data->black[i] == rc_data->black[j]
					&& style->bg_pixmap[i] == style->bg_pixmap[j]
					&& gdk_color_equal (&style->bg[i], &style->bg[j]))
				{
					break;
				}
			}
			gradient_set->redirect[i] = j;
		}
	}
}

void
xeno_gradient_set_unrealize (XenoGradientSet *gradient_set)
{
  #if !XENO_CONFIG_DIRTY_GRADIENTS
	XenoGradientApp	*app, *next_app;
  #endif
	XenoGradient	*gradient, *next_gradient;
	gint i;
	
	g_return_if_fail (gradient_set != NULL);
	
	for (gradient = gradient_set->gradients; gradient != NULL; gradient = next_gradient) {
		next_gradient = gradient->next;
	  #if XENO_CONFIG_DIRTY_GRADIENTS
		/* dissolve the list and unrealize */
		*gradient->ptr = NULL;
		gradient->ptr = NULL;
	  #if !XENO_CONFIG_COUNT_GRADIENT_STYLE
		gradient->style = NULL;
	  #endif
		for (i = 0; i < 5; ++i) {
			if (gradient->pixmap[i]) {
				gdk_pixmap_unref (gradient->pixmap[i]);
				gradient->pixmap[i] = NULL;
			}
		}
	  #else
		for (app = gradient->apps; app != NULL; app = next_app) {
			next_app = app->next;
			gtk_object_remove_data_by_id (GTK_OBJECT(app->widget), xeno_gradient_quark);
		}
	  #endif
	}
}

