#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <inttypes.h>
#include <errno.h>
#include <signal.h>
#include <assert.h>
#include <fcntl.h>
#include <termios.h>
#include <signal.h>
#include <setjmp.h>
#include <pthread.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <sys/vfs.h>

#include <linux/if.h>
#include <linux/if_tun.h>
#include <linux/if_bridge.h>

#include <uuid/uuid.h>

#include "xenner.h"

/* ------------------------------------------------------------------ */

static struct xenvm xen = {
    .pg_total    = MB_TO_PG(128),
    .debug       = 1,
    .corefd      = -1,
    .name        = "no-name",
    .vcpus       = 1,

    .qemu.upstream = 0,
    .qemu.path   = "/usr/" LIB "/xen/bin/qemu-dm",
    .qemu.kbd    = "en-us",
    .qemu.serial = "none",
};

/* ------------------------------------------------------------------ */

char *search_path[] = {
    ".",
    ".",
    "/usr/" LIB "/xenner",
    "/usr/local/" LIB "/xenner",
};
int search_path_length = sizeof(search_path)/sizeof(search_path[0]);
static char search_cwd[256];
static char search_exe[256];

static void prepare_search_path(struct xenvm *xen, char *argv0)
{
    char *h;
    
    getcwd(search_cwd, sizeof(search_cwd));
    search_path[0] = search_cwd;

    strncpy(search_exe, argv0, sizeof(search_exe));
    h = strrchr(search_exe,'/');
    if (h)
	*h = 0;
    search_path[1] = search_exe;
}

/* ------------------------------------------------------------------ */

#define MONITOR_GREETING  "xenner " VERSION " monitor\r\n"
#define MONITOR_ERROR     "unknown command\r\n"
#define MONITOR_ERROR_SUB "unknown sub-command\r\n"
#define MONITOR_PROMPT    "(qemu) "
#define MONITOR_HELP      \
    "help              - this text\r\n"                  \
    "info <item>       - query various informations\r\n" \
    "stop              - pause guest execution\r\n"      \
    "cont              - continue guest execution\r\n"   \
    "system_shutdown   - gracefull guest shutdown\r\n"   \
    "system_powerdown  - gracefull guest shutdown\r\n"   \
    "quit              - destoy guest and quit\r\n"      \
    ""

static int xwrite(int fd, const void *buf, size_t count)
{
    int rc, try;

    for (try = 0; try < 5; try++) {
	rc = write(fd, buf, count);
	if (rc == count)
	    break;
	if (rc == -1 && EINTR == errno)
	    continue;
	fprintf(stderr, "%s: write(mon, buf, %zd) = %d, errno %s\n",
		__FUNCTION__, count, rc, strerror(errno));
    }
    return rc;
}

static int qemu_monitor_info(struct xenvm *xen, int fd, char *line)
{
    static const char error[]  = MONITOR_ERROR_SUB;
    char msg[1024];
    int i, len;
    
    if (0 == strcmp(line, "cpus")) {
	for (i = 0, len = 0; i < xen->vcpus; i++) {
	    len += snprintf(msg+len, sizeof(msg)-len,
			    "%c CPU #%d: pc=0x%" PRIx64 " thread_id=%d\r\n",
			    (i == 0) ? '*' : ' ', 0,
			    (uint64_t)0,
			    xen->vcpu[i].thread_id);
	}
	xwrite(fd, msg, strlen(msg));

    } else {
	xwrite(fd, error, strlen(error));

    }
    return 0;
}

static int qemu_monitor_line(struct xenvm *xen, int fd, char *line)
{
    static const char nl[]     = "\r\n";
    static const char prompt[] = MONITOR_PROMPT;
    static const char error[]  = MONITOR_ERROR;
    static const char help[]   = MONITOR_HELP;

    d0printf("%s: \"%s\"\n", __FUNCTION__, line);
    xwrite(fd, nl, strlen(nl));

    if (0 == strcmp(line, "help")) {
	xwrite(fd, help, strlen(help));

    } else if (0 == strncmp(line, "info ", 5)) {
	qemu_monitor_info(xen, fd, line + 5);

    } else if (0 == strcmp(line, "stop")) {
	kvmbits_vcpus_stop(xen);

    } else if (0 == strcmp(line, "cont")) {
	kvmbits_vcpus_cont(xen);

    } else if ((0 == strcmp(line, "system_shutdown"))   ||
               (0 == strcmp(line, "system_powerdown")))  {
	kill(getpid(), SIGINT);

    } else if (0 == strcmp(line, "quit")) {
	kill(getpid(), SIGTERM);

    } else if (strlen(line)) {
	xwrite(fd, error, strlen(error));
 
    }
    xwrite(fd, prompt, strlen(prompt));
    return 0;
}

static int qemu_monitor_data(struct xenvm *xen, int fd, void *data)
{
    static const char msg[] = MONITOR_GREETING MONITOR_PROMPT;
    char buf[256];
    int rc, pos, i;

    rc = read(fd, buf, sizeof(buf));
    switch (rc) {
    case -1:
	if (EIO == errno) {
	    if (xen->mon_connected) {
		d1printf("%s: disconnected\n", __FUNCTION__);
		xen->mon_connected = 0;
	    }
	    return 0;
	} else if (EAGAIN == errno) {
	    if (!xen->mon_connected) {
		d1printf("%s: connected\n", __FUNCTION__);
		xwrite(fd, msg, strlen(msg));
	    }
	    xen->mon_connected = 1;
	    return 0;
	} else {
	    d1printf("%s: read: %s\n", __FUNCTION__, strerror(errno));
	}
	break;
    case 0:
	d1printf("%s: read: EOF\n", __FUNCTION__);
	break;
    default:
	buf[rc] = 0;
	xwrite(fd, buf, rc); /* echo */
	pos  = strlen(xen->mon_buffer);
	pos += snprintf(xen->mon_buffer+pos,
			sizeof(xen->mon_buffer)-pos, "%s", buf);

	/* try split into lines, failing that just return and wait for
	   more data  (unless someone overflows our buffer) */
	i = 0;
	while (i < pos) {
	    buf[i] = xen->mon_buffer[i];
	    if (buf[i] == '\r' || buf[i] == '\n') {
		buf[i] = '\0';
		qemu_monitor_line(xen, fd, buf);
		memmove(xen->mon_buffer, xen->mon_buffer+i+1, pos-i);
		pos -= i+1;
		i = 0;
		continue;
	    }
	    i++;
	}
	if (pos == sizeof(xen->mon_buffer)) {
	    d0printf("%s: zap flooded buffer\n", __FUNCTION__);
	    xen->mon_buffer[0] = '\0';
	}
	break;
    }
    return 0;
}

static int qemu_monitor_config(struct xenvm *xen, char *arg)
{
    char *dev = NULL;
    int tty = -1;

    d1printf("%s: %s\n", __FUNCTION__, arg);
    if (0 == strcmp(arg, "pty")) {
	tty = getpt();
	if (-1 == tty)
	    d1printf("%s: getpt: %s\n", __FUNCTION__, strerror(errno));
	if (-1 == grantpt(tty))
	    d1printf("%s: grantpt: %s\n", __FUNCTION__, strerror(errno));
	if (-1 == unlockpt(tty))
	    d1printf("%s: unlockpt: %s\n", __FUNCTION__, strerror(errno));
	dev = ptsname(tty);
	/* make read() return -EIO when unconnected, so we can
	 * figure when someone (libvirt ...) connects */
	close(open(dev, O_RDWR));
    }
    if (dev) {
	struct termios tio;
	
	d1printf("%s: %s\n", __FUNCTION__, dev);
	tcgetattr (tty, &tio);
	tio.c_lflag &= ~(ECHO|ICANON|ISIG);
	tio.c_cc[VMIN] = 1;
	tio.c_cc[VTIME] = 0;
	tcsetattr (tty, TCSAFLUSH, &tio);
	/* libvirt expects this on stderr */
	fprintf(stderr, "char device redirected to %s\n", dev);
    }
    return tty;
}

static int qemu_monitor_setup(struct xenvm *xen, int tty)
{
    xenner_register_cb(xen, "monitor",
		       tty, NULL, qemu_monitor_data);
    qemu_monitor_data(xen, tty, NULL);
    return 0;
}

/* ------------------------------------------------------------------ */

static int tap_fd_to_name(int fd, char *buf, int len)
{
    /* try to figure interface name from file handle */
    struct ifreq tun, sys;
    int sock = -1, ifn, match = 0;

    /* get tun dev mac address */
    memset(&tun, 0, sizeof(tun));
    if (-1 == ioctl(fd, SIOCGIFHWADDR, &tun))
	goto out;

    /*
     * Loop over interfaces to find the interface name with the
     * given mac address.
     * NOTE: There might be a bridge device with the same mac address,
     * trying to skip that one by heuristic ...
     */
    sock = socket(PF_INET, SOCK_STREAM, 0);
    if (sock < 0)
	goto out;
    for (ifn = 0; ifn < 65536; ifn++) {
	memset(&sys, 0, sizeof(sys));
	sys.ifr_ifindex = ifn;
	if (ioctl(sock, SIOCGIFNAME, &sys) < 0) {
	    if (ENODEV == errno)
		continue;
	    goto out;
	}
	if (ioctl(sock, SIOCGIFHWADDR, &sys) < 0)
	    goto out;
	if (0 != memcmp(tun.ifr_hwaddr.sa_data, sys.ifr_hwaddr.sa_data, 6))
	    continue;
	/* bridge interface guess work - start */
	if (NULL != strstr(sys.ifr_name, "br"))
	    continue;
	/* bridge interface guess work - end */
	strncpy(buf, sys.ifr_name, len);
	match++;
	break;
    }

out:
    if (-1 != sock)
	close(sock);
    return match;
}

static int qemu_parse_string(char *arg, char *tag, char *buf, int len)
{
    char *pos;
    int i = 0;

    pos = strstr(arg, tag);
    if (!pos)
	return -1;
    pos += strlen(tag);
    while (*pos != '\0' && *pos != ',' &&
	   *pos != '\r' && *pos != '\n' &&
	   i < len-1) {
	buf[i] = *pos;
	i++; pos++;
    }
    buf[i] = 0;
    return i;
}

static int qemu_parse_int(char *arg, char *tag, int def)
{
    char buf[32];

    if (qemu_parse_string(arg, tag, buf, sizeof(buf)) <= 0)
	return def;
    return atoi(buf);
}

static int qemu_net_config_save(struct xenvm *xen, char *arg)
{
    int i,n;

    n = sizeof(xen->qemu.nic)/sizeof(xen->qemu.nic[0]);
    for (i = 0; i < n; i++) {
	if (xen->qemu.nic[i])
	    continue;
	xen->qemu.nic[i] = arg;
	return 0;
    }
    return -1;
}

static int qemu_net_config_netbackd(struct xenvm *xen)
{
    char dummy[32], *arg;
    unsigned int vlan;
    int i,n;

    n = sizeof(xen->qemu.nic)/sizeof(xen->qemu.nic[0]);
    for (i = 0; i < n; i++) {
	arg = xen->qemu.nic[i];
	if (!arg)
	    break;

	d1printf("%s: %s\n", __FUNCTION__, arg);
	if (0 == strncmp(arg, "none", 4)) {
	    return 0;
	}
	if (0 == strncmp(arg, "user", 4)) {
	    d1printf("%s: user network not implemented\n", __FUNCTION__);
	    return 0;
	}
	
	vlan = qemu_parse_int(arg, "vlan=", 0);
	if (vlan >= sizeof(xen->nics)/sizeof(xen->nics[0])) {
	    d1printf("%s: vlan number too big (%u, %zd max)\n", __FUNCTION__,
		     vlan, sizeof(xen->nics)/sizeof(xen->nics[0])-1);
	    return 0;
	}
	if (!xen->nics[vlan].configured)
	    xen->nics[vlan].fd = -1;
	xen->nics[vlan].configured++;
	
	if (0 == strncmp(arg, "tap", 3)) {
	    xen->nics[vlan].fd = qemu_parse_int(arg, "fd=", -1);
	    qemu_parse_string(arg, "ifname=", xen->nics[vlan].ifname,
			      sizeof(xen->nics[vlan].ifname));
	    if (-1 != xen->nics[vlan].fd) {
		/* try figure interface name unless we got one passed via ifname= */
		if (0 == strlen(xen->nics[vlan].ifname))
		    tap_fd_to_name(xen->nics[vlan].fd, xen->nics[vlan].ifname,
				   sizeof(xen->nics[vlan].ifname));
	    }
	    d1printf("%s: %d: if=\"%s\"\n", __FUNCTION__, vlan,
		     xen->nics[vlan].ifname);
	    /* ignored ... */
	    qemu_parse_string(arg, "script=", dummy, sizeof(dummy));
	}
	
	if (0 == strncmp(arg, "nic", 3)) {
	    qemu_parse_string(arg, "macaddr=", xen->nics[vlan].mac,
			      sizeof(xen->nics[vlan].mac));
	    d1printf("%s: %d: mac=\"%s\"\n", __FUNCTION__, vlan,
		     xen->nics[vlan].mac);
	    /* ignored ... */
	    qemu_parse_string(arg, "model=", dummy, sizeof(dummy));
	}
    }
    return 0;
}

static int qemu_disk_config_save(struct xenvm *xen, char *arg)
{
    char boot[8];
    int index;
    int i,n;

    n = sizeof(xen->qemu.disk)/sizeof(xen->qemu.disk[0]);
    for (i = 0; i < n; i++) {
	if (xen->qemu.disk[i])
	    continue;
	xen->qemu.disk[i] = arg;

	index = qemu_parse_int(arg, "index=", 0);
	qemu_parse_string(arg, "boot=", boot, sizeof(boot));
	if (0 == strcmp(boot,"on"))
	    qemu_parse_string(arg, "file=", xen->bootdisk, sizeof(xen->bootdisk));
	else if (0 == index && 0 == strlen(xen->bootdisk))
	    qemu_parse_string(arg, "file=", xen->bootdisk, sizeof(xen->bootdisk));
	return 0;
    }
    return -1;
}

static int qemu_disk_config_blkbackd(struct xenvm *xen)
{
    char buf[32], dummy[32], interface[32];
    unsigned int index;
    char *arg;
    int i,n;

    n = sizeof(xen->qemu.disk)/sizeof(xen->qemu.disk[0]);
    
    for (i = 0; i < n; i++) {
	arg = xen->qemu.disk[i];
	if (!arg)
	    break;

	d1printf("%s: %s\n", __FUNCTION__, arg);
	
	if (-1 != qemu_parse_string(arg, "if=", interface, sizeof(interface))) {
	    if (0 != strcmp(interface, "xen")) {
		d1printf("%s: if != xen, ignoring disk\n", __FUNCTION__);
		continue;
	    }
	}

	index = qemu_parse_int(arg, "index=", 0);
	if (index >= sizeof(xen->disks)/sizeof(xen->disks[0])) {
	    d1printf("%s: index too big (%u, %zd max)\n", __FUNCTION__,
		     index, sizeof(xen->disks)/sizeof(xen->disks[0])-1);
	    return 0;
	}
	xen->disks[index].configured++;
	
	qemu_parse_string(arg, "file=", xen->disks[index].diskimage,
			  sizeof(xen->disks[index].diskimage));
	if (-1 != qemu_parse_string(arg, "media=", buf, sizeof(buf)))
	    if (0 == strcmp(buf, "cdrom"))
		xen->disks[index].cdrom = 1;
		
	/* ignored ... */
	qemu_parse_string(arg, "bus=", dummy, sizeof(dummy));
	qemu_parse_string(arg, "cyls=", dummy, sizeof(dummy));
	qemu_parse_string(arg, "snapshot=", dummy, sizeof(dummy));
	qemu_parse_string(arg, "cache=", dummy, sizeof(dummy));
    }
	
    return 0;
}

/* ------------------------------------------------------------------ */

static int qemu_vnc_config(struct xenvm *xen, char *arg)
{
    d1printf("%s: %s\n", __FUNCTION__, arg);
    xen->qemu.vnc = arg;
    return 0;
}

static int qemu_kbd_config(struct xenvm *xen, char *arg)
{
    d1printf("%s: %s\n", __FUNCTION__, arg);
    xen->qemu.kbd = arg;
    return 0;
}

static int qemu_serial_config(struct xenvm *xen, char *arg)
{
    d1printf("%s: %s\n", __FUNCTION__, arg);
    xen->qemu.serial = arg;
    return 0;
}

static void prepare_environment(struct xenvm *xen)
{
    char *shlib = "libxenctrl.so." XEN_SO_MAJOR;
    char buf[256];
    int i;

    for (i = 0; i < search_path_length; i++) {
	snprintf(buf, sizeof(buf), "%s/%s", search_path[i], shlib);
	if (0 != access(buf, R_OK))
	    continue;
	setenv("LD_LIBRARY_PATH", search_path[i], 1);
	d1printf("%s: LD_LIBRARY_PATH=%s\n", __FUNCTION__, search_path[i]);
	break;
    }
}

static int do_fork_exec(struct xenvm *xen, char **argv)
{
    char msg[1024];
    int i, len;
    pid_t pid;

    len = snprintf(msg, sizeof(msg), "%s: \"%s\"", __FUNCTION__, argv[0]);
    for (i = 1; argv[i]; i++)
	len += snprintf(msg+len, sizeof(msg)-len, " \"%s\"", argv[i]);
    d1printf("%s\n", msg);
    
    pid = fork();
    switch (pid) {
    case -1: /* Oops */
	d0printf("%s: fork: %s\n", __FUNCTION__, strerror(errno));
	return 0;
    case 0: /* child */
	execv(argv[0], argv);
	d1printf("%s: execv(%s): %s\n", __FUNCTION__, argv[0], strerror(errno));
	exit(1);
    default: /* parent */
	return pid;
    }
}

static void qemu_dm_setup(struct xenvm *xen)
{
    char domid[16];
    char *argv[] = {
	"/usr/" LIB "/xen/bin/qemu-dm",
	"-M", "xenpv",
	"-k", xen->qemu.kbd,
	"-vnc", xen->qemu.vnc,
#if 0
	"-serial", xen->qemu.serial,
#endif
	"-d", domid,
	"-domain-name", xen->name,
	NULL
    };

#if 1
    if (0 == strcmp(xen->qemu.serial, "pty"))
	fprintf(stderr, "char device redirected to /dev/null\n");
#endif

    snprintf(domid, sizeof(domid), "%d", xen->domid);
    prepare_environment(xen);
    xen->qemu_pid = do_fork_exec(xen, argv);
}

static void qemu_upstream_setup(struct xenvm *xen)
{
    char domid[16];
    char *argv[64];
    int argc, i, n;

    snprintf(domid, sizeof(domid), "%d", xen->domid);
    prepare_environment(xen);

    argc = 0;
    argv[argc++] = xen->qemu.path;
#if 0
    argv[argc++] = "-no-kvm";
    argv[argc++] = "-kernel";
    argv[argc++] =    "dummy";
#endif
    argv[argc++] = "-M";
    argv[argc++] =    "xenpv";
    argv[argc++] = "-domid";
    argv[argc++] =    domid;
    argv[argc++] = "-name";
    argv[argc++] =    xen->name;
    if (xen->qemu.vnc) {
	argv[argc++] = "-vnc";
	argv[argc++] =    xen->qemu.vnc;
	argv[argc++] = "-k";
	argv[argc++] =    xen->qemu.kbd;
    }
    argv[argc++] = "-serial";
    argv[argc++] =    xen->qemu.serial;

    n = sizeof(xen->qemu.nic)/sizeof(xen->qemu.nic[0]);
    for (i = 0; i < n; i++) {
	if (!xen->qemu.nic[i])
	    break;
	argv[argc++] = "-net";
	argv[argc++] = xen->qemu.nic[i];
    }
    
    n = sizeof(xen->qemu.disk)/sizeof(xen->qemu.disk[0]);
    for (i = 0; i < n; i++) {
	if (!xen->qemu.disk[i])
	    break;
	argv[argc++] = "-drive";
	argv[argc++] = xen->qemu.disk[i];
    }
    
    argv[argc++] = NULL;
    xen->qemu_pid = do_fork_exec(xen, argv);
}

int qemu_cleanup(struct xenvm *xen)
{
    if (xen->qemu_pid > 0)
	kill(xen->qemu_pid, SIGTERM);
    return 0;
}

/* ------------------------------------------------------------------ */

static int run_pygrub(struct xenvm *xen, char *bootloader)
{
    char out[32] = "/tmp/py.XXXXXX";
    char *argv[16];
    int argc = 0;
    char buf[1024], kernel[128], initrd[128], append[512];
    char *disk, *h;
    int st, rc, fd;
    int interactive;
    pid_t pid;

    if (0 == strlen(xen->bootdisk))
	return -1;

    fd = mkstemp(out);

    disk = xen->bootdisk;
    h = strchr(disk,':');
    if (h)
	disk = h+1;

    interactive = 1;
    if (xen->nostderr)
	interactive = 0;
    if (!isatty(2 /* stderr */))
	interactive = 0;

    argv[argc++] = bootloader ? bootloader : "pygrub";
    argv[argc++] = interactive ? "--interactive" : "--quiet";
    argv[argc++] = "--output";
    argv[argc++] = out;
    if (xen->kernel) {
	argv[argc++] = "--kernel";
	argv[argc++] = xen->kernel;
    }
    if (xen->ramdisk) {
	argv[argc++] = "--ramdisk";
	argv[argc++] = xen->ramdisk;
    }
    if (xen->cmdline) {
	argv[argc++] = "--args";
	argv[argc++] = xen->cmdline;
    }
    argv[argc++] = disk;
    argv[argc++] = NULL;

    logprintf(xen, "running bootloader %s on %s (%s mode) ...\n",
	      argv[0], xen->bootdisk,
	      interactive ? "interactive" : "quiet");
    pid = fork();
    switch (pid) {
    case -1: /* Oops */
	d0printf("%s: fork: %s\n", __FUNCTION__, strerror(errno));
	return -1;
    case 0: /* child */
	if (xen->nostderr) {
	    /* quiet mode -- connect stdin/out/err to /dev/null */
	    close(0); close(1); close(2);
	    open("/dev/null", O_RDWR); dup(0); dup(0);
	}
	execvp(argv[0], argv);
	d1printf("execvp(%s): %s\n", argv[0], strerror(errno));
	exit(1);
    default: /* parent */
	waitpid(pid, &st, 0);
	unlink(out);
	break;
    }
    logprintf(xen, "... bootloader is done.\n");

    rc = read(fd, buf, sizeof(buf));
    if (rc < 0)
	return -1;
    buf[rc] = 0;
    kernel[0] = 0;
    initrd[0] = 0;
    append[0] = 0;

    if (NULL != (h = strstr(buf, "(kernel")))
	sscanf(h, "(kernel %128[^)]", kernel);
    if (NULL != (h = strstr(buf, "(ramdisk")))
	sscanf(h, "(ramdisk %128[^)]", initrd);
    if (NULL != (h = strstr(buf, "(args")))
	if (1 != sscanf(h, "(args \"%512[^\"]", append))
	    sscanf(h, "(args %512[^)]", append);
    d1printf("%s: kernel=\"%s\" initrd=\"%s\" append=\"%s\"\n",
	     __FUNCTION__, kernel, initrd, append);

    if (strlen(kernel)) {
	xen->kernel = strdup(kernel);
	xen->unlink_kernel = 1;
    }
    if (strlen(initrd)) {
	xen->ramdisk = strdup(initrd);
	xen->unlink_ramdisk = 1;
    }
    if (strlen(append)) {
	xen->cmdline = strdup(append);
    }
    return 0;
}

static void run_xenconsole(struct xenvm *xen)
{
    char domid[16];
    char *argv[] = {
	"/usr/" LIB "/xen/bin/xenconsole", domid,
	NULL
    };

    snprintf(domid, sizeof(domid), "%d", xen->domid);
    do_fork_exec(xen, argv);
}

/* ------------------------------------------------------------------ */

static void sigsegv(int signo, siginfo_t *info, void *context)
{
    struct statfs fs;
    int mainthread;

    mainthread = pthread_equal(xen.mainthread, pthread_self());
    logprintf(&xen, "Oops: Got fatal signal %d (%s), %s, emergency shutdown\n",
	      signo, strsignal(signo),
	      mainthread ? "main thread" : "vcpu thread");

    /* stop all vcpus except myself */
    kvmbits_signal_vcpus(&xen, SIGUSR1, 0);
    logprintf(&xen, "VCPUs stopped.\n");

    /* check vmcore tmpfs */
    if (-1 == statfs(VMCORE_PATH, &fs)) {
	perror("statfs(" VMCORE_PATH ")");
    } else if (0 == fs.f_bavail) {
	logprintf(&xen, "I'm out of memory, tmpfs at %s is full.\n", VMCORE_PATH);
	logprintf(&xen, "  try: \"xenner-cleanup\" to wipe stale vmcore files.\n");
	logprintf(&xen, "  try: resize filesystem using \"%s\"\n",
		  "mount -o remount,size=<nbytes> /var/run/xenner");
    }

    /* cleanup and exit */
    if (mainthread) {
	xenner_cleanup(&xen, 0);
	exit(1);
    } else {
	logprintf(&xen, "Signaling main thread for shutdown.\n");
	pthread_mutex_unlock(&xen.biglock);
	pthread_kill(xen.mainthread, SIGTERM);
	pthread_kill(pthread_self(), SIGUSR1);
    }
}

static void openlog(struct xenvm *xen, char *filename)
{
    xen->logfile = fopen(filename, "w");
    if (!xen->logfile) {
	logprintf(xen, "open %s: %s\n", filename, strerror(errno));
	return;
    }
    setvbuf(xen->logfile, NULL, _IONBF, 0);
    logprintf(xen, "logging to %s\n", filename);
}

static void libvirt_caps(FILE *fp)
{
    static struct kvm_callbacks dummy;
    void *kvm;
 
    /* test for kvm */
    kvm = kvm_init(&dummy, NULL);
    if (NULL == kvm)
	return;
    if (!kvm_check_extension(kvm, KVM_CAP_SET_TSS_ADDR)) {
	kvm_finalize(kvm);
	return;
    }
    kvm_finalize(kvm);

    /* 32 bit guests */
    fprintf(fp,
	    "\n"
	    "  <guest>\n"
	    "    <os_type>xen</os_type>\n"
	    "    <arch name=\"i686\">\n"
	    "      <wordsize>32</wordsize>\n"
	    "      <emulator>/usr/bin/xenner</emulator>\n"
	    "      <domain type=\"kvm\"></domain>\n"
	    "      <machine>xenner</machine>\n"
	    "    </arch>\n"
	    "    <features>\n"
	    "      <pae/>\n"
	    "      <nonpae/>\n"
	    "    </features>\n"
	    "  </guest>\n");

#ifdef __x86_64__
    /* 64 bit guests */
    fprintf(fp,
	    "\n"
	    "  <guest>\n"
	    "    <os_type>xen</os_type>\n"
	    "    <arch name=\"x86_64\">\n"
	    "      <wordsize>64</wordsize>\n"
	    "      <emulator>/usr/bin/xenner</emulator>\n"
	    "      <domain type=\"kvm\"></domain>\n"
	    "      <machine>xenner</machine>\n"
	    "    </arch>\n"
	    "  </guest>\n"
	);
#endif
}

static void usage(FILE *fp)
{
    fprintf(fp,
	    "QEMU PC emulator version 0.9.1\n"
	    "/*\n"
	    " * No, it isn't.  This just makes libvirt qemu version detection happy,\n"
	    " * so you can manage xenner guests using libvirts qemu driver.\n"
	    " */\n"
	    "\n"
	    "xenner v" VERSION " -- run xen kernels in kvm\n"
	    "\n"
	    "usage: xenner [options]\n"
	    "options (qemu compatible):\n"
	    "   -help                    print this text\n"
	    "   -S                       start with vcpus stopped\n"
	    "   -kernel     <kernel>     xenified kernel image\n"
	    "   -initrd     <ramdisk>    ramdisk\n"
	    "   -append     <append>     kernel command line\n"
	    "   -name       <string>     set domain name\n"
	    "   -monitor    pty          monitor on pseudo tty\n"
	    "   -m          <memory>     vm memory in megabytes\n"
	    "   -smp        <cpus>       set number of vcpus\n"
	    "   -hd[abcd]   <file>       disk image\n"
	    "   -cdrom      <file>       cdrom image\n"
	    "   -drive      <config>     qemu style disk setup\n"
	    "   -net        <config>     qemu style network setup\n"
	    "   -serial     <config>     qemu style serial line (aka console) setup\n"
	    "   -vnc        <display>    vnc display (needs qemu-dm from xen-tools)\n"
	    "   -k          <kbd>        vnc clients keyboard map\n"
	    "\n"
	    "options (xenner only):\n"
	    "   -domid      <id>         set domain id\n"
	    "   -uuid       <uuid>       set domain uuid\n"
	    "   -network    <mac>,<br>   xenner style network setup\n"
	    "   -debug      <nr>         set debug level\n"
	    "   -logfile    <file>       set logfile\n"
	    "   -bootloader <app>        set bootloader\n"
	    "   -cap-on     <name>       enable kvm features\n"
	    "   -cap-off    <name>       disable kvm features\n"
	    "   -nostderr                send messages to logfile only\n"
	    "   -xenconsole              connect to guests xen console\n"
	    "   -dumpcore                don't catch fatal signals (sigsegv, sigbus)\n"
	    "   -libvirt-caps            print libvirt capability xml snippet\n"
	    "\n"
	    "pick device model:\n"
	    "   -qemu-path  <file>       path to qemu binary\n"
	    "   -qemu-dm                 qemu is xensource qemu-dm\n"
	    "   -qemu-upstream           qemu is upstream+patches\n"
	    "               => http://kraxel.fedorapeople.org/patches/\n"
	    "\n"
	    "Some more qemu-ish options are accepted but ignored,\n"
	    "so we don't error out when started by libvirt.\n"
	    "\n"
	    "When started without kernel specified xenner tries to\n"
	    "get one using the bootloader (pygrub by default).\n"
	    "\n"
	    "-- \n"
	    "(c) 2007,08 Gerd Hoffmann <kraxel@redhat.com>\n"
	    "\n");
}

int main(int argc, char *argv[])
{
    struct sigaction act, old;
    char drive[512];
    char *colors = NULL;
    char *uuid = NULL;
    char *bootloader = NULL;
    int monitor_fd = -1;
    int xenconsole = 0;
    int dumpcore   = 0;
    int opt, rc;

    xen.domid = getpid();
    INIT_LIST_HEAD(&xen.callbacks);
    
    for (opt = 1; opt < argc; opt++) {
	if (opt < argc-1) {
	    /* options with one argument */
	    if (0 == strcmp(argv[opt], "-boot")        ||
		0 == strcmp(argv[opt], "-parallel")    ||
		0 == strcmp(argv[opt], "-usbdevice"))   {
		/* ignore */
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-M")) {
		/*
		 * libvirt detection: calls us with -M xenner
		 * nothing special done (any more).
		 */
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-hda") ||
		       0 == strcmp(argv[opt], "-hdb") ||
		       0 == strcmp(argv[opt], "-hdc") ||
		       0 == strcmp(argv[opt], "-hdd")) {
		snprintf(drive, sizeof(drive), "file=%s,if=xen,index=%c,media=disk",
			 argv[opt+1], argv[opt][3] - 'a' + '0');
		qemu_disk_config_save(&xen, strdup(drive));
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-cdrom")) {
		snprintf(drive, sizeof(drive), "file=%s,if=xen,index=2,media=cdrom",
			 argv[opt+1]);
		qemu_disk_config_save(&xen, strdup(drive));
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-drive")) {
		qemu_disk_config_save(&xen, argv[opt+1]);
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-m")) {
		xen.pg_total = MB_TO_PG(atoi(argv[opt+1]));
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-k")) {
		qemu_kbd_config(&xen, argv[opt+1]);
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-smp")) {
		xen.vcpus = atoi(argv[opt+1]);
		if (xen.vcpus > sizeof(xen.vcpu)/sizeof(xen.vcpu[0]))
		    xen.vcpus = sizeof(xen.vcpu)/sizeof(xen.vcpu[0]);
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-name")) {
		snprintf(xen.name, sizeof(xen.name), "%s", argv[opt+1]);
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-monitor")) {
		monitor_fd = qemu_monitor_config(&xen, argv[opt+1]);
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-net")) {
		qemu_net_config_save(&xen, argv[opt+1]);
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-vnc")) {
		qemu_vnc_config(&xen, argv[opt+1]);
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-serial")) {
		qemu_serial_config(&xen, argv[opt+1]);
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-kernel")) {
		xen.kernel = argv[opt+1];
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-append")) {
		xen.cmdline = argv[opt+1];
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-initrd")) {
		xen.ramdisk = argv[opt+1];
		opt++; continue;

	    /* the ones below are xenner only */
	    } else if (0 == strcmp(argv[opt], "-domid")) {
		xen.domid = atoi(argv[opt+1]);
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-uuid")) {
		uuid = argv[opt+1];
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-debug")) {
		xen.debug = atoi(argv[opt+1]);
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-network")) {
		if (2 == sscanf(argv[opt+1], "%31[^,],%31s",
				xen.nics[0].mac,
				xen.nics[0].bridge))
		    xen.nics[0].configured = 1;
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-bootloader")) {
		bootloader = argv[opt+1];
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-logfile")) {
		openlog(&xen, argv[opt+1]);
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-cap-on")) {
		kvmbits_features_enable(argv[opt+1], 1);
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-cap-off")) {
		kvmbits_features_enable(argv[opt+1], 0);
		opt++; continue;

	    } else if (0 == strcmp(argv[opt], "-qemu-path")) {
		xen.qemu.path = argv[opt+1];
		opt++; continue;
	    }
	}
	/* options without arguments */
	if (0 == strcmp(argv[opt], "-usb")        ||
	    0 == strcmp(argv[opt], "-nographic")  ||
	    0 == strcmp(argv[opt], "-no-acpi"))    {
	    /* ignore */
	    continue;

	} else if (0 == strcmp(argv[opt], "-S")) {
	    xen.start_stopped = 1;
	    continue;

	} else if (0 == strcmp(argv[opt], "-nostderr")) {
	    xen.nostderr = 1;
	    continue;

	} else if (0 == strcmp(argv[opt], "-xenconsole")) {
	    xenconsole = 1;
	    continue;

	} else if (0 == strcmp(argv[opt], "-dumpcore")) {
	    dumpcore = 1;
	    continue;

	} else if (0 == strcmp(argv[opt], "-qemu-upstream")) {
	    xen.qemu.upstream = 1;
	    continue;

	} else if (0 == strcmp(argv[opt], "-qemu-dm")) {
	    xen.qemu.upstream = 0;
	    continue;

	} else if (0 == strcmp(argv[opt], "-libvirt-caps")) {
	    libvirt_caps(stdout);
	    exit(0);

	} else if (0 == strcmp(argv[opt], "-h") ||
		   0 == strcmp(argv[opt], "-help")) {
	    usage(stdout);
	    exit(0);
	}

	logprintf(&xen, "unknown option (or missing arg): %s\n", argv[opt]);
	if (xen.logfile)
	    usage(xen.logfile);
	if (!xen.nostderr)
	    usage(stderr);
	exit(1);
    }

    if (1 == argc) {
	/* no arguments, libvirt expects help text on stdout */
	usage(stdout);
	exit(0);
    }

    if (!xen.qemu.upstream) {
	qemu_disk_config_blkbackd(&xen);
	qemu_net_config_netbackd(&xen);
    }

    prepare_search_path(&xen, argv[0]);
    if (NULL == xen.kernel || bootloader)
	run_pygrub(&xen, bootloader);
    if (NULL == xen.kernel) {
	logprintf(&xen, "No kernel specified and pygrub failed.\n"); 
	exit(1);
    }

    loginit(colors);
    logprintf(&xen, "started as: ");
    for (opt = 0; opt < argc; opt++)
	logprintf(&xen, "\"%s\"%s", argv[opt], (opt == argc-1) ? "\n" : " "); 

    if (NULL == uuid) {
	uuid_t u;
	strcpy(xen.vm, "/vm/");
	uuid_generate(u);
	uuid_unparse(u, xen.vm+4);
    } else {
	sprintf(xen.vm, "/vm/%.36s", uuid);
    }

    /* signal handlers sigsegv and sigbus */
    xen.mainthread = pthread_self();
    if (!dumpcore) {
	memset(&act,0,sizeof(act));
	act.sa_flags = SA_SIGINFO | SA_RESETHAND;
	act.sa_sigaction = sigsegv;
	sigemptyset(&act.sa_mask);
	sigaction(SIGSEGV, &act, &old);
	sigaction(SIGBUS,  &act, &old);
    }

    /* init kvm */
    rc = kvmbits_init(&xen);
    if (xen.unlink_kernel)
	unlink(xen.kernel);
    if (xen.unlink_ramdisk)
	unlink(xen.ramdisk);
    if (0 != rc) {
	xenner_cleanup(&xen, 0);
	exit(1);
    }

    /* start other processes as needed */
    if (-1 != monitor_fd)
	qemu_monitor_setup(&xen, monitor_fd);
    if (xen.qemu.upstream)
	qemu_upstream_setup(&xen);
    else if (xen.qemu.vnc)
	qemu_dm_setup(&xen);
    if (xenconsole)
	run_xenconsole(&xen);

    /* go boot and run guest */
    banner_print(&xen, "boot");
    kvmbits_mainloop(&xen);
    exit(0);
}
