#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <inttypes.h>
#include <errno.h>
#include <sys/ioctl.h>

#include <linux/if_tun.h>

#include <xs.h>
#include <xen/io/xenbus.h>
#include <xen/io/xs_wire.h>

#include "xenner.h"
#include "mm.h"

/* ------------------------------------------------------------------ */

#define UUID0 "00000000-0000-0000-0000-000000000000"

static int xenstore_setup_host(struct xenvm *xen)
{
    char name[]  = "xenner-host";
    char domid[] = "0";
    char vm0[]   = "/vm/" UUID0;
    char uuid0[] = UUID0;
    unsigned int len;
    char *val;

    val = xs_read(xen->xenstore, 0, "/local/domain/0/name", &len);
    if (NULL != val) {
	/* already done */
	free(val);
	return 0;
    }

    xs_write(xen->xenstore, 0, "/vm/" UUID0 "/name",    name,  sizeof(name));
    xs_write(xen->xenstore, 0, "/vm/" UUID0 "/uuid",    uuid0, sizeof(uuid0));

    xs_write(xen->xenstore, 0, "/local/domain/0/name",  name,  sizeof(name));
    xs_write(xen->xenstore, 0, "/local/domain/0/domid", domid, sizeof(domid));
    xs_write(xen->xenstore, 0, "/local/domain/0/vm",    vm0,   sizeof(vm0));
    return 0;
}

static int xenstore_domain_set_str(struct xenvm *xen, char *relpath, char *val)
{
    char abspath[256];

    snprintf(abspath, sizeof(abspath), "/local/domain/%d/%s", xen->domid, relpath);
    if (!xs_write(xen->xenstore, 0, abspath, val, strlen(val)))
	d0printf("xs_write %s: failed\n", abspath);
    return 0;
}

static int xenstore_domain_set_int(struct xenvm *xen, char *relpath, int64_t ival)
{
    char val[32];

    snprintf(val, sizeof(val), "%" PRId64, ival);
    return xenstore_domain_set_str(xen, relpath, val);
}

static int xenstore_vm_set_str(struct xenvm *xen, char *relpath, char *val)
{
    char abspath[256];

    snprintf(abspath, sizeof(abspath), "%s/%s", xen->vm, relpath);
    if (!xs_write(xen->xenstore, 0, abspath, val, strlen(val)))
	d0printf("xs_write %s: failed\n", abspath);
    return 0;
}

static int xenstore_vm_set_int(struct xenvm *xen, char *relpath, int64_t ival)
{
    char val[32];

    snprintf(val, sizeof(val), "%" PRId64, ival);
    return xenstore_vm_set_str(xen, relpath, val);
}

static int xenstore_domain_permit(struct xenvm *xen)
{
    struct xs_permissions perms = {
	.id    = xen->domid,
	.perms = XS_PERM_WRITE,
    };
    char abspath[256];

    snprintf(abspath, sizeof(abspath), "/local/domain/%d", xen->domid);

    if (!xs_mkdir(xen->xenstore, 0, abspath)) {
	d0printf("%s: xs_mkdir %s: failed\n", __FUNCTION__, abspath);
	return -1;
    }
    if (!xs_set_permissions(xen->xenstore, 0, abspath, &perms, 1)) {
	d0printf("%s: xs_set_permissions failed\n", __FUNCTION__);
	return -1;
    }
    return 0;
}

static char *xenstore_device_protocol(struct xenvm *xen)
{
    static char *x86_32 = "x86_32-abi";
    static char *x86_64 = "x86_64-abi";

    switch (xen->mode) {
    case XENMODE_32:
    case XENMODE_PAE:
	return x86_32;
    case XENMODE_64:
	return x86_64;
    default:
	return NULL;
    }
}

static int xenstore_device_set_str(struct xenvm *xen, char *dev, char *node, char *val)
{
    char abspath[256];

    snprintf(abspath, sizeof(abspath), "%s/%s", dev, node);
    if (!xs_write(xen->xenstore, 0, abspath, val, strlen(val)))
	d0printf("xs_write %s: failed\n", abspath);
    return 0;
}

static int xenstore_device_set_int(struct xenvm *xen, char *dev, char *node, int ival)
{
    char abspath[256];
    char val[32];

    snprintf(abspath, sizeof(abspath), "%s/%s", dev, node);
    snprintf(val, sizeof(val), "%d", ival);
    if (!xs_write(xen->xenstore, 0, abspath, val, strlen(val)))
	d0printf("xs_write %s: failed\n", abspath);
    return 0;
}

static int xenstore_device_mkdir(struct xenvm *xen, char *dev, int p)
{
    struct xs_permissions perms = {
	.id    = xen->domid,
	.perms = p,
    };

    if (!xs_mkdir(xen->xenstore, 0, dev)) {
	d0printf("xs_mkdir %s: failed\n", dev);
	return -1;
    }

    if (!xs_set_permissions(xen->xenstore, 0, dev, &perms, 1)) {
	d0printf("%s: xs_set_permissions failed\n", __FUNCTION__);
	return -1;
    }
    return 0;
}

static int xenstore_device_dir(struct xenvm *xen,
			       char *ftype, char *btype, int vdev,
			       char *fe, char *be, int len)
{
    snprintf(fe, len, "/local/domain/%d/device/%s/%d",
	     xen->domid, ftype, vdev);
    snprintf(be, len, "/local/domain/0/backend/%s/%d/%d",
	     btype, xen->domid, vdev);

    xenstore_device_mkdir(xen, fe, XS_PERM_WRITE);
    xenstore_device_mkdir(xen, be, XS_PERM_READ);
    return 0;
}

static int xenstore_device_all(struct xenvm *xen, char *fe, char *be)
{
    /* frontend */
    xenstore_device_set_str(xen, fe, "protocol",
			    xenstore_device_protocol(xen));
    xenstore_device_set_int(xen, fe, "state",           XenbusStateInitialising);
    xenstore_device_set_int(xen, fe, "backend-id",      0);
    xenstore_device_set_str(xen, fe, "backend",         be);

    /* backend */
    xenstore_device_set_str(xen, be, "domain",          xen->name);
    xenstore_device_set_int(xen, be, "online",          1);
    xenstore_device_set_int(xen, be, "state",           XenbusStateInitialising);
    xenstore_device_set_int(xen, be, "frontend-id",     xen->domid);
    xenstore_device_set_str(xen, be, "frontend",        fe);

    return 0;
}

static int xenstore_device_blk(struct xenvm *xen, int index,
			       struct xendisk *disk)
{
    char fe[256], be[256];
    int vdev = 202 * 256 + 16 * index;
    char *devtype = disk->cdrom ? "cdrom" : "disk";
    char *mode    = disk->cdrom ? "r"     : "w";
    char dev[8];

    snprintf(dev, sizeof(dev), "xvd%c", 'a' + index);
    xenstore_device_dir(xen, "vbd", "blkbackd", vdev, fe, be, sizeof(fe));

    /* frontend */
    xenstore_device_set_int(xen, fe, "virtual-device",  vdev);
    xenstore_device_set_str(xen, fe, "device-type",     devtype);

    /* backend */
    xenstore_device_set_str(xen, be, "dev",             dev);
    xenstore_device_set_str(xen, be, "type",            "file");
    xenstore_device_set_str(xen, be, "params",          disk->diskimage);
    xenstore_device_set_str(xen, be, "mode",            mode);
/*
/local/domain/0/backend/vbd/3/51712/physical-device              :  "fd:7"
/local/domain/0/backend/vbd/3/51712/hotplug-status               :  "connected"
*/
    /* common stuff */
    return xenstore_device_all(xen, fe, be);
}

static int xenstore_device_net(struct xenvm *xen, int vdev,
			       struct xennic *nic)
{
    char fe[256], be[256];

    if (nic->fd != -1) {
	/* make persist and close, so netbackd can open it by name */
	ioctl(nic->fd, TUNSETPERSIST, 1);
	close(nic->fd);
	nic->fd = -1;
    }

    d1printf("%s: vnic %d: mac=\"%s\" if=\"%s\" br=\"%s\"\n", __FUNCTION__,
	     vdev, nic->mac, nic->ifname, nic->bridge);
    xenstore_device_dir(xen, "vif", "netbackd", vdev, fe, be, sizeof(fe));

    /* frontend */
    xenstore_device_set_int(xen, fe, "handle",     vdev);
    xenstore_device_set_str(xen, fe, "mac",        nic->mac);

    /* backend */
    xenstore_device_set_int(xen, be, "handle",     vdev);
    xenstore_device_set_str(xen, be, "mac",        nic->mac);
    if (nic->bridge && strlen(nic->bridge))
	xenstore_device_set_str(xen, be, "bridge", nic->bridge);
    if (nic->ifname && strlen(nic->ifname))
	xenstore_device_set_str(xen, be, "ifname", nic->ifname);

/*
/local/domain/0/backend/vif/3/0/script                           :  "/etc/xen/scripts/vif-bridge"
*/
    /* common stuff */
    return xenstore_device_all(xen, fe, be);
}

static int xenstore_device_vfb(struct xenvm *xen, int vdev, char *type)
{
    char fe[256], be[256];

    xenstore_device_dir(xen, "vfb", "vfb", vdev, fe, be, sizeof(fe));

    /* backend */
    xenstore_device_set_str(xen, be, "type",            type);
/*
/local/domain/0/backend/vfb/2/0/xauthority                       :  "/root/.Xauthority"
/local/domain/0/backend/vfb/2/0/display                          :  "localhost:10.0"
*/

    /* common stuff */
    return xenstore_device_all(xen, fe, be);
}

static int xenstore_device_vkbd(struct xenvm *xen, int vdev)
{
    char fe[256], be[256];

    xenstore_device_dir(xen, "vkbd", "vkbd", vdev, fe, be, sizeof(fe));
    return xenstore_device_all(xen, fe, be);
}

static int xenstore_device_console(struct xenvm *xen)
{
    char fe[256], be[256];

    xenstore_device_dir(xen, "console", "console", 0, fe, be, sizeof(fe));
    return xenstore_device_all(xen, fe, be);
}

static int xenstore_setup(struct xenvm *xen)
{
    char path[128];
    int i;

    xenstore_setup_host(xen);
    xenstore_domain_permit(xen);

    xenstore_vm_set_str(xen, "image/ostype",  "linux");
    xenstore_vm_set_str(xen, "image/kernel",  xen->kernel);
    if (xen->ramdisk)
	xenstore_vm_set_str(xen, "image/ramdisk", xen->ramdisk);
    if (xen->cmdline)
	xenstore_vm_set_str(xen, "image/cmdline", xen->cmdline);

    /* name + id */
    xenstore_vm_set_str(xen, "name",          xen->name);
    xenstore_vm_set_str(xen, "uuid",          xen->vm+4);
    xenstore_domain_set_str(xen, "name",      xen->name);
    xenstore_domain_set_int(xen, "domid",     xen->domid);
    xenstore_domain_set_str(xen, "vm",        xen->vm);

    /* memory */
    xenstore_domain_set_int(xen, "memory/target", xen->pg_guest * 4);
    xenstore_vm_set_int(xen, "memory",        PG_TO_MB(xen->pg_guest));
    xenstore_vm_set_int(xen, "maxmem",        PG_TO_MB(xen->pg_guest));

    /* cpus */
    for (i = 0; i < xen->vcpus; i++) {
	snprintf(path, sizeof(path), "cpu/%d/availability",i);
	xenstore_domain_set_str(xen, path, "online");
    }
    xenstore_vm_set_int(xen, "vcpu_avail",    xen->vcpus);
    xenstore_vm_set_int(xen, "vcpus",         xen->vcpus);

    /* console */
    xenstore_domain_set_str(xen, "console/type",
			    xen->qemu.upstream ? "ioemu" : "xenconsoled");
    xenstore_domain_set_int(xen, "console/limit",
			    128 * 1024);
    xenstore_domain_set_int(xen, "console/ring-ref",
			    xen->console_pfn + xen->mfn_guest);
    xenstore_domain_set_int(xen, "console/port",
			    xen->console_event);

    /* xenstore */
    xenstore_domain_set_int(xen, "store/ring-ref",
			    xen->xenstore_pfn + xen->mfn_guest);
    xenstore_domain_set_int(xen, "store/port",
			    xen->xenstore_event);

    /* devices */
    for (i = 0; i < sizeof(xen->disks)/sizeof(xen->disks[0]); i++) {
	if (!xen->disks[i].configured)
	    continue;
	xenstore_device_blk(xen, i, xen->disks + i);
    }
    for (i = 0; i < sizeof(xen->nics)/sizeof(xen->nics[0]); i++) {
	if (!xen->nics[i].configured)
	    continue;
	xenstore_device_net(xen, i, xen->nics + i);
    }
    if (xen->qemu.vnc) {
	xenstore_device_vfb(xen, 0, "vnc");
	xenstore_device_vkbd(xen, 0);
	xenstore_vm_set_str(xen, "vncpassword", "");
    }
    xenstore_device_console(xen);

    /* signal new domain */
    xs_introduce_domain(xen->xenstore,
			xen->domid,
			xen->xenstore_pfn + xen->mfn_guest,
			xen->xenstore_event);
    return 0;
}

static int xenstore_cleanup_be(struct xenvm *xen, char *btype)
{
    char abspath[256];

    snprintf(abspath, sizeof(abspath), "/local/domain/0/backend/%s/%d",
	     btype, xen->domid);
    return xs_rm(xen->xenstore, 0, abspath);
}

static int xenstore_cleanup(struct xenvm *xen)
{
    int disks = 0, nics = 0, i;
    char abspath[256];

    /* domain */
    xs_release_domain(xen->xenstore, xen->domid);
    snprintf(abspath, sizeof(abspath), "/local/domain/%d", xen->domid);
    xs_rm(xen->xenstore, 0, abspath);
    xs_rm(xen->xenstore, 0, xen->vm);

    /* backends */
    for (i = 0; i < sizeof(xen->disks)/sizeof(xen->disks[0]); i++)
	if (xen->disks[i].configured)
	    disks++;
    for (i = 0; i < sizeof(xen->nics)/sizeof(xen->nics[0]); i++)
	if (xen->nics[i].configured)
	    nics++;

    if (disks)
	xenstore_cleanup_be(xen, "blkbackd");
    if (nics)
	xenstore_cleanup_be(xen, "netbackd");
    if (xen->qemu.vnc) {
	xenstore_cleanup_be(xen, "vfb");
	xenstore_cleanup_be(xen, "vkbd");
    }
    xenstore_cleanup_be(xen, "console");

    return 0;
}

/* ------------------------------------------------------------------ */

/* before building domain */
int xenstore_init_early(struct xenvm *xen)
{
    xen->xenstore = xs_daemon_open();
    if (NULL == xen->xenstore) {
	d0printf("%s: can't connect to %s (xenstored not running?)\n",
		 __FUNCTION__, xs_daemon_socket());
	return -1;
    }

    xen->xenstore_event = evtchn_port(xen, "xenstore");
    return 0;
}

/* after building domain (shared pages known) */
int xenstore_init_late(struct xenvm *xen)
{
    xenstore_setup(xen);
    return 0;
}

int xenstore_shutdown(struct xenvm *xen, char *action)
{
    logprintf(xen, "trigger guest shutdown\n");
    xenstore_domain_set_str(xen, "control/shutdown", action);
    return 0;
}

void xenstore_fini(struct xenvm *xen)
{
    if (!xen->xenstore)
	return;
    xenstore_cleanup(xen);
    xs_daemon_close(xen->xenstore);
}
