#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <dlfcn.h>
#include <stdarg.h>
#include <assert.h>
#include <dirent.h>
#include <inttypes.h>
#include <fcntl.h>

#include <sys/stat.h>
#include <sys/mman.h>
#include <sys/types.h>

#include <xenctrl.h>

#include "xennerctrl.h"
#include "shared.h"

/* --------------------------------------------------------------------- */

static LIST_HEAD(doms);
static int domcount;
static pthread_mutex_t domlock;

/* --------------------------------------------------------------------- */

static int parse_elf_dom(struct xennerdom *dom)
{
    void *ptr;
    char *name;
    int i, size;

    if (dom->parsed)
	return 0;
    
    dom->vm_ehdr  = dom->vmcore;
    dom->vm_phdrs = dom->vmcore + dom->vm_ehdr->e_phoff;
    dom->vm_shdrs = dom->vmcore + dom->vm_ehdr->e_shoff;
    dom->vm_shstrtab = dom->vmcore +
	dom->vm_shdrs[dom->vm_ehdr->e_shstrndx].sh_offset;

    for (i = 0; i < dom->vm_ehdr->e_shnum; i++) {
	name = dom->vm_shstrtab + dom->vm_shdrs[i].sh_name;
	ptr  = dom->vmcore + dom->vm_shdrs[i].sh_offset;
#if 0
	fprintf(stderr, "  %d: %s at %" PRIx64 " +%" PRIx64 " (%p)\n", i,
		name, dom->vm_shdrs[i].sh_offset, dom->vm_shdrs[i].sh_size, ptr);
#endif
	if (0 == strcmp(name, ".addr.shared_info")) {
	    dom->shared_info = ptr;
	    dom->parsed++;
	}
	if (0 == strcmp(name, ".addr.grant_table")) {
	    size = dom->vm_shdrs[i].sh_size;
	    dom->grant_entries = size / sizeof(dom->grant_table[0]);
	    dom->grant_table = ptr;
	    dom->parsed++;
	}
	if (0 == strcmp(name, ".addr.vminfo")) {
	    dom->vminfo = ptr;
	    dom->parsed++;
	}
    }
    return 0;
}

static int init_dom(struct xennerdom *dom)
{
    struct stat st;

    snprintf(dom->filename, sizeof(dom->filename),
	     "/var/run/xenner/vmcore.%d", dom->domid);
    dom->fd = open(dom->filename, O_RDWR);
    if (-1 == dom->fd)
	return -1;
    fstat(dom->fd, &st);
    dom->uid         = st.st_uid;
    dom->gid         = st.st_gid;
    dom->vmcore_size = st.st_size;
    dom->vmcore = mmap(NULL, dom->vmcore_size, PROT_READ | PROT_WRITE,
		       MAP_SHARED, dom->fd, 0);
    if ((void*)-1 == dom->vmcore) {
	fprintf(stderr, "%s: mmap(%s,%" PRId64 "): %s\n", __FUNCTION__,
		dom->filename, (uint64_t)dom->vmcore_size, strerror(errno));
	dom->vmcore = NULL;
	close(dom->fd);
	dom->fd = -1;
	return -1;
    }
    return 0;
}

static int fini_dom(struct xennerdom *dom)
{
    if (dom->vmcore)
	munmap(dom->vmcore, dom->vmcore_size);
    if (-1 != dom->fd)
	close(dom->fd);
    return 0;
}

static struct xennerdom *find_dom(int domid)
{
    struct xennerdom *dom;
    struct list_head *item;

    pthread_mutex_lock(&domlock);
    list_for_each(item, &doms) {
	dom = list_entry(item, struct xennerdom, next);
	if (dom->domid == domid) {
	    pthread_mutex_unlock(&domlock);
	    return dom;
	}
    }
    pthread_mutex_unlock(&domlock);
    return NULL;
}

struct xennerdom *xenner_get_dom(int domid)
{
    struct xennerdom *dom;

    dom = find_dom(domid);
    if (!dom) {
	dom = malloc(sizeof(*dom));
	memset(dom,0,sizeof(*dom));
	dom->domid = domid;

	if (-1 == init_dom(dom)) {
	    free(dom);
	    return NULL;
	}

	pthread_mutex_lock(&domlock);
	list_add_tail(&dom->next, &doms);
	domcount++;
	pthread_mutex_unlock(&domlock);
	if (libxc_trace)
	    fprintf(stderr, "%s: %d\n", __FUNCTION__, dom->domid);
    }

    dom->refcount++;
    parse_elf_dom(dom);
    return dom;
}

void xenner_put_dom(struct xennerdom *dom)
{
    dom->refcount--;
    if (0 != dom->refcount)
	return;

    if (libxc_trace)
	fprintf(stderr, "%s: %d\n", __FUNCTION__, dom->domid);
    pthread_mutex_lock(&domlock);
    list_del(&dom->next);
    domcount--;
    pthread_mutex_unlock(&domlock);

    fini_dom(dom);
    free(dom);
}

