#ifndef longmode
# error do not include this directly but via emu.h
#endif

struct regs_64 {
    /* pushed onto stack before calling into C code */
    uint64_t rax;
    uint64_t rbx;
    uint64_t rcx;
    uint64_t rdx;
    uint64_t r8;
    uint64_t r9;
    uint64_t r10;
    uint64_t r11;
    uint64_t r12;
    uint64_t r13;
    uint64_t r14;
    uint64_t r15;
    uint64_t rsi;
    uint64_t rdi;
    uint64_t rbp;
    uint64_t trapno;
    /* trap / fault / int created */
    uint64_t error;
    uint64_t rip;
    uint64_t cs;
    uint64_t rflags;
    uint64_t rsp;
    uint64_t ss;
};

/* 64bit defines */
#define EMUNAME   "emu64"
#define regs      regs_64
#define fix_sel   fix_sel64
#define fix_desc  fix_desc64
#define ureg_t    uint64_t
#define sreg_t    int64_t
#define PRIxREG   PRIx64
#define tss(_v)   ((0xe000 >> 3) +  8 + (((_v)->id) << 2))
#define ldt(_v)   ((0xe000 >> 3) + 10 + (((_v)->id) << 2))

/* emu64.S */
extern uint64_t emu_pgd[];

/* emu-data.c */
extern struct idt_64 page_aligned xen_idt[256];
extern uint64_t *m2p;

extern uint64_t user_mode;
extern uint64_t kernel_cr3_mfn;
extern uint64_t user_cr3_mfn;

/* emu-main.c */
asmlinkage void do_int_80(struct regs_64 *regs);

/* emu-hcall.c */
void switch_mode(struct xen_cpu *cpu);
int is_kernel(struct xen_cpu *cpu);
asmlinkage void do_syscall(struct regs_64 *regs);

/* emu-mm.c */
void pgtable_walk(int level, uint64_t va, uint64_t root_mfn);
int pgtable_fixup_flag(uint64_t va, uint32_t flag);
void *map_page(uint64_t maddr);
void *fixmap_page(uint64_t maddr);
static inline void free_page(void *ptr) {}
uint64_t *find_pte_64(uint64_t va);

/* macros */
#define context_is_emu(_r)       (((_r)->cs & 0x03) == 0x00)
#define context_is_kernel(_v,_r) (((_r)->cs & 0x03) == 0x03 && is_kernel(_v))
#define context_is_user(_v,_r)   (((_r)->cs & 0x03) == 0x03 && !is_kernel(_v))

#define addr_is_emu(va)     (((va) >= XEN_M2P_64) && ((va) < XEN_DOM_64))
#define addr_is_kernel(va)  ((va) >= XEN_DOM_64)
#define addr_is_user(va)    ((va) < XEN_M2P_64)

/* inline asm bits */
static inline int wrmsr_safe(uint32_t msr, uint32_t ax, uint32_t dx)
{
    int ret;
    asm volatile("1:  wrmsr                \n"
		 "    xorl %0,%0           \n"
		 "2:  nop                  \n"

		 ".section .exfix, \"ax\"  \n"
		 "3:  mov $-1,%0           \n"
		 "    jmp 2b               \n"
		 ".previous                \n"

		 ".section .extab, \"a\"   \n"
		 "    .align 8             \n"
		 "    .quad 1b,3b          \n"
		 ".previous                \n"
		 : "=r" (ret)
		 : "c" (msr), "a" (ax), "d" (dx));
    return ret;
}

static inline int memcpy_pf(void *dest, const void *src, size_t bytes)
{
    int ret;

    asm volatile("    cld                  \n"
		 "91: rep movsb            \n"
		 "    xor %[ret],%[ret]    \n"
		 "98:                      \n"

		 ".section .exfix, \"ax\"  \n"
		 "99: mov $-1, %[ret]      \n"
		 "    jmp 98b              \n"
		 ".previous                \n"

		 ".section .extab, \"a\"   \n"
		 "    .align 8             \n"
		 "    .quad 91b,99b        \n"
		 ".previous                \n"
		 : [ ret ] "=a" (ret),
		   [ rsi ] "+S" (src),
		   [ rdi ] "+D" (dest),
		   [ rcx ] "+c" (bytes)
		 :
		 : "memory" );
    return ret;
}
